/*
 * Copyright (c) 2013 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jmerge_i.c -- Initialization Routines
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _MSDOS
#include <sys/param.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jmerge.h"

#define SCKARG 80
#define SIZEC  30

/*
 * init_w() -- initialize work area
 */
void init_w(struct s_work *w)

{
  init_finfo(&(w->key));
  init_finfo(&(w->out));
  init_finfo(&(w->err));
  init_finfo(&(w->stats));
  w->err.fp    = stderr;
  w->out.fp    = stdout;

  w->arg_verbose      = 0;
  w->arg_force        = FALSE;
  w->arg_sorted       = TRUE;
  w->arg_heading      = FALSE;
  w->arg_delm         = JLIB2_CHAR_NULL;
  w->arg_pause_reads  = PAUSE_RECS;
  w->arg_milliseconds = 0;
  w->num_files        = 0;

}  /* init_w() */

/*
 * print_c() - format the character or its hex value for printing
 */
char *print_c(char *str, int sizc, char c)
{
  unsigned char u = (unsigned char) c;

  switch (u)
    {
      case 7:
	snprintf(str, sizc, "%s", "BEL");
	return(str);
      case 9:
	snprintf(str, sizc, "%s", "TAB");
	return(str);
      case 32:
	snprintf(str, sizc, "%s", "space");
	return(str);
    }

  if ((u < 33) || (u > 126))
    {
      snprintf(str, sizc, "0x%02X", u);
      return(str);
    }

  snprintf(str, sizc, "%c", c);
  return(str);

} /* print_c() */

/*
 * save_fname() -- Check and Save File Name
 */
void save_fname(struct s_file_info *f, char *afname, char arg_val, int required)
{

  if (strlen(afname) < 1)
    {
      if (required == TRUE)
	{
	  fprintf(stderr, MSG_ERR_E004S, afname,   SWITCH_CHAR, arg_val);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
      return;
    }
  if (strcmp(afname, FILE_NAME_STD) == 0)
    {
      if (required == TRUE)
	{
	  fprintf(stderr, MSG_ERR_E004S, afname,   SWITCH_CHAR, arg_val);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
      return;
    }

  if (strlen(afname) > 0)
    {
      check_path_file(stderr, afname, arg_val);
      strncpy(f->fname, afname, PATH_MAX);
    }
  else
    {
      fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, arg_val);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* save_fname() */

/*
 * init_get_delm() -- translate a string into a delimiter
 */
char init_get_delm(FILE *fp, char *s)
{
  int d;
  int c = JLIB2_CHAR_NULL;

  if (s == (char *) NULL)
    return(c);

  if (strlen(s) == 1)
    {
      if ( ((*s) > 47)  && ((*s) < 58) ) /* 0 -- 9 */
	c = (*s) - 48;
      else
	c = (*s);
    }
  else
    {
      if (j2_is_numr(s) == (int) TRUE)
	{
	  d = atoi(s);
	  if ((d < 256) && (d > 0))
	    c = (char) d;
	  else
	    {
	      fprintf(fp, MSG_ERR_E049, s);
	      fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	}
      else
	{
	  fprintf(fp, MSG_ERR_E049, s);
	  fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  return(c);

} /* init_get_delm() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SCKARG];
  int opt       = 0;
  int i         = 0;
  int count_std = 0;

  memset(ckarg, '\0', SCKARG);
  snprintf(ckarg, (SCKARG - 1), "%c%c%c%c%c%c%c:%c:%c:%c:%c:%c:%c:",
	   ARG_FORCE, ARG_HELP, ARG_VERBOSE, ARG_VERSION, ARG_UNSORTED,
	   ARG_FIRST_HEADG,
	   ARG_INPUT, ARG_ERR,  ARG_OUT, ARG_DELM, ARG_KEY_STATS,
	   ARG_PAUSE, ARG_READS);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	case ARG_FORCE:
	  w->arg_force = TRUE;
	  break;
	case ARG_UNSORTED:
	  w->arg_sorted = FALSE;
	  break;
	case ARG_FIRST_HEADG:
	  w->arg_heading = TRUE;
	  break;
	case ARG_DELM:
	  w->arg_delm  = init_get_delm(stderr, optarg);
	  break;
	case ARG_HELP:
	  show_brief_help(stderr, PROG_NAME);
	  break;
	case ARG_VERBOSE:
	  w->arg_verbose++;
	  break;
	case ARG_VERSION:
	  show_rev(stderr, PROG_NAME);
	  break;
	case ARG_INPUT:
	  save_fname(&(w->key), optarg, ARG_INPUT, TRUE);
	  break;
	case ARG_KEY_STATS:
	  save_fname(&(w->stats), optarg, ARG_KEY_STATS, TRUE);
	  break;
	case ARG_ERR:
	  save_fname(&(w->err), optarg, ARG_ERR, FALSE);
	  break;
	case ARG_OUT:
	  save_fname(&(w->out), optarg, ARG_OUT, FALSE);
	  break;
	case ARG_PAUSE:
	  if (j2_is_numr(optarg) == (int) TRUE)
	    w->arg_milliseconds = atol(optarg);
	  else
	    {
	      fprintf(stderr, MSG_ERR_E008, optarg, SWITCH_CHAR, ARG_PAUSE);
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_READS:
	  if (j2_is_numr(optarg) == (int) TRUE)
	    w->arg_pause_reads = atol(optarg);
	  else
	    {
	      fprintf(stderr, MSG_ERR_E008, optarg, SWITCH_CHAR, ARG_READS);
	      exit(EXIT_FAILURE);
	    }
	  break;
	default:
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	  break;
	}
    }

  /*** Open err file ***/
  open_out(stderr, &(w->err), w->arg_force, FALSE);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->arg_verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->arg_verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->arg_verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(w->err.fp, MSG_ERR_E113);
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }


} /* END process_arg() */

/*
 * init() -- initialize
 */
void init(int argc, char **argv, struct s_work *w)

{
  char str[(SIZEC + 1)];

  memset(str, 0, (SIZEC + 1));
  init_w(w);

  process_arg(argc, argv, w);

  /*** validate and process some arguments ***/
  if (strlen(w->key.fname) < 1)
    {
      fprintf(w->err.fp, MSG_ERR_E022, SWITCH_CHAR, ARG_INPUT);
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }
  if ( (w->arg_milliseconds > 0) && (w->arg_pause_reads < 1) )
    {
      fprintf(w->err.fp, MSG_ERR_E097, SWITCH_CHAR, ARG_READS, SWITCH_CHAR, ARG_PAUSE);
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  if (w->arg_delm == JLIB2_CHAR_NULL)
    w->arg_delm = '|';

  /*** open other files ***/
  open_out(w->err.fp, &(w->out), w->arg_force, FALSE);
  if (strlen(w->stats.fname) > 0)
    open_out(w->err.fp, &(w->stats), w->arg_force, TRUE);

  if (open_in(w->err.fp, &(w->key), TRUE) == FALSE)
    exit(EXIT_FAILURE);

  /*** show arguments ***/
  if (w->arg_verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I193,
              (strlen(w->key.fname)  == 0 ? LIT_STDIN  : w->key.fname));
      if (strlen(w->stats.fname) > 0)
	fprintf(w->err.fp, MSG_INFO_I194, w->stats.fname);
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname)  == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname)  == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->arg_verbose);
      fprintf(w->err.fp, MSG_INFO_I140S, print_c(str, SIZEC, w->arg_delm));
      fprintf(w->err.fp, MSG_INFO_I087, (w->arg_force   == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I195, (w->arg_heading == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I196, (w->arg_sorted  == TRUE ? LIT_YES : LIT_NO));
      if (w->arg_milliseconds > 0)
	fprintf(w->err.fp, MSG_INFO_I157, w->arg_milliseconds, w->arg_pause_reads);
    }
  
}  /* end: init() */
