/*
 * Copyright (c) 2007 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jhead.h"

#define MAXARG 80

/*
 * verify_path() -- make sure dir/file name is OK
 */
int verify_path(char *fname)
{
  char *s;

  if (fname == (char *) NULL)
    return(PATH_IS_NULL);
  if (strlen(fname) < 1)
    return(PATH_IS_EMPTY);
  if (strlen(fname) > (PATH_MAX - 2))
    return(PATH_IS_BIG);
  if ((strlen(fname) > 1) && (*fname == '-'))
    return(PATH_INVALID);

  for (s = fname; (*s) != JLIB2_CHAR_NULL; s++)
    {
      if (isspace((int) (*s)))
	return(PATH_HAS_SPACE);
    }

  return(PATH_IS_VALID);

} /* verify_path() */

/*
 * check_path_file() -- Check and Save a Dir Name
 */
void check_path_file(FILE *efp, char *fname, char arg_switch)
{

  char sw = SWITCH_CHAR;
  char ag = arg_switch;

  if (arg_switch == JLIB2_CHAR_NULL)
    {
      sw = 'i';
      ag = 'n';
    }

  switch (verify_path(fname))
    {
      case PATH_IS_VALID:
	break;
      case PATH_IS_NULL:
	fprintf(efp, MSG_ERR_E004S, LIT_NULL,  sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_EMPTY:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_BIG:
	fprintf(efp, MSG_ERR_E004SB, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,   PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_HAS_SPACE:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_INVALID:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      default:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
    }

} /* check_path_file() */

/*
 * init_finfo() - clear file information
 */
void init_finfo(struct s_file_info *f)
{

  f->fp          = (FILE *) NULL;
  f->ok_to_close = FALSE;
  memset(f->fname, JLIB2_CHAR_NULL, (PATH_MAX + 1));

} /* init_finfo() */

/*
 * clear_work()
 */
void clear_work(work_area *w)

{

  char *e = (char *) NULL;

  w->num_files     = 0;
  w->force         = FALSE;
  w->verbose       = 0;
  w->show_ln       = FALSE;
  w->start_line    = HEAD_LINES_START;
  w->end_line      = 0L;
  w->show_lines    = HEAD_LINES_DEFAULT;
  w->lines_written = 0L;

  init_finfo(&(w->out));
  init_finfo(&(w->err));

  w->out.fp = stdout;
  w->err.fp = stderr;

  /*** default args based upon environment ***/
  e = getenv(ARG_JHEAD_N);
  if (e != (char *) NULL)
    {
      if (j2_is_numr(e) == TRUE)
	w->show_lines  = atol(e);
    }

} /* clear_work() */

/*
 * open_out() -- save the file name and check status
 */
void open_out(struct s_file_info *f, char *fname, int force)

{

  if (fname == (char *) NULL)
    return;
  if (strlen(fname) < 1)
    return;
  if (strcmp(fname, FILE_NAME_STD) == 0)
    return;
  if (f->ok_to_close == TRUE) /* file in use */
    {
      fprintf(stderr, MSG_ERR_E112, fname);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  if (force == FALSE)
    {
      if ( j2_f_exist(fname) )
	{
	  fprintf(stderr, MSG_ERR_E025, fname);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  f->fp = fopen(fname, "w");
  if (f->fp == (FILE *) NULL)
    {
      fprintf(stderr, MSG_ERR_E002, fname);
      fprintf(stderr, "\t%s\n", strerror(errno));
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  strncpy(f->fname, fname, PATH_MAX);
  f->ok_to_close = TRUE;
  return;

} /* open_out() */

/*
 * close_file() -- close an opened file
 */
void close_file(struct s_file_info *f)

{
  if (f->ok_to_close == TRUE)
    fclose(f->fp);

  init_finfo(f);

} /* close_file() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, work_area *w)

{

  char ckarg[MAXARG];
  int opt         = 0;
  int i           = 0;
  int count_std   = 0;
  char *ofile     = (char *) NULL;
  char *efile     = (char *) NULL;

  snprintf(ckarg, MAXARG, "%c%c%c%c%c%c:%c:%c:%c:%c:",
	  ARG_COUNT, ARG_FORCE, ARG_HELP, ARG_VERBOSE, ARG_VERSION,
	  ARG_ERR, ARG_OUT, ARG_NUMB, ARG_START, ARG_END);

  opterr = 1;  /* bypass print of messages from getopt() */
  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	  case ARG_COUNT:
	    w->show_ln = TRUE;
	    break;
	  case ARG_END:
	    w->end_line = atol(optarg);
	    if (w->end_line < 1)
	      {
		fprintf(stderr, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_NUMB);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_ERR:
	    if (efile == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_ERR);
		efile = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_FORCE:
	    w->force = TRUE;
	    break;
	  case ARG_HELP:
	    show_brief_help();
	    break;
	  case ARG_NUMB:
	    w->show_lines = atol(optarg);
	    if (w->show_lines < 1)
	      {
		fprintf(stderr, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_NUMB);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_OUT:
	    if (ofile == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_OUT);
		ofile = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_START:
	    w->start_line = atol(optarg);
	    if (w->start_line < 1)
	      {
		fprintf(stderr, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_START);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_VERBOSE:
	    w->verbose++;
	    break;
	  case ARG_VERSION:
	    show_rev();
	    break;
	  default:
	    fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(EXIT_FAILURE);
	    break;
	}
    }

  /*** if necessary open a real file ***/
  open_out(&(w->err), efile, w->force);
  open_out(&(w->out), ofile, w->force);

  /*** calculate number of lines ***/
  if (w->end_line > 0L)
    {
      if (w->start_line > w->end_line)
	{
	  fprintf(w->err.fp, MSG_ERR_E111, (long long int) w->start_line, (long long int) w->end_line);
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
      w->show_lines = w->end_line - w->start_line + 1;
    }

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* END process_arg() */

/*
 * init() -- setup for Run
 */
void init(work_area *w, int argc, char **argv)

{

  clear_work(w);

  process_arg(argc, argv, w);

  if (w->verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) < 1 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) < 1 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      if (w->start_line > 0L)
	fprintf(w->err.fp, MSG_INFO_I163, (long long int) w->start_line);
      if (w->end_line > 0L)
	fprintf(w->err.fp, MSG_INFO_I164, (long long int) w->end_line);
      else
	fprintf(w->err.fp, MSG_INFO_I165, (long long int) w->show_lines);
      fprintf(w->err.fp, MSG_INFO_I087, (w->force   == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I162, (w->show_ln == TRUE ? LIT_YES : LIT_NO));
    }

} /* process_arg() */
