/*
 * Copyright (c) 2007 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#ifdef OpenBSD
#include <err.h>
#endif
#ifdef __FreeBSD_version
#include <err.h>
#endif
#ifdef __NetBSD_Version__
#include <err.h>
#endif

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jhead.h"

/*
 * open_in() -- open in file
 */
int open_in(struct s_file_info *err, struct s_file_info *ifile, char *fname)

{

  if (fname == (char *) NULL)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strlen(fname) < 1)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strncmp(fname, FILE_NAME_STD, PATH_MAX) == 0)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (ifile->ok_to_close == TRUE) /* file in use */
    {
      fprintf(err->fp, MSG_ERR_E112, fname);
      fprintf(err->fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  ifile->fp = fopen(fname, "r");

  if (ifile->fp == (FILE *) NULL)
    {
      fprintf(err->fp, MSG_WARN_W002, fname, strerror(errno));
      return(FALSE);
    }
 
  strncpy(ifile->fname, fname, PATH_MAX);
  ifile->ok_to_close = TRUE;
  return(TRUE);

} /* open_in() */

/*
 * show_file_heading() -- Show run stats
 */
long int show_file_heading(FILE *fp, char *fname)

{
  long int writes = 0L;

  fprintf(fp, "%s\n", LIT_C80);
  writes++;

  if (fname == (char *) NULL)
    {
      fprintf(fp, "%s\n", LIT_STDIN);
      writes++;
    }
  else
    {
      if (strcmp(fname, FILE_NAME_STD) == 0)
	fprintf(fp, "%s\n", LIT_STDIN);
      else
	fprintf(fp, "%s\n", fname);
      writes++;
    }

  fprintf(fp, "%s\n", LIT_C80);
  writes++;

  return(writes);

} /* show_file_heading() */

/*
 * process_a_file()
 */
void process_a_file(work_area *w, char *fname,
                    char **buf,   size_t *bsiz)

{

  long int lines_read = 0L;
  long int shown = 0L;
  struct s_file_info ifile;

  if (w->verbose > 2)
    w->lines_written += show_file_heading(w->out.fp, fname);

  init_finfo(&ifile);

  if (open_in(&w->err, &ifile, fname) == FALSE)
    return;

  while (j2_getline(buf, bsiz, ifile.fp) > (ssize_t) -1)
    {
      lines_read++;
      if (lines_read >= w->start_line)
        {
          shown++;
	  if ( w->show_ln )
	    {
	      fprintf(w->out.fp, "%ld: %s", lines_read, (*buf));
	      w->lines_written++;
	    }
	  else
	    {
	      fprintf(w->out.fp, "%s", (*buf));
	      w->lines_written++;
	    }
          if (shown >= w->show_lines)
            break;
        }
    }

  if (w->verbose > 0)
    {
      fprintf(w->err.fp, MSG_INFO_I072, (long int) lines_read,
              (strlen(ifile.fname) < 1 ? LIT_STDIN : fname));
    }

  close_file(&ifile);

} /* process_a_file() */

/*
 * process_all()
 */
void process_all(int argc, char **argv, work_area *w)

{
  char *buf = (char *) NULL;
  size_t bsiz = (size_t) 0;
  int i;

  for (i = optind; i < argc; i++)
    process_a_file(w, argv[i], &buf, &bsiz);

  if (i == optind)
    process_a_file(w, FILE_NAME_STD, &buf, &bsiz);

  if (buf != (char *) NULL)
    free(buf);

} /* process_all() */

/*
 * main()
 */
int main(int argc, char **argv)

{
  time_t tstart = time(&tstart), tend;
  work_area w;

#ifdef OpenBSD
  if(pledge("stdio rpath wpath cpath",NULL) == -1)
    err(1,"pledge\n");
#endif

  init(&w, argc, argv);

  process_all(argc, argv, &w);

  /*** DONE ***/
  if (w.verbose > 0)
    {
      fprintf(w.err.fp, MSG_INFO_I080, (long int) w.lines_written,
	      (strlen(w.out.fname) < 1 ? LIT_STDOUT : w.out.fname));
      fprintf(w.err.fp, MSG_INFO_I152S, (long long int) (time(&tend) - tstart));
    }
  close_file(&(w.out));
  close_file(&(w.err));

  exit(EXIT_SUCCESS);

}  /* main() */
