/*
 * Copyright (c) 2003 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jgrep_s.c -- Search routines
 */

#define _GNU_SOURCE  /* needed for strcasestr() on Linux */

#ifndef _MSDOS
#include <sys/param.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "jgrep.h"

/*
 * get_field_value() - extract field value from a delimited record
 */
char *get_field_value(char *buf, size_t bsize, struct s_work *w)
{
  long int i  = 0L;
  long int j  = 0L;
  long int nd = 1L;
  size_t use_siz = w->field_siz - 5;
  char *loc = (char *) NULL;

  if (buf == (char *) NULL)
    return((char *) NULL);
  if (strlen(buf) < 1)
    return((char *) NULL);
  if ((w->col_search == 1) && (buf[0] == w->delm))
    return((char *) NULL);

  for ( ; buf[i] != JLIB2_CHAR_NULL; i++)
    {
      if (nd == w->col_search)
	break;
      if (buf[i] == w->delm)
	{
	  nd++;
	  j = i + 1;
	  if (buf[j] == JLIB2_CHAR_NULL)
	    return((char *) NULL);
	  if (buf[j] == w->delm)
	    loc = (char *) NULL;
	  else
	    loc = &(buf[j]);
	}
    }

  if (loc == (char *) NULL)
    return((char *) NULL);

  if (bsize >= use_siz)
    {
      if (w->field_val != (char *) NULL)
	free(w->field_val);
      w->field_siz = bsize * 2;
      use_siz = w->field_siz - 5;
      w->field_val = calloc(w->field_siz, sizeof(char));
      if (w->field_val == (char *) NULL)
	{
	  fprintf(w->err.fp, MSG_ERR_E080, strerror(errno));
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	}
    }
#if OpenBSD
  strlcpy(w->field_val, loc, w->field_siz);
#else
  /* strcpy is much faster and no worries about overflow, see above */
  strcpy(w->field_val, loc);
#endif

  loc = w->field_val;
  while ((*loc) != JLIB2_CHAR_NULL)
    {
      if (*loc == w->delm)
	{
	  *loc = JLIB2_CHAR_NULL;
	  break;
	}
      loc++;
    }

  return(w->field_val);

} /* get_field_value() */

/*
 * search_fixed() - search a record for a string of a specific size
 *                  and at a specific column (character)
 */
int search_fixed(char *buf, struct s_search *needle, size_t needle_siz, int ignore_case)
{

  if (ignore_case == TRUE)
    return(strncasecmp(buf, needle->str, needle_siz));
  else
    return(strncmp(buf, needle->str, needle_siz));

} /* search_fixed() */

/*
 * search_buf() - search a record for a string, return TRUE if found
 */
int search_buf(struct s_search *needle, char *buf, long int col_search, int ignore_case)
{
  int results = 0;
  char *found = (char *) NULL;
  char *s = buf;

  if (buf == (char *) NULL)
    return(FALSE);

  if (col_search == 0)
    {
      if (ignore_case == TRUE)
	found = strcasestr(buf, needle->str);
      else
	found = strstr(buf, needle->str);
      if (found == (char *) NULL)
	results = FALSE;
      else
	results = TRUE;
    }
  else
    {
      if (strlen(buf) < (size_t) col_search)
	return(FALSE);
      s = &(buf[col_search]);
      if (ignore_case == TRUE)
	results = strcmp(s, needle->str);
      else
	results = strcasecmp(s, needle->str);
      if (results == 0)
	results = TRUE;
      else
	results = FALSE;
    }

  return(results);

} /* search_buf() */

/*
 * search() -- see if the entry is in the buffer
 */
void search(struct s_work *w, struct s_search *needle, char *buf, size_t bsize, int *match_made)
{

  /*** process non-delimited record ***/
  if (w->delm == JLIB2_CHAR_NULL)
    {
      if (w->col_search < 1)
	{
	  if (search_buf(needle, buf, w->col_search, w->ignore_case) == TRUE)
	    goto matchfound;
	  goto no_match;
	}
      if (strlen(buf) < (size_t) w->col_search)
	goto no_match;
      if (search_fixed(&(buf[(w->col_search - 1)]), needle, strlen(needle->str), w->ignore_case) == 0)
	goto matchfound;
      goto no_match;
    }

  /*** searching specific field of a delimited record ***/
  if (get_field_value(buf, bsize, w) == (char *) NULL)
    goto no_match;

  if (w->ignore_case == TRUE)
    {
      if (strcasecmp(w->field_val, needle->str) == 0)
	goto matchfound;
    }
  else
    {
      if (strcmp(w->field_val, needle->str) == 0)
	goto matchfound;
    }
  goto no_match;

  no_match:
    needle->missing++;
    if (w->invert_match == TRUE)
      {
	w->print_results = TRUE;
      }
    return;

  matchfound:
    *match_made = TRUE;
    needle->found++;
    if (w->invert_match == FALSE)
      {
	w->print_results = TRUE;
      }
    return;

} /* search() */
