/*
 * Copyright (c) 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jgrep_r.c -- load search data to a linked list
 */

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jgrep.h"

/*
 * search_write() -- write search data data
 */
void search_write(FILE *fp, struct s_search *search_list)
{
  struct s_search *now    = search_list;

  fprintf(fp, "\nSearch Data Status:\n%s\n", LIT_DLINE_19);
  fprintf(fp, "%11s %11s %11s %s\n",
          "Search Rec", "Mactches", "No Match", "Search Value");
  fprintf(fp, "%11s %11s %11s %s\n",
          LIT_ULINE_11, LIT_ULINE_11, LIT_ULINE_11, LIT_ULINE_42);

  while (now != (struct s_search *) NULL)
    {
      fprintf(fp, "%11ld %11ld %11ld %s\n",
	      now->rec_num, now->found, now->missing, now->str);
      now = now->next;
    }
  fprintf(fp, "\n");

} /* search_write() */

/*
 * search_init() -- Initialize search info
 */
void search_init(struct s_search *r)
{

  r->found    = 0L;
  r->missing  = 0L;
  r->str       = (char *) NULL;
  r->next      = (struct s_search *) NULL;

} /* search_init() */

/*
 * search_free() -- free memory
 *                  this logic avoids a recursion stack overflow
 *                  Freeing some large lists caused a stack overflow
 */
void search_free(struct s_search **search_list)
{
  struct s_search *now   = (*search_list);
  struct s_search *prior = (*search_list);

  while (now != (struct s_search *) NULL)
    {
      prior = now;
      now = now->next;
      if (prior->str != (char *) NULL)
	free(prior->str);
      free(prior);
    }

  (*search_list) = (struct s_search *) NULL;

} /* search_free() */

/*
 * search_save() -- get memory and save data
 */
struct s_search *search_save(char *buf, long int rcount)
{
  struct s_search *search_list = (struct s_search *) NULL;

  search_list = calloc(1, sizeof (struct s_search));
  search_init(search_list);

  search_list->str           = strdup(buf);
  search_list->rec_num       = rcount;
  search_list->sort_exceeded = FALSE;

  return(search_list);

} /* search_save() */

/*
 * search_load() -- load text data into memory
 */
struct s_search *search_load(struct s_work *w, long int *rcount)
{
  long int rbypassed = 0L;
  size_t bsize = (size_t) 0;
  char *buf = (char *) NULL;
  struct s_search *search_list, *now, *new;

  search_list = now = new = (struct s_search *) NULL;

  if ( ! open_in(&(w->pattern.fp), w->pattern.fname, w->err.fp) )
    exit(FAILED_PROCESS);

  /* allocate initial memory (optional) */
  bsize = REC_SIZE_INIT;
  buf = (char *) calloc(REC_SIZE_INIT, sizeof(char));
  if (buf == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_ERR_E003, w->pattern.fname);
      fprintf(w->err.fp, "\t%s\n", strerror(errno));
      exit(FAILED_PROCESS);
    }

  while (getline(&buf, &bsize, w->pattern.fp) >= (ssize_t) 0)
    {
      (*rcount)++;
      j2_bye_nl(buf);
      j2_bye_ctlm(buf);
      if (strlen(buf) < 1)
	{
	  rbypassed++;
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_WARN_W037, *rcount, w->pattern.fname);
	  continue;
	}
      new = search_save(buf, *rcount);
      if (search_list == (struct s_search *) NULL)
	search_list = new;
      else
	now->next = new;
      now = new;
      memset(buf, 0, bsize);
    }

  if (buf != (char *) NULL)
    free(buf);

  if (w->verbose > 1)
    fprintf(w->err.fp, MSG_INFO_I072, (long int) w->search_reads,  w->pattern.fname);

  close_out(&(w->pattern));

  if (rbypassed == *rcount)
    {
      fprintf(w->err.fp, MSG_ERR_E110, w->pattern.fname);
      exit(FAILED_PROCESS);
    }
  return(search_list);

} /* search_load() */
