/*
 * Copyright (c) 2009 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jgrep_i.c -- Initialization Routines
 */
#ifndef _MSDOS
#include <sys/param.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jgrep.h"

#define SCKARG 120
#define SIZEC  30

/*
 * init_counts() -- initialize counts
 */
void init_counts(struct s_counts *c)

{
  c->reads   = 0L;
  c->writes  = 0L;
} /* init_counts() */

/*
 * init_w() -- initialize work area
 */
void init_w(struct s_work *w)

{
  init_finfo(&(w->out));
  init_finfo(&(w->err));
  init_finfo(&(w->pattern));
  w->err.fp    = stderr;
  w->out.fp    = stdout;
  w->num_files        = 0;
  w->verbose          = 0;
  w->files_specified  = FALSE;  /* file names specified on command line    */
  w->force            = FALSE;  /* force create fies even if they exists   */
  w->ignore_case      = FALSE;  /* ignore character case on search         */
  w->show_file        = FALSE;  /* show input file name on results         */
  w->print_line_num   = FALSE;  /* show input line number on results       */
  w->invert_match     = FALSE;  /* print items that did NOT match          */
  w->print_results    = FALSE;  /* print results (see quiet -q)            */
  w->entries_found    = FALSE;  /* were items selected for print ?         */
  w->quiet            = FALSE;  /* TRUE means do NOT print results         */
  w->col_search       = -1L;
  w->delm             = JLIB2_CHAR_NULL;
  w->pause_reads      = 0L;
  w->micro            = 0L;
  w->search_reads     = 0L;
  w->search_val       = (char *) NULL;
  w->field_siz        = 0;             /* only for delimited file processing */
  w->field_val        = (char *) NULL; /* only for delimited file processing */

}  /* init_w() */

/*
 * print_c() - format the character or its hex value for printing
 */
char *print_c(char *str, int sizc, char c)
{
  unsigned char u = (unsigned char) c;

  switch (u)
    {
      case 7:
	snprintf(str, sizc, "%s", "BEL");
	return(str);
      case 9:
	snprintf(str, sizc, "%s", "TAB");
	return(str);
      case 32:
	snprintf(str, sizc, "%s", "space");
	return(str);
    }

  if ((u < 33) || (u > 126))
    {
      snprintf(str, sizc, "0x%02X", u);
      return(str);
    }

  snprintf(str, sizc, "%c", c);
  return(str);

} /* print_c() */

/*
 * get_delm() -- translate a number or string into a delimiter
 */
char get_delm(FILE *fp, char *s)
{
  int d;
  int c = JLIB2_CHAR_NULL;


  if (strlen(s) == 1)
    {
      if ( ((*s) > 47)  && ((*s) < 58) ) /* 0 -- 9 */
	c = (*s) - 48;
      else
	c = (*s);
    }
  else
    {
      if (strcmp(s, "BEL") == 0)
	return(0x07);
      if (strcmp(s, "TAB") == 0)
	return(0x09);
      if (strcmp(s, "SPACE") == 0)
	return(0x20);
      if (j2_is_numr(s) == (int) TRUE)
	{
	  d = atoi(s);
	  if ((d < 256) && (d > 0))
	    c = (char) d;
	  else
	    {
	      fprintf(fp, MSG_ERR_E006, s, SWITCH_CHAR, ARG_DELM);
	      fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(FAILED_PROCESS);
	    }
	}
      else
	{
	  fprintf(fp, MSG_ERR_E006, s, SWITCH_CHAR, ARG_DELM);
	  fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	}
    }

  return(c);

} /* get_delm() */

/*
 * verify_path() -- make sure dir/file name ls OK
 */
int verify_path(char *pname)
{
  char *s;

  if (pname == (char *) NULL)
    return(PATH_IS_NULL);
  if (strcmp(pname, FILE_NAME_STDOUT) == 0)
    return(PATH_IS_NULL);
  if (strlen(pname) < 1)
    return(PATH_IS_EMPTY);
  if (strlen(pname) > (MAXPATHLEN - 2))
    return(PATH_IS_BIG);

  for (s = pname; (*s) != JLIB2_CHAR_NULL; s++)
    {
      if (isspace((int) (*s)))
	return(PATH_HAS_SPACE);
    }

  return(PATH_IS_VALID);

} /* verify_path() */

/*
 * save_path_file() -- Check and Save a File Name
 */
void save_path_file(int std_ok, char *full_filename, char *arg_val, char arg_switch)
{
  int path_ok = 0;
  int is_std  = FALSE;

  path_ok = verify_path(arg_val);

  switch (path_ok)
    {
      case PATH_IS_VALID:
	break;
      case PATH_IS_NULL:
	if (std_ok == FALSE)
	  {
	    fprintf(stderr, MSG_ERR_E004S, LIT_NULL,  SWITCH_CHAR, arg_switch);
	    fprintf(stderr, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(FAILED_PROCESS);
	  }
	is_std  = TRUE;
	break;
      case PATH_IS_EMPTY:
	if (std_ok == FALSE)
	  {
	    fprintf(stderr, MSG_ERR_E004S, arg_val,   SWITCH_CHAR, arg_switch);
	    fprintf(stderr, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, arg_switch);
	    exit(FAILED_PROCESS);
	  }
	is_std  = TRUE;
	break;
      case PATH_IS_BIG:
	fprintf(stderr, MSG_ERR_E004SB, arg_val,   SWITCH_CHAR, arg_switch);
	fprintf(stderr, MSG_ERR_E000,   PROG_NAME, SWITCH_CHAR, arg_switch);
	exit(FAILED_PROCESS);
      case PATH_HAS_SPACE:
	fprintf(stderr, MSG_ERR_E004S, arg_val,   SWITCH_CHAR, arg_switch);
	fprintf(stderr, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(FAILED_PROCESS);
      default:
	fprintf(stderr, MSG_ERR_E004S, LIT_NULL,  SWITCH_CHAR, arg_switch);
	fprintf(stderr, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(FAILED_PROCESS);
    }

  if (is_std == TRUE)
    memset(full_filename, 0, MAXPATHLEN);
  else
    strncpy(full_filename, arg_val, MAXPATHLEN);


} /* save_path_file() */

/*
 * get_numb() -- convert an argument to a long int
 */
long int get_numb(FILE *efp, long int low, long int high,
                  char *arg, char arg_sw)
{
  long int i = 0L;

  if (j2_is_numr(arg) == TRUE)
    i = atol(arg);
  else
    {
      fprintf(efp, MSG_ERR_E008, arg, SWITCH_CHAR, arg_sw);
      fprintf(efp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(FAILED_PROCESS);
    }

  if ( (i < low) || ((i > high) && (high > 0L)) )
    {
      fprintf(efp, MSG_ERR_E004, i, SWITCH_CHAR, arg_sw);
      fprintf(efp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(FAILED_PROCESS);
    }

  return(i);

} /* get_numb() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SCKARG];
  int opt;
  int i;

  snprintf(ckarg, SCKARG, "%c%c%c%c%c%c%c%c%c%c:%c:%c:%c:%c:%c:%c:",
	  ARG_FORCE,  ARG_HELP,      ARG_VERBOSE,     ARG_VERSION,
          ARG_IGNORE, ARG_SHOW_FILE, ARG_PREFIX_LINE, ARG_INVERT,  ARG_QUIET,
	  ARG_ERR,    ARG_OUT,       ARG_COL,         ARG_PAUSE,   ARG_READS,
          ARG_DELM,   ARG_PATERN_FILE);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	case ARG_COL:
	  if (w->col_search >= 0L)
	    {
	      fprintf(w->err.fp, MSG_ERR_E074, SWITCH_CHAR, opt);
	      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(FAILED_PROCESS);
	    }
	  w->col_search = get_numb(stderr, 1L, 0L, optarg, ARG_COL);
	  break;
	case ARG_PAUSE:
	  if (w->micro > 0L)
	    {
	      fprintf(w->err.fp, MSG_ERR_E074, SWITCH_CHAR, opt);
	      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(FAILED_PROCESS);
	    }
	  w->micro = get_numb(stderr, 1L, 600000000L, optarg, ARG_PAUSE);
	  break;
	case ARG_READS:
	  if (w->pause_reads > 0L)
	    {
	      fprintf(w->err.fp, MSG_ERR_E074, SWITCH_CHAR, opt);
	      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(FAILED_PROCESS);
	    }
	  w->pause_reads = get_numb(stderr, 1L, 0L, optarg, ARG_PAUSE);
	  break;
	case ARG_DELM:
	  w->delm = get_delm(stderr, optarg);
	  break;
	case ARG_FORCE:
	  w->force = TRUE;
	  break;
	case ARG_HELP:
	  show_brief_help(stderr, PROG_NAME);
	  break;
	case ARG_INVERT:
	  w->invert_match = TRUE;
	  break;
        case ARG_IGNORE:
	  w->ignore_case = TRUE;
	  break;
        case ARG_QUIET:
	  w->quiet = TRUE;
	  break;
        case ARG_SHOW_FILE:
	  w->show_file = TRUE;
	  break;
        case ARG_PREFIX_LINE:
	  w->print_line_num = TRUE;
	  break;
	case ARG_VERBOSE:
	  w->verbose++;
	  break;
	case ARG_VERSION:
	  show_rev(stderr, PROG_NAME);
	  break;
	case ARG_ERR:
	  save_path_file(TRUE,  w->err.fname, optarg, ARG_ERR);
	  break;
	case ARG_OUT:
	  save_path_file(TRUE,  w->out.fname, optarg, ARG_OUT);
	  break;
	case ARG_PATERN_FILE:
	  save_path_file(FALSE, w->pattern.fname, optarg, ARG_PATERN_FILE);
	  break;
	default:
	  fprintf(w->err.fp, MSG_ERR_E018B);
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	  break;
	}
    }

  /*** adjust some args ***/
  if (w->col_search < 0L)
    w->col_search = 0L;
  else
    {
      if (w->col_search < 1L)
	{
	  fprintf(w->err.fp, MSG_ERR_E004, (long int) w->col_search, SWITCH_CHAR, ARG_COL);
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	}
    }

  if ((w->pause_reads < 1) && (w->micro > 0))
    {
      fprintf(stderr, MSG_ERR_E097, SWITCH_CHAR, ARG_READS, SWITCH_CHAR, ARG_PAUSE);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(FAILED_PROCESS);
    }
  if ((w->pause_reads > 0) && (w->micro < 1))
    {
      fprintf(stderr, MSG_ERR_E097, SWITCH_CHAR, ARG_PAUSE, SWITCH_CHAR, ARG_READS);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(FAILED_PROCESS);
    }

  /*** open 'out' files ***/
  if ( ! open_out(stderr, &(w->err), PROG_NAME, w->force))
    w->err.fp = stderr;
  if ( ! open_out(w->err.fp, &(w->out), PROG_NAME, w->force) )
    w->out.fp = stdout;

  /*** set search pattern if no pattern file specified ***/
  if (strlen(w->pattern.fname) == 0)
    {
      if (argv[optind] != (char *) NULL)
	w->search_val = strdup(argv[optind]);
      if (w->search_val == (char *) NULL)
	{
	  fprintf(w->err.fp, MSG_ERR_E109);
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	}
    }

  if ((w->delm != JLIB2_CHAR_NULL) && (w->col_search < 1L))
    {
      fprintf(w->err.fp, MSG_ERR_E097, SWITCH_CHAR, ARG_COL, SWITCH_CHAR, ARG_DELM);
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(FAILED_PROCESS);
    }

  /* count files to processes */
  for (i = optind; i < argc; i++)
    {
      if ((w->search_val != (char *) NULL) && (i == optind))
	continue;
      (w->num_files)++;
      w->files_specified = TRUE;
      if (w->verbose > 0)
	fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
    }
  if (w->num_files == 0)
    {
      if (w->verbose > 0)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
      (w->num_files)++;
    }

} /* END process_arg() */

/*
 * init() -- initialize
 */
struct s_search *init(int argc, char **argv, struct s_work *w)

{
  struct s_search *search_list = (struct s_search *) NULL;
  char str[(SIZEC + 1)];

  memset(str, 0, (SIZEC + 1));
  init_w(w);

  process_arg(argc, argv, w);

  /*** show parameters ***/
  if (w->verbose > 0)
    {
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) < 1 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->out.fname) < 1 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      if (strlen(w->pattern.fname) > 0)
	fprintf(w->err.fp, MSG_INFO_I154,  w->pattern.fname);
      if (w->col_search > 0)
	fprintf(w->err.fp, MSG_INFO_I135S, w->col_search);
      if (w->delm != JLIB2_CHAR_NULL)
	fprintf(w->err.fp, MSG_INFO_I140S, print_c(str, SIZEC, w->delm));
      fprintf(w->err.fp, MSG_INFO_I087,  (w->force       == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I158,  (w->ignore_case == TRUE ? LIT_YES : LIT_NO));
      if (w->invert_match == FALSE)
	fprintf(w->err.fp, MSG_INFO_I159);
      else
	fprintf(w->err.fp, MSG_INFO_I160);
      if (w->quiet == TRUE)
	fprintf(w->err.fp, MSG_INFO_I161);
      if (w->search_val != (char *) NULL)
	fprintf(w->err.fp, MSG_INFO_I156, w->search_val);
      if (w->pause_reads > 0L)
	fprintf(w->err.fp, MSG_INFO_I157, w->micro, w->pause_reads);
    }

  /*** save search data ***/
  if (strlen(w->pattern.fname) > 0)
    search_list = search_load(w, &(w->search_reads));
  else
    search_list = search_save(w->search_val, 0);

  if (w->delm != JLIB2_CHAR_NULL)
    {
      w->field_siz = REC_SIZE_INIT;
      w->field_val = calloc(w->field_siz, sizeof(char));
      if (w->field_val == (char *) NULL)
	{
	  fprintf(w->err.fp, MSG_ERR_E080, strerror(errno));
	  fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(FAILED_PROCESS);
	}
    }

  return(search_list);

}  /* end: init() */
