/*
 * Copyright (c) 2003 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef JGREP_H

#define JGREP_H       "3.5 2025/09/13"
#define PROG_NAME     "jgrep"
#define REC_SIZE_INIT 256
#define MATCH_SUCCESS  EXIT_SUCCESS
#define MATCH_FAILED   1
#define FAILED_PROCESS 2

#ifndef JLIB2_CHAR_NULL
#define NO_JLIB 1
#define JLIB2_INT_NULL   ((int) '\0')
#define JLIB2_CHAR_NULL  ((char) '\0')
#define JLIB2_UCHAR_NULL ((unsigned char) '\0' )
#endif

#ifndef NULL
#define NULL '\0'
#endif
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#ifndef SSIZE_T
#define SSIZE_T ssize_t
#endif
#ifndef TAB_CHAR
#define TAB_CHAR 0x09
#endif

#ifdef MAXPATHLEN
#ifndef PATH_MAX
#define PATH_MAX MAXPATHLEN
#endif
#else
#ifdef PATH_MAX
#define MAXPATHLEN PATH_MAX
#endif
#endif

#define PATH_IS_VALID   0
#define PATH_IS_NULL    1
#define PATH_IS_EMPTY   2
#define PATH_IS_BIG     3
#define PATH_HAS_SPACE  4

/*** structures ***/
struct s_file_info
{
  FILE *fp;
  char fname[(MAXPATHLEN + 1)];
} ;

struct s_counts
{
  long int reads;
  long int writes;
} ;
struct s_search
{
  char *str;                      /* string to search for       */
  long int found;                 /* count of records found     */
  long int missing;               /* count of records not found */
  long int rec_num;               /* Record of the entry        */
  int sort_exceeded;              /* TRUE or FALSE              */
  struct s_search *next;
} ;

struct s_work
{
  struct s_file_info out;         /* default stdout                 */
  struct s_file_info err;         /* default stderr                 */
  struct s_file_info pattern;     /* search pattern file            */
  int num_files;                  /* # of files to process          */
  char delm;                      /* field delimiter                */
  int files_specified;            /* TRUE or FALSE                  */
  int verbose;                    /* TRUE or FALSE                  */
  int invert_match;               /* TRUE or FALSE                  */
  int sorted;                     /* TRUE or FALSE                  */
  int force;                      /* TRUE or FALSE                  */
  int ignore_case;                /* TRUE or FALSE                  */
  int show_file;                  /* TRUE or FALSE                  */
  int print_results;              /* TRUE or FALSE                  */
  int print_line_num;             /* TRUE or FALSE                  */
  int entries_found;              /* TRUE or FALSE                  */
  int quiet;                      /* TRUE or FALSE                  */
  long int col_search;            /* Column to start search at      */
  long int micro;                 /* pause micro seconds            */
  long int pause_reads;           /* pause after this many reads    */
  long int search_reads;          /* Entries read from pattern file */
  char *search_val;               /* search value                   */
  char *field_val;                /* field value being searched     */
  size_t field_siz;               /* allocated size of field_val    */
} ;

/*** If no j_lib ***/
#ifdef NO_JLIB
struct s_j2_datetime
{
  int month;    /* Month,           01 --12             */
  int dd;       /* Day of Month,    01 -- [28|29|30|31] */
  int yy;       /* two digit year,  00 -- 99            */
  int yyyy;     /* four digit year, 0000 -- 9999        */
  int hh;       /* Hour of the day, 00 -- 23            */
  int minutes;  /* minutes,         00 -- 59            */
  int ss;       /* seconds,         00 -- 59            */
  int mil;      /* milliseconds,    000 -- 999          */
  int tm_isdst; /* Daylight ?  0 = no, > 0 = yes        */
} ;

#define SWITCH_CHAR       '-'
#define ARG_COL           'C'  /* Column to start with               */
#define ARG_DELM          'd'  /* Field Delimiter                    */
#define ARG_ERR           'e'  /* Output Error File                  */
#define ARG_FORCE         'f'  /* force create files                 */
#define ARG_HELP          'h'  /* Show Help                          */
#define ARG_IGNORE        'i'  /* Ignore character case              */
#define ARG_OUT           'o'  /* Output File                        */
#define ARG_PATERN_FILE   'p'  /* Pattern File Name                  */
#define ARG_PAUSE         's'  /* Pause for for microseconds         */
#define ARG_PREFIX_LINE   'n'  /* Prefix output with line number     */
#define ARG_READS         'r'  /* after this many reads, do something*/
#define ARG_SHOW_FILE     'H'  /* Print file name                    */
#define ARG_QUIET         'q'  /* Quiet Mode                         */
#define ARG_INVERT        'R'  /* Invert (Reverse) Match             */
#define ARG_SORTED        'S'  /* Data is sorted                     */
#define ARG_VERBOSE       'v'  /* Verbose                            */
#define ARG_VERSION       'V'  /* Show Version Information           */
#define FILE_NAME_STDIN   "-"
#define FILE_NAME_STDOUT  "-"
#define LIT_C80           "::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::"
#define LIT_INFO_04       "Build: %s %s\n"
#define LIT_NO            "no"
#define LIT_NULL          "<null>"
#define LIT_REV           "Revision"
#define LIT_STDERR        "(standard error)"
#define LIT_STDIN         "(standard input)"
#define LIT_STDOUT        "(standard output)"
#define LIT_TOTAL         "Total"
#define LIT_DLINE_11      "==========="
#define LIT_DLINE_19      "==================="
#define LIT_ULINE_11      "-----------"
#define LIT_ULINE_42      "-----------------------------------------"
#define LIT_YES           "Yes"
#define MSG_ERR_E000      "Try '%s %c%c' for more information\n"
#define MSG_ERR_E002      "ERROR E002: Cannot open '%s' for write, processing aborted\n"
#define MSG_ERR_E003      "ERROR E003: Cannot allocate memory when processing file %s, processing aborted\n"
#define MSG_ERR_E004      "ERROR E004: value %ld invalid value for %c%c\n"
#define MSG_ERR_E004S     "ERROR E004: value '%s' invalid value for %c%c\n"
#define MSG_ERR_E004SB    "ERROR E004: value '%s' value for %c%c too large\n"
#define MSG_ERR_E006      "ERROR E006: '%s' is an invalid value for %c%c\n"
#define MSG_ERR_E008      "ERROR E008: '%s' is an invalid value for %c%c, must be numeric\n"
#define MSG_ERR_E018B     "ERROR E018: invalid argument supplied\n"
#define MSG_ERR_E018C     "ERROR E018: %c%c invalid argument\n"
#define MSG_ERR_E025      "ERROR E025: File %s cannot be created, already exists\n"
#define MSG_ERR_E074      "ERROR E074: 'Too many Arguments specified for %c%c\n"
#define MSG_ERR_E080      "ERROR E080: cannot allocate initial memory : %s\n"
#define MSG_ERR_E081      "ERROR E081: Required Argument '%c%c' is NOT numeric (%s)\n"
#define MSG_ERR_E097      "ERROR E097: Argument '%c%c' is Required when using Option '%c%c'\n"
#define MSG_ERR_E100      "ERROR E100: pledge() %d: %s\n"
#define MSG_ERR_E109      "ERROR E109: Search Term or Search File not Supplied\n"
#define MSG_ERR_E110      "ERROR E110: all data on file %s invalid\n"
#define MSG_INFO_I043L    "I043: Input  File                  : %s\n"
#define MSG_INFO_I072     "I072:    Lines Reads:  %9ld - File %s\n"
#define MSG_INFO_I080     "I080:   Lines Writes:  %9ld - File %s\n"
#define MSG_INFO_I081     "I081: Number of Input Files        : %d\n"
#define MSG_INFO_I086     "I086: Errors written to file       : %s\n"
#define MSG_INFO_I087     "I087: Force Create Output          ? %s\n"
#define MSG_INFO_I090     "I090: Output written to file       : %s\n"
#define MSG_INFO_I092     "I092: Verbose Level                : %d\n"
#define MSG_INFO_I096     "I096:     Read Bytes:  %9ld - File %s\n"
#define MSG_INFO_I096U    "I096:     Read Bytes:  %9lu - File %s\n"
#define MSG_INFO_I097     "I097:    Write Bytes:  %9ld - File %s\n"
#define MSG_INFO_I097U    "I097:    Write Bytes:  %9lu - File %s\n"
#define MSG_INFO_I106T    "I106: %s: Paused %ld microseconds after processing %ld Input Records\n"
#define MSG_INFO_I127     "I127: Parameters from default or Args: Pause Recs %ld - milliseconds %ld\n"
#define MSG_INFO_I135S    "I135: Search Column                : %ld\n"
#define MSG_INFO_I140S    "I140: Field Delimiter              : %s\n"
#define MSG_INFO_I152L    "I152:       Run Time:  %lld second(s)\n"
#define MSG_INFO_I154     "I154: Read Search Terms from file  : %s\n"
#define MSG_INFO_I155     "I155: Input File Sorted            ? %s\n"
#define MSG_INFO_I156     "I156: Search Value                 : %s\n"
#define MSG_INFO_I157     "I157: Pause %ld milliseconds after Reading %ld Records\n"
#define MSG_INFO_I158     "I158: Ignore Character Case        ? %s\n"
#define MSG_INFO_I159     "I159: Print Matching Lines\n"
#define MSG_INFO_I160     "I160: Print Lines that do NOT Match\n"
#define MSG_INFO_I161     "I161: Quiet, do not print results\n"
#define MSG_WARN_W002     "W002: Open Error Bypass File '%s' : %s\n"
#define MSG_WARN_W037     "W037: Bypassed Record %ld, no data from File %s\n"
#define USG_MSG_ARG_COL         "\t%c%c col\t\t: start line processing at column 'col'\n"
#define USG_MSG_ARG_DELM        "\t%c%c c\t\t: Field delimiter, the actual character OR\n"
#define USG_MSG_ARG_DELM_A      "\t\t\t  the decimal # representing the character.\n"
#define USG_MSG_ARG_DELM_B      "\t\t\t  Ex: 7 = '^G', 9 = TAB, 124 = '|' or just '|'.\n"
#define USG_MSG_ARG_ERR         "\t%c%c file\t\t: Write errors to file 'file', default stderr\n"
#define USG_MSG_ARG_FORCE       "\t%c%c\t\t: force create of files when found\n"
#define USG_MSG_ARG_HELP        "\t%c%c\t\t: Show brief help and exit\n"
#define USG_MSG_ARG_IGNORE      "\t%c%c\t\t: Ignore character case.\n"
#define USG_MSG_ARG_OUT         "\t%c%c file\t\t: Write output to file 'file', default stdout\n"
#define USG_MSG_ARG_PATERN_FILE "\t%c%c file\t\t: Obtain search text from 'file'\n"
#define USG_MSG_ARG_PAUSE_1     "\t%c%c n\t\t: Sleep 'n' microseconds based upon records processed\n"
#define USG_MSG_ARG_PREFIX_LINE "\t%c%c\t\t: prefix output with line number\n"
#define USG_MSG_ARG_READS       "\t%c%c n\t\t: Optional, after 'n' reads, pause\n"
#define USG_MSG_ARG_SHOW_FILE   "\t%c%c\t\t: Print File Name of each match\n"
#define USG_MSG_ARG_INVERT      "\t%c%c\t\t: Reverse Match, show lines with no match\n"
#define USG_MSG_ARG_QUIET       "\t%c%c\t\t: quiet, do not print results.\n"
#define USG_MSG_ARG_SORTED      "\t%c%c\t\t: Input is Sorted\n"
#define USG_MSG_ARG_VERBOSE_8   "\t%c%c\t\t: verbose level, each time specified level increases\n"
#define USG_MSG_ARG_VERSION     "\t%c%c\t\t: Show revision information and exit\n"
#define USG_MSG_OPTIONS         "Options\n"
#define USG_MSG_USAGE_8         "usage:\t%s [SEARCH-TEXT] [OPTIONS] FILE1 [FILE_2 ...]\n"

int j2_f_exist(char *file_name);
long int j2_rtw(char *buffer);
char *j2_get_prgname(char *argv_0, char *default_name);
SSIZE_T j2_getline(char **buf, size_t *n, FILE *fp);
long int j2_clr_str(char *s, char c, int size);
char *j2_d_fmtdt(char *datetime, size_t len);
int j2_is_numr(char *s);
void j2_sleepm(long int micro);
void j2_today(struct s_j2_datetime *dt);
long int j2_bye_last(char *x, char lastc);

#define j2_bye_nl(x)    j2_bye_last((x), '\n')
#define j2_bye_ctlm(x)  j2_bye_last((x), '\r')
#endif /* NO_JLIB */

/*** prototypes ***/
struct s_search *init(int, char **, struct s_work *);
void init_finfo(struct s_file_info *);
void show_brief_help(FILE *, char *);
void show_rev(FILE *, char *);
void process_arg(int, char **, struct s_work *);
int  open_out(FILE *, struct s_file_info *, char *, int);
void close_out(struct s_file_info *);
int  open_in(FILE **, char *, FILE *);
void close_in(FILE **, char *);
void init_counts(struct s_counts *c);
int verify_path(char *pname);
void search(struct s_work *w, struct s_search *needle, char *buf, size_t bsize, int *match_made);
struct s_search *search_load(struct s_work *w, long int *rcount);
struct s_search *search_save(char *buf, long int rcount);
void search_write(FILE *fp, struct s_search *search_list);
void search_free(struct s_search **search_list);

#endif /*  JGREP_H  */

/* END: jgrep.h */
