/*
 * Copyright (c) 2003 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jgrep.c -- Main routines
 */

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>

#ifdef OpenBSD
#include <err.h>
#endif
#ifdef __FreeBSD_version
#include <err.h>
#endif
#ifdef __NetBSD_Version__
#include <err.h>
#endif

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jgrep.h"

/*
 * add_counts() -- add counts
 */
void add_counts(struct s_counts *t, struct s_counts *c)

{
  t->reads  += c->reads;
  t->writes += c->writes;
} /* add_counts() */

/*
 * print_buf()
 */
void print_buf(struct s_work *w, struct s_counts *per,
	       char *buf,        char *fname_in, int match_made)
{
  char *fmt_in = (char *) NULL;

  w->entries_found = TRUE;  /* used to set return code */

  if (w->quiet == TRUE)
    return;
  if ((w->invert_match == TRUE) && (match_made == TRUE))
    return;

  per->writes++;

  if (w->show_file == TRUE)
    {
      if (strcmp(fname_in, FILE_NAME_STDIN) != 0)
	fmt_in = fname_in;
      fprintf(w->out.fp, "%s:",
	      (fmt_in == (char *) NULL ? LIT_STDIN  : fname_in));
    }

  if (w->print_line_num == TRUE)
    fprintf(w->out.fp, "%ld:", per->reads);
  fprintf(w->out.fp, "%s\n", buf);

} /* print_buf() */

/*
 * process_a_file()
 */
void process_a_file(struct s_work *w, struct s_counts *per,
                    struct s_search *search_list,
                    char *fname, char **buf, size_t *bsize)

{

  char *fmt_in  = (char *) NULL;
  FILE *fp;
  char fmt_dt[15];
  struct s_search *now = search_list;
  int match_made = FALSE;

  init_counts(per);

  if ( ! open_in(&fp, fname, w->err.fp) )
    return;

  /*** process data ***/
  while (j2_getline(buf, bsize, fp) > (ssize_t) -1)
    {
      match_made = FALSE;
      per->reads++;
      now = search_list;
      w->print_results = FALSE;
      j2_bye_nl(*buf);
      j2_bye_ctlm(*buf);
      if ((w->pause_reads > 0) && (w->verbose > 3))
	{
	  if ((per->reads % w->pause_reads) == 0)
	    {
	      fprintf(w->err.fp, MSG_INFO_I106T, j2_d_fmtdt(fmt_dt, 13), w->micro, per->reads);
	      fflush(w->err.fp);
	      j2_sleepm(w->micro);
	    }
	}
      while (now != (struct s_search *) NULL)
	{
	  search(w, now, *buf, *bsize, &match_made);
	  now = now->next;
	}
      if (w->print_results == TRUE)
	print_buf(w, per, *buf, fname, match_made);
    }

  /*** complete ***/
  close_in(&fp, fname);

  if (w->verbose > 1)
    {
      if (fname != (char *) NULL)
	{
	  if (strcmp(fname, FILE_NAME_STDIN) != 0)
	    fmt_in = fname;
	}
      fprintf(w->err.fp, MSG_INFO_I072, (long int) per->reads,
              (fmt_in == (char *) NULL ? LIT_STDIN  : fname));
      fprintf(w->err.fp, MSG_INFO_I080, (long int) per->writes,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
    }

} /* process_a_file() */

/*
 * process_all() -- Process all files
 */
void process_all(int argc, char **argv, struct s_work *w,
                 struct s_search *search_list)

{
  int i;
  char *buf = (char *) NULL;
  size_t bsiz = (size_t) REC_SIZE_INIT;
  struct s_counts totals;
  struct s_counts per_file;

  init_counts(&totals);

  /* allocate initial read buffer memory (optional) */
  buf = (char *) calloc(bsiz, sizeof(char));
  if (buf == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_ERR_E080, strerror(errno));
      return;
    }

  /* process files */
  for (i = optind; i < argc; i++)
    {
      if ((w->search_val != (char *) NULL) && (i == optind))
	continue;
      process_a_file(w, &per_file, search_list, argv[i], &buf, &bsiz);
      add_counts(&totals, &per_file);
    }

  if (w->files_specified == FALSE)
    {
      process_a_file(w, &per_file, search_list, FILE_NAME_STDIN, &buf, &bsiz);
      add_counts(&totals, &per_file);
    }

  /* show totals, no need if only 1 file processed */
  if ((w->num_files > 1) && (w->verbose > 1))
    {
      fprintf(w->err.fp, MSG_INFO_I072, (long int) totals.reads,  LIT_TOTAL);
      fprintf(w->err.fp, MSG_INFO_I080, (long int) totals.writes, LIT_TOTAL);
    }

#ifdef OpenBSD
  freezero(buf, bsiz);
#else
  if (buf != (char *) NULL)
    free(buf);
#endif

}  /* process_all() */

/*
 * main()
 */
int main(int argc, char **argv)

{
  time_t tstart = time(&tstart), tend;
  struct s_work w;
  struct s_search *search_list = (struct s_search *) NULL;

#ifdef OpenBSD
  if(pledge("stdio rpath wpath cpath",NULL) == -1)
    err(1, MSG_ERR_E100, 1, strerror(errno));
#endif

  search_list = init(argc, argv, &w);

  process_all(argc, argv, &w, search_list);

  if (w.verbose > 1)
    {
      if (w.verbose > 2)
	search_write(w.err.fp, search_list);
      fprintf(w.err.fp, MSG_INFO_I152L, (long long int) (time(&tend) - tstart));
    }

  if (w.search_val != (char *) NULL)
    free(w.search_val);
  if (w.field_val != (char *) NULL)
    free(w.field_val);
  search_free(&search_list);
  close_out(&(w.out));
  close_out(&(w.err));

  if (w.entries_found == TRUE)
    exit(MATCH_SUCCESS);
  exit(MATCH_FAILED);

}  /* main() */
