/*
 * Copyright (c) 2013 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _MSDOS
#include <sys/param.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jfile.h"

#define SIZ_ARG_FMT 80

/*
 * init_get_delm() -- translate a string into a delimiter
 */
int init_get_delm(FILE *fp, char *s)
{
  int d;
  int c = JLIB2_CHAR_NULL;

  if (s == (char *) NULL)
    return(c);

  if (strlen(s) == 1)
    {
      if ( ((*s) > 47)  && ((*s) < 58) ) /* 0 -- 9 */
	c = (*s) - 48;
      else
	c = (*s);
    }
  else
    {
      if (j2_is_numr(s) == (int) TRUE)
	{
	  d = atoi(s);
	  if ((d > -1) && (d < 128))
	    c = (char) d;
	  else
	    {
	      fprintf(fp, MSG_ERR_E049, s);
	      fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	}
      else
	{
	  fprintf(fp, MSG_ERR_E049, s);
	  fprintf(fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  return(c);

} /* init_get_delm() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SIZ_ARG_FMT];
  int opt       = 0;
  int i         = 0;
  int count_std = 0;
  char *fout    = (char *) NULL; 
  char *ferr    = (char *) NULL; 

  snprintf(ckarg, SIZ_ARG_FMT, "%c%c%c%c%c%c%c%c%c%c%c%c:%c:%c:", 
	         ARG_NL_ASCII, ARG_BOM_NO, ARG_PRINT_UTF8,
		 ARG_7BIT_FILE, ARG_8BIT_FILE, ARG_INVALID, ARG_UTF8_FILE,
		 ARG_FORCE, ARG_HELP, ARG_VERBOSE, ARG_VERSION,
		 ARG_ERR, ARG_OUT, ARG_DELM);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	  case ARG_FORCE:
	    w->force = (int) TRUE;
	    break;
	  case ARG_DELM:
	    w->delim    = (unsigned char) init_get_delm(w->err.fp, optarg);
	    w->is_delim = (int) TRUE;
	    break;
	  case ARG_UTF8_FILE:
	    w->show_utf_8 = (int) TRUE;
	    break;
	  case ARG_PRINT_UTF8:
	    w->show_real_utf8 = (int) TRUE;
	    break;
	  case ARG_BOM_NO:
	    w->flag_bom     = (int) TRUE;
	    break;
	  case ARG_7BIT_FILE:
	    w->show_7bit    = (int) TRUE;
	    break;
	  case ARG_8BIT_FILE:
	    w->show_8bit    = (int) TRUE;
	    break;
	  case ARG_INVALID:
	    w->show_invalid = (int) TRUE;
	    break;
	  case ARG_NL_ASCII:
	    w->nl_not_ascii = (int) FALSE;
	    break;
	  case ARG_HELP:
	    show_brief_help();
	    break;
	  case ARG_VERBOSE:
	    w->verbose++;
	    w->show_8bit    = TRUE;
	    w->show_invalid = TRUE;
	    w->show_utf_8   = TRUE;
	    break;
	  case ARG_VERSION:
	    show_rev();
	    break;
	  case ARG_ERR:
	    ferr = optarg;
	    break;
	  case ARG_OUT:
	    fout = optarg;
	    break;
	  default:
	    fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(EXIT_FAILURE);
	    break;
	}
    }

  /*** if necessary - save stdout/err files and open ***/
  open_out(&(w->err), ferr, w->force);
  open_out(&(w->out), fout, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 0)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 0)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* process_arg() */
