/*
 * Copyright (c) 2005 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _WIN32
#ifndef _MSDOS
#include <sys/param.h>
#endif
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>

#ifdef UNIX
#include <unistd.h>
#endif
#ifdef _MSDOS
#include <io.h>
#endif

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jexpand.h"

/*
 * verify_path() -- make sure dir/file name is OK
 */
int verify_path(char *fname)
{
  char *s;

  if (fname == (char *) NULL)
    return(PATH_IS_NULL);
  if (strlen(fname) < 1)
    return(PATH_IS_EMPTY);
  if (strlen(fname) > (PATH_MAX - 2))
    return(PATH_IS_BIG);
  if ((strlen(fname) > 1) && (*fname == '-'))
    return(PATH_INVALID);

  for (s = fname; (*s) != JLIB2_CHAR_NULL; s++)
    {
      if (isspace((int) (*s)))
	return(PATH_HAS_SPACE);
    }

  return(PATH_IS_VALID);

} /* verify_path() */

/*
 * check_path_file() -- Check and Save a Dir Name
 */
void check_path_file(FILE *efp, char *fname, char arg_switch)
{

  char sw = SWITCH_CHAR;
  char ag = arg_switch;

  if (arg_switch == JLIB2_CHAR_NULL)
    {
      sw = 'i';
      ag = 'n';
    }

  switch (verify_path(fname))
    {
      case PATH_IS_VALID:
	break;
      case PATH_IS_NULL:
	fprintf(efp, MSG_ERR_E004S, LIT_NULL,  sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_EMPTY:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_BIG:
	fprintf(efp, MSG_ERR_E004SB, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,   PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_HAS_SPACE:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_INVALID:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      default:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
    }

} /* check_path_file() */

/*
 * init_finfo() -- initialize out file structure
 */
void init_finfo(struct s_file_info *f)

{

  f->reads        = (unsigned long) 0;
  f->writes       = (unsigned long) 0;
  f->bytes        = (unsigned long) 0;
  f->ok_to_close  = FALSE;

  memset(f->fname, 0, (PATH_MAX + 1));

} /* init_finfo() */

/*
 * close_file() -- close a file
 */
void close_file(struct s_file_info *f)
{
  if (f->ok_to_close == TRUE)
    fclose(f->fp);

  init_finfo(f);

} /* close_file() */

/*
 * close_in() -- Open an input file
 */
void close_in(struct s_file_info *f)

{
  if (f->fname == (char *) NULL)
    return;
  if (strcmp(f->fname, FILE_NAME_STD) == 0)
    return;

  if (f->fp != (FILE *) NULL)
    {
      fclose(f->fp);
      f->fp = (FILE *) NULL;
    }

} /* close_in() */

/*
 * open_out() -- save the file name and check status
 */
void open_out(struct s_file_info *f, char *fname, int force)

{

  if (fname == (char *) NULL)
    return;
  if (strlen(fname) < 1)
    return;
  if (strcmp(fname, FILE_NAME_STD) == 0)
    return;
  if (f->ok_to_close == TRUE) /* file in use */
    {
      fprintf(stderr, MSG_ERR_E112, fname);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  if (force == FALSE)
    {
      if ( j2_f_exist(fname) )
	{
	  fprintf(stderr, MSG_ERR_E025, fname);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }


  f->fp = fopen(fname, "w");

  if (f->fp == (FILE *) NULL)
    {
      fprintf(stderr, MSG_ERR_E002, fname);
      fprintf(stderr, "\t%s\n", strerror(errno));
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  /*** success, save file name ***/
  strncpy(f->fname, fname, PATH_MAX);
  f->ok_to_close = TRUE;

} /* open_out() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[80];
  int opt       = 0;
  int i         = 0;
  int count_std = 0;
  char *fout    = (char *) NULL;
  char *ferr    = (char *) NULL;

  snprintf(ckarg, 79, "%c%c%c%c%c:%c:%c:",
                  ARG_FORCE, ARG_HELP, ARG_VERBOSE, ARG_VERSION,
                  ARG_TAB_SPACE, ARG_ERR, ARG_OUT);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	  case ARG_TAB_SPACE:
	    w->tab_size = atoi(optarg);
	    if (w->tab_size < 1)
	      {
		fprintf(w->err.fp, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_TAB_SPACE);
		fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_FORCE:
	    w->force = (int) TRUE;
	    break;
	  case ARG_HELP:
	    show_brief_help();
	    break;
	  case ARG_VERBOSE:
	    w->verbose++;
	    break;
	  case ARG_VERSION:
	    show_rev();
	    break;
	  case ARG_ERR:
	    if (ferr == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_ERR);
		ferr = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	    }

	    break;
	  case ARG_OUT:
	    if (fout == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_OUT);
		fout = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	      break;
	  default:
	    fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(EXIT_FAILURE);
	    break;
	}
    }

  /*** save stdout/err files and open ***/
  open_out(&(w->err), ferr, w->force);
  open_out(&(w->out), fout, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* process_arg() */

/*
 * clear_work()
 */
void clear_work(struct s_work *w)

{

  w->force       = FALSE;
  w->verbose     = 0;
  w->num_files   = 0;
  w->tab_size    = EXPAND_TAB_DEFAULT;

  init_finfo(&(w->out));
  init_finfo(&(w->err));
  w->err.fp = stderr;
  w->out.fp = stdout;

} /* clear_work() */

/*
 * init() -- setup for Run
 */
void init(struct s_work *w, char **argv, int argc)

{

  clear_work(w);

  process_arg(argc, argv, w);

  if (w->verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      fprintf(w->err.fp, MSG_INFO_I087, (w->force == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I167, w->tab_size);
    }

} /* process_arg() */
