/*
 * Copyright (c) 2005 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _WIN32
#ifndef _MSDOS
#include <sys/param.h>
#endif
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#ifndef INIX
#include <stdlib.h>
#ifndef _MSDOS
#include <unistd.h>
#endif
#endif

#ifdef _MSDOS
#include <getopt.h>
#include <io.h>
#endif

#ifdef OpenBSD
#include <err.h>
#endif

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jexpand.h"

/*
 * open_in() -- open in file
 */
int open_in(struct s_file_info *ferr, struct s_file_info *ifile, char *fname)

{
  if (ifile->ok_to_close == TRUE) /* file in use */
    {
      fprintf(ferr->fp, MSG_ERR_E112, fname);
      fprintf(ferr->fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      close_file(ferr);
      exit(EXIT_FAILURE);
    }

  if (fname == (char *) NULL)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strcmp(fname, FILE_NAME_STD) == 0)
    {
      ifile->fp = stdin;
      return((int) TRUE);
    }

  ifile->fp = fopen(fname, "r");

  if (ifile->fp == (FILE *) NULL)
    {
      fprintf(ferr->fp, MSG_WARN_W002, fname, strerror(errno));
      return((int) FALSE);
    }

  /*** success, save file name ***/
  strncpy(ifile->fname, fname, PATH_MAX);
  ifile->ok_to_close = TRUE;
  return((int) TRUE);

} /* open_in() */

/*
 * show_file_heading() -- Show run stats
 */
void show_file_heading(struct s_work *w,  char *fname)

{

  w->out.writes += 3;
  w->out.bytes += fprintf(w->out.fp, "%s\n", LIT_C80);

  if (fname == (char *) NULL)
    w->out.bytes += fprintf(w->out.fp, "%s\n", LIT_STDIN);
  else
    {
      if (strcmp(fname, FILE_NAME_STD) == 0)
	w->out.bytes += fprintf(w->out.fp, "%s\n", LIT_STDIN);
      else
	w->out.bytes += fprintf(w->out.fp, "%s\n", fname);
    }

  w->out.bytes += fprintf(w->out.fp, "%s\n", LIT_C80);

} /* show_file_heading() */

/*
 * show_stats() -- Show run stats
 */
void show_stats(FILE *fp, struct s_file_info *f, char *file_lit)
{

  static int show_heading = TRUE;

  if (show_heading == TRUE)
    {
      fprintf(fp, "%-10s %-13s %10s %s\n",
              LIT_LINES_READ, LIT_LINES_WRITES, LIT_BYTES, LIT_FILE);
      fprintf(fp, "%s %s %s %s\n",
              LIT_ULINE_10, LIT_ULINE_13, LIT_ULINE_10, LIT_ULINE_44);
      show_heading = FALSE;
    }

  fprintf(fp, "%10llu %13llu %10llu %s\n",
          (long long unsigned) f->reads, (long long unsigned) f->writes, 
          (long long unsigned) f->bytes,
	  (strlen(f->fname) < 1 ? file_lit : f->fname));

} /* show_stats() */

/*
 * mem_error() -- Ran out of memory, cleanup and abort
 */
void mem_error(struct s_work *w, struct s_file_info *ifile)
{
    fprintf(w->err.fp, MSG_ERR_E116, (long long int) ifile->reads,
            ifile->fname);

    show_stats(w->err.fp, ifile, LIT_STDIN);

    close_file((&w->out));
    close_file((&w->err));

    exit(EXIT_FAILURE);

} /* mem_error() */

/*
 * process_a_file()
 */
void process_a_file(struct s_work *w, char *fname, char **buf, size_t *buf_size)
{
  char *expanded = (char *) NULL;
  size_t  out_size = 0;
  SSIZE_T in_bytes = 0;
  struct s_file_info ifile;

  if (w->verbose > 2)
    show_file_heading(w, fname);

  init_finfo(&ifile);

  if (open_in(&(w->err), &ifile, fname) == FALSE)
    return;

  while ((in_bytes = j2_getline(buf, buf_size, ifile.fp)) > -1)
    {
      ifile.reads++;
      ifile.bytes += in_bytes;
      j2_bye_nl((*buf));
      if (j2_expand_tab(w->tab_size, &out_size, &expanded, (*buf)) == FALSE)
	mem_error(w, &ifile);
      w->out.bytes += fprintf(w->out.fp, "%s\n", expanded);
      (w->out.writes)++;
    }

  if (w->verbose > 0)
    show_stats(w->err.fp, &ifile, LIT_STDIN);

  close_file(&ifile);

  if (expanded != (char *) NULL)
    free(expanded);

} /* process_a_file() */

/*
 * main()
 */
int main(int argc, char **argv)
{
  time_t tstart = time(&tstart), tend;
  struct s_work w;
  int i           = 0;
  char *buf       = (char *) NULL;
  size_t buf_size = (size_t) 0;

#ifdef OpenBSD
  if(pledge("stdio rpath wpath cpath",NULL) == -1)
    err(1,"pledge\n");
#endif

  init(&w, argv, argc);

  /*** process all files ***/
  for (i = optind; i < argc; i++)
    process_a_file(&w, argv[i], &buf, &buf_size);
  if (i == optind)
    process_a_file(&w, FILE_NAME_STD, &buf, &buf_size);

  /*** DONE ***/
  if (w.verbose > 0)
    {
      show_stats(w.err.fp, &(w.out), LIT_STDOUT);
      fprintf(w.err.fp, "\n");
      fprintf(w.err.fp, MSG_INFO_I152L, (long long int) (time(&tend) - tstart));
      fprintf(w.err.fp, "\n%s\n", LIT_INFO_END);
    }

  close_file(&(w.err));
  close_file(&(w.out));

  if (buf != (char *) NULL)
    free(buf);
  exit(EXIT_SUCCESS);

}  /* main() */
