/*
 * Copyright (c) 2004 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* Sorted File Diff if Two Sorted Text Files*/

#define REV_JDIFF_C "1.13 2025/08/08"
#define PROG_NAME "jdiff"

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _MSDOS
#include <sys/param.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <time.h>

/* PATH_PAX is the one to use */
#ifdef MAXPATHLEN
#ifndef PATH_MAX
#define PATH_MAX MAXPATHLEN
#endif
#else
#ifdef PATH_MAX
#define MAXPATHLEN PATH_MAX
#endif
#endif

#define BUF_SIZE 256
#ifdef _MSDOS
#define MAX_ITERATION 999999
#else
#define MAX_ITERATION 20  // DEBUG 29999999
#endif

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif

#define NUM unsigned long long
#define SCKARG 80
#define SWITCH_CHAR '-'
#define FILE_STDIN  "-"

#define ARG_BUFF          'b'  /* Buffer Size                        */
#define ARG_ERR           'e'  /* Output Error File                  */
#define ARG_FORCE         'f'  /* force create files                 */
#define ARG_HELP          'h'  /* Show Help                          */
#define ARG_MAX_SHOW      'm'  /* Max lines to print                 */
#define ARG_OUT           'o'  /* Output File                        */
#define ARG_QUIET         'q'  /* do not show any output             */
#define ARG_VERBOSE       'v'  /* Verbose                            */
#define ARG_VERSION       'V'  /* Show Version Information           */

#define LIT_EOR      "******* End of Report *******"
#define LIT_INFO_04  "Build: %s %s\n"
#define LIT_NO       "No"
#define LIT_PROG     "diff Two Large Sorted Text Files"
#define LIT_REV      "Revision"
#define LIT_STDERR   "stderr"
#define LIT_STDIN    "stdin"
#define LIT_STDOUT   "stdout"
#define LIT_YES      "Yes"

#define MSG_E000    "Try '%s %c%c' for more information\n"
#define MSG_E001    "E001: Need just One or Two Text Files for Comparison\n"
#define MSG_E002    "E002: cannot open %s for read\n    %s\n"
#define MSG_E003    "E003: cannot use stdin, already in use\n"
#define MSG_E004    "E004: File %s not sorted\n"
#define MSG_E005    "E005: Record %llu out of order\n"
#define MSG_E006    "E006: eek!  I am lost to the World\n"
#define MSG_E007    "E004: value %lld invalid value for %c%c\n"
#define MSG_E008    "E008: '%s' is an invalid value for %c%c\n"
#define MSG_E009    "E009: 'Too many Arguments specified for %c%c\n"
#define MSG_E010    "E010: '%lld' is an invalid value for %c%c\n"
#define MSG_E011    "E011: Cannot get System Time: %s\n"
#define MSG_E012    "E012: cannot allocate %ld characters of memory\n"
#define MSG_E013    "E013: cannot use file %s, already in use\n"
#define MSG_E014    "E014: cannot open %s for write\n    File already exists.\n"

#define MSG_W101    "W101: Maximum Record Length Found: %-llu (some records truncated)\n"

#define MSG_I201    "I201: Seconds %7lld, Records Compared %15llu\n"
#define MSG_I202    "I202: %s: %llu: %s\n"
#define MSG_I203    "I203: %s: %llu: %s\n"  /* for entries if one file EOF */
#define MSG_I210    "I210: Text File Number %d         : %s\n"
#define MSG_I211    "I211: Output written to          : %s\n"
#define MSG_I212    "I212: Errors written to          : %s\n"
#define MSG_I213    "I213: Verbose Level              : %-d\n"
#define MSG_I214    "I214: Write Differances Found    ? %s\n"
#define MSG_I215    "I215: Force Create Files         ? %s\n"
#define MSG_I216LL  "I216: Max Record Length Used     : %-lld\n"
#define MSG_I217    "I217: Maximum Record Length Found: %-llu\n"
#define MSG_I218    "I218: Reads File %d               : %-llu\n"
#define MSG_I219    "I219: Differences Found          : %-llu\n"
#define MSG_I220    "I220: Output Lines Written       : %-llu\n"
#define MSG_I221LL  "I221: Max Output Lines to Write  : %-lld\n"
#define MSG_I221S   "I221: Max Output Lines to Write  : unlimited\n"
#define MSG_I230LL  "I230: Run Time                   : %lld seconds\n"

#define USG_MSG_USAGE_1          "usage:\t%s [OPTIONS]\n"
#define USG_MSG_OPTIONS          "Options\n"
#define USG_MSG_ARG_BUFF         "\t%c%c n\t\t: Allocate Read Buffer of size n\n"
#define USG_MSG_ARG_ERR          "\t%c%c file\t\t: Write errors to file 'file', default stderr\n"
#define USG_MSG_ARG_FORCE        "\t%c%c\t\t: force create of files when found\n"
#define USG_MSG_ARG_HELP         "\t%c%c\t\t: Show brief help and exit\n"
#define USG_MSG_ARG_MAX_SHOW     "\t%c%c n\t\t: Maximum Number of Differences to print\n"
#define USG_MSG_ARG_OUT          "\t%c%c file\t\t: Write output to file 'file', default stdout\n"
#define USG_MSG_ARG_QUIET        "\t%c%c\t\t: Do not print any output.\n"
#define USG_MSG_ARG_VERBOSE_8    "\t%c%c\t\t: verbose level, each time specified level increases\n"
#define USG_MSG_ARG_VERSION      "\t%c%c\t\t: Show revision information and exit\n"

struct s_file
{
  FILE *fp;
  NUM reads;
  NUM writes;
  char fname[(PATH_MAX + 1)];
  int  allow_close;    /* TRUE/FALSE */
  int  at_eof;         /* TRUE/FALSE */
  int  std_assigned;   /* TRUE/FALSE */
  char *buf;
  char *buf_last;
} ;

struct s_work
{
  int arg_verbose;
  int arg_quiet;
  NUM arg_show_max;
  size_t arg_buf_size;
  int arg_force;
  NUM count_diff;
  NUM max_size_found;
  time_t seconds_start;
  time_t seconds_now;
  struct s_file err;
  struct s_file f1;
  struct s_file f2;
  struct s_file out;
} ;
NUM rtw(char *buffer)

{
  char *last_non_space;
  char *b = buffer;

  if (buffer == (char *) NULL)
    return((NUM) 0);

  last_non_space = buffer;

  for ( ; (*buffer) != (char) 0x00; buffer++)
    {
      if ( ! isspace((int)(*buffer)) )
	last_non_space = buffer;
    }

  if ( ! isspace ((int) *last_non_space) )
    last_non_space++;

  (*last_non_space) = (char) 0x00;

  return((NUM) strlen(b));

} /* rtw() */

/*
 * is_numr() -- determines if all characters are numeric
 */
int is_numr(char *s)

{
  if (s == (char *) NULL)
    return(FALSE);

  for ( ; (*s) != (char) 0; s++)
    {
      if ( ! isdigit((int)(*s)) )
	return(FALSE);
    }

  return(TRUE);

} /* is_numr() */

/*
 * show_info()
 */
void show_info(FILE *fp, int rmode)
{

  switch (rmode)
    {
    case 2:
      fprintf(fp,"%s - %s\n", PROG_NAME, LIT_PROG);
      fprintf(fp,"\t%s %s: %s - ", LIT_REV, PROG_NAME, REV_JDIFF_C);
      fprintf(fp, LIT_INFO_04, __DATE__, __TIME__);
#ifdef OSTYPE
      fprintf(fp,"\t%s\n",OSTYPE);
#endif
      break;
    default:
      fprintf(fp, USG_MSG_USAGE_1, PROG_NAME);
      fprintf(fp, "\t%s\n", LIT_PROG);
      fprintf(fp, USG_MSG_OPTIONS);
      fprintf(fp, USG_MSG_ARG_BUFF,         SWITCH_CHAR, ARG_BUFF);
      fprintf(fp, USG_MSG_ARG_ERR,          SWITCH_CHAR, ARG_ERR);
      fprintf(fp, USG_MSG_ARG_FORCE,        SWITCH_CHAR, ARG_FORCE);
      fprintf(fp, USG_MSG_ARG_HELP,         SWITCH_CHAR, ARG_HELP);
      fprintf(fp, USG_MSG_ARG_MAX_SHOW,     SWITCH_CHAR, ARG_MAX_SHOW);
      fprintf(fp, USG_MSG_ARG_OUT,          SWITCH_CHAR, ARG_OUT);
      fprintf(fp, USG_MSG_ARG_QUIET,        SWITCH_CHAR, ARG_QUIET);
      fprintf(fp, USG_MSG_ARG_VERSION,      SWITCH_CHAR, ARG_VERSION);
      fprintf(fp, USG_MSG_ARG_VERBOSE_8,    SWITCH_CHAR, ARG_VERBOSE);
      break;
    }

  exit(2);

} /* show_info() */

/*
 * set_fname() -- save a file name, adjust for stdin
 */
void set_fname(char *dest, char *fname)
{

  if (strcmp(fname, FILE_STDIN) == 0)
    return; /* assume stdin, stderr or stdout */

  strncpy(dest, fname, PATH_MAX);

} /* set_fname() */

/*
 * init_file() -- clear s_file
 */
void init_file(struct s_file *f)

{
  memset(f->fname, 0, (PATH_MAX + 1));
  f->fp             = (FILE *) NULL;
  f->reads          = (NUM) 0;
  f->writes         = (NUM) 0;
  f->allow_close    = FALSE;
  f->buf            = (char *) NULL;
  f->buf_last       = (char *) NULL;
  f->at_eof         = FALSE;
  f->std_assigned   = FALSE;

} /* init_file() */

/*
 * get_ll() -- process long long argument
 */
long long get_ll(struct s_file *err, char *argval, char arg, long long current_val)
{
  long long val = 0LL;

  if (current_val != 0LL)
    {
      fprintf(err->fp, MSG_E009, SWITCH_CHAR, arg);
      fprintf(err->fp, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(2);
    }

  if (is_numr(argval) != TRUE)
    {
      fprintf(err->fp, MSG_E008, argval,   SWITCH_CHAR, arg);
      fprintf(err->fp, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(2);
    }

  val = atoll(argval);

  if (val < 0LL)
    {
      fprintf(err->fp, MSG_E007, val, SWITCH_CHAR, arg);
      fprintf(err->fp, MSG_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(2);
    }

  return(val);

} /* get_ll() */

/*
 * init_work() -- process arguments
 */
void init_work(struct s_work *w)
{
  init_file(&(w->out));
  init_file(&(w->err));
  init_file(&(w->f1));
  init_file(&(w->f2));

  w->arg_verbose    = 0;
  w->arg_quiet      = FALSE;
  w->arg_force      = TRUE;
  w->arg_show_max   = (NUM) 2000;
  w->count_diff     = (NUM) 0;
  w->max_size_found = (NUM) 0;
  w->arg_buf_size   = (size_t) BUF_SIZE;
  w->err.fp         = stderr;
  w->out.fp         = stdout;

  w->seconds_start  = time((time_t *) NULL);
  if (w->seconds_start < 0)
    {
      fprintf(stderr, MSG_E011, strerror(errno));
      exit(2);
    }

} /* init_work() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SCKARG];
  int opt;
  int i           = 0;
  int num_files   = 0;
  int set_max     = FALSE;
  int bsize_found = FALSE;
  long long max   = 0LL;
  size_t use_size = 0;

  init_work(w);

  snprintf(ckarg, SCKARG, "%c%c%c%c%c%c:%c:%c:%c:",
	   ARG_VERBOSE,  ARG_VERSION, ARG_HELP, ARG_QUIET, ARG_FORCE,
	   ARG_MAX_SHOW, ARG_BUFF,    ARG_ERR,  ARG_OUT);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	case ARG_BUFF:
	  if (bsize_found == TRUE)
	    {
	      fprintf(stderr, MSG_E009, SWITCH_CHAR, ARG_BUFF);
	      fprintf(stderr, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(2);
	    }
	  if (is_numr(optarg) != TRUE)
	    {
	      fprintf(stderr, MSG_E008, optarg,   SWITCH_CHAR, ARG_BUFF);
	      fprintf(stderr, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(2);
	    }
	  bsize_found = TRUE;
	  w->arg_buf_size = (size_t) atol(optarg);
	  break;
	case ARG_ERR:
	  if (strlen(w->err.fname) == 0)
	    set_fname(w->err.fname, optarg);
	  else
	    {
	      fprintf(stderr, MSG_E009, SWITCH_CHAR, ARG_ERR);
	      fprintf(stderr, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(2);
	    }
	  break;
	case ARG_OUT:
	  if (strlen(w->out.fname) == 0)
	    set_fname(w->out.fname, optarg);
	  else
	    {
	      fprintf(stderr, MSG_E009, SWITCH_CHAR, ARG_OUT);
	      fprintf(stderr, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(2);
	    }
	  break;
	case ARG_MAX_SHOW:
	  max = get_ll(&(w->err), optarg, ARG_MAX_SHOW, max);
	  set_max = TRUE;
	  break;
	case ARG_HELP:
	  show_info(stderr, 1);
	  break;
	case ARG_QUIET:
	  w->arg_quiet = TRUE;
	  break;
	case ARG_FORCE:
	  w->arg_force = TRUE;
	  break;
	case ARG_VERSION:
	  show_info(stderr, 2);
	  break;
	case ARG_VERBOSE:
	  w->arg_verbose++;
	  break;
	default:
	  fprintf(stderr, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(2);
	  break;
	}
    }

  /* set file names, expect 1 or 2 files */
  for (i = optind, num_files = 0; i < argc; i++)
    {
      num_files++;
      if (strlen(w->f1.fname) == 0)
	set_fname(w->f1.fname, argv[i]);
      else
	{
	  if (strlen(w->f2.fname) == 0)
	    set_fname(w->f2.fname, argv[i]);
	}
    }

  /* validations */
  if (set_max == TRUE)
    {
      if (max < 0LL)
	{
	  fprintf(w->err.fp, MSG_E010, max, SWITCH_CHAR, ARG_MAX_SHOW);
	  fprintf(w->err.fp, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(2);
	}
      w->arg_show_max = (NUM) max;
    }

  switch (num_files)
    {
    case 1:
      /* one file assumed to be stdin */
      break;
    case 2:
      /* both files supplied */
      break;
    default:
      fprintf(w->err.fp, MSG_E001);
      fprintf(w->err.fp, MSG_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(2);
      break;
    }

  /* Allocate buffer */
  use_size = w->arg_buf_size + 2;
  w->f1.buf = calloc(use_size, sizeof(char));
  if (w->f1.buf == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_E012, (long) w->arg_buf_size);
      exit(2);
    }
  w->f2.buf = calloc(use_size, sizeof(char));
  if (w->f2.buf == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_E012, (long) w->arg_buf_size);
      exit(2);
    }
  w->f1.buf_last = calloc(use_size, sizeof(char));
  if (w->f1.buf_last == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_E012, (long) w->arg_buf_size);
      exit(2);
    }
  w->f2.buf_last = calloc(use_size, sizeof(char));
  if (w->f2.buf_last == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_E012, (long) w->arg_buf_size);
      exit(2);
    }

} /* END process_arg() */

/*
 * close_file()
 */
void close_file(struct s_file *f)
{

  if (f->allow_close == TRUE)
    fclose(f->fp);
  f->allow_close = FALSE;
  f->fp = (FILE *) NULL;

} /* close_file() */

/*
 * open_read()
 */
int open_read(struct s_file *err, struct s_file *f)
{

  if (strlen(f->fname) < 1)
    {
      if (f->std_assigned == TRUE)
	{
	  fprintf(err->fp,  MSG_E003);
	  return(FALSE);
	}
      f->std_assigned = TRUE;
      f->fp = stdin;
      return(TRUE);
    }

  f->fp = fopen(f->fname, "r");
  if (f->fp == (FILE *) NULL)
    {
      fprintf(err->fp, MSG_E002, f->fname, strerror(errno));
      return(FALSE);
    }

  f->allow_close = TRUE;
  return(TRUE);

} /* END: open_read() */

/*
 * open_write()
 */
int open_write(struct s_file *f, char *stdtxt, int force)
{
  struct stat file_info;

  if (strlen(f->fname) < 1)
    {
      if (f->std_assigned == TRUE)
	{
	  fprintf(stderr,  MSG_E013, stdtxt);
	  return(FALSE);
	}
      f->std_assigned = TRUE;
      return(TRUE);
    }

  if (force == FALSE)
    {
      if (stat(f->fname, &file_info) == 0)
	{
	  fprintf(stderr, MSG_E014, f->fname);
	  exit(EXIT_FAILURE);
	}
    }

  if (f->allow_close == TRUE)
    {
	fprintf(stderr, MSG_E013, f->fname);
	exit(EXIT_FAILURE);
    }

  f->fp = fopen(f->fname, "w");
  if (f->fp == (FILE *) NULL)
    {
      fprintf(stderr, MSG_E002, f->fname, strerror(errno));
      return(FALSE);
    }

  f->allow_close = TRUE;
  return(TRUE);

} /* END: open_read() */

/*
 * open_all() -- process arguments
 */
void open_all(struct s_work *w)
{
  if (open_write(&(w->err), LIT_STDERR, w->arg_force) != TRUE)
    exit(2);

  if (w->arg_quiet == FALSE)
    {
      if (open_write(&(w->out), LIT_STDERR, w->arg_force) != TRUE)
	{
	  close_file(&(w->err));
	  exit(2);
	}
    }
  if (open_read(&(w->err), &(w->f1)) != TRUE)
    {
      close_file(&(w->out));
      close_file(&(w->err));
      exit(2);
    }
  if (open_read(&(w->err), &(w->f2)) != TRUE)
    {
      close_file(&(w->f1));
      close_file(&(w->out));
      close_file(&(w->err));
      exit(2);
    }

} /* open_all() */

/*
 * show_stats() -- show processing stats
 */
void show_stats(struct s_work *w)
{
    if (w->max_size_found > w->arg_buf_size)
      fprintf(w->err.fp, MSG_W101, (w->max_size_found - 1));

  if (w->arg_verbose > 2)
    {
      fprintf(w->err.fp, MSG_I210, 1,
	      (strlen(w->f1.fname) == 0 ? LIT_STDIN : w->f1.fname));
      fprintf(w->err.fp, MSG_I210, 2,
	      (strlen(w->f2.fname) == 0 ? LIT_STDIN : w->f2.fname));
      fprintf(w->err.fp, MSG_I211,
	      (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_I212,
	      (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_I213, w->arg_verbose);
      fprintf(w->err.fp, MSG_I214,
	      (w->arg_quiet == TRUE ? LIT_NO  : LIT_YES));
      fprintf(w->err.fp, MSG_I215,
	      (w->arg_force == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_I216LL, (long long int) w->arg_buf_size);
    }

  if (w->arg_verbose > 1)
    {
      if (w->max_size_found <= w->arg_buf_size)
	fprintf(w->err.fp, MSG_I217, (w->max_size_found - 1));
      fprintf(w->err.fp, MSG_I218, 1, w->f1.reads);
      fprintf(w->err.fp, MSG_I218, 2, w->f2.reads);
      fprintf(w->err.fp, MSG_I219, w->count_diff);
      fprintf(w->err.fp, MSG_I220, w->out.writes);
      if (w->arg_verbose > 2)
	{
	  if (w->arg_show_max > (NUM) 0)
	    fprintf(w->err.fp, MSG_I221LL, (long long int) w->arg_show_max);
	  else
	    fprintf(w->err.fp, MSG_I221S);
	}
    }

} /* show_stats() */

/*
 * read_rec() -- get next record
 */
int read_rec(struct s_file *err, struct s_file *f, NUM *max, size_t arg_buf_size, int free_mem)
{
  static char *b = (char *) NULL;
  static size_t bsize = (size_t) 0;
  ssize_t cread = (ssize_t) 0;

  if (free_mem == TRUE)
    {
      if (b != (char *) NULL)
	free(b);
      if (f->buf != (char *) NULL)
	free(f->buf);
      if (f->buf_last != (char *) NULL)
	free(f->buf_last);
      b = (char *) NULL;
      f->buf = (char *) NULL;
      f->buf_last = (char *) NULL;
      return(FALSE);
    }

  if (f->at_eof == TRUE)
    return(FALSE);

  if (b == (char *) NULL)
    {
      bsize = arg_buf_size + (size_t) 10;
      b = calloc(bsize, sizeof(char));
      if (b == (char *) NULL)
	{
	  fprintf(err->fp, MSG_E012, (long) bsize);
	  exit(2);
	}
    }

  strncpy(f->buf_last, f->buf, arg_buf_size);

  cread = getline(&b, &bsize, f->fp);

  if (cread < (ssize_t) 0)
    {
      f->at_eof = TRUE;
      memset(f->buf_last, 0, arg_buf_size);
      memset(f->buf,      0, arg_buf_size);
      return(FALSE);
    }

  f->reads++;

  memset(f->buf,  0, arg_buf_size);
  strncpy(f->buf, b, arg_buf_size);
  rtw(f->buf);

  if (cread > (ssize_t) (*max))
    (*max) = cread;

  if (strncmp(f->buf_last, f->buf, arg_buf_size) > 0)
    {
      fprintf(err->fp, MSG_E004, f->fname);
      fprintf(err->fp, MSG_E005, f->reads);
      exit(2);
    }

  return(TRUE);

} /* read_rec() */

/*
 * diff_both() -- show results of both files active
 */
void diff_both(struct s_work *w)
{
  int results = 0;

  results = strncmp(w->f1.buf,w->f2.buf, (w->arg_buf_size - 1));
  switch (results)
    {
    case 0:
      read_rec(&(w->err), &(w->f1), &(w->max_size_found), w->arg_buf_size, FALSE);
      read_rec(&(w->err), &(w->f2), &(w->max_size_found), w->arg_buf_size, FALSE);
      break;
    default:
      w->count_diff++;
      if (results < 0)
	{
	  if ((w->arg_quiet == FALSE) && ((w->arg_show_max == 0) || (w->out.writes < w->arg_show_max)))
	    {
	      w->out.writes++;
	      fprintf(w->out.fp, MSG_I202, "F1", w->f1.reads, w->f1.buf);
	    }
	  read_rec(&(w->err), &(w->f1), &(w->max_size_found), w->arg_buf_size, FALSE);
	}
      else
	{
	  if ((w->arg_quiet == FALSE) && ((w->arg_show_max == 0) || (w->out.writes < w->arg_show_max)))
	    {
	      w->out.writes++;
	      fprintf(w->out.fp, MSG_I202, "F2", w->f2.reads, w->f2.buf);
	    }
	  read_rec(&(w->err), &(w->f2), &(w->max_size_found), w->arg_buf_size, FALSE);
	}
      break;
    }

} /* diff_both() */

/*
 * diff_single() -- show results if only 1 file is active
 */
void diff_single(struct s_work *w)
{
  w->count_diff++;

  if (w->f1.at_eof == FALSE)
    {
      if (w->arg_quiet == FALSE)
	{
	  if ((w->arg_show_max == 0) || (w->out.writes < w->arg_show_max))
	    {
	      w->out.writes++;
	      fprintf(w->out.fp, MSG_I203, "F1", w->f1.reads, w->f1.buf);
	    }
	}
      read_rec(&(w->err), &(w->f1), &(w->max_size_found), w->arg_buf_size, FALSE);
      return;
    }

  if (w->f2.at_eof == FALSE)
    {
      if (w->arg_quiet == FALSE)
	{
	  if ((w->arg_show_max == 0) || (w->out.writes < w->arg_show_max))
	    {
	      w->out.writes++;
	      fprintf(w->out.fp, MSG_I203, "F2", w->f2.reads, w->f2.buf);
	    }
	}
      read_rec(&(w->err), &(w->f2), &(w->max_size_found), w->arg_buf_size, FALSE);
      return;
    }

  fprintf(w->err.fp, MSG_E006);
  exit(2);

} /* diff_single() */

/*
 * diff() -- process diff of 2 files
 */
void diff(struct s_work *w)
{
  NUM iteration = (NUM) 0;
  NUM show_mark = (NUM) 0;

  /*** get first record ***/
  read_rec(&(w->err), &(w->f1), &(w->max_size_found), w->arg_buf_size, FALSE);
  read_rec(&(w->err), &(w->f2), &(w->max_size_found), w->arg_buf_size, FALSE);

  /*** process diff ***/
  while((w->f1.at_eof == FALSE) || (w->f2.at_eof == FALSE))
    {
      iteration++;
      show_mark++;
      if ((show_mark > (NUM) MAX_ITERATION) && (w->arg_verbose > 3))
	{
	  show_mark = (NUM) 0;
	  w->seconds_now = time((time_t *) NULL);
	  fprintf(w->err.fp, MSG_I201, (long long) (w->seconds_now - w->seconds_start), iteration);
	}
      if ((w->f1.at_eof == FALSE) && (w->f2.at_eof == FALSE))
	diff_both(w);
      else
	diff_single(w);
    }

} /* diff() */

/*
 * main
 */
int main(int argc, char **argv)
{
  struct s_work w;

  process_arg(argc, argv, &w);
  open_all(&w);

  diff(&w);

  if (w.arg_verbose > 0)
    show_stats(&w);

  close_file(&(w.f1));
  close_file(&(w.f2));
  close_file(&(w.out));
  read_rec(&(w.err), &(w.f1), 0, 0, TRUE); /* release memory for f1 */
  read_rec(&(w.err), &(w.f2), 0, 0, TRUE); /* release memory for f2 */

  if (w.arg_verbose > 1)
    {
      w.seconds_now = time((time_t *) NULL);
      fprintf(w.err.fp, MSG_I230LL, (long long int) (w.seconds_now - w.seconds_start));
      fprintf(w.err.fp, "\n%s\n", LIT_EOR);
    }

  close_file(&(w.err));

  if (w.count_diff == (NUM) 0)
    exit(0);
  exit(1);

} /* main() */
