/*
 * Copyright (c) 2023 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software
 * for any purpose with or without fee is hereby granted,
 * provided that the above copyright notice and this permission
 * notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * jadd_i.c -- Initialization Routines
 */
#include <sys/param.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jadd.h"

#define SCKARG 80
#define SIZEC  30

/*** prototypes ***/
char get_delm(FILE *, char *, char *);

/*
 * init_vals() -- initialize values
 */
void init_vals(struct s_values *v)
{
  v->file_count     = (size_t) 0;
  v->reads          = (size_t) 0;
  v->writes         = (size_t) 0;
  v->current_value  = (NUMBER) 0;
  v->min            = (NUMBER) 0;
  v->max            = (NUMBER) 0;
  v->average        = (NUMBER) 0;
  v->total          = (NUMBER) 0;
  v->raw_value      = (char *) NULL;

} /* init_vals() */

/*
 * init_w() -- initialize work area
 */
void init_w(struct s_work *w)

{
  init_finfo(&(w->out));
  init_finfo(&(w->err));

  w->err.fp    = stderr;
  w->out.fp    = stdout;

  w->num_files        = 0;
  w->verbose          = 0;
  w->force            = (int) FALSE;
  w->decimal          = DECIMAL_POINT;
  w->thousands        = THOUSAND_SEP;
  w->delm             = JLIB2_CHAR_NULL;
  w->col              = -1;

}  /* init_w() */

/*
 * print_c() - format the character or its hex value for printing
 */
char *print_c(char *str, int sizc, unsigned char c)
{

  switch (c)
    {
      case 0:
	snprintf(str, sizc, "%s", "none");
	return(str);
      case 7:
	snprintf(str, sizc, "%s", "BEL");
	return(str);
      case 9:
	snprintf(str, sizc, "%s", "TAB");
	return(str);
      case 32:
	snprintf(str, sizc, "%s", "space");
	return(str);
    }

  if ((c < 33) || (c > 126))
    {
      snprintf(str, sizc, "0x%02X", c);
      return(str);
    }

  snprintf(str, sizc, "%c", c);
  return(str);

} /* print_c() */

/*
 * get_delm() -- translate a number or string into a delimiter
 */
char get_delm(FILE *fp, char *s, char *pname)
{
  int d;
  int c = JLIB2_CHAR_NULL;

  if (strlen(s) == 1)
    {
      if ( ((*s) > 47)  && ((*s) < 58) ) /* 0 -- 9 */
	c = (*s) - 48;
      else
	c = (*s);
    }
  else
    {
      if (j2_is_numr(s) == (int) TRUE)
	{
	  d = atoi(s);
	  if ((d < 256) && (d > 0))
	    c = (char) d;
	  else
	    {
	      fprintf(fp, MSG_ERR_E006, s, SWITCH_CHAR, ARG_DELM);
	      fprintf(fp, MSG_ERR_E000, pname, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	}
      else
	{
	  fprintf(fp, MSG_ERR_E006, s, SWITCH_CHAR, ARG_DELM);
	  fprintf(fp, MSG_ERR_E000, pname, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  return(c);

} /* get_delm() */

/*
 * process_arg() -- process arguments
 */
int process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SCKARG];
  int opt           = 0;
  int i             = 0;
  int count_std     = 0;
  long int li       = 0L;
  int col_specified = FALSE;
  char *efile       = (char *) NULL;
  char *ofile       = (char *) NULL;

  snprintf(ckarg, SCKARG, "%c%c%c%c%c:%c:%c:%c:%c:", 
	  ARG_FORCE, ARG_HELP, ARG_VERBOSE, ARG_VERSION,
	  ARG_COL, ARG_DECIMAL, ARG_DELM, ARG_ERR, ARG_OUT);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	case ARG_COL:
	  if (j2_is_numr(optarg) == TRUE)
	    {
	      li = atol(optarg) - 1;
	      if (li < 0)
		{
		  fprintf(stderr, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_COL);
		  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);;
		  exit(EXIT_FAILURE);
		}
	      w->col = atol(optarg) - 1;
	      col_specified = TRUE;
	    }
	  else
	    {
	      fprintf(stderr, MSG_ERR_E006, optarg, SWITCH_CHAR, ARG_COL);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);;
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_DECIMAL:
	  w->decimal = get_delm(stderr, optarg, LIT_DECIMAL_P);
	  if ((w->decimal != ',') && (w->decimal != '.') )
	    {
	      fprintf(stderr, MSG_ERR_E072, SWITCH_CHAR, ARG_DECIMAL);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);;
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_DELM:
	  w->delm = get_delm(stderr, optarg, PROG_NAME);
	  break;
	case ARG_FORCE:
	  w->force = (int) TRUE;
	  break;
	case ARG_HELP:
	  show_brief_help(stderr);
	  break;
	case ARG_VERBOSE:
	  w->verbose++;
	  break;
	case ARG_VERSION:
	  show_rev(stderr);
	  break;
	case ARG_ERR:
	  if (efile == (char *) NULL)
	    {
	      check_path_file(stderr, optarg, ARG_ERR);
	      efile = optarg;
	    }
	  else
	    {
	      fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_OUT:
	  if (ofile == (char *) NULL)
	    {
	      check_path_file(stderr, optarg, ARG_OUT);
	      ofile = optarg;
	    }
	  else
	    {
	      fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	  break;
	default:
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	  break;
	}
    }

  /*** open 'out' files ***/
  open_out(&(w->err), efile, w->force);
  open_out(&(w->out), ofile, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, MAXPATHLEN) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  /*** leave ***/
  return(col_specified);

} /* process_arg() */

/*
 * init() -- initialize
 */
void init(int argc, char **argv, struct s_work *w)

{

  int col_specified = FALSE;
  char str[(SIZEC + 1)];

  init_w(w);
  memset(str, 0, (SIZEC + 1));

  col_specified = process_arg(argc, argv, w);

  switch (w->decimal)
    {
    case '.':
      w->thousands = ',';
      break;
    case ',':
      w->thousands = '.';
      break;
    default:
      fprintf(w->err.fp, MSG_ERR_E101, w->decimal, '.', ',');
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
      break;
    }

  if ((w->delm != JLIB2_CHAR_NULL) && (col_specified == FALSE))
    {
      fprintf(w->err.fp, MSG_ERR_E097, SWITCH_CHAR, ARG_COL, SWITCH_CHAR, ARG_DELM);
      fprintf(w->err.fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }
  if (col_specified == FALSE) /* set default */
    w->col = 0;

  if (w->verbose > 1) /* show arguments */
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      fprintf(w->err.fp, MSG_INFO_I203, (long long int) w->col + 1);
      fprintf(w->err.fp, MSG_INFO_I094, w->decimal);
      fprintf(w->err.fp, MSG_INFO_I087, (w->force == TRUE ? LIT_YES : LIT_NO));
      if (w->delm != JLIB2_CHAR_NULL)
	fprintf(w->err.fp, MSG_INFO_I140S, print_c(str, SIZEC, w->delm));
    }
  
}  /* init() */
