#!/usr/pkg/bin/perl -w

# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: mt-xmlrpc.cgi,v 1.10 2002/03/18 20:43:56 btrott Exp $
use strict;

my($MT_DIR);
BEGIN {
    if ($0 =~ m!(.*[/\\])!) {
        $MT_DIR = $1;
    } else {
        $MT_DIR = './';
    }
    unshift @INC, $MT_DIR . 'lib';
    unshift @INC, $MT_DIR . 'extlib';
}

my($HAVE_XML_PARSER);
BEGIN {
    eval { require XML::Parser };
    $HAVE_XML_PARSER = $@ ? 0 : 1;
}

use XMLRPC::Transport::HTTP;
my $server = XMLRPC::Transport::HTTP::CGI->new;
$server->dispatch_to('blogger');
$server->handle;

## Following is the XML-RPC implementation of the Blogger API. The public
## methods are those that don't start with '_', and are documented in
## the User Manual (docs/mtmanual.html).

package MT::XMLRPC;
use MT::ErrorHandler;
BEGIN { @MT::XMLRPC::ISA = qw( MT::ErrorHandler ) }

use MT;
use MT::Util qw( first_n_words decode_html );

## This is sort of a hack. XML::Parser automatically makes everything
## UTF-8, and that is causing severe problems with the serialization
## of database records (what happens is this: we construct a string
## consisting of pack('N', length($string)) . $string. If the $string SV
## is flagged as UTF-8, the packed length is then upgraded to UTF-8,
## which turns characters with values greater than 128 into two bytes,
## like v194.129. And so on. This is obviously now what we want, because
## pack produces a series of bytes, not a string that should be mucked
## about with.)
##
## The following subroutine strips the UTF8 flag from a string, thus
## forcing it into a series of bytes. "pack 'C0'" is a magic way of
## forcing the following string to be packed as bytes, not as UTF8.

sub no_utf8 {
    for (@_) {
        $_ = pack 'C0A*', $_;
    }
}

sub _login {
    my $class = shift;
    my($user, $pass, $blog_id) = @_;
    require MT::Author;
    my $author = MT::Author->load({ name => $user }) or return;
    $author->is_valid_password($pass) or return;
    return $author unless $blog_id;
    require MT::Permission;
    my $perms = MT::Permission->load({ author_id => $author->id,
                                       blog_id => $blog_id });
    ($author, $perms);
}

sub _publish {
    my $class = shift;
    my($mt, $entry) = @_;
    require MT::Blog;
    my $blog = MT::Blog->load($entry->blog_id);
    $mt->rebuild_entry( Entry => $entry, Blog => $blog,
                        BuildDependencies => 1 )
        or return $class->error("Rebuild error: " . $mt->errstr);
    $mt->ping(Blog => $blog)
        or return $class->error("Ping error: " . $mt->errstr);
    1;
}

sub newPost {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $blog_id, $user, $pass, $content, $publish) = @_;
    no_utf8($blog_id, $content);
    unless ($HAVE_XML_PARSER) {
        $content = decode_html($content);
    }
    my $mt = MT->new;
    require MT::Blog;
    my $blog = MT::Blog->load($blog_id)
        or die "Invalid blog ID '$blog_id'\n";
    my($author, $perms) = __PACKAGE__->_login($user, $pass, $blog_id);
    die "Invalid login\n" unless $author;
    die "No posting privileges\n" unless $perms && $perms->can_post;
    require MT::Entry;
    my $entry = MT::Entry->new;
    $entry->blog_id($blog_id);
    $entry->author_id($author->id);
    $entry->status($publish ? MT::Entry::RELEASE() : MT::Entry::HOLD());
    $entry->allow_comments($blog->allow_comments_default);
    $entry->convert_breaks($blog->convert_paras);
    $entry->title( first_n_words($content, 5) );
    $entry->text($content);
    $entry->save;
    if ($publish) {
        __PACKAGE__->_publish($mt, $entry) or die __PACKAGE__->errstr;
    }
    SOAP::Data->type(string => $entry->id);
}

sub editPost {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $entry_id, $user, $pass, $content, $publish) = @_;
    no_utf8($content);
    unless ($HAVE_XML_PARSER) {
        $content = decode_html($content);
    }
    my $mt = MT->new;
    require MT::Entry;
    my $entry = MT::Entry->load($entry_id)
        or die "Invalid entry ID '$entry_id'\n";
    my($author, $perms) = __PACKAGE__->_login($user, $pass, $entry->blog_id);
    die "Invalid login\n" unless $author;
    die "Not privileged to edit entry\n"
        unless $perms && $perms->can_post ||
        ($perms->can_edit_all_posts && $entry->author_id == $author->id);
    $entry->status(MT::Entry::RELEASE()) if $publish;
    $entry->text($content);
    $entry->save;
    if ($publish) {
        __PACKAGE__->_publish($mt, $entry) or die __PACKAGE__->errstr;
    }
    SOAP::Data->type(boolean => 1);
}

sub getUsersBlogs {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $user, $pass) = @_;
    my $mt = MT->new;
    my($author) = __PACKAGE__->_login($user, $pass);
    die "Invalid login\n" unless $author;
    require MT::Permission;
    require MT::Blog;
    my $iter = MT::Permission->load_iter({ author_id => $author->id });
    my @res;
    while (my $perms = $iter->()) {
        next unless $perms->can_post;
        my $blog = MT::Blog->load($perms->blog_id);
        push @res, { url => SOAP::Data->type(string => $blog->site_url),
                     blogid => SOAP::Data->type(string => $blog->id),
                     blogName => SOAP::Data->type(string => $blog->name) };
    }
    \@res;
}

sub getUserInfo {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $user, $pass) = @_;
    my $mt = MT->new;
    my($author) = __PACKAGE__->_login($user, $pass);
    die "Invalid login\n" unless $author;
    my($fname, $lname) = split /\s+/, $author->name;
    { userid => SOAP::Data->type(string => $author->id),
      firstname => SOAP::Data->type(string => $fname),
      lastname => SOAP::Data->type(string => $lname),
      nickname => SOAP::Data->type(string => $author->nickname),
      email => SOAP::Data->type(string => $author->email),
      url => SOAP::Data->type(string => $author->url) };
}

sub getRecentPosts {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $blog_id, $user, $pass, $num) = @_;
    my $mt = MT->new;
    my($author, $perms) = __PACKAGE__->_login($user, $pass, $blog_id);
    die "Invalid login\n" unless $author;
    die "No posting privileges\n" unless $perms && $perms->can_post;
    require MT::Entry;
    my $iter = MT::Entry->load_iter({ blog_id => $blog_id },
        { 'sort' => 'created_on',
          direction => 'descend',
          limit => $num });
    my @res;
    while (my $entry = $iter->()) {
        my $co = sprintf "%04d%02d%02dT%02d:%02d:%02d",
            unpack 'A4A2A2A2A2A2', $entry->created_on;
        push @res, { dateCreated => SOAP::Data->type(dateTime => $co),
                     userid => SOAP::Data->type(string => $entry->author_id),
                     postid => SOAP::Data->type(string => $entry->id),
                     content => SOAP::Data->type(string => $entry->text) };
    }
    \@res;
}

sub deletePost {
    shift if UNIVERSAL::isa($_[0] => __PACKAGE__);
    my($appkey, $entry_id, $user, $pass, $publish) = @_;
    my $mt = MT->new;
    require MT::Entry;
    my $entry = MT::Entry->load($entry_id)
        or die "Invalid entry ID '$entry_id'\n";
    my($author, $perms) = __PACKAGE__->_login($user, $pass, $entry->blog_id);
    die "Invalid login\n" unless $author;
    die "Not privileged to delete entry\n"
        unless $perms && $perms->can_post ||
        ($perms->can_edit_all_posts && $entry->author_id == $author->id);
    $entry->remove;
    if ($publish) {
        __PACKAGE__->_publish($mt, $entry) or die __PACKAGE__->errstr;
    }
    SOAP::Data->type(boolean => 1);
}

## getTemplate and setTemplate are not applicable in MT's template
## structure, so they are unimplemented (they return a fault).
## We assign it twice to get rid of "setTemplate used only once" warnings.

sub getTemplate {
    die "Template methods are not implemented, due to differences between " .
        "the Blogger API and the Movable Type API.\n";
}
*setTemplate = *setTemplate = \&getTemplate;

## The above methods will be called as blogger.newPost, blogger.editPost,
## etc., because we are implementing Blogger's API. Thus, the empty
## subclass.
package blogger;
BEGIN { @blogger::ISA = qw( MT::XMLRPC ); }
