# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: Util.pm,v 1.52 2002/03/18 20:42:29 btrott Exp $

package MT::Util;
use strict;

use MT::ConfigMgr;
use MT::Request;
use Exporter;
@MT::Util::ISA = qw( Exporter );
use vars qw( @EXPORT_OK );
@EXPORT_OK = qw( start_end_day start_end_week start_end_month 
                 html_text_transform encode_html decode_html munge_comment
                 offset_time offset_time_list first_n_words
                 archive_file_for format_ts dirify remove_html
                 days_in wday_from_ts encode_js get_entry spam_protect
                 is_valid_email encode_php );

sub unix_from_ts {
    my @gm_args = @_;
    require Time::Local;
    my $gm;
    ## If timegm_nocheck is supported, we will use that to determine the
    ## timegm and skip the validity checks. If it is not supported, we just
    ## fall back to using standard timegm, but wrap in an eval to catch
    ## any errors.
    if (defined &Time::Local::timegm_nocheck) {
        $gm = Time::Local::timegm_nocheck(@gm_args);
    } else {
        eval { $gm = Time::Local::timegm(@gm_args) };
    }
    $gm;
}

sub wday_from_ts { (gmtime unix_from_ts(@_))[6] }

use vars qw( %Languages );
sub format_ts {
    my($format, $ts, $blog) = @_;
    my %f;
    my $lang = $blog && $blog->language ? $blog->language : 'en';
    my $cache = MT::Request->instance->cache('formats');
    unless ($cache) {
        MT::Request->instance->cache('formats', $cache = {});
    }
    if (my $f_ref = $cache->{$ts . $lang}) {
        %f = %$f_ref;
    } else {
        my $L = $Languages{$lang};
        my @ts = @f{qw( Y m d H M S )} = unpack 'A4A2A2A2A2A2', $ts;
        my $unix = unix_from_ts(@ts[5,4,3,2], $ts[1]-1, $ts[0]-1900);
        ($f{w}, $f{j}) = (gmtime $unix)[6, 7];
        $f{y} = substr $f{Y}, 2;
        $f{b} = substr $L->[1][$f{m}-1], 0, 3;
        $f{B} = $L->[1][$f{m}-1];
        $f{a} = substr $L->[0][$f{w}], 0, 3;
        $f{A} = $L->[0][$f{w}];
        ($f{e} = $f{d}) =~ s!^0! !;
        $f{I} = $f{H};
        $f{I} = $f{H};
        if ($f{I} > 12) {
            $f{I} -= 12;
            $f{p} = $L->[2][1];
        } elsif ($f{I} == 0) {
            $f{I} = 12;
            $f{p} = $L->[2][0];
        } elsif ($f{I} == 12) {
            $f{p} = $L->[2][1];
        } else {
            $f{p} = $L->[2][0];
        }
        $f{I} = sprintf "%02d", $f{I};
        ($f{k} = $f{H}) =~ s!^0! !;
        ($f{l} = $f{I}) =~ s!^0! !;
        $f{j} = sprintf "%03d", $f{j};
        $f{Z} = '';
        $cache->{$ts . $lang} = \%f;
    }

    $format =~ s!%(\w)!$f{$1}!g;
    $format;
}

{
    my @Days_In = ( -1, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 );
    sub days_in {
        my($m, $y) = @_;
        return $Days_In[$m] unless $m == 2;
        return $y % 4 == 0 && ($y % 100 != 0 || $y % 400 == 0) ?
            29 : 28;
    }
}

sub start_end_day {
    my $day = substr $_[0], 0, 8;
    return $day . '000000' unless wantarray;
    ($day . "000000", $day . "235959");
}

sub start_end_week {
    my($ts) = @_;
    my($y, $mo, $d, $h, $m, $s) = unpack 'A4A2A2A2A2A2', $ts;
    my $wday = wday_from_ts($s, $m, $h, $d, $mo-1, $y-1900);
    my($sd, $sm, $sy) = ($d - $wday, $mo, $y);
    if ($sd < 1) {
        $sm--;
        $sm = 12, $sy-- if $sm < 1;
        $sd += days_in($sm, $sy);
    }
    my $start = sprintf "%04d%02d%02d%s", $sy, $sm, $sd, "000000";
    return $start unless wantarray;
    my($ed, $em, $ey) = ($d + 6 - $wday, $mo, $y);
    if ($ed > days_in($em, $ey)) {
        $ed -= days_in($em, $ey);
        $em++;
        $em = 1, $ey++ if $em > 12;
    }
    my $end = sprintf "%04d%02d%02d%s", $ey, $em, $ed, "235959";
    ($start, $end);
}

sub start_end_month {
    my($ts) = @_;
    my($y, $mo) = unpack 'A4A2', $ts;
    my $start = sprintf "%04d%02d01000000", $y, $mo;
    return $start unless wantarray;
    my $end = sprintf "%04d%02d%02d235959", $y, $mo, days_in($mo, $y);
    ($start, $end);
}

sub offset_time_list { gmtime offset_time(@_) }

sub offset_time {
    my($ts, $blog, $dir) = @_;
    my $offset;
    if (defined $blog) {
        if (!ref($blog)) {
            require MT::Blog;
            $blog = MT::Blog->load($blog);
        }
        $offset = $blog && $blog->server_offset ? $blog->server_offset : 0;
    } else {
        $offset = MT::ConfigMgr->instance->TimeOffset;
    }
    $offset += 1 if (localtime $ts)[8];
    $offset *= -1 if $dir && $dir eq '-';
    $ts += $offset * 3600;
    $ts;
}

sub html_text_transform {
    my $str = shift;
    $str ||= '';
    my @paras = split /\r?\n\r?\n/, $str;
    for my $p (@paras) {
        $p =~ s!\r?\n!<br />\n!g;
        $p = "<p>$p</p>";
    }
    join "\n\n", @paras;
}

{
    my %Map = (':' => '&#58;', '@' => '&#64;', '.' => '&#46;');
    sub spam_protect {
        my($str) = @_;
        my $look = join '', keys %Map;
        $str =~ s!([$look])!$Map{$1}!g;
        $str;
    }
}

sub encode_js {
    my($str) = @_;
    return '' unless defined $str;
    $str =~ s!(['"])!\\$1!g;
    $str =~ s!\n!\\n!g;
    $str =~ s!\f!\\f!g;
    $str =~ s!\r!\\r!g;
    $str =~ s!\t!\\t!g;
    $str;
}

sub encode_php {
    my($str, $meth) = @_;
    return '' unless defined $str;
    if ($meth eq 'qq') {
        $str = encode_phphere($str);
        $str =~ s!"!\\"!g;    ## Replace " with \"
    } elsif (substr($meth, 0, 4) eq 'here') {
        $str = encode_phphere($str);
    } else {
        $str =~ s!'!\\'!g;    ## Replace ' with \'
        $str =~ s!\\!\\\\!;   ## Replace \ with \\
    }
    $str;
}

sub encode_phphere {
    my($str) = @_;
    $str =~ s!\\!\\\\!g;      ## Replace \ with \\
    $str =~ s!\$!\\\$!g;      ## Replace $ with \\$
    $str =~ s!\n!\\n!g;       ## Replace character \n with string \n
    $str =~ s!\r!\\r!g;       ## Replace character \r with string \r
    $str =~ s!\t!\\t!g;       ## Replace character \t with string \t
    $str;
}

{
    my $Have_Entities = eval 'use HTML::Entities; 1' ? 1 : 0;

    sub encode_html {
        my($html) = @_;
        return '' unless defined $html;
        $html =~ tr!\cM!!d;
        if ($Have_Entities) {
            $html = HTML::Entities::encode_entities($html);
        } else {
            $html =~ s!&!&amp;!g;
            $html =~ s!"!&quot;!g;
            $html =~ s!<!&lt;!g;
            $html =~ s!>!&gt;!g;
        }
        $html;
    }

    sub decode_html {
        my($html) = @_;
        return '' unless defined $html;
        $html =~ tr!\cM!!d;
        if ($Have_Entities) {
            $html = HTML::Entities::decode_entities($html);
        } else {
            $html =~ s!&quot;!"!g;
            $html =~ s!&lt;!<!g;
            $html =~ s!&gt;!>!g;
            $html =~ s!&amp;!&!g;
        }
        $html;
    }
}

sub remove_html {
    my($text) = @_;
    $text =~ s!<[^>]+>!!gs;
    $text;
}

sub dirify {
    my $s = lc $_[0];       ## lower-case.
    $s = remove_html($s);   ## remove HTML tags.
    $s =~ s!&[^;\s]+;!!g;   ## remove HTML entities.
    $s =~ s![^\w\s]!!g;     ## remove non-word/space chars.
    $s =~ tr! !_!s;         ## change space chars to underscores.
    $s;    
}

sub first_n_words {
    my($text, $n) = @_;
    $text = remove_html($text);
    my @words = split /\s+/, $text;
    my $max = @words > $n ? $n : @words;
    return join ' ', @words[0..$max-1];
}

sub munge_comment {
    my($text, $blog) = @_;
    unless ($blog->allow_comment_html) {
        $text = remove_html($text);
        if ($blog->autolink_urls) {
            $text =~ s!(http://\S+)!<a href="$1">$1</a>!g;
        }
    }
    $text;
}

sub archive_file_for {
    my($entry, $blog, $at, $cat, $map) = @_;
    return if $at eq 'None';
    my $file;
    unless ($map) {
        my $cache = MT::Request->instance->cache('maps');
        unless ($cache) {
            MT::Request->instance->cache('maps', $cache = {});
        }
        unless ($map = $cache->{$blog->id . $at}) {
            require MT::TemplateMap;
            $map = MT::TemplateMap->load({ blog_id => $blog->id,
                                           archive_type => $at,
                                           is_preferred => 1 });
            $cache->{$blog->id . $at} = $map if $map;
        }
    }
    my $file_tmpl = $map ? $map->file_template : '';
    my($ctx);
    if ($file_tmpl) {
        require MT::Template::Context;
        $ctx = MT::Template::Context->new;
        $ctx->stash('blog', $blog);
    }
    if ($at eq 'Individual') {
        if ($file_tmpl) {
            $ctx->stash('entry', $entry);
            $ctx->{current_timestamp} = $entry->created_on;
        } else {
            $file = sprintf("%06d", $entry->id);
        }
    } elsif ($at eq 'Daily') {
        if ($file_tmpl) {
            ($ctx->{current_timestamp}, $ctx->{current_timestamp_end}) =
                start_end_day($entry->created_on);
        } else {
            my $start = start_end_day($entry->created_on);
            my($year, $mon, $mday) = unpack 'A4A2A2', $start;
            $file = sprintf("%04d_%02d_%02d", $year, $mon, $mday);
        }
    } elsif ($at eq 'Weekly') {
        if ($file_tmpl) {
            ($ctx->{current_timestamp}, $ctx->{current_timestamp_end}) =
                start_end_week($entry->created_on);
        } else {
            my $start = start_end_week($entry->created_on);
            my($year, $mon, $mday) = unpack 'A4A2A2', $start;
            $file = sprintf("week_%04d_%02d_%02d", $year, $mon, $mday);
        }
    } elsif ($at eq 'Monthly') {
        if ($file_tmpl) {
            ($ctx->{current_timestamp}, $ctx->{current_timestamp_end}) =
                start_end_month($entry->created_on);
        } else {
            my $start = start_end_month($entry->created_on);
            my($year, $mon) = unpack 'A4A2', $start;
            $file = sprintf("%04d_%02d", $year, $mon);
        }
    } elsif ($at eq 'Category') {
        my $this_cat = $cat ? $cat : $entry->category;
        if ($file_tmpl) {
            $ctx->stash('archive_category', $this_cat);
        } else {
            my $label = '';
            if ($this_cat) {
                $label = dirify($this_cat->label);
            }
            $file = sprintf("cat_%s", $label);
        }
    } else {
        return $entry->error("Invalid Archive Type setting '$at'");
    }
    if ($file_tmpl) {
        require MT::Builder;
        my $build = MT::Builder->new;
        my $tokens = $build->compile($ctx, $file_tmpl) or return;
        defined($file = $build->build($ctx, $tokens)) or return;
    } else {
        my $ext = $blog->file_extension || 'html';
        $file .= '.' . $ext;
    }
    $file;
}

{
    my %Helpers = ( Monthly => \&start_end_month,
                    Weekly => \&start_end_week,
                    Daily => \&start_end_day,
                  );
    sub get_entry {
        my($ts, $blog_id, $at, $order) = @_;
        my($start, $end) = $Helpers{$at}->($ts);
        if ($order eq 'previous') {
            $order = 'descend';
            $ts = $start;
        } else {
            $order = 'ascend';
            $ts = $end;
        }
        my $entry = MT::Entry->load(
            { blog_id => $blog_id,
              status => MT::Entry::RELEASE() },
            { limit => 1,
              'sort' => 'created_on',
              direction => $order,
              start_val => $ts });
        $entry;
    }
}

sub is_valid_email {
    my($addr) = @_;
    if ($addr =~ /[ |\t|\r|\n]*\"?([^\"]+\"?@[^ <>\t]+\.[^ <>\t][^ <>\t]+)[ |\t|\r|\n]*/) {
        return $1;
    } else {
        return 0;
    }
}

%Languages = (
    en => [
            [ qw( Sunday Monday Tuesday Wednesday Thursday Friday Saturday ) ],
            [ qw( January February March April May June
                  July August September October November December ) ],
            [ qw( AM PM ) ],
          ],

    fr => [
            [ qw( Dimanche Lundi Mardi Mercredi Jeudi Vendredi Samedi ) ],
            [ ('Janvier', "F\xeavrier", 'Mars', 'Avril', 'Mai', 'Juin',
               'Juillet', "Ao\xfbt", 'Septembre', 'Octobre', 'Novembre',
               "D\xeacembre") ],
            [ 'du matin', 'du soir' ],
          ],

    es => [
            [ qw( Domingo Lunes Martes Miercoles Jueves Viernes Sabado ) ],
            [ qw( Enero Febrero Marzo Abril Mayo Junio Julio Agosto
                  Septiembre Octubre Noviembre Diciembre ) ],
            [ qw( AM PM ) ],
          ],

    pt => [
            [ ('domingo', 'segunda-feira', "ter\xe7a-feira", 'quarta-feira',
               'quinta-feira', 'sexta-feira', "s\xe1bado") ],
            [ ('janeiro', 'fevereiro', "mar\xe7o", 'abril', 'maio', 'junho',
               'julho', 'agosto', 'setembro', 'outubro', 'novembro',
               'dezembro' ) ],
            [ qw( AM PM ) ],
          ],

    nl => [
            [ qw( zondag maandag dinsdag woensag donderdag vrijdag
                  zaterdag ) ],
            [ qw( januari februari maart april mei juni juli augustus
                  september oktober november december ) ],
            [ qw( am pm ) ],
          ],


    se => [
            [ ("S\xf6ondag", "M\xe5dag", 'Tisdag', 'Onsdag', 'Torsdag',
               'Fredag', "L\xf6rdag") ],
            [ qw( Januari Februari Mars April Maj Juni Juli Augusti
                  September Oktober November December ) ],
            [ qw( FM EM ) ],
          ],

    de => [
            [ qw( Sonntag Montag Dienstag Mittwoch Donnerstag Freitag
                  Samstag ) ],
            [ ('Januar', 'Februar', "M\xe4erz", 'April', 'Mai', 'Juni',
               'Juli', 'August', 'September', 'Oktober', 'November',
               'December') ],
            [ qw( FM EM ) ],
          ],

    it => [
            [ ('Domenica', "Luned\xee", "Marted\xee", "Mercoled\xee",
               "Gioved\xee", "Venerd\xee", 'Sabato') ],
            [ qw( Gennaio Febbraio Marzo Aprile Maggio Giugno Luglio
                  Agosto Settembre Ottobre Novembre Dicembre ) ],
            [ qw( AM PM ) ],
          ],

    pl => [
            [ qw( niedziela poniedzialek wtorek sroda czwartek piatek
                  sobota ) ],
            [ qw( stycznia luty marca kwietnia maja czerwca lipca
                  sierpnia wrzesnia pazdziernika listopada grudnia ) ],
            [ qw( AM PM ) ],
          ],
);

1;
