# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: Context.pm,v 1.122 2002/03/18 21:35:29 btrott Exp $

package MT::Template::Context;
use strict;

use MT::Util qw( start_end_day start_end_week start_end_month
                 html_text_transform munge_comment archive_file_for
                 format_ts offset_time_list first_n_words dirify get_entry
                 encode_html encode_js remove_html wday_from_ts days_in
                 spam_protect encode_php );
use MT::ConfigMgr;
use MT::Request;
use MT::ErrorHandler;
@MT::Template::Context::ISA = qw( MT::ErrorHandler );

sub new {
    my $class = shift;
    my $ctx = bless {}, $class;
    $ctx->init(@_);
}

sub init {
    my $ctx = shift;
    $ctx->init_default_handlers;
    $ctx;
}

sub init_default_handlers {
    my $ctx = shift;
    $ctx->register_handler(CGIPath => \&_hdlr_cgi_path);
    $ctx->register_handler(Date => \&_hdlr_sys_date);
    $ctx->register_handler(Version => \&_hdlr_mt_version);

    $ctx->register_handler(BlogID => \&_hdlr_blog_id);
    $ctx->register_handler(BlogName => \&_hdlr_blog_name);
    $ctx->register_handler(BlogDescription => \&_hdlr_blog_description);
    $ctx->register_handler(BlogURL => \&_hdlr_blog_url);
    $ctx->register_handler(BlogArchiveURL => \&_hdlr_blog_archive_url);
    $ctx->register_handler(BlogHost => \&_hdlr_blog_host);
    $ctx->register_handler(BlogTimezone => \&_hdlr_blog_timezone);
    $ctx->register_handler(BlogEntryCount => \&_hdlr_blog_entry_count);
    $ctx->register_handler(BlogCommentCount => \&_hdlr_blog_comment_count);

    $ctx->register_handler(Entries => [ \&_hdlr_entries, 1 ]);
    $ctx->register_handler(EntryID => \&_hdlr_entry_id);
    $ctx->register_handler(EntryTitle => \&_hdlr_entry_title);
    $ctx->register_handler(EntryCategory => \&_hdlr_entry_category);
    $ctx->register_handler(EntryCategories => [ \&_hdlr_entry_categories, 1 ]);
    $ctx->register_handler(EntryBody => \&_hdlr_entry_body);
    $ctx->register_handler(EntryMore => \&_hdlr_entry_more);
    $ctx->register_handler(EntryExcerpt => \&_hdlr_entry_excerpt);
    $ctx->register_handler(EntryLink => \&_hdlr_entry_link);
    $ctx->register_handler(EntryAuthor => \&_hdlr_entry_author);
    $ctx->register_handler(EntryAuthorEmail => \&_hdlr_entry_author_email);
    $ctx->register_handler(EntryAuthorURL => \&_hdlr_entry_author_url);
    $ctx->register_handler(EntryAuthorLink => \&_hdlr_entry_author_link);
    $ctx->register_handler(EntryAuthorNickname => \&_hdlr_entry_author_nick);
    $ctx->register_handler(EntryDate => \&_hdlr_date);
    $ctx->register_handler(EntryCommentCount => \&_hdlr_entry_comments);
    $ctx->register_handler(EntryIfExtended => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(EntryIfAllowComments => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(EntryPrevious => [ \&_hdlr_entry_previous, 1 ]);
    $ctx->register_handler(EntryNext => [ \&_hdlr_entry_next, 1 ]);

    $ctx->register_handler(DateHeader => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(DateFooter => [ \&_hdlr_pass_tokens, 1 ]);

    $ctx->register_handler(ArchivePrevious => [ \&_hdlr_archive_prev_next, 1 ]);
    $ctx->register_handler(ArchiveNext => [ \&_hdlr_archive_prev_next, 1 ]);

    $ctx->register_handler(Include => \&_hdlr_include);

    $ctx->register_handler(ErrorMessage => \&_hdlr_error_message);

    $ctx->register_handler(GetVar => \&_hdlr_var);
    $ctx->register_handler(SetVar => \&_hdlr_var);

    $ctx->register_handler(Comments => [ \&_hdlr_comments, 1 ]);
    $ctx->register_handler(CommentID => \&_hdlr_comment_id);
    $ctx->register_handler(CommentEntryID => \&_hdlr_comment_entry_id);
    $ctx->register_handler(CommentName => \&_hdlr_comment_author);
    $ctx->register_handler(CommentIP => \&_hdlr_comment_ip);
    $ctx->register_handler(CommentAuthor => \&_hdlr_comment_author);
    $ctx->register_handler(CommentAuthorLink => \&_hdlr_comment_author_link);
    $ctx->register_handler(CommentEmail => \&_hdlr_comment_email);
    $ctx->register_handler(CommentURL => \&_hdlr_comment_url);
    $ctx->register_handler(CommentBody => \&_hdlr_comment_body);
    $ctx->register_handler(CommentOrderNumber => \&_hdlr_comment_order_num);
    $ctx->register_handler(CommentDate => \&_hdlr_date);
    $ctx->register_handler(CommentPreviewAuthor => \&_hdlr_comment_author);
    $ctx->register_handler(CommentPreviewIP => \&_hdlr_comment_ip);
    $ctx->register_handler(CommentPreviewAuthorLink =>
        \&_hdlr_comment_author_link);
    $ctx->register_handler(CommentPreviewEmail => \&_hdlr_comment_email);
    $ctx->register_handler(CommentPreviewURL => \&_hdlr_comment_url);
    $ctx->register_handler(CommentPreviewBody => \&_hdlr_comment_body);
    $ctx->register_handler(CommentPreviewDate => \&_hdlr_date);
    $ctx->register_handler(CommentPreviewState => \&_hdlr_comment_prev_state);
    $ctx->register_handler(CommentPreviewIsStatic =>
        \&_hdlr_comment_prev_static);

    $ctx->register_handler(ArchiveList => [ \&_hdlr_archives, 1 ]);
    $ctx->register_handler(ArchiveLink => \&_hdlr_archive_link);
    $ctx->register_handler(ArchiveTitle => \&_hdlr_archive_title);
    $ctx->register_handler(ArchiveCount => \&_hdlr_archive_count);
    $ctx->register_handler(ArchiveDate => \&_hdlr_date);
    $ctx->register_handler(ArchiveDateEnd => \&_hdlr_archive_date_end);
    $ctx->register_handler(ArchiveCategory => \&_hdlr_archive_category);

    $ctx->register_handler(ImageURL => \&_hdlr_image_url);
    $ctx->register_handler(ImageWidth => \&_hdlr_image_width);
    $ctx->register_handler(ImageHeight => \&_hdlr_image_height);

    $ctx->register_handler(Calendar => [ \&_hdlr_calendar, 1 ]);
    $ctx->register_handler(CalendarDay => \&_hdlr_calendar_day);
    $ctx->register_handler(CalendarCellNumber => \&_hdlr_calendar_cell_num);
    $ctx->register_handler(CalendarDate => \&_hdlr_date);
    $ctx->register_handler(CalendarWeekHeader => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(CalendarWeekFooter => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(CalendarIfBlank => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(CalendarIfToday => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(CalendarIfEntries => [ \&_hdlr_pass_tokens, 1 ]);
    $ctx->register_handler(CalendarIfNoEntries => [ \&_hdlr_pass_tokens, 1 ]);

    $ctx->register_handler(Categories => [ \&_hdlr_categories, 1 ]);
    $ctx->register_handler(CategoryID => \&_hdlr_category_id);
    $ctx->register_handler(CategoryLabel => \&_hdlr_category_label);
    $ctx->register_handler(CategoryArchiveLink => \&_hdlr_category_archive);
    $ctx->register_handler(CategoryCount => \&_hdlr_category_count);
}

sub post_process_handler {
    sub {
        my($ctx, $args, $str) = @_;
        if ($args) {
            if ($args->{'encode_html'}) {
                $str = encode_html($str);
            }
            if ($args->{'encode_js'}) {
                $str = encode_js($str);
            }
            if (my $meth = $args->{'encode_php'}) {
                $str = encode_php($str, $meth);
            }
            if ($args->{'remove_html'}) {
                $str = remove_html($str);
            }
            if ($args->{'dirify'}) {
                $str = dirify($str);
            }
            if ($args->{upper_case}) {
                $str = uc($str);
            }
            if ($args->{lower_case}) {
                $str = lc($str);
            }
            if (my $len = $args->{space_pad}) {
                $str = sprintf "%${len}s", $str;
            }
            if (my $len = $args->{zero_pad}) {
                $str = sprintf "%0${len}s", $str;
            }
            if (my $len = $args->{trim_to}) {
                $str = substr $str, 0, $len if $len < length($str);
            }
        }
        $str;
    }
}

sub stash {
    my $ctx = shift;
    my $key = shift;
    $ctx->{__stash}->{$key} = shift if @_;
    $ctx->{__stash}->{$key};
}

sub register_handler { $_[0]->{__handlers}{$_[1]} = $_[2] }
sub handler_for      {
    my $v = $_[0]->{__handlers}{$_[1]};
    ref($v) eq 'ARRAY' ? @$v : $v
}

sub _hdlr_include {
    my($arg, $cond) = @_[1,2];
    if (my $tmpl_name = $arg->{module}) {
        my $tmpl = MT::Template->load({ name => $tmpl_name,
                                        blog_id => $_[0]->stash('blog_id') })
            or return $_[0]->error("Can't find included template module " .
                                   "'$tmpl_name'");
        return $tmpl->build($_[0], $cond);
    } elsif (my $file = $arg->{file}) {
        my $blog = $_[0]->stash('blog');
        my @paths = ($file, map File::Spec->catfile($_, $file),
                            $blog->site_path, $blog->archive_path);
        my $path;
        for my $p (@paths) {
            $path = $p, last if -e $p && -r _;
        }
        return $_[0]->error("Can't find included file '$file'") unless $path;
        local *FH;
        open FH, $path
            or return $_[0]->error("Error opening included file '$file': $!");
        my $c = '';
        local $/; $c = <FH>;
        close FH;
        return $c;
    }
}

sub _hdlr_mt_version {
    require MT;
    MT->VERSION;
}

sub _hdlr_error_message {
    my $err = $_[0]->stash('error_message');
    defined $err ? $err : '';
}

sub _hdlr_var {
    my($ctx, $args) = @_;
    my $tag = $ctx->stash('tag');
    return $ctx->error("You used a <MT$tag> tag without any arguments.")
        unless keys %$args && $args->{name};
    if ($tag eq 'SetVar') {
        my $val = defined $args->{value} ? $args->{value} : '';
        $ctx->{__stash}{vars}{$args->{name}} = $val;
    } else {
        return $ctx->{__stash}{vars}{$args->{name}};
    }
    '';
}

sub _hdlr_cgi_path {
    my $path = MT::ConfigMgr->instance->CGIPath;
    $path .= '/' unless $path =~ m!/$!;
    $path;
}

sub _hdlr_blog_id { $_[0]->stash('blog')->id }
sub _hdlr_blog_name { $_[0]->stash('blog')->name }
sub _hdlr_blog_description {
    my $d = $_[0]->stash('blog')->description;
    defined $d ? $d : '';
}
sub _hdlr_blog_url { $_[0]->stash('blog')->site_url }
sub _hdlr_blog_archive_url { $_[0]->stash('blog')->archive_url }
sub _hdlr_blog_timezone {
    my $so = $_[0]->stash('blog')->server_offset;
    sprintf "%s%02d:00", $so < 0 ? '-' : '+', abs($so);
}
sub _hdlr_blog_host {
    my $host = $_[0]->stash('blog')->site_url;
    if ($host =~ m!^https?://([^/]+)/!) {
        return $1;
    } else {
        return '';
    }
}
sub _hdlr_blog_entry_count {
    my $blog_id = $_[0]->stash('blog')->id;
    require MT::Entry;
    scalar MT::Entry->count({ blog_id => $blog_id });
}
sub _hdlr_blog_comment_count {
    my $blog_id = $_[0]->stash('blog')->id;
    require MT::Comment;
    scalar MT::Comment->count({ blog_id => $blog_id });
}

sub _hdlr_entries {
    my($ctx, $args) = @_;
    require MT::Entry;
    my @entries;
    my $blog_id = $ctx->stash('blog_id');
    my($cat, $author, $saved_entry_stash);
    if (my $cat_name = $args->{category}) {
        require MT::Category;
        ## If this is a boolean lookup (like "Foo AND Bar"), we have to
        ## special-case the search. Then we stick the resulting list of
        ## entries into the stash so that it can be filtered using the
        ## mechanism below.
        if ($cat_name =~ /AND|OR/) {
            return $ctx->error("You can't use both AND and OR in the same " .
                               "expression ('$cat_name').")
                if $cat_name =~ /AND/ && $cat_name =~ /OR/;
            my @cats = split /\s*(?:AND|OR)\s*/, $cat_name;
            my %entries;
            require MT::Placement;
            for my $name (@cats) {
                my $cat = MT::Category->load({ label => $name,
                                               blog_id => $blog_id })
                    or return $ctx->error("No such category '$name'");
                my @place = MT::Placement->load({ category_id => $cat->id });
                for my $place (@place) {
                    $entries{$place->entry_id}++;
                }
            }
            my $is_and = $cat_name =~ /AND/;
            my $count = @cats;
            my @ids = $is_and ? grep { $entries{$_} == $count } keys %entries :
                                keys %entries;
            for my $entry_id (@ids) {
                push @entries, MT::Entry->load($entry_id);
            }
            $saved_entry_stash = $ctx->{__stash}{entries} || [];
            $ctx->{__stash}{entries} = \@entries;
            delete $args->{category};
        } else {
            $cat = MT::Category->load({ label => $cat_name,
                blog_id => $blog_id })
                or return $ctx->error("No such category '$cat_name'");
        }
    }
    if (my $author_name = $args->{author}) {
        require MT::Author;
        $author = MT::Author->load({ name => $author_name }) or
            return $ctx->error("No such author '$author_name'");
    }
    my $no_resort = 0;
    if (my $entries = $ctx->stash('entries')) {
        @entries = @$entries;
        if (%$args) {
            my @tmp;
            my $i = 0;
            my $n = $args->{lastn};
            ## If lastn is defined, and this is a Category archive, we need
            ## to sort the entries into descending order, because they will not
            ## be sorted by default. If this is not a Category archive, we
            ## need to flip the order of the array because we want the
            ## "N most recent entries"--and by default the array will be in
            ## oldest -> newest order.
            if ($n) {
                if ($ctx->{current_archive_type}) {
                    if ($ctx->{current_archive_type} eq 'Category') {
                        @entries = sort { $b->created_on cmp $a->created_on }
                                   @entries;
                    } else {
                        @entries = reverse @entries if $n;
                    }
                } else {
                    @entries = sort { $b->created_on cmp $a->created_on }
                               @entries;
                }
            }
            for my $e (@entries) {
                last if $n && $i >= $n;
                next unless !$cat || $e->is_in_category($cat);
                next unless !$author || $e->author_id == $author->id;
                push @tmp, $e;
                $i++;
            }
            @entries = @tmp;
        }
    } elsif (%$args) {
        my %terms = ( blog_id => $blog_id, status => MT::Entry::RELEASE() );
        $terms{author_id} = $author->id if $author;
        my %args;
        if ($cat) {
            require MT::Placement;
            $args{'join'} = [ 'MT::Placement', 'entry_id',
                              { category_id => $cat->id }, { unique => 1 } ];
        }
        if (my $last = $args->{lastn}) {
            $args{'sort'} = 'created_on';
            $args{direction} = 'descend';
            $args{limit} = $last;
            $args{offset} = $args->{offset} if $args->{offset};
        } elsif (my $days = $args->{days}) {
            my @ago = offset_time_list(time - 3600 * 24 * $days,
                $ctx->stash('blog_id'));
            my $ago = sprintf "%04d%02d%02d%02d%02d%02d",
                $ago[5]+1900, $ago[4]+1, @ago[3,2,1,0];
            $terms{created_on} = [ $ago ];
            %args = ( range => { created_on => 1 } );
        } elsif (my $n = $args->{recently_commented_on}) {
            $args{'join'} = [ 'MT::Comment', 'entry_id',
                { blog_id => $blog_id },
                { 'sort' => 'created_on',
                  direction => 'descend',
                  unique => 1,
                  limit => $n } ];
            $no_resort = 1;
        }
        @entries = MT::Entry->load(\%terms, \%args);
    } else {
        my $days = $ctx->stash('blog')->days_on_index;
        my @ago = offset_time_list(time - 3600 * 24 * $days,
            $ctx->stash('blog_id'));
        my $ago = sprintf "%04d%02d%02d%02d%02d%02d",
            $ago[5]+1900, $ago[4]+1, @ago[3,2,1,0];
        @entries = MT::Entry->load({ blog_id => $blog_id,
                                     created_on => [ $ago ],
                                     status => MT::Entry::RELEASE() },
            { range => { created_on => 1 } });
    }
    my $res = '';
    my $tok = $ctx->stash('tokens');
    my $builder = $ctx->stash('builder');
    unless ($no_resort) {
        my $so = $args->{sort_order} || $ctx->stash('blog')->sort_order_posts;
        my $col = $args->{sort_by} || 'created_on';
        @entries = $so eq 'ascend' ?
            sort { $a->$col() cmp $b->$col() } @entries :
            sort { $b->$col() cmp $a->$col() } @entries;
    }
    my($last_day, $next_day) = ('00000000') x 2;
    my $i = 0;
    for my $e (@entries) {
        local $ctx->{__stash}{entry} = $e;
        local $ctx->{current_timestamp} = $e->created_on;
        my $this_day = substr $e->created_on, 0, 8;
        my $next_day = $this_day;
        my $footer = 0;
        if (defined $entries[$i+1]) {
            $next_day = substr($entries[$i+1]->created_on, 0, 8);
            $footer = $this_day ne $next_day;
        } else { $footer++ }
        my $out = $builder->build($ctx, $tok, {
            DateHeader => ($this_day ne $last_day),
            DateFooter => $footer,
            EntryIfExtended => $e->text_more ? 1 : 0,
            EntryIfAllowComments => $e->allow_comments,
        });
        $last_day = $this_day;
        return $ctx->error( $builder->errstr ) unless defined $out;
        $res .= $out;
        $i++;
    }

    ## Restore a saved entry stash. This is basically emulating "local",
    ## which we can't use, because the local would be buried too far down
    ## in a conditional.
    if ($saved_entry_stash) {
        if (!@$saved_entry_stash) {
            delete $ctx->{__stash}{entries};
        } else {
            $ctx->{__stash}{entries} = $saved_entry_stash;
        }
    }
    $res;
}

sub _no_entry_error {
    return $_[0]->error("You used an '$_[1]' tag outside of the context of " .
                        "an entry; perhaps you mistakenly placed it outside " .
                        "of an 'MTEntries' container?");
}
sub _hdlr_entry_body {
    my $arg = $_[1];
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryBody');
    my $text = $e->text;
    $text = '' unless defined $text;
    if (exists $arg->{words}) {
        return first_n_words($text, $arg->{words});
    } else {
        my $convert_breaks = exists $arg->{convert_breaks} ?
            $arg->{convert_breaks} :
                defined $e->convert_breaks ? $e->convert_breaks :
                    $_[0]->stash('blog')->convert_paras;
        return $convert_breaks ? html_text_transform($text) : $text;
    }
}
sub _hdlr_entry_more {
    my $arg = $_[1];
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryMore');
    my $t = $e->text_more;
    my $convert_breaks = exists $arg->{convert_breaks} ?
        $arg->{convert_breaks} :
            defined $e->convert_breaks ? $e->convert_breaks :
                $_[0]->stash('blog')->convert_paras;
    $t = $convert_breaks ? html_text_transform($t) : $t;
    $t = '' unless defined $t;
    $t;
}
sub _hdlr_entry_title {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryTitle');
    $e->title;
}
sub _hdlr_entry_excerpt {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryExcerpt');
    if ($e->excerpt) {
        return $_[1] && $_[1]->{convert_breaks} ?
            html_text_transform($e->excerpt) : $e->excerpt;
    } elsif ($_[1] && $_[1]->{no_generate}) {
        return '';
    }
    my $words = $_[0]->stash('blog')->words_in_excerpt;
    $words = 20 unless defined $words && $words ne '';
    my $excerpt = _hdlr_entry_body($_[0], { words => $words });
    $excerpt ? $excerpt . '...' : '';
}
sub _hdlr_entry_author {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryAuthor');
    my $a = $e->author;
    $a ? $a->name || '' : '';
}
sub _hdlr_entry_author_nick {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryAuthorNickname');
    my $a = $e->author;
    $a ? $a->nickname || '' : '';
}
sub _hdlr_entry_author_email {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MT' . $_[0]->stash('tag'));
    my $a = $e->author;
    return '' unless $a && defined $a->email;
    $_[1] && $_[1]->{spam_protect} ? spam_protect($a->email) : $a->email;
}
sub _hdlr_entry_author_url {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MT' . $_[0]->stash('tag'));
    my $a = $e->author;
    $a ? $a->url || '' : '';
}
sub _hdlr_entry_author_link {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MT' . $_[0]->stash('tag'));
    my $a = $e->author;
    return '' unless $a;
    my $name = $a->name || '';
    if ($a->url) {
        return sprintf qq(<a target="_blank" href="%s">%s</a>), $a->url, $name;
    } elsif ($a->email) {
        my $str = "mailto:" . $a->email;
        $str = spam_protect($str) if $_[1] && $_[1]->{spam_protect};
        return sprintf qq(<a href="%s">%s</a>), $str, $name;
    } else {
        return $name;
    }
}
sub _hdlr_entry_id {
    my $args = $_[1];
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryID');
    $args && $args->{pad} ? (sprintf "%06d", $e->id) : $e->id;
}
sub _hdlr_entry_link {
    my $args = $_[1];
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryLink');
    my $arch = $_[0]->stash('blog')->archive_url;
    $arch .= '/' unless $arch =~ m!/$!;
    $arch . $e->archive_file($args ? $args->{archive_type} : ());
}
sub _hdlr_entry_category {
    my($ctx) = @_;
    my $e = $ctx->stash('entry')
        or return $ctx->_no_entry_error('MTEntryCategory');
    my $cat = $e->category;
    $cat ? $cat->label : '';
}

sub _hdlr_entry_categories {
    my($ctx) = @_;
    my $e = $ctx->stash('entry')
        or return $ctx->_no_entry_error('MTEntryCategories');
    my $cats = $e->categories;
    return '' unless $cats && @$cats;
    my $res = '';
    my $builder = $ctx->stash('builder');
    my $tokens = $ctx->stash('tokens');
    for my $cat (@$cats) {
        local $ctx->{__stash}->{category} = $cat;
        defined(my $out = $builder->build($ctx, $tokens))
            or return $ctx->error( $builder->errstr );
        $res .= $out;
    }
    $res;
}

sub _hdlr_entry_comments {
    my $e = $_[0]->stash('entry')
        or return $_[0]->_no_entry_error('MTEntryCommentCount');
    $e->comment_count;
}
sub _hdlr_entry_previous {
    my $ctx = $_[0];
    my $e = $ctx->stash('entry')
        or return $ctx->_no_entry_error('MTEntryPrevious');
    my $prev = $e->previous;
    my $res = '';
    if ($prev) {
        my $builder = $ctx->stash('builder');
        local $ctx->{__stash}->{entry} = $prev;
        local $ctx->{current_timestamp} = $prev->created_on;
        my %cond;
        $cond{EntryIfAllowComments} = $prev->allow_comments;
        $cond{EntryIfExtended} = $prev->text_more ? 1 : 0;
        my $out = $builder->build($ctx, $ctx->stash('tokens'), \%cond);
        return $ctx->error( $builder->errstr ) unless defined $out;
        $res .= $out;
    }
    $res;
}
sub _hdlr_entry_next {
    my $ctx = $_[0];
    my $e = $ctx->stash('entry')
        or return $ctx->_no_entry_error('MTEntryNext');
    my $next = $e->next;
    my $res = '';
    if ($next) {
        my $builder = $ctx->stash('builder');
        local $ctx->{__stash}->{entry} = $next;
        local $ctx->{current_timestamp} = $next->created_on;
        my %cond;
        $cond{EntryIfAllowComments} = $next->allow_comments;
        $cond{EntryIfExtended} = $next->text_more ? 1 : 0;
        my $out = $builder->build($ctx, $ctx->stash('tokens'), \%cond);
        return $ctx->error( $builder->errstr ) unless defined $out;
        $res .= $out;
    }
    $res;
}

sub _hdlr_pass_tokens {
    my($ctx, $args) = @_;
    $ctx->stash('builder')->build($ctx, $ctx->stash('tokens'));
}

sub _hdlr_sys_date {
    my $args = $_[1];
    my @ts = offset_time_list(time, $_[0]->stash('blog_id'));
    $args->{ts} = sprintf "%04d%02d%02d%02d%02d%02d",
        $ts[5]+1900, $ts[4]+1, @ts[3,2,1,0];
    _hdlr_date($_[0], $args);
}

sub _hdlr_date {
    my $args = $_[1];
    my $ts = $args->{ts} || $_[0]->{current_timestamp};
    my $tag = $_[0]->stash('tag');
    return $_[0]->error("You used an MT$tag tag without a date " .
                        "context set up.") unless defined $ts;
    my $format = $args->{'format'} || "%B %e, %Y %I:%M %p";
    format_ts($format, $ts, $_[0]->stash('blog'));
}

sub _no_comment_error {
    return $_[0]->error("You used an '$_[1]' tag outside of the context of " .
                        "a comment; perhaps you mistakenly placed it " .
                        "outside of an 'MTComments' container?");
}
sub _hdlr_comments {
    my($ctx, $args) = @_;
    my $blog_id = $ctx->stash('blog_id');
    my @comments;
    ## If there is a "lastn" arg, then we need to check if there is an entry
    ## in context. If so, grab the N most recent comments for that entry;
    ## otherwise, grab the N most recent comments for the entire blog.
    if (my $n = $args->{lastn}) {
        if (my $e = $ctx->stash('entry')) {
            ## Sort in descending order, then grab the first $n ($n most
            ## recent) comments.
            my $comments = $e->comments;
            @comments = sort { $b->created_on <=> $a->created_on } @$comments;
            my $max = $n - 1 > $#comments ? $#comments : $n - 1;
            @comments = @comments[0..$max];
        } else {
            require MT::Comment;
            @comments = MT::Comment->load({ blog_id => $blog_id },
                { 'sort' => 'created_on',
                  direction => 'descend',
                  limit => $n });
        }
    } else {
        my $e = $ctx->stash('entry')
            or return $_[0]->_no_entry_error('MTComments');
        my $comments = $e->comments;
        my $so = $ctx->stash('blog')->sort_order_comments;
        @comments = $so eq 'ascend' ?
            sort { $a->created_on <=> $b->created_on } @$comments :
            sort { $b->created_on <=> $a->created_on } @$comments;
    }
    my $html = '';
    my $builder = $ctx->stash('builder');
    my $tokens = $ctx->stash('tokens');
    my $i = 1;
    for my $c (@comments) {
        $ctx->stash('comment' => $c);
        local $ctx->{current_timestamp} = $c->created_on;
        $ctx->stash('comment_order_num', $i);
        my $out = $builder->build($ctx, $tokens);
        return $ctx->error( $builder->errstr ) unless defined $out;
        $html .= $out;
        $i++;
    }
    $html;
}
sub _hdlr_comment_id {
    my $args = $_[1];
    my $c = $_[0]->stash('comment')
        or return $_[0]->_no_comment_error('MTCommentID');
    $args && $args->{pad} ? (sprintf "%06d", $c->id) : $c->id;
}
sub _hdlr_comment_entry_id {
    my $args = $_[1];
    my $c = $_[0]->stash('comment')
        or return $_[0]->_no_comment_error('MTCommentEntryID');
    $args && $args->{pad} ? (sprintf "%06d", $c->entry_id) : $c->entry_id;
}
sub _hdlr_comment_author {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    defined $c->author ? $c->author : '';
}
sub _hdlr_comment_ip {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    defined $c->ip ? $c->ip : '';
}
sub _hdlr_comment_author_link {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    my $name = $c->author;
    $name = '' unless defined $name;
    if ($c->url) {
        return sprintf qq(<a target="_blank" href="%s">%s</a>), $c->url, $name;
    } elsif ($c->email) {
        my $str = "mailto:" . $c->email;
        $str = spam_protect($str) if $_[1] && $_[1]->{spam_protect};
        return sprintf qq(<a href="%s">%s</a>), $str, $name;
    } else {
        return $name;
    }
}
sub _hdlr_comment_email {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    return '' unless defined $c->email;
    $_[1] && $_[1]->{spam_protect} ? spam_protect($c->email) : $c->email;
}
sub _hdlr_comment_url {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    defined $c->url ? $c->url : '';
}
sub _hdlr_comment_body {
    my $tag = $_[0]->stash('tag');
    my $c = $_[0]->stash($tag =~ /Preview/ ? 'comment_preview' : 'comment')
        or return $_[0]->_no_comment_error('MT' . $tag);
    my $blog = $_[0]->stash('blog');
    my $t = munge_comment($c->text, $blog);
    my $convert_breaks = exists $_[1]->{convert_breaks} ?
        $_[1]->{convert_breaks} :
        $blog->convert_paras_comments;
    $convert_breaks ? html_text_transform($t) : $t;
}
sub _hdlr_comment_order_num { $_[0]->stash('comment_order_num') }
sub _hdlr_comment_prev_state { $_[0]->stash('comment_state') }
sub _hdlr_comment_prev_static { $_[0]->stash('comment_is_static') }

## Archives

{
    my $cur;
    my %TypeHandlers = (
        Individual => {
            group_end => sub { 1 },
            section_title => sub { $_[1]->title },
            section_timestamp => sub { $_[1]->created_on },
        },

        Daily => {
            group_end => sub {
                my $sod = start_end_day($_[1]->created_on,
                    $_[0]->stash('blog'));
                my $end = !$cur || $sod == $cur ? 0 : 1;
                $cur = $sod;
                $end;
            },
            section_title => sub {
                my $start =
                    start_end_day($_[1]->created_on, $_[0]->stash('blog'));
                _hdlr_date($_[0], { ts => $start, 'format' => "%B %e, %Y" });
            },
            section_timestamp => sub {
                start_end_day($_[1]->created_on, $_[0]->stash('blog'))
            },
            helper => \&start_end_day,
        },

        Weekly => {
            group_end => sub {
                my $sow = start_end_week($_[1]->created_on,
                    $_[0]->stash('blog'));
                my $end = !$cur || $sow == $cur ? 0 : 1;
                $cur = $sow;
                $end;
            },
            section_title => sub {
                my($start, $end) =
                    start_end_week($_[1]->created_on, $_[0]->stash('blog'));
                _hdlr_date($_[0], { ts => $start, 'format' => "%B %e, %Y" }) .
                ' - ' .
                _hdlr_date($_[0], { ts => $end, 'format' => "%B %e, %Y" });
            },
            section_timestamp => sub {
                start_end_week($_[1]->created_on, $_[0]->stash('blog'))
            },
            helper => \&start_end_week,
        },

        Monthly => {
            group_end => sub {
                my $som = start_end_month($_[1]->created_on,
                    $_[0]->stash('blog'));
                my $end = !$cur || $som == $cur ? 0 : 1;
                $cur = $som;
                $end;
            },
            section_title => sub {
                my $start =
                    start_end_month($_[1]->created_on, $_[0]->stash('blog'));
                _hdlr_date($_[0], { ts => $start, 'format' => "%B %Y" });
            },
            section_timestamp => sub {
                start_end_month($_[1]->created_on, $_[0]->stash('blog'));
            },
            helper => \&start_end_month,
        },
    );

    sub _hdlr_archive_prev_next {
        my $ctx = $_[0];
        my $tag = $ctx->stash('tag');
        my $is_prev = $tag eq 'ArchivePrevious';
        my $ts = $ctx->{current_timestamp}
            or return $ctx->error("You used <MT$tag> outside of " .
                                  "a date context.");
        my $at = $_[1]->{archive_type} || $ctx->{current_archive_type};
        return $ctx->error("<MT$tag> can be used only with " .
                           "Daily, Weekly, or Monthly archives.")
            unless $at eq 'Daily' || $at eq 'Weekly' || $at eq 'Monthly';
        my $res = '';
        my @arg = ($ts, $ctx->stash('blog_id'), $at);
        push @arg, $is_prev ? 'previous' : 'next';
        my $helper = $TypeHandlers{$at}{helper};
        if (my $entry = get_entry(@arg)) {
            my $builder = $ctx->stash('builder');
            local $ctx->{__stash}->{entries} = [ $entry ];
            my($start, $end) = $helper->($entry->created_on);
            local $ctx->{current_timestamp} = $start;
            local $ctx->{current_timestamp_end} = $end;
            defined(my $out = $builder->build($ctx, $ctx->stash('tokens')))
                or return $ctx->error( $builder->errstr );
            $res .= $out;
        }
        $res;
    }

    sub _hdlr_archives {
        my($ctx, $args) = @_;
        $cur = undef;
        require MT::Entry;
        my $blog = $ctx->stash('blog');
        my $at = $blog->archive_type;
        return '' if !$at || $at eq 'None';
        if (my $arg_at = $args->{archive_type}) {
            my %at = map { $_ => 1 } split /,/, $at;
            return $ctx->error(<<ERR) unless $at{$arg_at};
The archive type specified in MTArchiveList ('$arg_at') is not one of the chosen archive types in your blog configuration.
ERR
            $at = $arg_at;
        } elsif ($blog->archive_type_preferred) {
            $at = $blog->archive_type_preferred;
        } else {
            $at = (split /,/, $at)[0];
        }
        ## If we are producing a Category archive list, don't bother to
        ## handle it here--instead hand it over to <MTCategories>.
        return _hdlr_categories(@_) if $at eq 'Category';
        local $ctx->{current_archive_type} = $at;
        my %args;
        if ($at eq 'Category') {
            $args{'sort'} = 'category_id';
        } else {
            $args{'sort'} = 'created_on';
            $args{direction} = 'descend';
        }
        my $group_end = $TypeHandlers{$at}{group_end};
        my $sec_ts = $TypeHandlers{$at}{section_timestamp};
        my $iter = MT::Entry->load_iter({ blog_id => $blog->id,
                                          status => MT::Entry::RELEASE() },
                                        \%args);
        my @entries;
        my $tokens = $ctx->stash('tokens');
        my $builder = $ctx->stash('builder');
        my $res = '';
        my $i = 0;
        my $n = $args->{lastn};
        while (my $entry = $iter->()) {
            if ($group_end->($ctx, $entry) && @entries) {
                local $ctx->{__stash}{entries} = \@entries;
                my($start, $end) = $sec_ts->($ctx, $entries[0]);
                local $ctx->{current_timestamp} = $start;
                local $ctx->{current_timestamp_end} = $end;
                defined(my $out = $builder->build($ctx, $tokens)) or
                    return $ctx->error( $builder->errstr );
                $res .= $out;
                @entries = ();    ## Reset entry list
                last if $n && $i++ >= $n-1;
            }
            push @entries, $entry;
        }
        if (@entries) {
            local $ctx->{__stash}{entries} = \@entries;
            my($start, $end) = $sec_ts->($ctx, $entries[0]);
            local $ctx->{current_timestamp} = $start;
            local $ctx->{current_timestamp_end} = $end;
            defined(my $out = $builder->build($ctx, $tokens)) or
                return $ctx->error( $builder->errstr );
            $res .= $out;
        }
        $res;
    }

    sub _hdlr_archive_title {
        ## Since this tag can be called from inside <MTCategories>,
        ## we need a way to map this tag to <$MTCategoryLabel$>.
        return _hdlr_category_label(@_) if $_[0]->{inside_mt_categories};

        my($ctx) = @_;
        my $entries = $ctx->stash('entries');
        return $ctx->error('You used an <$MTArchiveLink$> tag outside of the ' .
                           'proper context.')
            unless $entries && ref($entries) eq 'ARRAY' &&
                   $ctx->{current_archive_type};
        return '' unless @$entries;
        if ($ctx->{current_archive_type} eq 'Category') {
            return $ctx->stash('archive_category')->label;
        } else {
            my $st = $TypeHandlers{$ctx->{current_archive_type}}{section_title};
            my $title = $st->($ctx, $entries->[0]);
            defined $title ? $title : '';
        }
    }
}

sub _hdlr_archive_date_end {
    my($ctx) = @_;
    my $end = $ctx->{current_timestamp_end}
        or return $_[0]->error('You used an <$MTArchiveDateEnd$> outside of ' .
                               'a Daily, Weekly, or Monthly context.');
    $_[1]{ts} = $end;
    _hdlr_date(@_);
}

sub _hdlr_archive_link {
    ## Since this tag can be called from inside <MTCategories>,
    ## we need a way to map this tag to <$MTCategoryArchiveLink$>.
    return _hdlr_category_archive(@_) if $_[0]->{inside_mt_categories};

    my($ctx) = @_;
    my $blog = $ctx->stash('blog');
    my $entries = $ctx->stash('entries');
    return $ctx->error('You used an <$MTArchiveLink$> tag outside of the ' .
                       'proper context.')
        unless $entries && ref($entries) eq 'ARRAY';
    my $entry = $entries->[0];
    my $at = $_[1]->{archive_type} || $ctx->{current_archive_type};
    my $arch = $blog->archive_url;
    $arch .= '/' unless $arch =~ m!/$!;
    $arch . archive_file_for($entry, $blog, $at);
}

sub _hdlr_archive_count {
    my $e = $_[0]->stash('entries');
    $e && ref($e) eq 'ARRAY' ? scalar @$e : 0;
}

sub _hdlr_archive_category {
    ## Since this tag can be called from inside <MTCategories>,
    ## we need a way to map this tag to <$MTCategoryLabel$>.
    return _hdlr_category_label(@_) if $_[0]->{inside_mt_categories};

    my($ctx) = @_;
    my $en = $ctx->stash('entries');
    my $cat = $en && $en->[0] ? $en->[0]->category :
        $ctx->stash('archive_category');
    $cat ? $cat->label : '';
}

sub _hdlr_image_url { $_[0]->stash('image_url') }
sub _hdlr_image_width { $_[0]->stash('image_width') }
sub _hdlr_image_height { $_[0]->stash('image_height') }

sub _hdlr_calendar {
    my($ctx, $args) = @_;
    my $blog_id = $ctx->stash('blog_id');
    my($prefix);
    my @ts = offset_time_list(time, $blog_id);
    my $today = sprintf "%04d%02d", $ts[5]+1900, $ts[4]+1;
    if ($prefix = $args->{month}) {
        if ($prefix eq 'this') {
            my $ts = $ctx->{current_timestamp}
                or return $ctx->error(qq(You used <MTCalendar month="this">) .
                    "outside of a date context.");
            $prefix = substr $ts, 0, 6;
        } else {
            return $ctx->error("Invalid month format: must be YYYYMM")
                unless length($prefix) eq 6;
        }
    } else {
        $prefix = $today;
    }
    my($cat_name, $cat);
    if ($cat_name = $args->{category}) {
        require MT::Category;
        $cat = MT::Category->load({ label => $cat_name, blog_id => $blog_id })
            or return $ctx->error("No such category '$cat_name'");
    } else {
        $cat_name = '';    ## For looking up cached calendars.
    }
    my $uncompiled = $ctx->stash('uncompiled');
    my $r = MT::Request->instance;
    my $calendar_cache = $r->cache('calendar');
    unless ($calendar_cache) {
        $r->cache('calendar', $calendar_cache = { });
    }
    if (exists $calendar_cache->{$prefix . $cat_name} &&
        $calendar_cache->{$prefix . $cat_name}{'uc'} eq $uncompiled) {
        return $calendar_cache->{$prefix . $cat_name}{output};
    }
    $today .= sprintf "%02d", $ts[3];
    my($start, $end) = start_end_month($prefix);
    my($y, $m) = unpack 'A4A2', $prefix;
    my $days_in_month = days_in($m, $y);
    my $pad_start = wday_from_ts(0, 0, 0, 1, $m-1, $y-1900);
    my $pad_end = 6 - wday_from_ts(0, 0, 0, $days_in_month, $m-1, $y-1900);
    require MT::Entry;
    my $iter = MT::Entry->load_iter({ blog_id => $blog_id,
                                      created_on => [ $start, $end ],
                                      status => MT::Entry::RELEASE() },
        { range => { created_on => 1 },
          'sort' => 'created_on',
          direction => 'ascend', });
    my @left;
    my $res = '';
    my $tokens = $ctx->stash('tokens');
    my $builder = $ctx->stash('builder');
    my $iter_drained = 0;
    for my $day (1..$pad_start+$days_in_month+$pad_end) {
        my $is_padding =
            $day < $pad_start+1 || $day > $pad_start+$days_in_month;
        my($this_day, @entries) = ('');
        local($ctx->{__stash}{entries}, $ctx->{__stash}{calendar_day},
              $ctx->{current_timestamp});
        local $ctx->{__stash}{calendar_cell} = $day;
        unless ($is_padding) {
            $this_day = $prefix . sprintf("%02d", $day - $pad_start);
            my $no_loop = 0;
            if (@left) {
                if (substr($left[0]->created_on, 0, 8) eq $this_day) {
                    @entries = @left;
                    @left = ();
                } else {
                    $no_loop = 1;
                }
            }
            unless ($no_loop || $iter_drained) {
                while (my $entry = $iter->()) {
                    next unless !$cat || $entry->is_in_category($cat);
                    my $e_day = substr $entry->created_on, 0, 8;
                    push(@left, $entry), last
                        unless $e_day eq $this_day;
                    push @entries, $entry;
                }
                $iter_drained++ unless @left;
            }
            $ctx->{__stash}{entries} = \@entries;
            $ctx->{current_timestamp} = $this_day . '000000';
            $ctx->{__stash}{calendar_day} = $day - $pad_start;
        }
        defined(my $out = $builder->build($ctx, $tokens, {
            CalendarWeekHeader => ($day-1) % 7 == 0,
            CalendarWeekFooter => $day % 7 == 0,
            CalendarIfEntries => !$is_padding && scalar @entries,
            CalendarIfNoEntries => !$is_padding && !(scalar @entries),
            CalendarIfToday => ($today eq $this_day),
            CalendarIfBlank => $is_padding,
        })) or
            return $ctx->error( $builder->errstr );
        $res .= $out;
    }
    $calendar_cache->{$prefix . $cat_name} =
        { output => $res, 'uc' => $uncompiled };
    $res;
}

sub _hdlr_calendar_day {
    my $day = $_[0]->stash('calendar_day')
        or return $_[0]->error('You used <$MTCalendarDay$> outside of the ' .
                               'proper context.');
    $day;
}

sub _hdlr_calendar_cell_num {
    my $day = $_[0]->stash('calendar_cell')
        or return $_[0]->error('You used <$MTCalendarCellNumber$> outside of ' .
                               'the proper context.');
    $day;
}

sub _hdlr_categories {
    my($ctx, $args) = @_;
    my $blog_id = $ctx->stash('blog_id');
    require MT::Category;
    require MT::Placement;
    my $iter = MT::Category->load_iter({ blog_id => $blog_id },
        { 'sort' => 'label', direction => 'ascend' });
    my $res = '';
    my $builder = $ctx->stash('builder');
    my $tokens = $ctx->stash('tokens');
    ## In order for this handler to double as the handler for
    ## <MTArchiveList archive_type="Category">, it needs to support
    ## the <$MTArchiveLink$> and <$MTArchiveTitle$> tags
    local $ctx->{inside_mt_categories} = 1;
    while (my $cat = $iter->()) {
        local $ctx->{__stash}->{category} = $cat;
        my @entries = MT::Entry->load({ blog_id => $blog_id,
                                        status => MT::Entry::RELEASE() },
                        { 'join' => [ 'MT::Placement', 'entry_id',
                                    { category_id => $cat->id } ],
                          'sort' => 'created_on',
                          direction => 'descend', });
        next unless @entries || $args->{show_empty};
        local $ctx->{__stash}{entries} = \@entries;
        local $ctx->{__stash}{category_count} = scalar @entries;
        defined(my $out = $builder->build($ctx, $tokens))
            or return $ctx->error( $builder->errstr );
        $res .= $out;
    }
    $res;
}

sub _hdlr_category_id {
    my $cat = $_[0]->stash('category')
        or return $_[0]->error('You used <$MTCategoryID$> outside of the ' .
                               'proper context.');
    $cat->id;
}

sub _hdlr_category_label {
    my $cat = $_[0]->stash('category')
        or return $_[0]->error('You used <$MTCategoryLabel$> outside of the ' .
                               'proper context.');
    defined $cat->label ? $cat->label : '';
}

sub _hdlr_category_count {
    my($ctx) = @_;
    my $cat = $ctx->stash('category')
        or return $ctx->error('You used <$MTCategoryLabel$> outside of the ' .
                              'proper context.');
    my($count);
    unless ($count = $ctx->stash('category_count')) {
        require MT::Placement;
        $count = MT::Placement->count({ category_id => $cat->id });
    }
    $count;
}

sub _hdlr_category_archive {
    my $cat = $_[0]->stash('category')
        or return $_[0]->error('You used <$MTCategoryArchiveLink$> outside ' .
                               'of the proper context.');
    my $blog = $_[0]->stash('blog');
    my $at = $blog->archive_type;
    return $_[0]->error('<$MTCategoryArchiveLink$> can be used only if ' .
                        'you have enabled Category archives.')
            unless $at =~ /Category/;
    my $arch = $blog->archive_url;
    $arch .= '/' unless $arch =~ m!/$!;
    $arch . archive_file_for(undef, $blog, 'Category', $cat);
}

1;
