# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without# permission from www.movabletype.org.
#
# $Id: Image.pm,v 1.6 2002/02/26 08:10:18 btrott Exp $

package MT::Image;
use strict;

use Image::Magick;
use File::Basename;

use MT::ErrorHandler;
@MT::Image::ISA = qw( MT::ErrorHandler );

sub new {
    my $class = shift;
    my $image = bless {}, $class;
    $image->init(@_)
        or return $class->error( $image->errstr );
    $image;
}

sub init {
    my $image = shift;
    my %param = @_;
    my %arg = ();
    if (my $type = $param{Type}) {
        %arg = (magick => lc($type));
    }
    my $magick = $image->{magick} = Image::Magick->new(%arg);
    if (my $file = $param{Filename}) {
        my @info;
        unless (@info = $magick->Ping($file)) {
            return $image->error("'$file' is not a valid image.");
        }
        $image->{width} = $info[0];
        $image->{height} = $info[1];
        my $x = $magick->Read($file);
        return $image->error("Reading file '$file' failed: $x") if $x;
    } elsif (my $blob = $param{Data}) {
        my $x = $magick->BlobToImage($blob);
        return $image->error("Reading image failed: $x") if $x;
        ($image->{width}, $image->{height}) = $magick->Get('width', 'height');
    }
    $image;
}

sub scale {
    my $image = shift;
    my %param = @_;
    my($w, $h) = ($image->{width}, $image->{height});
    if (my $pct = $param{Scale}) {
        ($w, $h) = (int($w * $pct / 100), int($h * $pct / 100));
    } else {
        if ($param{Width} && $param{Height}) {
            ($w, $h) = ($param{Width}, $param{Height});
        } else {
            my $x = $param{Width} || $w;
            my $y = $param{Height} || $h;
            my $w_pct = $x / $w;
            my $h_pct = $y / $h;
            my $pct = $x ? $w_pct : $h_pct;
            ($w, $h) = (int($w * $pct), int($h * $pct));
        }
    }
    my($thumb);
    my $magick = $image->{magick};
    my $err = $magick->Scale(width => $w, height => $h);
    return $image->error("Scaling to ${w}x${h} failed: $err") if $err;
    ($magick->ImageToBlob, $w, $h);
}

1;
__END__

=head1 NAME

MT::Image - Movable Type image manipulation routines

=head1 SYNOPSIS

    use MT::Image;
    my $img = MT::Image->new( Filename => '/path/to/image.jpg' );
    my $blob = $img->scale( Width => 100 );

    open FH, ">thumb.jpg" or die $!;
    binmode FH;
    print FH $blob;
    close FH;

=head1 DESCRIPTION

I<MT::Image> contains image manipulation routines using the I<ImageMagick>
library and I<Image::Magick> Perl module. Currently all this is used for is
to create thumbnails from uploaded images.

=head1 USAGE

=head2 MT::Image->new(%arg)

Constructs a new I<MT::Image> object. Returns the new object on success; on
error, returns C<undef>, and the error message is in C<MT::Image-E<gt>errstr>.

I<%arg> can contain:

=over 4

=item * Filename

The path to an image to load. This will be passed to I<Image::Magick> for the
actual loading of the image.

=item * Data

The actual contents of an image, already loaded from a file, a database,
etc. These contents will be passed directly to I<Image::Magick>.

=item * Type

The image format of the data in I<Data>. Older versions of I<Image::Magick>
do not auto-magically determine the format from the image data itself, so you
will need to help them along by providing the I<Type> argument. The value
should be a valid I<Image::Magick> format name.

=back

=head2 $img->scale(%arg)

Creates a thumbnail from the image represented by I<$img>; on success, returns
the binary contents of the thumbnail image. On error, returns C<undef>, and
the error message is in C<$img-E<gt>errstr>.

I<%arg> can contain:

=over 4

=item * Width

=item * Height

The width and height of the final image, respectively. If you provide only one
of these arguments, the other dimension will be scaled appropriately. If you
provide neither, the image will be scaled to C<100%> of the original (that is,
the same size). If you provide both, the image will likely look rather
distorted.

=item * Scale

To be used instead of I<Width> and I<Height>; the value should be a percentage
(ie C<100> to return the original image without resizing) by which both the
width and height will be scaled equally.

=back

=head1 AUTHOR & COPYRIGHT

Please see the I<MT> manpage for author, copyright, and license information.

=cut
