# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: Local.pm,v 1.5 2002/03/18 20:42:18 btrott Exp $

package MT::FileMgr::Local;
use strict;

use MT::FileMgr;
@MT::FileMgr::Local::ISA = qw( MT::FileMgr );

use Symbol;
use Fcntl qw( :flock );

sub get_data {
    my $fmgr = shift;
    my($from, $type) = @_;
    my $fh = gensym();
    open $fh, $from
        or return $fmgr->error("Opening local file '$from' failed: $!");
    if ($type && $type eq 'upload') {
        binmode($fh);
    }
    my($data);
    { local $/; $data = <$fh>; }
    close $fh;
    $data;
}

## $type is either 'upload' or 'output'
sub put {
    my $fmgr = shift;
    my($from, $to, $type) = @_;
    my $rv;
    if (!is_handle($from)) {
        my $fh = gensym();
        open $fh, $from
            or return $fmgr->error("Opening local file '$from' failed: $!");
        $rv = _write_file($fmgr, $fh, $to, $type);
        close $fh;
    } else {
        $rv = _write_file($fmgr, $from, $to, $type);
    }
    $rv;
}

*put_data = \&_write_file;

sub _write_file {
    my $fmgr = shift;
    my($from, $to, $type) = @_;
    local *FH;
    my $umask = oct($type && $type eq 'upload' ?
        $fmgr->{cfg}->UploadUmask : $fmgr->{cfg}->HTMLUmask);
    my $old = umask($umask);
    open FH, ">$to"
        or return $fmgr->error("Opening '$to' failed: $!");
    if ($type && $type eq 'upload') {
        binmode(FH);
        binmode($from) if is_handle($from);
    }
    flock FH, LOCK_EX;
    seek FH, 0, 0;
    truncate FH, 0;
    my $bytes = 0;
    if (is_handle($from)) {
        while (my $len = read $from, my($block), 8192) {
            print FH $block;
            $bytes += $len;
        }
    } else {
        print FH $from;
        $bytes = length($from);
    }
    close FH;
    umask($old);
    $bytes;
}

sub exists { -e $_[1] }

sub mkpath {
    my $fmgr = shift;
    my($path) = @_;
    require File::Path;
    my $umask = oct $fmgr->{cfg}->DirUmask;
    my $old = umask($umask);
    File::Path::mkpath([$path], 0, 0777);
    umask($old);
    1;
}

sub rename {
    my $fmgr = shift;
    my($from, $to) = @_;
    rename $from, $to
       or return $fmgr->error("Renaming '$from' to '$to' failed: $!");
    1;
}

sub is_handle {
    my $f = shift;
    my $fd = ref($f) || ref(\$f) eq 'GLOB' ? fileno($f) : undef;
    $fd;
}

1;
