# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: Entry.pm,v 1.41 2002/03/18 20:42:48 btrott Exp $

package MT::Entry;
use strict;

use MT::Author;
use MT::Category;
use MT::Placement;
use MT::Comment;
use MT::Util qw( archive_file_for );

use MT::Object;
@MT::Entry::ISA = qw( MT::Object );
__PACKAGE__->install_properties({
    columns => [
        'id', 'blog_id', 'status', 'author_id', 'allow_comments',
        'title', 'excerpt', 'text', 'text_more', 'convert_breaks',
## Have to keep this around for use in mt-upgrade.cgi.
        'category_id',
    ],
    indexes => {
        blog_id => 1,
        status => 1,
        author_id => 1,
        created_on => 1,
        modified_on => 1,
    },
    audit => 1,
    datasource => 'entry',
    primary_key => 'id',
});

use constant HOLD    => 1;
use constant RELEASE => 2;
use constant REVIEW => 3;

use Exporter;
*import = \&Exporter::import;
use vars qw( @EXPORT_OK );
@EXPORT_OK = qw( HOLD RELEASE REVIEW );

sub status_text {
    my $s = $_[0];
    $s == HOLD ? "draft" :
        $s == RELEASE ? "publish" :
            $s == REVIEW ? "review" : '';
}

sub next {
    my $entry = shift;
    unless ($entry->{__next}) {
        $entry->{__next} = MT::Entry->load(
            { blog_id => $entry->blog_id,
              status => RELEASE },
            { limit => 1,
              'sort' => 'created_on',
              direction => 'ascend',
              start_val => $entry->created_on });
    }
    $entry->{__next};
}

sub previous {
    my $entry = shift;
    unless ($entry->{__previous}) {
        $entry->{__previous} = MT::Entry->load(
            { blog_id => $entry->blog_id,
              status => RELEASE },
            { limit => 1,
              'sort' => 'created_on',
              direction => 'descend',
              start_val => $entry->created_on });
    }
    $entry->{__previous};
}

sub author {
    my $entry = shift;
    unless ($entry->{__author}) {
        $entry->{__author} = MT::Author->load($entry->author_id);
    }
    $entry->{__author};
}

sub category {
    my $entry = shift;
    unless ($entry->{__category}) {
        $entry->{__category} = MT::Category->load(undef,
            { 'join' => [ 'MT::Placement', 'category_id',
                        { entry_id => $entry->id,
                          is_primary => 1 } ] },
        );
    }
    $entry->{__category};
}

sub categories {
    my $entry = shift;
    unless ($entry->{__categories}) {
        $entry->{__categories} = [ MT::Category->load(undef,
            { 'join' => [ 'MT::Placement', 'category_id',
                        { entry_id => $entry->id } ] },
        ) ];
        $entry->{__categories} = [ sort { $a->label cmp $b->label }
                                   @{ $entry->{__categories} } ];
    }
    $entry->{__categories};
}

sub is_in_category {
    my $entry = shift;
    my($cat) = @_;
    my $cats = $entry->categories;
    for my $c (@$cats) {
        return 1 if $c->id == $cat->id;
    }
    0;
}

sub comments {
    my $entry = shift;
    unless ($entry->{__comments}) {
        $entry->{__comments} = [ MT::Comment->load({
            entry_id => $entry->id
        }) ];
    }
    $entry->{__comments};
}

sub comment_count {
    my $entry = shift;
    unless ($entry->{__comment_count}) {
        $entry->{__comment_count} = MT::Comment->count({
            entry_id => $entry->id
        });
    }
    $entry->{__comment_count};
}

sub archive_file {
    my $entry = shift;
    my($at) = @_;
    my $blog_id = $entry->blog_id;
    require MT::Blog;
    my $blog = MT::Blog->load($blog_id) or
        return $entry->error("Load of blog '$blog_id' failed: " .
            MT::Blog->errstr);
    unless ($at) {
        $at = $blog->archive_type_preferred || $blog->archive_type;
        return '' if !$at || $at eq 'None';
        my %at = map { $_ => 1 } split /,/, $at;
        for my $tat (qw( Individual Daily Weekly Monthly Category )) {
            $at = $tat if $at{$tat};
        }
    }
    archive_file_for($entry, $blog, $at);
}

sub remove {
    my $entry = shift;
    my $comments = $entry->comments;
    for my $comment (@$comments) {
        $comment->remove;
    }
    require MT::Placement;
    my @place = MT::Placement->load({ entry_id => $entry->id });
    for my $place (@place) {
        $place->remove;
    }
    $entry->SUPER::remove;
}

1;
__END__

=head1 NAME

MT::Entry - Movable Type entry record

=head1 SYNOPSIS

    use MT::Entry;
    my $entry = MT::Entry->new;
    $entry->blog_id($blog->id);
    $entry->status(MT::Entry::RELEASE());
    $entry->author_id($author->id);
    $entry->title('My title');
    $entry->text('Some text');
    $entry->save
        or die $entry->errstr;

=head1 DESCRIPTION

An I<MT::Entry> object represents an entry in the Movable Type system. It
contains all of the metadata about the entry (author, status, category, etc.),
as well as the actual body (and extended body) of the entry.

=head1 USAGE

As a subclass of I<MT::Object>, I<MT::Entry> inherits all of the
data-management and -storage methods from that class; thus you should look
at the I<MT::Object> documentation for details about creating a new object,
loading an existing object, saving an object, etc.

The following methods are unique to the I<MT::Entry> interface:

=head2 $entry->next

Loads and returns the next entry, where "next" is defined as the next record
in ascending chronological order (the entry posted after the current entry).
entry I<$entry>).

Returns an I<MT::Entry> object representing this next entry; if there is not
a next entry, returns C<undef>.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->previous

Loads and returns the previous entry, where "previous" is defined as the
previous record in ascending chronological order (the entry posted before the
current entry I<$entry>).

Returns an I<MT::Entry> object representing this previous entry; if there is
not a next entry, returns C<undef>.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->author

Returns an I<MT::Author> object representing the author of the entry
I<$entry>. If the author record has been removed, returns C<undef>.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->category

Returns an I<MT::Category> object representing the primary category of the
entry I<$entry>. If a primary category has not been assigned, returns
C<undef>.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->categories

Returns a reference to an array of I<MT::Category> objects representing the
categories to which the entry I<$entry> has been assigned (both primary and
secondary categories). If the entry has not been assigned to any categories,
returns a reference to an empty array.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->is_in_category($cat)

Returns true if the entry I<$entry> has been assigned to entry I<$cat>, false
otherwise.

=head2 $entry->comments

Returns a reference to an array of I<MT::Comment> objects representing the
comments made on the entry I<$entry>. If no comments have been made on the
entry, returns a reference to an empty array.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->comment_count

Returns the number of comments made on this entry.

Caches the return value internally so that subsequent calls will not have to
re-query the database.

=head2 $entry->archive_file([ $archive_type ])

Returns the name of/path to the archive file for the entry I<$entry>. If
I<$archive_type> is not specified, and you are using multiple archive types
for your blog, the path is created from the preferred archive type that you
have selected. If I<$archive_type> is specified, it should be one of the
following values: C<Individual>, C<Daily>, C<Weekly>, C<Monthly>, and
C<Category>.

=head1 DATA ACCESS METHODS

The I<MT::Entry> object holds the following pieces of data. These fields can
be accessed and set using the standard data access methods described in the
I<MT::Object> documentation.

=over 4

=item * id

The numeric ID of the entry.

=item * blog_id

The numeric ID of the blog in which this entry has been posted.

=item * author_id

The numeric ID of the author who posted this entry.

=item * status

The status of the entry, either Publish (C<2>) or Draft (C<1>).

=item * allow_comments

A boolean flag specifying whether comments are allowed on this entry. This
setting determines whether C<E<lt>MTEntryIfAllowCommentsE<gt>> containers are
displayed for this entry.

=item * convert_breaks

A boolean flag specifying whether line and paragraph breaks should be converted
when rebuilding this entry.

=item * title

The title of the entry.

=item * excerpt

The excerpt of the entry.

=item * text

The main body text of the entry.

=item * text_more

The extended body text of the entry.

=item * created_on

The timestamp denoting when the entry record was created, in the format
C<YYYYMMDDHHMMSS>. Note that the timestamp has already been adjusted for the
selected timezone.

=item * modified_on

The timestamp denoting when the entry record was last modified, in the
format C<YYYYMMDDHHMMSS>. Note that the timestamp has already been adjusted
for the selected timezone.

=back

=head1 DATA LOOKUP

In addition to numeric ID lookup, you can look up or sort records by any
combination of the following fields. See the I<load> documentation in
I<MT::Object> for more information.

=over 4

=item * blog_id

=item * status

=item * author_id

=item * created_on

=item * modified_on

=back

=head1 NOTES

=over 4

=item *

When you remove an entry using I<MT::Entry::remove>, in addition to removing
the entry record, all of the comments and placements (I<MT::Comment> and
I<MT::Placement> records, respectively) for this entry will also be removed.

=back

=head1 AUTHOR & COPYRIGHTS

Please see the I<MT> manpage for author, copyright, and license information.

=cut
