# Copyright 2001, 2002 Benjamin Trott. This code cannot be redistributed without
# permission from www.movabletype.org.
#
# $Id: ConfigMgr.pm,v 1.36 2002/02/18 08:29:56 btrott Exp $

package MT::ConfigMgr;
use strict;

use MT::ErrorHandler;
@MT::ConfigMgr::ISA = qw( MT::ErrorHandler );

{
    my $mgr;
    sub instance {
        return $mgr if $mgr;
        $mgr = __PACKAGE__->new;
    }
}

sub new {
    my $mgr = bless { __var => { } }, $_[0];
    $mgr->init;
    $mgr;
}

sub init {
    my $mgr = shift;
    $mgr->define('DataSource', Default => './db');
    $mgr->define('DBUser');
    $mgr->define('DBPassword');
    $mgr->define('TemplatePath', Default => './tmpl');
    $mgr->define('ImportPath', Default => './import');
    $mgr->define('ObjectDriver', Default => 'DBM');
    $mgr->define('Serializer', Default => 'MT');
    $mgr->define('SendMailPath', Default => '/usr/lib/sendmail');
    $mgr->define('TimeOffset', Default => 0);
    $mgr->define('StaticWebPath', Default => '');
    $mgr->define('CGIPath', Default => '/cgi-bin/');
    $mgr->define('MailTransfer', Default => 'sendmail');
    $mgr->define('SMTPServer', Default => 'localhost');
    $mgr->define('WeblogsPingURL', Default => 'http://rpc.weblogs.com/RPC2');
    $mgr->define('MTPingURL', Default => 'http://www.movabletype.org/update/');
    $mgr->define('CGIMaxUpload', Default => 1_000_000);
    $mgr->define('DBUmask', Default => '0111');
    $mgr->define('HTMLUmask', Default => '0111');
    $mgr->define('UploadUmask', Default => '0111');
    $mgr->define('DirUmask', Default => '0000');
    $mgr->define('NoTempFiles', Default => 0);
    $mgr->define('TempDir', Default => '/tmp');
    $mgr->define('EntriesPerRebuild', Default => 40);
    $mgr->define('UseNFSSafeLocking', Default => 0);
}

sub define {
    my $mgr = shift;
    my($var, %param) = @_;
    $mgr->{__var}->{$var} = undef;
    if (exists $param{Default}) {
        $mgr->{__var}->{$var} = $param{Default};
    }
}

sub get { $_[0]->{__var}->{ $_[1] } }
sub set {
    my $mgr = shift;
    my($var, $val) = @_;
    $mgr->{__var}->{$var} = $val;
}

sub read_config {
    my $class = shift;
    my($cfg_file) = @_;
    my $mgr = $class->instance;
    local(*FH, $_, $/);
    $/ = "\n";
    open FH, $cfg_file or
        return $class->error("Error opening file '$cfg_file': $!");
    while (<FH>) {
        chomp;
        next if !/\S/ || /^#/;
        my($var, $val) = $_ =~ /^\s*(\S+)\s+(.+)$/;
        $val =~ s/\s*$//;
        next unless $var && $val;
        return $class->error("$cfg_file:$.: variable '$var' not defined")
            unless exists $mgr->{__var}->{$var};
        $mgr->set($var, $val);
    }
    close FH;
    1;
}

sub DESTROY { }

use vars qw( $AUTOLOAD );
sub AUTOLOAD {
    my $mgr = $_[0];
    (my $var = $AUTOLOAD) =~ s!.+::!!;
    return $mgr->error("No such config variable '$var'")
        unless exists $mgr->{__var}->{$var};
    no strict 'refs';
    *$AUTOLOAD = sub {
        my $mgr = shift;
        @_ ? $mgr->set($var, $_[0]) : $mgr->get($var);
    };
    goto &$AUTOLOAD;
}

1;
__END__

=head1 NAME

MT::ConfigMgr - Movable Type configuration manager

=head1 SYNOPSIS

    use MT::ConfigMgr;
    my $cfg = MT::ConfigMgr->instance;

    $cfg->read_config('/path/to/mt.cfg')
        or die $cfg->errstr;

=head1 DESCRIPTION

I<MT::ConfigMgr> is a singleton class that manages the Movable Type
configuration file (F<mt.cfg>), allowing access to the config directives
contained therin.

=head1 USAGE

=head2 MT::ConfigMgr->instance

Returns the singleton I<MT::ConfigMgr> object. Note that when you want the
object, you should always call I<instance>, never I<new>; I<new> will construct
a B<new> I<MT::ConfigMgr> object, and that isn't what you want. You want the
object that has already been initialized with the contents of F<mt.cfg>. This
initialization is done by I<MT::new>.

=head2 $cfg->read_config($file)

Reads the config file at the path I<$file> and initializes the I<$cfg> object
with the directives in that file. Returns true on success, C<undef> otherwise;
if an error occurs you can obtain the error message with C<$cfg-E<gt>errstr>.

=head2 $cfg->define($directive [, %arg ])

Defines the directive I<$directive> as a valid configuration directive; you
must define new configuration directives B<before> you read the configuration
file, or else the read will fail.

=head1 CONFIGURATION DIRECTIVES

The following configuration directives are allowed in F<mt.cfg>. To get the
value of a directive, treat it as a method that you are calling on the
I<$cfg> object. For example:

    $cfg->CGIPath

To set the value of a directive, do the same as the above, but pass in a value
to the method:

    $cfg->CGIPath('http://www.foo.com/mt/');

The following are the allowed configuration directives:

=over 4

=item * CGIPath

Movable Type uses the I<CGIPath> setting to construct links back to CGI
scripts; for example, the MT tag E<lt>$MTCGIPath$E<gt> is substituted with the
value of the I<CGIPath> setting. You will need to change this value when you
first install MT; instructions for doing so are in the Installation
Instructions, in I<INSTALLING THE MOVABLE TYPE APPLICATION CODE>, Step 3.

Default value: none

Example:

    CGIPath http://www.your-site.com/path/to/mt/

=item * DataSource

The filesystem path to the F<db> directory, where your MT database files are
stored. You will probably need to change this when you first install MT;
instructions for doing so are in the Installation Instructions, in
I<INSTALLING THE MOVABLE TYPE APPLICATION CODE>, Step 8.

Default value: F<./db>

Example:

    DataSource ./db

=item * StaticWebPath

If you place all of your MT files in a cgi-bin directory, you will need to
situate the static files (F<docs>, F<images>, F<styles.css>) elsewhere, so
that the webserver will not try to execute them. The I<TROUBLESHOOTING>
section of the manual has more information.

Default value: none

Example:

    StaticWebPath /path/to/static-files/

=item * TemplatePath

The filesystem path to the F<tmpl> directory, which contains the front-end
templates used by the Movable Type application.

Default value: F<./tmpl>

Example:

    TemplatePath ./tmpl

=item * MailTransfer

If you would rather use SMTP than sendmail, you should set the MailTransfer
config setting to 'smtp' (as below). Possible values for MailTransfer are:
C<smtp>, C<sendmail>, and C<debug> (which just writes out mail messages to
STDERR, for debugging purposes).

Default value: C<sendmail>

Example:

    MailTransfer smtp

=item * SendMailPath

By default, Movable Type looks for sendmail in three locations:
F</usr/lib/sendmail>, F</usr/sbin/sendmail>, and F</usr/ucblib/sendmail>. If
your sendmail is in a different location, you can adjust the SendMailPath
configuration setting.

Default value: any of the above

Example:

    SendMailPath /usr/sbin/sendmail

=item * SMTPServer

The address of your SMTP server, to be used along with C<MailTransfer smtp>.

Default value: C<localhost>

Example:

    SMTPServer smtp.your-site.com

=item * NoTempFiles

By default, when writing to an output file (for example, one of your index or
archive pages), Movable Type will first write the data to a temp file, then
rename that temp file. In the case that the process writing the data dies
unexpectedly, this prevents the pages on your site from being erased. If you
do not like this behavior (because it requires you to set directory
permissions too liberally, for example), you can use C<NoTempFiles> to turn
it off.

Default value: C<0>

Example:

    NoTempFiles 1

=item * WeblogsPingURL

The URL used to send the XML-RPC I<weblogs.com> ping.

Default value: C<http://rpc.weblogs.com/RPC2>

Example:

    WeblogsPingURL http://some.alternate.weblogs.com.server/path/

=item * MTPingURL

The URL used to send the XML-RPC ping to I<movabletype.org> (if you have a
Recently Updated Key).

Default value: C<http://www.movabletype.org/update/>

Example:

    MTPingURL http://some.alternate.movabletype.org.server/path/

=item * CGIMaxUpload

When uploading files through Movable Type's upload mechanism, a ceiling is put
on the size of the files that can be uploaded to prevent denial-of-service
attacks.

Default value: C<1_000_000> (1MB)

Example:

    CGIMaxUpload 500_000

=item * DBUmask

=item * HTMLUmask

=item * UploadUmask

=item * DirUmask

When creating files and directories, Movable Type uses umask settings to
control the permissions set on the files.

Default values: C<0111> (DBUmask, HTMLUmask, UploadUmask), 0000 (DirUmask)

Example:

    DBUmask 0022

=item * TempDir

When processing uploaded files, if Movable Type notices that the file you
uploaded already exists, it will allow you to overwrite the original file, by
first asking for your confirmation. To do this, MT needs to write the uploaded
data to a temporary file. That temporary file is stored in the directory
specified by the TempDir setting.

Default value: C</tmp>

Example:

    TempDir /tmp/

=item * EntriesPerRebuild

When rebuilding individual archives, Movable Type splits up the rebuilding
process into segments, where each segment consists of rebuilding N entries.
The default value for N is 40, so by default, MT will rebuild 40 entries at
a time, then move on to the next 40, etc. You can change that value globally
here; for example, if you have a very stable server, you might wish to just
get it all done with in one batch.

Default value: C<40>

Example:

    EntriesPerRebuild 100

=item * ImportPath

The filesystem path to the F<import> directory, which is used when importing
entries and comments into the system--F<import> is the directory where the
files to be imported are placed.

Default value: F<./import>

Example:

    ImportPath ./import

=item * UseNFSSafeLocking

By default Movable Type uses Perl's I<flock()> function to lock your databases
while reading and writing. On systems using NFS-mounted directories, however,
Perl's I<flock()> may fail, unless the I<perl> executable has been built to
use I<fnctl(2)> instead of I<flock(2)>; and even then, it is not certain that
the locking will truly work.

Thus, if you have problems running Movable Type on systems using NFS, you can
use the I<UseNFSSafeLocking> directive to use simpler file locking that will
work over NFS.

Default value: C<0> (don't use NFS-safe locking)

Example:

    UseNFSSafeLocking 1

=back

=head1 AUTHOR & COPYRIGHT

Please see the I<MT> manpage for author, copyright, and license information.

=cut
