/*
  jugtailConf.c, written by Rhett "Jonzy" Jones 

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
 * Description:	Reads information from the jugtailConf.conf file.
 *		The format of this file is such that any text following
 *		the pound sign "#" is a comment, blank lines are ignored,
 *		and contains a variable name followed by an assignment.
 *		A maximum of 1024 character per line is allowed, as
 *		well as embedded quotes, quoted and non-quoted strings.
 *		A sample file follows:
 *		# <- Start of a comment
 *		hostname = some.machine.name  # Not quoted
 *		hostname "some.machine.name"  # Does the same thing
 *		var = this one and that one   # var now gets [this]
 *		var = "This one and that one" # var gets [the quoted string]
 *		var "A \"quoted\" word"       # var gets [A "quoted" word]
 *
 *		V: veronica token             # V gets [veronica token]
 *
 * Notes:	When adding variables for use by ReadConfFile() you need
 *		to do the following steps:
 *		Add the #define variable_number
 *		Add the #define variable_name
 *		Add the declaration of the variables
 *		Add the else if (!strcmp(...)) in ParseLine()
 *		Add the case variable_number in ReadConfFile()
 */

#include "stdinc.h"
#include "jugtail.h"

int ReadLine (FILE *fp, char *line, int lineSize);
static int HandleVeronicaToken (int token, char *s, char **varStr);
static int ParseLine (char *line, char **varStr);
char *MakeStr (char *oldStr, char *newStr);
int MakeInt (char *str);
void NullifyDisallowList (void);
int ReadConfFile (char *fileName);

#define BUFSIZE		1024

/* The variable-number to handle. */
#define COMMENTLINE		 -1
#define HOSTNAME		  1
#define JUGTAILHELP		  2
#define ERRORHOST		  3
#define PORT2USE		  4
#define MAXPROCS		  5
#define DELIMITERS		  6
#define MAXITEMS2RETURN		  7
#define DEFAULTBOOLOP		  8
#define JUGTAILHELPTITL		  9
#define USAGERROR		 10
#define WILDCARDERR		 11
#define TOMNYPRCSERR		 12
#define NOFORKERR		 13
#define GTSTRERR		 14
#define READERR			 15
#define TMPFILENAME		 16
#define CATCOMMAND		 17
#define SORTCOMMAND		 18
#define TOUCHCOMMAND		 19
#define DISALLOWCMD		 20

#ifdef VERSION2
#define EMAIL			 21
#define HOSTSTREE		 22
#endif

/* These aren't used by jugtail but exist in the veronica.ctl file. */
#define USERAGENTCMD		100
#define VERONICAINDEXCMD	101
#define VERONCIADATAFILECMD	102
#define VERONICADATAFILEPREPCMD	103
#define GOPHEREOFCOMMAND	200

/* The name of variable_name to handle. */
#define COMMENTSTR		""
#define HOSTNAMESTR		"hostname"
#define JUGTAILHELPSTR		"jugtailhelp"
#define ERRORHOSTSTR		"errorhost"
#define PORT2USESTR		"port2use"
#define MAXPROCSSTR		"maxprocs"
#define DELIMITERSSTR		"delimiters"
#define MAXITEMS2RETURNSTR	"maxitems2return"
#define DEFAULTBOOLOPSTR	"defaultboolop"
#define JUGTAILHELPTITLSTR	"jugtailhelptitl"
#define USAGERRORSTR		"usagerror"
#define WILDCARDERRSTR		"wildcarderr"
#define TOMNYPRCSERRSTR		"tomnyprcserr"
#define NOFORKERRSTR		"noforkerr"
#define GTSTRERRSTR		"gtstrerr"
#define READERRSTR		"readerr"
#define TMPFILENAMESTR		"tmpfilename"
#define CATCOMMANDSTR		"catcommand"
#define SORTCOMMANDSTR		"sortcommand"
#define TOUCHCOMMANDSTR		"touchcommand"
#define DISALLOWSTR		"Disallow:"

#ifdef VERSION2
#define EMAILSTR		"email"
#define HOSTSTREESTR		"hoststree"
#endif

/* These aren't used by jugtail but exist in the veronica.ctl file. */
#define USERAGENTSTR		"User-agent:"
#define VERONICAINDEXSTR	"veronicaindex:"
#define VERONCIADATAFILESTR	"veronica-datafile:"
#define VERONICADATAFILEPREPSTR	"veronica-datafile-prep:"
#define GOPHEREOFSTR		"."


/* These variables are defined in "jugtailConf.c". */
extern int debug;
extern FILE *rptPtr;
extern int ValidVariableCheck ();
extern void PrintJugtailConf ();
extern List *CreateNode (), *InsertNode (), *RemoveNode ();

/* These are defined in "path.c". */
extern List *disallowHead, *disallowTail;
extern char *vctlHost, *vctlPort;
extern void PrintTheList ();

#ifdef VERSION2
extern char *email, *hoststree;
#endif

/*****************************************************************************
 * ReadLine reads from the file 'fp' a maximum of 'lineSize' characters into
 * the string 'line' upto but not including end-of-file or end-of-line.
 * This routine returns the charcter read in.
 * Note:  If more them 'lineSize' character are found to exist on a line
 * a warning message is printed only once.
 ****************************************************************************/
int
ReadLine (FILE *fp, char *line, int lineSize)
     /* fp: Pointer to the file we are reading.
	line: Buffer for the line we read.
	lineSize: The size of our line buffer.  */
{
  int i, c;			/* A loop counter and a character. */

  /* Read the charcters up to the limit, eof-of-file or end-of-line. */
  for (i = 0; i < lineSize && (c = fgetc (fp)) != EOF && c != '\n'; i++)
    line[i] = (char) c;
  line[i] = '\0';

  if (debug)
    fprintf (rptPtr, "%s\n", line);

  if (i >= lineSize)		/* Yipes, tell the user their file is out to lunch. */
    {
      static int messagePosted = 0;	/* Have we told them yet? */

      if (!messagePosted)
	{
	  messagePosted = 1;
	  fprintf (rptPtr, "warning: line length exceeds %d buffer.\n",
		   lineSize + 1);
	  fprintf (rptPtr,
	   "         All charcters beyond this limit are discarded.\n");
	}
      while ((c = fgetc (fp)) != EOF && c != '\n');
    }
  else if (i && c == EOF)
    fprintf (rptPtr, "warning: No EOLN found, discarding [%s]\n", line);

  return (c);

}				/* ReadLine */

/*****************************************************************************
 * HandleVeronicaToken returns the veronica token of the variable
 * encountered. This routine also checks for a pound sign '#' in the string
 * 's'.  If a pound sign is encountered it is iterpreted as a comment and
 * all white space preceeding the pound sign is set to the null character.
 * The parameter 'value' ends up pointing to 's'.  This routine was written
 * to support the "veronica.ctl" commands which do not require double
 * quotes around a string which contains white space.
 ****************************************************************************/
static int
HandleVeronicaToken (int token, char *s, char **varStr)
     /* token: The token type.
	s: The string to put into 'varStr'.
	varStr: The value for the token 'token'. */
{
  char *pound;			/* The pound position in 's'. */

  /* If there is a pound null it out. */
  if ((pound = strchr (s, '#')))
    *pound = '\0';

  /* Null out all trailing isspace() characters. */
  for (pound = s + strlen (s) - 1; pound != s && isspace (*pound); pound--)
    *pound = '\0';

  *varStr = s;
  return (token);

}				/* HandleVeronicaToken */

/*****************************************************************************
 * ParseLine returns what type of variable we have encountered within the
 * jugtailConf.conf or veronica.ctl file.  This routine assigns to 'varStr' the
 * contents of what the encountered variable should get assigned, as well as
 * takes care of embedded quotes.  The "Get the variable contents" is kind of
 * nitty gritty but it handles quoted strings, embedded quotes, and
 * non-quoted strings.  Note: if the token is a veronica.ctl token the
 * routine HandleVeronicaToken() parses the value for the given token.
 ****************************************************************************/
static int
ParseLine (char *line, char **varStr)
     /* line: The line we are parsing.
        varStr: The string to assign the variable. */
{
  int varName = 0,		/* What variable did we encounter? */
    i, j,			/* Loop and index counters. */
    comment,			/* Are we doing a comment? */
    done,			/* Are we done with string contents? */
    escape,			/* Have we encountered an escape character? */
    quote;			/* Have we seen a quote? */
  char *s;			/* Place holder into 'line'. */

  /* Skip any leading whitespace. */
  while (isspace (*line))
    line++;

  if (!*line || *line == '#')	/* Comment line so bail out. */
    return (COMMENTLINE);

  for (s = line; *line && !isspace (*line) && *line != '=' && *line != '#';
       line++)
    /* Get the name of the variable. */ ;

  /* See if we got the variable name. */
  if (isspace (*line) || *line == '=')
    {
      *line++ = '\0';		/* 's' now contains the variable name. */

      /* Find out which variable we are looking at. */
      if (!strcmp (s, COMMENTSTR))
	return (COMMENTLINE);
      else if (!strcmp (s, HOSTNAMESTR))
	varName = HOSTNAME;
      else if (!strcmp (s, JUGTAILHELPSTR))
	varName = JUGTAILHELP;
      else if (!strcmp (s, ERRORHOSTSTR))
	varName = ERRORHOST;
      else if (!strcmp (s, PORT2USESTR))
	varName = PORT2USE;
      else if (!strcmp (s, MAXPROCSSTR))
	varName = MAXPROCS;
      else if (!strcmp (s, DELIMITERSSTR))
	varName = DELIMITERS;
      else if (!strcmp (s, MAXITEMS2RETURNSTR))
	varName = MAXITEMS2RETURN;
      else if (!strcmp (s, DEFAULTBOOLOPSTR))
	varName = DEFAULTBOOLOP;
      else if (!strcmp (s, JUGTAILHELPTITLSTR))
	varName = JUGTAILHELPTITL;
      else if (!strcmp (s, USAGERRORSTR))
	varName = USAGERROR;
      else if (!strcmp (s, WILDCARDERRSTR))
	varName = WILDCARDERR;
      else if (!strcmp (s, TOMNYPRCSERRSTR))
	varName = TOMNYPRCSERR;
      else if (!strcmp (s, NOFORKERRSTR))
	varName = NOFORKERR;
      else if (!strcmp (s, GTSTRERRSTR))
	varName = GTSTRERR;
      else if (!strcmp (s, READERRSTR))
	varName = READERR;
      else if (!strcmp (s, TMPFILENAMESTR))
	varName = TMPFILENAME;
      else if (!strcmp (s, CATCOMMANDSTR))
	varName = CATCOMMAND;
      else if (!strcmp (s, SORTCOMMANDSTR))
	varName = SORTCOMMAND;
      else if (!strcmp (s, TOUCHCOMMANDSTR))
	varName = TOUCHCOMMAND;
      else if (!strcmp (s, DISALLOWSTR))
	return (HandleVeronicaToken (DISALLOWCMD, line, varStr));
#ifdef VERSION2
      else if (!strcmp (s, EMAILSTR))
	varName = EMAIL;
      else if (!strcmp (s, HOSTSTREESTR))
	varName = HOSTSTREE;
#endif
      else if (!strcmp (s, USERAGENTSTR))
	return (HandleVeronicaToken (USERAGENTCMD, line, varStr));
      else if (!strcmp (s, VERONICAINDEXSTR))
	return (HandleVeronicaToken (VERONICAINDEXCMD, line, varStr));
      else if (!strcmp (s, VERONCIADATAFILESTR))
	return (HandleVeronicaToken (VERONCIADATAFILECMD, line, varStr));
      else if (!strcmp (s, VERONICADATAFILEPREPSTR))
	return (HandleVeronicaToken (VERONICADATAFILEPREPCMD, line, varStr));
      else if (!strcmp (s, GOPHEREOFSTR))
	return (HandleVeronicaToken (GOPHEREOFCOMMAND, line, varStr));
      else			/* Who the hell knows? */
	{
	  *varStr = line;
	  return (0);
	}

      /* Skip any white space or equal signs. */
      while (isspace (*line) || *line == '=')
	line++;

      /* Get the variable contents. */
      for (comment = done = escape = quote = i = j = 0, s = line;
	   !done && line[i] && !comment; i++)
	{
	  if (line[i] == '#' && !quote)
	    comment++;
	  if (line[i] == '\\')
	    {
	      escape++;
	      j++;
	    }
	  else if (!escape && !quote && isspace (line[i]))
	    {
	      done++;
	      j++;
	    }
	  else if (line[i] == '\"')
	    if (escape)
	      {
		escape = 0;
		line[i - j] = line[i];
	      }
	    else
	      {
		if (quote)
		  done++;
		else
		  quote++;
		j++;
	      }
	  else if (escape
		   && (line[i] == 'n' || line[i] == 'r' || line[i] == 't'))
	    {
	      escape = 0;
	      switch (line[i])
		{
		case 'n':
		  line[i - j] = '\n';
		  break;
		case 'r':
		  line[i - j] = '\r';
		  break;
		case 't':
		  line[i - j] = '\t';
		  break;
		}
	    }
	  else
	    line[i - j] = line[i];
	}
      line[i - j] = '\0';	/* 's' now contains the variable contents. */

      *varStr = s;
    }
  else				/* Hmm, somthing strange is going on. */
    {
      *varStr = s;
      return (0);
    }

  return (varName);

}				/* ParseLine */

/*****************************************************************************
 * MakeStr returns a copy of 'newStr' acquired in malloc space.  This
 * routine also frees up any memory occupied by 'oldStr' prior to the copy.
 ****************************************************************************/
char *
MakeStr (char *oldStr, char *newStr)
     /* oldStr: The string to copy into.
        newStr: The string we are copying. */
{

  if (oldStr)			/* Free up the memory. */
    free (oldStr);

  /* Get memory and copy the sucker. */
  if ((oldStr = malloc (strlen (newStr) + 1)))
    strcpy (oldStr, newStr);

  return (oldStr);

}				/* MakeStr */

/*****************************************************************************
 * MakeInt returns the integer representation of 'str'.
 ****************************************************************************/
int
MakeInt (char *str)
     /* str: The string with the integer. */
{
  int i;			/* The value to return. */

  for (i = 0; *str; str++)
    if (isdigit (*str))
      i = i * 10 + *str - '0';
  return (i);

}				/* MakeInt */

/*****************************************************************************
 * NullifyDisallowList simply frees up the memory occupied by the
 * disallow list and sets the head and tail of the list to NIL.
 * This routine was written to support the "Disallow:" directive in the
 * 'jugtailConf.conf' file and preventing jugtail from overdoseing on too
 * much memory when running as a search engine and receiving all those
 * SIGHUP signals.
 ****************************************************************************/
void
NullifyDisallowList (void)
{
  List *l;			/* A temp node in the list. */

  if (debug)
    PrintTheList ();

  for (l = disallowTail; l; l = disallowTail)
    disallowTail = RemoveNode (l);
  disallowHead = disallowTail;

  if (debug)
    PrintTheList ();

}				/* NullifyDisallowList */

/*****************************************************************************
 * ReadConfFile reads from the file 'fileName', which is the jugtailConf.conf
 * file, and initializes the variables stored within the file.  This routine
 * returns true if the file exists and false otherwise.
 ****************************************************************************/
int
ReadConfFile (char *fileName)
     /* fileName: Name of the jugtailConf.conf file. */
{
  FILE *fp;			/* Pointer to the conf file. */
  char line[BUFSIZE],		/* A line from the file. */
   *varStr;			/* The string variable from the file. */
  int command;			/* The command token ParseLine() found. */

  if (debug)
    fprintf (rptPtr, "In ReadConfFile(%s)\n", fileName);

  if ((fp = fopen (fileName, "r")))
    {
      while (ReadLine (fp, line, BUFSIZE - 1) != EOF)
	switch ((command = ParseLine (line, &varStr)))
	  {
	  case COMMENTLINE:
	    break;
	  case HOSTNAME:
	    hostname = MakeStr (hostname, varStr);
	    break;
	  case JUGTAILHELP:
	    jugtailhelp = MakeStr (jugtailhelp, varStr);
	    break;
	  case ERRORHOST:
	    errorhost = MakeStr (errorhost, varStr);
	    break;
	  case PORT2USE:
	    port2use = MakeInt (varStr);
	    break;
	  case MAXPROCS:
	    maxprocs = MakeInt (varStr);
	    break;
	  case DELIMITERS:
	    delimiters = MakeStr (jugtailhelp, varStr);
	    break;
	  case MAXITEMS2RETURN:
	    maxitems2return = MakeInt (varStr);
	    break;
	  case DEFAULTBOOLOP:
	    defaultboolop = MakeStr (defaultboolop, varStr);
	    break;
	  case JUGTAILHELPTITL:
	    jugtailhelptitl = MakeStr (jugtailhelptitl, varStr);
	    break;
	  case USAGERROR:
	    usagerror = MakeStr (usagerror, varStr);
	    break;
	  case WILDCARDERR:
	    wildcarderr = MakeStr (wildcarderr, varStr);
	    break;
	  case TOMNYPRCSERR:
	    tomnyprcserr = MakeStr (tomnyprcserr, varStr);
	    break;
	  case NOFORKERR:
	    noforkerr = MakeStr (noforkerr, varStr);
	    break;
	  case GTSTRERR:
	    gtstrerr = MakeStr (gtstrerr, varStr);
	    break;
	  case READERR:
	    readerr = MakeStr (readerr, varStr);
	    break;
	  case TMPFILENAME:
	    tmpfilename = MakeStr (tmpfilename, varStr);
	    break;
	  case CATCOMMAND:
	    catcommand = MakeStr (catcommand, varStr);
	    break;
	  case SORTCOMMAND:
	    sortcommand = MakeStr (sortcommand, varStr);
	    break;
	  case TOUCHCOMMAND:
	    touchcommand = MakeStr (touchcommand, varStr);
	    break;
	  case DISALLOWCMD:
	    if (!disallowHead)
	      disallowHead = disallowTail =
		CreateNode (varStr, vctlHost, vctlPort);
	    else
	      disallowTail =
		InsertNode (disallowTail,
			    CreateNode (varStr, vctlHost, vctlPort));
	    break;
#ifdef VERSION2
	  case EMAIL:
	    email = MakeStr (email, varStr);
	    break;
	  case HOSTSTREE:
	    hoststree = MakeStr (hoststree, varStr);
	    break;
#endif
	  case USERAGENTCMD:
	  case VERONICAINDEXCMD:
	  case VERONCIADATAFILECMD:
	  case VERONICADATAFILEPREPCMD:
	  case GOPHEREOFCOMMAND:
	    if (debug)
	      {
		fprintf (rptPtr, "Found veronica.ctl directive ");
		switch (command)
		  {
		  case USERAGENTCMD:
		    fprintf (rptPtr, "[%s]", USERAGENTSTR);
		    break;
		  case VERONICAINDEXCMD:
		    fprintf (rptPtr, "[%s]", VERONICAINDEXSTR);
		    break;
		  case VERONCIADATAFILECMD:
		    fprintf (rptPtr, "[%s]", VERONCIADATAFILESTR);
		    break;
		  case VERONICADATAFILEPREPCMD:
		    fprintf (rptPtr, "[%s]", VERONICADATAFILEPREPSTR);
		    break;
		  case GOPHEREOFCOMMAND:
		    fprintf (rptPtr, "(gopher end-of-file ) [%s]",
				    GOPHEREOFSTR);
		    break;
		  default:
		    break;
		  }
		fprintf (rptPtr, " - ignoring.\n\tValue is: [%s]\n", varStr);
	      }
	    break;
	  default:
	    fprintf (rptPtr,
		"warning: ParseLine found unknown directive [%s]\n", varStr);
	    break;
	  }
      fclose (fp);
    }
  else
    fprintf (rptPtr, "Could not open %s.  Using the default values.\n",
	     fileName);

  if (debug)
    PrintJugtailConf ();

  return (ValidVariableCheck ());

}				/* ReadConfFile */
