/*
  searchCmnds.c, written by Rhett "Jonzy" Jones 

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
 * Description:	Either builds an index into a datafile, where the datafile
 *	contains lines of gopher menu items (via jugtail), or performs
 *	boolean searches on words from the display string in the
 *	datafile as read from the resultant index file.
 *
 *	When building the index we read from the datafile and for
 *	each line we acquire the current line position in the file via
 *	ftell(), and extract the display string excluding the first
 *	character which is the item type, and break the display string
 *	into words.  We then dump the word followed by a tab and the
 *	position to a temporary file.  We then read from the temporary
 *	file and build a binary tree containing words and a list of
 *	postions, which is the line the word came from.  No word is
 *	duplicated in the tree.  And finaly dump the binary tree to
 *	a file with the following format:
 *	dataFileNameIndexWasBuiltFrom	numberOfnodes
 *	lenWord0	word0	0	1	3
 *	lenWord1	word1	2	5
 *	...
 *	lenThisWord	word_numberOfnodes-1	m	n
 *
 *	As of April 4, 1993, a hash table and an index table get built
 *	where the hash table as the following format:
 *	dataFileNameIndexWasBuiltFrom	numberOfnodes
 *	lenWord0	word0	position_in_index_table
 *	lenWord1	word1	position_in_index_table
 *	...
 *	lenThisWord	word_numberOfnodes-1	position_in_index_table
 *
 *	and the index table has the following format:
 *	number_of_positions	position1...position-number_of_positions
 *
 *	where: dataFileNameIndexWasBuiltFrom is the name of the data
 *	file this index was built from, numberOfnodes is the number of
 *	nodes in the tree, lenWordX is the number of characters in wordX
 *	including the terminating null, wordX is the word followed by
 *	the position in the data file the word came from.
 *
 *	When reading from the index file and doing boolean operations,
 *	we read the index into memory, and acquire the string the user
 *	wants to do a search on.  This string is then broken into words,
 *	using the same mechanisim as breaking the display string into
 *	words.  If a single word is found we print all lines from the
 *	datafile in which the word exists in the display string.  If two
 *	words are found it is taken to be word1 AND word2.  The boolean
 *	operations currently supported are AND, NOT, OR.  All boolean
 *	operations are evaluated left to right.
 *
 *	WARNING:  Attempting to change AND, NOT or OR to a set of
 *	different characters will violate the gopher protocol, and
 *	jugtail will not work in future versions.
 *
 */

#include "stdinc.h"
#include "utils.h"
#include "tree.h"
#include "sockets.h"

void LogMessage (int sockfd, char *message);
void MemoryCaution (int theSize, int limit, char *msg);
char *GetDisplayString (FILE * fp, long *pos);
int CreateWordsTree (char *fileName);
static ListType *GetPositions (long index);
static void WriteHashTables (TreeType * node);
void MakeHashTables (char *fileName, TreeType * root);
short ReservedWord (char *word);
short ParseSearchString (char *string);
static void PrintPositions (int sockfd, ListType *node,
			    long limit, long rangeStart, long rangeEnd);

#ifdef BOOLOP_DEBUG
static void PrintList (ListType * l);
#endif

ListType *DoOperation (short op, ListType * l1, ListType * l2);
void LogRequest (int sockfd);
static ListType *GetAllPositions (long index, char *what2Find,
				  char *asterik, int asterikPos, int sockfd);
void PostPositions (int sockfd, char *what2find);
static int VerifyDataBaseName (char *fName, char *dName);
short CreateElements (char *fileName);
void CleanUp (void);
void HangUpSignal (int);
void DoSearch (char *cmdLine, char *indexTable, char *logFile);
int ReapChild (void);
int PostMsg2Socket (char *str, int s);
static int MakeDaemon (void);
static int WritePIDfile (char *cmdLine);

/* from jugtailConf.c */
extern int ReadConfFile ();

int searchPort;
/* BUGS: The call to IP2Hostname () has the potention to overwrite memory. */

/* Uncomment the "#define NODEFUNCT" line if you don't want the <defunct>
 * process hanging around, or do a "make NODEFUNCT=-DNODEFUNCT".
 * NOTE:  Using "NODEFUNCT" will eat CPU cycles as long as there is a child
 *        process which is forked off.
 *        Not using "NODEFUNCT" is the more efficient way to use jugtail.
 *        The <defunct> process will be cleaned up the next time a connection
 *        is made, but will return as soon as the child processed as terminated
 *        and will not be taken care of until the ReapChild() routine can
 *        process it after another connection. */

/* #define NODEFUNCT */

/* Uncomment this if you want to look at the items after they are built. */
/* #define SHOW_ITEMS_DURING_DEBUG */
/* Uncomment this to assist in debugging the boolean operations. */
/* #define BOOLOP_DEBUG */
#define NOOP		 0	/* The NOOP operation. */
#define AND		 1	/* The AND operation. */
#define OR		 2	/* The OR  operation. */
#define NOT		 3	/* The NOT operation. */
#define MAXWRDSNCMNDS	20	/* The allowable words and commands. */
#define HASHEXT		".ih"	/* The hash table extention. */
#define INDXEXT		".ix"	/* The index table extention. */
#define NINEBACKSPACES	"\b\b\b\b\b\b\b\b\b"
Element *items = (Element *) NULL;	/* Array of words and file postions. */
long numElements,		/* Number of elements in 'items'. */
  elements2do;			/* Number of elements left to do. */
char dataFileName[512],		/* The name of the data file. */
  indexName[512],		/* Name of the index file. */
  hashName[512],		/* Name of the hast file. */
 *logFile = (char *) NULL,	/* Name of the file to log to. */
  *wrdsNcmds[MAXWRDSNCMNDS + 1],	/* Array of words and commands. */
  *hostname = (char *) 0,	/* Name of the machine jugtail is running on. */
  *jugtailhelp = (char *) 0,	/* Location of the "ABOUT" document. */
  *errorhost = (char *) 0,	/* Could be "jugtailhelp" or "\t\terror.host\t-1". */
  *delimiters = (char *) 0,	/* Character to delimit a word when doing a search. */
  *defaultboolop = (char *) 0,	/* The default boolean operation to perform. */
  *jugtailhelptitl = (char *) 0,	/* The title to the jugtail help document. */
  *usagerror = (char *) 0,	/* The usage error string. */
  *wildcarderr = (char *) 0,	/* Invalid wildcard usage string. */
  *tomnyprcserr = (char *) 0,	/* Too may processes string. */
  *noforkerr = (char *) 0,	/* The could not fork message. */
  *gtstrerr = (char *) 0,	/* The get string timed out mesage. */
  *readerr = (char *) 0,	/* The read error message. */
  *tmpfilename = (char *) 0,	/* The temporary file to use. */
  *catcommand = (char *) 0,	/* The cat command. */
  *sortcommand = (char *) 0,	/* The sort command. */
  *touchcommand = (char *) 0;	/* The touch command. */
int port2use,			/* The port to use as a search engine. */
#ifdef NODEFUNCT
  noDefunct = 1,		/* Do we remove the <defunct> process? */
#endif
  maxprocs,			/* Maximum number of jugtail processes. */
  maxitems2return;		/* The maximum items to return. */
short numCommands;		/* The number of words and commands. */
jmp_buf hupbuf;			/* The place to go if -HUP encountered. */
FILE *ifp = (FILE *) NULL,	/* Pointer to the index file. */
  *hfp = (FILE *) NULL;		/* Pointer to the hash file. */

/* The following are defined in "jugtailConf.c". */
extern FILE *rptPtr;		/* File pointer for error messages. */
extern char *userName,		/* Name of the user to run jugtail under. */
 *veronica;			/* Name of the veronica file. */
extern int debug,		/* Are we debugging? */
  time2process,			/* Do we calculate the time for a certain run? */
  menuFlag;			/* Do we display the status as we process? */
extern time_t startTime;	/* The time a run was started, for use with 'time2process'. */

extern void PostTime2Process ();	/* Defined in "jugtailConf.c". */

extern char *MakeStr ();	/* Defined in "jugtailConf.c". */
extern void NullifyDisallowList ();

extern char *vctlHost,		/* Defined in "path.c". */
 *vctlPort;
extern void PrintTheList ();

extern long lineNumber;		/* Defined in "getargs.c". */

extern char *SpecialCommand ();	/* Defined in "searchCmnds.c". */

void MemoryCaution ();

/*****************************************************************************
 * LogMessage logs the message 'message' to the end of the log file 'logFile'.
 * The logging is the same as that found in gopherd.c from the University of
 * Minnesota.
 ****************************************************************************/
void
LogMessage (int sockfd, char *message)
     /* sockfd: The socket file descriptor.
	message: The message to log. */
{
  static char hostName[256],	/* Name of the host we are talking to. */
    ip[256];			/* The hosts IP number.  Won't kill mem. */
  struct flock lock;		/* Info to lock the log file. */
  time_t theTime;		/* The current time. */
  char *cTime,			/* The calendar time. */
   *lineFeed,			/* Location of the line feed. */
    logEntry[2048];		/* The entry to place in the log. */
  int theLog = -1;		/* The file to log to. */

  if (!logFile && !debug)	/* No sense is using the CPU. */
    return;

  if (logFile)			/* Open the sucker. */
    theLog = open (logFile, O_WRONLY | O_APPEND | O_CREAT, 0644);

  hostName[0] = '\0';

  /* DANGER DANGER DANGER: if hostName or ip > 256 this will hoze memory. */
  if (sockfd > -1)
    IP2Hostname (sockfd, hostName, ip);

  time (&theTime);
  cTime = ctime (&theTime);
  if ((lineFeed = strchr (cTime, '\n'))) /* Get rid of the line feed. */
    *lineFeed = '\0';

  MemoryCaution (strlen (hostName) + strlen (message), 2000,
		 "error: LogMessage() attempted to overwrite memory");
  sprintf (logEntry, "%s %d %s : %s\n", cTime, getpid (), hostName,
	   message);

  if (theLog != -1)
    {
      lock.l_type = F_WRLCK;
      lock.l_whence = SEEK_SET;
      lock.l_start = lock.l_len = 0L;
      fcntl (theLog, F_SETLKW, &lock);	/* Lock the file so no one can write to it. */
      lseek (theLog, 0L, SEEK_END);	/* Make sure we are at the end of file. */
      write (theLog, logEntry, strlen (logEntry));
      lock.l_type = F_UNLCK;
      fcntl (theLog, F_SETLKW, &lock);	/* Unlock the file. */
    }

  if (logFile)			/* I guess we can close it now. */
    close (theLog);

  if (debug)
    fprintf (rptPtr, "%s\n", logEntry);


}				/* LogMessage */

/*****************************************************************************
 * MemoryCaution prints the message 'msg' to the logfile and aborts the
 * program if 'theSize' is greater the 'limit'.  Otherwise this routine
 * does nothing except ensure we don't overwrite memory.
 ****************************************************************************/
void
MemoryCaution (int theSize, int limit, char *msg)
     /* theSize: The size we are checking.
	limit: The limit 'size' can be.
	msg: The error message to post if need be. */
{
  if (theSize > limit)
    {
      LogMessage (-1, msg);
      if (debug)
	fprintf (rptPtr, "%s\n", msg);
      exit (-1);
    }

}				/* MemoryCaution */

/*****************************************************************************
 * GetDisplayString returns the display string portion of the gopher line
 * contained in 'str'.  I should mention the item type, which is the first
 * character in the display string, is skipped.
 ****************************************************************************/
char *
GetDisplayString (FILE *fp, long *pos)
     /* fp: The file we are reading.
	pos: Position in the file of the line. */
{
  static char buf[2048];	/* Buffer for the display string. */
  size_t len;			/* Number of characters to the tab. */
  static long line = 0;		/* The line number in the file for error reporting. */
  char *s;			/* Pointer to the line we acquire from the file. */
  short error = 0;		/* Did we encounter an error? */

  if (!(--elements2do % 10) && menuFlag)
    fprintf (stdout, "%s%9ld", NINEBACKSPACES, elements2do);

  *pos = ftell (fp);
  line++;

  if (fgets (buf, 2048, fp))
    {
    if ((s = strchr (buf, '\t')))
      if ((len = (size_t) (s - buf)) > 0)
	{
	  buf[len] = '\0';
	  return (buf + 1);
	}
      else
	error = 1;
    else
      error = 1;
    }

  if (error)
    {
      fprintf (rptPtr,
	       "%swarning: GetDisplayString found bad line, line = %ld.\n         ", NINEBACKSPACES, line);
      pos = 0;
      return ("");
    }

  return ((char *) NULL);

}				/* GetDisplayString */

/*****************************************************************************
 * CreateWordsTree creates a tree where each node of the tree contains a
 * word and a list of file positions, representing the line the word is
 * contained in.  It should be mentioned that the words are parsed from
 * the display string according to the 
 ****************************************************************************/
int
CreateWordsTree (char *fileName)
     /* fileName: Name of the data file. */
{
  long position;		/* Position in the file of the current line. */
  FILE *fpIn;			/* The data file we area reading from . */
  char *dStr,			/* The display string with no leading item type. */
    *word;			/* A word from dStr. */
  int error = 0;		/* Did we get an error? */

  if ((fpIn = fopen (fileName, "r")))
    {
      if (menuFlag)
	{
	  fprintf (stdout, "Building the words tree...\n");
	  fprintf (stdout, "%9ld", elements2do = NumberOfLines (fileName));
	}
      
      while ((dStr = GetDisplayString (fpIn, &position)))
	if (*dStr)
	  for (word = strtok (dStr, delimiters); word;
	       word = strtok ((char *) NULL, delimiters))
	    BuildTree (&root, StrToLower (word), position);
      fclose (fpIn);
      if (menuFlag)
	fprintf (stdout, "%swords tree is now built.\n", NINEBACKSPACES);
    }
  else
    error = fprintf (rptPtr, "error: CreateWordsTree could not open %s\n",
		     fileName);

  return (!error);

}				/* CreateWordsTree */

/*****************************************************************************
 * GetPositions returns a list of the file positions, which contains the
 * gopher information to send the client we are talking to, or do a given
 * operation on.
 ****************************************************************************/
static ListType *
GetPositions (long index)
{
  FILE *fp;
  ListType *list = (ListType *) NULL;
  register int i;
  int numPos;
  long where;

  if (debug)
    fprintf (rptPtr,
	     "In GetPositions with index = %ld, items[%ld].word = [%s], items[%ld].positions = %ld\n",
	     index, index, items[index].word, index,
	     items[index].positions->where);
  if ((fp = fopen (indexName, "r")))
    {
      if (!fseek (fp, items[index].positions->where, SEEK_SET))
	{
	  numPos = GetInt (fp);
#ifdef BOOLOP_DEBUG
	  if (debug)
	    fprintf (rptPtr, "\tnumPos = %d\n", numPos);
#endif
	  for (i = 0; i < numPos; i++)
	    {
	      where = GetLong (fp);
#ifdef BOOLOP_DEBUG
	      if (debug)
		fprintf (rptPtr, "\twhere = %ld\n", where);
#endif
	      list = BuildList (list, where);
	    }
	}
      else
	LogMessage (-1, "error: GetPositions had fseek fail");
      fclose (fp);
    }
  else
    {
      char s[512 + 50];		/* Make sure we don't walk on memory. */
      sprintf (s, "error: GetPositions could not open %s for reading",
	       indexName);
      LogMessage (-1, s);
    }

  return (list);

}				/* GetPositions */

/*****************************************************************************
 * WriteHashTables writes the information within the tree pointed to by 'node'
 * to the hash table and the index table.
 ****************************************************************************/
static void
WriteHashTables (TreeType *node)
     /* node: The node to process. */
{
  ListType *positions;		/* List with the positions. */

  if (node)
    {
      WriteHashTables (node->left);

      if (!(--elements2do % 10) && menuFlag)
	fprintf (stdout, "%s%9ld", NINEBACKSPACES, elements2do);

      fprintf (hfp, "%d\t%s\t%ld\n", (int) strlen (node->word) + 1,
	       node->word, (long) ftell (ifp));

      fprintf (ifp, "\t%ld", NumberOfListNodes (node->positions));
      for (positions = node->positions; positions;
	   positions = positions->next)
	fprintf (ifp, "\t%ld", positions->where);
      fprintf (ifp, "\n"), ++lineNumber;

      WriteHashTables (node->right);
    }

}				/* WriteHashTables */

/*****************************************************************************
 * MakeHashTables creates the hash and index table such that the hash table
 * contains the data file to index, the number of elements to create, followed
 * by each line containing the size of the word plus 1 for the null character,
 * a tab, the word, tab, and the file position in the the index table.  The
 * index table contains the number of file positions, tab, the file positions
 * seperated by tabs on a single line.  Each file position in the index table
 * is the position in the data file where the word found in the hash table
 * is referencing.
 ****************************************************************************/
void
MakeHashTables (char *fileName, TreeType *root)
     /* fileName: Name of the data file.
	root: The root of the tree.      */
{
  if (menuFlag)
    fprintf (stdout, "Building the hash tables...\n");

  /* Make sure we don't overwrite memory. */
  if ((strlen (fileName) + strlen (INDXEXT)) > 512)
    {
      fprintf (rptPtr, "error: attempt to overwrite memory.  %s too long\n",
	       fileName);
      exit (-1);
    }
  if ((strlen (fileName) + strlen (HASHEXT)) > 512)
    {
      fprintf (rptPtr, "error: attempt to overwrite memory.  %s too long\n",
	       fileName);
      exit (-1);
    }

  strcpy (indexName, fileName);
  strcat (indexName, INDXEXT);
  strcpy (hashName, fileName);
  strcat (hashName, HASHEXT);

  if (!(ifp = fopen (indexName, "w")))
    fprintf (rptPtr, "error: MakeHashTable could not open %s for writing\n",
	     indexName);
  if (!(hfp = fopen (hashName, "w")))
    fprintf (rptPtr, "error: MakeHashTable could not open %s for writing\n",
	     hashName);
  if (!ifp || !hfp)
    exit (-1);

  fprintf (hfp, "%s\t%ld\n", fileName, elements2do = NumberOfLeafs (root));
  if (menuFlag)
    fprintf (stdout, "         ");	/* To support use of the backspaces. */
  WriteHashTables (root);

  if (menuFlag)
    fprintf (stdout, "%shash tables are completed.\n", NINEBACKSPACES);

  fclose (ifp);
  fclose (hfp);
  ifp = hfp = (FILE *) NULL;

}				/* MakeHashTables */

/*****************************************************************************
 * ReservedWord returns true if 'word' is "AND", "OR", or "NOT", otherwise
 * it returns false.  If 'word' is "AND" this routine returns 1, if 'word' is
 * "OR" we return 2, if 'word' is "NOT" 3 gets returned.
 * WARNING:  Any attempt to change the set of characters representing AND,
 * OR, or NOT will not only violate gopher protocol but will also make
 * jugtail fail in future versions.
 ****************************************************************************/
short
ReservedWord (char *word)
     /* word: The word we are checking if reserved. */
{
  size_t len;			/* The length of 'word'. */
  len = strlen (word);

  switch (len)
    {
    case 2:
      if ((word[0] == 'O' || word[0] == 'o') &&
	  (word[1] == 'R' || word[1] == 'r'))
	{
	  word[0] = 'O';
	  word[1] = 'R';
	  return (OR);
	}
      break;
    case 3:
      if ((word[0] == 'A' || word[0] == 'a') &&
	  (word[1] == 'N' || word[1] == 'n') &&
	  (word[2] == 'D' || word[2] == 'd'))
	{
	  word[0] = 'A';
	  word[1] = 'N';
	  word[2] = 'D';
	  return (AND);
	}
      else if ((word[0] == 'N' || word[0] == 'n') &&
	       (word[1] == 'O' || word[1] == 'o') &&
	       (word[2] == 'T' || word[2] == 't'))
	{
	  word[0] = 'N';
	  word[1] = 'O';
	  word[2] = 'T';
	  return (NOT);
	}
      break;
    default:
      break;
    }

  return (NOOP);

}				/* ReservedWord */

/*****************************************************************************
 * ParseSearchString parses 'string' into words and or commands which we do
 * searches and boolean searches on.  All words and commands get placed into
 * the array 'commands'.  If any word is "and", "or", or "not" it is taken to
 * be the command AND, OR, or NOT respectivly.  If any 2 words are not
 * seperated with a command, the command AND is implied to be the seperating
 * command.
 ****************************************************************************/
short
ParseSearchString (char *string)
     /* string: The string we are parsing. */
{
  char *word;			/* The word extracted from 'string'. */
  short lastOneReserved,	/* Was the last word reserved? */
    thisOneReserved;		/* Is the current word reserved? */
  
#ifdef IN_THE_FUTURE
  /* In the future the database to use will exist before the tab. */
  if (time2process)
    time (&startTime);

  if (!CreateElements (indexTable))
    {
      if (!debug)
	fprintf (rptPtr, "error: DoSearch could not create index tree.\n");
      LogMessage (-1, "error: DoSearch could not create index tree");
      exit (-1);
    }

  if (time2process)
    PostTime2Process ();
#endif

  for (numCommands = 0, lastOneReserved = 1, word =
	 strtok (string, delimiters); numCommands < MAXWRDSNCMNDS && word;
       word = strtok ((char *) NULL, delimiters))
    {
      thisOneReserved = ReservedWord (word);
      if (!lastOneReserved && !thisOneReserved)
	{
	  wrdsNcmds[numCommands++] = defaultboolop;
	  wrdsNcmds[numCommands++] = StrToLower (word);
	}
      else if (thisOneReserved)
	wrdsNcmds[numCommands++] = word;
      else
	wrdsNcmds[numCommands++] = StrToLower (word);
      lastOneReserved = thisOneReserved;
    }

  return (numCommands);

}				/* ParseSearchString */

/*****************************************************************************
 * PrintPositions sends the line of text from the data file, the index table
 * was built from, whose postion in the file is specified by 'node->where'
 * to stdout.
 ****************************************************************************/
static void
PrintPositions (int sockfd, ListType *node,
		long limit, long rangeStart, long rangeEnd)
     /* sockfd: 	The socket file descriptor.
	node:		The list we are printing.
	limit:		The max number of items to return.
	rangeStart:	The start of a range.
	rangeEnd:	The end of a range.                */
{
  FILE *fp;			/* Pointer to the file with the actual data. */
  char buf[2048];		/* Buffer for the line of text. */
  register long i;		/* A loop counter. */
  long n;			/* The number of items to return. */

  /* If too many items create some links on the fly and send'em. */
  if (!rangeStart && limit == maxitems2return
      && (n = NumberOfListNodes (node)) > maxitems2return)
    {
      char s[1024],		/* A temporary string. */
        what[1024];		/* The search command - any special command. */

      /* Handle the "All 'n' items" directory. */
      sprintf (s, "1All %ld items\t?all", n);	/* <- Won't trash memory. */
      SendString (s);

      for (what[0] = i = 0; i < numCommands; i++)
	{
	  MemoryCaution (strlen (what) + strlen (wrdsNcmds[i]) + 1, 1024,
			 "error: PrintPositions() attempted to overwrite memory");
	  strcat (what, " ");
	  strcat (what, wrdsNcmds[i]);
	}

      SendString (what);
      sprintf (s, "\t%s\t%d\r\n", hostname, port2use);	/* Won't trash mem. */
      SendString (s);

      /* Now handle the range directories. */
      for (i = 0; i < n; i += limit)
	{
	  rangeStart = i + 1;
	  if ((i + limit) < n)
	    rangeEnd = i + limit;
	  else
	    rangeEnd = n;
	  sprintf (s, "1items %ld to %ld\t?range=%ld-%ld %s\t%s\t%d\r\n", rangeStart, rangeEnd, rangeStart, rangeEnd, what, hostname, port2use);	/* Won't trash mem. */
	  SendString (s);
	}

      sprintf (s, "TOO MANY ITEMS: %ld items found", n);	/* Won't trash mem. */
      LogMessage (sockfd, s);
      return;
    }

  if ((fp = fopen (dataFileName, "r")))
    {
      if (rangeStart)
	{
	  /* Traverse node to the rangeStart item. */
	  for (i = 0; i < rangeStart - 1 && node; i++, node = node->next);

	  /* Spit out range of items. */
	  for (; i < rangeEnd && node; i++, node = node->next)
	    if (!fseek (fp, node->where, SEEK_SET))
	      SendString (fgets (buf, 2048, fp));
	}
      else if (limit < 0)
	while (node)
	  {
	    if (!fseek (fp, node->where, SEEK_SET))
	      SendString (fgets (buf, 2048, fp));
	    node = node->next;
	  }
      else
	for (n = 0; n < limit && node; n++, node = node->next)
	  if (!fseek (fp, node->where, SEEK_SET))
	    SendString (fgets (buf, 2048, fp));

      fclose (fp);
    }
  else
    {
      char s[512];
      MemoryCaution (strlen (dataFileName), 450,
		     "error: PrintPositions() attempted to overwrite memory");
      sprintf (s, "error: PrintPositions could not open data file [%s]",
	       dataFileName);
      LogMessage (sockfd, s);
    }

}				/* PrintPositions */

#ifdef BOOLOP_DEBUG
/*****************************************************************************
 * PrintList prints the list 'l', and was written solely for debugging.
 ****************************************************************************/
static void
PrintList (ListType *l)
     /* l: The list to print. */
{
  if (debug)
    while (l)
      {
	fprintf (rptPtr, "%10ld,", l->where);
	l = l->next;
      }

}				/* PrintList */
#endif

/*****************************************************************************
 * DoOperation returns a list which is the result of "l1 op l2" where 'l1' and
 * 'l2' are lists, and 'op' is the operation to perform which is either the
 * AND, OR, or NOT operation.
 ****************************************************************************/
ListType *
DoOperation (short op, ListType *l1, ListType *l2)
     /* op: The operation [AND,OR,NOT]
	l1: A list we operate on
	l2: The other list we operate on. */
{
  ListType *result = (ListType *) NULL,	/* The result of "l1 op l2". */
    *t = (ListType *) NULL;	/* A pointer into 'l1' or 'l2'. */

  switch (op)
    {
    case AND:
      while (l1 && l2)
	if (l1->where == l2->where)
	  {
	    result = BuildList (result, l1->where);
	    l1 = l1->next;
	    l2 = l2->next;
	  }
	else if (l1->where < l2->where)
	  l1 = l1->next;
	else if (l1->where > l2->where)
	  l2 = l2->next;
      break;
    case OR:
      while (l1 && l2)
	if (l1->where == l2->where)
	  {
	    result = BuildList (result, l1->where);
	    l1 = l1->next;
	    l2 = l2->next;
	  }
	else if (l1->where < l2->where)
	  {
	    result = BuildList (result, l1->where);
	    l1 = l1->next;
	  }
	else if (l1->where > l2->where)
	  {
	    result = BuildList (result, l2->where);
	    l2 = l2->next;
	  }
      if (l1)
	t = l1;
      else if (l2)
	t = l2;
      while (t)
	{
	  result = BuildList (result, t->where);
	  t = t->next;
	}
      break;
    case NOT:
      while (l1 && l2)
	if (l1->where == l2->where)
	  {
	    l1 = l1->next;
	    l2 = l2->next;
	  }
	else if (l1->where < l2->where)
	  {
	    result = BuildList (result, l1->where);
	    l1 = l1->next;
	  }
	else if (l1->where > l2->where)
	  l2 = l2->next;
      while (l1)
	{
	  result = BuildList (result, l1->where);
	  l1 = l1->next;
	}
      break;
    default:
      result = (ListType *) NULL;
      break;
    }

  return (result);

}				/* DoOperation */

/*****************************************************************************
 * LogRequest simply writes the search request to the log file.
 ****************************************************************************/
void
LogRequest (int sockfd)
     /* sockfd: The socket to write to. */
{
  char buf[1024];		/* String to build to write out. */
  register short i;		/* A loop counter. */

  if (!logFile && !debug)	/* No sense is using the CPU. */
    return;
  
  /* The following will never overwrite memory. */
  sprintf (buf, "jugtail(%d) -> ", port2use);

  for (i = 0; i < numCommands; i++)
    if (strlen (buf) + strlen (wrdsNcmds[i]) + 1 < 1024)
      {
	strcat (buf, " ");
	strcat (buf, wrdsNcmds[i]);
      }
    else
      break;
  LogMessage (sockfd, buf);

}				/* LogRequest */

/*****************************************************************************
 * GetAllPositions acquires all the postitions of the partial word search by
 * finding the start and end position, and then OR'ing these positions into
 * a list which gets returned.  If 'what2Find' starts off with the wild card
 * character, the asterik, we post a message to the user and return nil.
 ****************************************************************************/
static ListType *
GetAllPositions (long index, char *what2Find,
		 char *asterik, int asterikPos, int sockfd)
     /* index: Index into the items array.
	what2Find: The word we are looking for.
	asterik: The position of the asterik.
	asterikPos: Number of characters to look at.
	sockfd: The socket file descriptor, errors only. */
{
  ListType *theList = (ListType *) NULL,	/* The list to return. */
    *list1 = (ListType *) NULL,	/* List of positions to OR against list2. */
    *list2 = (ListType *) NULL;	/* List of positions to OR against list1. */
  long start,		/* Start of the partial word match. */
    end;			/* End of the partial word match. */

  if (what2Find == asterik && *what2Find == '*')
    {
      char s[1024];
      MemoryCaution (strlen (wildcarderr) + strlen (errorhost), 1024,
		     "error: GetAllPositions() attempted to overwrite memory");
      sprintf (s, "%s%s", wildcarderr, errorhost);
      SendString (s);
      MemoryCaution (strlen (what2Find), 1000,
		     "error: GetAllPositions() attempted to overwrite memory");
      sprintf (s, "INVALID WILDCARD USAGE: %s", what2Find);
      LogMessage (sockfd, s);
      return ((ListType *) NULL);
    }

  /* Find the starting and ending positions of the positions to return. */
  for (start = index - 1;
       start >= 0 && !strncmp (what2Find, items[start].word, asterikPos);
       start--);
  for (end = index + 1;
       end < numElements && !strncmp (what2Find, items[end].word, asterikPos);
       end++);

  if (debug)
    fprintf (rptPtr, "GetAllPositions found starting position = %ld, and ending position = %ld\n", start + 1, end - 1);

  /* Process the positions we will be returning. */
  for (start++; start < end; start++)
    if (!list1)
      list1 = GetPositions (start);
    else if (!list2)
      {
	list2 = GetPositions (start);
	theList = DoOperation (OR, list1, list2);
	DestroyList (list1);
	DestroyList (list2);
	list1 = theList;
	list2 = (ListType *) NULL;
      }

  /* We may have only got one hit, so make sure we return the information. */
  if (!theList && list1)
    theList = list1;

  return (theList);

}				/* GetAllPositions */

/*****************************************************************************
 * PostPositions posts the result of doing the boolean operations on 'what2find'
 * and checking for membership in 'array'.
 ****************************************************************************/
void
PostPositions (int sockfd, char *what2find)
     /* sockfd: The socket to write to.
	what2find: String with words and operations. */
{
  short evaluate,		/* Do we evaluate the operation? */
    operater = NOOP,		/* Either [NOOP,AND,OR,NOT]. */
    reserved;			/* Is the current word reserved? */
  short i;			/* A loop counter. */
  long index,			/* Index into the items array. */
    rangeStart,			/* The start of a range. */
    rangeEnd,			/* The end of a range. */
    limit = maxitems2return;	/* The max number of items to return. */
  ListType *list1 = (ListType *) NULL,	/* List of positions to operate against list2. */
    *list2 = (ListType *) NULL,	/* List of positions to operate against list1. */
    *result = (ListType *) NULL,	/* The result of "list1 operation list2'. */
    *tList = (ListType *) NULL;	/* Temporary list to support partial word searches. */
  char *asterik;		/* Is this a partial word search? */
  int asterikPos;		/* Number of characters to the asterik. */

  if (!
      (what2find =
       SpecialCommand (what2find, &limit, &rangeStart, &rangeEnd)))
    return;
  if (ParseSearchString (what2find))
    {
      LogRequest (sockfd);
      for (evaluate = i = 0; i < numCommands; i++)
	if ((reserved = ReservedWord (wrdsNcmds[i])))
	  operater = reserved;
	else
	  if ((index =
	       BinarySearch (StrToLower (wrdsNcmds[i]), items, numElements,
			     &asterik, &asterikPos)) >= 0)
	  {
	    if (asterik)	/* We have a partial word search. */
	      tList =
		GetAllPositions (index, wrdsNcmds[i], asterik, asterikPos,
				 sockfd);
	    else
	      tList = GetPositions (index);
	    if (!list1 && !evaluate)
	      {
		result = list1 = tList;
		evaluate = 1;
	      }
	    else if (!list2)
	      {
		list2 = tList;

		result = DoOperation (operater, list1, list2);
#ifdef BOOLOP_DEBUG
		if (debug)
		  {
		    fprintf (rptPtr, "list1 ==> ");
		    PrintList (list1);
		    fprintf (rptPtr, "\nlist2 ==> ");
		    PrintList (list2);
		    fprintf (rptPtr, "\nresult ==> ");
		    PrintList (result);
		    fprintf (rptPtr, "\n");
		  }
#endif
		DestroyList (list1);
		DestroyList (list2);
		list1 = result;
		list2 = (ListType *) NULL;
		operater = NOOP;
	      }
	  }
	else
	  {
	    char message[256];
	    sprintf (message, "COULD NOT FIND [%s]", wrdsNcmds[i]);
	    LogMessage (sockfd, message);
	    if (operater == AND)
	      {
		DestroyList (result);
		result = list1 = (ListType *) NULL;
	      }
	    evaluate = 1;
	  }

      if (result)
	{
	  PrintPositions (sockfd, result, limit, rangeStart, rangeEnd);
	  DestroyList (result);
	}
    }

}				/* PostPositions */

/*****************************************************************************
 * VerifyDataBaseName verifies we are dealing with the correct database.
 * This routine returns true if we have the correct database and false
 * otherwise.
 ****************************************************************************/
static int
VerifyDataBaseName (char *fName, char *dName)
     /* fName: The root name of the database.
        dName: The name we should be dealing with. */
{
#if(0)				/* Commented out because it was failing. */
  char *str;			/* Position in fName where dName occurs. */

  if (str = strstr (fName, dName))
    {
      if (!strcmp (dName, str))
	strcpy (dName, fName);
      return (1);
    }
  return (0);
#else
  strcpy (dName, fName);
  return (1);
#endif

}				/* VerifyDataBaseName */

/*****************************************************************************
 * CreateElements returns true if we could create the dynamic array 'items'
 * and false othewise.
 ****************************************************************************/
short
CreateElements (char *fileName)
     /* fileName: The name of the file to read. */
{
  FILE *fp;			/* Pointer to the file we are reading. */
  long l;			/* A loop counter. */
  long where;			/* The postions in the "data" file. */
  short strLen;			/* The size of the word. */
  char s[1024];			/* A temporary do it all string */
  /* and cannot overwrite memory.  */

  /* Make sure we don't overwrite memory. */
  if ((strlen (fileName) + strlen (INDXEXT)) > 512)
    {
      fprintf (rptPtr, "error: attempt to overwrite memory.  %s too long\n",
	       fileName);
      exit (-1);
    }
  if ((strlen (fileName) + strlen (HASHEXT)) > 512)
    {
      fprintf (rptPtr, "error: attempt to overwrite memory.  %s too long\n",
	       fileName);
      exit (-1);
    }

  strcpy (indexName, fileName);
  strcat (indexName, INDXEXT);
  strcpy (hashName, fileName);
  strcat (hashName, HASHEXT);

  if (!(fp = fopen (fileName, "r")))
    {
      sprintf (s, "error: CreateElements could not open %s for reading\n",
	       fileName);
      LogMessage (-1, s);
    }
  if (!(hfp = fopen (hashName, "r")))
    {
      sprintf (s, "error: CreateElements could not open %s for reading\n",
	       hashName);
      LogMessage (-1, s);
    }
  if (!fp || !hfp)
    return (0);

  GetStr (hfp, dataFileName, 512);

  if (!VerifyDataBaseName (fileName, dataFileName))
    {
      sprintf (s, "error: incompatible database, looking for [%s]\n",
	       dataFileName);
      LogMessage (-1, s);
      return (0);
    }

  if (debug)
    {
      fprintf (rptPtr, "dataFileName = [%s]\n", dataFileName);
      fprintf (rptPtr, "veronicaFile = [%s]\n", veronica);
      fprintf (rptPtr, "indexName    = [%s]\n", indexName);
      fprintf (rptPtr, "hashName     = [%s]\n", hashName);
    }

  numElements = GetLong (hfp);
  if ((items = malloc (numElements * sizeof (Element))))
    for (l = 0; l < numElements; l++)
      {
	strLen = (short) GetInt (hfp);
	if ((items[l].word = malloc (strLen)))
	  {
	    GetStr (hfp, items[l].word, strLen);
	    where = GetLong (hfp);
	    items[l].positions = (ListType *) NULL;
	    items[l].positions = BuildList (items[l].positions, where);
	  }
	else
	  {
	    sprintf (s, "error: CreateElements could not get memory for string %ld\n", l);
	    LogMessage (-1, s);
	    return (0);
	  }
      }
  else
    {
      sprintf (s, "error: CreateElements could not get memory for the %ld items\n", numElements);
      LogMessage (-1, s);
      return (0);
    }

#ifdef SHOW_ITEMS_DURING_DEBUG
  if (debug)
    {
      fprintf (rptPtr, "items looks like:\n");
      for (l = 0; l < numElements; l++)
	fprintf (rptPtr, "\t[%ld]\t[%s]\t%ld\n", l, items[l].word,
		 items[l].positions->where);
    }
#endif

  fclose (fp);
  fclose (hfp);
  hfp = (FILE *) NULL;
  return (1);

}				/* CreateElements */

/*****************************************************************************
 * CleanUp simply frees up any memory used in the dynamic array 'items'.
 ****************************************************************************/
void
CleanUp (void)
{
  long i;			/* A loop counter. */

  for (i = 0; i < numElements; i++)
    {
      free (items[i].word);
      DestroyList (items[i].positions);
    }
  free (items);
  numElements = 0;

}				/* CleanUp */

/*****************************************************************************
 * HangUpSignal resets the hangup interrupt, touches the
 * /usr/local/etc/jugtail.pid file, and returns to the saved state.
 ****************************************************************************/
void
HangUpSignal (int sig)
{
  signal (SIGHUP, HangUpSignal);	/* Set things for next time. */

  LogMessage (-1, "SIGHUP signal encountered");
  if (debug)
    fprintf (rptPtr, "Releasing memory ...\n");

  CleanUp ();

  /* To support the "Disallow:" directive, but isn't used when    *
   * jugtail is a search engine.  Added here for clarity.         */
  vctlHost = vctlPort = "";

  NullifyDisallowList ();
  if (!ReadConfFile (JUGTAILCONF))
    {
      fprintf (rptPtr, "error: could not initialize the search engine\n");
      if (debug)
	fprintf (rptPtr, "error: InitSearchEngine() failed\n");
      exit (-1);
    }

  DoSystemCall (Mysprint (touchcommand, JUGTAILPID));

  if (debug)
    {
      PrintTheList ();
      fprintf (rptPtr, "Rebuilding the binary tree.\n");
    }

  longjmp (hupbuf, 0);		/* Jump to the saved state. */

}				/* HangUpSignal */

/*****************************************************************************
 * ReapChild returns 1 when a child process has died, and returns 0 if no
 * child processes exist or have not had a change of state.
 ****************************************************************************/
int
ReapChild (void)
{
  int status;			/* Status of the child process. */

  if (waitpid ((pid_t) -1, &status, WNOHANG | WUNTRACED) > 0)
    {
      if (debug)
	fprintf (rptPtr, "In ReapChild() with status = %d\n", status);
      return (1);
    }
  
  return (0);

}				/* ReapChild */

/*****************************************************************************
 * PostMsg2Socket returns true if it could write the string 'str' to the
 * socket 's', otherwise it returns false.
 ****************************************************************************/
int
PostMsg2Socket (char *str, int s)
     /* str: The string to post.
	s:   The socket to write to. */
{
  FILE *fp;

  if ((fp = fdopen (s, "w")))
    {
      fprintf (fp, "%s%s", str, errorhost);
      fflush (fp);
      fclose (fp);
      return (1);
    }
  return (0);

}				/* PostMsg2Socket */

/*****************************************************************************
 * MakeDaemon returns 0 if we were able to detach the program from a tty and
 * make it a daemon.  Otherwise this routine -1 signifying we could not
 * do it.
 * This code was taken from Advanced Programming in the UNIX Environment
 * by W. Richard Stevens.
 ****************************************************************************/
static int
MakeDaemon (void)
{
  pid_t pid;

  if (getpid () != (pid_t) 1)
    {
      if ((pid = fork ()) < 0)
	return (-1);
      else if (pid > 0)
	exit (0);		/* Parent goes bye-bye. */
    }

  /* Child continues. */

  setsid ();		/* Become session leader. */

  /* chdir("/");        No need, we're already where we want to be. */

  umask (0);		/* Clear our file mode creation mask. */

  return (0);


}				/* MakeDaemon */

/*****************************************************************************
 * WritePIDfile writes the jugtail.pid file, by default stored in
 * /usr/local/etc.  The format of the file is:
 *    Process_ID
 *    Command_Line_Invoking_jugtail_as_a_search_engine
 * This routine is used solely to assist in scripting the ability to send
 * SIGHUP to jugtail and keep a record as to how jugtail was previously
 * started as a search engine.  This routine returns true if the file could
 * be written and false otherwise.  If 'cmdLine' was acquired via the call
 * to GetArguments() in jugtailConf.c this routine also frees up the memory.
 * Thank you Kuypers@sri.ucl.ac.BE for the idea.
 ****************************************************************************/
static int
WritePIDfile (char *cmdLine)
     /* cmdLine: Command line starting jugtail. */
{
  FILE *fp;			/* Pointer to the jugtail.pid file. */
  char bufr[512];		/* Buffer for error messages. */

  if (cmdLine && (fp = fopen (JUGTAILPID, "w")))
    {
      fprintf (fp, "%d\n%s\n", getpid (), cmdLine);
      fclose (fp);
      free (cmdLine);
      return (1);
    }
  else
    {
      MemoryCaution (strlen (JUGTAILPID), 350,
		     "error: WritePIDfile() attempted to overwrite memory");
      sprintf (bufr, "error: Could not create %s\n", JUGTAILPID);
      LogMessage (-1, bufr);
      return (0);
    }

}				/* WritePIDfile */

/*****************************************************************************
 * DoSearch is the search server part of jugtail.  This routine never
 * returns.
 ****************************************************************************/
void
DoSearch (char *cmdLine, char *indexTable, char *logFile)
     /* cmdLine: The command line starting jugtail.
	indexTable: Name of the index table file.
	logFile: The file to log to. */
{
  int childspid,		/* The process ID of the child process. */
    s,				/* The socket file descriptor. */
    newS,			/* The new socket file descriptor. */
    addressLen;			/* Size of the address space. */
  struct sockaddr_in address;	/* Address of connecting entity. */
  char bufr[512];		/* Buffer for error messages. */
#ifdef NODEFUNCT
  int blocking,			/* Is accept blocking? */
    initialCntl,		/* Initial socket control settings. */
    initialProcs,		/* Initial number of processes allowed. */
    debugLoopVar;		/* A loop variant for debugging. */
#endif


  fprintf (rptPtr, "jugtail, Copyright 1993, 1994, University of Utah Computer Center\n");
  fprintf (rptPtr, "Using index table %s\n", indexTable);
  fprintf (rptPtr, "Using port %d\n", port2use);

  if (logFile)
    {
      if (debug)
	fprintf (rptPtr, "Logging to %s\n", logFile);
      MemoryCaution (strlen (indexTable), 490,
		     "error: DoSearch() attempted to overwrite memory");
      strcpy (bufr, "STARTED UP ON ");
      strcat (bufr, indexTable);
      LogMessage (-1, bufr);
    }

  if (debug)
    fprintf (rptPtr, "debug is on: NOT going in daemon mode.\n");
  else if (MakeDaemon ())	/* Detach from the tty if not running in the background. */
    LogMessage (-1, "error: jugtail cannot make itself daemon");

  if (cmdLine)
    {
      if (!debug)
	WritePIDfile (cmdLine);
      else
	fprintf (rptPtr, "debug is on: NOT writing [%s] to pid file [%s]\n",
		 cmdLine, JUGTAILPID);
      free (cmdLine);
    }

  if (userName)
    {
      struct passwd *pswd;
      if ((pswd = getpwnam (userName)))
	if (setuid (pswd->pw_uid))
	  {
	    sprintf (bufr, "error: could not setuid %ld (%s)\n", (long) pswd->pw_uid, userName);	/* Won't kill mem. */
	    LogMessage (-1, bufr);
	    exit (-1);
	  }
	else
	  {
	    sprintf (bufr, "Running with setuid %ld (%s)\n", (long) pswd->pw_uid, userName);	/* Won't kill mem. */
	    LogMessage (-1, bufr);
	  }
      else
	{
	  MemoryCaution (strlen (userName), 475,
			 "error: DoSearch() attempted to overwrite memory");
	  sprintf (bufr, "error: could not find user %s\n", userName);
	  LogMessage (-1, bufr);
	  exit (-1);
	}
    }

  if ((s = ListenerEstablished (port2use)) < 0)
    exit (-1);

  if (signal (SIGHUP, HangUpSignal) == SIG_ERR)
    {
      sprintf (bufr, "error: could not establish SIGHUP handler\n");
      if (debug)
	fprintf (rptPtr, "%s", bufr);
      LogMessage (-1, bufr);
      exit (-1);
    }
  setjmp (hupbuf);

#ifndef IN_THE_FUTURE
  if (time2process)
    time (&startTime);

  if (!CreateElements (indexTable))
    {
      LogMessage (-1, "error: DoSearch could not create index tree");
      if (debug)
	fprintf (rptPtr, "error: DoSearch could not create index tree\n");
      exit (-1);
    }

  if (time2process)
    PostTime2Process ();
#endif

#ifdef NODEFUNCT
  if (noDefunct)		/* Set things up to removed the <defunct> process. */
    {
      initialCntl = fcntl (s, F_GETFL, O_NDELAY);
      initialProcs = maxprocs;
      blocking = 1;
      if (debug)
	debugLoopVar = 0;
    }
#endif

  if (debug)
    fprintf (rptPtr, "Ready for incoming connections.\n");

  while (1)			/* Wait for and handle all connections. */
    {
      addressLen = sizeof (address);
      if ((newS = accept (s, (struct sockaddr *) &address, &addressLen)) < 0)
	{
#ifdef NODEFUNCT
	  if (errno != EWOULDBLOCK)
#endif
	    {
	      LogMessage (-1, "error: DoSearch could not accept");
	      exit (-1);
	    }
	}
      else
	{
	  maxprocs--;
	  if (maxprocs <= 0)
	    {
	      if (!PostMsg2Socket (tomnyprcserr, newS))
		LogMessage (-1, "error: could not open socket for writing");
	      if (debug)
		fprintf (rptPtr, "error: Too many jugtail processes.\n");
	      maxprocs++;
	    }
	  else if ((childspid = fork ()) < 0)
	    {
	      PostMsg2Socket (noforkerr, newS);
	      LogMessage (-1, "error: could not fork");
	      if (debug)
		fprintf (rptPtr, "error: could not fork\n");
	      maxprocs++;
	      sleep (50);
	    }
	  else if (!childspid)	/* Child process so */
	    {			/* close original socket */
	      close (s);
	      ProcessRequest (newS);

#ifdef IN_THE_FUTURE
	      CleanUp ();
#endif
	      exit (0);
	    }
	  close (newS);	/* Close the new socket we opened. */
	}

      if (debug)
	{
	  fprintf (rptPtr, "before ReapChild() maxprocs = %d\n", maxprocs);
#ifdef NODEFUNCT
	  if (noDefunct)
	    debugLoopVar++;
#endif
	}
      while (ReapChild ())
	maxprocs++;
      if (debug)
	{
	  fprintf (rptPtr, "after  ReapChild() maxprocs = %d", maxprocs);
#ifdef NODEFUNCT
	  if (noDefunct)
	    fprintf (rptPtr, " debugLoopVar = %d\n", debugLoopVar);
	  else
#endif
	    fprintf (rptPtr, "\n");
	}

#ifdef NODEFUNCT
      if (noDefunct)		/* Adjust accept() for blocking or non-blocking. */
	if (maxprocs < initialProcs)
	  {
	    if (blocking)
	      {
		fcntl (s, F_SETFL, O_NDELAY);
		blocking = 0;
	      }
	  }
	else
	  {
	    fcntl (s, F_SETFL, initialCntl);
	    blocking = 1;
	    if (debug)
	      debugLoopVar = 0;
	  }
#endif
    }

}				/* DoSearch */
