/*****************************************************************************
 * File:	sockets.c
 *
 * Author:	Rhett "Jonzy" Jones	jonzy@cc.utah.edu
 *
 * Date:	March 15, 1993
 *
 * Modified:	May 6, 1993, by Rhett "Jonzy" Jones.
 *		Modifed ProcessRequest() to adhere to the gopher protocol
 *		by sending error.host and -1 as the port if we encountered
 *		an error.  Done solely to prevent clients from crashing.
 *
 *		May 18, 1993, by Rhett "Jonzy" Jones.
 *		Added the #ifdef _POSIX_SOURCE in SendString() to prevent
 *		core dumps under A/UX.  Thank you hagberg@cumc.cornell.edu
 *
 *		November 14, 1993, by Rhett "Jonzy" Jones.
 *		Added use of "#ifdef NEXT_3.0" to support NeXT running 3.0.
 *		This patch was provided by hoymand@joe.uwex.edu
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Added the use of ERRORHOST to ProcessRequest() and now
 *		include jughead.cong.h.
 *
 *		May 5, 1994, by Rhett "Jonzy" Jones.
 *		Added support for Solaris compiler with use of #ifdef SVR4.
 *		Thanks cudma@csv.warwick.ac.uk.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Changed the use of #ifdef NEXT_3.0 to #ifdef NEXT.
 *
 *		May 10, 1994, by Rhett "Jonzy" Jones.
 *		Added use of GTSTRERR and READERR to assist in the
 *		localization of jughead to a given spoken language.
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code for lint.
 *
 *		May 23, 1994, by Rhett "Jonzy" Jones.
 *		Modified ContactHost() to no longer print except when
 *		debugging and altered the meaning of the return value.
 *		Added use of 'rptPtr' to write to the report file if
 *		need be.  Modified the returned value of SetUpReadNwriter()
 *		such that 0 implies no error, anyother value is the type
 *		of error.  Ensured we don't overwrite memory with use of
 *		MemoryCaution().
 *
 *		May 29, 1994, by Rhett "Jonzy" Jones.
 *		Added GetHostName() to support acquiring the name of
 *		the machine jughead is running on to make jughead more
 *		machine portable once compiled.
 *		Thank you speyer@mcc.com for the suggestion.
 *
 *		June 7, 1994, by Rhett "Jonzy" Jones.
 *		Modified GetHostName() to acquire the name in malloc space.
 *		Changed ERRORHOST to errorhost to support the new
 *		jughead.conf file.  Added use of 'hostname'.
 *
 *		June 28, 1994, by Rhett "Jonzy" Jones.
 *		Added #ifndef LINUX #include <sys/socketvar.h> to support
 *		linux compilation
 *
 *		July 23, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine SendBuffer() to support sending the
 *		datafile to veronica.
 *
 *		July 24, 1994, by Rhett "Jonzy" Jones.
 *		Removed inclusion of jughead.conf.h.
 *
 *		August 16, 1994, by Rhett "Jonzy" Jones.
 *		Modified SendString() to utilize fputs() instead of
 *		fprintf() depending on the appropriate #ifdef.
 *
 *		September 22, 1994, by Rhett "Jonzy" Jones.
 *		Added to routine PostContactHostError() solely for
 *		code consolidation.
 *
 *		September 24, 1994, by Rhett "Jonzy" Jones.
 *		Made TooMuchTime4Read() and env be static
 *		so they can exist with the same names in other
 *		files, and thus handle taking too long to read
 *		when playing a client.
 *
 *		November 16, 1994, by Rhett "Jonzy" Jones with code changes
 *		submitted by: paul@openage.com
 *		Added support for SCO compilation.
 *
 *		November 19, 1994, by Rhett "Jonzy" Jones with code changes
 *		submitted by: paul@openage.com
 *		Yipes!  Logic preprocessor error fixed.
 *
 * Description:	Contains routines dealing with sockets and communicating
 *		over the internet.
 *
 * Routines:		void	PostContactHostError(int error, char *hostStr,
 *							char *portStr);
 *			char	*GetHostName(char *hostDomainFromConf_h);
 *			void	IP2Hostname(int sockfd,char *host,char *ip);
 *			int	SetUpReadNwriter(int theSocket);
 *			void	CloseReadNwriter(void);
 *			int	ContactHost(char* theHost, int thePort);
 *			int	SendBuffer(char *buffer,int bufferSize);
 *			int	SendString(char *s);
 *			char	*GetString(FILE *ptr);
 *		static	void	TooMuchTime4Read(void);
 *			int	ListenerEstablished(int port);
 *			int	ProcessRequest(int sockfd);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <sys/types.h>
#include <stdlib.h>
#include <sys/socket.h>
#if !defined(LINUX) && !defined(SCO)
#	include <sys/socketvar.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/param.h>
#include <string.h>

#include "utils.h"

#ifdef SVR4
#	define bzero(p,l)	memset((p),0,(l))
#	define bcopy(p,q,l)	memmove((q),(p),(l))
#endif

#define READTIMEOUT	(5 * 60)
#define MAXLINE		512
#define RECVSIZE	4096
#define BUFFERSIZE	2048

static jmp_buf		env;

/* These are defined in "jughead.c". */
extern int	debug;
extern FILE	*wtPtr,
		*rdPtr,
		*rptPtr;
extern char	buffer[BUFFERSIZE];

extern char	*hostname,		/* Declared in "search.c". */
		*gtstrerr,
		*readerr;

extern char	*errorhost;		/* Declared in "search.c". */

extern void	PostPositions();	/* Defined in "search.c". */
extern void	LogMessage();		/* Defined in "search.c". */
extern void	MemoryCaution();	/* Defined in "search.c". */


/*****************************************************************************
 * PostContactHostError simply prints the result of any error encountered
 * with the call to ContactHost().
 ****************************************************************************/
void PostContactHostError(error,hostStr,portStr)
	int	error;		/* The error to report. */
	char	*hostStr,	/* The host we tried to connect to. */
		*portStr;	/* The port we used. */
{
	switch (error)
		{
		case -1:
			(void)fprintf(rptPtr,"error: SetUpReadNwriter could not open socket file for writing.\n");
			break;
		case -2:
			(void)fprintf(rptPtr,"error: SetUpReadNwriter could not open socket file for reading.\n");
			break;
		case -3:
			(void)fprintf(rptPtr,"error: ContactHost could not get the socket.\n");
			break;
		case -4:
			(void)fprintf(rptPtr,"error: ContactHost found unknown host [%s].\n",hostStr);
			break;
		case -5:
			(void)fprintf(rptPtr,"error: ContactHost could not connect to [%s] via port [%s].\n",
					hostStr,portStr);
			break;
		default:
			(void)fprintf(rptPtr,"error: ContactHost returned unknown error.\n");
			break;
		}

}	/* PostContactHostError */

/*****************************************************************************
 * GetHostName returns the name of the machine we are running on.
 * If HOSTDOMAIN is nil we conclude that the system call to hostname
 * returns our Fully Qualified Domain Name, otherwise HOSTDOMAIN
 * was set to append to what hostname returns to get the
 * Fully Qualified Domain Name.  The value returned is acquired in
 * malloc space.
 ****************************************************************************/
char *GetHostName(hostDomainFromConf_h)
	char		*hostDomainFromConf_h;	/* HOSTDOMAIN in jughead.conf.h */
{	char		theHostName[256],	/* Who we think we are. */
			*ourName;		/* The string we return. */
	struct hostent	*hostEntry;		/* Information about the host. */

	if (hostname)	/* Free up the memory. */
		{
		free(hostname);
		hostname = (char *)0;
		}

	theHostName[0] = '\0';
	if (gethostname(theHostName,256))	/* Yipes. */
		return((char *)0);
	if ((hostEntry = gethostbyname(theHostName)) && strlen(hostEntry->h_name) > strlen(theHostName))
		(void)strncpy(theHostName,hostEntry->h_name,256);
	else
		(void)strcat(theHostName,hostDomainFromConf_h);
	if (ourName = (char *)malloc((unsigned)strlen(theHostName) * sizeof(char) + 1))
		(void)strcpy(ourName,theHostName);
	return(ourName);

}	/* GetHostName */

/*****************************************************************************
 * IP2Hostname places the name of the host, as acquired from the IP
 * number, into 'host'.  This way we can log the name of the host instead
 * of the IP number.
 ****************************************************************************/
void IP2Hostname(sockfd,host,ip)
	int			sockfd;		/* Socket file descriptor. */
	char			*host;		/* The name of the host we are talking to. */
	char			*ip;		/* The hosts IP number. */
{	struct sockaddr_in	sin;		/* Info about the socket. */
	int			len;		/* The size of 'sin'. */
	struct hostent		*hostEntry;	/* Information about the host. */

	len = sizeof(sin);
	(void)getpeername(sockfd,&sin,&len);
	(void)strcpy(ip,(char *)inet_ntoa(sin.sin_addr));
	(void)strcpy(host,(char *)inet_ntoa(sin.sin_addr));

	hostEntry = gethostbyaddr((char *)&sin.sin_addr,sizeof(sin.sin_addr.s_addr),AF_INET);
     
	if (hostEntry)
		(void)strcpy(host,(char *)hostEntry->h_name);

}	/* IP2Hostname */

/*****************************************************************************
 * SetUpReadNwriter returns 0 if we can set up read and write pointers
 * to the socket 'theSocket'.  Otherwise the value returned is type of
 * error encountered, where -1 = couldn't setup for writing, and
 * -2 = couldn't setup for reading
 ****************************************************************************/
int SetUpReadNwriter(theSocket)
	int		theSocket;	/* The socket. */
{
	/* See if we can open a file for writing. */
	if (!(wtPtr = fdopen(theSocket,"w")))
		return(-1);

	/* See if we can open a file for reading. */
	if (!(rdPtr = fdopen(theSocket,"r")))
		return(-2);

	return(0);

}	/* SetUpReadNwriter */

/*****************************************************************************
 * CloseReadNwriter simply closes 'rdPtr' and 'wtPtr'.
 ****************************************************************************/
void CloseReadNwriter()
{
	(void)fclose(wtPtr);
	(void)fclose(rdPtr);

}	/* CloseReadNwriter */

/*****************************************************************************
 * ContactHost returns 0 if we were able to get a connection to 'theHost'
 * out port 'thePort', and open the files 'wrPtr' and 'rdPtr' for reading and
 * and writing information to and from 'theHost' out 'thePort'.  Otherwise
 * this routine returns a value representing the type of error encountered.
 * Returned values are:
 *   0 -> no error
 *  -1 -> SetUpReadNwriter could not open socket file for writing
 *  -2 -> SetUpReadNwriter could not open socket file for reading
 *  -3 -> ContactHost could not get the socket
 *  -4 -> ContactHost found unknown host 'theHost'
 *  -5 -> ContactHost could not connect 'theHost' via port 'thePort'
 ****************************************************************************/
int ContactHost(theHost,thePort)
	char			*theHost;	/* The host to connect with. */
	int			thePort;	/* The port to use. */
{	int			theSocket,	/* The socket. */
				error;		/* The error if encountered. */
	struct sockaddr_in	scktRec;	/* The socket address info. */
	struct hostent		*theHostEntry;	/* The host entry. */

	if (debug)
		(void)fprintf(rptPtr,"ContactHost attempting a connection to: %s %d\n",theHost,thePort);

	/* See if we can get a socket. */
	if ((theSocket = socket(PF_INET,SOCK_STREAM,0)) < 0)
		return(-3);

	/* Initialize our socket address information. */
	scktRec.sin_family = AF_INET;
	scktRec.sin_port = htons(thePort);
	if (theHostEntry = gethostbyname(theHost))
		bcopy(theHostEntry->h_addr,(char *)&scktRec.sin_addr.s_addr, 4);
	else
		return(-4);

	/* See if we can get a connection. */
	if (connect(theSocket,(struct sockaddr *)&scktRec,sizeof(scktRec)) < 0)
		return(-5);

	if (error = SetUpReadNwriter(theSocket))
		return(error);

	if (debug)
		(void)fprintf(rptPtr,"ContactHost has established connection\n");

	return(0);

}	/* ContactHost */

/*****************************************************************************
 * SendBuffer returns the number of bytes written to 'wtPtr', but first writes
 * the contents of 'buffer' to 'wtPtr' which is the machine we are talking to.
 ****************************************************************************/
int SendBuffer(buffer,bufferSize)
	char	*buffer;	/* The buffer we are sending to the machine. */
	int	bufferSize;	/* The number of bytes in 'buffer'. */
{	int	result;		/* The result of writting to the socket. */

	result = fwrite(buffer,1,bufferSize,wtPtr);
	(void)fflush(wtPtr);
	return(result);

}	/* SendBuffer */

/*****************************************************************************
 * SendString returns true no matter what, but first writes the string 's' to
 * 'wtPtr' which is the machine we are talking to.
 ****************************************************************************/
int SendString(s)
	char	*s;	/* The string we are sending to the machine. */
{
	(void)fputs(s,wtPtr);
	(void)fflush(wtPtr);
	return(1);

}	/* SendString */

/*****************************************************************************
 * GetString returns a line of text from either the socket or file we are
 * reading from, which is pointed to by the 'ptr'.  The string will contain at
 * most 'BUFFERSIZE' characters.  For more information on the contents of the
 * string returned by this routine consult the man page on 'fgets'.
 ****************************************************************************/
char *GetString(ptr)
	FILE	*ptr;	/* The file or socket we are reading from. */
{
	return(fgets(buffer,BUFFERSIZE,ptr));

}	/* GetString */

/*****************************************************************************
 * TooMuchTime4Read gets called only when the time period has elapsed when
 * waiting to read from our input..
 ****************************************************************************/
static void TooMuchTime4Read()
{
	if (debug)
		(void)fprintf(rptPtr,"In TooMuchTime4Read\n");

	longjmp(env,1);

}	/* TooMuchTime4Read */

/*****************************************************************************
 * ListenerEstablished returns -1 if we could NOT: create a socket, set up the socket
 * such that the address will be reused and will stick around for the client,
 * we can bind the socket to the port 'port', and we can set up to listen for
 * up to 5 connections out the port, a socket listening out port 'port'.
 * Otherwise we were able to do all of the above so we return the socket.
 ****************************************************************************/
int ListenerEstablished(port)
	int			port;		/* The port to use. */
{	struct sockaddr_in	sin;		/* The socket information. */
	struct linger		so_linger;	/* The SO_LINGER info. */
	int			so_reuseaddr,	/* The SO_REUSEADDR info. */
				sckt,		/* The socket to return. */
				error;		/* Did we get an error? */

	if (debug)
		(void)fprintf(rptPtr,"In ListenerEstablished\n");

	/* Initialize the variables. */
	so_reuseaddr = 1;
	error = so_linger.l_onoff = so_linger.l_linger = 0;
	bzero((char *)&sin,sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);
	sin.sin_port = htons(port);

	/* Create a socket binding the same local address and make sure the other will still be around. */
	if ((sckt = socket(AF_INET,SOCK_STREAM,0)) >= 0)
		if (setsockopt(sckt,SOL_SOCKET,SO_REUSEADDR,(char *)&so_reuseaddr,sizeof(so_reuseaddr)) >= 0)
			if (setsockopt(sckt,SOL_SOCKET,SO_LINGER,(char *)&so_linger,sizeof(so_linger)) >= 0)
				if (bind(sckt,(struct sockaddr *)&sin,sizeof(sin)) >= 0)
					if (listen(sckt,5) >= 0)
						return(sckt);
					else
						error = fprintf(rptPtr,"ListenerEstablished: can't listen 5\n");
				else
					error = fprintf(rptPtr,"ListenerEstablished: can't bind\n");
			else
				error = fprintf(rptPtr,"ListenerEstablished: can't setsockopt SO_LINGER\n");
		else
			error = fprintf(rptPtr,"ListenerEstablished: can't setsockopt REUSEADDR!\n");
	else
		error =  fprintf(rptPtr,"ListenerEstablished: can't socket\n");

	return(-error);

}	/* ListenerEstablished */

/*****************************************************************************
 * ProcessRequest gets and processes a request from a port.
 ****************************************************************************/
int ProcessRequest(sockfd)
	int		sockfd;			/* The socket file descriptor. */
{	char		*inputline,		/* The request to process. */
			bufr[1056];		/* Buffer for error messages. */
	int		length;			/* Length of the command line */

	if (SetUpReadNwriter(sockfd))
		{
		LogMessage(-1,"error: ProcessRequest could not setup read and writters");
		if (debug)
			(void)fprintf(rptPtr,"error: ProcessRequest could not setup read and writters\n");
		return(0);
		}

	/* Set things up so we don't wait for ever waiting to get a request. */
	(void)signal(SIGALRM,TooMuchTime4Read);
	(void)alarm(READTIMEOUT);

	if (setjmp(env))
		{
		MemoryCaution(strlen(gtstrerr) + strlen(errorhost),1056,
				"error: ProcessRequest attempted to overwrite memory");
		(void)sprintf(bufr,"%s%s",gtstrerr,errorhost);
		(void)SendString(bufr);
		exit(-1);
		}

	inputline = GetString(rdPtr);
	length = strlen(inputline);

	/* We got our request to deactivate the alarm. */
	(void)alarm(0);
	(void)signal(SIGALRM,SIG_IGN);

	if (length <= 0)
		{
		MemoryCaution(strlen(readerr) + strlen(errorhost),1056,
				"error: ProcessRequest attempted to overwrite memory");
		(void)sprintf(bufr,"%s%s",readerr,errorhost);
		(void)SendString(bufr);
		(void)close(sockfd);
		return(0);
		}

	(void)RemoveCRLF(inputline);

	if (debug)
		(void)fprintf(rptPtr,"ProcessRequest processing [%s]\n",inputline);

	PostPositions(sockfd,inputline);
	(void)SendString(".\r\n");
	CloseReadNwriter();
	return(1);

}	/* ProcessRequest */

