/*****************************************************************************
 * File:	search.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	March 6, 1993
 *
 * Modified:	March 15, 1993, by Rhett "Jonzy" Jones.
 *		Added the ability to handle the -HUP interrupt.
 *
 *		March 16, 1993, by Rhett "Jonzy" Jones.
 *		PrintPositions() will only return the collected items if
 *		the number of these items is less 1024.  Modified
 *		PostPositions() to watch for the case where A and B and C
 *		yeilds (A and B) = NIL, thus this result anded with C will
 *		no longer print the entire list C.  Oversight in my part.
 *
 *		March 27, 1993, by Rhett "Jonzy" Jones.
 *		Moved the definition of PORT2USE to the Makefile.
 *
 *		March 28, 1993, by Rhett "Jonzy" Jones.
 *		Added sun support with the inclusion of:
 *		#ifdef sun include <unistd.h>.
 *
 *		March 31, 1993, by Mic Kaczmarczik: mic@bongo.cc.utexas.edu
 *		Added support for an optional -p port flag when running as a
 *		search engine .  Added the variable 'searchPort as a parameter
 *		in DoSearch().
 *
 *		April 4, 1993, by Rhett "Jonzy" Jones.
 *		Added the variables elements2do, indexName, hashName, ifp, and
 *		hfp, the routines GetPositions(), WriteHashTables(), and
 *		MakeHashTables(),
 *		to support giving the user some feed back as to the status of
 *		the program when building the index tables and the use of a
 *		secondary hash table to limit the amount of memory used, and
 *		modified GetDisplayString(), CreateWordsTree()
 *
 *		May 5, 1993, by Rhett "Jonzy" Jones.
 *		Modified PostPositions(), and added GetAllPositions()
 *		to support partial word searches.
 *		Added the #define MAXITEMS2RETURN.
 *
 *		May 6, 1993, by Rhett "Jonzy" Jones.
 *		Modified LogMessage() to return if neither logFile or debug is
 *		true.  This was done for efficiency purposes only.
 *		Changed the loop counter in CleanUp() from an int to a long.
 *		Fixed a bug that caused a core dump if received the SIGHUP
 *		signal more than once.  This was rectified by making
 *		items[l].positions = NIL in CreateElements().  Oops!
 *		Fixed a potential bug in CreateElements() by making sure
 *		numElements
 *		gets a long value instead of a short.
 *		Added the #define SHOW_ITEMS_DURING_DEBUG to ease in debugging.
 *		Did some code cleaning up.
 *
 *		May 8, 1993, by Rhett "Jonzy" Jones.
 *		Added the routine PrintList(), and defined BOOLOP_DEBUG to
 *		assist in fixed a problem in PostPositions(), which gave
 *		incorrect results of "a* b* not c*".  Thank you
 *		doyle@liberty.uc.wlu.edu for bringing this to my attention.
 *
 *
 *		May 10, 1993, by Rhett "Jonzy" Jones.
 *		Modified PostPositions() to correct a problem of a search on
 *		"a and b and c", where if "a and b" evaluated to nothing, the
 *		result of anding this result with c produced c.  Example:
 *		"a and b and c" was the same as "a and b or c" if "a and b"
 *		evaluated to nothing.  Thanks doyle@liberty.uc.wlu.edu for
 *		bringing this to my attention.
 *		Changed the format of the logfile by altering the string
 *		"jughead-Search ->" to now be "jughead(port#) ->", where
 *		port# is the port jughead was started up with.  This change
 *		was at the request of doyle@liberty.uc.wlu.edu to indentify
 *		which jughead daemon did the logging when more than one
 *		jughead is logging to the same log file.  Added the variable
 *		'port2log' to support this feature.
 *
 *		May 12, 1993, by Rhett "Jonzy" Jones.
 *		doyle@liberty.uc.wlu.edu reported that "a and b" returns
 *		'b' if 'a' did not exist, so ... back to the drawing board to
 *		fix this one.
 *
 *		May 14, 1993, by Rhett "Jonzy" Jones.
 *		doyle@liberty.uc.wlu.edu reported that "a and b" returns
 *		'a' if 'b' did not exist.  I believe things are fixed now.
 *
 *		May 20, 1993, by Rhett "Jonzy" Jones.
 *		Added the inclusion of sys/types.h, sys/socket.h, time.h,
 *		and pwd.h.  Gave support for the -u username to change the
 *		process uid when running jughead as a search engine.
 *		Added the routine VerifyDataBaseName() to allow verification
 *		of the database while allowing for the database to reside in
 *		a directory other than jughead.  Thank you rzakon@mitre.org
 *		for bringing this to my attention.
 *
 *		May 22, 1993, by Rhett "Jonzy" Jones.
 *		Added DEFAULTBOOLOP, which is defined in the Makefile to
 *		allow for an easy change of the default boolean operator
 *		to use when no operator seperates word to search for.
 *
 *		May 23, 1993, by Rhett "Jonzy" Jones.
 *		Gave support for special commands.  For information on these
 *		commands consult either "searchCmnds.c", the man page, or
 *		"About.jughead".
 *
 *		May 23, 1993, by Rhett "Jonzy" Jones.
 *		At the request of doylej@liberty.uc.wlu.edu modified
 *		PrintPositions() to return a link when there are more than
 *		MAXITEMS2RETURN items.  The link has the form:
 *			Name=xxx items found.  Please consolidate your request
 *			Type=1
 *			Port=the port number jughead was started under
 *			Path=?all [the requested search]
 *			Host=THEHOST as defined in the Makefile
 *
 *
 *		May 24, 1993, by Rhett "Jonzy" Jones.
 *		At the request of doylej@liberty.uc.wlu.edu gave support
 *		for the "?range=start-stop what" special command.
 *
 *		November 14, 1993, by Rhett "Jonzy" Jones.
 *		Modified the code to support the 'm' flag when building the
 *		the hash tables.  This way no output will go to the screen
 *		and can thus help speed up a lengthy process.
 *		Support for this was requested by: doylej@liberty.uc.wlu.edu
 *
 *		December 8, 1993, by Rhett "Jonzy" Jones.
 *		Fixed a problem, which frutig@cr-sp.rnp.br reported, that
 *		resulted in an imcompatible database when using the -m flag
 *		inconjuction with the -B flag.
 *
 *		February 25, 1994, by Rhett "Jonzy" Jones.
 *		Added the use of MAXPROCS and numProcs to support the maximum
 *		number of forked jughead processes.  Renamed and rewrote
 *		ChildProcess() to ReapChild() to support maximum number of
 *		processes to fork off.  Added PostMsg2Socket() to send error
 *		messages back to the client.
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		May 1, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code by changing asterikPos in GetAllPositions()
 *		and PostPositions() from type size_t to an int.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Moved the definition of DELIMITERS and MAXITEMS2RETURN
 *		to jughead.conf.h and now include jughead.conf.h.
 *		Optimized some of the code.  Aded the use of ERRORHOST
 *		in GetAllPositions() and PostMsg2Socket().
 *
 *		May 5, 1994, by Rhett "Jonzy" Jones.
 *		Added code to automagicly run jughead in the background
 *		to detach itself from a tty, when not running with
 *		debugging turned on.  Done with the MakeDaemon() routine.
 *		Added support for Solaris compiler with use of #ifdef SVR4.
 *		Thanks cudma@csv.warwick.ac.uk.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Added use of #ifdef SOLARIS2 for better Solaris support/
 *		Thanks Maes@elec.ucl.ac.BE.
 *		Changed all calls to strcmp() to StrCmp() and strncmp()
 *		to StrnCmp() to support both ASCII and ISO-XXXX-Y character
 *		sets.  Removed use of LOCALE.
 *
 *		May 9, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine WritePIDfile() to write the jughead.pid
 *		file to assist in scripting and sending SIGHUP or starting
 *		jughead.  Thank you Kuypers@sri.ucl.ac.BE for the idea.
 *
 *		May 10, 1994, by Rhett "Jonzy" Jones.
 *		Added use of WILDCARDERR, TOMNYPRCSERR, and NOFORKERR to
 *		assist in the localization of jughead for a given spoken
 *		language.  A whole slew of code cleansing.
 *		Commented out VerifyDataBaseName(), because it needs some
 *		cleaning up.
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code for lint.
 *
 *		May 23, 1994, by Rhett "Jonzy" Jones.
 *		Added use of 'rptPtr' to write to this file instead of
 *		stderr if the -r flag is used.  Ensured we don't
 *		overwrite memory with the addition of MemoryCaution().
 *
 *		May 29, 1994, by Rhett "Jonzy" Jones.
 *		Changed use of HOSTDOMAIN to 'hostname' to assist in
 *		porting the compiled jughead to similar machines.
 *		Thanks speyer@mcc.com for the suggestion.
 *
 *		June 7, 1994, by Rhett "Jonzy" Jones.
 *		Many changes to support the jughead.conf file and reading
 *		this file, which contains variables, at SIGHUP.
 *
 *		June 11, 1994, by Rhett "Jonzy" Jones.
 *		Added #ifndef BSD to give support for BSD.
 *
 *		June 17, 1994, by Rhett "Jonzy" Jones.
 *		Added the #ifdef NEXT in MakeDaemon() to support
 *		NeXT system version 2.1.
 *
 *		June 24, 1994, by Rhett "Jonzy" Jones.
 *		Modified HangUpSignal() to touch /usr/local/etc/jughead.pid
 *		to allow a make script to run via cron and a dependency
 *		will exist to SIGHUP jughead.
 *
 *		July 24, 1994, by Rhett "Jonzy" Jones.
 *		Added the following variables so all variables in
 *		jughead.conf can be overrided via jughead.conf:
 *		delimiters, defaultboolop, jugheadhelptitl, usagerror,
 *		wildcarderr, tomnyprcserr, noforkerr, gtstrerr, readerr,
 *		tmpfilename, catcommand, rmcommand, sortcommand, touchcommand,
 *		port2use, maxprocs, maxitems2return.
 *		Removed 'searchPort' as a parameter to DoSearch() because this
 *		variable is now called 'port2use' and defined localy.
 *		Renamed 'port2log' to 'port2use'.  Why did I name it 'port2log'?
 *		Removed the InitSearchEngine() routine which now gets handled
 *		via the call to ReadConfFile().
 *		Removed inclusion of jughead.conf.h.
 *
 *		August 16, 1994, by Rhett "Jonzy" Jones.
 *		Fixed a problem of not being able to SIGHUP jughead when
 *		jughead was started via rc.local.
 *		Thank you Kuypers@sri.ucl.ac.be for bringing this to my
 *		attention, and thank you ldl@cs.utah.edu for finding the
 *		errors of my ways.
 *
 *		September 13, 1994, by Rhett "Jonzy" Jones.
 *		Added inclusion of <sys/errno.h> and <errno.h>,and modified
 *		DoSearch() to support removal of the <defunct> process which
 *		would exist until a connection was received.  Support for
 *		this feature will only occur when "#define NODEFUNCT"
 *		is not in comments or has been issued from the command line
 *		with the "make NODEFUNCT=-DNODEFUNCT", or "make nodefunct"
 *		command.
 *
 *		September 22, 1994, by Rhett "Jonzy" Jones.
 *		Minor change to report the name of the veronica file
 *		when in debug mode, and slight change to the way
 *		ReadConfFile() is called to support the "Disallow:"
 *		directive.
 *
 *		September 27, 1994, by Rhett "Jonzy" Jones.
 *		Removed all use of rm.
 *
 * Description:	Either builds an index into a datafile, where the datafile
 *		contains lines of gopher menu items (via jughead), or performs
 *		boolean searches on words from the display string in the
 *		datafile as read from the resultant index file.
 *
 *		When building the index we read from the datafile and for
 *		each line we acquire the current line position in the file via
 *		ftell(), and extract the display string excluding the first
 *		character which is the item type, and break the display string
 *		into words.  We then dump the word followed by a tab and the
 *		position to a temporary file.  We then read from the temporary
 *		file and build a binary tree containing words and a list of
 *		postions, which is the line the word came from.  No word is
 *		duplicated in the tree.  And finaly dump the binary tree to
 *		a file with the following format:
 *		dataFileNameIndexWasBuiltFrom	numberOfnodes
 *		lenWord0	word0	0	1	3
 *		lenWord1	word1	2	5
 *		...
 *		lenThisWord	word_numberOfnodes-1	m	n
 *
 *		As of April 4, 1993, a hash table and an index table get built
 *		where the hash table as the following format:
 *		dataFileNameIndexWasBuiltFrom	numberOfnodes
 *		lenWord0	word0	position_in_index_table
 *		lenWord1	word1	position_in_index_table
 *		...
 *		lenThisWord	word_numberOfnodes-1	position_in_index_table
 *
 *		and the index table has the following format:
 *		number_of_positions	position1...position-number_of_positions
 *
 *		where: dataFileNameIndexWasBuiltFrom is the name of the data
 *		file this index was built from, numberOfnodes is the number of
 *		nodes in the tree, lenWordX is the number of characters in wordX
 *		including the terminating null, wordX is the word followed by
 *		the position in the data file the word came from.
 *
 *		When reading from the index file and doing boolean operations,
 *		we read the index into memory, and acquire the string the user
 *		wants to do a search on.  This string is then broken into words,
 *		using the same mechanisim as breaking the display string into
 *		words.  If a single word is found we print all lines from the
 *		datafile in which the word exists in the display string.  If two
 *		words are found it is taken to be word1 AND word2.  The boolean
 *		operations currently supported are AND, NOT, OR.  All boolean
 *		operations are evaluated left to right.
 *
 *		WARNING:  Attempting to change AND, NOT or OR to a set of
 *		different characters will violate the gopher protocol, and
 *		jughead will not work in future versions.
 *
 * Routines:		void	 LogMessage(int sockfd,char *message);
 *			void	 MemoryCaution(int theSize,int limit,char *msg);
 *			char	 *GetDisplayString(FILE *fp,long *pos);
 *			int	 CreateWordsTree(char *fileName);
 *		static	ListType *GetPositions(long index);
 *		static	void	 WriteHashTables(TreeType *node);
 *			void	 MakeHashTables(char *fileName,TreeType *root);
 *			short	 ReservedWord(char *word);
 *			short	 ParseSearchString(char *string);
 *		static	void	 PrintPositions(ListType *node);
 *		static	void	 PrintList(ListType *l);
 *			ListType *DoOperation(short op,ListType *l1,
 *					      ListType *l2);
 *			void	 LogRequest(int sockfd);
 *		static ListType	 *GetAllPositions(long index,char *what2Find,
 *						  char *asterik,
 *						  int asterikPos,int sockfd);
 *			void	 PostPositions(char *what2find);
 *		static	int	 VerifyDataBaseName(char *fName,char *dName);
 *			short	 CreateElements(char *fileName);
 *			void	 CleanUp(void);
 *			void	 HangUpSignal(void);
 *			void	 DoSearch(char *cmdLine,char *indexTable,
 *					  char *logFile);
 *			void	 ReapChild(void);
 *			int	 PostMsg2Socket(char *str,int s);
 *		static	int	 MakeDaemon(void);
 *		static	int	 WritePIDfile(char *cmdLine);
 *				 int searchPort);
 *
 * Bugs:	The call to IP2Hostname() has the potention to overwrite
 *		memory.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

/* Uncomment the "#define NODEFUNCT" line if you don't want the <defunct>
 * process hanging around, or do a "make NODEFUNCT=-DNODEFUNCT".
 * NOTE:  Using "NODEFUNCT" will eat CPU cycles as long as there is a child
 *        process which is forked off.
 *        Not using "NODEFUNCT" is the more efficient way to use jughead.
 *        The <defunct> process will be cleaned up the next time a connection
 *        is made, but will return as soon as the child processed as terminated
 *        and will not be taken care of until the ReapChild() routine can
 *        process it after another connection. */
/* #define NODEFUNCT */

#include <pwd.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef NODEFUNCT
#	include <sys/errno.h>
#	include <errno.h>
#endif
#ifdef SVR4
#	include <wait.h>
#else
#	include <sys/wait.h>
#endif
#include <time.h>
#ifdef NEXT
#	include <libc.h>
#else
#	include <stdlib.h>
#endif
#include <string.h>
#include <fcntl.h>
#include <netinet/in.h>
#ifdef sun
#	include <unistd.h>
#endif

#include "sockets.h"
#include "tree.h"
#include "utils.h"

/* Uncomment this if you want to look at the items after they are built. */
/* #define SHOW_ITEMS_DURING_DEBUG */

/* Uncomment this to assist in debugging the boolean operations. */
/* #define BOOLOP_DEBUG */

#define NOOP		 0	/* The NOOP operation. */
#define AND		 1	/* The AND operation. */
#define OR		 2	/* The OR  operation. */
#define NOT		 3	/* The NOT operation. */
#define MAXWRDSNCMNDS	20	/* The allowable words and commands. */

#define HASHEXT		".ih"	/* The hash table extention. */
#define INDXEXT		".ix"	/* The index table extention. */
#define NINEBACKSPACES	"\b\b\b\b\b\b\b\b\b"

Element		*items = (Element *)NIL;	/* Array of words and file postions. */
long		numElements,			/* Number of elements in 'items'. */
		elements2do;			/* Number of elements left to do. */
char		dataFileName[512],		/* The name of the data file. */
		indexName[512],			/* Name of the index file. */
		hashName[512],			/* Name of the hast file. */
		*logFile = (char *)NIL,		/* Name of the file to log to. */
		*wrdsNcmds[MAXWRDSNCMNDS + 1],	/* Array of words and commands. */
		*hostname = (char *)0,		/* Name of the machine jughead is running on. */
		*jugheadhelp = (char *)0,	/* Location of the "About.jughead" document. */
		*errorhost = (char *)0,		/* Could be "jugheadhelp" or "\t\terror.host\t-1". */
		*delimiters = (char *)0,	/* Character to delimit a word when doing a search. */
		*defaultboolop = (char *)0,	/* The default boolean operation to perform. */
		*jugheadhelptitl = (char *)0,	/* The title to the jughead help document. */
		*usagerror = (char *)0,		/* The usage error string. */
		*wildcarderr = (char *)0,	/* Invalid wildcard usage string. */
		*tomnyprcserr = (char *)0,	/* Too may processes string. */
		*noforkerr = (char *)0,		/* The could not fork message. */
		*gtstrerr = (char *)0,		/* The get string timed out mesage. */
		*readerr = (char *)0,		/* The read error message. */
		*tmpfilename = (char *)0,	/* The temporary file to use. */
		*catcommand = (char *)0,	/* The cat command. */
		*sortcommand = (char *)0,	/* The sort command. */
		*touchcommand = (char *)0;	/* The touch command. */
int		port2use,			/* The port to use as a search engine. */
#ifdef NODEFUNCT
		noDefunct = 1,			/* Do we remove the <defunct> process? */
#endif
		maxprocs,			/* Maximum number of jughead processes. */
		maxitems2return;		/* The maximum items to return. */
short		numCommands;			/* The number of words and commands. */
jmp_buf		hupbuf;				/* The place to go if -HUP encountered. */
FILE		*ifp = (FILE *)NIL,		/* Pointer to the index file. */
		*hfp = (FILE *)NIL;		/* Pointer to the hash file. */

/* The following are defined in "jughead.c". */
extern FILE	*rptPtr;			/* File pointer for error messages. */
extern char	*userName,			/* Name of the user to run jughead under. */
		*veronica;			/* Name of the veronica file. */
extern int	debug,				/* Are we debugging? */
		time2process,			/* Do we calculate the time for a certain run? */
		menuFlag;			/* Do we display the status as we process? */
extern time_t	startTime;			/* The time a run was started, for use with 'time2process'. */

extern void	PostTime2Process();		/* Defined in "jughead.c". */

extern char	*MakeStr();			/* Defined in "jugheadConf.c". */
extern void	NullifyDisallowList();

extern char	*vctlHost,			/* Defined in "jugheadVctl.c". */
		*vctlPort;
extern void	PrintTheList();

extern long	lineNumber;			/* Defined in "dirTree.c". */

extern char	*SpecialCommand();		/* Defined in "searchCmnds.c". */

void MemoryCaution();

/*****************************************************************************
 * LogMessage logs the message 'message' to the end of the log file 'logFile'.
 * The logging is the same as that found in gopherd.c from the University of
 * Minnesota.
 ****************************************************************************/
void LogMessage(sockfd,message)
	int		sockfd;		/* The socket file descriptor. */
	char		*message;	/* The message to log. */
{	static char	hostName[256],	/* Name of the host we are talking to. */
			ip[256];	/* The hosts IP number.  Won't kill mem. */
	struct flock	lock;		/* Info to lock the log file. */
	time_t		theTime;	/* The current time. */
	char		*cTime,		/* The calendar time. */
			*lineFeed,	/* Location of the line feed. */
			logEntry[2048];	/* The entry to place in the log. */
	int		theLog = -1;	/* The file to log to. */

	if (!logFile && !debug)		/* No sense is using the CPU. */
		return;

	if (logFile)	/* Open the sucker. */
		theLog = open(logFile,O_WRONLY | O_APPEND | O_CREAT,0644);

	hostName[0] = '\0';

	/* DANGER DANGER DANGER: if hostName or ip > 256 this will hoze memory. */
	if (sockfd > -1)
		IP2Hostname(sockfd,hostName,ip);

	(void)time(&theTime);
	cTime = ctime(&theTime);
	if (lineFeed = strchr(cTime,'\n'))	/* Get rid of the line feed. */
		*lineFeed = '\0';

	MemoryCaution(strlen(hostName) + strlen(message),2000,
			"error: LogMessage() attempted to overwrite memory");
	(void)sprintf(logEntry,"%s %d %s : %s\n",cTime,getpid(),hostName,message);

	if (theLog != -1)
		{
		lock.l_type = F_WRLCK;
		lock.l_whence = SEEK_SET;
		lock.l_start = lock.l_len = 0L;
		(void)fcntl(theLog,F_SETLKW,&lock);		/* Lock the file so no one can write to it. */
		(void)lseek(theLog,0L,SEEK_END);		/* Make sure we are at the end of file. */
		(void)write(theLog,logEntry,strlen(logEntry));
		lock.l_type = F_UNLCK;
		(void)fcntl(theLog,F_SETLKW,&lock);		/* Unlock the file. */
		}

	if (logFile)	/* I guess we can close it now. */
		(void)close(theLog);

	if (debug)
		(void)fprintf(rptPtr,"%s\n",logEntry);


}	/* LogMessage */

/*****************************************************************************
 * MemoryCaution prints the message 'msg' to the logfile and aborts the
 * program if 'theSize' is greater the 'limit'.  Otherwise this routine
 * does nothing except ensure we don't overwrite memory.
 ****************************************************************************/
void MemoryCaution(theSize,limit,msg)
	int	theSize,	/* The size we are checking. */
		limit;		/* The limit 'size' can be. */
	char	*msg;		/* The error message to post if need be. */
{
	if (theSize > limit)
		{
		LogMessage(-1,msg);
		if (debug)
			(void)fprintf(rptPtr,"%s\n",msg);
		exit(-1);
		}

}	/* MemoryCaution */

/*****************************************************************************
 * GetDisplayString returns the display string portion of the gopher line
 * contained in 'str'.  I should mention the item type, which is the first
 * character in the display string, is skipped.
 ****************************************************************************/
char *GetDisplayString(fp,pos)
	FILE		*fp;		/* The file we are reading. */
	long		*pos;		/* Position in the file of the line. */
{	static char	buf[2048];	/* Buffer for the display string. */
	size_t		len;		/* Number of characters to the tab. */
	static long	line = 0;	/* The line number in the file for error reporting. */
	char		*s;		/* Pointer to the line we acquire from the file. */
	short		error = 0;	/* Did we encounter an error? */

	if (!(--elements2do % 10) && menuFlag)
		(void)fprintf(stdout,"%s%9ld",NINEBACKSPACES,elements2do);

	*pos = ftell(fp);
	line++;

	if (fgets(buf,2048,fp))
		if (s = strchr(buf,'\t'))
			if ((len = (size_t)(s - buf)) > 0)
				{
				buf[len] = '\0';
				return(buf + 1);
				}
			else
				error = 1;
		else
			error = 1;

	if (error)
		{
		(void)fprintf(rptPtr,"%swarning: GetDisplayString found bad line, line = %ld.\n         ",NINEBACKSPACES,line);
		pos = 0;
		return("");
		}

	return((char *)NIL);

}	/* GetDisplayString */

/*****************************************************************************
 * CreateWordsTree creates a tree where each node of the tree contains a
 * word and a list of file positions, representing the line the word is
 * contained in.  It should be mentioned that the words are parsed from
 * the display string according to the 
 ****************************************************************************/
int CreateWordsTree(fileName)
	char	*fileName;	/* Name of the data file. */
{	long	position;	/* Position in the file of the current line. */
	FILE	*fpIn;		/* The data file we area reading from .*/
	char	*dStr,		/* The display string with no leading item type. */		
		*word;		/* A word from dStr. */
	int	error = 0;	/* Did we get an error? */

	if (fpIn = fopen(fileName,"r"))
		{
		if (menuFlag)
			{
			(void)fprintf(stdout,"Building the words tree...\n");
			(void)fprintf(stdout,"%9ld",elements2do = NumberOfLines(fileName));
			}

		while (dStr = GetDisplayString(fpIn,&position))
			if (*dStr)
				for (word = strtok(dStr,delimiters); word; word = strtok((char *)NIL,delimiters))
					BuildTree(&root,StrToLower(word),position);
		(void)fclose(fpIn);
		if (menuFlag)
			(void)fprintf(stdout,"%swords tree is now built.\n",NINEBACKSPACES);
		}
	else
		error = fprintf(rptPtr,"error: CreateWordsTree could not open %s\n",fileName);

	return(!error);

}	/* CreateWordsTree */
	
/*****************************************************************************
 * GetPositions returns a list of the file positions, which contains the
 * gopher information to send the client we are talking to, or do a given
 * operation on.
 ****************************************************************************/
static ListType *GetPositions(index)
	long		index;
{	FILE		*fp;
	ListType	*list = (ListType *)NIL;
	register int	i;
	int		numPos;
	long		where;

	if (debug)
		(void)fprintf(rptPtr,"In GetPositions with index = %ld, items[%ld].word = [%s], items[%ld].positions = %ld\n",
				index,index,items[index].word,index,items[index].positions->where);
	if (fp = fopen(indexName,"r"))
		{
		if (!fseek(fp,items[index].positions->where,SEEK_SET))
			{
			numPos = GetInt(fp);
#ifdef BOOLOP_DEBUG
			if (debug)
				(void)fprintf(rptPtr,"\tnumPos = %d\n",numPos);
#endif
			for (i = 0; i < numPos; i++)
				{
				where = GetLong(fp);
#ifdef BOOLOP_DEBUG
				if (debug)
					(void)fprintf(rptPtr,"\twhere = %ld\n",where);
#endif
				list = BuildList(list,where);
				}
			}
		else
			LogMessage(-1,"error: GetPositions had fseek fail");
		(void)fclose(fp);
		}
	else
		{
		char	s[512 + 50];	/* Make sure we don't walk on memory. */
		(void)sprintf(s,"error: GetPositions could not open %s for reading",indexName);
		LogMessage(-1,s);
		}

	return(list);

}	/* GetPositions */

/*****************************************************************************
 * WriteHashTables writes the information within the tree pointed to by 'node'
 * to the hash table and the index table.
 ****************************************************************************/
static void WriteHashTables(node)
	TreeType	*node;		/* The node to process. */
{	ListType	*positions;	/* List with the positions. */

	if (node)
		{
		WriteHashTables(node->left);

		if (!(--elements2do % 10) && menuFlag)
			(void)fprintf(stdout,"%s%9ld",NINEBACKSPACES,elements2do);

		(void)fprintf(hfp,"%d\t%s\t%ld\n",(int)strlen(node->word) + 1,node->word,(long)ftell(ifp));

		(void)fprintf(ifp,"\t%ld",NumberOfListNodes(node->positions));
		for (positions = node->positions; positions; positions = positions->next)
			(void)fprintf(ifp,"\t%ld",positions->where);
		(void)fprintf(ifp,"\n"),++lineNumber;

		WriteHashTables(node->right);
		}

}	/* WriteHashTables */

/*****************************************************************************
 * MakeHashTables creates the hash and index table such that the hash table
 * contains the data file to index, the number of elements to create, followed
 * by each line containing the size of the word plus 1 for the null character,
 * a tab, the word, tab, and the file position in the the index table.  The
 * index table contains the number of file positions, tab, the file positions
 * seperated by tabs on a single line.  Each file position in the index table
 * is the position in the data file where the word found in the hash table
 * is referencing.
 ****************************************************************************/
void MakeHashTables(fileName,root)
	char		*fileName;	/* Name of the data file.*/
	TreeType	*root;		/* The root of the tree. */
{
	if (menuFlag)
		(void)fprintf(stdout,"Building the hash tables...\n");

	/* Make sure we don't overwrite memory. */
	if ((strlen(fileName) + strlen(INDXEXT)) > 512)
		{
		(void)fprintf(rptPtr,"error: attempt to overwrite memory.  %s too long\n",fileName);
		exit(-1);
		}
	if ((strlen(fileName) + strlen(HASHEXT)) > 512)
		{
		(void)fprintf(rptPtr,"error: attempt to overwrite memory.  %s too long\n",fileName);
		exit(-1);
		}

	(void)strcpy(indexName,fileName);
	(void)strcat(indexName,INDXEXT);
	(void)strcpy(hashName,fileName);
	(void)strcat(hashName,HASHEXT);

	if (!(ifp = fopen(indexName,"w")))
		(void)fprintf(rptPtr,"error: MakeHashTable could not open %s for writing\n",indexName);
	if (!(hfp = fopen(hashName,"w")))
		(void)fprintf(rptPtr,"error: MakeHashTable could not open %s for writing\n",hashName);
	if (!ifp || !hfp)
		exit(-1);

	(void)fprintf(hfp,"%s\t%ld\n",fileName,elements2do = NumberOfLeafs(root));
	if (menuFlag)
		(void)fprintf(stdout,"         ");	/* To support the use of the backspaces. */
	WriteHashTables(root);

	if (menuFlag)
		(void)fprintf(stdout,"%shash tables are completed.\n",NINEBACKSPACES);

	(void)fclose(ifp);
	(void)fclose(hfp);
	ifp = hfp = (FILE *)NIL;

}	/* MakeHashTables */

/*****************************************************************************
 * ReservedWord returns true if 'word' is "AND", "OR", or "NOT", otherwise
 * it returns false.  If 'word' is "AND" this routine returns 1, if 'word' is
 * "OR" we return 2, if 'word' is "NOT" 3 gets returned.
 * WARNING:  Any attempt to change the set of characters representing AND,
 * OR, or NOT will not only violate gopher protocol but will also make
 * jughead fail in future versions.
 ****************************************************************************/
short ReservedWord(word)
	char	*word;			/* The word we are checking if reserved. */
{	int	len = strlen(word);	/* The length of 'word'. */

	switch (len)
		{
		case 2:
			if ((word[0] == 'O' || word[0] == 'o') &&
			    (word[1] == 'R' || word[1] == 'r'))
	    			{
			    	word[0] = 'O'; word[1] = 'R';
				return(OR);
				}
			break;
		case 3:
			if ((word[0] == 'A' || word[0] == 'a') &&
			    (word[1] == 'N' || word[1] == 'n') &&
			    (word[2] == 'D' || word[2] == 'd'))
		    		{
		    		word[0] = 'A'; word[1] = 'N'; word[2] = 'D';
				return(AND);
				}
			else if ((word[0] == 'N' || word[0] == 'n') &&
				 (word[1] == 'O' || word[1] == 'o') &&
				 (word[2] == 'T' || word[2] == 't'))
		    		{
		    		word[0] = 'N'; word[1] = 'O'; word[2] = 'T';
				return(NOT);
				}
			break;
		default:
			break;
		}

	return(NOOP);

}	/* ReservedWord */

/*****************************************************************************
 * ParseSearchString parses 'string' into words and or commands which we do
 * searches and boolean searches on.  All words and commands get placed into
 * the array 'commands'.  If any word is "and", "or", or "not" it is taken to
 * be the command AND, OR, or NOT respectivly.  If any 2 words are not
 * seperated with a command, the command AND is implied to be the seperating
 * command.
 ****************************************************************************/
short ParseSearchString(string)
	char		*string;	/* The string we are parsing. */
{	char		*word;		/* The word extracted from 'string'. */
	short		lastOneReserved,/* Was the last word reserved? */
			thisOneReserved;/* Is the current word reserved? */

#ifdef IN_THE_FUTURE
	/* In the future the database to use will exist before the tab. */
	if (time2process)
		(void)time(&startTime);

	if (!CreateElements(indexTable))
		{
		if (!debug)
			(void)fprintf(rptPtr,"error: DoSearch could not create index tree.\n");
		LogMessage(-1,"error: DoSearch could not create index tree");
		exit(-1);
		}

	if (time2process)
		PostTime2Process();
#endif

	for (numCommands = 0, lastOneReserved = 1, word = strtok(string,delimiters);
	     numCommands < MAXWRDSNCMNDS && word;
	     word = strtok((char *)NIL,delimiters))
		{
		thisOneReserved = ReservedWord(word);
		if (!lastOneReserved && !thisOneReserved)
			{
			wrdsNcmds[numCommands++] = defaultboolop;
			wrdsNcmds[numCommands++] = StrToLower(word);
			}
		else if (thisOneReserved)
			wrdsNcmds[numCommands++] = word;
		else
			wrdsNcmds[numCommands++] = StrToLower(word);
		lastOneReserved = thisOneReserved;
		}

	return(numCommands);

}	/* ParseSearchString */

/*****************************************************************************
 * PrintPositions sends the line of text from the data file, the index table
 * was built from, whose postion in the file is specified by 'node->where'
 * to stdout.
 ****************************************************************************/
static void PrintPositions(sockfd,node,limit,rangeStart,rangeEnd)
	int		sockfd;		/* The socket file descriptor. */
	ListType	*node;		/* The list we are printing. */
	long		limit,		/* The max number of items to return. */
			rangeStart,	/* The start of a range. */
			rangeEnd;	/* The end of a range. */
{	FILE		*fp;		/* Pointer to the file with the actual data. */
	char		buf[2048];	/* Buffer for the line of text. */
	register long	i;		/* A loop counter. */
	long		n;		/* The number of items to return. */

	/* If too many items create some links on the fly and send'em.*/
	if (!rangeStart && limit == maxitems2return && (n = NumberOfListNodes(node)) > maxitems2return)
		{
		char	s[1024],	/* A temporary string. */
			what[1024];	/* The search command - any special command. */

		/* Handle the "All 'n' items" directory. */
		(void)sprintf(s,"1All %ld items\t?all",n);	/* <- Won't trash memory. */
		(void)SendString(s);

		for (what[0] = i = 0; i < numCommands; i++)
			{
			MemoryCaution(strlen(what) + strlen(wrdsNcmds[i]) + 1,1024,
					"error: PrintPositions() attempted to overwrite memory");
			(void)strcat(what," ");
			(void)strcat(what,wrdsNcmds[i]);
			}

		(void)SendString(what);
		(void)sprintf(s,"\t%s\t%d\r\n",hostname,port2use);	/* Won't trash mem. */
		(void)SendString(s);

		/* Now handle the range directories. */
		for (i = 0; i < n; i += limit)
			{
			rangeStart = i + 1;
			if ((i + limit) < n)
				rangeEnd = i + limit;
			else
				rangeEnd = n;
			(void)sprintf(s,"1items %ld to %ld\t?range=%ld-%ld %s\t%s\t%d\r\n",
				rangeStart,rangeEnd,rangeStart,rangeEnd,
				what,hostname,port2use);		/* Won't trash mem. */
			(void)SendString(s);
			}

		(void)sprintf(s,"TOO MANY ITEMS: %ld items found",n);	/* Won't trash mem. */
		LogMessage(sockfd,s);
		return;
		}

	if (fp = fopen(dataFileName,"r"))
		{
		if (rangeStart)
			{
			/* Traverse node to the rangeStart item. */
			for (i = 0; i < rangeStart - 1 && node; i++, node = node->next);

			/* Spit out range of items. */
			for ( ; i < rangeEnd && node; i++, node = node->next)
				if (!fseek(fp,node->where,SEEK_SET))
					(void)SendString(fgets(buf,2048,fp));
			}
		else if (limit < 0)
			while (node)
				{
				if (!fseek(fp,node->where,SEEK_SET))
					(void)SendString(fgets(buf,2048,fp));
				node = node->next;
				}
		else for (n = 0; n < limit && node; n++, node = node->next)
			if (!fseek(fp,node->where,SEEK_SET))
				(void)SendString(fgets(buf,2048,fp));

		(void)fclose(fp);
		}
	else
		{
		char	s[512];
		MemoryCaution(strlen(dataFileName),450,
				"error: PrintPositions() attempted to overwrite memory");
		(void)sprintf(s,"error: PrintPositions could not open data file [%s]",dataFileName);
		LogMessage(sockfd,s);
		}

}	/* PrintPositions */

#ifdef BOOLOP_DEBUG
/*****************************************************************************
 * PrintList prints the list 'l', and was written solely for debugging.
 ****************************************************************************/
static void PrintList(l)
	ListType	*l;		/* The list to print. */
{
	if (debug)
		while (l)
			{
			(void)fprintf(rptPtr,"%10ld,",l->where);
			l = l->next;
			}

}	/* PrintList */
#endif

/*****************************************************************************
 * DoOperation returns a list which is the result of "l1 op l2" where 'l1' and
 * 'l2' are lists, and 'op' is the operation to perform which is either the
 * AND, OR, or NOT operation.
 ****************************************************************************/
ListType *DoOperation(op,l1,l2)
	short		op;		/* The operation [AND,OR,NOT]. */
	ListType	*l1,		/* A list we operate on. */
			*l2;		/* The other list we operate on. */
{	ListType	*result = (ListType *)NIL,/* The result of "l1 op l2". */
			*t = (ListType *)NIL;	/* A pointer into 'l1' or 'l2'. */

	switch (op)
		{
		case AND:
			while (l1 && l2)
				if (l1->where == l2->where)
					{
					result = BuildList(result,l1->where);
					l1 = l1->next;
					l2 = l2->next;
					}
				else if (l1->where < l2->where)
					l1 = l1->next;
				else if (l1->where > l2->where)
					l2 = l2->next;
			break;
		case OR:
			while (l1 && l2)
				if (l1->where == l2->where)
					{
					result = BuildList(result,l1->where);
					l1 = l1->next;
					l2 = l2->next;
					}
				else if (l1->where < l2->where)
					{
					result = BuildList(result,l1->where);
					l1 = l1->next;
					}
				else if (l1->where > l2->where)
					{
					result = BuildList(result,l2->where);
					l2 = l2->next;
					}
			if (l1)
				t = l1;
			else if (l2)
				t = l2;
			while (t)
				{
				result = BuildList(result,t->where);
				t = t->next;
				}
			break;
		case NOT:
			while (l1 && l2)
				if (l1->where == l2->where)
					{
					l1 = l1->next;
					l2 = l2->next;
					}
				else if (l1->where < l2->where)
					{
					result = BuildList(result,l1->where);
					l1 = l1->next;
					}
				else if (l1->where > l2->where)
					l2 = l2->next;
			while (l1)
				{
				result = BuildList(result,l1->where);
				l1 = l1->next;
				}
			break;
		default:
			result = (ListType *)NIL;
			break;
		}

	return(result);

}	/* DoOperation */

/*****************************************************************************
 * LogRequest simply writes the search request to the log file.
 ****************************************************************************/
void LogRequest(sockfd)
	int		sockfd;		/* The socket to write to. */
{	char		buf[1024];	/* String to build to write out. */
	register short	i;		/* A loop counter. */

	if (!logFile && !debug)		/* No sense is using the CPU. */
		return;

	/* The following will never overwrite memory. */
	(void)sprintf(buf,"jughead(%d) -> ",port2use);

	for (i = 0; i < numCommands; i++)
		if (strlen(buf) + strlen(wrdsNcmds[i]) + 1 < 1024)
			{
			(void)strcat(buf," ");
			(void)strcat(buf,wrdsNcmds[i]);
			}
		else
			break;
	LogMessage(sockfd,buf);

}	/* LogRequest */

/*****************************************************************************
 * GetAllPositions acquires all the postitions of the partial word search by
 * finding the start and end position, and then OR'ing these positions into
 * a list which gets returned.  If 'what2Find' starts off with the wild card
 * character, the asterik, we post a message to the user and return nil.
 ****************************************************************************/
static ListType *GetAllPositions(index,what2Find,asterik,asterikPos,sockfd)
	long		index;		/* Index into the items array. */
	char		*what2Find,	/* The word we are looking for. */
			*asterik;	/* The position of the asterik. */
	int		asterikPos,	/* Number of characters to look at. */
			sockfd;		/* The socket file descriptor, errors only. */
{	ListType	*theList = (ListType *)NIL,	/* The list to return. */
			*list1   = (ListType *)NIL,	/* List of positions to OR against list2. */
			*list2   = (ListType *)NIL;	/* List of positions to OR against list1. */
	register long	start,		/* Start of the partial word match. */
			end;		/* End of the partial word match. */

	if (what2Find == asterik && *what2Find == '*')
		{
		char	s[1024];
		MemoryCaution(strlen(wildcarderr) + strlen(errorhost),1024,
				"error: GetAllPositions() attempted to overwrite memory");
		(void)sprintf(s,"%s%s",wildcarderr,errorhost);
		(void)SendString(s);
		MemoryCaution(strlen(what2Find),1000,
				"error: GetAllPositions() attempted to overwrite memory");
		(void)sprintf(s,"INVALID WILDCARD USAGE: %s",what2Find);
		LogMessage(sockfd,s);
		return((ListType *)NIL);
		}

	/* Find the starting and ending positions of the positions to return. */
	for (start = index - 1; start >= 0        && !StrnCmp(what2Find,items[start].word,asterikPos); start--);
	for (end   = index + 1; end < numElements && !StrnCmp(what2Find,items[ end ].word,asterikPos); end++);

	if (debug)
		(void)fprintf(rptPtr,"GetAllPositions found starting position = %ld, and ending position = %ld\n",start + 1,end - 1);

	/* Process the positions we will be returning. */
	for (start++; start < end; start++)
		if (!list1)
			list1 = GetPositions(start);
		else if (!list2)
			{
			list2 = GetPositions(start);
			theList = DoOperation(OR,list1,list2);
			DestroyList(list1);
			DestroyList(list2);
			list1 = theList;
			list2 = (ListType *)NIL;
			}

	/* We may have only got one hit, so make sure we return the information. */
	if (!theList && list1)
		theList = list1;

	return(theList);

}	/* GetAllPositions */

/*****************************************************************************
 * PostPositions posts the result of doing the boolean operations on 'what2find'
 * and checking for membership in 'array'.
 ****************************************************************************/
void PostPositions(sockfd,what2find)
	int		sockfd;		/* The socket to write to. */
	char		*what2find;	/* String with words and operations. */
{	short		evaluate,	/* Do we evaluate the operation? */
			operater = NOOP,/* Either [NOOP,AND,OR,NOT]. */
			reserved;	/* Is the current word reserved? */
	register short	i;		/* A loop counter. */
	long		index,		/* Index into the items array. */
			rangeStart,	/* The start of a range. */
			rangeEnd,	/* The end of a range. */
			limit   = maxitems2return;	/* The max number of items to return. */
	ListType	*list1  = (ListType *)NIL,	/* List of positions to operate against list2. */
			*list2  = (ListType *)NIL,	/* List of positions to operate against list1. */
			*result = (ListType *)NIL,	/* The result of "list1 operation list2'. */
			*tList  = (ListType *)NIL;	/* Temporary list to support partial word searches. */
	char 		*asterik;	/* Is this a partial word search? */
	int		asterikPos;	/* Number of characters to the asterik. */

	if (!(what2find = SpecialCommand(what2find,&limit,&rangeStart,&rangeEnd)))
		return;
	if (ParseSearchString(what2find))
		{
		LogRequest(sockfd);
		for (evaluate = i = 0; i < numCommands; i++)
			if (reserved = ReservedWord(wrdsNcmds[i]))
				operater = reserved;
			else if ((index = BinarySearch(StrToLower(wrdsNcmds[i]),items,numElements,&asterik,&asterikPos)) >= 0)
				{
				if (asterik)	/* We have a partial word search. */
					tList = GetAllPositions(index,wrdsNcmds[i],asterik,asterikPos,sockfd);
				else
					tList = GetPositions(index);
				if (!list1 && !evaluate)
					{
					result = list1 = tList;
					evaluate = 1;
					}
				else if (!list2)
					{
					list2 = tList;

					result = DoOperation(operater,list1,list2);
#ifdef BOOLOP_DEBUG
					if (debug)
						{
						(void)fprintf(rptPtr,"list1 ==> ");
						PrintList(list1);
						(void)fprintf(rptPtr,"\nlist2 ==> ");
						PrintList(list2);
						(void)fprintf(rptPtr,"\nresult ==> ");
						PrintList(result);
						(void)fprintf(rptPtr,"\n");
						}
#endif
					DestroyList(list1);
					DestroyList(list2);
					list1 = result;
					list2 = (ListType *)NIL;
					operater = NOOP;
					}
				}
			else
				{
				char	message[256];
				(void)sprintf(message,"COULD NOT FIND [%s]",wrdsNcmds[i]);
				LogMessage(sockfd,message);
				if (operater == AND)
					{
					DestroyList(result);
					result = list1 = (ListType *)NIL;
					}
				evaluate = 1;
				}

		if (result)
			{
			PrintPositions(sockfd,result,limit,rangeStart,rangeEnd);
			DestroyList(result);
			}
		}

}	/* PostPositions */

/*****************************************************************************
 * VerifyDataBaseName verifies we are dealing with the correct database.
 * This routine returns true if we have the correct database and false
 * otherwise.
 ****************************************************************************/
static int VerifyDataBaseName(fName,dName)
	char	*fName,		/* The root name of the database. */
		*dName;		/* The name we should be dealing with. */
{
#if(0)	/* Commented out because it was failing. */
	char	*str;		/* Position in fName where dName occurs. */

	if (str = strstr(fName,dName))
		{
		if (!StrCmp(dName,str))
			(void)strcpy(dName,fName);
		return(1);
		}
	return(0);
#else
	(void)strcpy(dName,fName);
	return(1);
#endif

}	/* VerifyDataBaseName */

/*****************************************************************************
 * CreateElements returns true if we could create the dynamic array 'items'
 * and false othewise.
 ****************************************************************************/
short CreateElements(fileName)
	char		*fileName;	/* The name of the file to read. */
{	FILE		*fp;		/* Pointer to the file we are reading. */
	register long	l;		/* A loop counter. */
	long		where;		/* The postions in the "data" file. */
	short		strLen;		/* The size of the word. */
	char		s[1024];	/* A temporary do it all string */
					/* and cannot overwrite memory.  */

	/* Make sure we don't overwrite memory. */
	if ((strlen(fileName) + strlen(INDXEXT)) > 512)
		{
		(void)fprintf(rptPtr,"error: attempt to overwrite memory.  %s too long\n",fileName);
		exit(-1);
		}
	if ((strlen(fileName) + strlen(HASHEXT)) > 512)
		{
		(void)fprintf(rptPtr,"error: attempt to overwrite memory.  %s too long\n",fileName);
		exit(-1);
		}

	(void)strcpy(indexName,fileName);
	(void)strcat(indexName,INDXEXT);
	(void)strcpy(hashName,fileName);
	(void)strcat(hashName,HASHEXT);

	if (!(fp = fopen(fileName,"r")))
		{
		(void)sprintf(s,"error: CreateElements could not open %s for reading\n",fileName);
		LogMessage(-1,s);
		}
	if (!(hfp = fopen(hashName,"r")))
		{
		(void)sprintf(s,"error: CreateElements could not open %s for reading\n",hashName);
		LogMessage(-1,s);
		}
	if (!fp || !hfp)
		return(0);

	(void)GetStr(hfp,dataFileName,512);

	if (!VerifyDataBaseName(fileName,dataFileName))
		{
		(void)sprintf(s,"error: incompatible database, looking for [%s]\n",dataFileName);
		LogMessage(-1,s);
		return(0);
		}

	if (debug)
		{
		(void)fprintf(rptPtr,"dataFileName = [%s]\n",dataFileName);
		(void)fprintf(rptPtr,"veronicaFile = [%s]\n",veronica);
		(void)fprintf(rptPtr,"indexName    = [%s]\n",indexName);
		(void)fprintf(rptPtr,"hashName     = [%s]\n",hashName);
		}

	numElements = GetLong(hfp);
	if (items = (Element *)malloc((unsigned)numElements * sizeof(Element)))
		for (l = 0; l < numElements; l++)
			{
			strLen = (short)GetInt(hfp);
			if (items[l].word = (char *)malloc((unsigned)strLen * sizeof(char)))
				{
				(void)GetStr(hfp,items[l].word,strLen);
				where = GetLong(hfp);
				items[l].positions = (ListType *)NIL;
				items[l].positions = BuildList(items[l].positions,where);
				}
			else
				{
				(void)sprintf(s,"error: CreateElements could not get memory for string %ld\n",l);
				LogMessage(-1,s);
				return(0);
				}
			}
	else
		{
		(void)sprintf(s,"error: CreateElements could not get memory for the %ld items\n",numElements);
		LogMessage(-1,s);
		return(0);
		}

#ifdef SHOW_ITEMS_DURING_DEBUG
	if (debug)
		{
		(void)fprintf(rptPtr,"items looks like:\n");
		for (l = 0; l < numElements; l++)
			(void)fprintf(rptPtr,"\t[%ld]\t[%s]\t%ld\n",l,items[l].word,items[l].positions->where);
		}
#endif

	(void)fclose(fp);
	(void)fclose(hfp);
	hfp = (FILE *)NIL;
	return(1);

}	/* CreateElements */

/*****************************************************************************
 * CleanUp simply frees up any memory used in the dynamic array 'items'.
 ****************************************************************************/
static void CleanUp()
{	register long	i;	/* A loop counter. */

	for (i = 0; i < numElements; i++)
		{
		free((char *)items[i].word);
		DestroyList(items[i].positions);
		}
	free((char *)items);
	numElements = 0;

}	/* CleanUp */

/*****************************************************************************
 * HangUpSignal resets the hangup interrupt, touches the
 * /usr/local/etc/jughead.pid file, and returns to the saved state.
 ****************************************************************************/
void HangUpSignal()
{
	(void)signal(SIGHUP,HangUpSignal);	/* Set things for next time. */

	LogMessage(-1,"SIGHUP signal encountered");
	if (debug)
		(void)fprintf(rptPtr,"Releasing memory ...\n");

	CleanUp();

	/* To support the "Disallow:" directive, but isn't used when	*
	 * jughead is a search engine.  Added here for clarity.		*/
	vctlHost = vctlPort = "";

	NullifyDisallowList();
	if (!ReadConfFile(JUGHEADCONF))
		{
		(void)fprintf(rptPtr,"error: could not initialize the search engine\n");
		if (debug)
			(void)fprintf(rptPtr,"error: InitSearchEngine() failed\n");
		exit(-1);
		}

	(void)DoSystemCall(Mysprint(touchcommand,JUGHEADPID));

	if (debug)
		{
		PrintTheList();
		(void)fprintf(rptPtr,"Rebuilding the binary tree.\n");
		}

	longjmp(hupbuf,0);			/* Jump to the saved state. */

}	/* HangUpSignal */

/*****************************************************************************
 * ReapChild returns 1 when a child process has died, and returns 0 if no
 * child processes exist or have not had a change of state.
 ****************************************************************************/
int ReapChild()
{
#ifdef NEXT
	union wait	status;		/* Status of the child process. */
#else
	int		status;		/* Status of the child process. */
#endif

#ifdef SVR4
#  ifdef SOLARIS2
	if (waitpid((pid_t)-1,&status,WNOHANG | WUNTRACED) > 0)
#  else
	if (waitid(P_ALL,0,NULL,WEXITED | WNOHANG | WSTOPPED) > 0)
#  endif
#else 
	if (wait3(&status,WNOHANG | WUNTRACED,NULL) > 0)
#endif
		{
#ifndef NEXT
		if (debug)
	       		(void)fprintf(rptPtr,"In ReapChild() with status = %d\n",status);
#endif
		return(1);
		}

	return(0);

}	/* ReapChild */

/*****************************************************************************
 * PostMsg2Socket returns true if it could write the string 'str' to the
 * socket 's', otherwise it returns false.
 ****************************************************************************/
int PostMsg2Socket(str,s)
	char	*str;		/* The string to post. */
	int	s;		/* The socket to write to. */
{	FILE	*fp;

	if (fp = fdopen(s,"w"))
		{
		(void)fprintf(fp,"%s%s",str,errorhost);
		(void)fflush(fp);
		(void)fclose(fp);
		return(1);
		}
	return(0);

}	/* PostMsg2Socket */

/*****************************************************************************
 * MakeDaemon returns 0 if we were able to detach the program from a tty and
 * make it a daemon.  Otherwise this routine -1 signifying we could not
 * do it.
 * This code was taken from Advanced Programming in the UNIX Environment
 * by W. Richard Stevens.
 ****************************************************************************/
static int MakeDaemon()
#ifdef NEXT
{       int     pid;
#else
{       pid_t   pid;
#endif

	if (getpid() != 1)
		{
		if ((pid = fork()) < 0)
			return(-1);
		else if (pid > 0)
			exit(0);    /* Parent goes bye-bye. */
		}

	/* Child continues. */

#ifndef BSD
	(void)setsid();			/* Become session leader. */
#endif

	/* chdir("/");			/* Change working directory. */
	/* No need, we're already where we want to be. */

	(void)umask(0);			/* Clear our file mode creation mask. */

	return(0);


}	/* MakeDaemon */

/*****************************************************************************
 * WritePIDfile writes the jughead.pid file, by default stored in
 * /usr/local/etc.  The format of the file is:
 *    Process_ID
 *    Command_Line_Invoking_jughead_as_a_search_engine
 * This routine is used solely to assist in scripting the ability to send
 * SIGHUP to jughead and keep a record as to how jughead was previously
 * started as a search engine.  This routine returns true if the file could
 * be written and false otherwise.  If 'cmdLine' was acquired via the call
 * to GetArguments() in jughead.c this routine also frees up the memory.
 * Thank you Kuypers@sri.ucl.ac.BE for the idea.
 ****************************************************************************/
static int WritePIDfile(cmdLine)
	char	*cmdLine;	/* Command line starting jughead. */
{	FILE	*fp;		/* Pointer to the jughead.pid file. */
	char	bufr[512];	/* Buffer for error messages. */

	if (cmdLine && (fp = fopen(JUGHEADPID,"w")))
		{
		(void)fprintf(fp,"%d\n%s\n",getpid(),cmdLine);
		(void)fclose(fp);
		free((char *)cmdLine);
		return(1);
		}
	else
		{
		MemoryCaution(strlen(JUGHEADPID),350,
				"error: WritePIDfile() attempted to overwrite memory");
		(void)sprintf(bufr,"error: Could not create %s\n",JUGHEADPID);
		LogMessage(-1,bufr);
		return(0);
		}

}	/* WritePIDfile */

/*****************************************************************************
 * DoSearch is the search server part of jughead.  This routine never
 * returns.
 ****************************************************************************/
void DoSearch(cmdLine,indexTable,logFile)
	char			*cmdLine,	/* The command line starting jughead. */
				*indexTable,	/* Name of the index table file. */
				*logFile;	/* The file to log to. */
{	int			childspid,	/* The process ID of the child process. */
				s,		/* The socket file descriptor. */
				newS,		/* The new socket file descriptor. */
				addressLen;	/* Size of the address space. */
	struct sockaddr_in	address;	/* Address of connecting entity. */
	char			bufr[512];	/* Buffer for error messages. */
#ifdef NODEFUNCT
	int			blocking,	/* Is accept blocking? */
				initialCntl,	/* Initial socket control settings. */
				initialProcs,	/* Initial number of processes allowed. */
				debugLoopVar;	/* A loop variant for debugging. */
#endif


	(void)fprintf(rptPtr,"jughead, Copyright 1993, 1994, University of Utah Computer Center\n");
	(void)fprintf(rptPtr,"Using index table %s\n",indexTable);
	(void)fprintf(rptPtr,"Using port %d\n",port2use);

	if (logFile)
		{
		if (debug)
			(void)fprintf(rptPtr,"Logging to %s\n",logFile);
		MemoryCaution(strlen(indexTable),490,
				"error: DoSearch() attempted to overwrite memory");
		(void)strcpy(bufr,"STARTED UP ON ");
		(void)strcat(bufr,indexTable);
		LogMessage(-1,bufr);
		}

	if (debug)
		(void)fprintf(rptPtr,"debug is on: NOT going in daemon mode.\n");
	else if (MakeDaemon())	/* Detach from the tty if not running in the background. */
		LogMessage(-1,"error: jughead cannot make itself daemon");

	if (cmdLine)
		{
		if (!debug)
			(void)WritePIDfile(cmdLine);
		else
			(void)fprintf(rptPtr,"debug is on: NOT writing [%s] to pid file [%s]\n",cmdLine,JUGHEADPID);
		free((char *)cmdLine);
		}

	if (userName)
		{
		struct passwd	*pswd;
		if (pswd = getpwnam(userName))
			if (setuid(pswd->pw_uid))
				{
				(void)sprintf(bufr,"error: could not setuid %ld (%s)\n",(long)pswd->pw_uid,userName);	/* Won't kill mem. */
				LogMessage(-1,bufr);
				exit(-1);
				}
			else
				{
				(void)sprintf(bufr,"Running with setuid %ld (%s)\n",(long)pswd->pw_uid,userName);	/* Won't kill mem. */
				LogMessage(-1,bufr);
				}
		else
			{
			MemoryCaution(strlen(userName),475,
					"error: DoSearch() attempted to overwrite memory");
			(void)sprintf(bufr,"error: could not find user %s\n",userName);
			LogMessage(-1,bufr);
			exit(-1);
			}
		}

	if ((s = ListenerEstablished(port2use)) < 0)
		exit(-1);

	if ((int)signal(SIGHUP,HangUpSignal) == -1)
		{
		(void)sprintf(bufr,"error: could not establish SIGHUP handler\n");
		if (debug)
			(void)fprintf(rptPtr,"%s",bufr);
		LogMessage(-1,bufr);
		exit(-1);
		}
	(void)setjmp(hupbuf);

#ifndef IN_THE_FUTURE
	if (time2process)
		(void)time(&startTime);

	if (!CreateElements(indexTable))
		{
		LogMessage(-1,"error: DoSearch could not create index tree");
		if (debug)
			(void)fprintf(rptPtr,"error: DoSearch could not create index tree\n");
		exit(-1);
		}

	if (time2process)
		PostTime2Process();
#endif

#ifdef NODEFUNCT
	if (noDefunct)	/* Set things up to removed the <defunct> process. */
		{
		initialCntl = fcntl(s,F_GETFL,O_NDELAY);
		initialProcs = maxprocs;
		blocking = 1;
		if (debug)
			debugLoopVar = 0;
		}
#endif

	if (debug)
		(void)fprintf(rptPtr,"Ready for incoming connections.\n");

	while (1)	/* Wait for and handle all connections. */
		{
		addressLen = sizeof(address);
		if ((newS = accept(s,(struct sockaddr *)&address,&addressLen)) < 0)
			{
#ifdef NODEFUNCT
			if (errno != EWOULDBLOCK)
#endif
				{
				LogMessage(-1,"error: DoSearch could not accept");
				exit(-1);
				}
			}
		else
			{
			maxprocs--;
			if (maxprocs <= 0)
				{
				if (!PostMsg2Socket(tomnyprcserr,newS))
					LogMessage(-1,"error: could not open socket for writing");
				if (debug)
					(void)fprintf(rptPtr,"error: Too many jughead processes.\n");
				maxprocs++;
				}
			else if ((childspid = fork()) < 0)
				{
				(void)PostMsg2Socket(noforkerr,newS);
				LogMessage(-1,"error: could not fork");
				if (debug)
					(void)fprintf(rptPtr,"error: could not fork\n");
				maxprocs++;
				sleep(50);
				}
			else if (!childspid)	/* Child process so */
				{		/* close original socket */
				(void)close(s);
				(void)ProcessRequest(newS);

#ifdef IN_THE_FUTURE
				CleanUp();
#endif
				exit(0);
				}
			(void)close(newS);		/* Close the new socket we opened. */
			}

		if (debug)
			{
			(void)fprintf(rptPtr,"before ReapChild() maxprocs = %d\n",maxprocs);
#ifdef NODEFUNCT
			if (noDefunct)
				debugLoopVar++;
#endif
			}
		while (ReapChild())
			maxprocs++;
		if (debug)
			{
			(void)fprintf(rptPtr,"after  ReapChild() maxprocs = %d",maxprocs);
#ifdef NODEFUNCT
			if (noDefunct)
				(void)fprintf(rptPtr," debugLoopVar = %d\n",debugLoopVar);
			else
#endif
				(void)fprintf(rptPtr,"\n");
			}

#ifdef NODEFUNCT
		if (noDefunct)	/* Adjust accept() for blocking or non-blocking. */
			if (maxprocs < initialProcs)
				{
				if (blocking)
					{
					(void)fcntl(s,F_SETFL,O_NDELAY);
					blocking = 0;
					}
				}
			else
				{
				(void)fcntl(s,F_SETFL,initialCntl);
				blocking = 1;
				if (debug)
					debugLoopVar = 0;
				}
#endif
		}

}	/* DoSearch */
