/*****************************************************************************
 * File:	jughead.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	February 17, 1993
 *
 * Modified:	February 27, 1993 by Rhett "Jonzy" Jones
 *		To support both the 'h' and 'v' flags, and to no longer
 * 		write recurrsivly to temporary files.  Now I only write
 *		to a single temporary file.
 *
 *		March 3, 1993, by Rhett "Jonzy" Jones
 *		Added support for the 'b' flag.
 *
 *		March 10, 1993, by Rhett "Jonzy" Jones.
 *		Added support for the 'B' and 'l' flags.
 *
 *		March 13, 1993, by Rhett "Jonzy" Jones.
 *		Fixed some bugs, optimized some routines, and rewrote
 *		DoSystemCall().  Removed Lowercase() and changed calls
 *		to the routine to StrToLower(), which is defined in
 *		"search.c".  Modified BeenHereB4() to handle the cases
 *		where the selector string is "/" and "1/", which is the
 *		same as "".
 *
 *		March 15, 1993, by Rhett "Jonzy" Jones.
 *		Relocated the changes made to BeenHereB4() to a new
 *		routine EmptyStringCheck() so all occurances of "/",
 *		"1/", and "1" get mapped to "".  I should mention I just
 *		discovered "1" gets handled like "/", "1/", and "".  Hmm.
 *
 *		March 18, 1993, by Rhett "Jonzy" Jones.
 *		Added HostField2Lower() to make sure the host field gets
 *		mapped to lowercase, which is needed when building the
 *		index so that sort will eliminate all duplicates.
 *		Fixed the system call to "sort" to only sort on the selector
 *		string and beyond.  Also defined CATCOMMAND, RMCOMMAND, and
 *		SORTCOMMAND for use when calling DoSystemCall().
 *
 *		March 21, 1993, by Rhett "Jonzy" Jones.
 *		GetParamsFromStdin() now only prints the message informing
 *		the user to enter control-D when finished, only if we are
 *		getting the information from stdin and not a file.
 *
 *		March 27, 1993, by Rhett "Jonzy" Jones.
 *		Moved the definitions of VERSION, CATCOMMAND, RMCOMMAND,
 *		SORTCOMMAND, and TMPFILENAME to the Makefile.  Changed the
 *		name of DoWeRecurse() to DoWeTraverse().  Added CantGet2Host()
 *		nogoHead, and nogoTail to support the ability of not attempting
 *		to connect to a host we could not connect to previously.
 *
 *		March 28, 1993, by Rhett "Jonzy" Jones.
 *		Added HandleIndention(), and modified DoWeTraverse() to print
 *		"<< see line # >>" with the proper indention if we have seen
 *		this directory before.  Added support for the -n flag.  If we
 *		received a line from a server which did not have the host or
 *		port string, the line is now not processed.  Fixed a problem of
 *		not comparing the hosts correctly, which was fixed by converting
 *		the incoming host name to lowercase.  If a menu path has been
 *		traversed previously, it will no longer be traversed again and a
 *		message stating what line to view will be printed with the
 *		proper indention.  When not running to build the index table
 *		as a search engine, jughead now prints a note if hosts were
 *		encountered that could not be connected to, and just prior to
 *		termination displays the hosts it could not connect to.
 *
 *		March 31, 1993, by Mic Kaczmarczik: mic@bongo.cc.utexas.edu
 *		Added support for an optional -p port flag when running as a
 *		search engine.  Added the variable 'searchPort', modified
 *		UsageError(), GetArguments() and the call to DoSearch() in
 *		main().
 *
 *		April 1, 1993, by Rhett "Jonzy" Jones.
 *		Added support for the -t flag which prints the time required
 *		to either read in the index table, build the data file, or
 *		process a menu from a given gopher server.  Added support for
 *		the -a and -A flags which print the dirTree with the number of
 *		directories served or the directories served respectivly.
 *		Made 'buildIndex' and 'doSearch' global variables instead of
 *		local to main() and adjusted GetArguments() to reflect the
 *		change.  No longer sort the datafile to datafile.sorted when
 *		using the -b option.  Instead dataFile gets sorted into
 *		dataFile.
 *
 *		April 20, 1993, by Rhett "Jonzy" Jones.
 *		Added LooksGood() to make sure we received a valid line of
 *		of data from the server.  This routine was written because
 *		it was found the PC Gopher Server did not follow gopher
 *		protocol.  Added the variable menuFlag to prevent printing of
 *		the menus when building the dataFile.  Fixed a bug that would
 *		not correctly validate the port for the exceptions host, which
 *		is the host and port to not traverse.
 *
 *		May 3, 1993, by Rhett "Jonzy" Jones.
 *		Modified LooksGood() to immediatly return false if we acquired
 *		a null string.  This fixed a core dump when attempting to
 *		retrieve an empty directory from a gopher+ server.  Thanks
 *		lindner@boombox.micro.umn.edu for the bug report.
 *
 *		May 5, 1993, by Rhett "Jonzy" Jones.
 *		Added support for -X which is a selector string to not traverse.
 *		A quick hack to not traverse any selector strings that begin
 *		with "ftp:".  Example: jughead soar.cc.utah.edu -X "ftp:*".
 *		The variable used for this support is 'exceptionSelStr' and
 *		does supports wildcarding.
 *
 *		May 20, 1993, by Rhett "Jonzy" Jones.
 *		Moved UsageError(), PrintVersion(), GetParamsFromStdin(), and
 *		GetArguments() to getargs.c, due in part to rewritting some of
 *		these routines.  Removed initialization of global variables at
 *		the time of declaration, and added InitializeTheWorld() to
 *		accommodate the initialization.  Added the variable username
 *		to support -u username and allow jughead to act as a search
 *		engine running with a uid of username.
 *
 *		May 22, 1993, by Rhett "Jonzy" Jones.
 *		Removed the variables 'exceptionHost', 'exceptionPort', and
 *		'exceptionSelStr', and the routine ValidSelStr(), added the
 *		variable 'nogos' to support multiple exception items.  Thus
 *		jughead now can support multiple hosts and ports not to traverse
 *		as well as multiple selection strings to traverse.
 *
 *		November 14, 1993, by Rhett "Jonzy" Jones.
 *		Modified DoWeTraverse() to support wildcarding of any hosts not
 *		to search with the -x flag.  The port is still required.
 *		Thank you gopher@pereiii.uji.es for the patch.
 *
 *		Modified ProcessMenu() such that no gopher+ ASK block directory
 *		will be searched.  Thank you doylej@liberty.uc.wlu.edu for
 *		bringing this to my attention.
 *
 *		February 25, 1994, by Rhett "Jonzy" Jones.
 *		Added the use of MAXPROCS and maxProcs to support the maximum
 *		number of forked jughead processes.
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		April 30, 1994, by Rhett "Jonzy" Jones.
 *		Maes@elec.ucl.ac.BE reported a typographical error on my
 *		part in the call to setlocale().  Had to fix this one
 *		real fast.  Oops.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Moved many definitions from the Makefile into
 *		jughead.conf.h and now include jughead.conf.h.
 *		Optimized some of the code.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Changed all calls to strcmp() to StrCmp() and strncmp()
 *		to StrnCmp() to support both ASCII and ISO-XXXX-Y character
 *		sets.  Removed use of LOCALE.
 *
 *		May 9, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine GetCommandLine() to get a copy of the
 *		command line prior to parsing it so we can write the
 *		command line to the jughead.pid file to assist in starting
 *		jughead.  A reminder as to how jughead was started as a
 *		search engine.  The file also contains the process ID to
 *		assist in scripting and sending SIGHUP to jughead.
 *		Thank you Kuypers@sri.ucl.ac.BE for the idea.
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		A little bit of code cleaning.  Expanded use of the variable
 *		'logFile' to generate a report to assist in logging
 *		the results of building the datafile via a script for
 *		use with cron.  Removed use of 'onlyDoHosts' and 'onlyDoHostsT',
 *		and removed use of the 'list' variable in DoWeTraverse().
 *		Cleaned up the code for lint.  Commented out the PrintPath(),
 *		BuildPath(), InitPath(), and DestroyPath() routine definitions.
 *
 *		May 23, 1994, by Rhett "Jonzy" Jones.
 *		Modifed the call to PrintTree() to prints the results to
 *		'logFile` if the -l flag is given.  Cleaned up ProcessMenu()
 *		so use of the -h or -H flags will hinder use of any other flags.
 *		Cleaned up use of the -m flag as well.
 *		Removed all use and occurances of PrintPath(), BuildPath(),
 *		InitPath(), and DestroyPath() and variables path[] and
 *		pathPrinted[] to support a cleaner version of PopPath(),
 * 		PrintPath(), and PushPath() which are defined in "path.c".
 *		Now terminate if we're going to overwrite memory.
 *
 *		June 11, 1994, by Rhett "Jonzy" Jones.
 *		Added #ifdef NEED_MALLOC_H to give support for BSD.
 *
 *		July 24, 1994 by Rhett "Jonzy" Jones.
 *		Renamed 'searchPort' to 'port2use' for clarification, and
 *		moved the declaration to search.c.  Removed 'port2use' as
 *		a parameter to DoSearch().  Renamed 'maxProcs' to 'maxprocs'
 *		and the declaration to search.c.
 *		InitializeTheWorld() now returns an int instead of void,
 *		as well as calls ReadConfFile().
 *		Added the routines ValidVariableCheck() and PrintJugheadConf().
 *
 *		September 20, 1994, by Rhett "Jonzy" Jones.
 *		Gave support for the -V flag which is used in conjunction
 *		with the -b flag to build a database file for veronica
 *		which does not contain any "Disallow" gopher paths.
 *
 *		September 22, 1994, by Rhett "Jonzy" Jones.
 *		Add the routine SpankDuplicateEntries() to ensure there is
 *		not a gopher0 and gopher+ item pointing to the same item.
 *		This routine removed the gopher0 item.
 *		Added the inclusion of <errno.h> and did some code
 *		cleansing.
 *		Altered the way ReadConfFile() gets called to the
 *		"Disallow:" directive can be utilized in jughead.conf.
 *
 *		September 24, 1994, by Rhett "Jonzy" Jones.
 *		Altered the parameters to the system call sort() and
 *		SpankDuplicateEntries(), now make the system call to rm(),
 *		and no longer call rename().  rename() does not work
 *		across mounted devices.  Hello I'm awake.
 *		Added code to prevent waiting for ever for a server reply,
 *		which includes the TooMuchTime4Read() routine.
 *
 *		September 25, 1994, by Rhett "Jonzy" Jones.
 *		Added the ControlCAbort() routine and appropriate signal
 *		handling to ensure the temporary file is discarded prior
 *		to termination if SIGINT (control-c) is encountered.
 *
 *		September 27, 1994, by Rhett "Jonzy" Jones.
 *		Removed all use of rm.
 *
 *		October 9, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine Unlink() to ensure unlink() won't
 *		dump core if the file to remove does not exist.
 *
 *		November 16, 1994, by Rhett "Jonzy" Jones with code changes
 *		submitted by: paul@openage.com
 *		Added support for SCO compilation.
 *
 * Description:	Jonzy's Universal Gopher Hierarchy Excavation And Display.
 *		Excavates through gopher menus and displays the hierarchy
 *		of the menus encountered.
 *
 * Routines:		void	HandleIndention(int indent);
 *			short	Host2Search(char *s);
 *			List	*CreateNode(char *sStr,char *hStr,char *pStr);
 *			List	*InsertNode(List *tail,List *node);
 *			List	*RemoveNode(List *tail);
 *			short	BeenHereB4(char *sStr,char *hStr,char *pStr,
 *					   List *head);
 *			short	CantGet2Host(List *nogoList,char *sStr,char *Str,
 *					     char *pStr);
 *			short	DoWeTraverse(char type,char *sStr,char *hStr,
 *					     char *pStr,int indent);
 *			void	PrintItem(char *dStr,int indent);
 *			char	*EmptyStringCheck(char *sStr);
 *			char	*HostField2Lower(char *line);
 *		static	void	TooMuchTime4Read(void);
 *			int	LooksGood(char **rsltLine,FILE *rdPtr);
 *			int	Unlink(char *path);
 *			void	ProcessMenu(char *selStr,char *hostStr,
 *					    char *portStr,int indent);
 *			void	PostTime2Process(void);
 *			int	ValidVariableCheck();
 *			void	SpankDuplicateEntries(char *f1,char *f2)
 *			void	PrintJugheadConf();
 *			int	InitializeTheWorld();
 *			char	*GetCommandLine(int argc,char *argv);
 *		static	void	ControlCAbort(void);
 *			int	main(int argc,char *argv[]);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#ifdef NEED_MALLOC_H
#	include <malloc.h>
#endif
#ifdef NEXT
#	include <libc.h>
#else
#	include <stdlib.h>
#endif
#include <signal.h>
#include <stdio.h>
#include <string.h>
#ifndef SCO
#	include <sys/file.h>
#endif
#include <sys/stat.h>
#include <sys/wait.h>
#include <time.h>
#include <setjmp.h>
#include <errno.h>
#ifdef SCO
#	include <sys/file.h>
#endif

#include "dirTree.h"
#include "jughead.conf.h"
#include "jughead.h"
#include "sockets.h"
#include "tree.h"

/* #define DOPATHS */

#define READTIMEOUT	(5 * 60)
static jmp_buf		env;

extern void	DoSearch();		/* Defined in "search.c". */
extern void	MakeHashTables();	/* Defined in "search.c". */
extern char	*GetString();		/* Defined in "sockets.c". */

/* These routines are defined in "path.c". */
extern int	PopPath();
extern int	PrintPath();
extern int	PushPath();

/* These routines are defined in "jugheadConf.c". */
extern char	*MakeStr();
extern int 	MakeInt();
extern int	ReadConfFile();

FILE	*wtPtr,			/* File pointer for writing to 'theHost'. */
	*rdPtr,			/* File pointer for reading from 'theHost'. */
	*rptPtr = stderr;	/* Pointer to the logFile file. */
char	buffer[BUFFERSIZE],	/* The output from the machine. */
	*searchHosts[MAXHOSTS],	/* Hosts to search. */
	*initialHost,		/* The initial host we connected to. */
	*selStr,		/* The selector string. */
	*hostStr,		/* The host string. */
	*portStr,		/* The port string. */
	*fileName,		/* Name of the file to open or create. */
	*userName,		/* Name of the user to use with -S. */
	*veronica;		/* Name of the veronica dataFile. */
int	numSearchHosts,		/* The number of hosts to search. */
	debug,			/* Are we debugging? */
	info4dirsOnly,		/* Do we print host info for directories only? */
	info4allItems,		/* Do we print host info for all the items? */
	listHosts,		/* Do we print only the hosts accessed? */
	listHostsNPorts,	/* Do we print only the hosts and ports accessed? */
	buildDataFile,		/* Are we building the dataFile? */
	menuFlag,		/* Do we display the menus? */
	printLineNumbers,	/* Do we print line numbers? */
	printDTree,		/* Do we print the dirTree with the number of directories served? */
	printDTreeDirs,		/* Do we print the dirTree along with the directories served? */
	time2process;		/* Do we calculate the time for a certain run? */
short	buildIndex,		/* Do we build the index table? */
	doSearch;		/* Are we doing the search stuff? */
List	*nogos = (List *)NIL,	/* Information about who to not traverse.*/
	*head = (List *)NIL,	/* The head of the current menu path list. */
	*tail = (List *)NIL,	/* The tail of the current menu path list. */
	*nogoHead = (List *)NIL,/* Head of the can't connect to host and port list. */
	*nogoTail = (List *)NIL;/* Tail of the can't connect to host and port list. */
long	seenDirB4;		/* Have we seen this before? */
time_t	startTime;		/* The time a run was started, for use with 'time2process'. */

/* The root of the veronica tree, which is a list of gopher servers we need	*
 * to try and grab the veronica.ctl file from to acquire the "Disallow" paths.	*/
TreeType	*vRoot = (TreeType *)NIL;

extern long	lineNumber;	/* Defined in "dirTree.c". */

extern char	*vctlHost,	/* Defined in "jugheadVctl.c". */
		*vctlPort;
extern void	PrintTheList();

extern void	CreateVeronicaFile();	/* Defined in "jugheadVctl.c". */

/*****************************************************************************
 * HandleIndention prints 'indent' number of indentions to stdout.  This
 * routine is used to keep directory contents lined up horizontaly.
 ****************************************************************************/
void HandleIndention(indent)
	int		indent;		/* The level of indention. */
{	register int	numIndents;	/* The number of indentions to add to path. */

	/* Indent the data 'indent' number of indentions. */
	for (numIndents = 0; numIndents < indent; numIndents++)
		(void)fprintf(stdout,"%s",INDENTSTR);

}	/* HandleIndention */

/*****************************************************************************
 * Host2Search returns true if the host 's' is a member of 'searchHosts',
 * otherwise it returns false.
 ****************************************************************************/
short Host2Search(s)
	char		*s;	/* Is this in 'searchHosts'? */
{	register int	i;	/* A loop counter. */

	for (i = 0; i < numSearchHosts; i++)
		if (StrRcmp(searchHosts[i],s))
			return(1);

	return(0);

}	/* Host2Search */

/*****************************************************************************
 * CreateNode returns a newly created and initialized node for the list.
 ****************************************************************************/
List *CreateNode(sStr,hStr,pStr)
	char	*sStr,		/* The selector string. */
		*hStr,		/* The host string. */
		*pStr;		/* The port string. */
{	List	*node;		/* The node we are returning. */

	if (node = (List *)malloc(sizeof(List)))
		if (node->info.sStr = (char *)malloc((unsigned)strlen(sStr) + 1))
			{
			(void)strcpy(node->info.sStr, sStr);
			if (node->info.hStr = (char *)malloc((unsigned)strlen(hStr) + 1))
				{
				(void)strcpy(node->info.hStr, hStr);

				if (node->info.pStr = (char *)malloc((unsigned)strlen(pStr) + 1))
					{
					(void)strcpy(node->info.pStr, pStr);
					node->next = node->last = (List *)NIL;
					return(node);
					}
				else
					{
					free((char *)node->info.hStr);	node->info.hStr = (char *)NIL;
					free((char *)node->info.sStr);	node->info.sStr = (char *)NIL;
					free((char *)node);		node = (List *)NIL;
					}
				}
			else
				{
				free((char *)node->info.sStr);	node->info.sStr = (char *)NIL;
				free((char *)node);		node = (List *)NIL;
				}
			}
		else
			{
			free((char *)node);	node = (List *)NIL;
			}

	(void)fprintf(rptPtr,"error: CreateNode could not get memory for the node.\n");
	return(node);

}	/* CreateNode */

/*****************************************************************************
 * InsertNode inserts the node 'node' on the end of the list 'list'.
 ****************************************************************************/
List *InsertNode(tail,node)
	List	*tail,		/* The tail of the list we are inserting into. */
		*node;		/* The node we are inserting. */
{
	if (!node)
		return(tail);

	tail->next = node;
	node->last = tail;
	return(node);

}	/* InsertNode */

/*****************************************************************************
 * RemoveNode removes and frees up the memory occupied by the last item
 * in the list, which is pointed to by 'tail'.
 ****************************************************************************/
List *RemoveNode(tail)
	List	*tail;		/* The tail of the list we are inserting into. */
{	List	*node;		/* The node to remove. */

	node = tail;
	tail = tail->last;
	if (tail)
		tail->next = (List *)NIL;
	free((char *)node->info.sStr);		node->info.sStr = (char *)NIL;
	free((char *)node->info.hStr);		node->info.hStr = (char *)NIL;
	free((char *)node->info.pStr);		node->info.pStr = (char *)NIL;
	free((char *)node);			node = (List *)NIL;
	return(tail);

}	/* RemoveNode */

/*****************************************************************************
 * BeenHereB4 returns true if 'sStr', 'hStr', and 'pStr' are in the list
 * pointed to by 'list'.  This is done so we will not travese a menu
 * we have already looking at.  Otherwise it returns false because we
 * are not looking at nor have been in this part of the tree.
 ****************************************************************************/
short BeenHereB4(sStr,hStr,pStr,list)
	char	*sStr,		/* The selector string. */
		*hStr,		/* The host string. */
		*pStr;		/* The port string. */
	List	*list;		/* Head of the list. */
{
	while (list)		/* See if we have been here before. */
		{
		if (!StrCmp(list->info.sStr,sStr))
			if (!StrCmp(list->info.hStr,hStr))
				if (!StrCmp(list->info.pStr,pStr))
					return(1);
		list = list->next;
		}
	return(0);

}	/* BeenHereB4 */

/*****************************************************************************
 * CantGet2Host returns true if we could not connect to 'hStr' out port 'pstr'
 * previously.  Otherwise it returns false.  This routine calls BeenHereB4()
 * and was written for readability only.
 ****************************************************************************/
short CantGet2Host(nogoList,sStr,hStr,pStr)
	List	*nogoList;	/* List of hosts we can't connect to. */
	char	*sStr,		/* The selector string. */
		*hStr,		/* The host to connect to. */
		*pStr;		/* The port to use. */
{
	return(BeenHereB4(sStr,hStr,pStr,nogoList));

}	/* CantGet2Host */

/*****************************************************************************
 * DoWeTraverse returns true if we can look at this directory in the menu.
 * Otherwise it returns false.  In otherwords return true if we have not
 * looked at this directory yet, and the current entry is a directory,
 * and the directory is from one of the hosts contained in 'hosts2search'.
 ****************************************************************************/
short DoWeTraverse(type,sStr,hStr,pStr,indent)
	char	type,		/* What type of entry is this? */
 		*sStr,		/* The selector string. */
		*hStr,		/* The host string. */
		*pStr;		/* The port string. */
	int	indent;		/* The level to indent if need be. */
{	List	*t;		/* A temporary list of nogos.*/
	char	*asterik;	/* Where is the asterik? */
	size_t	numChars;	/* Number of characters to the asterik. */

	for (t = nogos; t; t = t->next)
		if (t->info.sStr[0])
			{
			if (asterik = strchr(t->info.sStr,'*'))
				numChars = (size_t)(asterik - t->info.sStr);
			else
				numChars = strlen(t->info.sStr);
			if (!StrnCmp(t->info.sStr,sStr,numChars))
				return(0);
			}
		else if (t->info.hStr)
			{
			if (!StrCmp(pStr,t->info.pStr) &&
			   (!StrCmp(hStr,t->info.hStr) || !StrCmp("*", t->info.hStr)))
				return(0);
			}

	WaterTree(sStr,hStr,pStr);

	if (type == A_DIRECTORY && Host2Search(hStr))
		if (!(seenDirB4 = InDirTree(dirRoot,PSTR)))
			{
			dirRoot = BuildDirTree(dirRoot);
			return(1);
			}
		else if (menuFlag)
			{
			HandleIndention(indent + 2 * printLineNumbers);
			(void)fprintf(stdout,"<< see line %ld >>\n",seenDirB4),++lineNumber;
			}
	return(0);

}	/* DoWeTraverse */

/*****************************************************************************
 * PrintItem prints the item with the proper number of indentions, as well
 * as prints what type of item it is just like the unix gopher from Minnesota.
 ****************************************************************************/
void PrintItem(dStr,indent)
	char	*dStr;		/* The display string. */
	int	indent;		/* The level of indention. */
{
	if (printLineNumbers)	/* User wants the line numbers printed. */
		(void)fprintf(stdout,"%7ld ",lineNumber + 1);

	HandleIndention(indent);

	/* Print the menu title like a gopher menu. */
	(void)fprintf(stdout,"%s",dStr + 1);
	switch (dStr[0])
		{
		case A_DIRECTORY:
			(void)fprintf(stdout,"/");
			break;
		case A_CSO:
			(void)fprintf(stdout," <CSO>");
			break;
		case A_TN3270:
			(void)fprintf(stdout," <3270>");
			break;
		case A_TELNET:
			(void)fprintf(stdout," <TEL>");
			break;
		case A_INDEX:
			(void)fprintf(stdout," <?>");
			break;
		case A_SOUND:
			(void)fprintf(stdout," <)");
			break;
		case A_FILE:
			(void)fprintf(stdout,".");
			break;
		case A_PCBIN:
			(void)fprintf(stdout," <PC Bin>");
			break;
		case A_UNIXBIN:
			(void)fprintf(stdout," <Bin>");
			break;
		case A_IMAGE:
		case A_GIF:
			(void)fprintf(stdout," <Picture>");
			break;
		case A_MACHEX:
			(void)fprintf(stdout," <HQX>");
			break;
		default:
			(void)fprintf(stdout," <~%c~>",dStr[0]);
			break;
		}

}	/* PrintItem */

/*****************************************************************************
 * EmptyStringCheck checks if the string 'sStr' is either "/", "1/", or "1",
 * and if it is we return the emptystring "".  Otherwise 'sStr' gets returned
 * unchanged.  The reason for this routine is to prevent a recursive menu
 * loop from happening.  The gopher protocol says you must send the empty
 * string to acquire the menu.  Well it looks like a unix server handles
 * "", "/", "1/", and "1" all the same.  Hmm.
 ****************************************************************************/
static char *EmptyStringCheck(sStr)
	char	*sStr;		/* The string we are testing. */
{

	if (!StrCmp(sStr,"/") || !StrCmp(sStr,"1/") || !StrCmp(sStr,"1"))
		return(EMPTYSTRING);
	return(sStr);

}	/* EmptyStringCheck */

/*****************************************************************************
 * HostField2Lower returns the string passed in with the host field converted
 * to lowercase.
 ****************************************************************************/
char *HostField2Lower(line)
	char	*line;		/* The line we are looking at. */
{	char	*s;		/* A position in 'line'. */


	for (s = strchr(strchr(line,'\t') + 1,'\t') + 1; *s && *s != '\t'; s++)
		*s = tolower(*s);

	return(line);

}	/* HostField2Lower */

/*****************************************************************************
 * TooMuchTime4Read gets called only when the time period has elapsed when
 * waiting to read from our input..
 ****************************************************************************/
static void TooMuchTime4Read()
{
	if (debug)
		(void)fprintf(rptPtr,"In TooMuchTime4Read\n");

	longjmp(env,1);

}	/* TooMuchTime4Read */

/*****************************************************************************
 * LooksGood returns true if we received a valid line from 'rdPtr', otherwise
 * it returns false.  A valid line has the form:
 *	dStr\tsStr\thStr\tpStr[\tmore]\r\n
 * a invalid line, or we are done, has the form:
 *	.\r\n
 * or any line with no tabs, etc.
 * This routine was written because the PC Gopher server does not follow
 * gopher protocol, and sends the string:
 *	Error: Could not find requested file\r\n.\r\n
 * when it should look something like:
 *	Error: could not find requested file\t\terror.host\t-1\r\n.\r\n
 ****************************************************************************/
static int LooksGood(rsltLine,rdPtr)
	char	**rsltLine;	/* The line we are getting from 'rdPtr'. */
	FILE	*rdPtr;		/* The file or socket we are reading from. */
{	char	*s;		/* A position in 'rsltLine'. */
	int	numTabs;	/* How many tabs are there? */


	/* Set things up so we don't wait for ever waiting to read. */
	(void)signal(SIGALRM,TooMuchTime4Read);
	(void)alarm(READTIMEOUT);
	if (setjmp(env))
		{
		if (debug)
			(void)fprintf(rptPtr,"Too much time waiting to read\n");
		return(0);
		}

	/* Get the line of data from the server and make sure we got something. */
	if (!(*rsltLine = GetString(rdPtr)))
		return(0);

	/* We got our request so deactivate the alarm. */
	(void)alarm(0);
	(void)signal(SIGALRM,SIG_IGN);

	/* Check of we are done. */
	if (strstr(*rsltLine,".\r"))
		return(0);

	/* See if we have at least 3 tabs. */
	for (s = *rsltLine, numTabs = 0; *s; s++)
		if (*s == '\t')
			numTabs++;
	if (numTabs > 2)
		return(1);

	/* We got something that is totaly out to lunch. */
	return(0);

}	/* LooksGood */

/*****************************************************************************
 * Unlink removes the directory entry names by 'path'.  This basicly does
 * same thing "man 2 unlink" is supposed to do, except this routine first
 * checks if 'path' exists prior to calling unlink() to prevent a core
 * dump if 'path' does not exist.  If the file 'path' does not exist this
 * routine returns 0, otherwise it returns the result of unlink().
 ****************************************************************************/
int Unlink(path)
	char	*path;		/* Pathway to the file to remove. */
{	FILE	*fp;		/* Pointer to the file to remove. */

	if (fp = fopen(path,"r"))
		{
		(void)fclose(fp);
		return(unlink(path));
		}

	return(0);
	
}	/* Unlink */

/*****************************************************************************
 * ProcessMenu processes the menu entity which should be a directory.  If
 * the entity is not a directory ... well you know what they say, "garabage
 * in ... garbage out."
 * This code is ugly, but oh well such is life.
 ****************************************************************************/
void ProcessMenu(selStr,hostStr,portStr,indent)
	char	*selStr,	/* Selection string to send. */
		*hostStr,	/* The host to contact. */
		*portStr;	/* The port to use. */
	int	indent;		/* The level of indentation. */
{	char	**data,		/* The data acquired from the host via the file. */
		*rsltLine,	/* The resultant line of data. */
		*dStr,		/* The display field. */
		*sStr,		/* The selector field. */
		*hStr,		/* The host field. */
		*pStr,		/* The port field. */
		*gPlusStr;	/* The gopher+ field. */
	FILE	*fPtr;		/* Pointer to the menu file. */
	int	numItems,	/* Number of items in the menu. */
		error = 0,	/* Did we encounter an error? */
		i;		/* A loop counter. */

	if (!head)
		head = tail = CreateNode(selStr,hostStr,OnlyDigits(portStr));
	else
		tail = InsertNode(tail,CreateNode(selStr,hostStr,OnlyDigits(portStr)));

	if (!dirRoot && !indent)
		{
		(void)PushPath("Gopher root",selStr,hostStr,portStr);
		WaterTree(selStr,hostStr,portStr);
		indent++;
		lineNumber++;
		if (menuFlag)			/* User wants the menu displayed. */
			{
			if (printLineNumbers)	/* User wants line numbers too. */
				(void)fprintf(stdout,"%7ld ",lineNumber);
			(void)fprintf(stdout,"Gopher root is [%s] port = %s",hostStr,portStr);
			if (selStr[0])
				(void)fprintf(stdout," branch %s",selStr);
			(void)fprintf(stdout,"\n");
			}
		dirRoot = BuildDirTree(dirRoot);
		}

	if (!(error = ContactHost(hostStr,Str2Int(portStr))))
		{
		/* Write the information to a temporary file. */
		if (fPtr = fopen(tmpfilename,"w"))
			{
			(void)SendString(selStr);
			(void)SendString("\r\n");
			for (numItems = 0; LooksGood(&rsltLine,rdPtr); numItems++)
				(void)fprintf(fPtr,"%s",(buildDataFile) ? HostField2Lower(rsltLine) : rsltLine);
			CloseReadNwriter();
			(void)fclose(fPtr);
			if (buildDataFile)
				(void)DoSystemCall(Mysprint(catcommand,tmpfilename,fileName));
			if (debug)
				(void)fprintf(rptPtr,"Connections are now closed with %d item%c found\n",
					numItems,numItems > 1 ? 's' : '\0');
			}
		else
			error = fprintf(rptPtr,"error: ProcessMenu cannot create [%s]\n",tmpfilename);

		if (veronica)
			{
			char	s[1024];
			if ((strlen(hostStr) + strlen(portStr) + 2) < 1024)
				{
				(void)sprintf(s,"%s\t%s",hostStr,portStr);
				BuildTree(&vRoot,s,(long)-1);
				}
			else		/* Better safe than sorry. */
				{
				(void)fprintf(rptPtr,"error: attempted to overwrite memory\n");
				exit(1);
				}
			}

		/* Read the temporary file into a dynamic array. */
		if (!error && (data = (char **)malloc((unsigned)numItems * sizeof(char *))))
			{
			if (fPtr = fopen(tmpfilename,"r"))
				{
				for (i = 0; i < numItems && !error; i++)
					if (data[i] = (char *)malloc((unsigned)(strlen(rsltLine = GetString(fPtr)) + 1) * sizeof(char)))
						(void)strcpy(data[i],rsltLine);
					else
						{
						error = fprintf(rptPtr,"error: ProcessMenu cannot get memory for the %dth element\n");
						for ( ; i; i--)
							free((char *)data[i]);
						free((char *)data);
						}
				(void)fclose(fPtr);
				if (Unlink(tmpfilename))
					(void)fprintf(rptPtr,"error: %d could delete [%s]\n",tmpfilename,errno);
				}
			else
				error = fprintf(rptPtr,"error: ProcessMenu cannot read [%s]\n",tmpfilename);
			}
		else if (!error && numItems)
			error = fprintf(rptPtr,"error: ProcessMenu cannot get memory for the %d data items\n",numItems);

		if (!error)
			{
			for (i = 0; i < numItems; i++)
				{
				dStr = MyStrTok(data[i],'\t');
				sStr = EmptyStringCheck(MyStrTok((char *)NULL,'\t'));
				hStr = StrToLower(MyStrTok((char *)NULL,'\t'));

				/* Make sure we are gopher+ savvy. */
				pStr = OnlyDigits(gPlusStr = MyStrTok((char *)NULL,'\0'));
				gPlusStr = pStr + strlen(pStr) + 1;

				/* Make sure we were able to parse the line correctly. */
				if (!hStr[0] || !pStr[0])	/* Yipes!  Skip this item. */
					continue;

				/* Now make sure we don't try to traverse a gopher+ ASK block directory. */
				if (dStr[0] == A_DIRECTORY && strchr(gPlusStr,'?'))
					{
					if (debug)
						(void)fprintf(rptPtr,"Encountered an ASK block directory: SKIPPING\n");
					continue;
					}

				if (debug)
					{
					(void)fprintf(rptPtr,"\tdStr = [%s]\n",dStr);
					(void)fprintf(rptPtr,"\tsStr = [%s]\n",sStr);
					(void)fprintf(rptPtr,"\thStr = [%s]\n",hStr);
					(void)fprintf(rptPtr,"\tpStr = [%s]\n",pStr);
					}

				if (listHosts)
					{
					char	s[1024];
					if ((strlen(hStr) + strlen(pStr) + 2) < 1024)
						{
						(void)sprintf(s,"%s\t%s",hStr,pStr);
						BuildTree(&root,s,(long)-1);
						}
					else		/* Better safe than sorry. */
						{
						(void)fprintf(rptPtr,"error: attempted to overwrite memory\n");
						exit(1);
						}
					}

				if (menuFlag && Host2Search(hStr))
					{
					PrintItem(dStr,indent);
					if (info4allItems || (info4dirsOnly && dStr[0] == A_DIRECTORY))
						(void)fprintf(stdout,"\t%s %s",hStr,pStr);
					(void)fprintf(stdout,"\n"),++lineNumber;
					}

				if (DoWeTraverse(dStr[0],sStr,hStr,pStr,indent + 1))
					{
					(void)PushPath(dStr + 1,sStr,hStr,pStr);
					if (!CantGet2Host(nogoHead,EMPTYSTRING,hStr,pStr))
						ProcessMenu(sStr,hStr,pStr,indent + 1);
					else
						{
						(void)fprintf(rptPtr,"warning: ProcessMenu could not previously connect to %s %s\n",hStr,pStr);
						(void)PrintPath();
						}
					(void)PopPath();
					}
				}
			/* Free up the memory we acquired. */
			for (i = 0; i < numItems; i++)
				free((char *)data[i]);
			free((char *)data);
			}
		}
	else		/* Keep track of hosts we can't connect to. */
		{
		PostContactHostError(error,hostStr,portStr);
		(void)PrintPath();

		if (!nogoHead)
			nogoHead = nogoTail = CreateNode(EMPTYSTRING,hostStr,portStr);
		else
			nogoTail = InsertNode(nogoTail,CreateNode(EMPTYSTRING,hostStr,portStr));
		}

	if (head != tail)
		tail = RemoveNode(tail);

}	/* ProcessMenu */

/*****************************************************************************
 * PostTime2Process simply prints the time required for a given run.  Where
 * the run could be building the datafile, building the tables, or starting
 * up as a search engine.
 ****************************************************************************/
void PostTime2Process()
{	time_t	theTime,	/* The current time. */
		timeUsed;	/* The time in seconds required for a given run. */
	int	hours,		/* The hours for the process. */
		minutes,	/* The minutes for the process. */
		seconds;	/* The seconds for the process. */

	(void)time(&theTime);
	timeUsed = theTime - startTime;
	if (doSearch)
		(void)fprintf(rptPtr,"The time required to load the index table took ");
	else if (buildIndex)
		(void)fprintf(rptPtr,"The time required to build the index table took ");
	else
		(void)fprintf(rptPtr,"The time required to process the menus of %s took ",initialHost);
	hours = (int)(timeUsed / 3600);
	timeUsed -= hours * 3600;
	minutes = (int)(timeUsed / 60);
	timeUsed -= minutes * 60;
	seconds = (int)timeUsed;
	(void)fprintf(rptPtr,"%d:%d:%d\n",hours,minutes,seconds);

}	/* PostTime2Process */

/*****************************************************************************
 * ValidVariableCheck returns true if the variables are valid and false
 * otherwise.  This routine is a redunant valididy checker and gets called
 * via ReadConfFile().
 ****************************************************************************/
int ValidVariableCheck()
{	int	error = 0;	/* Did we get an error? */

	if (!hostname)
		if (!(hostname = GetHostName(HOSTDOMAIN)))
			{
			(void)fprintf(rptPtr,"error: cannot get name of this machine\n");
			(void)fprintf(rptPtr,"       HOSTDOMAIN = \"%s\" in jughead.conf.h may be wrong.\n",HOSTDOMAIN);
			error = 1;
			}
	if (!jugheadhelp)
		jugheadhelp = MakeStr(jugheadhelp,JUGHEADHELP);
	if (!errorhost)
		errorhost = MakeStr(errorhost,ERRORHOST);
	if ((strlen(USAGERROR) + strlen(jugheadhelp)) > 1000 ||
            (strlen(VERSION)   + strlen(jugheadhelp)) >  975)
		{
		(void)fprintf(rptPtr,"error: too many characters in [%s], [%s], or [%s]\n",
				jugheadhelp,USAGERROR,VERSION);
		error = 1;
		}

	return(!error);

}	/* ValidVariableCheck */

/*****************************************************************************
 * SpankDuplicateEntries is a quick and dirty hack to ensure no duplicate
 * entries exist in the data file.  The system call to 'sort' works great
 * except duplicate entries can exist if one is a gopher0 item and the other
 * a gopher+ item.  So, what this routine does is determine if we have 2
 * entries pointing to the same gopher0 and gopher+ item, and if so keeps
 * the gopher+ item.  Basicly what I am doing here is grabbing two lines
 * from the data file, and parsing each line into the display part 'dStrN'
 * and the rest 'restN', where 'restN' is the selector string, host string,
 * port string and optional gopher+ items.  A duplicate item has the
 * selector, host, and port which are the same, and when this routine gets
 * called the only thing different will be the gopher+ stuff at the end of
 * the line.  Then I move the second line into the first line and do the
 * comparision stuff all over again.
 * 
 * I should mention that by the time this routine is called the data file
 * will have been sorted such that if there exists a gopher0 and gopher+
 * items pointing to the same thing, the first line is the gopher0 item and
 * the second line is the gopher+ item.
 *
 * This routine does NOT make the most effecient use of memory, due to the
 * copying of 'line2' into 'line1', and the duplicated call to MyStrTok() to
 * re-parse the line.  Oh well!  At least the sucker works.
 *
 * This routine removes the duplicate entries from file 'f1' and stores the
 * results in file 'f2'.
 ****************************************************************************/
void SpankDuplicateEntries(f1,f2)
	char	*f1,			/* Temporary file name. */
		*f2;			/* Name of the data file. */
{	FILE	*fin,			/* Pointer to the data file. */
		*fout;			/* Pointer to the temp file. */
	char	line1[BUFFERSIZE],	/* The first line we're looking at. */
		line2[BUFFERSIZE],	/* The second line we looking at. */
		*dStr1, *rest1,		/* The parsed first line. */
		*dStr2, *rest2;		/* The parsed second line. */
	int	done = 0;		/* Are we done yet? */

	if (fin = fopen(f1,"r"))
		{
		if (fout = fopen(f2,"w"))
			{
			if (ReadLine(fin,line1,BUFFERSIZE) != EOF)
				{
				/* Parse the first line. */
				dStr1 = MyStrTok(line1,'\t');
				rest1 = MyStrTok((char *)NULL,'\0');

				while (!done && (ReadLine(fin,line2,BUFFERSIZE) != EOF))
					{
					/* Parse the second line. */
					dStr2 = MyStrTok(line2,'\t');
					rest2 = MyStrTok((char *)NULL,'\0');

					if (!StrnCmp(rest1,rest2,strlen(rest1) - 2))
						{
						(void)fprintf(fout,"%s\t%s\n",dStr2,rest2);
						done = ReadLine(fin,line1,BUFFERSIZE) == EOF;
						}
					else
						{
						(void)fprintf(fout,"%s\t%s\n",dStr1,rest1);
						(void)sprintf(line1,"%s\t%s",dStr2,rest2);
						}

					/* Re-parse the first line.  Yuck! */
					dStr1 = MyStrTok(line1,'\t');
					rest1 = MyStrTok((char *)NULL,'\0');
					}
				(void)fprintf(fout,"%s\t%s\n",dStr1,rest1);
				}
			(void)fclose(fout);
			}
		else
			(void)fprintf(rptPtr,"error: could not open [%s] for writing.\n",f2);
		(void)fclose(fin);
		}
	else
		(void)fprintf(rptPtr,"error: could not open [%s] for reading.\n",f1);

}	/* SpankDuplicateEntries */

/*****************************************************************************
 * PrintJugheadConf prints the jughead configuration variables if debug is
 * true.
 ****************************************************************************/
void PrintJugheadConf()
{
	if (debug)
		{
		(void)fprintf(rptPtr,"Using [%s] as hostname\n",hostname ? hostname : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as jugheadhelp\n",jugheadhelp ? jugheadhelp : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as errorhost\n",errorhost ? errorhost : "NULL");
		(void)fprintf(rptPtr,"Using [%d] as port2use\n",port2use);
		(void)fprintf(rptPtr,"Using [%d] as maxprocs\n",maxprocs);
		(void)fprintf(rptPtr,"Using [%s] as delimiters\n",delimiters ? delimiters : "NULL");
		(void)fprintf(rptPtr,"Using [%d] as maxitems2return\n",maxitems2return);
		(void)fprintf(rptPtr,"Using [%s] as defaultboolop\n",defaultboolop ? defaultboolop : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as jugheadhelptitl\n",jugheadhelptitl ? jugheadhelptitl : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as usagerror\n",usagerror ? usagerror : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as wildcarderr\n",wildcarderr ? wildcarderr : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as tomnyprcserr\n",tomnyprcserr ? tomnyprcserr : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as noforkerr\n",noforkerr ? noforkerr : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as gtstrerr\n",gtstrerr ? gtstrerr : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as readerr\n",readerr ? readerr : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as tmpfilename\n",tmpfilename ? tmpfilename : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as catcommand\n",catcommand ? catcommand : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as sortcommand\n",sortcommand ? sortcommand : "NULL");
		(void)fprintf(rptPtr,"Using [%s] as touchcommand\n",touchcommand ? touchcommand : "NULL");
		}

}	/* PrintJugheadConf */

/*****************************************************************************
 * InitializeTheWorld initializes the global variables to either 0, NIL, or
 * the respective default value and always returns true.
 ****************************************************************************/
int InitializeTheWorld()
{
	initialHost = selStr = hostStr = portStr =
	fileName = userName = logFile = veronica = (char *)NIL;

	numSearchHosts = debug = info4dirsOnly = info4allItems = listHosts =
	listHostsNPorts = buildDataFile = printLineNumbers = printDTree =
	printDTreeDirs = time2process = buildIndex = doSearch = 0;

	menuFlag = 1;

	selStr = EMPTYSTRING;
	portStr = DEFAULTPORT;

	head = tail = nogoHead = nogoTail = (List *)NIL;

	hostname	= MakeStr(hostname,HOSTDOMAIN);
	jugheadhelp	= MakeStr(jugheadhelp,JUGHEADHELP);
	errorhost	= MakeStr(errorhost,ERRORHOST);
	port2use	= PORT2USE;
	maxprocs	= MAXPROCS;
	delimiters	= MakeStr(delimiters,DELIMITERS);
	maxitems2return	= MAXITEMS2RETURN;
	defaultboolop	= MakeStr(defaultboolop,DEFAULTBOOLOP);
	jugheadhelptitl	= MakeStr(jugheadhelptitl,JUGHEADHELPTITL);
	usagerror	= MakeStr(usagerror,USAGERROR);
	wildcarderr	= MakeStr(wildcarderr,WILDCARDERR);
	tomnyprcserr	= MakeStr(tomnyprcserr,TOMNYPRCSERR);
	noforkerr	= MakeStr(noforkerr,NOFORKERR);
	gtstrerr	= MakeStr(gtstrerr,GTSTRERR);
	readerr		= MakeStr(readerr,READERR);
	tmpfilename	= MakeStr(tmpfilename,TMPFILENAME);
	catcommand	= MakeStr(catcommand,CATCOMMAND);
	sortcommand	= MakeStr(sortcommand,SORTCOMMAND);
	touchcommand	= MakeStr(touchcommand,TOUCHCOMMAND);

	return(1);

}	/* InitializeTheWorld */

/*****************************************************************************
 * GetCommandLine returns a string which makes up the command line.
 * This routine is used so we write a copy of the command line to the
 * jughead.pid file which allows an easy use for SIGHUP and restarting
 * jughead.  The file jughead.pid is only used when jughead is started
 * as a search engine with the -S flag.
 ****************************************************************************/
char *GetCommandLine(argc,argv)
	int	argc;
	char	*argv[];
{	int	i,		/* A loop counter. */
		len;		/* Length of the string to create. */
	char	*str;		/* The string to return. */

	for (i = len = 0; i < argc; i++)
		len += strlen(argv[i]) + 1;

	if (str = (char *)malloc((unsigned)len * sizeof(char)))
		{
		for (i = 0; i < len; i++)	/* Null the whole string. */
			str[i] = '\0';

		for (i = 0; i < argc; i++)	/* Make up the string. */
			{
			(void)strcat(str,argv[i]);
			if (i < (argc - 1))
				(void)strcat(str," ");
			}

		return(str);			/* Return the sucker. */
		}
	else					/* What the hell? */
		{
		(void)fprintf(rptPtr,"error: GetCommandLine could not get memory\n");
		return((char *)NIL);
		}

}	/* GetCommandLine */

/*****************************************************************************
 * ControlCAbort does some cleaning up and is only called if Control-C
 * is encountered and we are not debugging.
 ****************************************************************************/
static void ControlCAbort()
{
	if (debug)
		(void)fprintf(rptPtr,"\nControl-C encountered: debug is on, not cleaning up.\n");
	else
		{
		(void)fprintf(rptPtr,"\nControl-C encountered: cleaning up and aborting.\n");

		/* Make sure the temporary file is history. */
		if (Unlink(tmpfilename))
			(void)fprintf(rptPtr,"error: %d could delete [%s]\n",tmpfilename,errno);
		}
	exit(1);

}	/* ControlCAbort */

/*****************************************************************************
 * main is the heart of this program and is simply the traffic director
 * calling the appropriate routines the user wants jughead to do.
 ****************************************************************************/
int main(argc,argv)
	int	argc;
	char	*argv[];
{	char	*cmdLine;		/* The entire command line. */
	time_t	clock;			/* The current time. */

	if (!InitializeTheWorld())
		{
		(void)fprintf(rptPtr,"error: could not initialize the environment\n");
		if (debug)
			(void)fprintf(rptPtr,"could not initialize the environment\n");
		exit(1);
		}

	cmdLine = GetCommandLine(argc,argv);

	if (!GetArguments(argc,argv,&fileName,&logFile))	/* Get out of Dodge. */
		return(-1);

	/* Handle the jughead.conf configuration file if it exists. */
	if (initialHost)
		vctlHost = initialHost;		/* To support the "Disallow:" directive. */
	else
		vctlHost = EMPTYSTRING;		/* Must be running as a search engine. */
	vctlPort = portStr;			/* To support the "Disallow:" directive. */
	if (debug)
		{
		PrintJugheadConf();
		(void)fprintf(rptPtr,"vctlHost = [%s], vctlPort = [%s]\n",vctlHost,vctlPort);
		}
	(void)ReadConfFile(JUGHEADCONF);
	if (debug)
		PrintTheList();

	if (doSearch)		/* Running with the -S flag. */
		DoSearch(cmdLine,fileName,logFile);	/* Never returns. */

	/* Set things up so we clean up if Control-C is encountered. */
	(void)signal(SIGINT,ControlCAbort);

	if (time2process)	/* Running with the -t flag. */
		(void)time(&startTime);

	if (logFile)		/* Open the logFile file for appending. */
		{
		if (!(rptPtr = fopen(logFile,"a")))
			(void)fprintf(rptPtr = stderr,"warning: could not open [%s] writing to stderr\n",logFile);
		(void)time(&clock);
		(void)fprintf(rptPtr,"------------------------------------------------------\n");
		(void)fprintf(rptPtr,"%s%s\n",ctime(&clock),cmdLine);
		}

	if (buildIndex)		/* Build the index tables and bail out. */
		if (CreateWordsTree(fileName))
			{
			MakeHashTables(fileName,root);
			if (time2process)
				PostTime2Process();
			return(0);
			}
		else
			return(-1);

	if (cmdLine)		/* Don't need this anymore. */
		free((char *)cmdLine);

	if (buildDataFile)	/* Make sure we start with a fresh data file. */
		if (Unlink(tmpfilename))
			(void)fprintf(rptPtr,"error: %d could delete [%s]\n",tmpfilename,errno);

	ProcessMenu(selStr,hostStr,portStr,FIRSTMENU);

	if (buildDataFile)	/* Make sure we remove duplicate entries. */
		{
		if (!logFile)
			(void)fprintf(rptPtr,"Removing any duplicates from [%s].\n",fileName);
		(void)DoSystemCall(Mysprint(sortcommand,tmpfilename,fileName));
		SpankDuplicateEntries(tmpfilename,fileName);
		if (Unlink(tmpfilename))
			(void)fprintf(rptPtr,"error: %d could delete [%s]\n",tmpfilename,errno);
		}

	if (listHosts || listHostsNPorts)	/* Print the hosts tree. */
		{
		(void)fprintf(rptPtr,"Hosts %saccessed via %s\n",
				(listHostsNPorts) ? "and ports " : "",hostStr);
		PrintTree(root,listHosts + listHostsNPorts);
		}

	if (printDTree)		/* Print the directory tree. */
		{
		(void)fprintf(rptPtr,"Directory tree with the %s\n",
				(printDTreeDirs) ? "directories being served" : "number of directories served");
		PrintDirTree(dirRoot,PSTR);
		}

	if (nogoHead)		/* Print the hosts we couldn't connect with. */
		{
		(void)fprintf(rptPtr,"Could not connect to the following host(s):\n");
		while (nogoHead)
			{
			(void)fprintf(rptPtr,"    %s %s\n",nogoHead->info.hStr,nogoHead->info.pStr);
			nogoHead = nogoHead->next;
			}
		}

	if (buildDataFile)	/* Report on the number of items we have. */
		{
		if (!logFile)
			{
			(void)fprintf(rptPtr,"[%s] now contains the information with duplicates removed\n",
					fileName);
			(void)fprintf(rptPtr,"and points to %ld different gopher menu items\n",
					NumberOfLines(fileName));
			}
		else
			{
			(void)time(&clock);
			(void)fprintf(rptPtr,"jughead points to %ld gopher menu items and was rebuilt %s",
					NumberOfLines(fileName),ctime(&clock));
			}
		}

	if (veronica)		/* Build the data file for veronica. */
		CreateVeronicaFile(vRoot);

	if (logFile)		/* All finished with the log. */
		(void)fclose(rptPtr);

	/* Make sure the temporary file is history. */
	if (Unlink(tmpfilename))
		(void)fprintf(rptPtr,"error: %d could delete [%s]\n",tmpfilename,errno);

	if (time2process)	/* Finished with the -t flag. */
		PostTime2Process();

	return(0);

}	/* main */
