/*	$Id: serve.c,v 1.6 2005/08/09 15:06:23 aon Exp $	*/

/*
 * Copyright (c) 2005 Antti Nyknen <aon@iki.fi>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <sys/socket.h>

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "serve.h"

#ifndef MAXBSIZE
#define MAXBSIZE (64 * 1024)
#endif

extern const char *docroot;

static const char rcsid[] = "$Id: serve.c,v 1.6 2005/08/09 15:06:23 aon Exp $";

static char	*getdescr(char *, char *);
static void	 fserve_error(FILE *, char *);

/*
 * Determine a Gopher type field from a filename.
 */
char
gettype(char *filename)
{
	char *c;

	c = strrchr(filename, '.');
	if (c == NULL)
		return('0');
	c++;

	if (strcmp(c, "hqx") == 0)
		return('4');
	if (strcmp(c, "tar") == 0 || strcmp(c, "rar") == 0 ||
	    strcmp(c, "zip") == 0) 
		return('5');
	if (strcmp(c, "uu") == 0)
		return('6');
	/*
	 * Some common binary formats to be listed as BIN.
	 * Not sure if this makes any difference, though.
	 */
	if (strcmp(c, "bz2") == 0 || strcmp(c, "gz") == 0 ||
	    strcmp(c, "iso") == 0 || strcmp(c, "mp3") == 0 ||
	    strcmp(c, "Z") == 0)
		return('9');
	if (strcmp(c, "gif") == 0)
		return('g');
	if (strcmp(c, "jpg") == 0 || strcmp(c, "png") == 0)
		return('I');

	return('0');
}

/*
 * Serve a directory index to the client.
 */
void
serve_directory(int fd, char *path)
{
	char type, extramap_path[MAXPATHLEN];
	char server_name[NI_MAXHOST], server_port[6];
	int error, emap;
	DIR *dh;
	FILE *fh;
	struct dirent *de;
	struct sockaddr_storage server_ss;
	socklen_t namelen;

	namelen = sizeof(server_ss);
	error = getsockname(fd, (struct sockaddr *)&server_ss, &namelen);
	if (error < 0) {
		syslog(LOG_ERR, "getsockname: %m");
		_exit(EXIT_FAILURE);
	}

	error = getnameinfo((struct sockaddr *)&server_ss, namelen,
	    server_name, sizeof(server_name), server_port, sizeof(server_port),
	    NI_NUMERICSERV);
	if (error < 0) {
		syslog(LOG_ERR, "getnameinfo: %m");
		_exit(EXIT_FAILURE);
	}

	fh = fdopen(fd, "w");
	if (fh == NULL) {
		syslog(LOG_ERR, "fdopen: %m");
		_exit(EXIT_FAILURE);
	}

	dh = opendir(path);
	if (dh == NULL) {
		fserve_error(fh, "No such directory.");
		_exit(EXIT_FAILURE);
	}

	emap = 0;
	while ((de = readdir(dh)) != NULL) {
		if (strcmp(de->d_name, ".extramap") == 0) {
			emap = 1;
			continue;
		}
			
		if (de->d_name[0] == '.')
			continue;

		type = 3;
		if (de->d_type == DT_DIR) 
			type = '1';
		else if (de->d_type == DT_REG)
			type = gettype(de->d_name);
		if (type != '3') {
			if (strcmp(path, ".") == 0)
				(void)fprintf(fh, "%c%s\t/%s\t%s\t%s\r\n",
		    	    	    type, getdescr(de->d_name, path),
				    de->d_name, server_name, server_port);
			else
				(void)fprintf(fh, "%c%s\t/%s/%s\t%s\t%s\r\n",
				    type, getdescr(de->d_name, path),
				    path, de->d_name, server_name,
				    server_port);
		}
	}

	(void)fflush(fh);

	if (emap == 1) {
		(void)snprintf(extramap_path, MAXPATHLEN, "%s/.extramap", path);
		serve_file(fd, extramap_path);
	}

	fprintf(fh, ".\r\n");
	fclose(fh);
}

/*
 * fserve_error wrapper for use with file descriptors.
 */
void
serve_error(int fd, char *errmsg)
{
	FILE *fh;

	fh = fdopen(fd, "w");
	if (fh == NULL) {
		syslog(LOG_ERR, "fdopen: %m");
		_exit(EXIT_FAILURE);
	}

	fserve_error(fh, errmsg);
	fclose(fh);
}

void
serve_file(int fd, char *path)
{
	char buf[MAXBSIZE];
	int rfd;
	ssize_t nb;

	rfd = open(path, O_RDONLY, 0);
	if (rfd < 0) {
		serve_error(fd, "Can't open file.");
		_exit(EXIT_FAILURE);
	}

	while ((nb = read(rfd, buf, MAXBSIZE)) != -1 && nb != 0)
		write(fd, buf, nb);

	(void)close(rfd);
}

/*
 * Send an error message to the client.
 */
static void
fserve_error(FILE *fh, char *errstr)
{
	fprintf(fh, "3%s\terror.host\t1\r\n.\r\n", errstr);
}

/*
 * Read the description of a file.
 */
static char *
getdescr(char *file, char *dir)
{
	FILE *fh;
	char buf[BUFSIZ], dfp[MAXPATHLEN], *p;

	(void)snprintf(dfp, sizeof(dfp), "%s/.descriptions", dir);
	
	fh = fopen(dfp, "r");
	if (fh == NULL)
		return(strdup(file));

	while (fgets(buf, sizeof(buf), fh) != NULL) {
		if ((p = strchr(buf, '\n')) == NULL)
			continue;
		*p = '\0';
		if (strncmp(buf, file, strlen(file)) == 0) {
			p = strchr(buf, '/');
			if (p == NULL || strlen(p) < 2)
				return(strdup(file));
			return(strdup(p+1));
		}
	}

	return(strdup(file));
}
