/*	$Id: main.c,v 1.8 2005/08/09 16:09:56 aon Exp $	*/

/*
 * Copyright (c) 2005 Antti Nyknen <aon@iki.fi>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <errno.h>
#include <netdb.h>
#include <poll.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include <netinet/in.h>

#include "conf.h"
#include "serve.h"

#ifndef INFTIM
#define INFTIM (-1)
#endif /* not INFTIM */

extern struct config config;
static FILE *al_fh;

static const char rcsid[] = "$Id: main.c,v 1.8 2005/08/09 16:09:56 aon Exp $";

static void	 handle_connection(int, struct sockaddr *, socklen_t);
static void	 sigchld_handler(int);
static void	 usage(void);
static char 	*spath(char *);
int		 main(int, char *[]);

/*
 * shrewd -- serve files using the Internet Gopher Protocol (RFC1436).
 */
int
main(int argc, char *argv[])
{
	char *config_path = "/etc/shrewd.conf";
	struct addrinfo hints, *res, *res0;
	struct sockaddr_storage client_addr;
	struct pollfd *pfds;
	socklen_t addrlen;
	struct passwd *pwd = NULL;
	int ch, error, fd, *fds, fflag, i, nsock, on = 1;

	fflag = 0;
	while ((ch = getopt(argc, argv, "c:f")) != -1) {
		switch (ch) {
		case 'c':
			config_path = optarg;
			break;
		case 'f':
			fflag = 1;
			break;
		case '?':
		default:
			usage();
			/* NOTREACHED */
		}
	}

	parse_config(config_path);
	if (fflag == 0)
		if(daemon(1, 0) < 0) {
			perror("daemon");
			exit(EXIT_FAILURE);
		}

	signal(SIGCHLD, sigchld_handler);

	openlog("shrewd", (fflag == 1) ? LOG_PERROR : 0, LOG_DAEMON);
	syslog(LOG_INFO, "invoked by UID %u", getuid());

	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AI_PASSIVE;
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;

	error = getaddrinfo(NULL,
	    config.port[0] == '\0' ? "gopher" : config.port, &hints, &res0);
	if (error < 0) {
		syslog(LOG_ERR, "getaddrinfo: %s", gai_strerror(error));
		exit(EXIT_FAILURE);
	}

	nsock = 0;
	for (res = res0; res; res = res->ai_next)
		nsock++;

	fds = (int *)malloc(sizeof(int) * nsock);
	pfds = (struct pollfd *)malloc(sizeof(struct pollfd) * nsock);
	if (fds == NULL || pfds == NULL) {
		syslog(LOG_ERR, "malloc: %m");
		exit(EXIT_FAILURE);
	}

	nsock = 0;
	for (res = res0; res; res = res->ai_next) {
		fds[nsock] = socket(res->ai_family, res->ai_socktype,
		    res->ai_protocol);
		if (fds[nsock] < 0) {
			syslog(LOG_ERR, "socket: %m");
			continue;
		}

		error = setsockopt(fds[nsock], SOL_SOCKET, SO_REUSEADDR, 
		    &on, sizeof(on));
		if (error < 0) {
			syslog(LOG_ERR, "setsockopt: %m");
			close(fds[nsock]);
			fds[nsock] = -1;
			continue;
		}

#ifdef IPV6_V6ONLY
		error = setsockopt(fds[nsock], IPPROTO_IPV6, IPV6_V6ONLY,
		    &on, sizeof(on));
		if (error < 0 && res->ai_protocol == AF_INET6) {
			syslog(LOG_ERR, "setsockopt: %m");
			close(fds[nsock]);
			fds[nsock] = -1;
			continue;
		}
#endif /* IPV6_V6ONLY */

		error = bind(fds[nsock], res->ai_addr, res->ai_addrlen);
		if (error < 0) {
			syslog(LOG_ERR, "bind: %m");
			close(fds[nsock]);
			fds[nsock] = -1;
			continue;
		}

		error = listen(fds[nsock], 32);
		if (error < 0) {
			close(fds[nsock]);
			fds[nsock] = -1;
			continue;
		}

		pfds[nsock].fd = fds[nsock];
		pfds[nsock].events = POLLIN;
		nsock++;
	}
	if (nsock == 0) {
		syslog(LOG_ERR, "Couldn't establish any listener sockets.");
		exit(EXIT_FAILURE);
	}
	free(fds);
	freeaddrinfo(res0);

	if (config.user[0] != '\0') {
		pwd = getpwnam(config.user);
		if (pwd == NULL) {
			syslog(LOG_ERR, "getpwnam: %m");
			exit(EXIT_FAILURE);
		}
		endpwent();
	}

	if (config.chroot_dir[0] != '\0') {
		if (chdir(config.chroot_dir) < 0) {
			syslog(LOG_ERR, "chdir to %s: %m", config.chroot_dir);
			exit(EXIT_FAILURE);
		}
		if (chroot(config.chroot_dir) < 0) {
			syslog(LOG_ERR, "chroot to %s: %m", config.chroot_dir);
			exit(EXIT_FAILURE);
		}
		syslog(LOG_INFO, "chrooted to %s", config.chroot_dir);
	}

	if (config.user[0] != '\0') {
		if (setgid(pwd->pw_gid) < 0) {
			syslog(LOG_ERR, "setgid: %m");
			exit(EXIT_FAILURE);
		}
		if (setuid(pwd->pw_uid) < 0) {
			syslog(LOG_ERR, "setuid: %m");
			exit(EXIT_FAILURE);
		}
		syslog(LOG_INFO, "set uid/gid to %u/%u",
		    pwd->pw_uid, pwd->pw_gid);
	}

	if (config.accesslog_file[0] != '\0') {
		al_fh = fopen(config.accesslog_file, "a");
		if (al_fh == NULL) {
			syslog(LOG_ERR, "%s: %m", config.accesslog_file);
			exit(EXIT_FAILURE);
		}
	}


	error = chdir(config.document_root);
	if (error < 0) {
		syslog(LOG_ERR, "chdir %s: %m", config.document_root);
		exit(EXIT_FAILURE);
	}

	for (;;) {
		if (poll(pfds, (nfds_t)nsock, INFTIM) < 0) {
			if (errno == EINTR)
				continue;
			syslog(LOG_ERR, "poll: %m");
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < nsock; i++)
			if ((pfds[i].revents & POLLIN) != 0) {
				addrlen = (socklen_t)sizeof(client_addr);
				fd = accept(pfds[i].fd,
				(struct sockaddr *)&client_addr, &addrlen);
				handle_connection(fd,
				    (struct sockaddr *)&client_addr,
				    addrlen);
				close(fd);
			}
	}
	/* NOTREACHED */

	return(1);
}

/*
 * Handle an incoming gopher client connection.
 */
static void
handle_connection(int fd, struct sockaddr *client_addr, socklen_t addrlen)
{
	char *path, *nl, selector[256], name[NI_MAXHOST];
	int error;
	time_t tv;
	ssize_t nbytes;
	struct stat sb;

	switch(fork()) {
	case 0:
		break;
	case -1:
		syslog(LOG_ERR, "fork: %m");
		exit(EXIT_FAILURE);
	default:
		return;
	}

	nbytes = read(fd, selector, sizeof(selector)-1);
	selector[nbytes] = '\0';

	error = getnameinfo(client_addr, addrlen, name, NI_MAXHOST,
	    NULL, 0, NI_NUMERICHOST);
	if (error < 0) {	
		syslog(LOG_ERR, "getnameinfo: %s", gai_strerror(errno));
		_exit(EXIT_FAILURE);
	}

	nl = strstr(selector, "\r\n");
	if (nl == NULL)
		_exit(EXIT_FAILURE);
	*nl = '\0';

	if (config.accesslog_file[0] != '\0') {
		tv = time(NULL);
		(void)fprintf(al_fh, "%s : %s : %s",
		    name, selector, ctime(&tv));
		(void)fflush(al_fh);
	}

	path = NULL;
	if (nl == selector)
		path = strdup(".");
	else {
		path = spath(selector);
		if (path == NULL) {
			serve_error(fd, "Invalid path.");
			_exit(EXIT_FAILURE);
		}
	}

	if (stat(path, &sb) < 0) {
		serve_error(fd, "No such file or directory.");
		_exit(EXIT_FAILURE);
	}

	if ((S_IROTH | sb.st_mode) != sb.st_mode) {
		serve_error(fd, "File isn't world readable.");
		_exit(EXIT_FAILURE);
	}

	if (S_ISDIR(sb.st_mode))
		serve_directory(fd, path);
	else if (S_ISREG(sb.st_mode))
		serve_file(fd, path);

	free(path);
	_exit(EXIT_SUCCESS);
}

/*
 * SIGCHLD handler.
 */
static void
sigchld_handler(int c)
{
	int old_errno, rv;

	old_errno = errno;
	while((rv = waitpid(-1, NULL, WNOHANG)) > 0 || (rv == -1 &&
	    errno == EINTR));
	errno = old_errno;
}
		

/*
 * Print usage message, exit. 
 */
static void
usage(void)
{
	(void)fprintf(stderr, "Usage: shrewd [-f] [-c config]\n");
	exit(EXIT_FAILURE);
}

/*
 * Copy a path from dst to src if it's valid, making it
 * docroot-relative.
 */
static char *
spath(char *src)
{
        if (src == NULL || src[0] != '/' || strstr(src, "..") != NULL ||
            strstr(src, "/.") != NULL || strlen(src) > MAXPATHLEN)
                return(NULL);

        if (strlen(src) == 1)
                /* It's the document root. */
                return(".");

	/* Strip the preceding slash. */
	return(strdup(src + 1));
}
