//
// LogFormatter.java
// Part of the Aftershock Project, see README for details
// Copyright 2003 Rob Linwood (rob@linwood.us)
//

package aftershock.log;

import aftershock.io.GopherConnection;
import aftershock.io.GopherRequest;
import aftershock.engine.Response;

import java.util.Locale;
import java.util.Date;
import java.text.DateFormat;

public class LogFormatter {

  /* 
   * Format string tokens:
   * 
   * For all format strings:
   * {DATE}     - date, formatted to appropriate locale
   * {TIME}     - time, as above
   * {NEWLINE}  - newline character
   *
   * For access formats:
   * {REMOTEIP} - IP Address of client
   * {SELECTOR} - selector used
   * {PROTOCOL} - either "gopher" or "gopher+"
   * {ERROR}    - the text of the error condition, if any
   *
   * For Exception formats:
   * {EXCEPTION} - Exception message
   * {STACK}     - stack trace from exception
   *
   * For Error and Info formats:
   * {MESSAGE} - the message passed
   */

  static Locale currentLocale = null;

  static {
    currentLocale = Locale.getDefault();
  }

  public static String format( String formatString ) {
    Date today = new Date();
    DateFormat dateFormatter = DateFormat.getDateInstance(DateFormat.SHORT,
							  currentLocale);
    String date = dateFormatter.format( today );

    DateFormat timeFormatter =
      DateFormat.getTimeInstance(DateFormat.DEFAULT, currentLocale);

    String time = timeFormatter.format( today );
    /*
     * Because String.replaceAll() takes a regex pattern as its first
     * argument, the curly braces have to be escaped with a \
     */

    String formatted = formatString
      .replaceAll("\\{DATE\\}", date)
      .replaceAll("\\{TIME\\}", time)
      .replaceAll("\\{NEWLINE\\}", "\n");


    return formatted;
  }

  public static String format( Exception e, String formatString ) {
    StringBuffer stack = new StringBuffer(200); // guess
    StackTraceElement[] eles = e.getStackTrace();

    for( int i=0; i<eles.length; i++ ) {
      stack.append( eles[i].toString() ).append( "\n" );
    }
    
    String formatted = format(formatString)
      .replaceAll("\\{EXCEPTION\\}", e.getMessage() )
      .replaceAll("\\{STACK\\}", stack.toString() );

    return formatted;
  }

  public static String format( String msg, String formatString ) {
    String formatted = format(formatString)
      .replaceAll("\\{MESSAGE\\}", msg);

    return formatted;
  }

  public static String format( GopherConnection conn, String formatString ) {
    GopherRequest req = conn.getRequest();
    
    String remoteIP = conn.getChannel().socket().getInetAddress().toString();
    String selector = req.getSelector().replaceAll("\\r\\n", " [nl] " );
    String protocol = req.isGopherPlus() ? "gopher+" : "gopher";

    
    
    String formatted = format(formatString)
      .replaceAll("\\{REMOTEIP\\}", remoteIP)
      .replaceAll("\\{SELECTOR\\}", selector)
      .replaceAll("\\{PROTOCOL\\}", protocol)
      .replaceAll("\\{ERROR\\}", "");

    return formatted;
  }
}
