//
// GopherListener.java
// Based on soma2.http.HttpListener
// Part of the Aftershock Project, see README for details
// Copyright 1998-2003 Rob Linwood (rob@linwood.us)
//


package aftershock.io;

import aftershock.conf.Configuration;

import java.net.*;
import java.io.*;
import java.util.*;
import java.nio.*;
import java.nio.channels.*;
import java.nio.charset.*;

/**
 * Listen for Gopher connections.
 * 
 */

public class GopherListener {

  int port;
  int bufferSize;

  String charsetName = null;
  Charset charset = null;
  CharsetEncoder encoder = null;
  CharsetDecoder decoder = null;

  Selector serverSelector = null;

  String hostName = null;

  String lineEnd = null;
  int lineEndLength = 1;

  public GopherListener( Configuration config ) throws Exception {

    // Other assertions are handled in the set*() methods
    assert config.getCharsetName() != null;
    assert config.getHostName() != null;

    this.hostName = config.getHostName();
    this.charsetName = config.getCharsetName();
    setPort( config.getPortNumber() );
    setBufferSize( config.getListenerBufferSize() );

    serverSelector = Selector.open();
    
    charset = Charset.forName( charsetName );
    encoder = charset.newEncoder();
    decoder = charset.newDecoder();     

    final Configuration xConfig = config;
    Thread shutdownThread = new Thread() {
	public void run() {
	  
	  if( !serverSelector.isOpen() ) {
	    xConfig.getLogManager().logError("Tried to close an already-closed"+
					    " serverSelector" );
	  }

	  try {
	    serverSelector.close();
	  } catch( IOException e ) {
	    xConfig.getLogManager().logError( "I/O Error occured on Selector" +
					     "shutdown. See exception log." );
	    xConfig.getLogManager().logException( e );
	  }
	}
      };
  
    try {
      Runtime.getRuntime().addShutdownHook( shutdownThread );
    } catch( Exception e ) {
      config.getLogManager().logException( e );
    }
  }

  public ServerSocketChannel setup() throws Exception {
    ServerSocketChannel ssc = ServerSocketChannel.open();
    InetSocketAddress isa = 
      new InetSocketAddress( InetAddress.getByName(hostName), port );

    ssc.socket().bind( isa );
    ssc.configureBlocking( false );
    ssc.register( serverSelector, SelectionKey.OP_ACCEPT );

    return ssc;
  }
 	   
  private void serve( ServerSocketChannel ssc, GopherResponseFactory factory ) 
    throws Exception {

    while( (serverSelector.select() > 0) ) {
      Iterator keysIter = serverSelector.selectedKeys().iterator();

      while( keysIter.hasNext() ) {
	SelectionKey key = (SelectionKey)keysIter.next();
	keysIter.remove();
	
	if( key.isAcceptable() ) {
	  SocketChannel socket = ssc.accept();
	  socket.configureBlocking( false );
	  socket.register( serverSelector, SelectionKey.OP_READ );
	} else if( key.isReadable() ) {
	  if( key.attachment() == null ) {
	    key.attach( new GopherConnection((SocketChannel)key.channel()) );
	  }
	  processRead( key, factory );
	}
      }
    }
  }
  
  private void processRead( SelectionKey key, GopherResponseFactory factory ) 
    throws Exception {
    
    SocketChannel channel = (SocketChannel)key.channel();

    ByteBuffer dbuf = ByteBuffer.allocateDirect( bufferSize );
    
    channel.read( dbuf );
    dbuf.flip();
    
    String text = decoder.decode(dbuf).toString();
    text = chop( text );
    //    text = text.trim();
    //System.out.println( "Received: \"" + text + "\"" );

    GopherConnection conn = (GopherConnection)key.attachment();
    assert conn != null;
    GopherRequest request = new GopherRequest( text );
    conn.request = request;

    factory.createResponse().respond( conn );
  }
  
  public void listen( GopherResponseFactory factory ) {
    assert factory != null;
    try {
      ServerSocketChannel ssc = setup();
      while( true ) {
	serve( ssc, factory );
      }
    } catch( Exception e ) {
      // This is triggered most often by shutting down the server
      // (for example, with Ctrl-C), so for now its ignored.
//        System.err.println("Exception in serve().  Stack follows");
//        e.printStackTrace(System.err);

      return;
    }
  }

  public int getPort() {
    return port;
  }

  public void setPort( int p ) {
    assert p > 0;
    port = p;
  }

  public int getBufferSize() {
    return bufferSize;
  }

  public void setBufferSize( int size ) {
    assert size > 0;
    bufferSize = size;
  }

  private String chop( String str ) {

    if( str.endsWith("\r\n") ) {
      return str.substring( 0, str.length()-2 );
    } else if( str.endsWith("\n") ) {
      return str.substring( 0, str.length()-1 );
    } else if( str.endsWith("\r") ) {
      return str.substring( 0, str.length()-1 );
    } else {
      return str;
    }
  }

}
