//
// Response.java
// Part of the Aftershock Project, see README for details
// Copyright 2003 Rob Linwood (rob@linwood.us)
//

package aftershock.engine;

import aftershock.io.GopherResponse;
import aftershock.io.GopherRequest;
import aftershock.io.GopherConnection;
import aftershock.io.Pipe;
import aftershock.conf.Configuration;
import aftershock.cache.FileCache;
import aftershock.cache.CacheEntry;
import aftershock.util.BufferUtilities;
import aftershock.Aftershock;

import java.nio.ByteBuffer;
import java.io.FileNotFoundException;

public class Response implements GopherResponse {

  /*
   * Class variables
   */

  static ByteBuffer errorBadSelector = null;
  static ByteBuffer errorNotFound = null;
  static ByteBuffer gopherPlusDir = null;

  static {
    errorNotFound = BufferUtilities.toByteBuffer( "3File Not Found\tNotFound\terror.host\t1\r\n", Aftershock.asciiEncoder );
    errorBadSelector = BufferUtilities.toByteBuffer( "3Bad Selector\tBadSelector\terror.host\t1\r\n", Aftershock.asciiEncoder );
  }



  /*
   * Instance variables
   */
  Configuration config = null;
  // Error Status


  Response( Configuration config ) {
    this.config = config;

    gopherPlusDir = 
      BufferUtilities.toByteBuffer( "+-1\r\n+INFO: 1Main menu (non-gopher+)\t\t"+config.getHostName()+"\t"+config.getPortNumber()+"\r\n+ADMIN:\r\n Admin: Server Administrator\r\n Server:\r\n+VIEWS:\r\n application/gopher+-menu: <512b>\r\n+ABSTRACT:\r\n This gopher supports standard gopher access only. Use this\r\n kludge to disable gopher+ client requests by your client.\r\n", 				    Aftershock.asciiEncoder );    
  }

  public void respond( GopherConnection connection ) {
    ByteBuffer file = null;
    boolean sendDot = false;

    GopherRequest req = connection.getRequest();

    // Log Request
    config.getLogManager().logAccess( connection );

    if( req.isGopherPlus() ) {
      // Send fake Gopher+ menu
      file = gopherPlusDir;
      sendDot = true;
    } else {

      String selector = req.getSelector();
      
      String fileName = config.getGopherRoot() + selector;

      CacheEntry entry = null;
    
      if( selector.charAt(0) == '.' ) {
	file = errorBadSelector;
      } else {
	entry = config.getCache().getEntry( fileName );
      }

      if( entry == null ) {
	try {
	  entry = config.getCache().addFile( fileName ) ;
	  file = entry.getFile();
	  if( entry.isDirectory() || entry.getFileType().equals("0") ) {
	    sendDot = true;
	  }
	} catch( FileNotFoundException e ) {
	  file = errorNotFound;
	  config.getLogManager().logError( "File not found: " + fileName );
	} catch( SecurityException e ) {
	  file = errorNotFound;
	  config.getLogManager().logError( "Security error: " + fileName );
	} catch( Exception e ) {
	  config.getLogManager().logError( "Can't add file to cache: " +
					   fileName + "\n" + 
					   "See exception log for more info" );
	  config.getLogManager().logException( e );
	}
      } else {

	file = entry.getFile();

	if( entry.isDirectory() || entry.getFileType().equals("0") ) {
	  sendDot = true;
	}
      }

    }
    assert file != null : "Response: file is null";

    Pipe.send( file, connection.getChannel() );
    if( sendDot ) {
      Pipe.sendDot( connection.getChannel() );
    }
    try {
      connection.getChannel().close();
    } catch( Exception e ) {
      config.getLogManager().logException( e );
    }
  }

}
