//
// DirectoryLister.java
// Part of the Aftershock Project, see README for details
// Copyright 2003 Rob Linwood (rob@linwood.us)
//

package aftershock.engine;

import aftershock.util.BufferUtilities;
import aftershock.util.FileUtilities;
import aftershock.Aftershock;
import aftershock.conf.Configuration;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Arrays;
import java.nio.ByteBuffer;


public class DirectoryLister {

  public static ByteBuffer list( File dir, Configuration config )
    throws SecurityException {

    assert dir.isDirectory() : "DirectoryLister.list() called on non-directory";

    // Check whether a link file exists in this directory
    assert config.getLinkFileName() != null : "linkFileName is null";
    File linkFile = new File( dir, config.getLinkFileName() );
    if( linkFile.exists() ) {
      //      System.err.println( "Reading link file..." );
      return readLinkFile( linkFile, config );
    }

    File[] files = dir.listFiles( config.getFileFilter() );

    // 35 is a guess at how long the average line will be
    StringBuffer output = new StringBuffer( files.length * 35);

    for( Iterator iter=Arrays.asList(files).iterator(); iter.hasNext(); ) {
      File file = (File)iter.next();
      String fileName = file.getName();

      String relName = FileUtilities.getPathRelativeTo( file, 
							config.getGopherRoot() );

      if( file.isDirectory() ) {
	output.append( "1" + fileName + "\t" + relName + 
		       "\t" + config.getHostName() +
		       "\t" + config.getPortNumber() + "\r\n" );
      } else {
	String type = FileTypeManager.getFileType( fileName );
	if( type == null ) {
	  // No type given, assume text
	  type = "0";
	}
	output.append( type + fileName + "\t" + relName + 
		       "\t" + config.getHostName() +
		       "\t" + config.getPortNumber() + "\r\n" );
      }
    }

    return BufferUtilities.toByteBuffer( output.toString(), 
					 Aftershock.asciiEncoder );
  }


  /**
   * Read in a Link file and format it as a directory listing.
   */
  static ByteBuffer readLinkFile( File linkFile, Configuration config ) {
    BufferedReader in = null;

    try {
      in = new BufferedReader( new FileReader(linkFile) );
    } catch( FileNotFoundException e ) {
      System.err.println( "Error: Tried to read non-existant file: " + 
			  linkFile.getPath() );
      config.getLogManager().logError( "Tried to read non-existant file: " + 
				       linkFile.getPath() );
      config.getLogManager().logException( e );
      return null;
    }

    String linkName = null;
    String linkPath = null;
    String linkType = "i";
    String linkHost = config.getHostName();
    String linkPort = config.getPortNumber() + "";

    StringBuffer output = new StringBuffer();

    try {
      String input = null;
      while( (input = in.readLine()) != null ) {
	input = input.trim();
	if( input.length() > 0 ) { // skip blank lines
	  if( input.charAt(0) != '#' ) { // skip comments
	    int eqlPos = input.indexOf( '=' );
	    if( eqlPos != -1 ) {
	      String key = input.substring( 0, eqlPos ).toLowerCase();
	      String value = input.substring( eqlPos+1 ).trim();

	      if( key.equalsIgnoreCase("name") ) {
		// If linkName is not null, then that means that a record
		// has already been read in.  If so, format it.
		if( linkName != null ) {
		  output.append( linkType + linkName + "\t" +
				 linkPath + "\t" +
				 linkHost + "\t" +
				 linkPort + "\r\n" );

		  // Reset values to the default
		  linkName = null;
		  linkPath = null;
		  linkType = "i";
		  linkHost = config.getHostName();
		  linkPort = config.getPortNumber() + "";
		  
		}
		linkName = value;
	      } else if( key.equalsIgnoreCase("type") ) {
		linkType = value;
	      } else if( key.equalsIgnoreCase("path") ) {
		linkPath = value;
	      } else if( key.equalsIgnoreCase("host") ) {
		linkHost = value;
	      } else if( key.equalsIgnoreCase("port") ) {
		linkPort = value;
	      }
	    }
	  }
	}
      }
      output.append( linkType + linkName + "\t" +
		     linkPath + "\t" +
		     linkHost + "\t" +
		     linkPort + "\r\n" );
      
    } catch( IOException e ) {
      System.err.println( e );
    }

    return BufferUtilities.toByteBuffer( output.toString(), 
					 Aftershock.asciiEncoder );    

  }

}
