//
// Configuration.java
// Part of the Aftershock Project, see README for details
// Copyright 2003 Rob Linwood (rob@linwood.us)
//

package aftershock.conf;

import aftershock.cache.FileCache;
import aftershock.util.RegexFileFilter;
import aftershock.log.LogManager;
import aftershock.engine.FileTypeManager;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.FileFilter;
import java.util.Hashtable;

/**
 * <code>Configuration</code> stores configuration information, as well as
 * holding pointers to global "manager" objects (such as 
 * {@link FileCache FileCache} and {@link LogManager LogManager}).
 *
 * @author Rob Linwood (rob@linwood.us)
 */

public class Configuration {
  Hashtable variables = new Hashtable();
  
  public static final int DEFAULT_GOPHER_PORT = 70;
  static final int DEFAULT_BUFFER_SIZE = 1024;
  static final String DEFAULT_CHARSET_NAME = "US-ASCII";
  static final String DEFAULT_HOST_NAME = "127.0.0.1";

  int portNum = DEFAULT_GOPHER_PORT;
  String gopherRoot = null;
  String hostName = DEFAULT_HOST_NAME;
  FileCache cache = new FileCache( this );
  RegexFileFilter fileFilter = new RegexFileFilter();
  String linkFileName = ".Link";

  /** Check the timestamps of cached files to see if they've changed? */
  boolean checkCache = false;

  /** Buffer size to use for the gopher listener */
  int listenerBufferSize = DEFAULT_BUFFER_SIZE;
  
  /** Charcater set to use */
  String charsetName = DEFAULT_CHARSET_NAME;
  
  /** The LogManager used to handle all logging needs */
  LogManager logManager = null;
 
  /** The name of the access log file (where hits are recorded) */
  String accessLogFile = null;
  /** The name of the error log file (where server errors are recorded) */
  String errorLogFile = null;
  /** The name of the exception log file (where stack traces are recorded) */
  String exceptionLogFile = null;
  /** The name of the info log file (for informational messages) */
  String infoLogFile = null;

  /** The format to use when printing access log entries */
  String accessLogFormat = "{DATE} {TIME} - {SELECTOR} - {REMOTEIP} - {PROTOCOL}"; 
  /** The format to use when printing error log entries */
  String errorLogFormat = "{DATE} {TIME} - {MESSAGE}";
  /** The format to use when printing exception log entries */
  String exceptionLogFormat = "{DATE} {TIME} - {EXCEPTION} {NEWLINE} {STACK}";
  /** The format to use when printing info log entries */
  String infoLogFormat = "{DATE} {TIME} - {MESSAGE}";

  /** The operating system user name to switch to after all privleged
      operations are performed */
  String userName = null;

  public Configuration( String fileName ) throws FileNotFoundException {
   read( fileName );
  }

  public void read( String fileName ) throws FileNotFoundException {
    BufferedReader in = null;

    in = new BufferedReader( new FileReader(fileName) );

    try {
      String input = null;
      while( (input = in.readLine()) != null ) {
	input = input.trim();
	if( input.length() > 0 ) { // skip blank lines
	  if( input.charAt(0) != '#' ) { // skip comments
	    int colonPos = input.indexOf( ':' );
	    if( colonPos != -1 ) {
	      String key = input.substring( 0, colonPos );
	      String value = input.substring( colonPos+1 ).trim();

	      if( key.equalsIgnoreCase("Filter") ) {
		fileFilter.addFilter( value );
	      } else {
		variables.put( key.trim().toLowerCase(), value );
	      }

	    }
	  }
	}
      }
    } catch( IOException e ) {
      System.err.println( e );
    }


    hostName = getVariable( "hostName" );
    gopherRoot = getVariable( "gopherRoot" );

    userName = getVariable( "userName" );

    String linkVar = getVariable( "linkFileName" );
    if( linkVar != null ) {
      linkFileName = linkVar;
    }    

    String cacheVar = getVariable( "checkCacheTimes" );
    if( cacheVar != null && cacheVar.equalsIgnoreCase("true") ) {
      checkCache = true;
    }

    String charsetVar = getVariable( "charsetName" );
    if( charsetVar != null ) {
      charsetName = charsetVar;
    }

    String listBuffSizeVar = getVariable( "listenerBufferSize" );
    if( listBuffSizeVar != null ) {
      int lbs;
      try {
	lbs = Integer.parseInt( listBuffSizeVar );
	listenerBufferSize = lbs;
      } catch( NumberFormatException e ) {
	// ignore for now - this setting is not mandatory
      }
    }

    String fileTypesFile = getVariable( "fileTypesFile" );
    if( fileTypesFile != null ) {
      try {
	FileTypeManager.read( fileTypesFile );
      } catch( Exception e ) {
	System.out.println( e );
      }
    }

    // Store log-related variables
    accessLogFile = getVariable("accessLogFile");
    errorLogFile = getVariable("errorLogFile");
    exceptionLogFile = getVariable("exceptionLogFile");
    infoLogFile = getVariable("infoLogFile");
    
    String accessLogFormatVar = getVariable("accessLogFormat");
    if(accessLogFormatVar != null ) {accessLogFormat = accessLogFormatVar;}
    String errorLogFormatVar = getVariable("errorLogFormat");
    if(errorLogFormatVar != null ) {errorLogFormat = errorLogFormatVar;}
    String exceptionLogFormatVar = getVariable("exceptionLogFormat");
    if(exceptionLogFormatVar != null ) {exceptionLogFormat = exceptionLogFormatVar;}
    String infoLogFormatVar = getVariable("infoLogFormat");
    if(infoLogFormatVar != null ) {infoLogFormat = infoLogFormatVar;}

    logManager = new LogManager( this );
   

    try {
      portNum = Integer.parseInt( getVariable("portNum") );
    } catch( NumberFormatException e ) {
      System.out.println( e );
      System.exit( 1 );
    }

  }

  public String getVariable( String name ) {
    return (String)variables.get( name.toLowerCase() );
  }

  public int getPortNumber() { return portNum; }

  public String getGopherRoot() { return gopherRoot; }

  public String getHostName() { return hostName; }

  public FileCache getCache() { return cache; }

  public FileFilter getFileFilter() { return fileFilter; }

  public String getLinkFileName() { return linkFileName; }

  public boolean checkCacheTimes() { return checkCache; }

  public String getUserName() { return userName; }

  public LogManager getLogManager() { return logManager; }

  public String getCharsetName() { return charsetName; }

  public int getListenerBufferSize() { return listenerBufferSize; }

  public String getAccessLogFile() { return accessLogFile; }
  public String getAccessLogFormat() { return accessLogFormat; }
  public String getErrorLogFile() { return errorLogFile; }
  public String getErrorLogFormat() { return errorLogFormat; }
  public String getExceptionLogFile() { return exceptionLogFile; }
  public String getExceptionLogFormat() { return exceptionLogFormat; }
  public String getInfoLogFile() { return infoLogFile; }
  public String getInfoLogFormat() { return infoLogFormat; }
}
