//
// FileCache.java
// Based on soma2.webserver.cache.FileCache
// Part of the Aftershock Project, see README for details
// Copyright 1998-2003 Rob Linwood (rob@linwood.us)
//

package aftershock.cache;

import aftershock.conf.Configuration;
import aftershock.util.FileUtilities;

import java.io.FileNotFoundException;
import java.io.File;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.Hashtable;

public class FileCache {

  static final int DEFAULT_MAXIMUM_SIZE = 10000000;
  static final int DEFAULT_HASH_SIZE = 50;

  int maximumSize = DEFAULT_MAXIMUM_SIZE;
  Map cache = null;
  Configuration config = null;
  
  
  /**
   * Creates a new <code>FileCache</code> of the default maximum size.
   */
  public FileCache( Configuration config ) {
    this( config, DEFAULT_MAXIMUM_SIZE );
  }


  /**
   * Creates a new <code>FileCache</code> with the given maximum size
   *
   * @param maximumSize the maximum size, in bytes, you wish to allow the cache
   *                    to be, or -1 to represent an unlimited size
   */
  public FileCache( Configuration config, int maximumSize ) {
    this.config = config;
    setMaxSize( maximumSize );

    cache = new Hashtable( DEFAULT_HASH_SIZE );
  }

  
  /**
   * Sets the maximum size of the cache to the given number of bytes.
   *
   * @param maximumSize the maximum size, in bytes, you wish to allow the cache
   *                    to be, or -1 to represent an unlimited size
   */   
  public void setMaxSize( int maximumSize ) {
    this.maximumSize = maximumSize;
  }


  /**
   * Returns the maximum available size of the cache.  If it returns -1, then
   * the cache size is unlimited
   *
   * @return the cache size in bytes, or -1 if it is unlimited
   */
  public int getMaxSize() {
    return maximumSize;
  }


  /**
   * Add a file to the cache.  This method creates a 
   * {@link CacheEntry <code>CacheEntry</code>} object and adds it to the 
   * file cache.
   *
   * @param fileName the name of the file to add
   * @param mimeType the MIME type of the file
   * @return the {@link CacheEntry <code>CacheEntry</code>} created for the
   *         file.
   * @throws java.lang.Throwable
   */
  public CacheEntry addFile( String fileName ) 
    throws FileNotFoundException, SecurityException {
    CacheEntry entry = null;

    assert fileName != null : "FileCache.addFile() : fileName is null";

    File file = new File( fileName );

    if( file.isDirectory() ) {
      entry = CacheEntry.createDirectoryEntry( file, config );
    } else {
      entry = CacheEntry.createFileEntry( file, config );
    }

    cache.put( fileName, entry );
    return entry;
  }


  /**
   * Return the {@link CacheEntry <code>CacheEntry</code>} by the given name
   * which is stored in the cache. If no such file can be found, it returns
   * null.
   *
   * @param fileName the name of the file to return
   * @return a <code>CacheEntry</code> by the given name, or null if not found
   * @see #getFile(String fileName) 
   */
  public CacheEntry getEntry( String fileName ) {
    return (CacheEntry)cache.get( fileName );
  }
  

  /**
   * Return the {@link ByteBuffer <code>ByteBuffer</code>} by the given name
   * which is stored in the cache.  If no such file can be found, it returns
   * null.
   *
   * @param fileName the name of the file to return
   * @return a <code>ByteBuffer</code> by the given name, or null if not found
   * @see #getEntry(String fileName)
   */
  public ByteBuffer getFile( String fileName ) {
    CacheEntry entry = getEntry( fileName );

    if( entry != null ) {

      // Do we see if the file has been updated since cached?
      if( config.checkCacheTimes() ) {
	if( FileUtilities.getFileTime(entry.fileName) > entry.timeStamp ) {
	  // File has changed
	  cache.remove( fileName );
	  CacheEntry newEntry = null;
	  try {
	    newEntry = addFile( fileName );
	  } catch( Exception e ) {
	    config.getLogManager()
	      .logError("Error reloading file " + fileName + 
			" into cache. Using old cached copy instead. See " +
			"exception log for more" );
	    config.getLogManager().logException( e );
	    return entry.getFile();
	  }
	  return newEntry.getFile();
	} else {
	  return entry.getFile();
	}
      } else {
	return entry.getFile();
      }
    } else {
      return null;
    }
  }

}
