//
// CacheEntry.java
// Based on soma2.webserver.cache.CacheEntry
// Part of the Aftershock Project, see README for details
// Copyright 1998-2003 Rob Linwood (rob@linwood.us)
//

package aftershock.cache;

import aftershock.util.FileUtilities;
import aftershock.engine.DirectoryLister;
import aftershock.engine.FileTypeManager;
import aftershock.conf.Configuration;

import java.io.FileInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.GregorianCalendar;

public class CacheEntry {

  int hits = 0;
  long timeStamp;
  String fileName = null;
  ByteBuffer fileBytes = null;
  String fileType = null;
  boolean isDir = false;

  CacheEntry( String fileName  ) {
    this.fileName = fileName;
    hits = 0;
    timeStamp = new GregorianCalendar().getTimeInMillis();
  }


  static CacheEntry createFileEntry( File file, Configuration config ) 
    throws FileNotFoundException, SecurityException {

    CacheEntry entry = new CacheEntry( file.getName() );

    entry.fileType = FileTypeManager.getFileType( file.getName() );

    FileInputStream fis = new FileInputStream( file );
    FileChannel channel = fis.getChannel();
    
    try {
      entry.fileBytes = ByteBuffer.allocate( (int)channel.size() );
      channel.read( entry.fileBytes );
    } catch( IOException e ) {
      config.getLogManager().logException( e );
      System.err.println( "Exception in channel.read( fileBytes ) " );
      e.printStackTrace( System.err );
      System.exit( 1 );
    }

    
    return entry;
  }

  
  static CacheEntry createDirectoryEntry( File file, Configuration config )
    throws SecurityException {
    CacheEntry entry = new CacheEntry( file.getName() );

    entry.fileBytes = DirectoryLister.list( file, config );
    entry.isDir = true;
    return entry;
  }

  
  public void hit() {
    hits++;
  }

  public int getHits() {
    return hits;
  }

  public long getTime() {
    return timeStamp;
  }

  public ByteBuffer getFile() {
    return fileBytes;
  }

  public String getFileName() {
    return fileName;
  }

  public String getFileType() {
    return fileType;
  }

  public boolean isDirectory() {
    return isDir;
  }
}
