/*
  dirTree.h, written by Rhett "Jonzy" Jones 

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* Description:	Contains various utlity routines.  See "Routines" for more
 *		information.
 */

#include "stdinc.h"

char *StrToLower (char *str);
long GetLong (FILE * fp);
int GetInt (FILE * fp);
char *GetStr (FILE *fp, char *str, short limit);
int Str2Int (char *s);
char *Reverse (char *s, char *r);
short StrRcmp (char *s1, char *s2);
char *Mysprint (char *format, ...);
int DoSystemCall (char *command);
char *MyStrTok (char *str, char delimeter);
char *OnlyDigits (char *s);
char *RemoveCRLF (char *line);

extern FILE *rptPtr;		/* Defined in "jugtailConf.c". */

/*****************************************************************************
 * NumberOfLines returns the number of lines in the file "fileName".
 ****************************************************************************/
long
NumberOfLines (char *fileName)
     /* fileName: Name of the file to count lines. */
{
  FILE *fp;			/* Pointer the the file 'fileName'. */
  long numLines = 0;		/* The number of lines to return. */
  int c;

  if ((fp = fopen (fileName, "r")))
    {
      while ((c = fgetc (fp)) != EOF)
	if (c == '\n')
	  numLines++;
      fclose (fp);
    }
  else
    fprintf (rptPtr, "error: NumberOfLines could not open %s\n", fileName);
  return (numLines);

}				/* NumberOfLines */

/*****************************************************************************
 * StrToLower returns the string 'str' passed in, which gets converted to
 * lower case.  This routine supports the ISO Latin character set 1.
 * Thank you Maes@elec.ucl.ac.BE for the code modifications.
 ****************************************************************************/
char *
StrToLower (char *str)
     /* str: The string we are converted to lower case. */
{
  char *s;		/* Pointer to the position we are converting. */
  unsigned char c;      /* To support ISO latin set 1 characters. */

  for (s = str; *s; s++)
    if (isascii (*s))
      *s = tolower (*s);
    else
      switch (c = (unsigned char) *s)
	{
	case 208:		/* ETH */
	case 215:		/* MULT */
	  break;		/* Nothing to do with these. */
	default:
	  if (c > 191 && c < 222)	/* Uppercase */
	    *s = c + 32;
	}

  return (str);

}				/* StrToLower */

/*****************************************************************************
 * GetLong returns a long acquired from the stream 'fp'.  This routine
 * supports a '+' or '-' as input as the first character to set the sign.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
long
GetLong (FILE *fp)
     /* fp: The stream we are getting input from. */
{
  long i;			/* The value we return. */
  int c,			/* A character from the stream. */
    sign = 1;			/* Initial sign of the result. */

  while (isspace (c = fgetc (fp)));	/* Skip any whitespace. */
  if (c == '+' || c == '-')	/* Check the sign. */
    {
      sign = (c == '+') ? 1 : -1;
      c = fgetc (fp);
    }
  for (i = 0; isdigit (c); c = fgetc (fp))	/* Get the number. */
    i = i * 10 + c - '0';
#if(0)
  if (c != EOF)			/* Put the character back. */
    ungetc (c, fp);
#endif
  return (i * sign);

}				/* GetLong */

/*****************************************************************************
 * GetInt returns an int acquired from the stream 'fp'.  This routine
* supports a '+' or '-' as input as the first character to set the sign.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
int
GetInt (FILE *fp)
     /* fp: The stream we are getting input from. */
{
  long i;			/* The value we return. */
  int c,			/* A character from the stream. */
    sign = 1;			/* Initial sign of the result. */

  while (isspace (c = fgetc (fp)));	/* Skip any whitespace. */
  if (c == '+' || c == '-')	/* Check the sign. */
    {
      sign = (c == '+') ? 1 : -1;
      c = fgetc (fp);
    }
  for (i = 0; isdigit (c); c = fgetc (fp))	/* Get the number. */
    i = i * 10 + c - '0';
#if(0)
  if (c != EOF)			/* Put the character back. */
    ungetc (c, fp);
#endif
  return (i * sign);


}				/* GetInt */

/*****************************************************************************
 * GetStr returns a string acquired from the stream 'fp'.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
char *
GetStr (FILE *fp, char *str, short limit)
     /* fp: The stream we are getting input from.
	str: Where to place the characters.
	limit: Max chars to get. */
{
  int i;			/* A loop counter. */
  int c;			/* A character from the stream. */

  while (isspace (c = fgetc (fp)));	/* Skip any whitespace. */
  for (i = 0; i < limit && !isspace (c); c = fgetc (fp))
    str[i++] = (char) c;
  str[i] = '\0';
#if(0)
  if (c != EOF)			/* Put the character back. */
    ungetc (c, fp);
#endif
  return (str);

}				/* GetStr */

/*****************************************************************************
 * Str2Int returns the integer equivilent of str.
 ****************************************************************************/
int
Str2Int (char *s)
     /* s: The string we are converting to an integer. */
{
  int i;			/* The value returned by this routine. */

  for (i = 0; *s && isdigit (*s); s++)
    i = i * 10 + *s - '0';
  return (i);

}				/* Str2Int */

/*****************************************************************************
 * Reverse copies 's' into 'r' in reverse order and returns 'r'.
 ****************************************************************************/
char *
Reverse (char *s, char *r)
     /* s: The string in forward order.
	r: The result in reverse order. */
{
  int i;			/* A loop counter. */
  int l;			/* The length of 's'. */

  for (l = strlen (s), i = 0; i < l; i++)
    r[i] = s[l - i - 1];
  r[l] = '\0';
  return (r);

}				/* Reverse */

/*****************************************************************************
 * StrRcmp returns true if 's1' in reverse is contained at the begining of
 * 's2' in reverse.  Otherwise it returns false.
 * Example: say 's1' = ".some.site", and 's2' = "this.some.site", if you reverse
 * both of these strings you get
 * 's1' = "etis.emos."
 * 's2' = "etis.emos.siht"  which evalutaes to true.  If 's1' is not contained
 * in 's2' we return false.
 * This routine was written to support the use of domains as hosts contained
 * in 'hosts2search'.
 ****************************************************************************/
short
StrRcmp (char *s1, char *s2)
     /* s1: The string which must be in s2 (in reverse order).
	s2: The string which must contain s1. */
{
  char r1[255],			/* The string s1 in reverse. */
    r2[255];			/* The string s2 in reverse. */

  for (s1 = Reverse (s1, r1), s2 = Reverse (s2, r2); *s1 && *s2; s1++, s2++)
    if (*s1 != *s2)
      return (0);
  if (*s1 && !*s2)
    return (0);
  return (1);

}				/* StrRcmp */

 /*VARARGS*/
/*****************************************************************************
 * Mysprint takes a variable number of arguments similar to fprintf() or
 * or vsprintf() inparticular, and puts the formated information into a string
 * and returns the string. 
 ****************************************************************************/
char *
Mysprint (char *format, ...)
{
  static char str[1024];	/* The string we send to the system. */
  int result;			/* The result of calling vsprintf. */
  va_list ap;			/* The arguments beyond 'format'. */

  va_start (ap, format);
  result = vsprintf (str, format, ap);
  va_end (ap);

  if (result < 0)
    {
      fprintf (rptPtr, "error: Mysprint had vsprintf fail.\n");
      return ((char *) NULL);
    }
  
  return (str);
  
}				/* Mysprint */

 /*ARGSUSED*/
/*****************************************************************************
 * DoSystemCall does a system on the result of 'format'.
 ****************************************************************************/
int
DoSystemCall (char *command)
     /* command: The command to send to system(). */
{
  int result,			/* The result of calling vsprintf. */
    error = 0;			/* Did things turn out ok? */

  if (command)
    if ((result = system (command)))
      error =
	fprintf (rptPtr, "error: DoSystemCall [%s] failed with %d\n", command,
		 result);
  return (error);

}				/* DoSystemCall */

/*****************************************************************************
 * MyStrTok returns breaks 'str' into tokens as per the token seperator
 * 'delimeter'.  This routine basicly does the samething as strtok().
 ****************************************************************************/
char *
MyStrTok (char *str, char delimeter)
     /* str: The string to tokenize.
	delimeter: The delimeter character. */
{
  static char *theStr,		/* Keep a copy the string to tokenize. */
   *s,				/* A temporary pointer. */
   *token;			/* The token to return. */

  if (str)			/* A new string to tokenize. */
    theStr = str;

  token = theStr;		/* This is the token. */

  /* Break up theStr into a token. */
  if (*theStr != delimeter)
    for (s = theStr; *s && *s != delimeter; s++)
      ;

  /* Nill out the token and do some cleaning up. */
  *s++ = '\0';
  theStr = s;
  return (token);

}				/* MyStrTok */

/*****************************************************************************
 * OnlyDigits returns a null terminated string containing only the digit
 * characters contained in 's'.  What we do is convert the first non-digit
 * character in 's' to the null character and return the string.
 ****************************************************************************/
char *
OnlyDigits (char *s)
{
  char *digits;			/* The string with digits only. */
  
  for (digits = s; *s; s++)
    if (!isdigit (*s))
      {
	*s = '\0';
	return (digits);
      }

  return (digits);

}				/* OnlyDigits */

/*****************************************************************************
 * RemoveCRLF removes the carriage return or line feed by changing it to
 * the null character.
 ****************************************************************************/
char *
RemoveCRLF (char *line)
{
  char *s;		/* Pointer into 'line'. */

  for (s = line; *s && *s != '\r' && *s != '\n'; s++)
    ;				/* Go to the CR or LF position. */
  *s = '\0';
  return (s);

}				/* RemoveCRLF */

