package PGopherLib::Misc;

use strict;
use warnings;
no warnings 'exec';
use Socket;
use Fcntl qw/F_GETFL F_SETFL O_NONBLOCK/;
use POSIX qw/:errno_h/;
use PGopherLib::Globals qw/$PAGER $BOOKMARKS $PROMPT @history @menu
	$rows $cols $sigint $sigpipe $sigwinch $utf8 $dumb/;
use PGopherLib::Init qw/quit/;
use PGopherLib::TTY qw/newtty oldtty getwinsize/;
use PGopherLib::VT100 qw/cls clreol curpos appkeys/;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(msg prompt errmsg mysystem myconnect isutf8 escape
	unescape splithpts help getbookmark menu);

my @help = split(/\n/, <<'EOF');
        (# = menu item number)

        #       View/Download menu item
        d#      Download menu item
        p#      Print menu item
        ?[#]    Show information about menu or menu item
        .       Redisplay current menu
        ,       Go back one menu
        b       Bookmarks
        B[#]    Add menu or menu item to bookmarks
        h       Command help (this text)
        !       Shell
        q       Quit

        Any other character sequence is taken as a server to connect to       
        in the format `host[:port][/type[selector]]'.

        Press Control-C to abort a transfer in progress.
EOF

my %icons = (
    '0' => '[TXT]',
    '1' => '[DIR]',
    '2' => '[CSO]',
    '3' => '[ERR]',
    '4' => '[HQX]',
    '5' => '[ARC]',
    '6' => '[UUE]',
    '7' => '[NDX]',
    '8' => '[TEL]',
    '9' => '[BIN]',
    'T' => '[IBM]',
    'g' => '[GIF]',
    'I' => '[IMG]',
    'h' => '[HTM]',
    'P' => '[PDF]',
    's' => '[SND]',
    ';' => '[MOV]',
    'i' => '     '
);

my %escapes;
foreach (0..255) {
    $escapes{chr($_)} = sprintf("%%%02X", $_);
}

my ($mhdr, $mdata, $mmode);


sub chksigwinch {
    $sigwinch and do {getwinsize; $sigwinch = 0};
}

sub msg {
    $dumb and return print "\n" . shift;
    curpos($rows, 1); clreol; print shift;
}

sub prompt {
    my $prompt = shift;
    chksigwinch;
    msg $prompt;
    !$dumb and $/ = "\015";
    $_ = "";
    while (1) {
	if (my $nr = sysread(STDIN, my $c, 1)) {
	    $_ .= $c;
	    $c eq $/ and last;
	} elsif (defined($nr)) {
	    quit;
	} elsif ($sigwinch) {
	    $sigwinch = 0;
	    getwinsize and $prompt =~ /> $/ and do
		    {&show; chksigwinch; msg $prompt};
	}
    }
    chomp;
    !$dumb and $/ = "\012";
    !$dumb and msg $prompt;
    s/^\s*(.*?)\s*$/$1/;
    return $_;
}

sub errmsg {
    prompt "Error: " . shift() . " ";
}

sub mysystem {
    oldtty;
    my $r = system(shift);
    newtty;
    return $r;
}

sub myconnect {
    my ($host, $port) = @_;
    my $iaddr = inet_aton($host) or do
	    {errmsg "no host `$host'"; return undef};
    my $paddr = sockaddr_in($port, $iaddr);
    socket(SOCK, PF_INET, SOCK_STREAM, 6) or do
	    {errmsg "socket: $!"; return undef};
    CONNECT: {
	my $flags = fcntl(SOCK, F_GETFL, 0) or do
		{errmsg "fcntl: $!"; last CONNECT};
	fcntl(SOCK, F_SETFL, $flags | O_NONBLOCK) or do
		{errmsg "fcntl: $!"; last CONNECT};
	$sigint = 0;
	my $start = time;
	while (! connect(SOCK, $paddr)) {
	    if ($! != EINPROGRESS and $! != EALREADY) {
		$! == EISCONN and last;
		# FreeBSD returns EINVAL instead of ECONNREFUSED
		$! == EINVAL and $! = ECONNREFUSED;
		errmsg "connect: $!";
		last CONNECT;
	    }
	    $sigint and last CONNECT;
	    if (time - $start >= 10) {
		errmsg "connect: timeout";
		last CONNECT;
	    }
	    select(undef, undef, undef, 0.1);
	}
	fcntl(SOCK, F_SETFL, $flags) or do
		{errmsg "fcntl: $!"; last CONNECT};
	return \*SOCK;
    }
    close(SOCK);
    return undef;
}

sub isutf8 {
    return shift =~ /^(?:[\x00-\x7F] |		# ASCII
	    [\xC2-\xDF][\x80-\xBF] |		# non-overlong 2-byte
	    \xE0[\xA0-\xBF][\x80-\xBF] |	# excluding overlongs
	    [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2} |	# straight 3-byte
	    \xED[\x80-\x9F][\x80-\xBF] |	# excluding surrogates
	    \xF0[\x90-\xBF][\x80-\xBF]{2} |	# planes 1-3
	    [\xF1-\xF3][\x80-\xBF]{3} |		# planes 4-15
	    \xF4[\x80-\x8F][\x80-\xBF]{2}	# plane 16
    )*$/x;
}

sub escape {
    my $text = shift;
    $text =~ s/([^A-Za-z0-9\-_.!~*'()\/])/$escapes{$1}/g;
    return $text;
}

sub unescape {
    my $text = shift;
    $text =~ s/%([0-9A-Fa-f]{2})/chr(hex($1))/eg;
    return $text; 
}

sub splithpts {
    my ($host, $port, $type, $sel) = (shift =~
	    m#^([^:/]+)(:(\d+))?(/(.)(.*))?$#)[0,2,4,5] or return ();
    defined $type or $type = "1";
    defined($port) or $port = ($type eq "2") ? 105 :
	    ($type =~ /[8T]/) ? 23 : 70;
    defined($sel) or $sel = "";
    return ($host, $port, $type, $sel);
}

sub chopl {
    my $s = shift;
    length($s) > $cols and $s = substr($s, 0, $cols - 2) . "..";
    $s;
}

sub center {
    my $text = shift;
    return " " x ($cols/2 - length($text)/2) . $text;
}

sub show {
    cls;
    my @text = ("\n", center(chopl($mhdr)) . "\n", "\n");
    my $n = 1;
    foreach (@$mdata) {
	my $s;
	if ($mmode) {
	    my ($type, $desc) = @$_;
	    $s = sprintf("%4s %s %s", ($type eq "i" ? "" : $n),
		    (exists $icons{$type} ? $icons{$type} : "[???]"),
		    $desc);
	} else {
	    $s = $_;
	}
	push @text, chopl($s) . "\n";
	$n++;
    }
    if (scalar(@text) > $rows - 2) {
	oldtty; appkeys 1;
	open(PIPE, "|$PAGER") or do {
	    newtty; appkeys 0;
	    errmsg "open: $!";
	    return 0;
	};
	$utf8 and binmode PIPE, ':utf8';
	$sigint = $sigpipe = 0;
	foreach (@text) {
	    last if $sigint || $sigpipe;
	    print PIPE;
	}
	close(PIPE);
	newtty; appkeys 0;
	!$dumb and print "\n";
	return 2;
    } else {
	print foreach (@text);
	return 1;
    }
}

sub help {
    ($mhdr, $mdata, $mmode) = ("COMMAND HELP", \@help, 0);
    my $r = show or return;
    $r == 1 || $PAGER !~ /^less\s*/ and
	    prompt "Press RETURN to continue ";
}

sub getbookmark {
    -f $BOOKMARKS or return;
    open(FILE, $BOOKMARKS) or do {errmsg "open: $!"; return ()};
    $utf8 and binmode FILE, ':utf8';
    my @bookmarks = ();
    while (<FILE>) {
	chomp;
	/^\s*$/ and next;
	my ($desc, $hpts) = split(/\t/, $_, 2);
	my ($host, $port, $type, $sel) = splithpts($hpts) or do {
	    close(FILE);
	    errmsg "invalid string `$hpts' in bookmark file";
	    return ();
	};
	push @bookmarks, [$type, $desc, $sel, $host, $port];
    }
    close(FILE);
    ($mhdr, $mdata, $mmode) = ("BOOKMARKS", \@bookmarks, 1);
    show or return ();
    ($_ = prompt("bookmarks> ")) eq "" and return ();
    !/^\d+$/ and do {errmsg "invalid entry"; return ()};
    $_ < 1 || $_ > scalar(@bookmarks) and do
	    {errmsg "invalid entry"; return ()};
    return @{$bookmarks[$_ - 1]};
}

sub menu {
    chksigwinch;
    scalar(@history) or return cls;
    my ($type, $desc) = @{$history[$#history]};
    $type eq "7" and $desc = "Search Results";
    ($mhdr, $mdata, $mmode) = ("MENU" . ($desc ? " [$desc]" : ""),
	    \@menu, 1);
    show;
}

1;
