/** @file wlan_wmm.c
  * @brief This file contains functions for WMM.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	10/04/05: Add Doxygen format comments
	
********************************************************/
#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

#define IPTOS_OFFSET 5

static u8  wmm_tos2ac[8] = {
		AC_PRIO_BE,
		AC_PRIO_BK,
		AC_PRIO_BK,
		AC_PRIO_BE,
		AC_PRIO_VI,
		AC_PRIO_VI,
		AC_PRIO_VO,
		AC_PRIO_VO
};

static u8  wmm_ac_downgrade[MAX_AC_QUEUES] = {
		AC_PRIO_BK,
		AC_PRIO_BE,
		AC_PRIO_VI,
		AC_PRIO_VO
};

/* This mapping table will be useful if bit-flip is needed */
static u8 wmm_tos2priority[8] = {
/*	Priority   DSCP   DSCP   DSCP	WMM
		    P2     P1     P0 	AC    */
	0x00,	/*  0	   0	  0	AC_BE */
	0x01,	/*  0	   0	  1	AC_BK */
	0x02,	/*  0	   1	  0	AC_BK */
	0x03,	/*  0	   1	  1	AC_BE */
	0x04,	/*  1	   0	  0	AC_VI */
	0x05,	/*  1	   0	  1	AC_VI */
	0x06,	/*  1	   1	  0	AC_VO */
	0x07	/*  1	   1	  1	AC_VO */
};

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/
/** 
 *  @brief pop up the highest skb from wmm queue
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
static void wmm_pop_highest_prio_skb(wlan_private *priv)
{
	int i;
	wlan_adapter *Adapter = priv->adapter;
	u8 ac;

	for (i = 0; i < MAX_AC_QUEUES; i++) {
		ac = Adapter->CurBssParams.wmm_queue_prio[i];
		if(!list_empty((struct list_head*) 
					&Adapter->wmm.TxSkbQ[ac])) {
			Adapter->CurrentTxSkb = Adapter->wmm.TxSkbQ[ac].next;
			list_del((struct list_head *) Adapter->
						wmm.TxSkbQ[ac].next);
			break;
		}
	}
}

/** 
 *  @brief send cmd to FW to announce package available
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
static void wmm_send_prio_pkt_avail(wlan_private *priv)
{
#if 0	/* WMM_PRIO_PKT_AVAIL command not supported for now */
	int i;

	for (i = 0; i < MAX_AC_QUEUES; i++) {
		ac = Adapter->CurBssParams.wmm_queue_prio[i];
		if(!list_empty((struct list_head*)
					&priv->adapter->wmm.TxSkbQ[ac]))
			break;

	if(i >= MAX_AC_QUEUES)		/* No High prio packets available */
		return;

	priv->adapter->priopktavail.PacketAC = ac;

	PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_WMM_PRIO_PKT_AVAIL,
		0, 0, 0, NULL);
#endif
}

/** 
 *  @brief implement WMM enable command
 *  @param priv       pointer to wlan_private
 *  @param wrq        pointer to user data
 *  @return 	      WLAN_STATUS_SUCCESS if success; otherwise <0
*/
int wlan_wmm_enable_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	u32		flags;
	int		ret;
	int data,data1;

	ENTER();
	if(wrq->u.data.length == 0)
		return -EINVAL;
	copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
	switch(data) {
	case CMD_DISABLED: /* disable */
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected)
			return -EPERM; 

		spin_lock_irqsave(&Adapter->CurrentTxLock, flags);
		Adapter->wmm.required = 0;

		if (!Adapter->wmm.enabled) {
			spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);
			return WLAN_STATUS_SUCCESS;
		}
		else
			Adapter->wmm.enabled = 0;

		if (Adapter->CurrentTxSkb) {
			kfree_skb(Adapter->CurrentTxSkb);
			OS_INT_DISABLE;
			Adapter->CurrentTxSkb = NULL;
			OS_INT_RESTORE;
			priv->stats.tx_dropped++;
		}

		/* Release all skb's in all the queues */
		wmm_cleanup_queues(priv);
		
		spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);

		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WMM_ENABLE;

		SetMacPacketFilter(priv);
		break;

	case CMD_ENABLED: /* enable */
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected)
			return -EPERM; 

		spin_lock_irqsave(&Adapter->CurrentTxLock, flags);
		
		Adapter->wmm.required = 1;
		
		spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);

		break;
	case CMD_GET:
		wrq->u.data.length = 1;
		data1 = Adapter->wmm.required;
		if (copy_to_user(wrq->u.data.pointer, &data1,sizeof(int))) {
			PRINTM(INFO, "Copy to user failed\n");
			ret = -EFAULT;
		}

		break;
	default:
		PRINTM(INFO, "Invalid option\n");
		return -EINVAL;
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief implement read/set/remove TSPEC
 *  @param priv       pointer to wlan_private
 *  @param wrq        pointer to user data
 *  @return 	      WLAN_STATUS_SUCCESS if success; otherwise <0
*/
int wlan_do_wmm_tspec_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 				ret = WLAN_STATUS_SUCCESS;
	u16				Action;
	HostCmd_DS_802_11_WMM_TSPEC	*tSpec = &priv->adapter->tspec;

	Action = req->ifr_data[1] | (req->ifr_data[2] << 8);

	switch(Action) {
	case HostCmd_ACT_GEN_GET:
		ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_WMM_GET_TSPEC, 0,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL);
		HEXDUMP("Tspec Conf GET", (u8*)tSpec, sizeof(HostCmd_DS_802_11_WMM_TSPEC));
		if (copy_to_user(req->ifr_data + SKIP_TYPE_SIZE, tSpec,
				sizeof(HostCmd_DS_802_11_WMM_TSPEC))) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}
		break;
	case HostCmd_ACT_GEN_SET:
		memset(tSpec, 0, sizeof(HostCmd_DS_COMMAND));
		if (copy_from_user(tSpec, req->ifr_data + SKIP_TYPE_SIZE,
				sizeof(HostCmd_DS_802_11_WMM_TSPEC))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
		HEXDUMP("Tspec Conf SET", (u8*)tSpec, sizeof(HostCmd_DS_802_11_WMM_TSPEC));
		ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_WMM_ADD_TSPEC, 0,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL);
		break;
	case HostCmd_ACT_GEN_REMOVE:
		ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_WMM_REMOVE_TSPEC, 0,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL);
		break;
	default:
		PRINTM(INFO, "Invalid Command\n");
		return -EINVAL;
	}

	return ret;
}

/** 
 *  @brief Set WMM IE
 *  @param priv       pointer to wlan_private
 *  @param wrq        pointer to user data containging WMM IE
 *  @return 	      WLAN_STATUS_SUCCESS if success; otherwise <0
*/
int wlan_do_wmm_para_ie_ioctl(wlan_private *priv, struct ifreq *req)
{
	u16	Action;
	u8	*para_ie = priv->adapter->wmm.Para_IE;

	Action = req->ifr_data[1] | (req->ifr_data[2] << 8);

	switch (Action) {
	case HostCmd_ACT_GEN_GET:
		if (copy_to_user(req->ifr_data + SKIP_TYPE_SIZE, para_ie, 
					WMM_PARA_IE_LENGTH)) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}

		HEXDUMP("Para IE Conf GET", (u8*)para_ie, WMM_PARA_IE_LENGTH);

		break;
	case HostCmd_ACT_GEN_SET:
		if (priv->adapter->MediaConnectStatus ==
					WlanMediaStateConnected)
			return -EPERM;
	
		HEXDUMP("Para IE Conf SET", (u8*)para_ie, WMM_PARA_IE_LENGTH);

		if (copy_from_user(para_ie, req->ifr_data + SKIP_TYPE_SIZE, 
					WMM_PARA_IE_LENGTH)) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
		break;
	default:
		PRINTM(INFO, "Invalid Option\n");
		return -EINVAL;
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Read/Set WMM ACK policy
 *  @param priv       pointer to wlan_private
 *  @param wrq        pointer to user data
 *  @return 	      WLAN_STATUS_SUCCESS if success; otherwise <0
*/
int wlan_do_wmm_ack_policy_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 	ret = WLAN_STATUS_SUCCESS, i, index;
	HostCmd_DS_802_11_WMM_ACK_POLICY	
				*ackPolicy = &priv->adapter->ackpolicy;

	memset(ackPolicy, 0, sizeof(HostCmd_DS_COMMAND));

	if (copy_from_user(ackPolicy, req->ifr_data + SKIP_TYPE,
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}
  
	HEXDUMP("Ack Policy Conf", (u8*)ackPolicy, 
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY));

	switch(ackPolicy->Action) {
	case HostCmd_ACT_GET:
		for(i=0; i < WMM_ACK_POLICY_PRIO; ++i) {
			ackPolicy->AC = i;

			if((ret = PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_WMM_ACK_POLICY, 0,
				HostCmd_OPTION_WAITFORRSP,
				0, NULL))) {

				LEAVE();
				PRINTM(MSG, "PrepareAndSend Failed\n");
				return ret;
			}

			index = SKIP_TYPE_ACTION + (i * 2);
			if (copy_to_user(req->ifr_data + index, 
					(u8*)(&ackPolicy->AC), 16)) {
				PRINTM(MSG, "Copy from user failed\n");
				return -EFAULT;
			}

			HEXDUMP("Ack Policy Conf", (u8*)ackPolicy + SKIP_ACTION, 
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY));
		}


		break;
	case HostCmd_ACT_SET:
		ackPolicy->AC = req->ifr_data[SKIP_TYPE_ACTION];
		ackPolicy->AckPolicy = req->ifr_data[SKIP_TYPE_ACTION + 1];

		if((ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_WMM_ACK_POLICY, 0,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL))) {

			LEAVE();
			return ret;
		}

		if (copy_to_user(req->ifr_data + SKIP_TYPE, ackPolicy,
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		HEXDUMP("Ack Policy Conf", (u8*)ackPolicy, 
			sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY));

		break;
	default:
		PRINTM(MSG, "Invalid Action\n");
		return -EINVAL;
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Implement cmd HostCmd_DS_802_11_WMM_TSPEC
 *  @param priv       pointer to wlan_private
 *  @param cmd        pointer to CMD buffer 
 *  @param cmdno      cmd ID
 *  @param InfoBuf    pointer to cmd data
 *  @return 	      WLAN_STATUS_SUCCESS
*/
int wlan_cmd_802_11_wmm_tspec(wlan_private *priv,
			HostCmd_DS_COMMAND *cmd, u16 cmdno, void *InfoBuf)
{

	cmd->Command = wlan_cpu_to_le16(cmdno);	
	cmd->Size = wlan_cpu_to_le16(
			sizeof(HostCmd_DS_802_11_WMM_TSPEC) + S_DS_GEN);

	memcpy(&cmd->params.tspec, &priv->adapter->tspec, 
				sizeof(HostCmd_DS_802_11_WMM_TSPEC));

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Implement cmd HostCmd_DS_802_11_WMM_ACK_POLICY
 *  @param priv       pointer to wlan_private
 *  @param cmd        pointer to CMD buffer 
 *  @param cmdno      cmd ID
 *  @param InfoBuf    pointer to cmd data
 *  @return 	      WLAN_STATUS_SUCCESS
*/
int wlan_cmd_802_11_wmm_ack_policy(wlan_private *priv,
			HostCmd_DS_COMMAND *cmd, u16 action, void *InfoBuf)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_WMM_ACK_POLICY);
	cmd->Size = wlan_cpu_to_le16(
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY) + 
							S_DS_GEN);

	memcpy(&cmd->params.ackpolicy, &priv->adapter->ackpolicy, 
				sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY));

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Implement cmd HostCmd_CMD_802_11_WMM_GET_STATUS
 *  @param priv       pointer to wlan_private
 *  @param cmd        pointer to CMD buffer 
 *  @param cmdno      cmd ID
 *  @param InfoBuf    pointer to cmd data
 *  @return 	      WLAN_STATUS_SUCCESS
*/
int wlan_cmd_802_11_wmm_get_status(wlan_private *priv,
			HostCmd_DS_COMMAND *cmd, u16 action, void *InfoBuf)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_WMM_GET_STATUS);	
	cmd->Size  = wlan_cpu_to_le16(
			sizeof(HostCmd_DS_802_11_WMM_GET_STATUS)+S_DS_GEN);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Implement cmd HostCmd_CMD_802_11_WMM_PRIO_PKT_AVAIL
 *  @param priv       pointer to wlan_private
 *  @param cmd        pointer to CMD buffer 
 *  @param cmdno      cmd ID
 *  @param InfoBuf    pointer to cmd data
 *  @return 	      WLAN_STATUS_SUCCESS
*/
int wlan_cmd_802_11_wmm_prio_pkt_avail(wlan_private *priv,
			HostCmd_DS_COMMAND *cmd, u16 action, void *InfoBuf)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_WMM_PRIO_PKT_AVAIL);
	cmd->Size = wlan_cpu_to_le16(
			sizeof(HostCmd_DS_802_11_WMM_PRIO_PKT_AVAIL)+S_DS_GEN);

	cmd->params.priopktavail.PacketAC = priv->adapter->
						priopktavail.PacketAC;

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief get WMM Status
 *  @param priv       pointer to wlan_private
 *  @return 	      WLAN_STATUS_SUCCESS; WLAN_STATUS_FAILURE
*/
int sendWMMStatusChangeCmd(wlan_private *priv)
{
	return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_WMM_GET_STATUS,
			0, 0, 0, NULL);
}

/** 
 *  @brief check if wmm TX queue is empty
 *  @param priv       pointer to wlan_private
 *  @return 	      0 for not empty; 1 for empty
*/
int wmm_lists_empty(wlan_private *priv)
{
	int i;

	for (i = 0; i < MAX_AC_QUEUES; i++)
		if (!list_empty((struct list_head*)
					&priv->adapter->wmm.TxSkbQ[i]))
			return 0;

	return 1;
}

/** 
 *  @brief Cleanup wmm TX queue
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
void wmm_cleanup_queues(wlan_private *priv)
{
	int 		i;
	struct sk_buff	*delNode, *Q;

	for (i = 0; i < MAX_AC_QUEUES; i++) {
		Q = &priv->adapter->wmm.TxSkbQ[i];
		
		while (!list_empty((struct list_head *) Q)) {
			delNode = Q->next;
			list_del((struct list_head *)delNode);
			kfree_skb(delNode);
		}
	}
}


/** 
 *  @brief Add skb to WMM queue
 *  @param priv       pointer to wlan_private
 *  @param skb        pointer to sk_buff
 *  @return 	      N/A
*/
void wmm_map_and_add_skb(wlan_private *priv, struct sk_buff *skb)
{
	wlan_adapter 	*Adapter = priv->adapter;
	u8		tos, ac0, ac;
	struct ethhdr	*eth = (struct ethhdr *)skb->data;

	switch (eth->h_proto)
	{
	case __constant_htons(ETH_P_IP):
		PRINTM(INFO, "packet type ETH_P_IP: %04x, tos=%#x prio=%#x\n",eth->h_proto,skb->nh.iph->tos,skb->priority);
		tos = IPTOS_PREC(skb->nh.iph->tos) >> IPTOS_OFFSET;
		break;
	case __constant_htons(ETH_P_ARP):
		PRINTM(INFO, "ARP packet %04x\n",eth->h_proto);
	default:
		tos = 0;
		break;
	}

	ac0 = wmm_tos2ac[tos];
	ac = wmm_ac_downgrade[ac0];
	
	skb->priority = wmm_tos2priority[tos];
	PRINTM(INFO, "wmm_map: tos=%#x, ac0=%#x ac=%#x, prio=%#x\n",tos,ac0,ac,skb->priority);

	/* Access control of the current packet not the Lowest */
	if(ac > AC_PRIO_BE)
		Adapter->wmm.fw_notify = 1;

	list_add_tail((struct list_head *) skb,
			(struct list_head *) &Adapter->wmm.TxSkbQ[ac]);
}


/** 
 *  @brief Initialize WMM priority queue
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
void wmm_setup_queue_priorities(wlan_private *priv)
{
	wlan_adapter *Adapter = priv->adapter;
#ifdef WMM_AIFS
	pWMM_PARA_IE pIe = (pWMM_PARA_IE)Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex].Wmm_IE;

	HEXDUMP("Wmm_IE in association response", (u8 *)pIe,
		Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex].Wmm_ie_len);
	if (pIe->OuiSubtype == WMM_OUISUBTYPE_PARA) {
		u16 cwmax, cwmin, avg_back_off;
		u16 tmp[4];
		int i, j, n = 0;
		PRINTM(INFO, "WMM Parameter IE: version=%d, QoSInfo Parameter Set Count=%d, Reserved=%#x\n",
	    		pIe->Version,
	    		pIe->QoSInfo.ParaSetCount,
	    		pIe->Reserved);
		PRINTM(INFO, "BE AC_Parameters: ACI=%d, ACM=%d, AIFSN=%d\n",
	    		pIe->AC_Paras_BE.ACI_AIFSN.ACI,
	    		pIe->AC_Paras_BE.ACI_AIFSN.ACM,
	    		pIe->AC_Paras_BE.ACI_AIFSN.AIFSN);
		PRINTM(INFO, "                  ECWmin=%d, ECWmax=%d, Txop_Limit=%d\n",
	    		pIe->AC_Paras_BE.ECW.ECW_Min,
	    		pIe->AC_Paras_BE.ECW.ECW_Max,
               		wlan_le16_to_cpu(pIe->AC_Paras_BE.Txop_Limit)<<WMM_TXOP_LIMIT_UNITS_SHIFT);
		cwmax = (1 << pIe->AC_Paras_BE.ECW.ECW_Max) - 1;
		cwmin = (1 << pIe->AC_Paras_BE.ECW.ECW_Min) - 1;
		avg_back_off = (cwmin >> 1) + pIe->AC_Paras_BE.ACI_AIFSN.AIFSN;
		PRINTM(INFO, "CWmax=%d CWmin=%d Avg Back-off=%d\n",cwmax,cwmin,avg_back_off);
		Adapter->CurBssParams.wmm_queue_prio[n] = AC_PRIO_BE;
		tmp[n++] = avg_back_off;
		PRINTM(INFO, "BK AC_Parameters: ACI=%d, ACM=%d, AIFSN=%d\n",
	    		pIe->AC_Paras_BK.ACI_AIFSN.ACI,
	    		pIe->AC_Paras_BK.ACI_AIFSN.ACM,
	    		pIe->AC_Paras_BK.ACI_AIFSN.AIFSN);
		PRINTM(INFO, "                  ECWmin=%d, ECWmax=%d, Txop_Limit=%d\n",
	    		pIe->AC_Paras_BK.ECW.ECW_Min,
	    		pIe->AC_Paras_BK.ECW.ECW_Max,
               		wlan_le16_to_cpu(pIe->AC_Paras_BK.Txop_Limit)<<WMM_TXOP_LIMIT_UNITS_SHIFT);
		cwmax = (1 << pIe->AC_Paras_BK.ECW.ECW_Max) - 1;
		cwmin = (1 << pIe->AC_Paras_BK.ECW.ECW_Min) - 1;
		avg_back_off = (cwmin >> 1) + pIe->AC_Paras_BK.ACI_AIFSN.AIFSN;
		PRINTM(INFO, "CWmax=%d CWmin=%d Avg Back-off=%d\n",cwmax,cwmin,avg_back_off);
		Adapter->CurBssParams.wmm_queue_prio[n] = AC_PRIO_BK;
		tmp[n++] = avg_back_off;
		PRINTM(INFO, "VI AC_Parameters: ACI=%d, ACM=%d, AIFSN=%d\n",
	    		pIe->AC_Paras_VI.ACI_AIFSN.ACI,
	    		pIe->AC_Paras_VI.ACI_AIFSN.ACM,
	    		pIe->AC_Paras_VI.ACI_AIFSN.AIFSN);
		PRINTM(INFO, "                  ECWmin=%d, ECWmax=%d, Txop_Limit=%d\n",
	    		pIe->AC_Paras_VI.ECW.ECW_Min,
	    		pIe->AC_Paras_VI.ECW.ECW_Max,
               		wlan_le16_to_cpu(pIe->AC_Paras_VI.Txop_Limit)<<WMM_TXOP_LIMIT_UNITS_SHIFT);
		cwmax = (1 << pIe->AC_Paras_VI.ECW.ECW_Max) - 1;
		cwmin = (1 << pIe->AC_Paras_VI.ECW.ECW_Min) - 1;
		avg_back_off = (cwmin >> 1) + pIe->AC_Paras_VI.ACI_AIFSN.AIFSN;
		PRINTM(INFO, "CWmax=%d CWmin=%d Avg Back-off=%d\n",cwmax,cwmin,avg_back_off);
		Adapter->CurBssParams.wmm_queue_prio[n] = AC_PRIO_VI;
		tmp[n++] = avg_back_off;
		PRINTM(INFO, "VO AC_Parameters: ACI=%d, ACM=%d, AIFSN=%d\n",
	    		pIe->AC_Paras_VO.ACI_AIFSN.ACI,
	    		pIe->AC_Paras_VO.ACI_AIFSN.ACM,
	    		pIe->AC_Paras_VO.ACI_AIFSN.AIFSN);
		PRINTM(INFO, "                  ECWmin=%d, ECWmax=%d, Txop_Limit=%d\n",
	    		pIe->AC_Paras_VO.ECW.ECW_Min,
	    		pIe->AC_Paras_VO.ECW.ECW_Max,
               		wlan_le16_to_cpu(pIe->AC_Paras_VO.Txop_Limit)<<WMM_TXOP_LIMIT_UNITS_SHIFT);
		cwmax = (1 << pIe->AC_Paras_VO.ECW.ECW_Max) - 1;
		cwmin = (1 << pIe->AC_Paras_VO.ECW.ECW_Min) - 1;
		avg_back_off = (cwmin >> 1) + pIe->AC_Paras_VO.ACI_AIFSN.AIFSN;
		PRINTM(INFO, "CWmax=%d CWmin=%d Avg Back-off=%d\n",cwmax,cwmin,avg_back_off);
		Adapter->CurBssParams.wmm_queue_prio[n] = AC_PRIO_VO;
		tmp[n++] = avg_back_off;
		HEXDUMP("avg_back_off  ", (u8 *)tmp, sizeof(tmp));
		HEXDUMP("wmm_queue_prio", Adapter->CurBssParams.wmm_queue_prio,
			sizeof(Adapter->CurBssParams.wmm_queue_prio));
		/* bubble sort */
		for (i=0; i < n; i++) {
		    for (j=1; j < n - i; j++) {
			if (tmp[j-1] > tmp[j]) {
				SWAP_U16(tmp[j-1],tmp[j]);
				SWAP_U8(Adapter->CurBssParams.wmm_queue_prio[j-1],
					Adapter->CurBssParams.wmm_queue_prio[j]);
			}
			else if (tmp[j-1] == tmp[j]) {
				if (Adapter->CurBssParams.wmm_queue_prio[j-1] <
					Adapter->CurBssParams.wmm_queue_prio[j]) {
					SWAP_U8(Adapter->CurBssParams.wmm_queue_prio[j-1],
						Adapter->CurBssParams.wmm_queue_prio[j]);
				}
			}
		    }
		}
		HEXDUMP("avg_back_off  ", (u8 *)tmp, sizeof(tmp));
		HEXDUMP("wmm_queue_prio", Adapter->CurBssParams.wmm_queue_prio,
			sizeof(Adapter->CurBssParams.wmm_queue_prio));
	}
	else if (((pWMM_PARA_IE)pIe)->OuiSubtype == WMM_OUISUBTYPE_IE)
#endif /* WMM_AIFS */
	{
		/* default queue priorities: VO->VI->BE->BK */
		Adapter->CurBssParams.wmm_queue_prio[0] = AC_PRIO_VO;
		Adapter->CurBssParams.wmm_queue_prio[1] = AC_PRIO_VI;
		Adapter->CurBssParams.wmm_queue_prio[2] = AC_PRIO_BE;
		Adapter->CurBssParams.wmm_queue_prio[3] = AC_PRIO_BK;
	}
}

/** 
 *  @brief Downgrade WMM priority queue
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
void wmm_setup_ac_downgrade(wlan_private *priv)
{
	wlan_adapter *Adapter = priv->adapter;
	u8 ac0, ac;
	int i, j;

	PRINTM(INFO, "wmm.acstatus = %#x\n", Adapter->wmm.acstatus);

	for (i = 0; i < MAX_AC_QUEUES; i++) {	/* default settings without ACM */
		wmm_ac_downgrade[i] = (u8)i;
	}
	HEXDUMP("wmm_ac_downgrade default", wmm_ac_downgrade, sizeof(wmm_ac_downgrade));

	for (i = 0; i < MAX_AC_QUEUES - 1; i++) {	/* skip the last one */
		ac0 = Adapter->CurBssParams.wmm_queue_prio[i];	/* original AC */
		ac = Adapter->CurBssParams.wmm_queue_prio[i+1];	/* downgraded AC */
		if (Adapter->wmm.acstatus & (1<<ac0)) {
			for (j = 0; j < MAX_AC_QUEUES; j++) {
				if (ac0 == wmm_ac_downgrade[j]) {
					PRINTM(INFO, "wmm_setup_ac_downgrade: i=%d j=%d ac0=%d ac=%d\n",i,j,ac0,ac);
					wmm_ac_downgrade[j] = ac;
				}
			}
			PRINTM(INFO, "wmm_setup_ac_downgrade: AC %#x has been downgraded to %#x\n",ac0,ac);
		}
	}

	HEXDUMP("wmm_ac_downgrade", wmm_ac_downgrade, sizeof(wmm_ac_downgrade));

	ac0 = Adapter->CurBssParams.wmm_queue_prio[MAX_AC_QUEUES - 1];
	if (Adapter->wmm.acstatus & (1<<ac0)) {
		PRINTM(MSG, "wmm_setup_ac_downgrade: ignored ACM for the AC %#x as it has the lowest priority\n",ac0);
	}
}

/** 
 *  @brief handle TX data
 *  @param priv       pointer to wlan_private
 *  @return 	      N/A
*/
void wmm_process_tx(wlan_private * priv)
{
	wlan_adapter   *Adapter = priv->adapter;
	u32		flags;

	OS_INTERRUPT_SAVE_AREA;	/* Needed for Threadx; Dummy for Linux */

	ENTER();
    
#ifdef PS_REQUIRED
	if ((Adapter->PSState == PS_STATE_SLEEP)
		|| (Adapter->PSState == PS_STATE_PRE_SLEEP)
	) {
		PRINTM(INFO, "In PS State %d"
			" - Not sending the packet\n", Adapter->PSState);
		goto done;
	}
#endif

	spin_lock_irqsave(&Adapter->CurrentTxLock, flags);

	if(priv->wlan_dev.dnld_sent) {

		if(priv->adapter->wmm.fw_notify) {
			wmm_send_prio_pkt_avail(priv);
			priv->adapter->wmm.fw_notify = 0;
		}

		spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);
		
		goto done;
	}

	wmm_pop_highest_prio_skb(priv);
	spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);

	if (Adapter->CurrentTxSkb) {
		wlan_process_tx(priv);
	}
		
	
done:
	LEAVE();
}
