/** @file  wlan_wext.c 
  * @brief This file contains ioctl functions
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	10/10/05: Add Doxygen format comments
	
********************************************************/


#include	"include.h"

#define WAIT_FOR_SCAN_RRESULT_MAX_TIME (10 * HZ)
/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/
int SSIDcmp(WLAN_802_11_SSID *ssid1, WLAN_802_11_SSID *ssid2);
static int get_active_data_rates(wlan_adapter *Adapter, WLAN_802_11_RATES rates);

static inline int wlan_scan_type_ioctl(wlan_private *priv, struct iwreq *wrq);

/** 
 *  @brief Find a character in a string.
 *   
 *  @param s	   A pointer to string
 *  @param c	   Character to be located 
 *  @param dlen    the length of string
 *  @return 	   A pointer to the first occurrence of c in string, or NULL if c is not found.
 */
static void *wlan_memchr(void *s, int c, int n)
{
	const u8	*p = s;

	while(n-- != 0) {
		if((u8)c == *p++) {
			return (void *) (p - 1);
		}
	}
	return NULL;
}


#if WIRELESS_EXT > 14
/** 
 *  @brief Convert mw value to dbm value
 *   
 *  @param mw	   the value of mw
 *  @return 	   the value of dbm
 */
static int mw_to_dbm(int mw)
{
	if (mw < 2)
		return 0;
	else if (mw < 3)
		return 3;
	else if (mw < 4)
		return 5;
	else if (mw < 6)
		return 7;
	else if (mw < 7)
		return 8;
	else if (mw < 8)
		return 9;
	else if (mw < 10)
		return 10;
	else if (mw < 13)
		return 11;
	else if (mw < 16)
		return 12;
	else if (mw < 20)
		return 13;
	else if (mw < 25)
		return 14;
	else if (mw < 32)
		return 15;
	else if (mw < 40)
		return 16;
	else if (mw < 50)
		return 17;
	else if (mw < 63)
		return 18;
	else if (mw < 79)
		return 19;
	else if (mw < 100)
		return 20;
	else
		return 21;
}
#endif

/** 
 *  @brief Find the channel frequency power info with specific channel
 *   
 *  @param adapter 	A pointer to wlan_adapter structure
 *  @param band		it can be BAND_A, BAND_G or BAND_B
 *  @param channel      the channel for looking	
 *  @return 	   	A pointer to CHANNEL_FREQ_POWER structure or NULL if not find.
 */
static CHANNEL_FREQ_POWER *find_cfp_by_band_and_channel(wlan_adapter *adapter, 
							u8 band, u16 channel)
{
	CHANNEL_FREQ_POWER	*cfp = NULL;
	REGION_CHANNEL		*rc;
	int	count = sizeof(adapter->region_channel) / 
				sizeof(adapter->region_channel[0]);
	int i, j;

	for (j = 0; !cfp && (j < count); j++) {
		rc = &adapter->region_channel[j];

#ifdef ENABLE_802_11D
		if( adapter->State11D.Enable11D == ENABLE_11D ) {
			rc = &adapter->universal_channel[j];
		}
#endif
		if(!rc->Valid || !rc->CFP)
			continue;
#ifdef MULTI_BANDS
		switch (rc->Band) {
			case BAND_A:
				switch (band) {
					case BAND_A:	// matching BAND_A
						break;
					default:
						continue;
				}
				break;
			case BAND_B:
			case BAND_G:
				switch (band) {
					case BAND_B:	//matching BAND_B/G
					case BAND_G:
						break;
					default:
						continue;
				}
				break;
			default:
				continue;
		}
#else
		if (rc->Band != band)
			continue;
#endif
		for (i = 0; i < rc->NrCFP; i++) {
			if (rc->CFP[i].Channel == channel) {
				cfp = &rc->CFP[i];
				break;
			}
		}
	}

	if (!cfp && channel)
		PRINTM(INFO, "find_cfp_by_band_and_channel(): cannot find "
			"cfp by band %d & channel %d\n",band,channel);

	return cfp;
}
			
/** 
 *  @brief Find the channel frequency power info with specific frequency
 *   
 *  @param adapter 	A pointer to wlan_adapter structure
 *  @param band		it can be BAND_A, BAND_G or BAND_B
 *  @param freq	        the frequency for looking	
 *  @return 	   	A pointer to CHANNEL_FREQ_POWER structure or NULL if not find.
 */
static CHANNEL_FREQ_POWER *find_cfp_by_band_and_freq(wlan_adapter *adapter, 
							u8 band, u32 freq)
{
	CHANNEL_FREQ_POWER *cfp = NULL;
	REGION_CHANNEL *rc;
	int count = sizeof(adapter->region_channel) / 
				sizeof(adapter->region_channel[0]);
	int i, j;

	for (j = 0; !cfp && (j < count); j++) {
		rc = &adapter->region_channel[j];

#ifdef ENABLE_802_11D
		if( adapter->State11D.Enable11D == ENABLE_11D ) {
			rc = &adapter->universal_channel[j];
		}
#endif

		if(!rc->Valid || !rc->CFP)
			continue;
#ifdef MULTI_BANDS
		switch (rc->Band) {
			case BAND_A:
				switch (band) {
					case BAND_A:	// matching BAND_A
						break;
					default:
						continue;
				}
				break;
			case BAND_B:
			case BAND_G:
				switch (band) {
					case BAND_B:
					case BAND_G:
						break;
					default:
						continue;
				}
				break;
			default:
				continue;
		}
#else
		if (rc->Band != band)
			continue;
#endif
		for (i = 0; i < rc->NrCFP; i++) {
			if (rc->CFP[i].Freq == freq) {
				cfp = &rc->CFP[i];
				break;
			}
		}
	}

	if(!cfp && freq)
		PRINTM(INFO, "find_cfp_by_band_and_freql(): cannot find cfp by "
				"band %d & freq %d\n",band,freq);

	return cfp;
}
	

#ifdef MANF_CMD_SUPPORT
/** 
 *  @brief Manufacturing command ioctl function
 *   
 *  @param priv 	A pointer to wlan_private structure
 *  @param userdata	A pointer to user buf
 *  @return 	   	WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int wlan_mfg_command(wlan_private * priv, void *userdata)
{
	PkHeader	*pkHdr;
	int		len, ret;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	// creating the cmdbuf
	if (Adapter->mfg_cmd == NULL) {
		PRINTM(INFO, "Creating cmdbuf\n");
		if (!(Adapter->mfg_cmd = kmalloc(2000, GFP_KERNEL))) {
			PRINTM(INFO, "kmalloc failed!\n");
			return WLAN_STATUS_FAILURE;
		}
		Adapter->mfg_cmd_len = 2000;
	}
	
	// get PktHdr from userdata
	if (copy_from_user(Adapter->mfg_cmd, userdata, sizeof(PkHeader))) {
		PRINTM(INFO, "copy from user failed :PktHdr\n");
		return WLAN_STATUS_FAILURE;
	}

	// get the size
	pkHdr = (PkHeader *) Adapter->mfg_cmd;
	len = pkHdr->len;

	PRINTM(INFO, "cmdlen = %d\n", (u32) len);

	while (len >= Adapter->mfg_cmd_len) {
		kfree(Adapter->mfg_cmd);

		if (!(Adapter->mfg_cmd = kmalloc(len + 256, GFP_KERNEL))) {
			PRINTM(INFO, "kmalloc failed!\n");
			return WLAN_STATUS_FAILURE;
		}

		Adapter->mfg_cmd_len = len + 256;
	}

	// get the whole command from user
	if (copy_from_user(Adapter->mfg_cmd, userdata, len)) {
		PRINTM(INFO, "copy from user failed :PktHdr\n");
		return WLAN_STATUS_FAILURE;
	}

	ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_MFG_COMMAND,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	// copy it back to user
	if (Adapter->mfg_cmd_resp_len > 0) {
		if (copy_to_user(userdata, Adapter->mfg_cmd, len)) {
			PRINTM(INFO, "copy to user failed \n");
			return WLAN_STATUS_FAILURE;
		}
	}

	LEAVE();
	return ret;
}
#endif


/*
 *    WEP      WPA      WPA2    ad-hoc  encrypt                       Network
 *  enabled  enabled  enabled    AES     mode    Privacy  WPA  WPA2  Compatible
 *     0        0        0        0      NONE       0      0    0	yes	No security
 *     1        0        0        0      NONE       1      0    0	yes	Static WEP
 *     0        1        0        0       x         1x     1    x	yes	WPA
 *     0        0        1        0       x         1x     x    1	yes	WPA2
 *     0        0        0        1      NONE       1      0    0	yes	Ad-hoc AES
 *     0        0        0        0     !=NONE      1      0    0	yes	Dynamic WEP
 */
/** 
 *  @brief This function checks if network compatible.
 *  
 *  @param Adapter	A pointer to wlan_adapter
 *  @param index	index in BSSID list
 *  @param mode		encryption mode
 *  @return 	   	index in SSID list
 */
static inline int IsNetworkCompatible(wlan_adapter *Adapter, 
						int index, int mode)
{
	ENTER();
	
#ifdef WMM
	if(Adapter->wmm.required
		&& Adapter->BSSIDList[index].Wmm_IE[0] == WMM_IE) {
		Adapter->wmm.enabled = 1; 
	} else {
		Adapter->wmm.enabled = 0; 
	}
#endif /* WMM */

	if (Adapter->BSSIDList[index].InfrastructureMode == mode) {
		if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPDisabled
#ifdef WPA
			&& !Adapter->SecInfo.WPAEnabled
#ifdef WPA2
			&& !Adapter->SecInfo.WPA2Enabled
#endif
			&& Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0] != WPA_IE
#ifdef WPA2
			&& Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0] != WPA2_IE
#endif
#endif
#ifdef ADHOCAES
			&& !Adapter->AdhocAESEnabled
#endif
			&& Adapter->SecInfo.EncryptionMode == CIPHER_NONE
			&& !Adapter->BSSIDList[index].Privacy
		) {
			/* no security */
			LEAVE();
			return index;
		}
		else if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled
#ifdef WPA
			&& !Adapter->SecInfo.WPAEnabled
#ifdef WPA2
			&& !Adapter->SecInfo.WPA2Enabled
#endif	//WPA2
#endif	//WPA
#ifdef ADHOCAES
			&& !Adapter->AdhocAESEnabled
#endif
			&& Adapter->BSSIDList[index].Privacy
		) {
			/* static WEP enabled */
			LEAVE();
			return index;
		}
#ifdef WPA
		else if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPDisabled
			&& Adapter->SecInfo.WPAEnabled
#ifdef WPA2
			&& !Adapter->SecInfo.WPA2Enabled
#endif	//WPA2
			&& Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0] == WPA_IE
#ifdef ADHOCAES
			&& !Adapter->AdhocAESEnabled
#endif
			/* Privacy bit may NOT be set in some APs like LinkSys WRT54G
			   && Adapter->BSSIDList[index].Privacy */
		) {
			/* WPA enabled */
#ifdef WPA2
			PRINTM(INFO, "IsNetworkCompatible() WPA: index=%d wpa_ie=%#x wpa2_ie=%#x "
				"WEP=%s WPA=%s WPA2=%s EncMode=%#x privacy=%#x\n",
				index,
				Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
				Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0],
				(Adapter->SecInfo.WEPStatus==Wlan802_11WEPEnabled)?"e":"d",
				(Adapter->SecInfo.WPAEnabled)?"e":"d",
				(Adapter->SecInfo.WPA2Enabled)?"e":"d",
				Adapter->SecInfo.EncryptionMode,
				Adapter->BSSIDList[index].Privacy);
#endif
			LEAVE();
			return index;
		}
#ifdef WPA2
		else if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPDisabled
			&& !Adapter->SecInfo.WPAEnabled
			&& Adapter->SecInfo.WPA2Enabled
			&& Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0] == WPA2_IE
#ifdef ADHOCAES
			&& !Adapter->AdhocAESEnabled
#endif
			/* Privacy bit may NOT be set in some APs like LinkSys WRT54G
			&& Adapter->BSSIDList[index].Privacy */
		) {
			/* WPA2 enabled */
			PRINTM(INFO, "IsNetworkCompatible() WPA2: index=%d wpa_ie=%#x wpa2_ie=%#x "
				"WEP=%s WPA=%s WPA2=%s EncMode=%#x privacy=%#x\n",
				index,
				Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
				Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0],
				(Adapter->SecInfo.WEPStatus==Wlan802_11WEPEnabled)?"e":"d",
				(Adapter->SecInfo.WPAEnabled)?"e":"d",
				(Adapter->SecInfo.WPA2Enabled)?"e":"d",
				Adapter->SecInfo.EncryptionMode,
				Adapter->BSSIDList[index].Privacy);
			LEAVE();
			return index;
		}
#endif
#endif
#ifdef ADHOCAES
		else if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPDisabled
#ifdef WPA
			&& !Adapter->SecInfo.WPAEnabled
#ifdef WPA2
			&& !Adapter->SecInfo.WPA2Enabled
#endif
			&& Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0] != WPA_IE
#ifdef WPA2
			&& Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0] != WPA2_IE
#endif/* WPA2 */
#endif/*WPA */
			&& Adapter->AdhocAESEnabled
			&& Adapter->SecInfo.EncryptionMode == CIPHER_NONE
			&& Adapter->BSSIDList[index].Privacy
		) {
			/* Ad-hoc AES enabled */
			LEAVE();
			return index;
		}
#endif
		else if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPDisabled
#ifdef WPA
			&& !Adapter->SecInfo.WPAEnabled
#ifdef WPA2
			&& !Adapter->SecInfo.WPA2Enabled
#endif	/* WPA2 */
			&& Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0] != WPA_IE
#ifdef WPA2
			&& Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0] != WPA2_IE
#endif	/* WPA2 */
#endif	/* WPA */
#ifdef ADHOCAES
			&& !Adapter->AdhocAESEnabled
#endif
			&& Adapter->SecInfo.EncryptionMode != CIPHER_NONE
			&& Adapter->BSSIDList[index].Privacy
		) {
			/* dynamic WEP enabled */
#ifdef WPA
#ifdef WPA2
			PRINTM(INFO, "IsNetworkCompatible() dynamic WEP: index=%d wpa_ie=%#x wpa2_ie=%#x "
				"EncMode=%#x privacy=%#x\n",
				index,
				Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
				Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0],
				Adapter->SecInfo.EncryptionMode,
				Adapter->BSSIDList[index].Privacy);
#else
			PRINTM(INFO, "IsNetworkCompatible() dynamic WEP: index=%d wpa_ie=%#x EncMode=%#x "
				"privacy=%#x\n",
				index,
				Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
				Adapter->SecInfo.EncryptionMode,
				Adapter->BSSIDList[index].Privacy);
#endif
#endif
			LEAVE();
			return index;
		}

		/* security doesn't match */
#ifdef WPA
#ifdef WPA2
		PRINTM(INFO, "IsNetworkCompatible() FAILED: index=%d wpa_ie=%#x wpa2_ie=%#x "
			"WEP=%s WPA=%s WPA2=%s EncMode=%#x privacy=%#x\n",
			index,
			Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
			Adapter->BSSIDList[index].wpa2_supplicant.Wpa_ie[0],
			(Adapter->SecInfo.WEPStatus==Wlan802_11WEPEnabled)?"e":"d",
			(Adapter->SecInfo.WPAEnabled)?"e":"d",
			(Adapter->SecInfo.WPA2Enabled)?"e":"d",
			Adapter->SecInfo.EncryptionMode,
			Adapter->BSSIDList[index].Privacy);
#else
		PRINTM(INFO, "IsNetworkCompatible() FAILED: index=%d wpa_ie=%#x "
			"WEP=%s WPA=%s EncMode=%#x privacy=%#x\n",
			index,
			Adapter->BSSIDList[index].wpa_supplicant.Wpa_ie[0],
			(Adapter->SecInfo.WEPStatus==Wlan802_11WEPEnabled)?"e":"d",
			(Adapter->SecInfo.WPAEnabled)?"e":"d",
			Adapter->SecInfo.EncryptionMode,
			Adapter->BSSIDList[index].Privacy);
#endif
#endif
		LEAVE();
		return -ECONNREFUSED;
	}

	/* mode doesn't match */
	LEAVE();
	return -ENETUNREACH;
}

/** 
 *  @brief This function finds bssid in ssid list.
 *  
 *  @param Adapter	A pointer to wlan_adapter
 *  @param bssid	bssid
 *  @param mode		encryption mode
 *  @return 	   	index in BSSID list
 */

static int FindBSSIDInList(wlan_adapter *Adapter, u8 *bssid, int mode)
{
	int	i;

	if (!bssid)
		return -EFAULT;

	PRINTM(INFO, "Num of BSSIDs = %d\n", Adapter->ulNumOfBSSIDs);

	for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
		if (!memcmp(Adapter->BSSIDList[i].MacAddress, bssid, 
								ETH_ALEN)) {
			switch (mode) {
			case Wlan802_11Infrastructure :
			case Wlan802_11IBSS :
				return IsNetworkCompatible(Adapter, i, mode);
			default :
				return i;
			}
		}
	}

	return -ENETUNREACH;
}

/** 
 *  @brief This function finds the best SSID in the Scan List depending on current mode.
 *  
 *  @param Adapter	A pointer to wlan_adapter
 *  @return 	   	index in BSSID list
 */
static int FindBestSSIDInList(wlan_adapter *Adapter)
{
	int	i, mode = Adapter->InfrastructureMode;
	int	bestnet = -ENETUNREACH;
	u8	bestrssi = 0;
	
	ENTER();

	PRINTM(INFO, "Num of BSSIDs = %d\n", Adapter->ulNumOfBSSIDs);

	for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
		switch (mode) {
		case Wlan802_11Infrastructure :
		case Wlan802_11IBSS :
			if (IsNetworkCompatible(Adapter, i, mode) >= 0) {
				if (SCAN_RSSI(Adapter->BSSIDList[i].Rssi) > bestrssi) {
					bestrssi = SCAN_RSSI(Adapter->BSSIDList[i].Rssi);
					bestnet = i;
				}
			}
			break;
		case Wlan802_11AutoUnknown :
		default :
			if (SCAN_RSSI(Adapter->BSSIDList[i].Rssi) > bestrssi) {
				bestrssi = SCAN_RSSI(Adapter->BSSIDList[i].Rssi);
				bestnet = i;
			}
			break;
		}
	}

	LEAVE();
	return bestnet;
}



/** 
 *  @brief Stop the Adhoc Network
 *   
 *  @param priv		 	A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static inline int StopAdhocNetwork(wlan_private *priv)
{
	return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_STOP,
			0,  HostCmd_OPTION_WAITFORRSP,
			0, NULL);
}

/** 
 *  @brief Send Deauthentication Request 
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static inline int SendDeauthentication(wlan_private *priv)
{
#ifdef REASSOC_CMD_SUPPORT
	wlan_adapter *Adapter = priv->adapter;

    	/* If a reassociation attempt is in progress, do not send the deauth */
    	if (Adapter->reassocAttempt){
    	       	return WLAN_STATUS_SUCCESS;
    	}
#endif
	return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DEAUTHENTICATE,
			0,  HostCmd_OPTION_WAITFORRSP,
			0, NULL);
}


/** 
 *  @brief Update Current Channel 
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int UpdateCurrentChannel(wlan_private *priv)
{
    int ret;
    
    /*
    ** the channel in f/w could be out of sync, get the current channel
    */
    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                                HostCmd_OPT_802_11_RF_CHANNEL_GET,
                                HostCmd_OPTION_WAITFORRSP, 0,
                                NULL);
    
    PRINTM(INFO, "Current Channel = %d\n", priv->adapter->Channel);

    return ret;
}

/** 
 *  @brief Set Current Channel 
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param channel		The channel to be set. 
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int SetCurrentChannel(wlan_private *priv, int channel)
{
    PRINTM(INFO, "Set Channel = %d\n", channel);

    /* 
    **  Current channel is not set to AdhocChannel requested, set channel
    */
    return (PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                                  HostCmd_OPT_802_11_RF_CHANNEL_SET,
                                  HostCmd_OPTION_WAITFORRSP, 0,
                                  &channel));
}

/** 
 *  @brief Change Adhoc Channel
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param channel		The channel to be set. 
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int ChangeAdhocChannel(wlan_private *priv, int channel)
{
    	int             ret      = WLAN_STATUS_SUCCESS;
	wlan_adapter    *Adapter = priv->adapter;

    	Adapter->AdhocChannel = channel;

    	UpdateCurrentChannel(priv);

    	if (Adapter->Channel == Adapter->AdhocChannel){
        	/* AdhocChannel is set to the current Channel already */
        	LEAVE();
        	return WLAN_STATUS_SUCCESS;
    	}

    	PRINTM(INFO, "Updating Channel from %d to %d\n", 
	Adapter->Channel, Adapter->AdhocChannel);

    	SetCurrentChannel(priv, Adapter->AdhocChannel);

    	UpdateCurrentChannel(priv);

    	if (Adapter->Channel != Adapter->AdhocChannel){
        	PRINTM(INFO, "Failed to updated Channel to %d, channel = %d\n", 
                Adapter->AdhocChannel, Adapter->Channel);
        	LEAVE();
        	return WLAN_STATUS_FAILURE;
    	}

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		int			i;
		WLAN_802_11_SSID	curAdhocSsid;
		
		PRINTM(INFO, "Channel Changed while in an IBSS\n");

		/* Copy the current ssid */
		memcpy(&curAdhocSsid, &Adapter->CurBssParams.ssid, 
				sizeof(WLAN_802_11_SSID));
		
		/* Exit Adhoc mode */
		PRINTM(INFO, "In ChangeAdhocChannel(): Sending Adhoc Stop\n");
		ret = StopAdhocNetwork(priv);

		if (ret) {
			LEAVE();
			return ret;
		}

		SendSpecificScan(priv, &curAdhocSsid);

		// find out the BSSID that matches the current SSID 
		i = FindSSIDInList(Adapter, &curAdhocSsid, NULL, 
				Wlan802_11IBSS);

		if (i >= 0) {
			PRINTM(INFO, "SSID found at %d in List," 
					"so join\n", i);
			JoinAdhocNetwork(priv, &curAdhocSsid, i);
		} else {
			// else send START command
			PRINTM(INFO, "SSID not found in list, "
					"so creating adhoc with ssid = %s\n",
					curAdhocSsid.Ssid);
			StartAdhocNetwork(priv, &curAdhocSsid);
		}	// end of else (START command)
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Find the AP with specific ssid in the scan list
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param pSSID                A pointer to AP's ssid   
 *  @return 	   		WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int FindBestNetworkSsid(wlan_private *priv, WLAN_802_11_SSID *pSSID)
{
	wlan_adapter	*Adapter = priv->adapter;
	int		i, ret = WLAN_STATUS_SUCCESS;	
	
	ENTER();
	
	memset(pSSID, 0, sizeof(WLAN_802_11_SSID));
	
	wlan_scan_networks(priv);

	PRINTM(INFO, "AUTOASSOCIATE: Scan complete - doing associate...\n");

	i = FindBestSSIDInList(Adapter);
	
	if (i >= 0) {
		PWLAN_802_11_BSSID	pReqBSSID;

		pReqBSSID = &Adapter->BSSIDList[i];
		memcpy(pSSID, &pReqBSSID->Ssid, sizeof(WLAN_802_11_SSID));

		/* Make sure we are in the right mode */
		if (Adapter->InfrastructureMode == Wlan802_11AutoUnknown) {
			Adapter->InfrastructureMode =
				pReqBSSID->InfrastructureMode;

			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_SNMP_MIB,
				       	HostCmd_PENDING_ON_SET_OID,
					HostCmd_OPTION_WAITFORRSP,
					OID_802_11_INFRASTRUCTURE_MODE,
					NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
		}
	} 
	
	if (!pSSID->SsidLength) {
		ret = WLAN_STATUS_FAILURE;
	}

	LEAVE();
	return ret;
}



#ifdef WPA
/** 
 *  @brief Set WPA key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_encode_wpa(struct net_device *dev, 
			struct iw_request_info *info,
				struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
#ifdef DEEP_SLEEP
	wlan_adapter		*Adapter = priv->adapter;
#endif
	PWLAN_802_11_KEY	pKey;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "():%s IOCTLS called when station is"
					" in DeepSleep\n",__FUNCTION__);
		return -EBUSY;
	}
#endif

	pKey = (PWLAN_802_11_KEY) extra;

	HEXDUMP("Key buffer: ", extra, dwrq->length);

	HEXDUMP("KeyMaterial: ", (u8 *) pKey->KeyMaterial,
			pKey->KeyLength);

	// current driver only supports key length of up to 32 bytes
	if (pKey->KeyLength > MRVL_MAX_WPA_KEY_LENGTH) {
		PRINTM(INFO, " Error in key length \n");
		return WLAN_STATUS_FAILURE;
	}

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_KEY_MATERIAL,
		HostCmd_ACT_SET,
		HostCmd_OPTION_WAITFORRSP,
		KEY_INFO_ENABLED, pKey);
	if (ret) {
		LEAVE();
		return ret;
	}

	LEAVE();
	return ret;
}
#endif

/*
 *  iwconfig ethX key on:	WEPEnabled;
 *  iwconfig ethX key off:	WEPDisabled;
 *  iwconfig ethX key [x]:	CurrentWepKeyIndex = x; WEPEnabled;
 *  iwconfig ethX key [x] kstr:	WepKey[x] = kstr;
 *  iwconfig ethX key kstr:	WepKey[CurrentWepKeyIndex] = kstr;
 *
 *  all:			Send command SET_WEP;
 				SetMacPacketFilter;
 */

/** 
 *  @brief Set WEP key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_encode_nonwpa(struct net_device *dev,
			       struct iw_request_info *info,
				       struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	MRVL_WEP_KEY		*pWep;
	WLAN_802_11_SSID	ssid;
	int			index, PrevAuthMode;

	ENTER();

	pWep = &Adapter->WepKey[Adapter->CurrentWepKeyIndex];
	PrevAuthMode = Adapter->SecInfo.AuthenticationMode;

	index = (dwrq->flags & IW_ENCODE_INDEX) - 1;

	if(index >= 4) {
		PRINTM(INFO, "Key index #%d out of range.\n", index + 1);
		return -EINVAL;
	}

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "():%s IOCTLS called when station is"
					" in DeepSleep\n",__FUNCTION__);
		return -EBUSY;
	}
#endif

	PRINTM(INFO, "Flags=0x%x, Length=%d Index=%d CurrentWepKeyIndex=%d\n", 
			dwrq->flags, dwrq->length, index, Adapter->CurrentWepKeyIndex);

	if (dwrq->length > 0) {
		/* iwconfig ethX key [n] xxxxxxxxxxx 
		 * Key has been provided by the user 
		 */

		/*
		 * Check the size of the key 
		 */

		if (dwrq->length > MAX_KEY_SIZE) {
			return -EINVAL;
		}

		/*
		 * Check the index (none -> use current) 
		 */

		if (index < 0 || index > 3)		//invalid index or no index
			index = Adapter->CurrentWepKeyIndex;
		else					//index is given & valid
			pWep = &Adapter->WepKey[index];

		/*
		 * Check if the key is not marked as invalid 
		 */
		if (!(dwrq->flags & IW_ENCODE_NOKEY)) {
			/* Cleanup */
			memset(pWep, 0, sizeof(MRVL_WEP_KEY));

			/* Copy the key in the driver */
			memcpy(pWep->KeyMaterial, extra, dwrq->length);

			/* Set the length */
			if (dwrq->length > MIN_KEY_SIZE) {
				pWep->KeyLength = MAX_KEY_SIZE;
			} else {
				if (dwrq->length > 0) {
					pWep->KeyLength = MIN_KEY_SIZE;
				} else {
					/* Disable the key */
					pWep->KeyLength = 0;
				}
			}
			pWep->KeyIndex = index;

			if(Adapter->SecInfo.WEPStatus != Wlan802_11WEPEnabled) {
				/*
				 * The status is set as Key Absent 
				 * so as to make sure we display the 
				 * keys when iwlist ethX key is 
				 * used - MPS 
				 */

				Adapter->SecInfo.WEPStatus = 
					Wlan802_11WEPKeyAbsent;
			}

			PRINTM(INFO, "KeyIndex=%u KeyLength=%u\n", 
					pWep->KeyIndex,
					pWep->KeyLength);
			HEXDUMP("WepKey", (u8 *) pWep->KeyMaterial, 
					pWep->KeyLength);
		}
	} else {
		/*
		 * No key provided so it is either enable key, 
		 * on or off */
		if (dwrq->flags & IW_ENCODE_DISABLED) {
			PRINTM(INFO, "******** iwconfig ethX key off **********\n");

			Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
			if (Adapter->SecInfo.AuthenticationMode == Wlan802_11AuthModeShared)
				Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
		} else {
			/* iwconfig ethX key [n]
			 * iwconfig ethX key on 
			 * Do we want to just set the transmit key index ? 
			 */

			if (index < 0 || index > 3) {
				PRINTM(INFO, "******** iwconfig ethX key on **********\n");
				index = Adapter->CurrentWepKeyIndex;
			}
			else {
				PRINTM(INFO, "******** iwconfig ethX key [x=%d] **********\n",index);
				Adapter->CurrentWepKeyIndex = index;
			}

			/* Copy the required key as the current key */
			pWep = &Adapter->WepKey[index];

			if (!pWep->KeyLength) {
				PRINTM(INFO, "Key not set,so cannot enable it\n");
				return -EPERM;
			}
	    
			Adapter->SecInfo.WEPStatus = Wlan802_11WEPEnabled;

			HEXDUMP("KeyMaterial", (u8 *) pWep->KeyMaterial,
					pWep->KeyLength);
		}
	}

	if(pWep->KeyLength) {
		ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SET_WEP,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_ADD_WEP, NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	}

	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled) {
		Adapter->CurrentPacketFilter |= HostCmd_ACT_MAC_WEP_ENABLE;
	} else {
		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WEP_ENABLE;
	}

	SetMacPacketFilter(priv);

	if (dwrq->flags & IW_ENCODE_RESTRICTED) {
		/* iwconfig ethX restricted key [1] */
		Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeShared;
		PRINTM(INFO, "Authentication mode restricted!\n");
	} else if (dwrq->flags & IW_ENCODE_OPEN) {
		/* iwconfig ethX key [2] open */
		Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
		PRINTM(INFO, "Authentication mode open!\n");
	}

	/*
	 * If authentication mode changed - de-authenticate, set authentication
	 * method and re-associate if we were previously associated.
	 */
	if (Adapter->SecInfo.AuthenticationMode != PrevAuthMode) {
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected &&
				Adapter->InfrastructureMode == 
				Wlan802_11Infrastructure) {

			/* keep a copy of the ssid associated with */
			memcpy(&ssid, &Adapter->CurBssParams.ssid, sizeof(ssid));

			/*
			 * De-authenticate from AP 
			 */
				
			ret = SendDeauthentication(priv);

			if (ret) {
				LEAVE();
				return ret;
			}

		} else {
			/* reset ssid */
			memset(&ssid, 0, sizeof(ssid));
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Set RX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param Mode			RF antenna mode
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int SetRxAntenna(wlan_private *priv, int Mode)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	if (Mode != RF_ANTENNA_1 && Mode != RF_ANTENNA_2
					&& Mode != RF_ANTENNA_AUTO) {
		return -EINVAL;
	}

	Adapter->RxAntennaMode = Mode;

	PRINTM(INFO, "SET RX Antenna Mode to 0x%04x\n", Adapter->RxAntennaMode);
			
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_SET_RX, HostCmd_OPTION_WAITFORRSP,
			0, &Adapter->RxAntennaMode);
	return ret;
}

/** 
 *  @brief Set TX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param Mode			RF antenna mode
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int SetTxAntenna(wlan_private *priv, int Mode)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	if ((Mode != RF_ANTENNA_1) && (Mode != RF_ANTENNA_2)
			&& (Mode != RF_ANTENNA_AUTO)) {
		return -EINVAL;
	}
		
	Adapter->TxAntennaMode = Mode; 

	PRINTM(INFO, "SET TX Antenna Mode to 0x%04x\n", Adapter->TxAntennaMode);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_SET_TX, HostCmd_OPTION_WAITFORRSP,
			0, &Adapter->TxAntennaMode);

	return ret;
}

/** 
 *  @brief Get RX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param buf			A pointer to recieve antenna mode
 *  @return 	   		length of buf 
 */
static int GetRxAntenna(wlan_private *priv, char *buf)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	// clear it, so we will know if the value 
	// returned below is correct or not.
	Adapter->RxAntennaMode = 0;	

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_GET_RX, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	PRINTM(INFO, "Get Rx Antenna Mode:0x%04x\n", Adapter->RxAntennaMode);

	LEAVE();
			
	return sprintf(buf, "0x%04x", Adapter->RxAntennaMode) + 1;
}


/** 
 *  @brief Get TX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param buf			A pointer to recieve antenna mode
 *  @return 	   		length of buf 
 */
static int GetTxAntenna(wlan_private *priv, char *buf)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	
	// clear it, so we will know if the value 
	// returned below is correct or not.
	Adapter->TxAntennaMode = 0;	

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_GET_TX, HostCmd_OPTION_WAITFORRSP,
			0, NULL);		

	if (ret) {
		LEAVE();
		return ret;
	}

	PRINTM(INFO, "Get Tx Antenna Mode:0x%04x\n", Adapter->TxAntennaMode);

	LEAVE();

	return sprintf(buf, "0x%04x", Adapter->TxAntennaMode) + 1;
}


/** 
 *  @brief Send Deauth Request
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_send_deauth(wlan_private *priv)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;
	
	ENTER();
	
    	if (Adapter->InfrastructureMode == Wlan802_11Infrastructure &&
		Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		
		ret = SendDeauthentication(priv);

	} else {
		LEAVE();
		return -ENOTSUPP;
	}

	LEAVE();
	return ret;
}


/** 
 *  @brief Stop Adhoc Network
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_do_adhocstop_ioctl(wlan_private *priv)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->InfrastructureMode == Wlan802_11IBSS && 
		Adapter->MediaConnectStatus == WlanMediaStateConnected) {

		ret = StopAdhocNetwork(priv);

	} else {
		LEAVE();
		return -ENOTSUPP;
	}

	LEAVE();
	
	return WLAN_STATUS_SUCCESS;
		
}

/** 
 *  @brief Set Radio On/OFF
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @option 			Radio Option
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_radio_ioctl(wlan_private *priv, u8 option)
{
	int		ret = 0;
	wlan_adapter	*Adapter	= priv->adapter;

	ENTER();

	if (Adapter->RadioOn != option) {
		PRINTM(INFO, "Switching %s the Radio\n", option ? "On" : "Off");
		Adapter->RadioOn = option;
		
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RADIO_CONTROL,
				HostCmd_ACT_GEN_SET, HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
	}

	LEAVE();
	return ret;
}

#ifdef REASSOCIATION
/** 
 *  @brief Set Auto Reassociation On
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int reassociation_on(wlan_private *priv)
{
	wlan_adapter	*Adapter	= priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "():%s IOCTLS called when station is"
					" in DeepSleep\n",__FUNCTION__);
		return -EBUSY;
	}
#endif

	Adapter->Reassoc_on = TRUE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Auto Reassociation Off
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int reassociation_off(wlan_private *priv)
{
	wlan_adapter		*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "():%s IOCTLS called when station is"
					" in DeepSleep\n",__FUNCTION__);
		return -EBUSY;
	}
#endif
	
	if (Adapter->TimerIsSet == TRUE) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}	

	Adapter->Reassoc_on = FALSE;
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* REASSOCIATION */

/** 
 *  @brief Set Idle Off
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlanidle_off(wlan_private *priv)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_SSID	*AdhocSsid;

	ENTER();

	if (Adapter->MediaConnectStatus == WlanMediaStateDisconnected 
			&& Adapter->PreviousSSID.Ssid[0] != '\0') {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
			PRINTM(INFO, "Previous SSID = %s\n", 
					Adapter->PreviousSSID.Ssid);
			
			ret = wlan_associate(priv, &Adapter->PreviousSSID);

		} else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
	
			/* Copy the current ssid */
			memcpy(&AdhocSsid, &Adapter->PreviousSSID, 
						sizeof(WLAN_802_11_SSID));
			ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_AD_HOC_START,
				0, HostCmd_OPTION_WAITFORRSP,
				0, AdhocSsid);
		}
	}
	/* else it is connected */
	
	PRINTM(INFO, "\nwlanidle is off");
	LEAVE();
	return ret;
}

/** 
 *  @brief Set Idle On
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlanidle_on(wlan_private *priv)
{
	int			ret = 0;
	wlan_adapter		*Adapter = priv->adapter;

	ENTER();

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
			PRINTM(INFO, "Previous SSID = %s\n", 
					Adapter->PreviousSSID.Ssid);
			memmove(&Adapter->PreviousSSID, 
					&Adapter->CurBssParams.ssid,
						sizeof(WLAN_802_11_SSID));
 			wlan_send_deauth(priv);
		
		} else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
			ret = StopAdhocNetwork(priv);
		}

	}

#ifdef REASSOCIATION
	if (Adapter->TimerIsSet == TRUE) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}	
#endif /* REASSOCIATION */

	PRINTM(INFO, "\nwlanidle is on");

	LEAVE();
	return ret;
}

/** 
 *  @brief Set Region
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param region_code		region code
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_set_region(wlan_private *priv, u16 region_code)
{
	int i;

	ENTER();
	
	for (i = 0; i < MRVDRV_MAX_REGION_CODE; i++) {
		// use the region code to search for the index
		if (region_code == RegionCodeToIndex[i]) {
			priv->adapter->RegionTableIndex = (u16) i;
			priv->adapter->RegionCode = region_code;
			break;
		}
	}

	// if it's unidentified region code
	if (i >= MRVDRV_MAX_REGION_CODE) {
		PRINTM(INFO, "Region Code not identified\n");
		LEAVE();
		return WLAN_STATUS_FAILURE;
	}

#ifdef MULTI_BANDS
	if (wlan_set_regiontable(priv, priv->adapter->RegionCode, 
					priv->adapter->config_bands)) {
#else
	if (wlan_set_regiontable(priv, priv->adapter->RegionCode, 0)) {
#endif
		LEAVE();
		return -EINVAL;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Copy Rates
 *   
 *  @param dest                 A pointer to Dest Buf
 *  @param src		        A pointer to Src Buf
 *  @param len                  The len of Src Buf
 *  @return 	   	        Number of Rates copyed 
 */
static inline int CopyRates(u8 *dest, int pos, u8 *src, int len)
{
	int	i;

	for (i = 0; i < len && src[i]; i++, pos++) {
		if (pos >= sizeof(WLAN_802_11_RATES))
			break;
		dest[pos] = src[i];
	}

	return pos;
}

/** 
 *  @brief Get active data rates
 *   
 *  @param Adapter              A pointer to wlan_adapter structure
 *  @param rate		        The buf to return the active rates
 *  @return 	   	        The number of Rates
 */
static int get_active_data_rates(wlan_adapter *Adapter, WLAN_802_11_RATES rates)
{
	int	k = 0;

	ENTER();
	
	if (Adapter->MediaConnectStatus != WlanMediaStateConnected) {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {	
			//Infra. mode
#ifdef	MULTI_BANDS
			if (Adapter->config_bands & BAND_G) {
				PRINTM(INFO, "Infra mBAND_G=%#x\n", BAND_G);
				k = CopyRates(rates, k, SupportedRates_G, 
						sizeof(SupportedRates_G));
			} else {
				if (Adapter->config_bands & BAND_B) {
					PRINTM(INFO, "Infra mBAND_B=%#x\n", BAND_B);
					k = CopyRates(rates, k, 
						SupportedRates_B, 
						sizeof(SupportedRates_B));
				}
				if (Adapter->config_bands & BAND_A) {
					PRINTM(INFO, "Infra mBAND_A=%#x\n", BAND_A);
					k = CopyRates(rates, k, 
						SupportedRates_A, 
						sizeof(SupportedRates_A));
				}
			}
#else //MULTI_BANDS
			PRINTM(INFO, "Infra\n");
			k = CopyRates(rates, k, SupportedRates, 
						sizeof(SupportedRates));
#endif //MULTI_BANDS
		} else {
			//ad-hoc mode
#ifdef	MULTI_BANDS
			if (Adapter->config_bands & BAND_G) {
				PRINTM(INFO, "Adhoc mBAND_G=%#x\n", BAND_G);
				k = CopyRates(rates, k, AdhocRates_G, 
							sizeof(AdhocRates_G));
			} else if (Adapter->config_bands & BAND_B) {
				PRINTM(INFO, "Adhoc mBAND_B=%#x\n", BAND_B);
				k = CopyRates(rates, k, AdhocRates_B, 
							sizeof(AdhocRates_B));
			}
			if (Adapter->config_bands & BAND_A) {
				PRINTM(INFO, "Adhoc mBAND_A=%#x\n", BAND_A);
				k = CopyRates(rates, k, AdhocRates_A, 
							sizeof(AdhocRates_A));
			}
#else //MULTI_BANDS
#ifdef G_RATE
			PRINTM(INFO, "Adhoc G\n");
			k = CopyRates(rates, k, AdhocRates_G, 
							sizeof(AdhocRates_G));
#else //G_RATE
			PRINTM(INFO, "Adhoc B\n");
			k = CopyRates(rates, k, AdhocRates_B, 
							sizeof(AdhocRates_B));
#endif //G_RATE
#endif //MULTI_BANDS
		}
	} else {
		k = CopyRates(rates, 0, Adapter->CurBssParams.DataRates, 
					Adapter->CurBssParams.NumOfRates);
	}

	LEAVE();

	return k;
}

/** 
 *  @brief Get/Set Firmware wakeup method
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_txcontrol(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	int data;	
	ENTER();

	if((int) wrq->u.data.length == 0) {
	    if(copy_to_user(wrq->u.data.pointer, &Adapter->PktTxCtrl, sizeof(u32))) {
		PRINTM(MSG, "copy_to_user failed!\n");
		return -EFAULT;
	    }	    
	} else{
	     if((int) wrq->u.data.length > 1) {
	  	PRINTM(MSG, "ioctl too many args!\n");
	   	return -EFAULT;
	     }
	     copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
	     (u32)Adapter->PktTxCtrl = data;
	}

	wrq->u.data.length = 1;
			
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef ATIMGEN			
/** 
 *  @brief Enable/Disable atim generation
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_atimgen( wlan_private *priv, struct iwreq *wrq )
{
	int data; 
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

	if ( wrq->u.data.length == 0 ) {
		return -EINVAL;
	}
	copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
	PRINTM(INFO, "Enable ATIMGEN: %s\n", (data==1)?"Enable":"Disable");
	switch ( data ) {
		case CMD_ENABLED:
			Adapter->ATIMEnabled = TRUE;
			PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_GENERATE_ATIM,
				HostCmd_ACT_SET,
				HostCmd_OPTION_WAITFORRSP,
				0,
				NULL);
			break;
		case CMD_DISABLED:
			Adapter->ATIMEnabled = FALSE;
			PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_GENERATE_ATIM,
				HostCmd_ACT_SET,
				HostCmd_OPTION_WAITFORRSP,
				0,
				NULL);
			break;
		default:			
			PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_GENERATE_ATIM,
				HostCmd_ACT_GET,
				HostCmd_OPTION_WAITFORRSP,
				0,
				NULL);
		
			break;
	}

	data = (Adapter->ATIMEnabled==TRUE)?CMD_ENABLED:CMD_DISABLED;
	copy_to_user(wrq->u.data.pointer, &data, sizeof(int) );
	wrq->u.data.length = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif

#ifdef WMM_UAPSD
/** 
 *  @brief Enable/Disable atim uapsd null package generation
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_null_pkg_gen( wlan_private *priv, struct iwreq *wrq )
{
	int data;
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

	if ( wrq->u.data.length == 0 ) {
		return -EINVAL;
	}
	copy_from_user(&data,wrq->u.data.pointer,sizeof(int) );

	PRINTM(INFO, "Enable UAPSD NULL PKG: %s\n", 
			(data==CMD_ENABLED)?"Enable":"Disable");
	
	switch ( data ) {
		case CMD_ENABLED:
			Adapter->wmm.gen_null_pkg = TRUE;
			break;
		case CMD_DISABLED:
			Adapter->wmm.gen_null_pkg = FALSE;
			break;
		default:			
			break;
	}

	data = (Adapter->wmm.gen_null_pkg ==TRUE)?CMD_ENABLED:CMD_DISABLED;
	copy_to_user(wrq->u.data.pointer, &data, sizeof(int) );
	wrq->u.data.length = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif


#ifdef	linux
#include	"wlan_version.h"

#define GETLOG_BUFSIZE  300

#define MAX_SCAN_CELL_SIZE      (IW_EV_ADDR_LEN + \
				MRVDRV_MAX_SSID_LENGTH + \
				IW_EV_UINT_LEN + IW_EV_FREQ_LEN + \
				IW_EV_QUAL_LEN + MRVDRV_MAX_SSID_LENGTH + \
				IW_EV_PARAM_LEN + 40) /* 40 for WPAIE */



/** 
 *  @brief Commit handler: called after a bunch of SET operations
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_config_commit(struct net_device *dev, 
		struct iw_request_info *info, 
		char *cwrq, char *extra)
{
	ENTER();

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get protocol name 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_name(struct net_device *dev, struct iw_request_info *info, 
		char *cwrq, char *extra)
{
	const char *cp;
	char comm[6] = {"COMM-"};
	char mrvl[6] = {"MRVL-"};
	int cnt;

	ENTER();

	strcpy(cwrq, mrvl);

	cp = strstr(driver_version, comm);
	if (cp == driver_version)	//skip leading "COMM-"
		cp = driver_version + strlen(comm);
	else
		cp = driver_version;

	cnt = strlen(mrvl);
	cwrq += cnt;
	while (cnt < 16 && (*cp != '-')) {
		*cwrq++ = toupper(*cp++);
		cnt++;
	}
	*cwrq = '\0';

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get frequency
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_freq(struct net_device *dev, struct iw_request_info *info, 
		struct iw_freq *fwrq, char *extra)
{
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	CHANNEL_FREQ_POWER	*cfp;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCGIWFREQ: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

#ifdef MULTI_BANDS
	cfp = find_cfp_by_band_and_channel(Adapter, Adapter->CurBssParams.band, 
						Adapter->CurBssParams.channel);
#else
	cfp = find_cfp_by_band_and_channel(Adapter, 0, 
						Adapter->CurBssParams.channel);
#endif
	
	if (!cfp) {
		if (Adapter->CurBssParams.channel)
			PRINTM(INFO, "Invalid channel=%d\n", 
					Adapter->CurBssParams.channel);
		return -EINVAL;
	} 
	
	fwrq->m = (long) cfp->Freq * 100000;
	fwrq->e = 1;

	PRINTM(INFO, "freq=%u\n",fwrq->m);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Connect to the AP or Ad-hoc Network with specific bssid
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_wap(struct net_device *dev,	struct iw_request_info *info,
		struct sockaddr *awrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	static const u8 	bcast[ETH_ALEN] = 
					{ 255, 255, 255, 255, 255, 255 };
	WLAN_802_11_SSID	SSID;
	u8			reqBSSID[ETH_ALEN];	
	int 			i = 0;
	
	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWAP: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (awrq->sa_family != ARPHRD_ETHER)
		return -EINVAL;			

	PRINTM(INFO, "sa_data: %02x:%02x:%02x:%02x:%02x:%02x\n",
			awrq->sa_data[0], awrq->sa_data[1],
			awrq->sa_data[2], awrq->sa_data[3],
			awrq->sa_data[4], awrq->sa_data[5]);
	
#ifdef REASSOCIATION
	// cancel re-association timer if there's one
	if (Adapter->TimerIsSet == TRUE) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}
#endif /* REASSOCIATION */

	memset(&SSID, 0, sizeof(WLAN_802_11_SSID));
	
	if (!memcmp(bcast, awrq->sa_data, ETH_ALEN)) {
		i = FindBestSSIDInList(Adapter);
		if (i >= 0) {
			memcpy(reqBSSID, Adapter->BSSIDList[i].MacAddress, 
					ETH_ALEN);
		}
	} else {
		memcpy(reqBSSID, awrq->sa_data, ETH_ALEN);

		PRINTM(INFO, "Required bssid = %x:%x:%x:%x:%x:%x\n", 
				reqBSSID[0], reqBSSID[1],
				reqBSSID[2], reqBSSID[3],
				reqBSSID[4], reqBSSID[5]);
		
		// Search for index position in list for requested MAC
		i = FindBSSIDInList(Adapter, reqBSSID, 
				Adapter->InfrastructureMode);
  
		// If BSSID not found, do scan one more time.
		if (i < 0)
		{
			wlan_scan_networks(priv);

  			i = FindBSSIDInList(Adapter, reqBSSID, 
				Adapter->InfrastructureMode);
		}
	}	
	
	if (i >= 0) {
		memcpy(&SSID, &Adapter->BSSIDList[i].Ssid, sizeof(SSID.Ssid));
	} else {
		PRINTM(INFO, "MAC address not found in BSSID List\n");
		return i;
	}

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		ret = SendDeauthentication(priv);

		if (ret) {
			LEAVE();
			return ret;
		}
	}
	
	Adapter->RequestViaBSSID = TRUE;
	memcpy(Adapter->RequestedBSSID, reqBSSID, ETH_ALEN);
	
	if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
		ret = wlan_associate(priv, &SSID);

		if (ret) {
			LEAVE();
			return ret;
		}
	} else {
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			/* Exit Adhoc mode */
			ret = StopAdhocNetwork(priv);

			if (ret) {
				LEAVE();
				return ret;
			}
		}

		JoinAdhocNetwork(priv, &SSID, i);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get current BSSID
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_wap(struct net_device *dev, struct iw_request_info *info, 
		struct sockaddr *awrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		memcpy(awrq->sa_data, Adapter->CurrentBSSID, ETH_ALEN);
	} else {
		memset(awrq->sa_data, 0, ETH_ALEN);
	}
	awrq->sa_family = ARPHRD_ETHER;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Adapter Node Name
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_nick(struct net_device *dev, struct iw_request_info *info, 
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	/*
	 * Check the size of the string 
	 */
	
	if (dwrq->length > 16) {
		return -E2BIG;
	}

	memset(Adapter->nodeName, 0, sizeof(Adapter->nodeName));
	memcpy(Adapter->nodeName, extra, dwrq->length);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Adapter Node Name
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_nick(struct net_device *dev, struct iw_request_info *info, 
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	/*
	 * Get the Nick Name saved 
	 */

	strncpy(extra, Adapter->nodeName, 16);

	extra[16] = '\0';

	/*
	 * If none, we may want to get the one that was set 
	 */

	/*
	 * Push it out ! 
	 */
	dwrq->length = strlen(extra) + 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set RTS threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_rts(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		rthr = vwrq->value;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWRTS: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (vwrq->disabled) {
		Adapter->RTSThsd = rthr = MRVDRV_RTS_MAX_VALUE;
	} else {
		if (rthr < MRVDRV_RTS_MIN_VALUE || rthr > MRVDRV_RTS_MAX_VALUE)
			return -EINVAL;
		Adapter->RTSThsd = rthr;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_RTS_THRESHOLD, &rthr);

	LEAVE();
	return ret;
}


/** 
 *  @brief Get RTS threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_rts(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCGIWRTS: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (!Adapter->RTSThsd) {
		ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
				0,HostCmd_OPTION_WAITFORRSP,
				OID_802_11_RTS_THRESHOLD, NULL);
		if (ret) {
			LEAVE();
			return ret;
		}
	}

	vwrq->value = Adapter->RTSThsd;
	vwrq->disabled = ((vwrq->value < MRVDRV_RTS_MIN_VALUE)
			 || (vwrq->value > MRVDRV_RTS_MAX_VALUE));
	vwrq->fixed = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Fragment threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_frag(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	int		fthr = vwrq->value;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWFRAG: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (vwrq->disabled) {
		Adapter->FragThsd = fthr = MRVDRV_FRAG_MAX_VALUE;
	} else {
		if (fthr < MRVDRV_FRAG_MIN_VALUE || fthr > MRVDRV_FRAG_MAX_VALUE)
			return -EINVAL;
		Adapter->FragThsd = fthr;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_FRAGMENTATION_THRESHOLD,
			&fthr);
	LEAVE();
	return ret;
}

/** 
 *  @brief Get Fragment threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_frag(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCGIWFRAG: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (!Adapter->FragThsd) {
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_SNMP_MIB, 0,
				HostCmd_OPTION_WAITFORRSP,
				OID_802_11_FRAGMENTATION_THRESHOLD,
				NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	}

	vwrq->value = Adapter->FragThsd;
	vwrq->disabled = ((vwrq->value < MRVDRV_FRAG_MIN_VALUE)
			 || (vwrq->value > MRVDRV_FRAG_MAX_VALUE));
	vwrq->fixed = 1;

	LEAVE();
	return ret;
}

/** 
 *  @brief Get Wlan Mode
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_mode(struct net_device *dev,
	      struct iw_request_info *info, u32 * uwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*adapter = priv->adapter;

	ENTER();

	switch (adapter->InfrastructureMode) {
		case Wlan802_11IBSS:
			*uwrq = IW_MODE_ADHOC;
			break;

		case Wlan802_11Infrastructure:
			*uwrq = IW_MODE_INFRA;
			break;

		default:
		case Wlan802_11AutoUnknown:
			*uwrq = IW_MODE_AUTO;
			break;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get Encryption key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_encode(struct net_device *dev,
			struct iw_request_info *info,
				struct iw_point *dwrq, u8 * extra)
{

	wlan_private	*priv = dev->priv;
	wlan_adapter	*adapter = priv->adapter;
	int		index = (dwrq->flags & IW_ENCODE_INDEX);

	ENTER();

	PRINTM(INFO, "flags=0x%x index=%d length=%d CurrentWepKeyIndex=%d\n", 
			dwrq->flags, index, dwrq->length, 
			adapter->CurrentWepKeyIndex);

	dwrq->flags = 0;

	/*
	 * Check encryption mode 
	 */

	switch (adapter->SecInfo.AuthenticationMode) {
		case Wlan802_11AuthModeOpen:
			dwrq->flags = IW_ENCODE_OPEN;
			break;

		case Wlan802_11AuthModeShared:
		case Wlan802_11AuthModeNetworkEAP:
			dwrq->flags = IW_ENCODE_RESTRICTED;
			break;
		default:
			dwrq->flags = IW_ENCODE_DISABLED | IW_ENCODE_OPEN;
			break;
	}

	if ((adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled)
		|| (adapter->SecInfo.WEPStatus == Wlan802_11WEPKeyAbsent)
#ifdef WPA
		|| adapter->SecInfo.WPAEnabled
#ifdef WPA2
		|| adapter->SecInfo.WPA2Enabled
#endif
#endif
		) {
		dwrq->flags &= ~IW_ENCODE_DISABLED;
	} else {
		dwrq->flags |= IW_ENCODE_DISABLED;
	}

	memset(extra, 0, 16);

	if (!index) {
		/* Handle current key request	*/
		if ((adapter->WepKey[adapter->CurrentWepKeyIndex].KeyLength) &&
				(adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled)) {
			index = adapter->WepKey[adapter->CurrentWepKeyIndex].KeyIndex;
			memcpy(extra, adapter->WepKey[index].KeyMaterial,
					adapter->WepKey[index].KeyLength);
			dwrq->length = adapter->WepKey[index].KeyLength;
			/* return current key */
			dwrq->flags |= (index + 1);
			/* return WEP enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
#ifdef WPA
		} else if ((adapter->SecInfo.WPAEnabled)
#ifdef WPA2
		|| (adapter->SecInfo.WPA2Enabled)
#endif
		) {
			/* return WPA enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
#endif
		} else {
			dwrq->flags |= IW_ENCODE_DISABLED;
		}
	} else {
		/* Handle specific key requests */
		index--;
		if (adapter->WepKey[index].KeyLength) {
			memcpy(extra, adapter->WepKey[index].KeyMaterial,
					adapter->WepKey[index].KeyLength);
			dwrq->length = adapter->WepKey[index].KeyLength;
			/* return current key */
			dwrq->flags |= (index + 1);
			/* return WEP enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
#ifdef WPA
		} else if ((adapter->SecInfo.WPAEnabled)
#ifdef WPA2
		|| (adapter->SecInfo.WPA2Enabled)
#endif
		) {
			/* return WPA enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
#endif
		} else {
			dwrq->flags |= IW_ENCODE_DISABLED;
		}
	}

	dwrq->flags |= IW_ENCODE_NOKEY;

	PRINTM(INFO, "Key:%02x:%02x:%02x:%02x:%02x:%02x KeyLen=%d\n",
			extra[0], extra[1], extra[2],
			extra[3], extra[4], extra[5], dwrq->length);

#ifdef WPA
	if (adapter->EncryptionStatus == Wlan802_11Encryption2Enabled
			&& !dwrq->length) {
		dwrq->length = MAX_KEY_SIZE;
	}
#endif

	PRINTM(INFO, "Return flags=0x%x\n", dwrq->flags);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get TX Power
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_txpow(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCGIWTXPOW: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_RF_TX_POWER,
			HostCmd_ACT_TX_POWER_OPT_GET,
			HostCmd_OPTION_WAITFORRSP,
		      	0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

  	PRINTM(INFO, "TXPOWER GET %d dbm.\n", Adapter->TxPowerLevel);
	vwrq->value = Adapter->TxPowerLevel;
	vwrq->fixed = 1;
	if (Adapter->RadioOn) {
		vwrq->disabled = 0;
		vwrq->flags = IW_TXPOW_DBM;
	} else {
		vwrq->disabled = 1;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Set TX Retry Count
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_retry(struct net_device *dev, struct iw_request_info *info,
	       struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private   *priv = dev->priv;
	wlan_adapter   *adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWRETRY: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (vwrq->flags == IW_RETRY_LIMIT) {
		/* The MAC has a 4-bit Total_Tx_Count register
		   Total_Tx_Count = 1 + Tx_Retry_Count */
#define TX_RETRY_MIN 0
#define TX_RETRY_MAX 14
		if (vwrq->value < TX_RETRY_MIN || vwrq->value > TX_RETRY_MAX)
			return -EINVAL;

		/* Adding 1 to convert retry count to try count */
		adapter->TxRetryCount = vwrq->value + 1;

		ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB, 
				0, HostCmd_OPTION_WAITFORRSP,
				OID_802_11_TX_RETRYCOUNT,
				NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	} else {
		return -EOPNOTSUPP;
	}

	LEAVE();
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get TX Retry Count
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_retry(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private   *priv = dev->priv;
	wlan_adapter   *adapter = priv->adapter;

	ENTER();

	vwrq->disabled = 0;
	if (!vwrq->flags) {
		vwrq->flags = IW_RETRY_LIMIT;
		/* Subtract 1 to convert try count to retry count */
		vwrq->value = adapter->TxRetryCount - 1;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Sort Channels
 *   
 *  @param freq 		A pointer to iw_freq structure
 *  @param num		        number of Channels
 *  @return 	   		NA
 */
static inline void sort_channels(struct iw_freq *freq, int num)
{
	int i, j;
	struct iw_freq temp;

	for (i = 0; i < num; i++)
		for (j = i + 1; j < num; j++)
			if (freq[i].i > freq[j].i) {
				temp.i = freq[i].i;
				temp.m = freq[i].m;
				
				freq[i].i = freq[j].i;
				freq[i].m = freq[j].m;
				
				freq[j].i = temp.i;	
				freq[j].m = temp.m;	
			}
}

/* data rate listing
	MULTI_BANDS:
		abg		a	b	b/g
   Infra 	G(12)		A(8)	B(4)	G(12)
   Adhoc 	A+B(12)		A(8)	B(4)	B(4)

	non-MULTI_BANDS:
		   		 	b	b/g
   Infra 	     		    	B(4)	G(12)
   Adhoc 	      		    	B(4)	B(4)
 */
/** 
 *  @brief Get Range Info
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_range(struct net_device *dev, struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{
	int			i, j;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	struct iw_range		*range = (struct iw_range *) extra;
	CHANNEL_FREQ_POWER	*cfp;
	WLAN_802_11_RATES	rates;

#ifdef ENABLE_802_11D
	BOOLEAN		flag = FALSE;
#endif

	ENTER();

	dwrq->length = sizeof(struct iw_range);
	memset(range, 0, sizeof(struct iw_range));

	range->min_nwid = 0;
	range->max_nwid = 0;

	memset(rates, 0, sizeof(rates));
	range->num_bitrates = get_active_data_rates(Adapter, rates);

	for (i=0; i < MIN(range->num_bitrates,IW_MAX_BITRATES) && rates[i]; i++) {
		range->bitrate[i] = (rates[i] & 0x7f) * 500000;
	}
	range->num_bitrates = i;
	PRINTM(INFO, "IW_MAX_BITRATES=%d num_bitrates=%d\n",IW_MAX_BITRATES,range->num_bitrates);

	range->num_frequency = 0;
#ifdef ENABLE_802_11D
	if( wlan_get_state_11d( priv) == ENABLE_11D &&
			Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		u8 chan_no;
		u8 band;
		
		parsed_region_chan_11d_t *parsed_region_chan = &Adapter->parsed_region_chan;

		if( parsed_region_chan == NULL ) {
			PRINTM(INFO, "11D:parsed_region_chan is NULL\n");
			LEAVE();
			return 0;
		}
		band = parsed_region_chan->band;
		PRINTM(INFO, "band=%d NoOfChan=%d\n", band, parsed_region_chan->NoOfChan);

		for (i = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
				&& (i < parsed_region_chan->NoOfChan); i++) {
			chan_no = parsed_region_chan->chanPwr[i].chan;
			PRINTM(INFO, "chan_no=%d\n", chan_no );
			range->freq[range->num_frequency].i = (long) chan_no;
			range->freq[range->num_frequency].m = 
					(long) chan_2_freq(chan_no, band) * 100000;
			range->freq[range->num_frequency].e = 1;
			range->num_frequency++;
		}
		flag = TRUE;
	}
	if ( !flag )
#endif
	{
		for (j = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
			&& (j < sizeof(Adapter->region_channel) 
				/ sizeof(Adapter->region_channel[0]));
				j++) {
			cfp = Adapter->region_channel[j].CFP;
			for (i = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
				&& Adapter->region_channel[j].Valid 
				&& cfp 
				&& (i < Adapter->region_channel[j].NrCFP);
					i++) {
				range->freq[range->num_frequency].i = (long) cfp->Channel;
				range->freq[range->num_frequency].m = (long) cfp->Freq * 100000;
				range->freq[range->num_frequency].e = 1;
				cfp++;
				range->num_frequency++;
			}
		}
	}

	PRINTM(INFO, "IW_MAX_FREQUENCIES=%d num_frequency=%d\n",IW_MAX_FREQUENCIES,
							range->num_frequency);

	range->num_channels = range->num_frequency;
	
	sort_channels(&range->freq[0], range->num_frequency);
	
	/*
	 * Set an indication of the max TCP throughput in bit/s that we can
	 * expect using this interface 
	 */
	if (i > 2)
		range->throughput = 5000 * 1000;
	else
		range->throughput = 1500 * 1000;

	range->min_rts = MRVDRV_RTS_MIN_VALUE;
	range->max_rts = MRVDRV_RTS_MAX_VALUE;
	range->min_frag = MRVDRV_FRAG_MIN_VALUE;
	range->max_frag = MRVDRV_FRAG_MAX_VALUE;

	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;
	range->num_encoding_sizes = 2;
	range->max_encoding_tokens = 4;

	range->min_pmp = 1000000;
	range->max_pmp = 120000000;	
	range->min_pmt = 1000;		
	range->max_pmt = 1000000;	
	range->pmp_flags = IW_POWER_PERIOD;
	range->pmt_flags = IW_POWER_TIMEOUT;
	range->pm_capa = IW_POWER_PERIOD | IW_POWER_TIMEOUT | IW_POWER_ALL_R;

	/*
	 * Minimum version we recommend 
	 */
	range->we_version_source = 15;

	/*
	 * Version we are compiled with 
	 */
	range->we_version_compiled = WIRELESS_EXT;

	range->retry_capa = IW_RETRY_LIMIT;
	range->retry_flags = IW_RETRY_LIMIT | IW_RETRY_MAX;

	range->min_retry = TX_RETRY_MIN;
	range->max_retry = TX_RETRY_MAX;

	/*
	 * Set the qual, level and noise range values 
	 */
	/*
	 * need to put the right values here 
	 */
	range->max_qual.qual = 10;
	range->max_qual.level = 0;
	range->max_qual.noise = 0;
	range->sensitivity = 0;

	/*
	 * Setup the supported power level ranges 
	 */
	memset(range->txpower, 0, sizeof(range->txpower));
	range->txpower[0] = 5;
	range->txpower[1] = 7;
	range->txpower[2] = 9;
	range->txpower[3] = 11;
	range->txpower[4] = 13;
	range->txpower[5] = 15;
	range->txpower[6] = 17;
	range->txpower[7] = 19;

	range->num_txpower = 8;
	range->txpower_capa = IW_TXPOW_DBM;
#ifdef WPA
	range->txpower_capa |= IW_TXPOW_RANGE;
#endif

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef PS_REQUIRED
/** 
 *  @brief  Set power management 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_power(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWPOWER: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	/* PS is currently supported only in Infrastructure Mode 
	 * Remove this check if it is to be supported in IBSS mode also 
	 */
#ifndef AD_HOC_PS
	if (Adapter->InfrastructureMode != Wlan802_11Infrastructure)
		return -EINVAL;
#endif

	if (vwrq->disabled) {
		Adapter->PSMode = Wlan802_11PowerModeCAM;
		if (Adapter->PSState != PS_STATE_FULL_POWER) {
			PSWakeup(priv, HostCmd_OPTION_WAITFORRSP);
		}

		return 0;
	}

	if ((vwrq->flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
		PRINTM(INFO, "Setting power timeout command is not supported\n");
		return -EINVAL;
	} else if ((vwrq->flags & IW_POWER_TYPE) == IW_POWER_PERIOD) {
		PRINTM(INFO, "Setting power period command is not supported\n");
		return -EINVAL;
	}

	if (Adapter->PSMode != Wlan802_11PowerModeCAM) {
		return WLAN_STATUS_SUCCESS;
	}
     
	Adapter->PSMode = Wlan802_11PowerModeMAX_PSP;
     
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		PSSleep(priv, HostCmd_OPTION_WAITFORRSP);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Get power management 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_power(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		mode;

	ENTER();

	mode = Adapter->PSMode;

	if ((vwrq->disabled = (mode == Wlan802_11PowerModeCAM))
			|| Adapter->MediaConnectStatus == 
					WlanMediaStateDisconnected) {
		LEAVE();
		return WLAN_STATUS_SUCCESS;
	}

	vwrq->value = 0;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif

/** 
 *  @brief  Set sensitivity threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_sens(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	ENTER();
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Get sensitivity threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_FAILURE
 */
static int wlan_get_sens(struct net_device *dev,
	      struct iw_request_info *info, struct iw_param *vwrq,
	      char *extra)
{
	ENTER();
	LEAVE();
	return WLAN_STATUS_FAILURE;
}


/** 
 *  @brief  Get Scan Data
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_scan(struct net_device *dev, struct iw_request_info *info, 
                         struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	char			*current_ev = extra;
	char			*end_buf = extra + IW_SCAN_MAX_DATA;
	wlan_private		*priv = dev->priv;
	struct iw_event		iwe;	/* Temporary buffer */
	char			*current_val;	/* For rates */
	int			i, j;
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_BSSID	*pBSSIDList;
#if defined(WPA) || (WIRELESS_EXT > 14)
	u8			buf[16+256*2], *ptr;
#endif
	CHANNEL_FREQ_POWER	*cfp;

	ENTER();

	PRINTM(INFO, "ulNumOfBSSIDs = %d\n", Adapter->ulNumOfBSSIDs);

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCGIWSCAN: IOCTLS called when station is"
               " in DeepSleep\n");
		return -EBUSY;
	}
#endif
    
#if WIRELESS_EXT > 13
	/* The old API using SIOCGIWAPLIST had a hard limit of IW_MAX_AP. 
	 * The new API using SIOCGIWSCAN is only limited by buffer size
	 * WE-14 -> WE-16 the buffer is limited to IW_SCAN_MAX_DATA bytes
	 * which is 4096 :-).
	 */ 
	for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
		if ((current_ev + MAX_SCAN_CELL_SIZE) >= end_buf) {
			PRINTM(INFO, "i=%d break out: current_ev=%p end_buf=%p "
				"MAX_SCAN_CELL_SIZE=%d\n",
				i,current_ev,end_buf,MAX_SCAN_CELL_SIZE);
			break;
		}

		pBSSIDList = &Adapter->BSSIDList[i];
		
		PRINTM(INFO, "Current Ssid: %s\n", Adapter->CurBssParams.ssid.Ssid);

#ifdef MULTI_BANDS
		cfp = find_cfp_by_band_and_channel(Adapter, 
				pBSSIDList->bss_band, pBSSIDList->Channel);
#else
		cfp = find_cfp_by_band_and_channel(Adapter, 0, 
						pBSSIDList->Channel);
#endif
		if (!cfp) {
			PRINTM(INFO, "Invalid channel number %d\n",
						pBSSIDList->Channel);
			continue;
		}

		/* First entry *MUST* be the AP MAC address */
		iwe.cmd = SIOCGIWAP;
		iwe.u.ap_addr.sa_family = ARPHRD_ETHER;
		memcpy(iwe.u.ap_addr.sa_data, &Adapter->
				BSSIDList[i].MacAddress,
				ETH_ALEN);
		PRINTM(INFO, "Before current_ev\n");
		iwe.len = IW_EV_ADDR_LEN;
		current_ev = iwe_stream_add_event(current_ev,
				end_buf, &iwe, iwe.len);
		PRINTM(INFO, "After SIOCGIWAP\n");

		//Add the ESSID 
		iwe.u.data.length = Adapter->
			BSSIDList[i].Ssid.SsidLength;

		if (iwe.u.data.length > 32)
			iwe.u.data.length = 32;

		iwe.cmd = SIOCGIWESSID;
		iwe.u.data.flags = 1;
		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		current_ev = iwe_stream_add_point(current_ev,
				end_buf, &iwe,
				Adapter->BSSIDList[i].Ssid.Ssid);
		PRINTM(INFO, "After SIOCGIWAP1\n");

		//Add mode 
		iwe.cmd = SIOCGIWMODE;
		iwe.u.mode = Adapter->BSSIDList[i].
			InfrastructureMode + 1;
		iwe.len = IW_EV_UINT_LEN;
		current_ev = iwe_stream_add_event(current_ev,
				end_buf, &iwe, iwe.len);
		PRINTM(INFO, "After SIOCGIWAP2\n");

		//frequency 
		iwe.cmd = SIOCGIWFREQ;

		iwe.u.freq.m = (long) cfp->Freq * 100000;

		iwe.u.freq.e = 1;

		iwe.len = IW_EV_FREQ_LEN;
		current_ev = iwe_stream_add_event(current_ev,
					  end_buf, &iwe, iwe.len);

		/* Add quality statistics */
		iwe.cmd = IWEVQUAL;
		iwe.u.qual.level = SCAN_RSSI(Adapter->BSSIDList[i].Rssi);
		iwe.u.qual.qual = 0;
		if (Adapter->NF[TYPE_BEACON][TYPE_NOAVG] == 0)
			iwe.u.qual.noise = MRVDRV_NF_DEFAULT_SCAN_VALUE;
		else
			iwe.u.qual.noise = CAL_NF(Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);

		if ((Adapter->InfrastructureMode == Wlan802_11IBSS) &&
				!SSIDcmp(&Adapter->CurBssParams.ssid,
					&Adapter->BSSIDList[i].Ssid)
				&& Adapter->AdhocCreate) {
			ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_RSSI,
					0, HostCmd_OPTION_WAITFORRSP,
					0, NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
			iwe.u.qual.level =
				CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE, 
					Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);
		}

		iwe.len = IW_EV_QUAL_LEN;
		current_ev = iwe_stream_add_event(current_ev,
				end_buf, &iwe, iwe.len);

		/* Add encryption capability */
		iwe.cmd = SIOCGIWENCODE;
		if (Adapter->BSSIDList[i].Privacy)
			iwe.u.data.flags = IW_ENCODE_ENABLED | IW_ENCODE_NOKEY;
		else
			iwe.u.data.flags = IW_ENCODE_DISABLED;

		iwe.u.data.length = 0;

		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		current_ev = iwe_stream_add_point(current_ev,
				end_buf, &iwe,
				Adapter->BSSIDList->Ssid.Ssid);

		/*
		 * Rate : stuffing multiple values in a single event require a
		 * bit more of magic - Jean II 
		 */
		current_val = current_ev + IW_EV_LCP_LEN;

		iwe.cmd = SIOCGIWRATE;
		/* Those two flags are ignored... */
		iwe.u.bitrate.fixed = iwe.u.bitrate.disabled = 0;
		iwe.u.bitrate.value = 0;
		/* Bit rate given in 500 kb/s units (+ 0x80) */
		for (j = 0; j < sizeof(Adapter->BSSIDList[i].SupportedRates);
				j++){
			int	rate;
			if (Adapter->BSSIDList[i].SupportedRates[j] == 0) {
				break;
			}
			rate = (Adapter->BSSIDList[i].SupportedRates[j] &
					0x7F) * 500000;
			if (rate > iwe.u.bitrate.value) {
				iwe.u.bitrate.value = rate;
			}
	
			iwe.u.bitrate.value = (Adapter->BSSIDList[i].
					SupportedRates[j] & 0x7f) * 500000;
			iwe.len = IW_EV_PARAM_LEN;
			current_ev = iwe_stream_add_value(current_ev, 
					current_val, end_buf,
					&iwe, iwe.len);

		}
		if ((Adapter->BSSIDList[i].InfrastructureMode ==
				Wlan802_11IBSS) &&
				!SSIDcmp(&Adapter->CurBssParams.ssid,
				&Adapter->BSSIDList[i].Ssid) &&
				Adapter->AdhocCreate) {
			iwe.u.bitrate.value = 22 * 500000;
		}
		iwe.len = IW_EV_PARAM_LEN;
		current_ev = iwe_stream_add_value(current_ev,
				current_val, end_buf,
				&iwe, iwe.len);

		/* Add new value to event */
		current_val = current_ev + IW_EV_LCP_LEN;

#ifdef WPA
#ifdef WPA2
		if (Adapter->BSSIDList[i].wpa2_supplicant.Wpa_ie[0] == WPA2_IE) {
			memset(&iwe, 0, sizeof(iwe));
			memset(buf, 0, sizeof(buf));
			ptr = buf;
			ptr += sprintf(ptr, "rsn_ie=");

			for (j = 0; j < Adapter->BSSIDList[i].wpa2_supplicant.Wpa_ie_len; 
					j++) {
				ptr += sprintf(ptr, "%02x",
				Adapter->BSSIDList[i].wpa2_supplicant.Wpa_ie[j]);
			}
			iwe.u.data.length = strlen(buf);

			PRINTM(INFO, "iwe.u.data.length %d\n",iwe.u.data.length);
			PRINTM(INFO, "WPA2 BUF: %s \n",buf);

			iwe.cmd = IWEVCUSTOM;
			iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
			current_ev = iwe_stream_add_point(current_ev, end_buf, 
				&iwe, buf);
		}
#endif	//WPA2
		if (Adapter->BSSIDList[i].wpa_supplicant.Wpa_ie[0] == WPA_IE) {
			memset(&iwe, 0, sizeof(iwe));
			memset(buf, 0, sizeof(buf));
			ptr = buf;
			ptr += sprintf(ptr, "wpa_ie=");

			for (j = 0; j < Adapter->BSSIDList[i].wpa_supplicant.Wpa_ie_len; 
					j++) {
				ptr += sprintf(ptr, "%02x",
				Adapter->BSSIDList[i].wpa_supplicant.Wpa_ie[j]);
			}
			iwe.u.data.length = strlen(buf);

			PRINTM(INFO, "iwe.u.data.length %d\n",iwe.u.data.length);
			PRINTM(INFO, "WPA BUF: %s \n",buf);

			iwe.cmd = IWEVCUSTOM;
			iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
			current_ev = iwe_stream_add_point(current_ev, end_buf, 
				&iwe, buf);
		}

#endif	//WPA

#ifdef WMM
		if (Adapter->BSSIDList[i].Wmm_IE[0] == WMM_IE) {
			memset(&iwe, 0, sizeof(iwe));
			memset(buf, 0, sizeof(buf));
			ptr = buf;
			ptr += sprintf(ptr, "wmm_ie=");

			for (j = 0; j < Adapter->BSSIDList[i].Wmm_ie_len; j++) {
				ptr += sprintf(ptr, "%02x",
				Adapter->BSSIDList[i].Wmm_IE[j]);
			}
			iwe.u.data.length = strlen(buf);

			PRINTM(INFO, "iwe.u.data.length %d\n",iwe.u.data.length);
			PRINTM(INFO, "WMM BUF: %s \n",buf);

			iwe.cmd = IWEVCUSTOM;
			iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
			current_ev = iwe_stream_add_point(current_ev, end_buf, 
				&iwe, buf);
		}
#endif	//WMM

#if WIRELESS_EXT > 14
#ifdef MULTI_BANDS
		memset(&iwe, 0, sizeof(iwe));
		memset(buf, 0, sizeof(buf));
		ptr = buf;
		ptr += sprintf(ptr, "band=");

		memset(&iwe, 0, sizeof(iwe));
		
		if (Adapter->BSSIDList[i].bss_band == BAND_A) {
			ptr += sprintf(ptr, "a");
		} else {
			ptr += sprintf(ptr, "bg");
		}
		
		iwe.u.data.length = strlen(buf);

		PRINTM(INFO, "iwe.u.data.length %d\n",iwe.u.data.length);
		PRINTM(INFO, "BUF: %s \n",buf);

		iwe.cmd = IWEVCUSTOM;
		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		current_ev = iwe_stream_add_point(current_ev, end_buf, 
                                          &iwe, buf);
#endif

		if(Adapter->BSSIDList[i].extra_ie != 0){
            		memset(&iwe, 0, sizeof(iwe));
            		memset(buf, 0, sizeof(buf));
            		ptr = buf;
            		ptr += sprintf(ptr, "extra_ie");
            		iwe.u.data.length = strlen(buf);

            		PRINTM(INFO, "iwe.u.data.length %d\n",iwe.u.data.length);
            		PRINTM(INFO, "BUF: %s \n",buf);
            
            		iwe.cmd = IWEVCUSTOM;
            		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
            		current_ev = iwe_stream_add_point(current_ev, end_buf, 
                                              &iwe, buf);
        	}

#ifdef TIMESTAMP_SCAN_RESULTS
        	/*
        	**
        	** Add the TSF/Timestamp of the received scan as a custom field
        	**
        	*/
		memset(&iwe, 0, sizeof(iwe));
		memset(buf, 0, sizeof(buf));
		ptr = buf;
        	ptr += sprintf(ptr, "timestamp=");
        	for (j = 0; j < sizeof(Adapter->BSSIDList[i].TimeStamp); j++)
        	{
            		ptr += sprintf(ptr, "%02x", Adapter->BSSIDList[i].TimeStamp[j]);
        	}
		memset(&iwe, 0, sizeof(iwe));
		iwe.u.data.length = strlen(buf);
        
		PRINTM(INFO, "timestamp scan buf: %s\n", buf);
        
		iwe.cmd = IWEVCUSTOM;
		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		current_ev = iwe_stream_add_point(current_ev, end_buf, 
                                          &iwe, buf);
#endif
#endif
        
		current_val = current_ev + IW_EV_LCP_LEN;
		/*
		 * Check if we added any event 
		 */
		if ((current_val - current_ev) > IW_EV_LCP_LEN)
			current_ev = current_val;
	}
    
	dwrq->length = (current_ev - extra);
	dwrq->flags = 0;
    
	LEAVE();
#endif
	return WLAN_STATUS_SUCCESS;
}


#ifdef IWGENIE_SUPPORT
#if (WIRELESS_EXT >= 18)   
/** 
 *  @brief  Set IE 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_gen_ie(struct net_device *dev, 
                           struct iw_request_info *info, 
                           struct iw_param *vwrq, 
                           char *extra)
{
    /*
    ** Functionality to be added later
    */
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Get IE 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_gen_ie(struct net_device *dev, 
                           struct iw_request_info *info, 
                           struct iw_point *dwrq,
                           char *extra)
{
    /*
    ** Functionality to be added later
    */
    return WLAN_STATUS_SUCCESS;
}
#else

/** 
 *  @brief  Append/Reset IE buffer. 
 *   
 *  pass an opaque block of data, expected to be IEEE TLVs, to the driver 
 *  for eventual passthrough to the firmware in an associate/join (and potentially start) command.
 *  Data is appended to an existing buffer and then wrapped in a passthrough
 *  TLV in the command API to the firmware.  The firmware treats the data
 *  as a transparent passthrough to the transmitted management frame.
 *  @param dev                  A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure	
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_gen_ie_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter			*Adapter = priv->adapter;
    	int ret = WLAN_STATUS_SUCCESS;
    
    	/* If the passed length is zero, reset the buffer */
    	if (wrq->u.data.length == 0)
    	{
        	Adapter->genIeBufferLen = 0;
    	}
    	else
    	{
        	/* Verify that the passed length is not larger than the available 
        	**   space remaining in the buffer
        	*/
        	if (wrq->u.data.length < 
            		(sizeof(Adapter->genIeBuffer) - Adapter->genIeBufferLen))
        	{
            		/* Append the passed data to the end of the genIeBuffer */
            		copy_from_user(Adapter->genIeBuffer + Adapter->genIeBufferLen,wrq->u.data.pointer,
				       	wrq->u.data.length);
            
            		/* Increment the stored buffer length by the size passed */
            		Adapter->genIeBufferLen += wrq->u.data.length;
        	}
        	else
        	{
            		/* Passed data does not fit in the remaining buffer space */
            		ret = WLAN_STATUS_FAILURE;
        	}
    	}

    	/* Return 0 for success, -1 for error case */
    	return ret;
}


/** 
 *  @brief  Get IE buffer from driver 
 *   
 *  used to pass an opaque block of data, expected to be
 *  IEEE TLVs, back to the application.  Currently the data block passed
 *  back to the application is the saved association response retrieved 
 *  from the firmware.
 *  @param dev                  A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure	
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_gen_ie_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter			*Adapter = priv->adapter;
    	WLAN_802_11_ASSOCIATION_INFORMATION* pAssocInfo 
        	= (WLAN_802_11_ASSOCIATION_INFORMATION*)Adapter->AssocInfoBuffer;
    	int copySize; 
    
    	/*
    	** Set the amount to copy back to the application as the minimum of the 
    	**   available data or the buffer provided by the application
    	*/
    	copySize = MIN(pAssocInfo->ResponseIELength, wrq->u.data.length);

    	if (copySize)
    	{
        	/*
        	** If the copy size is non-zero, copy the IEEE TLVs in the association
        	**   response back to the application
        	*/
       		copy_to_user(wrq->u.data.pointer,(u8*)pAssocInfo + pAssocInfo->OffsetResponseIEs, 
               	copySize);
    	}

    	/* Returned copy length */
    	wrq->u.data.length = copySize;

 	/* No error on return */
    	return WLAN_STATUS_SUCCESS;
}
#endif
#endif

#ifdef REASSOC_CMD_SUPPORT
/** 
 *  @brief  Set Reassociate Info
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param wrq 		        A pointer to iwreq structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_reassociate_ioctl(struct net_device *dev, struct iwreq *wrq)
{
    	wlan_private    *priv    = dev->priv;
	wlan_adapter    *Adapter = priv->adapter;
    	wlan_ioctl_reassociation_info reassocInfo;
    	u8 zeroMac[] = {0, 0, 0, 0, 0, 0};
    	struct sockaddr bssidSockAddr;
    	struct iw_point dwrq;

    	int attemptBSSID = 0;
    	int attemptSSID  = 0;
    	int retcode      = 0;

    	ENTER();

    	Adapter->reassocAttempt = TRUE;

    	/* Fail if the data passed is not what we expect */
    	if (wrq->u.data.length != sizeof(reassocInfo))
    	{
        	retcode = -EINVAL;
    	}
    	else if (copy_from_user(&reassocInfo, wrq->u.data.pointer,
                            sizeof(reassocInfo)) != 0)
    	{
        	/* copy_from_user failed  */
        	PRINTM(INFO, "Reassoc: copy from user failed\n");
        	retcode = -EFAULT;
    	}
    	else
    	{
        	/*
        	**  Copy the passed CurrentBSSID into the driver adapter structure.
        	**    This is later used in a TLV in the associate command to 
        	**    trigger the reassociation in firmware.
        	*/
        	memcpy(Adapter->reassocCurrentAp, reassocInfo.CurrentBSSID,
               	sizeof(Adapter->reassocCurrentAp));

        	/* Check if the desired BSSID is non-zero */
        	if (memcmp(reassocInfo.DesiredBSSID, zeroMac, sizeof(zeroMac)) != 0)
        	{
            		/* Set a flag for later processing */
            		attemptBSSID = TRUE;
        	}

        	/* Check if the desired SSID is set */
        	if (reassocInfo.DesiredSSID[0])
        	{
            		/* Make sure there is a null at the end of the SSID string */
            		reassocInfo.DesiredSSID[sizeof(reassocInfo.DesiredSSID) - 1] = 0;

            		/* Set a flag for later processing */
            		attemptSSID = TRUE;
        	}

        	/* 
        	**  Debug prints
        	*/
        	PRINTM(INFO, "Reassoc: BSSID(%d), SSID(%d)\n",attemptBSSID, attemptSSID);
        	PRINTM(INFO, "Reassoc: CBSSID(%02x:%02x:%02x:%02x:%02x:%02x)\n",
                	reassocInfo.CurrentBSSID[0],
                	reassocInfo.CurrentBSSID[1],
                	reassocInfo.CurrentBSSID[2],
                	reassocInfo.CurrentBSSID[3],
                	reassocInfo.CurrentBSSID[4],
                	reassocInfo.CurrentBSSID[5]);
        	PRINTM(INFO, "Reassoc: DBSSID(%02x:%02x:%02x:%02x:%02x:%02x)\n",
                	reassocInfo.DesiredBSSID[0],
                	reassocInfo.DesiredBSSID[1],
                	reassocInfo.DesiredBSSID[2],
                	reassocInfo.DesiredBSSID[3],
                	reassocInfo.DesiredBSSID[4],
                	reassocInfo.DesiredBSSID[5]);
        	PRINTM(INFO, "Reassoc: DSSID = %s\n", reassocInfo.DesiredSSID);


        	/*
        	**  If a BSSID was provided in the reassociation request, attempt
        	**    to reassociate to it first 
       	 	*/
        	if (attemptBSSID)
        	{
            		memset(bssidSockAddr.sa_data, 0x00, sizeof(bssidSockAddr.sa_data));
            		memcpy(bssidSockAddr.sa_data, reassocInfo.DesiredBSSID,
                   	sizeof(bssidSockAddr.sa_data));
            		bssidSockAddr.sa_family = ARPHRD_ETHER;

            		/* Attempt to reassociate to the BSSID passed */        
            		retcode = wlan_set_wap(dev, NULL, &bssidSockAddr, NULL);
        	}

        	/*
        	**  If a SSID was provided, and either the BSSID failed or a BSSID
        	**    was not provided, attemtp to reassociate to the SSID
        	*/
        	if (attemptSSID && (retcode != 0 || !attemptBSSID))
        	{
            		reassocInfo.DesiredSSID[sizeof(reassocInfo.DesiredSSID) - 1] = 0;
            		dwrq.length = strlen(reassocInfo.DesiredSSID) + 1;
            		dwrq.flags = 1; /* set for a specific association */

            		/* Attempt to reassociate to the SSID passed */        
            		retcode = wlan_set_essid(dev, NULL, &dwrq,
                                     reassocInfo.DesiredSSID);
        	}


        	/* Reset the adapter control flag and current AP to zero */
        	memset(Adapter->reassocCurrentAp, 0x00, 
               	sizeof(Adapter->reassocCurrentAp));
        	Adapter->reassocAttempt = FALSE;
    	}

    	return retcode;
}
#endif

/*
 * iwconfig settable callbacks 
 */
static const iw_handler wlan_handler[] = {
    (iw_handler) wlan_config_commit,        /* SIOCSIWCOMMIT */
    (iw_handler) wlan_get_name,             /* SIOCGIWNAME */
    (iw_handler) NULL,                      /* SIOCSIWNWID */
    (iw_handler) NULL,                      /* SIOCGIWNWID */
    (iw_handler) wlan_set_freq,             /* SIOCSIWFREQ */
    (iw_handler) wlan_get_freq,             /* SIOCGIWFREQ */
    (iw_handler) wlan_set_mode,             /* SIOCSIWMODE */
    (iw_handler) wlan_get_mode,             /* SIOCGIWMODE */
    (iw_handler) wlan_set_sens,             /* SIOCSIWSENS */
    (iw_handler) wlan_get_sens,             /* SIOCGIWSENS */
    (iw_handler) NULL,                      /* SIOCSIWRANGE */
    (iw_handler) wlan_get_range,            /* SIOCGIWRANGE */
    (iw_handler) NULL,                      /* SIOCSIWPRIV */
    (iw_handler) NULL,                      /* SIOCGIWPRIV */
    (iw_handler) NULL,                      /* SIOCSIWSTATS */
    (iw_handler) NULL,                      /* SIOCGIWSTATS */
#if WIRELESS_EXT > 15
    iw_handler_set_spy,                     /* SIOCSIWSPY */
    iw_handler_get_spy,                     /* SIOCGIWSPY */
    iw_handler_set_thrspy,                  /* SIOCSIWTHRSPY */
    iw_handler_get_thrspy,                  /* SIOCGIWTHRSPY */
#else  /* WIRELESS_EXT > 15 */
#ifdef WIRELESS_SPY
    (iw_handler) wlan_set_spy,              /* SIOCSIWSPY */
    (iw_handler) wlan_get_spy,              /* SIOCGIWSPY */
#else  /* WIRELESS_SPY */
    (iw_handler) NULL,                      /* SIOCSIWSPY */
    (iw_handler) NULL,                      /* SIOCGIWSPY */
#endif /* WIRELESS_SPY */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
#endif /* WIRELESS_EXT > 15 */
    (iw_handler) wlan_set_wap,              /* SIOCSIWAP */
    (iw_handler) wlan_get_wap,              /* SIOCGIWAP */
    (iw_handler) NULL,                      /* -- hole -- */
    //(iw_handler) wlan_get_aplist,         /* SIOCGIWAPLIST */
    NULL,                                   /* SIOCGIWAPLIST */
#if WIRELESS_EXT > 13
    (iw_handler) wlan_set_scan,             /* SIOCSIWSCAN */
    (iw_handler) wlan_get_scan,             /* SIOCGIWSCAN */
#else  /* WIRELESS_EXT > 13 */
    (iw_handler) NULL,                      /* SIOCSIWSCAN */
    (iw_handler) NULL,                      /* SIOCGIWSCAN */
#endif /* WIRELESS_EXT > 13 */
    (iw_handler) wlan_set_essid,            /* SIOCSIWESSID */
    (iw_handler) wlan_get_essid,            /* SIOCGIWESSID */
    (iw_handler) wlan_set_nick,             /* SIOCSIWNICKN */
    (iw_handler) wlan_get_nick,             /* SIOCGIWNICKN */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) wlan_set_rate,             /* SIOCSIWRATE */
    (iw_handler) wlan_get_rate,             /* SIOCGIWRATE */
    (iw_handler) wlan_set_rts,              /* SIOCSIWRTS */
    (iw_handler) wlan_get_rts,              /* SIOCGIWRTS */
    (iw_handler) wlan_set_frag,             /* SIOCSIWFRAG */
    (iw_handler) wlan_get_frag,             /* SIOCGIWFRAG */
    (iw_handler) wlan_set_txpow,            /* SIOCSIWTXPOW */
    (iw_handler) wlan_get_txpow,            /* SIOCGIWTXPOW */
    (iw_handler) wlan_set_retry,            /* SIOCSIWRETRY */
    (iw_handler) wlan_get_retry,            /* SIOCGIWRETRY */
#ifdef WPA                                  
    (iw_handler) NULL,                      /* -- hole -- */
#else  /* WPA */
    (iw_handler) wlan_set_encode,           /* SIOCSIWENCODE */
#endif                                      
    (iw_handler) wlan_get_encode,           /* SIOCGIWENCODE */
#ifdef PS_REQUIRED                          
    (iw_handler) wlan_set_power,            /* SIOCSIWPOWER */
    (iw_handler) wlan_get_power,            /* SIOCGIWPOWER */
#endif /* PS_REQUIRED */                    
#if (WIRELESS_EXT >= 18)                    
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
#if defined(IWGENIE_SUPPORT)                
    (iw_handler) wlan_set_gen_ie,           /* SIOCSIWGENIE */
    (iw_handler) wlan_get_gen_ie,           /* SIOCGIWGENIE */
#else /* IWGENIE_SUPPORT */                 
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
#endif /* IWGENIE_SUPPORT */
#endif /* WIRELESSS_EXT >= 18 */
};

/*
 * iwpriv settable callbacks 
 */

static const iw_handler wlan_private_handler[] = {
	NULL,			/* SIOCIWFIRSTPRIV */
};

static const struct iw_priv_args wlan_private_args[] = {
	/*
	 * { cmd, set_args, get_args, name } 
	 */
	{
	WLANEXTSCAN,
	IW_PRIV_TYPE_INT,
	IW_PRIV_TYPE_CHAR | 2,
	"extscan"
	},

	{
	WLANCISDUMP,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE | 512,
	"getcis"
	},

     	{
	WLANSCAN_TYPE,
	IW_PRIV_TYPE_CHAR | 8,
	IW_PRIV_TYPE_CHAR | 8,
	"scantype"
	},

	{ 
	WLAN_SETONEINT_GETONEINT,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	""
	},
	{ 
	WLANNF, 
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"getNF"
	},
	{ 
	WLANRSSI, 
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"getRSSI"
	},
#ifdef THROUGHPUT_TEST
	{ 
	WLANTHRUPUT, 
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"thruput"
	},
#endif /* THROUGHPUT_TEST */

#ifdef WMM
	{ 
	WLANWMM_ENABLE,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"wmm"
	},
#endif /* WMM */

#ifdef BG_SCAN 
	{ 
	WLANBGSCAN, 
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"bgscan"
	},
#endif /* BG_SCAN */

#ifdef BULVERDE_SDIO
	{ 
	WLANSDIOCLOCK,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"sdioclock"
	},
#endif /* BULVERDE_SDIO */

#ifdef ENABLE_802_11D
	{
	WLANENABLE11D,		
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"enable11d"
	},
#endif

#ifdef ADHOC_GRATE
	{ 
	WLANADHOCGRATE,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"adhocgrate"
	},
#endif

#ifdef ENABLE_802_11H
	{
	WLAN_11H_ENABLE,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"enable11h"
	},

	{
	WLAN_11H_SETLOCALPOWER,		
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"setpowercons"
	},
#endif
#ifdef WMM_UAPSD
	{
	WLAN_WMM_QOSINFO,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"wmm_qosinfo"
	},
#endif
#ifdef PS_REQUIRED
	{
	WLAN_LISTENINTRVL,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"lolisteninter"
	},
#ifdef FW_WAKEUP_METHOD
	{
	WLAN_FW_WAKEUP_METHOD,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"fwwakeupmethod"
	},
#endif
#endif

	{
	WLAN_TXCONTROL,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"txcontrol"
	},
#ifdef ATIMGEN			
	{
	WLANATIMGEN,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"atimgen"
	},
#endif

#ifdef WMM_UAPSD
	{
	WLANNULLGEN,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"uapsdnullgen"
	},
#endif
	
	/* Using iwpriv sub-command feature */
	{ 
	WLAN_SETONEINT_GETNONE,  /* IOCTL: 24 */
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE, 
	""
	},
	{ 
	WLAN_SUBCMD_SETRXANTENNA, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setrxant"
	},
	{ 
	WLAN_SUBCMD_SETTXANTENNA, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE, 
	"settxant"
	},
	{
	WLANSETAUTHALG,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"authalgs",
	},
	{
	WLANSET8021XAUTHALG,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"8021xauthalgs",
	},
	{
	WLANSETENCRYPTIONMODE,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"encryptionmode",
	},
	{ 
	WLANSETREGION,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setregioncode"
	},
	{ 
	WLAN_SET_LISTEN_INTERVAL,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setlisteninter"
	},
#ifdef PS_REQUIRED
	{ 
	WLAN_SET_MULTIPLE_DTIM,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setmultipledtim"
	},
#endif
	{ 
	WLAN_SET_ATIM_WINDOW,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"atimwindow"
	},
	{ 
	WLANSETBCNAVG, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setbcnavg"
	},
	{ 
	WLANSETDATAAVG, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setdataavg"
	},
	{ 
	WLAN_SETNONE_GETONEINT, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	""
	},
	{ 
	WLANGETREGION,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getregioncode"
	},
	{ 
	WLAN_GET_LISTEN_INTERVAL,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getlisteninter"
	},
	{ 
	WLAN_GET_MULTIPLE_DTIM,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getmultipledtim"
	},
	{ 
	WLANGETBCNAVG,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getbcnavg"
	},
	{ 
	WLAN_SETNONE_GETTWELVE_CHAR,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12,
	""
	},
	{ 
	WLAN_SUBCMD_GETRXANTENNA, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12, 
	"getrxant"
	},
	{ 
	WLAN_SUBCMD_SETTXANTENNA, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12, 
	"gettxant"
	},
#ifdef DEEP_SLEEP
	{
	WLANDEEPSLEEP,
	IW_PRIV_TYPE_CHAR | 1, 
	IW_PRIV_TYPE_CHAR | 6, 
	"deepsleep"
	},
#endif

#ifdef HOST_SLEEP
	{
	WLANHOSTSLEEPCFG,
	IW_PRIV_TYPE_CHAR | 15,
	IW_PRIV_TYPE_NONE,
	"hostsleepcfg"
	},
#endif

	{
	WLAN_SETNONE_GETNONE,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	""
	},
	{
	WLANDEAUTH,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"deauth"
	},
	{
	WLANADHOCSTOP,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"adhocstop"
	},
	{
	WLANRADIOON,		
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"radioon"
	},
	{
	WLANRADIOOFF,		
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"radiooff"
	},
#ifdef ADHOCAES
	{
	WLANREMOVEADHOCAES,		
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"rmaeskey"
	},
#endif
#ifdef CIPHER_TEST
	{
	WLANCIPHERTEST,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"cipher_test"
	},
#endif
	{
	WLAN_SETNONE_GETBYTE,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE,
	""
	},
#ifdef REASSOCIATION
	{
	WLANREASSOCIATIONAUTO,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE,
	"reasso-on"
	},
	{
	WLANREASSOCIATIONUSER,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE,
	"reasso-off"
	},
#endif /* REASSOCIATION */
	{
	WLANWLANIDLEON,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE,
	"wlanidle-on"
	},
	{
	WLANWLANIDLEOFF,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE,
	"wlanidle-off"
	},
	{
	WLAN_SET64CHAR_GET64CHAR,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	""
	},
	{
	WLANSLEEPPARAMS,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"sleepparams"
	},

#ifdef ENABLE_802_11H
	{
	WLAN_11H_REQUESTTPC,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"requesttpc"
	},
	{
	WLAN_11H_SETPOWERCAP,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"powercap"
	},
#endif
#ifdef ENABLE_MEAS
	{
	WLAN_MEASREQ,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"measreq"
	},
#endif
#ifdef BCA
	{
	WLAN_BCA_TIMESHARE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"bca-ts"
	},
#endif
	{
	WLANSCAN_MODE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"scanmode"
	},
	{ 
	WLAN_GET_ADHOC_STATUS,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"getadhocstatus"
	},
#if defined(IWGENIE_SUPPORT) && (WIRELESS_EXT < 18)
	{
    WLAN_SET_GEN_IE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"setgenie"
	},
    {
	WLAN_GET_GEN_IE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"getgenie"
	},
#endif
#ifdef REASSOC_CMD_SUPPORT
    {
    WLAN_REASSOCIATE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"reassociate"
	},
#endif
	{
	WLAN_SETWORDCHAR_GETNONE,
	IW_PRIV_TYPE_CHAR | 32,
	IW_PRIV_TYPE_NONE,
	""
	},
#ifdef ADHOCAES
	{
	WLANSETADHOCAES,
	IW_PRIV_TYPE_CHAR | 32,
	IW_PRIV_TYPE_NONE,
	"setaeskey"
	},
#endif /* end of ADHOCAES */ 
	{
	WLAN_SETNONE_GETWORDCHAR,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | 128,
	""
	},
#ifdef ADHOCAES
	{
	WLANGETADHOCAES,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | 128,
	"getaeskey"
	},
#endif /* end of ADHOCAES */ 
	{
	WLANVERSION,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | 128,
	"version"
	},
	{
	WLANSETWPAIE,
	IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 24,
	IW_PRIV_TYPE_NONE,
	"setwpaie"
	},
	{ 
	WLAN_SETTENCHAR_GETNONE,
	IW_PRIV_TYPE_CHAR | 10, 
	IW_PRIV_TYPE_NONE,
	""
	},
#ifdef MULTI_BANDS
	{ 
	WLAN_SET_BAND,
	IW_PRIV_TYPE_CHAR | 10, 
	IW_PRIV_TYPE_NONE,
	"setband"
	},
	{ 
	WLAN_SET_ADHOC_CH,
	IW_PRIV_TYPE_CHAR | 10, 
	IW_PRIV_TYPE_NONE,
	"setadhocch"
	},
#endif
#ifdef ENABLE_802_11H
	{ 
	WLAN_11H_CHANSWANN,
	IW_PRIV_TYPE_CHAR | 10, 
	IW_PRIV_TYPE_NONE,
	"chanswann"
	},
#endif /* ENABLE_802_11H */
	{ 
	WLAN_SETNONE_GETTENCHAR,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 10, 
	""
	},
#ifdef MULTI_BANDS
	{ 
	WLAN_GET_BAND,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 10, 
	"getband"
	},
	{ 
	WLAN_GET_ADHOC_CH,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 10, 
	"getadhocch"
	},
#endif
	{
	WLANGETLOG,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | GETLOG_BUFSIZE,
	"getlog"
	},
        {
        WLAN_SET_GET_SIXTEEN_INT,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        ""
        },
        {
        WLAN_TPCCFG,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "tpccfg"
        },
        {
        WLAN_POWERCFG,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "powercfg"
        },
#ifdef AUTO_FREQ_CTRL
	{
	WLAN_AUTO_FREQ_SET,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"setafc"
	},
	{
	WLAN_AUTO_FREQ_GET,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"getafc"
	},
#endif
	{
	WLAN_SCANPROBES,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"scanprobes"
	},
#ifdef LED_GPIO_CTRL	
        {
        WLAN_LED_GPIO_CTRL,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "ledgpio"
        },
#endif	
	{ 
	WLAN_SLEEP_PERIOD,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"sleeppd"
	},
	{ 
	WLAN_ADAPT_RATESET,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"rateadapt"
	},

	{ 
	WLAN_INACTIVITY_TIMEOUT,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"inactivityto"
	},
	{ 
	WLANSNR, 
	IW_PRIV_TYPE_INT | 16, 
	IW_PRIV_TYPE_INT | 16, 
	"getSNR"
	},
	{ 
	WLAN_GET_RATE,
	IW_PRIV_TYPE_INT | 16, 
	IW_PRIV_TYPE_INT | 16, 
	"getrate"
	},
};

struct iw_handler_def wlan_handler_def = {
	num_standard:sizeof(wlan_handler) / sizeof(iw_handler),
	num_private:sizeof(wlan_private_handler) / sizeof(iw_handler),
	num_private_args:sizeof(wlan_private_args) /
	sizeof(struct iw_priv_args),
	standard:(iw_handler *) wlan_handler,
	private:(iw_handler *) wlan_private_handler,
	private_args:(struct iw_priv_args *) wlan_private_args,
};

#ifdef STDCMD
/** 
 *  @brief wlan stdcmd ioctl handler
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param req		        A pointer to ifreq structure
 *  @param cmd			command 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_stdcmd_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
	u8		*tempResponseBuffer;
	CmdCtrlNode	*pCmdNode;
	HostCmd_DS_GEN	gencmd, *pCmdPtr;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	u16		wait_option = 0;
	ENTER();

	/*
	 * Get a free command control node 
	 */
	if (!(pCmdNode = GetFreeCmdCtrlNode(priv))) {
		PRINTM(INFO, "Failed GetFreeCmdCtrlNode\n");
		return -ENOMEM;
	}

	if (!(tempResponseBuffer = kmalloc(3000, GFP_KERNEL))) {
		PRINTM(INFO, "ERROR: Failed to allocate response buffer!\n");
		return -ENOMEM;
	}

	wait_option |= HostCmd_OPTION_WAITFORRSP;
	
	SetCmdCtrlNode(priv, pCmdNode, 0, wait_option, NULL);

	pCmdPtr = (HostCmd_DS_GEN *) pCmdNode->BufVirtualAddr;

	/*
	 * Read to header to get the size 
	 */
	copy_from_user(&gencmd, req->ifr_data, sizeof(gencmd));

	/*
	 * Copy the whole command into the command buffer 
	 */
	copy_from_user(pCmdPtr, req->ifr_data, gencmd.Size);

	pCmdNode->pdata_buf = tempResponseBuffer;
	pCmdNode->CmdFlags |= CMD_F_STDCMD;

	pCmdPtr->SeqNum = ++priv->adapter->SeqNum;
	pCmdPtr->Result = 0;

	PRINTM(INFO, "STDCMD Command: 0x%04x Size: %d SeqNum: %d\n",
			pCmdPtr->Command, pCmdPtr->Size, pCmdPtr->SeqNum);
	HEXDUMP("Command Data", (u8 *) (pCmdPtr), MIN(32, pCmdPtr->Size));
	PRINTM(INFO, "Copying data from : (user)0x%p -> 0x%p(driver)\n",
			req->ifr_data, pCmdPtr);

	QueueCmd(Adapter, pCmdNode, TRUE);
	wake_up_interruptible(&priv->MainThread.waitQ);

	if (wait_option & HostCmd_OPTION_WAITFORRSP) {
		/* Sleep until response is generated by FW */
		wait_event_interruptible(pCmdNode->cmdwait_q, 
						pCmdNode->CmdWaitQWoken);
	}
		
	/* Copy the response back to user space */
	pCmdPtr = (HostCmd_DS_GEN *) tempResponseBuffer;

	if (copy_to_user(req->ifr_data, tempResponseBuffer, pCmdPtr->Size))
		PRINTM(INFO, "ERROR: copy_to_user failed!\n");

	kfree(tempResponseBuffer);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#endif	/* STDCMD */
/** 
 *  @brief Get SNR 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_snr(wlan_private *priv, struct iwreq *wrq)
{
  	int ret = WLAN_STATUS_SUCCESS;
  	wlan_adapter *Adapter = priv->adapter;
  	int data[4];

  	ENTER();
  	memset(data,0,sizeof(data));
  	if(wrq->u.data.length)
		copy_from_user(data,wrq->u.data.pointer,MIN(wrq->u.data.length,4) * sizeof(int));
  	if ((wrq->u.data.length == 0) || (data[0] == 0) || (data[0] == 1))
  	{ 
    		ret = PrepareAndSendCommand(priv, 
          		HostCmd_CMD_802_11_RSSI,
          		0, HostCmd_OPTION_WAITFORRSP,
          		0, NULL);

    		if (ret) 
    		{
      			LEAVE();
      			return ret;
    		}
  	}

  	if (wrq->u.data.length == 0)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_NOAVG];
    		data[1] = Adapter->SNR[TYPE_BEACON][TYPE_AVG];
      		data[2] = Adapter->SNR[TYPE_RXPD][TYPE_NOAVG];
    		data[3] = Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
    		copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 4);
    		wrq->u.data.length = 4;
  	}
  	else if (data[0] == 0)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_NOAVG];
    		copy_to_user(wrq->u.data.pointer,data,sizeof(int));
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 1)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_AVG];
    		copy_to_user(wrq->u.data.pointer,data,sizeof(int));
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 2)
  	{
      		data[0] = Adapter->SNR[TYPE_RXPD][TYPE_NOAVG];
   		copy_to_user(wrq->u.data.pointer,data,sizeof(int));
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 3)
  	{
    		data[0] = Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
    		copy_to_user(wrq->u.data.pointer,data,sizeof(int));
    		wrq->u.data.length = 1;
  	}
  	else
  	{
    		return -ENOTSUPP;
  	}
    
  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

#ifdef BG_SCAN 
/** 
 *  @brief Enable/Disable BG Scan
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param enable               TRUE-enalbe, FALSE-disable 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_bg_scan_enable(wlan_private *priv, BOOLEAN enable)
{
	int		ret;

	ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_BG_SCAN_CONFIG,
			0, HostCmd_OPTION_WAITFORRSP,
			0, &enable);
	return ret;
}
#endif /* BG_SCAN */



#ifdef THROUGHPUT_TEST
/** 
 *  @brief set up throughput test
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq               	A pointer to iwreq structure 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_thruput(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter    *Adapter = priv->adapter;
	static u32	startJiffi;
	static u32	startT, endT, TXthruput, RXthruput;
	int data = 0;

	ENTER();
	
	if (wrq->u.data.length) {
		copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
		PRINTM(INFO, "The value is %d\n", data);		

		Adapter->ThruputTest = data;
		switch(data) {
		case 0:
			endT = get_utimeofday();
			PRINTM(MSG, "StartJiffi %lu, EndJiffi %lu, Num times Tx %d,"
				       "Num time Rx %d, diff %lu\n",
					startJiffi, jiffies, Adapter->NumTransfersTx,
					Adapter->NumTransfersRx, (jiffies - startJiffi));
			TXthruput = (Adapter->NumTransfersTx * 1530 * 8) / ((endT - startT) / 1000);
			RXthruput = (Adapter->NumTransfersRx * 1530 * 8) / ((endT - startT) / 1000);
			PRINTM(MSG, "StartT=%lu.%lu.%lu, EndT=%lu.%lu.%lu, Num times Tx %d, "
				       "Num time Rx %d, diff %lu.%lu.%lu, "
					"TXthruput=%lu.%lu Mbps, RXthruput=%lu.%lu Mbps\n",
					startT/1000000,(startT%1000000)/1000,(startT%1000000)%1000,
					endT/1000000,(endT%1000000)/1000,(endT%1000000)%1000,
 					Adapter->NumTransfersTx,Adapter->NumTransfersRx,
					(endT-startT)/1000000,((endT-startT)%1000000)/1000,((endT-startT)%1000000)%1000,
					TXthruput/1000,TXthruput%1000,
					RXthruput/1000,RXthruput%1000
					);
			break;
		case 1:
		case 3:
			startJiffi = jiffies;
			startT = get_utimeofday();
			priv->adapter->NumTransfersRx = 0;
			priv->adapter->NumTransfersTx = 0;
			priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
			priv->adapter->IntCounter = 1;
			wake_up_interruptible(&priv->MainThread.waitQ);
			break;
		case 2:
			startJiffi = jiffies;
			startT = get_utimeofday();
			priv->adapter->NumTransfersRx = 0;
			priv->adapter->NumTransfersTx = 0;
			priv->adapter->IntCounter = 1;
			wake_up_interruptible(&priv->MainThread.waitQ);
			break;
		default:
			break;
		}
	}

	return 0;
}
#endif /* THROUGHPUT */

/** 
 *  @brief Get RSSI 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_rssi(wlan_private *priv, struct iwreq *wrq)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;
	int		temp;
	int data = 0;
	
	ENTER();
			
	if(wrq->u.data.length) {
		copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
		if ((data == 0) || (data == 1)) { 
			ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_RSSI,
					0,HostCmd_OPTION_WAITFORRSP,
					0, NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
		}

		switch (data) {
		case 0:
		temp = CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
					Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
				break;
		case 1:
		temp = CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_AVG],
				Adapter->NF[TYPE_BEACON][TYPE_AVG]);
				break;
		case 2:
		temp = CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_NOAVG],
					Adapter->NF[TYPE_RXPD][TYPE_NOAVG]);
				break;
		case 3:
		temp = CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE,
				Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);
				break;
		default:
			return -ENOTSUPP;
		}
		
		if (copy_to_user(wrq->u.data.pointer, 
					&temp, sizeof(int))) {
			PRINTM(INFO, "copy_to_user failed!\n");
			return -EFAULT;
		}
		wrq->u.data.length = 1;
	} else
		return -EINVAL;
 
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get NF
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_nf(wlan_private *priv, struct iwreq *wrq)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;
	int		temp;
	int data = 0;

	ENTER();

	if(wrq->u.data.length) {
		copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
		if ((data == 0) || (data == 1) ) { 
			ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_RSSI,
					0,HostCmd_OPTION_WAITFORRSP,
					0, NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
		}

		switch (data) {
			case 0:
				temp = Adapter->NF[TYPE_BEACON][TYPE_NOAVG];
				break;
			case 1:
				temp = Adapter->NF[TYPE_BEACON][TYPE_AVG];
				break;
			case 2:
				temp = Adapter->NF[TYPE_RXPD][TYPE_NOAVG];
				break;
			case 3:
				temp = Adapter->
					NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
				break;
			default:
				return -ENOTSUPP;
		}

		temp = CAL_NF(temp);
		
		PRINTM(INFO, "***temp = %d\n", temp);		
		if (copy_to_user(wrq->u.data.pointer, 
					&temp, sizeof(int))) {
			PRINTM(INFO, "copy_to_user failed!\n");
			return -EFAULT;
		}
		wrq->u.data.length = 1;
	} else
		return -EINVAL;
		
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef ADHOCAES
/** 
 *  @brief Remove AES key
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_remove_aes(wlan_private *priv)
{
	wlan_adapter	*Adapter = priv->adapter;	
#ifdef WPA
	WLAN_802_11_KEY	key;
#endif
	int 	ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if(Adapter->InfrastructureMode != Wlan802_11IBSS ||
			Adapter->MediaConnectStatus == 
					WlanMediaStateConnected)
		return -EOPNOTSUPP;
					
	Adapter->AdhocAESEnabled = FALSE;

#ifdef WPA
	memset(&key, 0, sizeof(WLAN_802_11_KEY));
	PRINTM(INFO, "WPA2: DISABLE AES_KEY\n");
	key.KeyLength = WPA_AES_KEY_LEN;
	key.KeyIndex = 0x40000000;

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_KEY_MATERIAL,
		HostCmd_ACT_SET,
		HostCmd_OPTION_WAITFORRSP,
		!(KEY_INFO_ENABLED), &key);
#endif
	
	LEAVE();
	
	return ret;
}
#endif

/** 
 *  @brief scan network with specific ssid 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_extscan_ioctl(wlan_private *priv, struct ifreq *req)
{
	WLAN_802_11_SSID Ext_Scan_SSID;
	union iwreq_data wrqu;

	ENTER();

	if (copy_from_user(&Ext_Scan_SSID, req->ifr_data,
				sizeof(Ext_Scan_SSID))) {
		PRINTM(INFO, "copy of SSID for ext scan from user failed \n");
		LEAVE();
 		return -EFAULT;
	}
	SendSpecificScan(priv, &Ext_Scan_SSID);

	memset(&wrqu, 0, sizeof(union iwreq_data));
#ifdef linux
	wireless_send_event(priv->wlan_dev.netdev, SIOCGIWSCAN, &wrqu , NULL);
#endif

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Support Rates
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_getrate_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_RATES	rates;
	int rate[16];
	int			i;

	ENTER();

	memset(rates, 0, sizeof(rates));
	memset(rate,0,sizeof(rate));
	wrq->u.data.length = get_active_data_rates(Adapter, rates);
	if(wrq->u.data.length > 16)
		wrq->u.data.length = 16;

	for (i = 0; i < wrq->u.data.length; i++)
	{
		rates[i] &= ~0x80;
		rate[i] = rates[i];
	}

	if (copy_to_user(wrq->u.data.pointer, rate, wrq->u.data.length * sizeof(int))) {
		PRINTM(INFO, "Copy to user failed\n");
		return -EFAULT;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Adhoc Status
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_adhoc_status_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	char 		status[64];
	wlan_adapter	*Adapter = priv->adapter;
	
	memset(status, 0, sizeof(status));

	switch (Adapter->InfrastructureMode) {
	case Wlan802_11IBSS:
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			if (Adapter->AdhocCreate == TRUE)
				memcpy(&status, "AdhocStarted", sizeof(status));
			else
				memcpy(&status, "AdhocJoined", sizeof(status));
		} else {
			memcpy(&status, "AdhocIdle", sizeof(status));
		}
		break;
	case Wlan802_11Infrastructure:	
		memcpy(&status, "InfraMode", sizeof(status));
		break;
	default:	
		memcpy(&status, "AutoUnknownMode", sizeof(status));
		break;
	}

	PRINTM(INFO, "Status = %s\n", status);
	wrq->u.data.length = strlen(status) + 1;

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, 
					&status, wrq->u.data.length))
			return -EFAULT;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Driver Version
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_version_ioctl(wlan_private *priv, struct ifreq *req)
{
	int	len;
	char	buf[128];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	get_version(priv->adapter, buf, sizeof(buf) - 1);

	len = strlen(buf);
	if (wrq->u.data.pointer) {
		if(copy_to_user(wrq->u.data.pointer, buf, len)){
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
		wrq->u.data.length = len;
	}

	PRINTM(INFO, "wlan version: %s\n", buf);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Read/Write adapter registers
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_regrdwr_ioctl(wlan_private *priv, struct ifreq *req)
{
	wlan_ioctl_regrdwr	regrdwr;
	wlan_offset_value	offval;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if(copy_from_user(&regrdwr, req->ifr_data,	sizeof(regrdwr))){
		PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl from user failed \n");
		LEAVE();
 		return -EFAULT;	
	}

	if(regrdwr.WhichReg == REG_EEPROM) {
		PRINTM(MSG, "Inside RDEEPROM\n");
		Adapter->pRdeeprom = (char *) kmalloc(regrdwr.NOB, GFP_KERNEL);
		if(!Adapter -> pRdeeprom)
			return -ENOMEM;
		/* +14 is for Action, Offset, and NOB in 
		 * response */
			PRINTM(INFO, "Action:%d Offset: %x NOB: %02x\n",
						regrdwr.Action, 
						regrdwr.Offset, regrdwr.NOB);

			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_EEPROM_ACCESS,
					regrdwr.Action,HostCmd_OPTION_WAITFORRSP,
					0, &regrdwr);
	
			if (ret) {
				LEAVE();
				return ret;
			}

			mdelay(10);

			/*
			 * Return the result back to the user 
			 */

			memcpy(&regrdwr.Value, Adapter->pRdeeprom, regrdwr.NOB);

			if (regrdwr.Action == HostCmd_ACT_GEN_READ) {
				if(copy_to_user(req->ifr_data, &regrdwr, 
				sizeof(regrdwr) + regrdwr.NOB - 1))
				{
					PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl to user failed \n");
					LEAVE();
 					return -EFAULT;	
				}
			}
			
			if(Adapter -> pRdeeprom)
				kfree(Adapter -> pRdeeprom);

			return WLAN_STATUS_SUCCESS;
	}

	offval.offset = regrdwr.Offset;
	offval.value = (regrdwr.Action) ? regrdwr.Value : 0x00;

	PRINTM(INFO, "RegAccess: %02x Action:%d "
			"Offset: %04x Value: %04x\n",
			regrdwr.WhichReg, regrdwr.Action, 
			offval.offset, offval.value);

	/*
	 * regrdwr.WhichReg should contain the command that
	 * corresponds to which register access is to be 
	 * performed HostCmd_CMD_MAC_REG_ACCESS 0x0019
	 * HostCmd_CMD_BBP_REG_ACCESS 0x001a 
	 * HostCmd_CMD_RF_REG_ACCESS 0x001b 
	 */
	ret = PrepareAndSendCommand(priv, regrdwr.WhichReg, 
			regrdwr.Action,HostCmd_OPTION_WAITFORRSP,
			0, &offval);

	if (ret) {
		LEAVE();
		return ret;
	}

	mdelay(10);

	/*
	 * Return the result back to the user 
	 */
	regrdwr.Value = Adapter->OffsetValue.value;
	if (regrdwr.Action == HostCmd_ACT_GEN_READ) {
		if(copy_to_user(req->ifr_data, &regrdwr, sizeof(regrdwr))){
			PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl to user failed \n");
			LEAVE();
 			return -EFAULT;	
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
		
#if (defined SD) || (defined SPI)
/** 
 *  @brief Cmd52 read/write register
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_cmd52rdwr_ioctl(wlan_private *priv, struct ifreq *req)
{
	u8 buf[7];
	u8 rw, func, dat = 0xff;
	u32 reg;

	ENTER();

	copy_from_user(buf, req->ifr_data, sizeof(buf));

	rw = buf[0];
	func = buf[1];
	reg = buf[5];
	reg = (reg << 8) + buf[4];
	reg = (reg << 8) + buf[3];
	reg = (reg << 8) + buf[2];

	if(rw != 0)
		dat = buf[6];
	
	PRINTM(INFO, "rw=%d func=%d reg=0x%08X dat=0x%02X\n",rw,func,	reg,dat);

	if(rw == 0) {
		if (sbi_read_ioreg(priv, func, reg, &dat) < 0) {
			PRINTM(INFO, "sdio_read_ioreg: reading register 0x%X failed\n"
									,reg);
			dat = 0xff;
		}
	}
	else {
		if (sbi_write_ioreg(priv, func, reg, dat) < 0) {
			PRINTM(INFO, "sdio_read_ioreg: writing register 0x%X failed\n"
									,reg);
			dat = 0xff;
		}
	}
	copy_to_user(req->ifr_data, &dat, sizeof(dat));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Cmd53 read/write register
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_cmd53rdwr_ioctl(wlan_private *priv, struct ifreq *req)
{
	return -EINVAL;
}
#endif


#ifdef CIPHER_TEST
/** 
 *  @brief cipher test
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_cipher_test(wlan_private *priv, struct iwreq *wrq)
{
  	int ret = WLAN_STATUS_SUCCESS;
  	HostCmd_DS_802_11_KEY_ENCRYPT cmd;
	ENTER();

  	memset(&cmd, 0, sizeof(cmd));
	copy_from_user(&cmd, wrq->u.data.pointer, sizeof(cmd));
	HEXDUMP("key_encrypt", (u8 *)&cmd, sizeof(cmd));

	ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_KEY_ENCRYPT,
			HostCmd_ACT_SET,
		 	HostCmd_OPTION_WAITFORRSP,
			0, &cmd);

  	copy_to_user(wrq->u.data.pointer, &cmd, sizeof(cmd));
	LEAVE();
	return ret;
}
#endif

#ifdef ADHOCAES
/** 
 *  @brief Convert ascii string to Hex integer
 *     
 *  @param d                    A pointer to integer buf
 *  @param s			A pointer to ascii string 
 *  @param dlen			the length o fascii string
 *  @return 	   	        number of integer  
 */
static int ascii2hex(u8 *d, char *s, u32 dlen)
{
	int i;
	u8 n;
    
	memset(d, 0x00, dlen);

	for (i = 0; i < dlen * 2; i++) {
		if ((s[i] >= 48) && (s[i] <= 57))
			n = s[i] - 48;
		else if ((s[i] >= 65) && (s[i] <= 70))
			n = s[i] - 55;
		else if ((s[i] >= 97) && (s[i] <= 102))
			n = s[i] - 87;
		else
			break;
		if ((i % 2) == 0)
			n = n * 16;
		d[i / 2] += n;
	}

	return i;
}

/** 
 *  @brief Set adhoc aes key
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setadhocaes_ioctl(wlan_private *priv, struct ifreq *req)
{
	u8  key_ascii[32];
	u8  key_hex[16];
	int ret = 0;
	struct iwreq *wrq = (struct iwreq *) req;
	wlan_adapter *Adapter = priv->adapter;

#ifdef WPA
	WLAN_802_11_KEY		key;
#endif

	ENTER();

	if (Adapter->InfrastructureMode != Wlan802_11IBSS)
		return -EOPNOTSUPP;

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected)
		return -EOPNOTSUPP;

	if(copy_from_user(key_ascii, wrq->u.data.pointer, sizeof(key_ascii))){
		PRINTM(INFO, "wlan_setadhocaes_ioctl copy from user failed \n");
		LEAVE();
 		return -EFAULT;	
	}
  
	Adapter->AdhocAESEnabled = TRUE;
	ascii2hex(key_hex, key_ascii, sizeof(key_hex));

	HEXDUMP("wlan_setadhocaes_ioctl", key_hex, sizeof(key_hex));

#ifdef WPA
	PRINTM(INFO, "WPA2: ENABLE AES_KEY\n");
	key.KeyLength = WPA_AES_KEY_LEN;
	key.KeyIndex = 0x40000000;
	memcpy(key.KeyMaterial, key_hex, key.KeyLength );

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_KEY_MATERIAL,
		HostCmd_ACT_SET,
	 	HostCmd_OPTION_WAITFORRSP,
 		KEY_INFO_ENABLED, &key);
#endif

	LEAVE();
	return ret;
}

/** 
 *  @brief Get adhoc aes key   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_getadhocaes_ioctl(wlan_private *priv, struct ifreq *req)
{
	u8 *tmp;
	u8 key_ascii[33];
	u8 key_hex[16];
	int i, ret = 0;
	struct iwreq *wrq = (struct iwreq *) req;
#ifdef WPA
	wlan_adapter *Adapter = priv->adapter;
	WLAN_802_11_KEY		key;
#endif

	ENTER();

	memset(key_hex, 0x00, sizeof(key_hex));

#ifdef WPA
	PRINTM(INFO, "WPA2: ENABLE AES_KEY\n");
	key.KeyLength = WPA_AES_KEY_LEN;
	key.KeyIndex = 0x40000000;
	memcpy(key.KeyMaterial, key_hex, key.KeyLength );

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_KEY_MATERIAL,
		HostCmd_ACT_GET,
		HostCmd_OPTION_WAITFORRSP,
 		KEY_INFO_ENABLED, &key);
#endif

	if (ret) {
		LEAVE();
		return ret;
	}

#ifdef WPA
	memcpy(key_hex, Adapter->aeskey.KeyParamSet.Key, sizeof(key_hex));
#endif

	HEXDUMP("wlan_getadhocaes_ioctl", key_hex, sizeof(key_hex));

	wrq->u.data.length = sizeof(key_ascii) + 1;

	memset(key_ascii, 0x00, sizeof(key_ascii));
	tmp = key_ascii;
            
	for (i = 0; i < sizeof(key_hex); i++) 
		tmp += sprintf(tmp, "%02x", key_hex[i]);

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &key_ascii, 
					sizeof(key_ascii))) {
			PRINTM(INFO, "copy_to_user failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return ret;
}
#endif

#ifdef WPA
/** 
 *  @brief Set/Get WPA IE   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setwpaie_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (wrq->u.data.length) {
		if (wrq->u.data.length > sizeof(Adapter->Wpa_ie)){
			PRINTM(INFO, "failed to copy WPA IE, too big \n");
			return -EFAULT;
		}
		if (copy_from_user(Adapter->Wpa_ie, wrq->u.data.pointer,
							wrq->u.data.length)) {
			PRINTM(INFO, "failed to copy WPA IE \n");
			return -EFAULT;
		}
		Adapter->Wpa_ie_len = wrq->u.data.length;
		PRINTM(INFO, "Set Wpa_ie_len=%d IE=%#x\n", Adapter->Wpa_ie_len, 
							Adapter->Wpa_ie[0]);
		HEXDUMP("Wpa_ie", Adapter->Wpa_ie, Adapter->Wpa_ie_len);
		if (Adapter->Wpa_ie[0] == WPA_IE)
			Adapter->SecInfo.WPAEnabled = TRUE;
#ifdef WPA2
		else if (Adapter->Wpa_ie[0] == WPA2_IE)
			Adapter->SecInfo.WPA2Enabled = TRUE;
#endif //WPA2
		else {
			Adapter->SecInfo.WPAEnabled = FALSE;
#ifdef WPA2
			Adapter->SecInfo.WPA2Enabled = FALSE;
#endif //WPA2
		}
	} else {
		memset(Adapter->Wpa_ie, 0, sizeof(Adapter->Wpa_ie));
		Adapter->Wpa_ie_len = wrq->u.data.length;
		PRINTM(INFO, "Reset Wpa_ie_len=%d IE=%#x\n", Adapter->Wpa_ie_len,
							Adapter->Wpa_ie[0]);
		Adapter->SecInfo.WPAEnabled = FALSE;
#ifdef WPA2
		Adapter->SecInfo.WPA2Enabled = FALSE;
#endif //WPA2
	}

	// enable/disable RSN in firmware if WPA is enabled/disabled
	// depending on variable Adapter->SecInfo.WPAEnabled is set or not
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_ENABLE_RSN,
			HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	LEAVE();
	return ret;
}
#endif //WPA

/** 
 *  @brief Set atim window   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_atim_window_ioctl(wlan_private *priv, struct ifreq *req)
{
	wlan_adapter	*Adapter = priv->adapter;
	struct	ifreq	*treq;

	ENTER();
	
	PRINTM(INFO, "WLAN_SET_ATIM_WINDOW\n");
	treq = (struct ifreq *)((u8 *)req + 4);
	Adapter->AtimWindow = (int)treq->ifr_data;
	Adapter->AtimWindow = MIN(Adapter->AtimWindow, 50);
	PRINTM(INFO, "AtimWindow is set to %d\n", Adapter->AtimWindow);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Region  
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setregion_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		region_code;
	struct	ifreq	*treq;

	ENTER();

	treq = (struct ifreq *)((u8 *)req + 4);
	region_code = (int) treq->ifr_data;

	if (wlan_set_region(priv, (u16) region_code))
		return -EFAULT;
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Set Listen Interval
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_listen_interval_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		interval;
	struct	ifreq	*treq;

	ENTER();
					
	treq = (struct ifreq *)((u8 *)req + 4);
	interval = (int) treq->ifr_data;
					
	priv->adapter->ListenInterval = (u16) interval;
		
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set RX antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_setrxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct	ifreq	*treq;
	PRINTM(INFO, "WLAN_SUBCMD_SETRXANTENNA\n");
	treq = (struct ifreq *)((u8 *)req + 4);
	return SetRxAntenna(priv, (int)treq->ifr_data);
}


/** 
 *  @brief Set TX antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_settxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct	ifreq	*treq;
	PRINTM(INFO, "WLAN_SUBCMD_SETTXANTENNA\n");
	treq = (struct ifreq *)((u8 *)req + 4);
	return SetTxAntenna(priv, (int)treq->ifr_data);
}

#ifdef PS_REQUIRED
/** 
 *  @brief Set multiple dtim
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_multiple_dtim_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct	ifreq	*treq;
	u32		mdtim;
	int		ret = -EINVAL;

	ENTER();
	
	treq = (struct ifreq *)((u8 *)req + 4);
	if (treq) {
		mdtim = (u32)treq->ifr_data;
		if (((mdtim >= MRVDRV_MIN_MULTIPLE_DTIM) &&
			(mdtim <= MRVDRV_MAX_MULTIPLE_DTIM))
			|| (mdtim == MRVDRV_IGNORE_MULTIPLE_DTIM)
		) {
			priv->adapter->MultipleDtim = mdtim;
			ret = WLAN_STATUS_SUCCESS;
		}
	}

	if (ret)
		PRINTM(INFO, "Invalid parameter, MultipleDtim not changed.\n");

	LEAVE();
	return ret;
}
#endif

/** 
 *  @brief Set authentication mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setauthalg_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 	alg;
	struct	ifreq	*treq;
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;
	
	ENTER();

	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		treq = (struct ifreq *)((u8 *)req + 4);
		alg = (int)treq->ifr_data;
	} else {	
		//from wpa_supplicant subcmd
		copy_from_user(&alg,wrq->u.data.pointer,sizeof(alg));
	}
	
	PRINTM(INFO, "auth alg is %#x\n",alg);

	switch (alg) {
		case AUTH_ALG_SHARED_KEY:
			Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeShared;
			break;
		case AUTH_ALG_NETWORK_EAP:
			Adapter->SecInfo.AuthenticationMode = 
						Wlan802_11AuthModeNetworkEAP;
			break;
		case AUTH_ALG_OPEN_SYSTEM:
		default:
			Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
			break;
	}
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set 802.1x authentication mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set8021xauthalg_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 		alg;
	struct	ifreq	*treq;
	struct iwreq	*wrq = (struct iwreq *) req;
	
	ENTER();
	
	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		treq = (struct ifreq *)((u8 *)req + 4);
		alg = (int)treq->ifr_data;
	} else {	
		//from wpa_supplicant subcmd
		copy_from_user(&alg,wrq->u.data.pointer,sizeof(int));
	}
	PRINTM(INFO, "802.1x auth alg is %#x\n",alg);
	priv->adapter->SecInfo.Auth1XAlg = alg;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Encryption mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setencryptionmode_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		mode;
	struct	ifreq	*treq;
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();
	
	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		treq = (struct ifreq *)((u8 *)req + 4);
		mode = (int)treq->ifr_data;
	}
	else {	
		//from wpa_supplicant subcmd
		copy_from_user(&mode,wrq->u.data.pointer,sizeof(int));
	}
	PRINTM(INFO, "encryption mode is %#x\n",mode);
	priv->adapter->SecInfo.EncryptionMode = mode;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Rx antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_getrxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		len;
	char		buf[8];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	PRINTM(INFO, "WLAN_SUBCMD_GETRXANTENNA\n");
	len = GetRxAntenna(priv, buf);

	wrq->u.data.length = len;
	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &buf, len)) {
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Tx antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_gettxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		len;
	char		buf[8];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	PRINTM(INFO, "WLAN_SUBCMD_GETTXANTENNA\n");
	len = GetTxAntenna(priv, buf);

	wrq->u.data.length = len;
	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &buf, len)) {
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef DEEP_SLEEP
/** 
 *  @brief Get/Set DeepSleep mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_deepsleep_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		ret = WLAN_STATUS_SUCCESS;
	char		status[128];
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	
#ifdef PS_REQUIRED
	// check if it is in IEEE PS mode or not
	if (Adapter->PSMode != Wlan802_11PowerModeCAM) {
		PRINTM(MSG, "Cannot process Deep Sleep command in IEEE Power"
							" Save mode.\n");
		return -EINVAL;
	}
#endif

	if (req->ifr_data[0] == '0') {
		PRINTM(INFO, "Exit Deep Sleep Mode.\n");
		sprintf(status, "setting to off ");
		SetDeepSleep(priv, FALSE);	
	} else if (req->ifr_data[0] == '1') {
		PRINTM(INFO, "Enter Deep Sleep Mode.\n");
		sprintf(status, "setting to on ");
		SetDeepSleep(priv, TRUE);
	} else if (req->ifr_data[0] == '2') {
		PRINTM(INFO, "Get Deep Sleep Mode.\n");
		if (Adapter->IsDeepSleep == TRUE) {
			sprintf(status, "on ");
		} else {
			sprintf(status, "off ");
		}
	} else {
		PRINTM(INFO, "unknown option = %d\n",req->ifr_data[0]);
		return  -EINVAL;
	}

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &status, strlen(status)))
			return -EFAULT;
		wrq->u.data.length = strlen(status);
	}

	LEAVE();
	return ret;
}
#endif

#ifdef HOST_SLEEP
/** 
 *  @brief Config hostsleep parameter
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_hostsleepcfg_ioctl(wlan_private *priv, struct iwreq *wrq) 
{
	wlan_adapter	*Adapter = priv->adapter;
	char		buf[16];
	int		ret = WLAN_STATUS_SUCCESS;
	int 		gpio, gap;
	

	memset(buf, 0, sizeof(buf));
	if (copy_from_user(buf, wrq->u.data.pointer, 
				MIN(sizeof(buf)-1,
				wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}
	buf[15] = 0;

	sscanf(buf,"%x %x %x",&Adapter->HostSleepCfgParam.conditions,
		&gpio, &gap );
	Adapter->HostSleepCfgParam.gpio = (u8)gpio;
	Adapter->HostSleepCfgParam.gap	 = (u8)gap;

	PRINTM(MSG, "conditions=0x%x gpio=%d gap=%d\n",
		Adapter->HostSleepCfgParam.conditions,
		Adapter->HostSleepCfgParam.gpio,
		Adapter->HostSleepCfgParam.gap);

	ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_HOST_SLEEP_CFG,
				0, HostCmd_OPTION_WAITFORRSP,
				0, &Adapter->HostSleepCfgParam);

	return ret;
}
#endif

#ifdef SUBSCRIBE_EVENT_CTRL
/** 
 *  @brief subscribe event
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_subscribe_event(wlan_private *priv, struct ifreq *req)
{
	EventSubscribe		*Events;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret;
	
	ENTER();

	Events = kmalloc(sizeof(EventSubscribe), GFP_KERNEL);
	if(!Events){
		PRINTM(MSG, "kmalloc no memory !!\n");
		return -ENOMEM;
		
	}
	
	if (copy_from_user(Events, req->ifr_data + SKIP_CMDNUM,
			       sizeof(EventSubscribe))) {
			PRINTM(INFO, "Copy from user failed\n");
			kfree(Events);
			return -EFAULT;
	}
	HEXDUMP("SubscribeEvents: ", req->ifr_data + SKIP_CMDNUM,
					sizeof(EventSubscribe));

	ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_SUBSCRIBE_EVENT,
				0,HostCmd_OPTION_WAITFORRSP,
			       	0, Events);

	if (Events->Action == HostCmd_ACT_GET) {
		PRINTM(INFO, "Suscribed Events = 0x%X\n", Adapter->GetEvents);
		if (copy_to_user(req->ifr_data + SKIP_CMDNUM, 
					&Adapter->GetEvents, sizeof(u16))) {
			PRINTM(INFO, "Copy to user Failed\n");
			kfree(Events);
			return -EFAULT;
		}
	}

	kfree(Events);

	LEAVE();
	return ret;
}
#endif

#ifdef BG_SCAN 
/** 
 *  @brief config BGSCAN parameter
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_bg_scan_config_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;
	u8		*buf;

	ENTER();
	
	PRINTM(INFO, "Action = %d\n", 
		(u8) *(req->ifr_data + SKIP_CMDNUM + SKIP_SIZE));

	switch((u8) *(req->ifr_data + SKIP_CMDNUM + SKIP_SIZE)) {
	case HostCmd_ACT_GEN_GET:
		buf = kmalloc(Adapter->bgScanConfigSize + SKIP_SIZE, GFP_KERNEL);
		if(!buf){
			PRINTM(MSG, "kmalloc no memory !!\n");
			return -ENOMEM;
		}
		memcpy(buf, &Adapter->bgScanConfigSize, SKIP_SIZE);
		memcpy(buf + SKIP_TYPE_SIZE, Adapter->bgScanConfig, 
						Adapter->bgScanConfigSize);

		if (copy_to_user(req->ifr_data, buf, 
					Adapter->bgScanConfigSize + SKIP_SIZE)) {
			PRINTM(INFO, "Copy from user failed\n");
			kfree(buf);
			return -EFAULT;
		}

		kfree(buf);

		break;
	case HostCmd_ACT_GEN_SET:
		Adapter->bgScanConfigSize = (u16) *(req->ifr_data + SKIP_CMDNUM);
		PRINTM(INFO, "bgscanConfigSize = %d\n", Adapter->bgScanConfigSize);
		
		if(Adapter->bgScanConfig)
			kfree(Adapter->bgScanConfig);

		if(!(Adapter->bgScanConfig = 
				kmalloc(Adapter->bgScanConfigSize, GFP_KERNEL))) {
			PRINTM(MSG, "kmalloc no memory !!\n");
			Adapter->bgScanConfigSize = 0;
			return -ENOMEM;
		}

		HEXDUMP("treq", req->ifr_data + SKIP_CMDNUM + SKIP_SIZE,
						Adapter->bgScanConfigSize);
		
		if (copy_from_user(Adapter->bgScanConfig, req->ifr_data + 
					SKIP_CMDNUM + SKIP_SIZE, 
					Adapter->bgScanConfigSize)) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		break;
	}

	LEAVE();

	return ret;
}
#endif /* BG_SCAN */

#ifdef CAL_DATA
/** 
 *  @brief Get/Set Cal data ext
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_caldata_ext_ioctl(wlan_private *priv, struct ifreq *req)
{
	int			ret = WLAN_STATUS_SUCCESS;
	u8			buf[8];
	HostCmd_DS_802_11_CAL_DATA_EXT	CalData;
	
	ENTER();
	
	memset(&CalData, 0, sizeof(HostCmd_DS_802_11_CAL_DATA_EXT));
			
	if (copy_from_user(&CalData.Action, req->ifr_data + SKIP_CMDNUM,
						sizeof(HostCmd_DS_802_11_CAL_DATA_EXT))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	HEXDUMP("Cal data ext", (u8 *)&CalData.Action, 
					sizeof(HostCmd_DS_802_11_CAL_DATA_EXT));

	PRINTM(INFO, "CalData.Action = 0x%0X\n", CalData.Action);
	
	switch (CalData.Action) {
	case WLANSET_CAL_DATA_EXT:
		ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_CAL_DATA_EXT,
					HostCmd_ACT_GEN_SET,	
					HostCmd_OPTION_WAITFORRSP,
					0, &CalData);

		if (ret) {
			LEAVE();
			return ret;
		}
					
		if (copy_to_user(req->ifr_data, &priv->adapter->caldataext,
					sizeof(HostCmd_DS_802_11_CAL_DATA_EXT) + 
					sizeof(HostCmd_DS_GEN))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		buf[0] = 0x01;
		buf[1] = 0x00;
		break;
			
	case WLANGET_CAL_DATA_EXT:
		ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_CAL_DATA_EXT,	
					HostCmd_ACT_GEN_GET,
					HostCmd_OPTION_WAITFORRSP,
					0, &CalData);

		if (ret) {
			LEAVE();
			return ret;
		}

		if (copy_to_user(req->ifr_data, &priv->adapter->caldataext,
					sizeof(HostCmd_DS_802_11_CAL_DATA_EXT) +
					sizeof(HostCmd_DS_GEN))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
			break;
	}

	LEAVE();
	return ret;
}
#endif	/* CAL_DATA */

#ifdef MULTI_BANDS
/** 
 *  @brief Set band 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_do_setband_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int		cnt, i;
	char		buf[16];
	u16		tmp = 0;
	wlan_adapter	*Adapter = priv->adapter;
				
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) 
		return -EOPNOTSUPP;
				
	memset(buf, '\0', sizeof(buf));
	if(copy_from_user(buf, wrq->u.data.pointer, 
				MIN(wrq->u.data.length, sizeof(buf)-1))){
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	cnt = strlen(buf);

	if (!cnt || cnt > 3)
		return -EINVAL;

	for (i = 0; i < cnt; i++) {
		switch (buf[i]) {
			case 'B' :
			case 'b' :
				tmp |= BAND_B; 
				break;
			case 'G' :
			case 'g' :
				tmp |= BAND_G; 
				break;
			case 'A' :
			case 'a' :
				tmp |= BAND_A; 
				break;
			default:
			return -EINVAL;
		}
	}
		
	/* Validate against FW support */
	if ((tmp | Adapter->fw_bands) &
					~Adapter->fw_bands)
		return -EINVAL;
	
	/* To support only <a/b/bg/abg> */
	if (!tmp || (tmp == BAND_G) || (tmp == (BAND_A | BAND_B)) ||
					(tmp == (BAND_A | BAND_G)))
		return -EINVAL;
	
	if (wlan_set_regiontable(priv, Adapter->RegionCode, tmp)) {
		return -EINVAL;
		}

	Adapter->config_bands = tmp;

	if (Adapter->config_bands & BAND_A) {
		Adapter->adhoc_start_band = BAND_A; 
        Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL_A;
	}
	else if (Adapter->config_bands & BAND_G) {
		Adapter->adhoc_start_band = BAND_G; 
		Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
	}
	else {
		Adapter->adhoc_start_band = BAND_B; 
		Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
	}

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set adhoc channel 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_setadhocch_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	/* iwpriv ethX setaddocchannel "a nnn" */
	u32	vals[2];
	u16	band;
	u8	chan;
	wlan_adapter	*Adapter = priv->adapter;

	if(copy_from_user(vals, wrq->u.data.pointer, 
		MIN(wrq->u.data.length, sizeof(vals)))){
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;			

	}
	switch (vals[0]) { /* 1st byte is band */
		case 'B' :
		case 'b' :
			band = BAND_B; 
			break;
		case 'G' :
		case 'g' :
			band = BAND_G; 
			break;
		case 'A' :
		case 'a' :
			band = BAND_A; 
			break;
		default:
			return -EINVAL;
	}

	/* Validate against FW support */
	if ((band | Adapter->fw_bands) & ~Adapter->fw_bands)
		return -EINVAL;

	chan = vals[1];

	if (!find_cfp_by_band_and_channel(Adapter, band, chan)) {
		PRINTM(INFO, "Invalid channel number %d\n", chan);
		return -EINVAL;
	}
		
	Adapter->adhoc_start_band = band; 
	Adapter->AdhocChannel   = chan;

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get band 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_get_band_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	char	status[10], *ptr;
	u16	val = priv->adapter->config_bands;

	ENTER();

	memset(status, 0, sizeof(status));
	ptr = status;

	if (val & BAND_A)
	*ptr++ = 'a'; 

	if (val & BAND_B)
	*ptr++ = 'b';

	if (val & BAND_G)
	*ptr++ = 'g';

	*ptr = '\0';
				
	PRINTM(INFO, "Status = %s\n", status);
	wrq->u.data.length = strlen(status) + 1;

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, 
					&status, wrq->u.data.length))
			return -EFAULT;
		}

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get adhoc channel 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_get_adhocch_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	u32		status[2];
	wlan_adapter	*Adapter = priv->adapter;
	u16		band = Adapter->adhoc_start_band;

	ENTER();
	
	if (band & BAND_A) {
		status[0] = 'a';
	} else if (band & BAND_B) {
		status[0] = 'b';
	} else if (band & BAND_G) { 
		status[0] = 'g';
	}

	status[1] = Adapter->AdhocChannel;

	if (wrq->u.data.pointer) {
		wrq->u.data.length = sizeof(status);

		if (copy_to_user(wrq->u.data.pointer, status,
							wrq->u.data.length))
			return -EFAULT;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#endif 

/** 
 *  @brief Get/Set sleep period 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_sleep_period(wlan_private *priv, struct iwreq *wrq)
{
	int ret;		
	int data;
	wlan_adapter *Adapter = priv->adapter;
	HostCmd_DS_802_11_SLEEP_PERIOD	sleeppd;
       
	ENTER();

	if (wrq->u.data.length > 1)
		return -ENOTSUPP;
	
	memset(&sleeppd, 0, sizeof(sleeppd));
	memset(&Adapter->sleep_period, 0, sizeof(SleepPeriod));

        if (wrq->u.data.length == 0) {
		sleeppd.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
        } else {
		copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
		/* sleep period is between 0 or 10 ... 60 */
		if ((data <= 60 && data >= 10) || (data == 0)){
			sleeppd.Action = 
				wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			sleeppd.Period = data;
		} else
			return -EINVAL;
        }

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SLEEP_PERIOD,
			0, HostCmd_OPTION_WAITFORRSP,
			0, (void *) &sleeppd);
	
	data = (int) Adapter->sleep_period.period;
	copy_to_user(wrq->u.data.pointer,&data,sizeof(int));
        wrq->u.data.length = 1;

	LEAVE();
	return ret;
}

/** 
 *  @brief Get/Set adapt rate 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_adapt_rateset(wlan_private *priv, struct iwreq *wrq)
{
	int			ret;
	wlan_adapter		*Adapter = priv->adapter;
	int data[2];

	ENTER();
	memset(data,0,sizeof(data));
	if (!wrq->u.data.length) {
		PRINTM(INFO, "Get ADAPT RATE SET\n");
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RATE_ADAPT_RATESET,
				HostCmd_ACT_GEN_GET, HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
		data[0] = Adapter->EnableHwAuto;
		data[1] = Adapter->RateBitmap;
		copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 2);
#define GET_TWO_INT	2
		wrq->u.data.length = GET_TWO_INT;
	} else {
		PRINTM(INFO, "Set ADAPT RATE SET\n");
		if(wrq->u.data.length > 2)
			return -EINVAL;
		copy_from_user(data,wrq->u.data.pointer,sizeof(int) * wrq->u.data.length);
		Adapter->EnableHwAuto = data[0];
		Adapter->RateBitmap = data[1];
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RATE_ADAPT_RATESET,
				HostCmd_ACT_GEN_SET,HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
	}

	LEAVE();
	return ret;	
}

/** 
 *  @brief Get/Set inactivity timeout
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_inactivity_timeout(wlan_private *priv, struct iwreq *wrq)
{
	int ret;		
    	int data = 0;
	u16		timeout = 0;
	ENTER();
	if (wrq->u.data.length > 1)	
		return -ENOTSUPP;
	
        if (wrq->u.data.length == 0) {
		/* Get*/
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_INACTIVITY_TIMEOUT,
				HostCmd_ACT_GET, HostCmd_OPTION_WAITFORRSP,
				0, &timeout);
		data = timeout;
		copy_to_user(wrq->u.data.pointer,&data,sizeof(int));
        } else {
		/* Set */
		copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
		timeout =data;
		ret = PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_INACTIVITY_TIMEOUT,
				HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
				0, &timeout);
	}

        wrq->u.data.length = 1;

	LEAVE();
	return ret;
}


/** 
 *  @brief Get LOG
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_getlog_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int		ret;
	char		buf[GETLOG_BUFSIZE-1];
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, " GET STATS\n"); 

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_GET_LOG,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	if (wrq->u.data.pointer) {
		sprintf(buf, "\n  mcasttxframe %u failed %u retry %u " 
				"multiretry %u framedup %u "
				"rtssuccess %u rtsfailure %u ackfailure %u\n"
				"rxfrag %u mcastrxframe %u fcserror %u "
				"txframe %u wepundecryptable %u ",
				Adapter->LogMsg.mcasttxframe,
				Adapter->LogMsg.failed,
				Adapter->LogMsg.retry,
				Adapter->LogMsg.multiretry,
				Adapter->LogMsg.framedup,
				Adapter->LogMsg.rtssuccess,
				Adapter->LogMsg.rtsfailure,
				Adapter->LogMsg.ackfailure,
				Adapter->LogMsg.rxfrag,
				Adapter->LogMsg.mcastrxframe,
				Adapter->LogMsg.fcserror,
				Adapter->LogMsg.txframe,
				Adapter->LogMsg.wepundecryptable);
		wrq->u.data.length = strlen(buf) + 1;
		copy_to_user(wrq->u.data.pointer, buf, wrq->u.data.length);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


#ifdef CF
/** 
 *  @brief Read/Write CF register
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_regcfrdwr_ioctl(wlan_private *priv, struct ifreq *req)
{
	wlan_ioctl_cfregrdwr regrdwr;

	ENTER();

	copy_from_user(&regrdwr, req->ifr_data, 
					sizeof(regrdwr));

	if (regrdwr.Action) {
		PRINTM(INFO, "Write value into CF register\n");
		sbi_write_cfreg(priv, regrdwr.Offset, regrdwr.Value);
	} else {
		PRINTM(INFO, "Read value from CF register\n");
		regrdwr.Value = sbi_read_cfreg(priv, 
		regrdwr.Offset);
		copy_to_user(req->ifr_data, &regrdwr, 
						sizeof(regrdwr));
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif


/** 
 *  @brief config sleep parameters
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_sleep_params_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int ret;
	wlan_adapter 	*Adapter = priv->adapter;
	wlan_ioctl_sleep_params_config  sp;

	ENTER();

	memset(&sp, 0, sizeof(sp));
	
	if (copy_from_user(&sp, wrq->u.data.pointer,
				MIN(sizeof(sp), wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}
	
	memcpy(&Adapter->sp, &sp.Error, sizeof(SleepParams));
	
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SLEEP_PARAMS,
			sp.Action, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (!ret && !sp.Action && wrq->u.data.pointer) {
		memcpy(&sp.Error, &Adapter->sp, sizeof(SleepParams));
		copy_to_user(wrq->u.data.pointer, &sp, sizeof(sp));
		wrq->u.data.length = sizeof(sp);
	}

	LEAVE();
	return ret;
}

/** 
 *  @brief Read the CIS Table
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_getcis_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 		ret = WLAN_STATUS_SUCCESS;
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (wrq->u.data.pointer) {
		copy_to_user(wrq->u.data.pointer, Adapter->CisInfoBuf, 
							Adapter->CisInfoLen);
		wrq->u.data.length = Adapter->CisInfoLen;
	}

	LEAVE();
	return ret;
}


#ifdef BCA
/** 
 *  @brief Set BCA timeshare
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_bca_timeshare_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int 				ret;
	wlan_adapter 			*Adapter = priv->adapter;
	wlan_ioctl_bca_timeshare_config bca_ts;

	ENTER();

	memset(&bca_ts, 0, sizeof(HostCmd_DS_802_11_BCA_TIMESHARE));
	
	if (copy_from_user(&bca_ts, wrq->u.data.pointer,
				MIN(sizeof(bca_ts), wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	PRINTM(INFO, "TrafficType=%x TimeShareInterva=%x BTTime=%x\n", 
			bca_ts.TrafficType, bca_ts.TimeShareInterval, 
			bca_ts.BTTime);

	ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_BCA_CONFIG_TIMESHARE,
			bca_ts.Action, HostCmd_OPTION_WAITFORRSP,
			0, &bca_ts);

	if (!ret && !bca_ts.Action && wrq->u.data.pointer) {
		copy_to_user(wrq->u.data.pointer, &Adapter->bca_ts, 
								sizeof(bca_ts));
		wrq->u.data.length = sizeof(HostCmd_DS_802_11_BCA_TIMESHARE);
	}

	LEAVE();
	return ret;
}
#endif /* BCA */

/** 
 *  @brief Set scan type
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static inline int wlan_scan_type_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	u8	buf[12];
	u8	*option[] = { "active", "passive", "get", };
	int	i, max_options = (sizeof(option)/sizeof(option[0]));
	int	ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef ENABLE_802_11D
	if( wlan_get_state_11d( priv) == ENABLE_11D ) {
		PRINTM(INFO, "11D: Cannot set scantype when 11D enabled\n");
		return -EFAULT;
	}
#endif

	memset(buf, 0, sizeof(buf));

	if (copy_from_user(buf, wrq->u.data.pointer, MIN(sizeof(buf), 
					wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}


	PRINTM(INFO, "Scan Type Option = %s\n", buf);

	buf[sizeof(buf)-1] = '\0';

	for (i = 0; i < max_options; i++) {
		if (!strcmp(buf, option[i]))
			break;
	}

	switch (i) {
		case 0:
			Adapter->ScanType = HostCmd_SCAN_TYPE_ACTIVE;
			break;
		case 1:
			Adapter->ScanType = HostCmd_SCAN_TYPE_PASSIVE;
			break;
		case 2:
			wrq->u.data.length = 
				strlen(option[Adapter->ScanType]) + 1;

			if (copy_to_user(wrq->u.data.pointer, 
						option[Adapter->ScanType],
						wrq->u.data.length)) {
				PRINTM(INFO, "Copy to user failed\n");
				ret = -EFAULT;
			}

			break;
		default:
			PRINTM(INFO, "Invalid Scan Type Ioctl Option\n");
			ret = -EINVAL;
			break;
	}

	LEAVE();
	return ret;
}

/** 
 *  @brief Set scan mode
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_scan_mode_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter *Adapter = priv->adapter;
	u8	buf[12];
	u8	*option[] = { "bss", "ibss", "any","get"};
	int	i, max_options = (sizeof(option)/sizeof(option[0]));
	int	ret = WLAN_STATUS_SUCCESS;

	ENTER();

	memset(buf, 0, sizeof(buf));

	if (copy_from_user(buf, wrq->u.data.pointer, MIN(sizeof(buf), 
					wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	PRINTM(INFO, "Scan Mode Option = %s\n", buf);

	buf[sizeof(buf)-1] = '\0';

	for (i = 0; i < max_options; i++) {
		if (!strcmp(buf, option[i]))
			break;
	}

	switch (i) {

		case 0:
			Adapter->ScanMode = HostCmd_BSS_TYPE_BSS;
			break;
		case 1:
			Adapter->ScanMode = HostCmd_BSS_TYPE_IBSS;
			break;
		case 2:
			Adapter->ScanMode = HostCmd_BSS_TYPE_ANY;
			break;
		case 3:

			wrq->u.data.length = 
				strlen(option[Adapter->ScanMode-1]) + 1;

			PRINTM(INFO, "Get Scan Mode Option = %s\n", 
					option[Adapter->ScanMode-1]);

			PRINTM(INFO, "Scan Mode Length %d\n", wrq->u.data.length);

			if (copy_to_user(wrq->u.data.pointer, 
						option[Adapter->ScanMode-1],
							wrq->u.data.length)) {
				PRINTM(INFO, "Copy to user failed\n");
				ret = -EFAULT;
			}
			PRINTM(INFO, "GET Scan Type Option after copy = %s\n", 
					wrq->u.data.pointer);

			break;

		default:
			PRINTM(INFO, "Invalid Scan Mode Ioctl Option\n");
			ret = -EINVAL;
			break;
	}

	LEAVE();
	return ret;
}

#ifdef ADHOC_GRATE
/** 
 *  @brief Get/Set Adhoc G Rate
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_do_set_grate_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	int 		data,data1;
	
	ENTER();
    	if(wrq->u.data.length == 0)
		return -EINVAL;
	copy_from_user(&data1,wrq->u.data.pointer,sizeof(int));
	switch (data1) {
		case 0:
			Adapter->adhoc_grate_enabled = FALSE;
			break;
		case 1:
			Adapter->adhoc_grate_enabled = TRUE;
			break;
		case 2:
			data = Adapter->adhoc_grate_enabled;
			copy_to_user(wrq->u.data.pointer, &data, sizeof(int) );
			wrq->u.data.length = 1;
			break;
		default:
			return -EINVAL;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* ADHOC_GRATE */

#ifdef PS_REQUIRED
#ifdef FW_WAKEUP_METHOD
/** 
 *  @brief Get/Set Firmware wakeup method
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_cmd_fw_wakeup_method( wlan_private *priv, struct iwreq *wrq )
{
	wlan_adapter	*Adapter = priv->adapter;
	u16		action;
	u16 		method;
	int		ret;
	int 		data;
	
	ENTER();

	if (wrq->u.data.length == 0 || !wrq->u.data.pointer) {
		action = HostCmd_ACT_GET;
	}
	else {
		action = HostCmd_ACT_SET;
		copy_from_user(&data, wrq->u.data.pointer,sizeof(int));
		switch (data ) {
		case 0:
			Adapter->fwWakeupMethod = WAKEUP_FW_UNCHANGED;
			break;
		case 1:
			Adapter->fwWakeupMethod = WAKEUP_FW_THRU_INTERFACE;
			break;
		case 2:
			Adapter->fwWakeupMethod = WAKEUP_FW_THRU_GPIO;
			break;
		default:
			return -EINVAL;
		}
	}

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_FW_WAKEUP_METHOD, action,
		HostCmd_OPTION_WAITFORRSP, 0,
		&Adapter->fwWakeupMethod);

	if (action == HostCmd_ACT_GET) {
		method = Adapter->fwWakeupMethod;
		copy_to_user(wrq->u.data.pointer, &method, sizeof(method));
		wrq->u.data.length = 1;
	}

	LEAVE();
	return ret;
}
#endif
#endif


/********************************************************
		Global Functions
********************************************************/
/** 
 *  @brief ioctl function - entry point
 *  
 *  @param dev		A pointer to net_device structure
 *  @param req	   	A pointer to ifreq structure
 *  @param cmd 		command
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
int wlan_do_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
	int		subcmd = 0;
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)
			&& (cmd != WLANDEEPSLEEP)
			/* here we exclude all sub-cmds under SET1INT_GET1INT cmd
			   in order to get SDIOCLOCK sub-cmd to work; need a fix later */
			&& (cmd != WLAN_SETONEINT_GETONEINT)
		) {
		PRINTM(MSG, "():%s IOCTLS called when station is"
					" in DeepSleep\n",__FUNCTION__);
		return -EBUSY;
	}
#endif

	PRINTM(INFO, "wlan_do_ioctl: ioctl cmd = 0x%x\n", cmd);
	switch (cmd) {
	case WLANEXTSCAN: 
		ret = wlan_extscan_ioctl(priv, req);
		break;
#ifdef STDCMD
	case WLANSTDCMD:
		ret = wlan_stdcmd_ioctl(dev, req, cmd);
		break;
#endif

	case WLANCISDUMP:	/* Read CIS Table  */
		ret = wlan_do_getcis_ioctl(priv, req);
		break;
		
	case WLANSCAN_TYPE :
		PRINTM(INFO, "Scan Type Ioctl\n");
		ret = wlan_scan_type_ioctl(priv, wrq);
		break;

#ifdef MANF_CMD_SUPPORT
	case WLANMANFCMD:
		PRINTM(INFO, "Entering the Manufacturing ioctl SIOCCFMFG\n");
		ret = wlan_mfg_command(priv, (void *) req->ifr_data);
		
		PRINTM(INFO, "Manufacturing Ioctl %s\n", (ret) ? "failed":"success");
		break;
#endif

	case WLANREGRDWR:	/* Register read write command */
		ret = wlan_regrdwr_ioctl(priv, req);
		break;

#if (defined SD) || (defined SPI)
	case WLANCMD52RDWR:	/* CMD52 read/write command */
		ret = wlan_cmd52rdwr_ioctl(priv, req);
		break;

	case WLANCMD53RDWR:	/* CMD53 read/write command */
		ret = wlan_cmd53rdwr_ioctl(priv, req);
		break;
#endif

#ifdef WPA
	case SIOCSIWENCODE:	/* set encoding token & mode for WPA */
		ret = wlan_set_encode(dev, NULL, &(wrq->u.data),
							wrq->u.data.pointer);
		break;
#endif
	case WLAN_SETNONE_GETNONE : /* set WPA mode on/off ioctl #20 */
		switch(wrq->u.data.flags) {
		case WLANDEAUTH :
			PRINTM(INFO, "Deauth\n");
			wlan_send_deauth(priv);
			break;
			
		case WLANADHOCSTOP:
			PRINTM(INFO, "Adhoc stop\n");
			ret = wlan_do_adhocstop_ioctl(priv);
			break;

		case WLANRADIOON:
			wlan_radio_ioctl(priv, RADIO_ON);
			break;
			
		case WLANRADIOOFF:
			wlan_radio_ioctl(priv, RADIO_OFF);
			break;
#ifdef ADHOCAES
		case WLANREMOVEADHOCAES:
			ret = wlan_remove_aes(priv);
			break;
#endif
#ifdef CIPHER_TEST
    		case WLANCIPHERTEST:
      			ret = wlan_cipher_test(priv, wrq);
      			break;
#endif
		}	/* End of switch */
		break;

	case WLAN_SETWORDCHAR_GETNONE :
		switch(wrq->u.data.flags) {
#ifdef ADHOCAES
		case WLANSETADHOCAES:
			ret = wlan_setadhocaes_ioctl(priv, req);
			break;
#endif /* end of ADHOCAES */
		}
		break;

	case WLAN_SETNONE_GETWORDCHAR:
		switch(wrq->u.data.flags) {
#ifdef ADHOCAES
		case WLANGETADHOCAES:
			ret = wlan_getadhocaes_ioctl(priv, req);
			break;
#endif /* end of ADHOCAES */
		case WLANVERSION:	/* Get driver version */
			ret = wlan_version_ioctl(priv, req);
			break;
		}
		break;
#ifdef WPA
	case WLANSETWPAIE:
		ret = wlan_setwpaie_ioctl(priv, req);
		break;
#endif

	case WLAN_SETONEINT_GETONEINT:
		switch(wrq->u.data.flags) {
		case WLANNF :		   
			ret  = wlan_get_nf(priv, wrq);
			break;
				
		case WLANRSSI :		
			ret = wlan_get_rssi(priv, wrq);
			break;
#ifdef WMM
		case WLANWMM_ENABLE:
			ret = wlan_wmm_enable_ioctl(priv, wrq);
			break;
#endif /* WMM */
#ifdef THROUGHPUT_TEST
		case WLANTHRUPUT:
			ret = wlan_thruput(priv, wrq);
			break;
#endif /* THROUGHTPUT_TEST */

#ifdef BG_SCAN
		case WLANBGSCAN:
			if (wrq->u.data.length) {
				int data,data1;
				copy_from_user(&data1,wrq->u.data.pointer,sizeof(int));
				switch (data1) {
					case CMD_DISABLED:
						PRINTM(INFO, "Background scan is set to disable\n");
						ret = wlan_bg_scan_enable(priv, FALSE);
						break;
					case CMD_ENABLED:
						PRINTM(INFO, "Background scan is set to enable\n");
						ret = wlan_bg_scan_enable(priv, TRUE);
						break;
					case CMD_GET:
						data = (Adapter->bgScanConfig->Enable == TRUE) ? 
							CMD_ENABLED : CMD_DISABLED;
						wrq->u.data.length = 1;
						copy_to_user(wrq->u.data.pointer, &data, sizeof(int));
						break;
					default:			
						ret = -EINVAL;
						PRINTM(INFO, "Background scan: wrong parameter\n");
						break;
				}
			}
			else {
				ret = -EINVAL;
				PRINTM(INFO, "Background scan: wrong parameter\n");
			}
			break;
#endif /* BG_SCAN */

#ifdef BULVERDE_SDIO
		case WLANSDIOCLOCK:
			if (wrq->u.data.length) {
				int data;
				copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
				switch (data) {
					case CMD_DISABLED:
						PRINTM(INFO, "Bulverde SDIO clock is turned off\n");
						ret = wlan_sdio_clock(priv, FALSE);
						break;
					case CMD_ENABLED:
						PRINTM(INFO, "Bulverde SDIO clock is turned on\n");
						ret = wlan_sdio_clock(priv, TRUE);
						break;
					case CMD_GET: /* need an API in sdio.c to get STRPCL */
					default:			
						ret = -EINVAL;
						PRINTM(INFO, "sdioclock: wrong parameter\n");
						break;
				}
			}
			else {
				ret = -EINVAL;
				PRINTM(MSG, "sdioclock scan: wrong parameter\n");
			}
			break;
#endif /* BULVERDE_SDIO */
#ifdef ENABLE_802_11D
		case WLANENABLE11D:
			ret = wlan_cmd_enable_11d( priv, wrq );
			break;
#endif

#ifdef ADHOC_GRATE
		case WLANADHOCGRATE:
			ret = wlan_do_set_grate_ioctl(priv, wrq);
			break;
#endif			

#ifdef ENABLE_802_11H
		case WLAN_11H_ENABLE:
			ret = wlan_11h_ioctl_enable( priv, wrq );
			break;
        
		case WLAN_11H_SETLOCALPOWER:
            		ret = wlan_11h_ioctl_get_local_power( priv, wrq );
			break;
#endif

#ifdef WMM_UAPSD
		case WLAN_WMM_QOSINFO:
			{
				int data;
				if (wrq->u.data.length == 1)
				{
					copy_from_user(&data, wrq->u.data.pointer,sizeof(int));
					Adapter->wmm.qosinfo = (u8)data; 
				}
				else
				{
					data = (int) Adapter->wmm.qosinfo;
					copy_to_user(wrq->u.data.pointer,&data,sizeof(int));
					wrq->u.data.length = 1;
				}
			}
			break;
#endif
#ifdef PS_REQUIRED
		case WLAN_LISTENINTRVL:
			if(!wrq->u.data.length) {				
				int data;
				PRINTM(INFO, "Get LocalListenInterval Value\n");
#define GET_ONE_INT	1				
				data = Adapter->LocalListenInterval;
				copy_to_user(wrq->u.data.pointer,&data,sizeof(int));
				wrq->u.data.length = GET_ONE_INT;
			} else {
				int data;
				copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
				PRINTM(INFO, "Set LocalListenInterval = %d\n",data);
#define MAX_U16_VAL	65535
				if(data > MAX_U16_VAL) {
					PRINTM(INFO, "Exceeds U16 value\n");
					return -EINVAL;
				}
				Adapter->LocalListenInterval = data;
			}
			break;
#ifdef FW_WAKEUP_METHOD
		case WLAN_FW_WAKEUP_METHOD:
			ret = wlan_cmd_fw_wakeup_method( priv, wrq );
			break;		
#endif
#endif
		case WLAN_TXCONTROL:
			ret = wlan_txcontrol(priv, wrq ); //adds for txcontrol ioctl
			break;

#ifdef ATIMGEN			
		case  WLANATIMGEN:
			ret = wlan_atimgen(priv, wrq );
			/* enable/disable atim generation*/
			break;
#endif

#ifdef WMM_UAPSD
		case  WLANNULLGEN:
			ret = wlan_null_pkg_gen(priv, wrq );
			/* enable/disable null pkg generation*/
			break;
#endif

		default:
			ret = -EOPNOTSUPP;
			break;
		}
		break;	
	
	case WLAN_SETONEINT_GETNONE:
		/* The first 4 bytes of req->ifr_data is sub-ioctl number
		 * after 4 bytes sits the payload.
		 */
		subcmd = wrq->u.data.flags;	//from wpa_supplicant subcmd
		
		if(!subcmd)
			subcmd = (int)req->ifr_data;	//from iwpriv subcmd
		
		switch (subcmd) {
		case WLAN_SUBCMD_SETRXANTENNA : /* SETRXANTENNA */
			ret = wlan_subcmd_setrxantenna_ioctl(priv, req);
			break;
				
		case WLAN_SUBCMD_SETTXANTENNA : /* SETTXANTENNA */
			ret = wlan_subcmd_settxantenna_ioctl(priv, req);
			break;
				
		case WLAN_SET_ATIM_WINDOW:
			ret = wlan_set_atim_window_ioctl(priv, req);
			break;
		case WLANSETBCNAVG:
			priv->adapter->bcn_avg_factor = 
				(int) ((struct ifreq *)((u8 *)req + 4))->ifr_data;
			ret = WLAN_STATUS_SUCCESS;
			break;
		case WLANSETDATAAVG:
			priv->adapter->data_avg_factor = 
				(int) ((struct ifreq *)((u8 *)req + 4))->ifr_data;
			ret = WLAN_STATUS_SUCCESS;
			break;
		case WLANSETREGION:
			ret = wlan_setregion_ioctl(priv, req);
			break;
				
		case WLAN_SET_LISTEN_INTERVAL:
			ret = wlan_set_listen_interval_ioctl(priv, req);
			break;
				
#ifdef PS_REQUIRED
		case WLAN_SET_MULTIPLE_DTIM:
			ret = wlan_set_multiple_dtim_ioctl(priv, req);
			break;
#endif

		case WLANSETAUTHALG:
			ret = wlan_setauthalg_ioctl(priv, req);
			break;
		
		case WLANSET8021XAUTHALG:
			ret = wlan_set8021xauthalg_ioctl(priv, req);
			break;

		case WLANSETENCRYPTIONMODE:
			ret = wlan_setencryptionmode_ioctl(priv, req);
			break;
		default:
			ret = -EOPNOTSUPP;
			break;
		}
		
		break;

	case WLAN_SETNONE_GETTWELVE_CHAR: /* Get Antenna settings */
		/* 
		 * We've not used IW_PRIV_TYPE_FIXED so sub-ioctl number is
		 * in flags of iwreq structure, otherwise it will be in
		 * mode member of iwreq structure.
		 */
		switch ((int)wrq->u.data.flags) {
		case WLAN_SUBCMD_GETRXANTENNA: /* Get Rx Antenna */
			ret = wlan_subcmd_getrxantenna_ioctl(priv, req);
			break;

		case WLAN_SUBCMD_GETTXANTENNA:  /* Get Tx Antenna */
			ret = wlan_subcmd_gettxantenna_ioctl(priv, req);
			break;
		}
		break;
		
	case WLAN_SETNONE_GETBYTE:
		switch(wrq->u.data.flags) {
#ifdef REASSOCIATION
		case WLANREASSOCIATIONAUTO:
			reassociation_on(priv);
			break;
		case WLANREASSOCIATIONUSER:
			reassociation_off(priv);
			break;
#endif /* REASSOCIATION */
		case WLANWLANIDLEON:
			wlanidle_on(priv);
			break;
		case WLANWLANIDLEOFF:
			wlanidle_off(priv);
			break;
		}	
		
		break;		
		
#ifdef DEEP_SLEEP
	case WLANDEEPSLEEP:
		ret = wlan_deepsleep_ioctl(priv, req);
		break;
#endif

#ifdef HOST_SLEEP
	case WLANHOSTSLEEPCFG:
		ret = wlan_do_hostsleepcfg_ioctl(priv, wrq); 
		break;
#endif
			
	case WLAN_SET64CHAR_GET64CHAR:
		switch ((int)wrq->u.data.flags) {
#ifdef ENABLE_802_11H
		case WLAN_11H_REQUESTTPC:
			ret = wlan_11h_ioctl_request_tpc(priv, wrq);
			break;
		case WLAN_11H_SETPOWERCAP:
			ret = wlan_11h_ioctl_set_power_cap(priv, wrq);
			break;
#endif
#ifdef ENABLE_MEAS
		case WLAN_MEASREQ:
			ret = wlan_meas_ioctl_send_req(priv, wrq);
			break;
#endif

		case WLANSLEEPPARAMS:
			ret = wlan_sleep_params_ioctl(priv, wrq);
			break;

#ifdef BCA
		case WLAN_BCA_TIMESHARE:
			ret = wlan_bca_timeshare_ioctl(priv, wrq);
			break;
#endif
		case WLANSCAN_MODE :
			PRINTM(INFO, "Scan Mode Ioctl\n");
			ret = wlan_scan_mode_ioctl(priv, wrq);
			break;
		
		case WLAN_GET_ADHOC_STATUS:
			ret = wlan_get_adhoc_status_ioctl(priv, wrq);
			break;
#if defined(IWGENIE_SUPPORT) && (WIRELESS_EXT < 18)                    
        	case WLAN_SET_GEN_IE:
            		ret = wlan_set_gen_ie_ioctl(priv, wrq);
            		break;
        	case WLAN_GET_GEN_IE:
            		ret = wlan_get_gen_ie_ioctl(priv, wrq);
            		break;
#endif
#ifdef REASSOC_CMD_SUPPORT
        	case WLAN_REASSOCIATE:
            		ret = wlan_reassociate_ioctl(dev, wrq);
            		break;
#endif
        	}
        	break;
	case WLAN_SETCONF_GETCONF:
		PRINTM(INFO, "The WLAN_SETCONF_GETCONF=0x%x is %d\n",WLAN_SETCONF_GETCONF, *(req -> ifr_data));
		switch ((int)*(req -> ifr_data)) {
#ifdef CAL_DATA
		case CAL_DATA_EXT_CONFIG:
		ret = wlan_do_caldata_ext_ioctl(priv, req);
		break;
#endif /* CAL_DATA */
#ifdef BG_SCAN
		case BG_SCAN_CONFIG:
			ret = wlan_do_bg_scan_config_ioctl(priv, req);
			break;
#endif /* BG_SCAN */

#ifdef SUBSCRIBE_EVENT_CTRL
		case SUBSCRIBE_EVENT:
			ret = wlan_do_subscribe_event(priv, req);
			break;
#endif

#ifdef WMM
		case WMM_TSPEC:
			ret = wlan_do_wmm_tspec_ioctl(priv, req);
			break;
		case WMM_ACK_POLICY:
			ret = wlan_do_wmm_ack_policy_ioctl(priv, req);
			break;	
		case WMM_PARA_IE:
			ret = wlan_do_wmm_para_ie_ioctl(priv, req);
			break;
#endif /* WMM */
		}
		break;

	case WLAN_SETNONE_GETONEINT:
		switch((int)req->ifr_data) {
		case WLANGETBCNAVG:
			req->ifr_data = (char *)((u32)Adapter->bcn_avg_factor);
			break;

		case WLANGETREGION:
			req->ifr_data = (char *)((u32)Adapter->RegionCode);
			break;

		case WLAN_GET_LISTEN_INTERVAL:
			req->ifr_data = (char *)((u32)Adapter->ListenInterval);
			break;

#ifdef PS_REQUIRED
		case WLAN_GET_MULTIPLE_DTIM:
			req->ifr_data = (char *)((u32)Adapter->MultipleDtim);
			break;
#endif
		default:
			ret = -EOPNOTSUPP;

		}

		break;
		
	case WLAN_SETTENCHAR_GETNONE:
		switch ((int)wrq->u.data.flags) {
#ifdef MULTI_BANDS
		case WLAN_SET_BAND:
			ret = wlan_do_setband_ioctl(priv, wrq);
			break;

		case WLAN_SET_ADHOC_CH:
			ret = wlan_do_setadhocch_ioctl(priv, wrq);
			break;
#endif

#ifdef ENABLE_802_11H
		case WLAN_11H_CHANSWANN:
			ret = wlan_11h_ioctl_chan_sw_ann(priv, wrq);
			break;
#endif
		}
		break;

	case WLAN_SETNONE_GETTENCHAR:
		switch ((int)wrq->u.data.flags) {
#ifdef MULTI_BANDS
		case WLAN_GET_BAND:
			ret = wlan_do_get_band_ioctl(priv, wrq);
			break;

		case WLAN_GET_ADHOC_CH:
			ret = wlan_do_get_adhocch_ioctl(priv, wrq);
			break;
#endif
		}
		break;

    	case WLANGETLOG:
        	ret = wlan_do_getlog_ioctl(priv, wrq);
        	break;

#ifdef CF
	case WLANREGCFRDWR:
		ret = wlan_do_regcfrdwr_ioctl(priv, req);
		break;
#endif

  	case WLAN_SET_GET_SIXTEEN_INT:
    		switch ((int)wrq->u.data.flags) {
      		case WLAN_TPCCFG:
     	 	{
        		int data[5];
			HostCmd_DS_802_11_TPC_CFG cfg;
        		memset(&cfg, 0, sizeof(cfg));
        		if ((wrq->u.data.length > 1) && (wrq->u.data.length != 5))
          			return WLAN_STATUS_FAILURE;

        		if (wrq->u.data.length == 0){
          			cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
        		}
        		else
        		{
  	  			copy_from_user(data,wrq->u.data.pointer, sizeof(int) * 5);
			        cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
          			cfg.Enable  = data[0];
          			cfg.UseSNR  = data[1];
          			cfg.P0      = data[2];
          			cfg.P1      = data[3];
          			cfg.P2      = data[4];
        		}

        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_TPC_CFG,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &cfg);

        		data[0] = cfg.Enable;
        		data[1] = cfg.UseSNR;
        		data[2] = cfg.P0;
        		data[3] = cfg.P1;
        		data[4] = cfg.P2;
			copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 5);
        		wrq->u.data.length = 5;
      		}
      		break;
  
      		case WLAN_POWERCFG:
      		{
      			int data[4];
        		HostCmd_DS_802_11_PWR_CFG cfg;
        		memset(&cfg, 0, sizeof(cfg));
        		if ((wrq->u.data.length > 1) && (wrq->u.data.length != 4))
          			return WLAN_STATUS_FAILURE;
        		if (wrq->u.data.length == 0){
				cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
        		}
        		else
        		{
	  			copy_from_user(data,wrq->u.data.pointer, sizeof(int) * 4);		
          			cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
          			cfg.Enable  = data[0];
          			cfg.PA_P0   = data[1];
          			cfg.PA_P1   = data[2];
          			cfg.PA_P2   = data[3];
        		}
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_PWR_CFG,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &cfg);
        		data[0] = cfg.Enable;
        		data[1] = cfg.PA_P0;
        		data[2] = cfg.PA_P1;
        		data[3] = cfg.PA_P2; 
			copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 4);
        		wrq->u.data.length = 4;
      		}
      		break;
#ifdef AUTO_FREQ_CTRL
  		case WLAN_AUTO_FREQ_SET: 
      		{
			int data[3];
        		HostCmd_DS_802_11_AFC afc;
        		memset(&afc, 0, sizeof(afc));
			if (wrq->u.data.length != 3)
		 		return WLAN_STATUS_FAILURE;
			copy_from_user(data,wrq->u.data.pointer,sizeof(int)*3);
        		afc.afc_auto    = data[0];

        		if (afc.afc_auto != 0) {
          			afc.afc_thre    = data[1];
          			afc.afc_period  = data[2];
        		} else {
          			afc.afc_toff    = data[1];
          			afc.afc_foff    = data[2];
        		}
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_SET_AFC,0, 
				HostCmd_OPTION_WAITFORRSP,0,(void *) &afc);
      		}
      		break;
		case WLAN_AUTO_FREQ_GET: 
      		{
        		int data[3];
        		HostCmd_DS_802_11_AFC afc;
        		memset(&afc, 0, sizeof(afc));
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_GET_AFC,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &afc);
        		data[0] = afc.afc_auto;
        		data[1] = afc.afc_toff;
        		data[2] = afc.afc_foff;
			copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 3);
        		wrq->u.data.length = 3;
      		}
      		break;
#endif
		case WLAN_SCANPROBES: 
      		{
			int data;      
        		if (wrq->u.data.length > 0){
	  			copy_from_user(&data,wrq->u.data.pointer,sizeof(int));
          			Adapter->ScanProbes = data;
			}
        		else
			{
          			data = Adapter->ScanProbes;
	  			copy_to_user(wrq->u.data.pointer,&data,sizeof(int));
			}
        		wrq->u.data.length = 1;
      		}
      		break;
#ifdef LED_GPIO_CTRL
    		case WLAN_LED_GPIO_CTRL:
    		{
      			int i;
      			int data[16];
      
      			HostCmd_DS_802_11_LED_CTRL ctrl;
      			MrvlIEtypes_LedGpio_t *gpio = (MrvlIEtypes_LedGpio_t *) ctrl.data;

      			memset(&ctrl, 0, sizeof(ctrl));
        		if (wrq->u.data.length > MAX_LEDS*2)
        			return -ENOTSUPP;
      			if ((wrq->u.data.length % 2) != 0)
        			return -ENOTSUPP;
      			if (wrq->u.data.length == 0) {
        			ctrl.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
      			} else {
				copy_from_user(data,wrq->u.data.pointer,sizeof(int)*wrq->u.data.length);
        			ctrl.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
        			ctrl.NumLed = wlan_cpu_to_le16(0);
        			gpio->Header.Type = wlan_cpu_to_le16(TLV_TYPE_LED_GPIO);
        			gpio->Header.Len  = wrq->u.data.length;
        			for (i=0; i < wrq->u.data.length; i+=2) {
          				gpio->LedPin[i/2].Led = data[i];
         	 			gpio->LedPin[i/2].Pin = data[i+1];
        			}
      			}
      			ret = PrepareAndSendCommand( priv, HostCmd_CMD_802_11_LED_GPIO_CTRL,0, 
					HostCmd_OPTION_WAITFORRSP,0, (void *) &ctrl);
        		for (i=0; i < gpio->Header.Len; i+=2) {
        			data[i] = gpio->LedPin[i/2].Led;
        			data[i+1] = gpio->LedPin[i/2].Pin;
      			}
      			copy_to_user(wrq->u.data.pointer,data,sizeof(int)* gpio->Header.Len); 	
      			wrq->u.data.length = gpio->Header.Len;
    		}
    		break;
#endif
		case WLAN_SLEEP_PERIOD:
			ret = wlan_sleep_period(priv, wrq);
			break;
		case WLAN_ADAPT_RATESET:
			ret = wlan_adapt_rateset(priv, wrq);
			break;
		case WLAN_INACTIVITY_TIMEOUT:
			ret = wlan_inactivity_timeout(priv, wrq); 
			break;
		case WLANSNR :
			ret = wlan_get_snr(priv, wrq);
			break;
		case WLAN_GET_RATE:
			ret = wlan_getrate_ioctl(priv, wrq);
			break;
  		}
  		break;
	default:
 		ret = -EINVAL;
		break;
	}
	LEAVE();
	return ret;
}


/** 
 *  @brief Get wireless statistics
 *  
 *  @param dev		A pointer to net_device structure
 *  @return 	   	A pointer to iw_statistics buf
 */
struct iw_statistics *wlan_get_wireless_stats(struct net_device *dev)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	priv->wstats.status = Adapter->InfrastructureMode;
	priv->wstats.discard.retries = priv->stats.tx_errors;

	/* send RSSI command to get beacon RSSI/NF, valid only if associated */
	ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_RSSI,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	priv->wstats.qual.level =
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
		Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	priv->wstats.qual.noise = CAL_NF(Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	if (Adapter->NF[TYPE_BEACON][TYPE_NOAVG] == 0
		&& Adapter->MediaConnectStatus == WlanMediaStateConnected)
		priv->wstats.qual.noise = MRVDRV_NF_DEFAULT_SCAN_VALUE;
	else
		priv->wstats.qual.noise = 
				CAL_NF(Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	priv->wstats.qual.qual = 0;

	PRINTM(INFO, "Signal Level = %#x\n", priv->wstats.qual.level);
	PRINTM(INFO, "Noise = %#x\n", priv->wstats.qual.noise);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_GET_LOG,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (!ret) {
		priv->wstats.discard.code = Adapter->LogMsg.wepundecryptable;
		priv->wstats.discard.fragment = Adapter->LogMsg.fcserror;
		priv->wstats.discard.retries = Adapter->LogMsg.retry;
		priv->wstats.discard.misc = Adapter->LogMsg.ackfailure;
	}

	return &priv->wstats;
}

#endif //linux



/** 
 *  @brief compare ssid 
 *  
 *  @param ssid1	A pointer to ssid
 *  @param ssid2        A pointer to ssid 
 *  @return 	   	0--ssid is same, otherwise is different
 */
int SSIDcmp(WLAN_802_11_SSID *ssid1, WLAN_802_11_SSID *ssid2)
{
	if (!ssid1 || !ssid2)
		return -1;

	if (ssid1->SsidLength != ssid2->SsidLength)
		return -1;

	return memcmp(ssid1->Ssid, ssid2->Ssid, ssid1->SsidLength);
}

/** 
 *  @brief This function finds ssid in ssid list.
 *  
 *  @param Adapter	A pointer to wlan_adapter
 *  @param ssid		ssid
 *  @param bssid	bssid
 *  @param mode		encryption mode
 *  @return 	   	index in BSSID list
 */
int FindSSIDInList(wlan_adapter *Adapter,
			WLAN_802_11_SSID *ssid, u8 *bssid, int mode)
{
	int	i, j;
	int	net = -ENETUNREACH;
	u8	bestrssi = 0;

	PRINTM(INFO, "Num of BSSIDs = %d\n", Adapter->ulNumOfBSSIDs);

	for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
		if (!SSIDcmp(&Adapter->BSSIDList[i].Ssid, ssid) &&
				(!bssid || 
				 !memcmp(Adapter->BSSIDList[i].
					MacAddress, bssid, ETH_ALEN))) {
			switch (mode) {
			case Wlan802_11Infrastructure :
			case Wlan802_11IBSS :
				j = IsNetworkCompatible(Adapter, i, mode);

				if (j >= 0) {
					if (bssid) {
						return i;
					}

					if (SCAN_RSSI(Adapter->BSSIDList[i].Rssi)
							> bestrssi) {
						bestrssi = 
							SCAN_RSSI(Adapter->BSSIDList[i].Rssi);
						net = i;
					}
				} else {
					if (net == -ENETUNREACH) {
						net = j;
					}
				}
				break;
			case Wlan802_11AutoUnknown :
			default :
				if (SCAN_RSSI(Adapter->BSSIDList[i].Rssi)
							> bestrssi) {
					bestrssi = 
						SCAN_RSSI(Adapter->BSSIDList[i].Rssi);
					net = i;
				}
				break;
			}
		}
	}

	return net;
}

/*  @brief associate to the AP with specific ssid 
 *   
 *  @param adapter 	A pointer to wlan_private structure
 *  @ssid 		A pointer to AP's ssid
 *  @return 	   	WLAN_STATUS_SUCCESS-success, otherwise fail
 */

int wlan_associate(wlan_private *priv, WLAN_802_11_SSID *ssid)
{
	int ret = WLAN_STATUS_SUCCESS;
	wlan_adapter *Adapter = priv->adapter;

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AUTHENTICATE,
		0,HostCmd_OPTION_WAITFORRSP,
		0, &Adapter->CurrentBSSID);

	if (ret) {
		LEAVE();
		return ret;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_ASSOCIATE,
			0, HostCmd_OPTION_WAITFORRSP,
			0, ssid);
	return ret;
}

/** 
 *  @brief Set frequency
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param info			A pointer to iw_request_info structure 
 *  @param fwrq			A pointer to iw_freq structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int wlan_set_freq(struct net_device *dev, struct iw_request_info *info, 
		struct iw_freq *fwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int             rc = -EINPROGRESS;	/* Call commit handler */
	CHANNEL_FREQ_POWER	*cfp;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWFREQ: IOCTLS called when station is "
							"in DeepSleep\n");
		return -EBUSY;
	}
#endif

#ifdef MULTI_BANDS
	if (Adapter->is_multiband) {
		PRINTM(MSG, "SIOCSIWFREQ: IOCTLS call not supported in "
				"MultiBands mode, use private command "
				"'setadhocch' instead.\n");
		return -ENOTSUPP;
	}
#endif

	/*
	 * If setting by frequency, convert to a channel 
	 */
	if (fwrq->e == 1) {

		long	f = fwrq->m / 100000;
		int	c = 0;

		cfp = find_cfp_by_band_and_freq(Adapter, 0, f);
		if (!cfp) {
			PRINTM(INFO, "Invalid freq=%ld\n", f);
			return -EINVAL;
		} 
		
		c = (int) cfp->Channel;

		if (c < 0) 
			return -EINVAL;
	
		fwrq->e = 0;
		fwrq->m = c;
	}

	/*
	 * Setting by channel number 
	 */
	if (fwrq->m > 1000 || fwrq->e > 0) {
		rc = -EOPNOTSUPP;
	} else {
		int	channel = fwrq->m;

		cfp = find_cfp_by_band_and_channel(Adapter, 0, channel);
		if (!cfp) {
			rc = -EINVAL;
		} else {
			if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
				rc = ChangeAdhocChannel(priv, channel);
				/*  If station is WEP enabled, send the 
				 *  command to set WEP in firmware
				 */
				if (Adapter->SecInfo.WEPStatus == 
							Wlan802_11WEPEnabled) {
					PRINTM(INFO, "set_freq: WEP Enabled\n");
					ret = PrepareAndSendCommand(priv,
						HostCmd_CMD_802_11_SET_WEP, 
						0,HostCmd_OPTION_WAITFORRSP
						, OID_802_11_ADD_WEP, 
						NULL);

					if (ret) {
						LEAVE();
						return ret;
					}
						
					Adapter->CurrentPacketFilter |=	
						HostCmd_ACT_MAC_WEP_ENABLE;

					SetMacPacketFilter(priv);
				}
			} else {
				rc = -EOPNOTSUPP;
			}
		}
	}

	LEAVE();
	return rc;
}


/** 
 *  @brief scan an AP with specific ssid 
 *   
 *  @param adapter 		A pointer to wlan_private structure
 *  @param RequestedSSID	A pointer to AP's ssid
 *  @return 	   		WLAN_STATUS_SUCCESS-success, otherwise fail
 */

int SendSpecificScan(wlan_private *priv, WLAN_802_11_SSID *RequestedSSID)
{
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (!Adapter->bIsScanInProgress) {
		/* store the SSID info temporarily */
		if (RequestedSSID) {
			/* Specific BSSID */
			memcpy(&(Adapter->AttemptedSSIDBeforeScan),
				RequestedSSID, sizeof(WLAN_802_11_SSID));

			memcpy(&Adapter->SpecificScanSSID, RequestedSSID,
						sizeof(WLAN_802_11_SSID));
		}

		Adapter->SetSpecificScanSSID = TRUE;
		
#ifdef PROGRESSIVE_SCAN
		Adapter->ChannelsPerScan = MRVDRV_CHANNELS_PER_ACTIVE_SCAN;
#endif
		wlan_scan_networks(priv);
		Adapter->SetSpecificScanSSID = FALSE;
#ifdef PROGRESSIVE_SCAN
		Adapter->ChannelsPerScan = MRVDRV_CHANNELS_PER_SCAN;
#endif
	}

	if (Adapter->SurpriseRemoved)
		return WLAN_STATUS_FAILURE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef DEEP_SLEEP
/** 
 *  @brief Set Deep Sleep 
 *   
 *  @param adapter 	A pointer to wlan_private structure
 *  @param bDeepSleep	TRUE--enalbe deepsleep, FALSE--disable deepsleep
 *  @return 	   	WLAN_STATUS_SUCCESS-success, otherwise fail
 */

int SetDeepSleep(wlan_private *priv, BOOLEAN bDeepSleep)
{
	int		ret =  WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

#ifdef PS_REQUIRED
	// check if it is in IEEE PS mode or not
	if (Adapter->PSMode != Wlan802_11PowerModeCAM) {
		LEAVE();
		return  WLAN_STATUS_SUCCESS;
	}
#endif

	if (bDeepSleep == TRUE) {
		if (Adapter->IsDeepSleep != TRUE) {
			PRINTM(INFO, "Deep Sleep : sleep\n");

			
			// note: the command could be queued and executed later
			//       if there is command in prigressing.
			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_DEEP_SLEEP, 0,
					HostCmd_OPTION_WAITFORRSP,
					0, NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
			os_stop_queue(priv);
			os_carrier_off(priv);
		}
	}
	else {
		if (Adapter->IsDeepSleep == TRUE) {
			PRINTM(INFO, "Deep Sleep : wakeup\n");

			if (Adapter->IntCounterSaved)
				Adapter->IntCounter = Adapter->IntCounterSaved;

			if (sbi_exit_deep_sleep(priv))
				PRINTM(INFO, "Deep Sleep : wakeup failed\n");
			

			if (Adapter->IsDeepSleep == TRUE) {
			
				if(interruptible_sleep_on_timeout(&Adapter->ds_awake_q,
						       	WAIT_FOR_SCAN_RRESULT_MAX_TIME) == 0)
					PRINTM(MSG, "ds_awake_q: timer expired\n");
			}

			if(Adapter->IntCounter)
				wake_up_interruptible(&priv->MainThread.waitQ);
		}
	}
	
	LEAVE();
	return  WLAN_STATUS_SUCCESS;
}
#endif // DEEP_SLEEP

/** 
 *  @brief Start an Adhoc Network
 *   
 *  @param priv		 	A pointer to wlan_private structure
 *  @param AdhocSSID		The ssid of the Adhoc Network
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int StartAdhocNetwork(wlan_private * priv, WLAN_802_11_SSID * AdhocSSID)
{
    	int     ret = 0;
    	wlan_adapter    *Adapter = priv->adapter;

	ENTER();

    	Adapter->AdhocCreate = TRUE;
	
	if (!Adapter->capInfo.ShortPreamble) {
		PRINTM(INFO, "AdhocStart: Long Preamble\n");
		Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
	} else {
		PRINTM(INFO, "AdhocStart: Short Preamble\n");
		Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
	}
		
	SetRadioControl(priv);

#ifdef ENABLE_802_11H
	if (Adapter->adhoc_start_band == BAND_A)
    	{
        	Adapter->AdhocChannel = wlan_11h_get_adhoc_start_channel(priv);
        
       	 	/*
        	** Check if the region and channel requires a channel availability 
        	**   check.
        	*/
	        if (wlan_11h_radar_detect_required(priv, Adapter->AdhocChannel)){
            		/*
            		** Radar detection is required for this channel, make sure
            		**   11h is activated in the firmware
            		*/
            		wlan_11h_activate(priv, TRUE);

            		/* Check for radar on the channel */
            		if (wlan_11h_radar_detected(priv, Adapter->AdhocChannel)){
                		LEAVE();
                		return WLAN_STATUS_FAILURE;
            		}
        	}
    	}
#endif

    	PRINTM(INFO, "Adhoc Channel = %d\n", Adapter->AdhocChannel);
	PRINTM(INFO, "CurBssParams.channel = %d\n", Adapter->CurBssParams.channel);
	PRINTM(INFO, "CurBssParams.band = %d\n", Adapter->CurBssParams.band);

    	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_START,
                                0, HostCmd_OPTION_WAITFORRSP,
                                0, AdhocSSID);

    	LEAVE();
    	return ret;
}

/** 
 *  @brief Join an Adhoc Network
 *   
 *  @param priv		 	A pointer to wlan_private structure
 *  @param AdhocSSID		The ssid of the Adhoc Network
 *  @param i                    The index of BSSID List 
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int JoinAdhocNetwork(wlan_private * priv, WLAN_802_11_SSID * AdhocSSID, int i)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_BSSID	*pBSSIDList;
 
	ENTER();

	pBSSIDList = &Adapter->BSSIDList[i];

	/*Use ShortPreamble only when both creator and card supports 
	 short preamble*/	
	if (!pBSSIDList->Cap.ShortPreamble 
				|| !Adapter->capInfo.ShortPreamble) {
		PRINTM(INFO, "AdhocJoin: Long Preamble\n");
		Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
	}
	else {
		PRINTM(INFO, "AdhocJoin: Short Preamble\n");
		Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
	}
		
	SetRadioControl(priv);
	
	Adapter->Channel = pBSSIDList->Channel;
    
	PRINTM(INFO, "Joining on Channel %d\n", Adapter->Channel);
	PRINTM(INFO, "CurBssParams.channel = %d\n", Adapter->CurBssParams.channel);
	PRINTM(INFO, "CurBssParams.band = %c\n", Adapter->CurBssParams.band);

	Adapter->AdhocCreate = FALSE;

	// store the SSID info temporarily
	memset(&Adapter->AttemptedSSIDBeforeScan, 0, sizeof(WLAN_802_11_SSID));
	memcpy(&Adapter->AttemptedSSIDBeforeScan,
			AdhocSSID, sizeof(WLAN_802_11_SSID));

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_JOIN,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_SSID,
			&Adapter->AttemptedSSIDBeforeScan);

	LEAVE();
	return ret;
}


/** 
 *  @brief Set essid 
 *   
 *  @param dev 			A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param dwrq                 A pointer to iw_point structure
 *  @param extra		A pointer to extra data buf   
 *  @return 	   		WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int wlan_set_essid(struct net_device *dev, struct iw_request_info *info,
	       struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_SSID	ssid, *pRequestSSID;
	int			i;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWESSID: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif
    
#ifdef REASSOCIATION
	// cancel re-association timer if there's one
	if (Adapter->TimerIsSet == TRUE) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}
#endif /* REASSOCIATION */

	/* Check the size of the string */
	if (dwrq->length > IW_ESSID_MAX_SIZE + 1) {
		return -E2BIG ;
	}
	
	memset(&ssid, 0, sizeof(WLAN_802_11_SSID));

	/*
	 * Check if we asked for `any' or 'particular' 
	 */
	if (!dwrq->flags) {
		if (FindBestNetworkSsid(priv, &ssid)) {
			PRINTM(INFO, "Could not find best network\n");
			return WLAN_STATUS_SUCCESS;
		}
	} else {
		/* Set the SSID */
		memcpy(ssid.Ssid, extra, dwrq->length);
		ssid.SsidLength = dwrq->length - 1;
	}

	pRequestSSID = &ssid;

	PRINTM(INFO, "Requested new SSID = %s\n",
			(pRequestSSID->SsidLength > 0) ?
			(char *) pRequestSSID->Ssid : "NULL");

	if (!pRequestSSID->SsidLength || pRequestSSID->Ssid[0] < 0x20) {
		PRINTM(INFO, "Invalid SSID - aborting set_essid\n");
		return -EINVAL;
	}

	// If the requested SSID is not a NULL string, 
	// do ASSOCIATION or JOIN

	// if it's infrastructure mode
	if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
		PRINTM(INFO, "Got the ssid for comparision %s\n",
				pRequestSSID->Ssid);

#ifdef PROGRESSIVE_SCAN
		i = -1;
#else
		i = FindSSIDInList(Adapter, pRequestSSID, NULL,
				Wlan802_11Infrastructure);
#endif
		if (i < 0) {
			// If the SSID is not found in the list, 
			// do an specific scan with the specific SSID. 
			// This will scan for AP's which have 
			// SSID broadcasting disabled.
			PRINTM(INFO, "0 APs in scan list or SSID not found in"
					" scan list, trying active scan; " 
					"do scan first\n");

			SendSpecificScan(priv, pRequestSSID);

			i = FindSSIDInList(Adapter, pRequestSSID, NULL,
					Wlan802_11Infrastructure);
		}

		if (i >= 0){
			PRINTM(INFO, "SSID found in scan list ... associating...\n");
            
			if (Adapter->MediaConnectStatus == WlanMediaStateConnected){
				PRINTM(INFO, "Already Connected ..\n");
				ret = SendDeauthentication(priv);

				if (ret){
					LEAVE();
					return ret;
				}
			}
			
			memcpy(Adapter->CurrentBSSID,Adapter->BSSIDList[i].MacAddress,ETH_ALEN);
            
			Adapter->RequestViaBSSID = FALSE;
            
			ret = wlan_associate(priv, pRequestSSID);

			if (ret) {
				LEAVE();
				return ret;
			}
		}
        	else  /* i >= 0 */
        	{
            		return i; /* return -ENETUNREACH, passed from FindSSIDInList */
        	}
	} 
    	else 
    	{
		// it's ad hoc mode
		/* If the requested SSID matches current SSID return */
		if (!SSIDcmp(&Adapter->CurBssParams.ssid, pRequestSSID))
			return WLAN_STATUS_SUCCESS;

		SendSpecificScan(priv, pRequestSSID);

		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			/*
			 * Exit Adhoc mode 
			 */
			PRINTM(INFO, "Sending Adhoc Stop\n");
			ret = StopAdhocNetwork(priv);

			if (ret) {
				LEAVE();
				return ret;
			}

		}
		// find out the BSSID that matches the SSID given in
		// pdata_buf
		i = FindSSIDInList(Adapter, pRequestSSID, NULL, 
				Wlan802_11IBSS);

		Adapter->RequestViaBSSID = FALSE;
		
		if (i >= 0) {
			PRINTM(INFO, "SSID found at %d in List," 
					"so join\n", i);
			JoinAdhocNetwork(priv, &ssid, i);
		} else {
			// else send START command
			PRINTM(INFO, "SSID not found in list, "
					"so creating adhoc with ssid = %s\n", 
					ssid.Ssid);

			StartAdhocNetwork(priv, &ssid);
		}	// end of else (START command)
	}		// end of else (Ad hoc mode)

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief use index to get the data rate
 *   
 *  @param index                The index of data rate
 *  @return 	   		data rate or 0 
 */
u32 index_to_data_rate(u8 index)
{
	if (index >= sizeof(WlanDataRates))
		index = 0;

	return WlanDataRates[index];
}

/** 
 *  @brief use rate to get the index
 *   
 *  @param rate                 data rate
 *  @return 	   		index or 0 
 */
u8 data_rate_to_index(u32 rate)
{
	u8	*ptr;

	if (rate)
		if ((ptr = wlan_memchr(WlanDataRates, (u8)rate, 
						sizeof(WlanDataRates))))
			return (ptr - WlanDataRates);

	return 0;
}


/** 
 *  @brief set data rate
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_rate(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	u32			data_rate;
	u16			action;
	int 			ret = WLAN_STATUS_SUCCESS;
	WLAN_802_11_RATES	rates;
	u8			*rate;
    
	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWRATE: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	PRINTM(INFO, "Vwrq->value = %d\n", vwrq->value);

	if (vwrq->value == -1) {
		action = HostCmd_ACT_SET_TX_AUTO;	// Auto
		Adapter->Is_DataRate_Auto = TRUE;
		Adapter->DataRate = 0;
	} else {
		if (vwrq->value % 100000) {
			return -EINVAL;
		}

		data_rate = vwrq->value / 500000;

		memset(rates, 0, sizeof(rates));
		get_active_data_rates(Adapter, rates);
		rate = rates;
		while (*rate) {
			PRINTM(INFO, "Rate=0x%X  Wanted=0x%X\n",*rate,data_rate);
			if ((*rate & 0x7f) == (data_rate & 0x7f))
				break;
			rate++;
		}
		if (!*rate) {
			PRINTM(MSG, "The fixed data rate 0x%X is out "
					"of range.\n",data_rate);
			return -EINVAL;
		}

		Adapter->DataRate = data_rate;
		action = HostCmd_ACT_SET_TX_FIX_RATE;
		Adapter->Is_DataRate_Auto = FALSE;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DATA_RATE, 
			action,	HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	LEAVE();
	return ret;
}

/** 
 *  @brief get data rate
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_get_rate(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->Is_DataRate_Auto) {
		vwrq->fixed = 0;
	} else {
		vwrq->fixed = 1;
	}
	
	vwrq->value = Adapter->DataRate * 500000;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief set wireless mode 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_mode(struct net_device *dev,
	      struct iw_request_info *info, u32 * uwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	
	WLAN_802_11_NETWORK_INFRASTRUCTURE	WantedMode;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWMODE: IOCTLS called when station is in DeepSleep\n");
		return -EBUSY;
	}
#endif

	switch (*uwrq) {
		case IW_MODE_ADHOC:
			PRINTM(INFO, "Wanted Mode is ad-hoc: current DataRate=%#x\n",Adapter->DataRate);
			WantedMode = Wlan802_11IBSS;
#ifdef MULTI_BANDS
			if (Adapter->adhoc_start_band == BAND_A) 
#ifdef ENABLE_802_11H
				Adapter->AdhocChannel = wlan_11h_get_adhoc_start_channel(priv);
#else
				Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL_A;
#endif
			else
#endif
				Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
			break;

		case IW_MODE_INFRA:
			PRINTM(INFO, "Wanted Mode is Infrastructure\n");
			WantedMode = Wlan802_11Infrastructure;
			break;

		case IW_MODE_AUTO:
			PRINTM(INFO, "Wanted Mode is Auto\n");
			WantedMode = Wlan802_11AutoUnknown;
			break;

		default:
			PRINTM(INFO, "Wanted Mode is Unknown: 0x%x\n", *uwrq);
			return -EINVAL;
	}

	if (Adapter->InfrastructureMode == WantedMode ||
			WantedMode == Wlan802_11AutoUnknown) {
		PRINTM(INFO, "Already set to required mode! No change!\n");

		Adapter->InfrastructureMode = WantedMode;

		LEAVE();
		return WLAN_STATUS_SUCCESS;
	}

	if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
#ifdef PS_REQUIRED
		if (Adapter->PSState != PS_STATE_FULL_POWER) {
			PSWakeup(priv, HostCmd_OPTION_WAITFORRSP);
		}
		Adapter->PSMode = Wlan802_11PowerModeCAM;
#endif
	}
	
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
			ret = SendDeauthentication(priv);

			if (ret) {
				LEAVE();
				return ret;
			}
		} else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
			/* If current mode is Adhoc, clean stale information */
			ret = StopAdhocNetwork(priv);

			if (ret) {
				LEAVE();
				return ret;
			}
		}
	} 

	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled) {
		/* If there is a key with the specified SSID, 
		 * send REMOVE WEP command, to make sure we clean up
		 * the WEP keys in firmware
		 */
		ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SET_WEP,
			0, HostCmd_OPTION_WAITFORRSP
			, OID_802_11_REMOVE_WEP,
			NULL);

		if (ret) {
			LEAVE();
			return ret;
		}

		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WEP_ENABLE;

		SetMacPacketFilter(priv);
	}

	Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
	Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;

	Adapter->InfrastructureMode = WantedMode;

	ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SNMP_MIB,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_INFRASTRUCTURE_MODE,
			NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	LEAVE();
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Encryption key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_encode(struct net_device *dev,
		struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{

	PWLAN_802_11_KEY pKey = NULL;

	ENTER();
	
	if (dwrq->length > MAX_KEY_SIZE) {
		pKey = (PWLAN_802_11_KEY) extra;

		if(pKey->KeyLength <= MAX_KEY_SIZE) {
			//dynamic WEP
			dwrq->length = pKey->KeyLength;
			dwrq->flags = pKey->KeyIndex + 1;
			return wlan_set_encode_nonwpa(dev, info, dwrq, 
					pKey->KeyMaterial);
		}
		else {
#ifdef	WPA
			//WPA
			return wlan_set_encode_wpa(dev, info, dwrq, extra);
#endif
		}
	}
	else {
		//static WEP
		PRINTM(INFO, "Setting WEP\n");
		return wlan_set_encode_nonwpa(dev, info, dwrq, extra);
	}

	return -EINVAL;
}


/** 
 *  @brief set tx power 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_txpow(struct net_device *dev, struct iw_request_info *info,
			   		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private *priv= dev->priv;
	wlan_adapter *Adapter= priv->adapter;

	int wlan_radio_ioctl(wlan_private *priv, u8 option);

	u16	dbm;
	
	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWTXPOW: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif
	
	if(vwrq->disabled) {
		wlan_radio_ioctl(priv, RADIO_OFF);
		return WLAN_STATUS_SUCCESS;
	}

	Adapter->Preamble = HostCmd_TYPE_AUTO_PREAMBLE;

	wlan_radio_ioctl(priv, RADIO_ON);

#if WIRELESS_EXT > 14
	if ((vwrq->flags & IW_TXPOW_TYPE) == IW_TXPOW_MWATT) {
		dbm = (u16) mw_to_dbm(vwrq->value);
	}
	else
#endif
		dbm = (u16) vwrq->value;


	/* auto tx power control */

	if (vwrq->fixed == 0)
		dbm = 0xffff;

  	PRINTM(INFO, "<1>TXPOWER SET %d dbm.\n", dbm);

	ret = PrepareAndSendCommand(priv, 
		HostCmd_CMD_802_11_RF_TX_POWER,
		HostCmd_ACT_TX_POWER_OPT_SET_LOW,
		HostCmd_OPTION_WAITFORRSP,
		0, (void *) &dbm);

	LEAVE();
	return ret;
}



/** 
 *  @brief scan network  
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_scan_networks(wlan_private *priv)
{
	wlan_adapter	*Adapter = priv->adapter;
	int 		i;
	int		ret = WLAN_STATUS_SUCCESS;
#ifdef BG_SCAN	
	BOOLEAN         bBgScan;
#endif 	

	ENTER();

	os_stop_queue(priv);
	os_carrier_off(priv);

	Adapter->ulNumOfBSSIDs = 0;

#ifdef BG_SCAN
	bBgScan = priv->adapter->bgScanConfig->Enable;
	if(priv->adapter->bgScanConfig->Enable == TRUE) {
		wlan_bg_scan_enable(priv, FALSE);
	}
#endif /* BG_SCAN */

	for (i = 0; i < sizeof(Adapter->region_channel) / 
			sizeof(Adapter->region_channel[0]); i++) {

#ifdef ENABLE_802_11D
		if( wlan_get_state_11d( priv) == ENABLE_11D &&
			Adapter->MediaConnectStatus != WlanMediaStateConnected) {
/* Scan all the supported chan for the first scan */
			if (!Adapter->universal_channel[i].Valid)
				continue;
			Adapter->cur_region_channel = 
				&(Adapter->universal_channel[i]);

/* clear the parsed_region_chan for the first scan  */
			memset( &Adapter->parsed_region_chan, 0,
					sizeof(parsed_region_chan_11d_t) );
			
		}
		else
#endif
		{
			if (!Adapter->region_channel[i].Valid)
				continue;
			Adapter->cur_region_channel = 
					&(Adapter->region_channel[i]);
		}

		PRINTM(INFO, "cur_region_channel: NrCFP=%d\n", 
				Adapter->cur_region_channel->NrCFP );

		ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SCAN,
				0, 0, 0, NULL);

		if (ret) {
			break;
		}

		if (interruptible_sleep_on_timeout(&Adapter->scan_q,
				       			WAIT_FOR_SCAN_RRESULT_MAX_TIME) == 0)
			PRINTM(MSG, "Adapter->scan_q: timer expired\n");
	}

#ifdef BG_SCAN
	if(bBgScan == TRUE) {
		wlan_bg_scan_enable(priv, TRUE);
	}
#endif /* BG_SCAN */

	os_carrier_on(priv);
	os_start_queue(priv);

	LEAVE();
	return ret;
}

/** 
 *  @brief Scan Network
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_scan(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	union iwreq_data	wrqu;

	ENTER();

#ifdef DEEP_SLEEP
	if ((Adapter->IsDeepSleep == TRUE)) {
		PRINTM(MSG, "SIOCSIWSCAN: IOCTLS called when station is"
							" in DeepSleep\n");
		return -EBUSY;
	}
#endif

	if (!Adapter->bIsScanInProgress) {
		memset(Adapter->BSSIDList, 0,
			sizeof(WLAN_802_11_BSSID) * 
			MRVDRV_MAX_BSSID_LIST);

		if (!wlan_scan_networks(priv)) {
			memset(&wrqu, 0, sizeof(union iwreq_data));
#ifdef linux
			wireless_send_event(priv->wlan_dev.netdev, SIOCGIWSCAN, &wrqu , NULL);
#endif
		}
	}

	if (Adapter->SurpriseRemoved)
		return 	WLAN_STATUS_FAILURE;

	LEAVE();
	return 	WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get current essid 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_get_essid(struct net_device *dev, struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	/*
	 * Note : if dwrq->flags != 0, we should get the relevant SSID from
	 * the SSID list... 
	 */

	/*
	 * Get the current SSID 
	 */
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		memcpy(extra, Adapter->CurBssParams.ssid.Ssid,
				Adapter->CurBssParams.ssid.SsidLength);
		extra[Adapter->CurBssParams.ssid.SsidLength] = '\0';
	} else {
		memset(extra, 0, 32);
		extra[Adapter->CurBssParams.ssid.SsidLength] = '\0';
	}
	/*
	 * If none, we may want to get the one that was set 
	 */

	/* To make the driver backward compatible with WPA supplicant v0.2.4 */
	if (dwrq->length == 32)  /* check with WPA supplicant buffer size */
		dwrq->length = MIN(Adapter->CurBssParams.ssid.SsidLength, 
							IW_ESSID_MAX_SIZE);
	else 		
		dwrq->length = Adapter->CurBssParams.ssid.SsidLength + 1;

	dwrq->flags = 1;		/* active */

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef	linux
/** 
 *  @brief Get version 
 *   
 *  @param adapter              A pointer to wlan_adapter structure
 *  @param version		A pointer to version buffer
 *  @param maxlen		max length of version buffer
 *  @return 	   		NA
 */
void get_version(wlan_adapter *adapter, char *version, int maxlen)
{
	union {
		u32	l;
		u8	c[4];
	} ver;
	char	fwver[32];

	ver.l = adapter->FWReleaseNumber;
	if(ver.c[3] == 0)
		sprintf(fwver, "%u.%u.%u",
			ver.c[2], ver.c[1], ver.c[0]);
	else
		sprintf(fwver, "%u.%u.%u.p%u",
			ver.c[2], ver.c[1], ver.c[0],ver.c[3]);

	snprintf(version, maxlen, driver_version, fwver);
}

#ifdef BULVERDE_SDIO
/** 
 *  @brief Enable/Disable SDIO clock 
 *   
 *  @param dev                  A pointer to wlan_private structure
 *  @param on               	TRUE-on , FALSE-off 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_sdio_clock(wlan_private *priv, BOOLEAN on)
{
	if (on)
		return start_bus_clock(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);
	else
		return stop_bus_clock_2(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);
}
#endif /* BULVERDE_SDIO */
#endif //linux

