/**
*** @file wlan_meas.h
**
**  @brief Interface for the measurement module implemented in wlan_meas.c
**
**  Driver interface functions and type declarations for the measurement module
**    implemented in wlan_meas.c
**  
**  @sa wlan_meas.c
**
**  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
*/

/*****************************************************************************/
/*  Change Log:
** 
**      9/30/05: Initial revision            
**
**
*/
/*****************************************************************************/

#ifndef _WLAN_MEAS_H
#define _WLAN_MEAS_H

#include  "wlan_defs.h"

/**        
*** @brief Include extra 11h measurement types (not currently supported 
***                                             in firmware)
**/        
#define  WLAN_MEAS_EXTRA_11H_TYPES   0

/**        
*** @brief Include extra 11k measurement types (not currently implemented)
**/        
#define  WLAN_MEAS_EXTRA_11K_TYPES   0

/**        
*** @brief Enumeration of measurement types, including max supported 
***        enum for 11h/11k
**/        
typedef enum
{
    WLAN_MEAS_BASIC             = 0,  //!< 11h: Basic
#if WLAN_MEAS_EXTRA_11H_TYPES
    /* Not supported in firmware */
    WLAN_MEAS_CCA               = 1,  //!< 11h: CCA
    WLAN_MEAS_RPI               = 2,  //!< 11h: RPI
#endif
#if WLAN_MEAS_EXTRA_11K_TYPES
    /* Future 11k extensions */
    WLAN_MEAS_CHAN_LOAD         = 3,  //!< 11k: Channel Load
    WLAN_MEAS_NOISE_HIST        = 4,  //!< 11k: Noise Histogram 
    WLAN_MEAS_BEACON            = 5,  //!< 11k: Beacon 
    WLAN_MEAS_FRAME             = 6,  //!< 11k: Frame
    WLAN_MEAS_HIDDEN_NODE       = 7,  //!< 11k: Hidden Node
    WLAN_MEAS_MEDIUM_SENS_HIST  = 8,  //!< 11k: Medium Sense Histogram
    WLAN_MEAS_STA_STATS         = 9,  //!< 11k: Station Statistics
#endif

    WLAN_MEAS_NUM_TYPES,              //!< Number of enumerated measurements

#if WLAN_MEAS_EXTRA_11H_TYPES
    WLAN_MEAS_11H_MAX_TYPE      = WLAN_MEAS_RPI,       //!< Max 11h measurement
#else
    WLAN_MEAS_11H_MAX_TYPE      = WLAN_MEAS_BASIC,     //!< Max 11h measurement
#endif

#if WLAN_MEAS_EXTRA_11K_TYPES
    /* Future 11k extensions */
    WLAN_MEAS_11K_MAX_TYPE      = WLAN_MEAS_STA_STATS, //!< Max 11k measurement
#endif

} __ATTRIB_PACK__ MeasType_t;


/**        
*** @brief Mode octet of the measurement request element (7.3.2.21)
**/        
typedef struct
{
#ifdef BIG_ENDIAN
    u8 rsvd5_7           : 3;  //!< Reserved
    u8 durationMandatory : 1;  //!< 11k: duration spec. for meas. is mandatory
    u8 report            : 1;  //!< 11h: en/disable report rcpt. of spec. type
    u8 request           : 1;  //!< 11h: en/disable requests of specified type
    u8 enable            : 1;  //!< 11h: enable report/request bits
    u8 parallel          : 1;  //!< 11k: series or parallel with previous meas
#else
    u8 parallel          : 1;  //!< 11k: series or parallel with previous meas
    u8 enable            : 1;  //!< 11h: enable report/request bits
    u8 request           : 1;  //!< 11h: en/disable requests of specified type
    u8 report            : 1;  //!< 11h: en/disable report rcpt. of spec. type
    u8 durationMandatory : 1;  //!< 11k: duration spec. for meas. is mandatory
    u8 rsvd5_7           : 3;  //!< Reserved
#endif

} __ATTRIB_PACK__ MeasReqMode_t;

/**        
*** @brief Common measurement request structure (7.3.2.21.1 to 7.3.2.21.3)
**/        
typedef struct
{
    u8   channel;    //!< Channel to measure
    u64  startTime;  //!< TSF Start time of measurement (0 for immediate)
    u16  duration;   //!< TU duration of the measurement
    
} __ATTRIB_PACK__ MeasReqCommonFormat_t;

/**        
*** @brief Basic measurement request structure (7.3.2.21.1)
**/        
typedef MeasReqCommonFormat_t MeasReqBasic_t;

/**        
*** @brief CCA measurement request structure (7.3.2.21.2)
**/        
typedef MeasReqCommonFormat_t MeasReqCCA_t;

/**        
*** @brief RPI measurement request structure (7.3.2.21.3)
**/        
typedef MeasReqCommonFormat_t MeasReqRPI_t;


/**        
*** @brief Union of the availble measurement request types.  Passed in the 
***        driver/firmware interface.
**/        
typedef union
{
    MeasReqBasic_t basic; //!< Basic measurement request
    MeasReqCCA_t   cca;   //!< CCA measurement request
    MeasReqRPI_t   rpi;   //!< RPI measurement request

} MeasRequest_t;

               
/**        
*** @brief Mode octet of the measurement report element (7.3.2.22)
**/        
typedef struct 
{
#ifdef BIG_ENDIAN
    u8  rsvd3_7   : 5;   //!< Reserved
    u8  refused   : 1;   //!< Measurement refused
    u8  incapable : 1;   //!< Incapable of performin measurement
    u8  late      : 1;   //!< Start TSF time missed for measurement
#else
    u8  late      : 1;   //!< Start TSF time missed for measurement
    u8  incapable : 1;   //!< Incapable of performin measurement
    u8  refused   : 1;   //!< Measurement refused
    u8  rsvd3_7   : 5;   //!< Reserved
#endif

} __ATTRIB_PACK__ MeasRptMode_t;


/**
***  @brief Map octet of the basic measurment report (7.3.2.22.1)
**/        
typedef struct 
{
#ifdef BIG_ENDIAN
    u8  rsvd5_7         : 3;   //!< Reserved
    u8  unmeasured      : 1;   //!< Channel is unmeasured
    u8  radar           : 1;   //!< Radar detected on channel
    u8  unidentifiedSig : 1;   //!< Unidentified signal found on channel
    u8  OFDM_Preamble   : 1;   //!< OFDM preamble detected on channel
    u8  BSS             : 1;   //!< At least one valid MPDU received on channel
#else
    u8  BSS             : 1;   //!< At least one valid MPDU received on channel
    u8  OFDM_Preamble   : 1;   //!< OFDM preamble detected on channel
    u8  unidentifiedSig : 1;   //!< Unidentified signal found on channel
    u8  radar           : 1;   //!< Radar detected on channel
    u8  unmeasured      : 1;   //!< Channel is unmeasured
    u8  rsvd5_7         : 3;   //!< Reserved
#endif
    
} __ATTRIB_PACK__ MeasRptBasicMap_t;


/**        
*** @brief Basic measurement report (7.3.2.22.1)
**/        
typedef struct
{
    u8   channel;              //!< Channel to measured
    u64  startTime;            //!< Start time (TSF) of measurement
    u16  duration;             //!< Duration of measurement in TUs
    MeasRptBasicMap_t  map;    //!< Basic measurement report
    
} __ATTRIB_PACK__ MeasRptBasic_t;
    

/**        
*** @brief CCA measurement report (7.3.2.22.2)
**/        
typedef struct
{
    u8   channel;              //!< Channel to measured            
    u64  startTime;            //!< Start time (TSF) of measurement
    u16  duration;             //!< Duration of measurement in TUs 
    u8   busyFraction;         //!< Fractional duration CCA indicated chan busy
    
} __ATTRIB_PACK__ MeasRptCCA_t;
    

/**        
*** @brief RPI measurement report (7.3.2.22.3)
**/        
typedef struct
{
    u8   channel;            //!< Channel to measured            
    u64  startTime;          //!< Start time (TSF) of measurement
    u16  duration;           //!< Duration of measurement in TUs 
    u8   density[8];         //!< RPI Density histogram report
    
} __ATTRIB_PACK__ MeasRptRPI_t;


/**        
*** @brief Union of the availble measurement report types.  Passed in the 
***        driver/firmware interface.
**/        
typedef union
{
    MeasRptBasic_t  basic;   //!< Basic measurement report
    MeasRptCCA_t    cca;     //!< CCA measurement report
    MeasRptRPI_t    rpi;     //!< RPI measurement report

} MeasReport_t;



/**        
*** @brief Structure passed to firmware to perform a measurement
**/        
typedef struct 
{
    u8            macAddr[MRVDRV_ETH_ADDR_LEN];  //!< Reporting STA address  
    u8            dialogToken;                   //!< Measurement dialog toke
    MeasReqMode_t reqMode;                       //!< Report mode            
    MeasType_t    measType;                      //!< Measurement type       
    MeasRequest_t req;                           //!< Measurement request data

} __ATTRIB_PACK__ HostCmd_DS_MEASUREMENT_REQUEST;


/**        
*** @brief Structure passed back from firmware with a measurement report,
***        also can be to send a measurement report to another STA
**/        
typedef struct
{
    u8            macAddr[MRVDRV_ETH_ADDR_LEN];  //!< Reporting STA address
    u8            dialogToken;                   //!< Measurement dialog token
    MeasRptMode_t rptMode;                       //!< Report mode 
    MeasType_t    measType;                      //!< Measurement type
    MeasReport_t  rpt;                           //!< Measurement report data

} __ATTRIB_PACK__ HostCmd_DS_MEASUREMENT_REPORT;


#ifndef WLANCONFIG
/**
*** @brief Driver measurement state held in wlan_priv structure
**  
**  Used to record a measurement request that the driver is pending on 
**    the result (received measurement report).
*/
typedef struct
{
    /**
    *** Dialog token of a pending measurement request/report.  Used to
    ***   block execution while waiting for the specific dialog token
    **/
    u8 meas_rpt_pend_on;

    /**
    *** Measurement report received from the firmware that we were pending on
    **/
    HostCmd_DS_MEASUREMENT_REPORT meas_rpt_returned;

    /**
    *** OS wait queue used to suspend the requesting thread
    **/
    wait_queue_head_t  meas_rpt_waitQ;

} wlan_meas_state_t;


/* Initialize the measurement code on startup */
extern void wlan_meas_init( wlan_private *priv  );

/* Send a given measurement request to the firmware, report back the result */
extern int
wlan_meas_util_send_req( wlan_private* priv, 
                         HostCmd_DS_MEASUREMENT_REQUEST* pMeasReq,
                         int wait_for_resp_timeout,
                         HostCmd_DS_MEASUREMENT_REPORT* pMeasRpt );

/* Setup a measurement command before it is sent to the firmware */
extern int wlan_meas_cmd_process( wlan_private* priv, 
                                  HostCmd_DS_COMMAND* pCmdPtr, 
                                  const void* pInfoBuf);

/* Handle a given measurement command response from the firmware */
extern int wlan_meas_cmdresp_process( wlan_private* priv,
                                      const HostCmd_DS_COMMAND* resp );

/* Process an application ioctl for sending a measurement request */
extern int wlan_meas_ioctl_send_req( wlan_private* priv, 
                                     struct iwreq* wrq );

#endif /* WLANCONFIG */

#endif
