/** @file wlan_main.c
  *  
  * @brief This file contains the major functions in WLAN
  * driver. It includes init, exit, open, close and main
  * thread etc..
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  * 
  */
/**
  * @mainpage M-WLAN Linux Driver
  *
  * @section overview_sec Overview
  *
  * The M-WLAN is a Linux reference driver for Marvell
  * 802.11 (a/b/g) WLAN chipset.
  * 
  * @section copyright_sec Copyright
  *
  * Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  *
  */
/********************************************************
Change log:
	09/30/05: Add Doxygen format comments
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

#ifdef CONFIG_PM
static struct pm_dev *wlan_pm_dev = NULL;
#endif 

#define WLAN_TX_PWR_DEFAULT		20 		/*100mW*/

#define WLAN_TX_PWR_US_DEFAULT		20 		/*100mW*/
#define WLAN_TX_PWR_JP_DEFAULT		16 		/*50mW*/
#define WLAN_TX_PWR_FR_DEFAULT		20 		/*100mW*/
#define WLAN_TX_PWR_EMEA_DEFAULT	20 		/*100mW*/

/* Format { Channel, Frequency (MHz), MaxTxPower } */
/* Band: 'B/G', Region: USA FCC/Canada IC */
static CHANNEL_FREQ_POWER channel_freq_power_US_BG[] = {
			{1, 2412, WLAN_TX_PWR_US_DEFAULT},   
			{2, 2417, WLAN_TX_PWR_US_DEFAULT},
			{3, 2422, WLAN_TX_PWR_US_DEFAULT}, 
			{4, 2427, WLAN_TX_PWR_US_DEFAULT}, 
			{5, 2432, WLAN_TX_PWR_US_DEFAULT}, 
			{6, 2437, WLAN_TX_PWR_US_DEFAULT}, 
			{7, 2442, WLAN_TX_PWR_US_DEFAULT}, 
			{8, 2447, WLAN_TX_PWR_US_DEFAULT}, 
			{9, 2452, WLAN_TX_PWR_US_DEFAULT},
			{10, 2457, WLAN_TX_PWR_US_DEFAULT},
			{11, 2462, WLAN_TX_PWR_US_DEFAULT}
};

/* Band: 'B/G', Region: Europe ETSI */
static CHANNEL_FREQ_POWER channel_freq_power_EU_BG[] = {
			{1, 2412, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{2, 2417, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{3, 2422, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{4, 2427, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{5, 2432, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{6, 2437, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{7, 2442, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{8, 2447, WLAN_TX_PWR_EMEA_DEFAULT},
			{9, 2452, WLAN_TX_PWR_EMEA_DEFAULT},
			{10, 2457, WLAN_TX_PWR_EMEA_DEFAULT},
			{11, 2462, WLAN_TX_PWR_EMEA_DEFAULT},
			{12, 2467, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{13, 2472, WLAN_TX_PWR_EMEA_DEFAULT} 
};

/* Band: 'B/G', Region: Spain */
static CHANNEL_FREQ_POWER channel_freq_power_SPN_BG[] = {
			{10, 2457, WLAN_TX_PWR_DEFAULT}, 
			{11, 2462, WLAN_TX_PWR_DEFAULT} 
};

/* Band: 'B/G', Region: France */
static CHANNEL_FREQ_POWER channel_freq_power_FR_BG[] = {
			{10, 2457, WLAN_TX_PWR_FR_DEFAULT}, 
			{11, 2462, WLAN_TX_PWR_FR_DEFAULT},
			{12, 2467, WLAN_TX_PWR_FR_DEFAULT}, 
			{13, 2472, WLAN_TX_PWR_FR_DEFAULT} 
};

/* Band: 'B/G', Region: Japan */
static CHANNEL_FREQ_POWER channel_freq_power_JPN_BG[] = {
			{1, 2412, WLAN_TX_PWR_JP_DEFAULT}, 
			{2, 2417, WLAN_TX_PWR_JP_DEFAULT}, 
			{3, 2422, WLAN_TX_PWR_JP_DEFAULT}, 
			{4, 2427, WLAN_TX_PWR_JP_DEFAULT}, 
			{5, 2432, WLAN_TX_PWR_JP_DEFAULT}, 
			{6, 2437, WLAN_TX_PWR_JP_DEFAULT}, 
			{7, 2442, WLAN_TX_PWR_JP_DEFAULT}, 
			{8, 2447, WLAN_TX_PWR_JP_DEFAULT},
			{9, 2452, WLAN_TX_PWR_JP_DEFAULT}, 
			{10, 2457, WLAN_TX_PWR_JP_DEFAULT},
			{11, 2462, WLAN_TX_PWR_JP_DEFAULT}, 
			{12, 2467, WLAN_TX_PWR_JP_DEFAULT},
			{13, 2472, WLAN_TX_PWR_JP_DEFAULT}, 
			{14, 2484, WLAN_TX_PWR_JP_DEFAULT} 
};

#ifdef MULTI_BANDS
/* Band: 'A', Region: USA FCC, Canada IC, Spain, France */
static CHANNEL_FREQ_POWER channel_freq_power_A[] = {
			{36, 5180, WLAN_TX_PWR_US_DEFAULT}, 
			{40, 5200, WLAN_TX_PWR_US_DEFAULT}, 
			{44, 5220, WLAN_TX_PWR_US_DEFAULT}, 
			{48, 5240, WLAN_TX_PWR_US_DEFAULT}, 
			{52, 5260, WLAN_TX_PWR_US_DEFAULT}, 
			{56, 5280, WLAN_TX_PWR_US_DEFAULT}, 
			{60, 5300, WLAN_TX_PWR_US_DEFAULT}, 
			{64, 5320, WLAN_TX_PWR_US_DEFAULT}, 
#if defined(FCC_DFS_APPROVAL) && defined(ENABLE_802_11H)
            /* FCC_DFS_APPROVAL place holder -- remove when appropriate */
			{100, 5500, WLAN_TX_PWR_US_DEFAULT}, 
			{104, 5520, WLAN_TX_PWR_US_DEFAULT}, 
			{108, 5540, WLAN_TX_PWR_US_DEFAULT}, 
			{112, 5560, WLAN_TX_PWR_US_DEFAULT}, 
			{116, 5580, WLAN_TX_PWR_US_DEFAULT}, 
			{120, 5600, WLAN_TX_PWR_US_DEFAULT}, 
			{124, 5620, WLAN_TX_PWR_US_DEFAULT}, 
			{128, 5640, WLAN_TX_PWR_US_DEFAULT}, 
			{132, 5660, WLAN_TX_PWR_US_DEFAULT}, 
			{136, 5680, WLAN_TX_PWR_US_DEFAULT}, 
			{140, 5700, WLAN_TX_PWR_US_DEFAULT},
#endif
			{149, 5745, WLAN_TX_PWR_US_DEFAULT}, 
			{153, 5765, WLAN_TX_PWR_US_DEFAULT}, 
			{157, 5785, WLAN_TX_PWR_US_DEFAULT},
			{161, 5805, WLAN_TX_PWR_US_DEFAULT}, 
			{165, 5825, WLAN_TX_PWR_US_DEFAULT}
};
	
/* Band: 'A', Region: Europe ETSI */
static CHANNEL_FREQ_POWER channel_freq_power_EU_A[] = {
			{36, 5180, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{40, 5200, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{44, 5220, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{48, 5240, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{52, 5260, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{56, 5280, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{60, 5300, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{64, 5320, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{100, 5500, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{104, 5520, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{108, 5540, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{112, 5560, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{116, 5580, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{120, 5600, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{124, 5620, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{128, 5640, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{132, 5660, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{136, 5680, WLAN_TX_PWR_EMEA_DEFAULT}, 
			{140, 5700, WLAN_TX_PWR_EMEA_DEFAULT}
};
	
/* Band: 'A', Region: Japan */
static CHANNEL_FREQ_POWER channel_freq_power_JPN_A[] = {
			{8, 5040, WLAN_TX_PWR_JP_DEFAULT}, 
			{12, 5060, WLAN_TX_PWR_JP_DEFAULT}, 
			{16, 5080, WLAN_TX_PWR_JP_DEFAULT},
			{34, 5170, WLAN_TX_PWR_JP_DEFAULT}, 
			{38, 5190, WLAN_TX_PWR_JP_DEFAULT}, 
			{42, 5210, WLAN_TX_PWR_JP_DEFAULT}, 
			{46, 5230, WLAN_TX_PWR_JP_DEFAULT}, 
};
#endif /* MULTI BANDS */

/********************************************************
		Global Variables
********************************************************/

/**
 * the structure for channel, frequency and power
 */
typedef struct _region_cfp_table { 
	u8 region;
	CHANNEL_FREQ_POWER *cfp_BG;
	int cfp_no_BG;
#ifdef MULTI_BANDS
	CHANNEL_FREQ_POWER *cfp_A;
	int cfp_no_A;
#endif /* MULTI BANDS */
} region_cfp_table_t;

/**
 * the structure for the mapping between region and CFP
 */
region_cfp_table_t region_cfp_table[] = {
	{ 0x10, /*US FCC*/ 
		channel_freq_power_US_BG, 
		sizeof(channel_freq_power_US_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_A,  
		sizeof(channel_freq_power_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
	{ 0x20,	/*CANADA IC*/
		channel_freq_power_US_BG, 
		sizeof(channel_freq_power_US_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_A,  
		sizeof(channel_freq_power_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
	{ 0x30,	/*EU*/
		channel_freq_power_EU_BG, 
		sizeof(channel_freq_power_EU_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_EU_A,  
		sizeof(channel_freq_power_EU_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
	{ 0x31,	/*SPAIN*/
		channel_freq_power_SPN_BG, 
		sizeof(channel_freq_power_SPN_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_A,  
		sizeof(channel_freq_power_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
	{ 0x32,	/*FRANCE*/
		channel_freq_power_FR_BG, 
		sizeof(channel_freq_power_FR_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_A,  
		sizeof(channel_freq_power_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
	{ 0x40,	/*JAPAN*/
		channel_freq_power_JPN_BG, 
		sizeof(channel_freq_power_JPN_BG)/sizeof(CHANNEL_FREQ_POWER),
#ifdef MULTI_BANDS
      		channel_freq_power_JPN_A,  
		sizeof(channel_freq_power_JPN_A)/sizeof(CHANNEL_FREQ_POWER),
#endif /* MULTI BANDS */
	},
/*Add new region here */	
};

/**
 * the rates supported by the card
 */
u8	WlanDataRates[WLAN_SUPPORTED_RATES] = 
		{ 0x02, 0x04, 0x0B, 0x16, 0x00, 0x0C, 0x12, 
		  0x18, 0x24, 0x30, 0x48, 0x60, 0x6C, 0x00 };

#ifdef MULTI_BANDS
u8	SupportedRates[A_SUPPORTED_RATES];

/**
 * the rates supported in B mode
 */
u8	SupportedRates_B[B_SUPPORTED_RATES] =
		{ 0x82, 0x84, 0x0b, 0x16, 0, 0, 0, 0 };

/**
 * the rates supported in G mode
 */
u8	SupportedRates_G[G_SUPPORTED_RATES] =
		{ 0x82, 0x84, 0x8b, 0x96, 0x0c, 0x12, 0x18, 0x24, 0x30, 0x48, 0x60, 0x6c, 0}; 

/**
 * the rates supported in A mode
 */
u8	SupportedRates_A[A_SUPPORTED_RATES] =
		{ 0x8c, 0x12, 0x98, 0x24, 0xb0, 0x48, 0x60, 0x6c, 0};

/**
 * the rates supported in A mode for ad-hoc
 */
u8	AdhocRates_A[A_SUPPORTED_RATES] =
		{ 0x8c, 0x12, 0x98, 0x24, 0xb0, 0x48, 0x60, 0x6c, 0};
#else /* not MULTI_BANDS */
#ifdef	G_RATE
/**
 * the rates supported
 */
u8	SupportedRates[G_SUPPORTED_RATES] =
		{ 0x82, 0x84, 0x8b, 0x96, 0x0c, 0x12, 0x18, 0x24, 0x30, 0x48, 0x60, 0x6c, 0}; 
#else
/**
 * the rates supported
 */
u8	SupportedRates[B_SUPPORTED_RATES] =
		{ 0x82, 0x84, 0x0b, 0x16, 0, 0, 0, 0 };
#endif
#endif /* MULTI_BANDS */

#ifdef ADHOC_GRATE
/**
 * the rates supported for ad-hoc G mode
 */
u8	AdhocRates_G[G_SUPPORTED_RATES] =
		{ 0x82, 0x84, 0x8b, 0x96, 0x0c, 0x12, 0x18, 0x24, 0x30, 0x48, 0x60, 0x6c, 0};
#else /* ADHOC_GRATE */
/**
 * the rates supported for ad-hoc B mode
 */
u8	AdhocRates_G[4] =
		{ 0x82, 0x84, 0x8b, 0x96};
#endif /* ADHOC_GRATE */

/**
 * the rates supported for ad-hoc B mode
 */
u8	AdhocRates_B[4] =
		{ 0x82, 0x84, 0x8b, 0x96};

/**
 * the global variable of a pointer to wlan_private
 * structure variable
 */
wlan_private   *wlanpriv = NULL;

u32	DSFreqList[15] = {
	0, 2412000, 2417000, 2422000, 2427000, 2432000, 2437000, 2442000,
	2447000, 2452000, 2457000, 2462000, 2467000, 2472000, 2484000
};

/**
 * the table to keep region code
 */
u16	RegionCodeToIndex[MRVDRV_MAX_REGION_CODE] =
		{ 0x10, 0x20, 0x30, 0x31, 0x32, 0x40 };

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function opens the network device
 *  
 *  @param dev     A pointer to net_device structure
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int wlan_open(struct net_device *dev)
{
	wlan_private   *priv = (wlan_private *) dev->priv;
	wlan_adapter   *adapter = priv->adapter;

	ENTER();
	
	memcpy(dev->dev_addr, adapter->PermanentAddr, ETH_ALEN);
	
	MOD_INC_USE_COUNT;

	priv->open = TRUE;

	if (priv->adapter->MediaConnectStatus == WlanMediaStateConnected)
		os_carrier_on(priv);
	else
		os_carrier_off(priv);

	os_start_queue(priv);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function closes the network device
 *  
 *  @param dev     A pointer to net_device structure
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int wlan_close(struct net_device *dev)
{
	wlan_private   *priv = dev->priv;
	
	ENTER();

	/* Flush all the packets upto the OS before stopping */
	wlan_send_rxskbQ(priv);
	os_stop_queue(priv);
	os_carrier_off(priv);

	MOD_DEC_USE_COUNT;

	priv->open = FALSE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef CONFIG_PM

/** 
 *  @brief This function is a callback function. it is called by
 *  kernel to enter or exit power saving mode.
 *  
 *  @param pmdev   A pointer to pm_dev
 *  @param pmreq   pm_request_t
 *  @param pmdata  A pointer to pmdata
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_pm_callback(struct pm_dev *pmdev, pm_request_t pmreq,
			    void *pmdata)
{
	wlan_private 		*priv = wlanpriv;
	wlan_adapter 		*Adapter = priv->adapter;
	struct net_device 	*dev = priv->wlan_dev.netdev;
#ifdef DEEP_SLEEP
	static BOOLEAN		OS_Enable_DS = FALSE;
#endif

        PRINTM(INFO, "WPRM_PM_CALLBACK: pmreq = %d.\n", pmreq);

	switch (pmreq) {
		case PM_SUSPEND:
                        PRINTM(INFO, "WPRM_PM_CALLBACK: enter PM_SUSPEND.\n");

#ifdef WPRM_DRV
                        /* check WLAN_HOST_WAKEB */
                        if(wprm_wlan_host_wakeb_is_triggered()) {
                                PRINTM(MSG, "exit on GPIO-1 triggered.\n");
                                return WLAN_STATUS_FAILURE;
                        }
#endif
 
                        /* in associated mode */
                        if(Adapter->MediaConnectStatus == WlanMediaStateConnected) {
#ifdef PS_REQUIRED
                                if(
                                	(Adapter->PSState != PS_STATE_SLEEP)
#ifdef HOST_SLEEP
					|| !Adapter->bWakeupDevRequired
					|| (Adapter->WakeupTries != 0)
#endif
				) {
					PRINTM(INFO, "wlan_pm_callback: can't enter sleep mode\n");
                                        return WLAN_STATUS_FAILURE;
                                }
                                else
#endif
				{

                                        /*
                                         * Detach the network interface
                                         * if the network is running
                                         */
                                        if (netif_running(dev)) {
                                                netif_device_detach(dev);
                                                PRINTM(INFO, "netif_device_detach().\n");
                                        }
#ifdef BULVERDE_SDIO
                                        /* Stop SDIO bus clock */
                                        stop_bus_clock_2(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);
#endif
                                        sbi_suspend(priv);
                                }
                                break;
                        }
 
                        /* in non associated mode */

			/*
			 * Detach the network interface 
			 * if the network is running
			 */
			if (netif_running(dev))
				netif_device_detach(dev);

			/* 
			 * Storing and restoring of the regs be taken care 
			 * at the driver rest will be done at wlan driver
			 * this makes driver independent of the card
			 */
#ifdef DEEP_SLEEP
			if (Adapter->IsDeepSleep == FALSE) {
				SetDeepSleep(priv, TRUE);
				OS_Enable_DS = TRUE;
			}
#endif
			
			sbi_suspend(priv);

			break;

		case PM_RESUME:
                        /* in associated mode */
                        if(Adapter->MediaConnectStatus == WlanMediaStateConnected) {
#ifdef HOST_SLEEP
                                if(Adapter->bWakeupDevRequired == FALSE) {
                                        /* could never happen */
                                        PRINTM(MSG, "wlan_pm_callback: serious error.\n");
                                }
                                else
#endif
				{
                                        /*
                                         * Bring the inteface up first
                                         * This case should not happen still ...
                                         */
                                        sbi_resume(priv);

#ifdef BULVERDE_SDIO
                                        /*
                                         * Start SDIO bus clock
                                         */
                                        start_bus_clock(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);

#endif
                                        /*
                                         * Attach the network interface
                                         * if the network is running
                                         */
                                        if (netif_running(dev)) {
                                                netif_device_attach(dev);
                                                PRINTM(INFO, "after netif_device_attach().\n");
                                        }
                                        PRINTM(INFO, "After netif attach, in associated mode.\n");
                                }
                        	break;
                        }
 
                        /* in non associated mode */

#ifdef WPRM_DRV
                        /* Background scan support */
                        WPRM_DRV_TRACING_PRINT();
                        /* check if WLAN_HOST_WAKEB triggered, turn on SDIO_CLK */
                        if(wprm_wlan_host_wakeb_is_triggered()) { /* WLAN_HSOT_WAKEB is triggered */
                                if(wlan_sdio_clock(priv, TRUE)) {
                                        PRINTM(MSG, "wlan_pm_callback: in PM_RESUME, wlan sdio clock turn on fail\n");
                                }
                                WPRM_DRV_TRACING_PRINT();
                        }
#endif
			/*
			 * Bring the inteface up first 
			 * This case should not happen still ...
			 */

#ifdef DEEP_SLEEP
			if (OS_Enable_DS == TRUE) {
#ifdef WPRM_DRV
                                /* if need to wakeup FW, then trigger HOST_WLAN_WAKEB first */
                                wprm_trigger_host_wlan_wakeb(1);
#endif
			}
#endif

			sbi_resume(priv);

#ifdef DEEP_SLEEP
			if (OS_Enable_DS == TRUE) {
				SetDeepSleep(priv, FALSE);
				OS_Enable_DS = FALSE;
			}
#endif
		      
			if (netif_running(dev))
				netif_device_attach(dev);

                        PRINTM(INFO, "after netif attach, in NON associated mode.\n");
			break;
	}

	return WLAN_STATUS_SUCCESS;
}
#endif /* CONFIG_PM */

/** 
 *  @brief This function handles packet transmission
 *  
 *  @param skb     A pointer to sk_buff structure
 *  @param dev     A pointer to net_device structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	int             ret;
	wlan_private   *priv = dev->priv;
#ifdef OMAP1510_TIMER_DEBUG
	int            i;
#endif /* OMAP1510_TIMER_DEBUG */

    	ENTER();

#ifdef OMAP1510_TIMER_DEBUG
	for(i=0; i<10; ++i)
		tm_ind[i] = 0;

	outl(0x22, OMAP1510_TIMER1_BASE);
	outl(0xFFFFFFFF, OMAP1510_TIMER1_BASE + 0x04);
	outl(0x23, OMAP1510_TIMER1_BASE);

	times[0][(tm_ind[0])++] = inw(0xFFFEC500 + 0x08);
#endif /* OMAP1510_TIMER_DEBUG */

	if (wlan_tx_packet(priv, skb)) {
		/* Transmit failed */
		ret = WLAN_STATUS_FAILURE;
		goto done;
	} else {
		/* Transmit succeeded */
#ifdef WMM
		if (!priv->adapter->wmm.enabled) {
#endif /* WMM */
			UpdateTransStart(dev);
			os_stop_queue(priv);
#ifdef WMM
		}
#endif
	}
    
    	ret = WLAN_STATUS_SUCCESS;
done:

	LEAVE();
	return ret;
}

#ifdef linux	
/** 
 *  @brief This function handles the timeout of packet
 *  transmission
 *  
 *  @param dev     A pointer to net_device structure
 *  @return 	   n/a
 */
static void wlan_tx_timeout(struct net_device *dev)
{
	wlan_private   *priv = (wlan_private *) dev->priv;

	ENTER();

	priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
	UpdateTransStart(dev);

	if (priv->adapter->CurrentTxSkb) {
		wake_up_interruptible(&priv->MainThread.waitQ);
	}
	else {
		os_start_queue(priv);
	}

	LEAVE();
}
#endif	/*linux */

/** 
 *  @brief This function returns the network statistics
 *  
 *  @param dev     A pointer to wlan_private structure
 *  @return 	   A pointer to net_device_stats structure
 */
static struct net_device_stats *wlan_get_stats(struct net_device *dev)         
{
	wlan_private   *priv = (wlan_private *) dev->priv;

	return &priv->stats;
}

/** 
 *  @brief This function sets the MAC address to firmware.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param pRxPD   A pointer to RxPD structure of received packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_set_mac_address(struct net_device *dev, void *addr)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = (wlan_private *) dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	struct sockaddr	*pHwAddr = (struct sockaddr *)addr;
	
	ENTER();
	
	memset(Adapter->CurrentAddr, 0, MRVDRV_ETH_ADDR_LEN); 
	
	/* dev->dev_addr is 8 bytes */
	HEXDUMP("dev->dev_addr:", dev->dev_addr, ETH_ALEN);

	HEXDUMP("addr:", pHwAddr->sa_data, ETH_ALEN);
	memcpy(Adapter->CurrentAddr, pHwAddr->sa_data, ETH_ALEN);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_MAC_ADDRESS,
		HostCmd_ACT_SET,
		HostCmd_OPTION_WAITFORRSP,
		0, NULL);

	if (ret) {
		PRINTM(INFO, "set mac address failed.\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	HEXDUMP("Adapter->MacAddr:", Adapter->CurrentAddr, ETH_ALEN);
	memcpy(dev->dev_addr, Adapter->CurrentAddr, ETH_ALEN);

done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function sets multicast addresses to firmware
 *  
 *  @param dev     A pointer to net_device structure
 *  @return 	   n/a
 */
static void wlan_set_multicast_list(struct net_device *dev)
{
	wlan_private   *priv = dev->priv;
	wlan_adapter   *Adapter = priv->adapter;
	int             OldPacketFilter;

	ENTER();

	OldPacketFilter = Adapter->CurrentPacketFilter;

	if (dev->flags & IFF_PROMISC) {
		PRINTM(INFO, "Enable Promiscuous mode\n");
		Adapter->CurrentPacketFilter |= 
			HostCmd_ACT_MAC_PROMISCUOUS_ENABLE;
		Adapter->CurrentPacketFilter &=
			~(HostCmd_ACT_MAC_ALL_MULTICAST_ENABLE |
					HostCmd_ACT_MAC_MULTICAST_ENABLE);
	} else {
		/* Multicast */
		Adapter->CurrentPacketFilter &=
			~HostCmd_ACT_MAC_PROMISCUOUS_ENABLE;

		if (dev->flags & IFF_ALLMULTI || dev->mc_count > 
					MRVDRV_MAX_MULTICAST_LIST_SIZE) {
			PRINTM(INFO, "Enabling All Multicast!\n");
			Adapter->CurrentPacketFilter |=
				HostCmd_ACT_MAC_ALL_MULTICAST_ENABLE;
			Adapter->CurrentPacketFilter &=
				~HostCmd_ACT_MAC_MULTICAST_ENABLE;
		} else {
			Adapter->CurrentPacketFilter &=
				~HostCmd_ACT_MAC_ALL_MULTICAST_ENABLE;

			if (!dev->mc_count) {
				PRINTM(INFO, "No multicast addresses - "
						"disabling multicast!\n");
				Adapter->CurrentPacketFilter &=
					~HostCmd_ACT_MAC_MULTICAST_ENABLE;
			} else {
				int	i;
				
				Adapter->CurrentPacketFilter |=
					HostCmd_ACT_MAC_MULTICAST_ENABLE;

				Adapter->NumOfMulticastMACAddr = 
					CopyMulticastAddrs(Adapter, dev);

				PRINTM(INFO, "Multicast addresses: %d\n", 
							dev->mc_count);
				
				for (i = 0; i < dev->mc_count; i++) {
					PRINTM(INFO, "Multicast address %d:"
						"%x %x %x %x %x %x\n", i,
						Adapter->MulticastList[i][0],
						Adapter->MulticastList[i][1],
						Adapter->MulticastList[i][2],
						Adapter->MulticastList[i][3],
						Adapter->MulticastList[i][4],
						Adapter->MulticastList[i][5]);
				}
				/* set multicast addresses to firmware */
				PrepareAndSendCommand(priv, HostCmd_CMD_MAC_MULTICAST_ADR,
							HostCmd_ACT_GEN_SET,
							0, 0, NULL);
			}
		}
	}

	if (Adapter->CurrentPacketFilter != OldPacketFilter) {
		SetMacPacketFilter(priv);
	}

	LEAVE();
}

#ifdef REASSOCIATION
/** 
 *  @brief This function handles re-association. it is triggered
 *  by re-assoc timer.
 *  
 *  @param data    A pointer to wlan_thread structure
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int wlan_reassociation_thread(void *data)
{
	wlan_thread	*thread = data;
	wlan_private	*priv = thread->priv;
	wlan_adapter	*Adapter = priv->adapter;
	wait_queue_t	wait;

	OS_INTERRUPT_SAVE_AREA;

    	ENTER();
    
	wlan_activate_thread(thread);
	init_waitqueue_entry(&wait, current);

	for (;;) {
		add_wait_queue(&thread->waitQ, &wait);
		OS_SET_THREAD_STATE(TASK_INTERRUPTIBLE);

		PRINTM(INFO, "Re-asso Thread sleeping...\n");
		
		schedule();
	
		OS_SET_THREAD_STATE(TASK_RUNNING);
		remove_wait_queue(&thread->waitQ, &wait);

		if (thread->state == WLAN_THREAD_STOPPED)
			break;

		PRINTM(INFO, "Re-assoc Thread waking up...\n");

		if (Adapter->InfrastructureMode != Wlan802_11Infrastructure) {
			PRINTM(MSG, "Reassoc: non infra mode is not supported\n");
			continue;
		}
				
		if (Adapter->MediaConnectStatus != WlanMediaStateDisconnected) {
			PRINTM(MSG, "Reassoc: Adapter->MediaConnectStatus is wrong\n");
			continue;
		}
				
		PRINTM(INFO, "Reassoc: Required ESSID :%s\n", Adapter->PreviousSSID.Ssid);
	
		PRINTM(INFO, "Reassoc: Performing Active Scan @ %lu\n", os_time_get());
		
		SendSpecificScan(priv, &Adapter->PreviousSSID);
	
		{
			u8      *pReqBSSID = NULL;
			int	i;

			PRINTM(INFO, "Reassoc: Adapter->ulNumOfBSSIDs= %d\n", 
					Adapter->ulNumOfBSSIDs);

			if (Adapter->RequestViaBSSID)
				pReqBSSID = Adapter->RequestedBSSID;
  
			i = FindSSIDInList(Adapter, &Adapter->PreviousSSID,
					pReqBSSID, Adapter->InfrastructureMode);

			if (i >= 0) {
				PRINTM(INFO, "Reassoc: Found the AP, trying to assoc.\n");
					
				wlan_associate(priv, &Adapter->PreviousSSID);
			}		
		}

		if (Adapter->MediaConnectStatus == WlanMediaStateDisconnected) {
			PRINTM(INFO, "Reassoc: No AP found or assoc failed."
				"Restarting re-assoc Timer @ %lu\n", os_time_get());

			Adapter->TimerIsSet = TRUE;
			ModTimer(&Adapter->MrvDrvTimer, MRVDRV_TIMER_10S);
		}
	}

	wlan_deactivate_thread(thread);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* REASSOCIATION */

/** 
 *  @brief This function pops rx_skb from the rx queue.
 *  
 *  @param RxSkbQ  A pointer to rx_skb queue
 *  @return 	   A pointer to skb
 */
static struct sk_buff* wlan_pop_rx_skb(struct sk_buff *RxSkbQ)
{
	struct sk_buff* skb_data = NULL;

	if(!list_empty((struct list_head *) RxSkbQ)) {
		skb_data = RxSkbQ->next;	
		list_del((struct list_head *) RxSkbQ->next);
	}

	return skb_data;
}

/** 
 *  @brief This function hanldes the major job in WLAN driver.
 *  it handles the event generated by firmware, rx data received
 *  from firmware and tx data sent from kernel.
 *  
 *  @param data    A pointer to wlan_thread structure
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int wlan_service_main_thread(void *data)
{
	wlan_thread	*thread = data;
	wlan_private	*priv = thread->priv;
	wlan_adapter	*Adapter = priv->adapter;
	wait_queue_t    wait;
	u8              ireg = 0;

	OS_INTERRUPT_SAVE_AREA;	

	ENTER();

	wlan_activate_thread(thread);
	
	init_waitqueue_entry(&wait, current);

#ifdef ENABLE_MEAS
    wlan_meas_init(priv);
#endif

#ifdef ENABLE_802_11H
    PRINTM(INFO, "11H: init 11H\n");
    wlan_11h_init( priv );
#endif

	for (;;) {
		PRINTM(INFO, "main-thread 111: IntCounter=%d "
			"CurrentTxSkb=%p dnld_sent=%d\n",
		       	Adapter->IntCounter, 
			Adapter->CurrentTxSkb,
		       	priv->wlan_dev.dnld_sent);

		add_wait_queue(&thread->waitQ, &wait);
		OS_SET_THREAD_STATE(TASK_INTERRUPTIBLE);

		TX_DISABLE;
		
		if (
#ifdef PS_REQUIRED
			(Adapter->PSState == PS_STATE_SLEEP
#ifdef HOST_SLEEP
				&& !Adapter->bWakeupDevRequired
#endif /* HOST_SLEEP */
			) ||
#endif /* PS_REQUIRED */
			(
			!Adapter->IntCounter 
#ifdef WMM
			&& (priv->wlan_dev.dnld_sent || !Adapter->wmm.enabled || 
				wmm_lists_empty(priv))
#endif /* WMM */
			&& (priv->wlan_dev.dnld_sent || !Adapter->CurrentTxSkb)
			&& (priv->wlan_dev.dnld_sent || Adapter->CurCmd ||
				list_empty(&Adapter->CmdPendingQ))
			)
		) {
			PRINTM(INFO, "main-thread sleeping... Conn=%d IntC=%d PS_Mode=%d PS_State=%d\n",
				Adapter->MediaConnectStatus,
			       	Adapter->IntCounter,
				Adapter->PSMode,
			       	Adapter->PSState);			
#ifdef _MAINSTONE
			MST_LEDDAT1 = get_utimeofday();
#endif
			TX_RESTORE;
			schedule();
		} else {
			TX_RESTORE;
		}

		PRINTM(INFO, "main-thread 222 (waking up): IntCounter=%d CurrentTxSkb=%p "
				"dnld_sent=%d\n",
			       	Adapter->IntCounter, 
				Adapter->CurrentTxSkb,
				priv->wlan_dev.dnld_sent);

		OS_SET_THREAD_STATE(TASK_RUNNING);
		remove_wait_queue(&thread->waitQ, &wait);

		PRINTM(INFO, "main-thread 333: IntCounter=%d CurrentTxSkb=%p "
				"dnld_sent=%d\n",
			       	Adapter->IntCounter, 
				Adapter->CurrentTxSkb,
				priv->wlan_dev.dnld_sent);

		if ((thread->state == WLAN_THREAD_STOPPED) || 
					Adapter->SurpriseRemoved) {
			PRINTM(INFO, "main-thread: break from main thread: SurpriseRemoved=0x%x\n",
					Adapter->SurpriseRemoved);
			break;
		}

		if (Adapter->IntCounter) {
			OS_INT_DISABLE;
			Adapter->IntCounter = 0;
			OS_INT_RESTORE;
		
			if (sbi_get_int_status(priv, &ireg)) {
				PRINTM(INFO, "main-thread: reading HOST_INT_STATUS_REG failed\n");
				continue;
			}
			Adapter->HisRegCpy |= ireg;
		}
#ifdef HOST_SLEEP
#ifdef PS_REQUIRED
		else if (Adapter->PSState == PS_STATE_SLEEP
				&& Adapter->bWakeupDevRequired
#ifdef FW_WAKEUP_TIME
				&& (wt_pwrup_sending == 0L)
#endif
		) {
#ifdef DEEP_SLEEP
#ifdef FW_WAKEUP_TIME
			wt_pwrup_sending = get_utimeofday();
#endif
			Adapter->WakeupTries++;
			/* we borrow deep_sleep wakeup code for time being */
			if(sbi_exit_deep_sleep(priv))
				PRINTM(MSG, "main-thread: wakeup dev failed\n");
#endif //DEEP_SLEEP
			continue;
		}
#endif //PS_REQUIRED
#endif //HOST_SLEEP

		PRINTM(INFO, "main-thread 444: IntCounter=%d CurrentTxSkb=%p "
				"dnld_sent=%d\n",
			       	Adapter->IntCounter, 
				Adapter->CurrentTxSkb,
				priv->wlan_dev.dnld_sent);
		
		/* Command response? */
		if (Adapter->HisRegCpy & HIS_CmdUpLdRdy) {
			PRINTM(INFO, "main-thread: Cmd response ready.\n");

			OS_INT_DISABLE;
			Adapter->HisRegCpy &= ~HIS_CmdUpLdRdy;
			OS_INT_RESTORE;

			wlan_process_rx_command(priv);
		}

		/* Any received data? */
		if (Adapter->HisRegCpy & HIS_RxUpLdRdy) {
			PRINTM(INFO, "main-thread: Rx Packet ready.\n");
			
			OS_INT_DISABLE;
			Adapter->HisRegCpy &= ~HIS_RxUpLdRdy;
			OS_INT_RESTORE;

#ifndef THROUGHPUT_TEST
			wlan_send_rxskbQ(priv);
#else
			Adapter->NumTransfersRx += 1;
#endif /* THROUGHPUT_TEST */
		}

		/* Any Card Event */
		if (Adapter->HisRegCpy & HIS_CardEvent) {
			PRINTM(INFO, "main-thread: Card Event Activity.\n");

			OS_INT_DISABLE;
			Adapter->HisRegCpy &= ~HIS_CardEvent;
			OS_INT_RESTORE;

			if (sbi_read_event_cause(priv)) {
				PRINTM(MSG, "main-thread: sbi_read_event_cause failed.\n");
				continue;
			}
			wlan_process_event(priv);
		}

#ifdef PS_REQUIRED
		/* Check if we need to confirm Sleep Request received previously */
		if (Adapter->PSState == PS_STATE_PRE_SLEEP) {
			if (!priv->wlan_dev.dnld_sent && !Adapter->CurCmd) {
                            if(Adapter->MediaConnectStatus == WlanMediaStateConnected) {
				PRINTM(INFO, "main_thread: PRE_SLEEP--IntCounter=%d CurrentTxSkb=%p "
					"dnld_sent=%d CurCmd=%p, confirm now\n",
				       	Adapter->IntCounter, 
					Adapter->CurrentTxSkb,
					priv->wlan_dev.dnld_sent,
					Adapter->CurCmd);

				PSConfirmSleep(priv, (u16) Adapter->PSMode);
			    }
			    else {
				/* workaround for firmware sending deauth/linkloss event
					immediately after sleep request, remove this after
					firmware fixes it */
				Adapter->PSState = PS_STATE_AWAKE;
				PRINTM(MSG, "main-thread: ignore PS_SleepConfirm in non-connected state\n");
			    }
			}
		}

		/* The PS state is changed during processing of Sleep Request event above */
		if ((priv->adapter->PSState == PS_STATE_SLEEP)
			|| (priv->adapter->PSState == PS_STATE_PRE_SLEEP)
		) {
			continue;
		}
#endif

		/* Execute the next command */
		if (!priv->wlan_dev.dnld_sent && !Adapter->CurCmd) {
			ExecuteNextCommand(priv);
		}

#ifdef THROUGHPUT_TEST
		if (!priv->wlan_dev.dnld_sent && 
				(Adapter->ThruputTest & 0x01)) {
			sbi_host_to_card(priv, MVMS_DAT, G_buf, 1530);
			Adapter->NumTransfersTx += 1;
		}
#endif /* THROUGHPUT_TEST */
#ifdef WMM
		if (Adapter->wmm.enabled) {
			if (!wmm_lists_empty(priv)) {
#ifdef WMM_UAPSD
			if ((Adapter->PSState == PS_STATE_FULL_POWER)
				       	|| (Adapter->wmm.no_more_packet == 0))
#endif
				wmm_process_tx(priv);
			}
		} else {
#endif /* WMM */
			if (!priv->wlan_dev.dnld_sent && Adapter->CurrentTxSkb) {
				PRINTM(INFO, "main-thread: Tx Download Ready!\n");
				wlan_process_tx(priv);
			}
#ifdef WMM
		}
#endif
	}

	wlan_deactivate_thread(thread);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/**
 * @brief This function adds the card. it will probe the
 * card, allocate the wlan_priv and initialize the device. 
 *  
 *  @param card    A pointer to card
 *  @return 	   A pointer to wlan_private structure
 */
static wlan_private *wlan_add_card(void *card)
{
	struct net_device *dev = NULL;
	wlan_private   *priv = NULL;

	ENTER();

	/* probe the card */
	if (sbi_probe_card(card) < 0) {
		PRINTM(MSG, "NO card found!\n");
		return NULL;
	}

	/* Allocate an Ethernet device and register it */
	if (!(dev = init_etherdev(dev, sizeof(wlan_private)))) {
		PRINTM(MSG, "Init ethernet device failed!\n");
		return NULL;
	}

	priv = dev->priv;
	
	/* allocate buffer for wlan_adapter */
	if (!(priv->adapter = kmalloc(sizeof(wlan_adapter), GFP_KERNEL))) {
		PRINTM(MSG, "Allocate buffer for wlan_adapter failed!\n");
		goto err_freeadapter;
	}

	/* init wlan_adapter */
	memset(priv->adapter, 0, sizeof(wlan_adapter));

	priv->wlan_dev.netdev = dev;
	priv->wlan_dev.card = card;
	wlanpriv = priv;

	SET_MODULE_OWNER(dev);

	/* Setup the OS Interface to our functions */
	dev->open = wlan_open;
	dev->hard_start_xmit = wlan_hard_start_xmit;
	dev->stop = wlan_close;
	dev->do_ioctl = wlan_do_ioctl;
	dev->set_mac_address = wlan_set_mac_address;

#ifdef linux 

#define	WLAN_WATCHDOG_TIMEOUT	(2 * HZ)

	dev->tx_timeout = wlan_tx_timeout;
	dev->get_stats = wlan_get_stats;
	dev->watchdog_timeo = WLAN_WATCHDOG_TIMEOUT;

#ifdef	WIRELESS_EXT
	dev->get_wireless_stats = wlan_get_wireless_stats;
	dev->wireless_handlers = (struct iw_handler_def *) &wlan_handler_def;
#endif
#endif  /* linux */	

	dev->flags |= IFF_BROADCAST | IFF_MULTICAST;
	dev->set_multicast_list = wlan_set_multicast_list;

#ifdef MANF_CMD_SUPPORT
	/* Required for the mfg command */
	init_waitqueue_head(&priv->adapter->mfg_cmd_q);
#endif
    
	init_waitqueue_head(&priv->adapter->scan_q);
	init_waitqueue_head(&priv->adapter->ds_awake_q);

#ifdef CONFIG_PM
  	if(!(wlan_pm_dev = pm_register(PM_UNKNOWN_DEV, 0, wlan_pm_callback)))
		PRINTM(MSG, "Failed to register PM callback\n");
#endif

 	INIT_LIST_HEAD(&priv->adapter->CmdFreeQ);
        INIT_LIST_HEAD(&priv->adapter->CmdPendingQ);

	PRINTM(INFO, "Starting kthread...\n");
	priv->MainThread.priv = priv;
	wlan_create_thread(wlan_service_main_thread,
		       		&priv->MainThread,
				"wlan_main_service");

	ConfigureThreadPriority();

#ifdef REASSOCIATION
	priv->ReassocThread.priv = priv;
	wlan_create_thread(wlan_reassociation_thread,
		       		&priv->ReassocThread,
				"wlan_reassoc_service");
#endif /* REASSOCIATION */

	/*
	 * Register the device. Fillup the private data structure with
	 * relevant information from the card and request for the required
	 * IRQ. 
	 */

	if (sbi_register_dev(priv) < 0) {
		PRINTM(FATAL, "Failed to register wlan device!\n");
		goto err_unregisternet;
	}

#ifdef linux
	PRINTM(WARN, "%s: Marvell Wlan 802.11 Adapter "
			"revision 0x%02X at IRQ %i\n", dev->name,
			priv->adapter->chip_rev, dev->irq);

	wlan_proc_entry(priv, dev);
#ifdef PROC_DEBUG
	wlan_debug_entry(priv, dev);
#endif
#endif

	/* Get the CIS Table */
	sbi_get_cis_info(priv);
	
	/* init FW and HW */
	if (wlan_init_fw(priv)) {
		PRINTM(INFO, "Firmware Init Failed\n");
		goto err_unregisterdev;
	}

	LEAVE();
	return priv;

err_unregisterdev:
	sbi_unregister_dev(priv);
err_unregisternet:
err_freeadapter:
	unregister_netdev(dev);

	/* Stop the thread servicing the interrupts */
	wake_up_interruptible(&priv->MainThread.waitQ);
	wlan_terminate_thread(&priv->MainThread);

#ifdef REASSOCIATION
	wake_up_interruptible(&priv->ReassocThread.waitQ);
	wlan_terminate_thread(&priv->ReassocThread);
#endif /* REASSOCIATION */

	kfree(priv->adapter);
	wlanpriv = NULL;

	LEAVE();
	return NULL;
}

/** 
 *  @brief This function removes the card.
 *  
 *  @param priv    A pointer to card
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int wlan_remove_card(void *card)
{
	wlan_private		*priv = wlanpriv;
	wlan_adapter		*Adapter = priv->adapter;
	struct net_device	*dev;
#ifdef WPA
	union	iwreq_data	wrqu;
#endif

	ENTER();

	dev = priv->wlan_dev.netdev;

	wake_up_interruptible(&Adapter->scan_q);
	wake_up_interruptible(&Adapter->ds_awake_q);

	if (Adapter->CurCmd) {
		PRINTM(INFO, "Wake up current cmdwait_q\n");
		wake_up_interruptible(&Adapter->CurCmd->cmdwait_q);
	}

	Adapter->CurCmd = NULL;
	
#ifdef PS_REQUIRED
	if (Adapter->PSMode == Wlan802_11PowerModeMAX_PSP) {
		Adapter->PSMode = Wlan802_11PowerModeCAM;
		PSWakeup(priv, HostCmd_OPTION_WAITFORRSP);
	}
#endif

#ifdef WPA
	memset(wrqu.ap_addr.sa_data, 0xaa, ETH_ALEN);
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;
#ifdef linux
	wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL);
#endif
#endif

	/* Disable interrupts on the card as we cannot handle them after RESET */
	sbi_disable_host_int(priv);

	PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RESET,
			HostCmd_ACT_HALT, 0, 0, NULL);

	os_sched_timeout(200);

#ifdef CONFIG_PM
        pm_unregister(wlan_pm_dev);
#endif

	/* Flush all the packets upto the OS before stopping */
	wlan_send_rxskbQ(priv);
	os_stop_queue(priv);
	os_carrier_off(priv);

	/* Stop the thread servicing the interrupts */
	wake_up_interruptible(&priv->MainThread.waitQ);
	wlan_terminate_thread(&priv->MainThread);

#ifdef REASSOCIATION
	wake_up_interruptible(&priv->ReassocThread.waitQ);
	wlan_terminate_thread(&priv->ReassocThread);
#endif /* REASSOCIATION */

	Adapter->SurpriseRemoved = TRUE;

#ifdef linux
#ifdef PROC_DEBUG
	wlan_debug_remove(priv);
#endif
	wlan_proc_remove("wlan");
#endif
	
	PRINTM(INFO, "unregester dev\n");
    	sbi_unregister_dev(priv);

	PRINTM(INFO, "Free Adapter\n");
    	wlan_free_adapter(priv);
    
	priv->wlan_dev.netdev = NULL;
	wlanpriv = NULL;

	/* Last reference is our one */
	PRINTM(INFO, "refcnt = %d\n", atomic_read(&dev->refcnt));
	
	os_schedule(10);
	
	PRINTM(INFO, "netdev_finish_unregister: %s%s.\n", dev->name,
	       (dev->features & NETIF_F_DYNALLOC)?"":", old style");

	unregister_netdev(dev);

	PRINTM(INFO, "Unregister finish\n");

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/********************************************************
		Global Functions
********************************************************/

/**
 * @brief This function sends the rx packets to the os from the skb queue
 *
 * @param priv	A pointer to wlan_private structure
 * @return	n/a
 */
void wlan_send_rxskbQ(wlan_private *priv)
{
	struct sk_buff	*skb;

	if (priv->adapter) {
		while((skb = wlan_pop_rx_skb(&priv->adapter->RxSkbQ)))
			ProcessRxedPacket(priv, skb);
	}
}

/** 
 *  @brief This function finds the CFP in 
 *  region_cfp_table based on region and band parameter.
 *  
 *  @param region  The region code
 *  @param band	   The band
 *  @param cfp_no  A pointer to CFP number
 *  @return 	   A pointer to CFP
 */
CHANNEL_FREQ_POWER *wlan_get_region_cfp_table(u8 region, u8 band, int *cfp_no)
{
	int  i;

	ENTER();
			
	for ( i=0; i< sizeof(region_cfp_table)/sizeof(region_cfp_table_t); 
		i++ ) {
		PRINTM(INFO, "region_cfp_table[i].region=%d\n", region_cfp_table[i].region );
		if (region_cfp_table[i].region == region ) {
#ifdef MULTI_BANDS
			if (band & (BAND_B | BAND_G)) 
#endif
			{
				*cfp_no = region_cfp_table[i].cfp_no_BG;
				LEAVE();
				return region_cfp_table[i].cfp_BG;
			}
#ifdef MULTI_BANDS
			else {
				if (band & BAND_A) {
					*cfp_no = region_cfp_table[i].cfp_no_A;
					LEAVE();
					return region_cfp_table[i].cfp_A;
				}
				else {
					PRINTM(INFO, "Error Band[%x]\n", band );
					LEAVE();
					return NULL;
				}
			}
#endif
		}
	}
	
	LEAVE();	
	return NULL;
}

/** 
 *  @brief This function sets region table. 
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param region  The region code
 *  @param band	   The band
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_set_regiontable(wlan_private *priv, u8 region, u8 band)
{
	wlan_adapter	*Adapter = priv->adapter;
	int		i = 0;

	CHANNEL_FREQ_POWER *cfp;
	int 		cfp_no;

	ENTER();
	
	memset(Adapter->region_channel, 0, sizeof(Adapter->region_channel));

#ifdef MULTI_BANDS
	if (band & (BAND_B | BAND_G)) 
#endif
	{
#ifdef MULTI_BANDS
		cfp = wlan_get_region_cfp_table( region, BAND_G|BAND_B, &cfp_no);
#else
		cfp = wlan_get_region_cfp_table( region, band, &cfp_no);
#endif
		if ( cfp != NULL ) {
			Adapter->region_channel[i].NrCFP = cfp_no;
			Adapter->region_channel[i].CFP = cfp; 
		}
		else {
			PRINTM(INFO, "wrong region code %#x in Band B-G\n",region);
			return WLAN_STATUS_FAILURE;
		}
		Adapter->region_channel[i].Valid	= TRUE;
		Adapter->region_channel[i].Region	= region;
#ifdef MULTI_BANDS
		Adapter->region_channel[i].Band	= 
			(band & BAND_G) ? BAND_G : BAND_B;
#else
		Adapter->region_channel[i].Band		= band;
#endif
		i++;
	}
#ifdef MULTI_BANDS
	if (band & BAND_A) {
		cfp = wlan_get_region_cfp_table( region, BAND_A, &cfp_no);
		if ( cfp != NULL ) {
			Adapter->region_channel[i].NrCFP = cfp_no;
			Adapter->region_channel[i].CFP = cfp; 
		}
		else {
			PRINTM(INFO, "wrong region code %#x in Band A\n",region);
			return WLAN_STATUS_FAILURE;
		}
		Adapter->region_channel[i].Valid	= TRUE;
		Adapter->region_channel[i].Region	= region;
		Adapter->region_channel[i].Band		= BAND_A;
	}
#endif
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the interrupt. it will change PS
 *  state if applicable. it will wake up main_thread to handle
 *  the interrupt event as well.
 *  
 *  @param dev     A pointer to net_device structure
 *  @return 	   n/a
 */
void wlan_interrupt(struct net_device *dev)
{
	wlan_private   *priv = dev->priv;

	ENTER();

	PRINTM(INFO, "wlan_interrupt: IntCounter=%d\n",
			priv->adapter->IntCounter);

	priv->adapter->IntCounter++;

#ifdef HOST_SLEEP
	priv->adapter->WakeupTries = 0;
#endif

#ifdef PS_REQUIRED
	if(priv->adapter->PSState == PS_STATE_SLEEP) {
#ifdef BULVERDE_SDIO
		ps_sleep_confirmed = 0;
#endif
		priv->adapter->PSState = PS_STATE_AWAKE;
	}
#endif

#ifdef DEEP_SLEEP
	if (priv->adapter->IsDeepSleep == TRUE) {
		priv->adapter->IsDeepSleep = FALSE;
		PRINTM(MSG, "Interrupt received in DEEP SLEEP mode!\n");
		os_carrier_on(priv);
		os_start_queue(priv);
	}
#endif /* DEEP_SLEEP */

	wake_up_interruptible(&priv->MainThread.waitQ);

	LEAVE();
}

/** 
 *  @brief This function initializes module.
 *  
 *  @param	   n/a    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_init_module(void)
{
	int ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (sbi_register(wlan_add_card, wlan_remove_card, NULL) == NULL) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

done: 
	LEAVE();
	return ret;
}

/** 
 *  @brief This function cleans module
 *  
 *  @param priv    n/a
 *  @return 	   n/a
 */
void wlan_cleanup_module(void)
{
	ENTER();

	sbi_unregister();

	LEAVE();
}

module_init(wlan_init_module);
module_exit(wlan_cleanup_module);

MODULE_DESCRIPTION("M-WLAN Driver");
MODULE_AUTHOR("Marvell Semiconductor Inc.");
