/** @file wlan_fw.c
  * @brief This file contains the initialization for FW
  * and HW 
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

#ifdef THROUGHPUT_TEST
u8	G_buf[1600];
#endif /* THROUGHPUT_TEST */

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function downloads firmware image, gets
 *  HW spec from firmware and set basic parameters to
 *  firmware.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_setup_station_hw(wlan_private *priv)
{
	int 		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter   *adapter = priv->adapter;
	u16		dbm = MRVDRV_MAX_TX_POWER;
#ifndef THROUGHPUT_TEST
 	u8           TempAddr[MRVDRV_ETH_ADDR_LEN];
#endif /* THROUGHPUT_TEST */
	
    	ENTER();
    
	sbi_disable_host_int(priv);

#ifdef	HELPER_IMAGE
	/* Download the helper */
	ret = sbi_prog_helper(priv);
#else
	ret = sbi_prog_firmware(priv);
#endif

	if (ret) {
		PRINTM(INFO, "Bootloader in invalid state!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#ifdef	HELPER_IMAGE
	/* Download the main firmware via the helper firmware */
	if (sbi_prog_firmware_w_helper(priv)) {
		PRINTM(INFO, "Wlan Firmware download failed!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;		
	}
#endif

	/* check if the fimware is downloaded successfully or not */
	if (sbi_verify_fw_download(priv)) {
		PRINTM(INFO, "Firmware failed to be active in time!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#define RF_REG_OFFSET 0x07
#define RF_REG_VALUE  0xc8

    	sbi_enable_host_int(priv);

#ifndef THROUGHPUT_TEST

#ifdef GSPI
	g_bus_mode_reg = 0x06 \
			| B_BIT_4; /* This will be useful in B1 GSPI h/w
		       		      will not affect B0 as this bit is 
		      		      reserved. */
	g_dummy_clk_ioport = 1;
	g_dummy_clk_reg = 1;
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_GSPI_BUS_CONFIG,
			    HostCmd_ACT_GEN_SET,
			    HostCmd_OPTION_WAITFORRSP,
			    0, NULL);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	PRINTM(INFO, "Setting for %d %d\n", g_dummy_clk_ioport, g_dummy_clk_reg);
#endif /* GSPI */

	/*
	 * Read MAC address from HW
	 * permanent address is not read first before it gets set if
	 * NetworkAddress entry exists in the registry, so need to read it
	 * first
	 * 
	 * use permanentaddr as temp variable to store currentaddr if
	 * NetworkAddress entry exists
	 */
	memmove(TempAddr, adapter->CurrentAddr, MRVDRV_ETH_ADDR_LEN);
	memset(adapter->CurrentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);
	memset(adapter->PermanentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_GET_HW_SPEC, 
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	if (TempAddr[0] != 0xff) {
		/*
		 * an address was read, so need to send GET_HW_SPEC again to
		 * set the soft mac
		 */

		memmove(adapter->CurrentAddr, TempAddr, MRVDRV_ETH_ADDR_LEN);

		ret = PrepareAndSendCommand(priv, HostCmd_CMD_GET_HW_SPEC,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

		if (ret) {
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}

	SetMacPacketFilter(priv);

#ifdef PCB_REV4
	wlan_read_write_rfreg(priv);
#endif

	/* set the max tx power */
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_TX_POWER,
		HostCmd_ACT_TX_POWER_OPT_SET_LOW,
		HostCmd_OPTION_WAITFORRSP, 0,
		&dbm);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#ifdef FW_WAKEUP_METHOD
	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_FW_WAKEUP_METHOD, HostCmd_ACT_GET,
		HostCmd_OPTION_WAITFORRSP, 0,
		&priv->adapter->fwWakeupMethod);
		
	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif

#endif /* THROUGHPUT_TEST */

#ifndef THROUGHPUT_TEST
	/* Get the supported Data Rates */
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DATA_RATE,
			HostCmd_ACT_GET_TX_RATE,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#else
	memset(G_buf, 0xaa, 1600);
#endif /* THROUGHPUT_TEST */

	ret = WLAN_STATUS_SUCCESS;
done:	
	LEAVE();

	return (ret);
}

/** 
 *  @brief This function initializes timers.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void init_sync_objects(wlan_private *priv)
{
	wlan_adapter   *Adapter = priv->adapter;

	InitializeTimer(&Adapter->MrvDrvCommandTimer,
			MrvDrvCommandTimerFunction, priv);
	Adapter->CommandTimerIsSet = FALSE;

#ifdef REASSOCIATION
	/* Initialize the timer for the reassociation */
	InitializeTimer(&Adapter->MrvDrvTimer,
		       	MrvDrvTimerFunction, priv);
 	Adapter->TimerIsSet = FALSE;
#endif /* REASSOCIATION */

	return;
}

/** 
 *  @brief This function allocates buffer for the member of adapter
 *  structure like command buffer and BSSID list.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_allocate_adapter(wlan_private *priv)
{
	u32           ulBufSize;
	PWLAN_802_11_BSSID pTempBSSIDList;
	wlan_adapter   *Adapter = priv->adapter;

	/* Allocate buffer to store the BSSID list */
	ulBufSize = sizeof(WLAN_802_11_BSSID) * MRVDRV_MAX_BSSID_LIST;
	if (!(pTempBSSIDList = kmalloc(ulBufSize, GFP_KERNEL))) {
		wlan_free_adapter(priv);
		return WLAN_STATUS_FAILURE;
	}

	Adapter->BSSIDList = pTempBSSIDList;
	memset(Adapter->BSSIDList, 0, ulBufSize);

	spin_lock_init(&Adapter->QueueSpinLock);

	/* Allocate the command buffers */
	AllocateCmdBuffer(priv);

#ifdef PS_REQUIRED
	AllocatePSConfirmBuffer(priv);
#endif

	return WLAN_STATUS_SUCCESS;
}

/**
 *  @brief This function initializes the adapter structure
 *  and set default value to the member of adapter.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void wlan_init_adapter(wlan_private *priv)
{
	wlan_adapter	*Adapter = priv->adapter;
	int		i;
#ifdef BG_SCAN
	/**
	 * Default configuration for BG scan command
	 */
	static u8 dfltBgScanCfg[] = {
		0x01, 0x00,                                  	//Action
 		0x00,						//Enable
		0x03,						//BssType
		0x0E,						//ChannelsPerScan
		0x00,						//DiscardWhenFull
		0x00, 0x00,                                     //Reserved
		0x64, 0x00, 0x00, 0x00,				//Scan Interval				
		0x01, 0x00, 0x00, 0x00,		                //StoreCondition
		0x01, 0x00, 0x00, 0x00,                         //ReportConditions
		0x0E, 0x00,					//MaxScanResults

		0x01, 0x01, 0x4d, 0x00,                         //ChannelList  
		0x00, 0x0b, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x01, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x02, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x03, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x04, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x05, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x06, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x07, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x08, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x09, 0x00, 0x06, 0x00, 0x64, 0x00,
		0x00, 0x0a, 0x00, 0x06, 0x00, 0x64, 0x00
	};
#endif /* BG_SCAN */

	Adapter->ScanProbes = 0;

	Adapter->bcn_avg_factor  = DEFAULT_BCN_AVG_FACTOR;
	Adapter->data_avg_factor = DEFAULT_DATA_AVG_FACTOR;

	/* ATIM params */
  	Adapter->AtimWindow = 0;
	Adapter->ATIMEnabled = FALSE;

	Adapter->MediaConnectStatus = WlanMediaStateDisconnected;
	Adapter->LinkSpeed = MRVDRV_LINK_SPEED_1mbps;
	memset(Adapter->CurrentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);

	/* Status variables */
	Adapter->HardwareStatus = WlanHardwareStatusInitializing;

	/* scan type */
    	Adapter->ScanType = HostCmd_SCAN_TYPE_ACTIVE;

	/* scan mode */
	Adapter->ScanMode = HostCmd_BSS_TYPE_ANY;

    	/* 802.11 specific */
	Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
	for(i = 0; i < sizeof(Adapter->WepKey)/sizeof(Adapter->WepKey[0]); i++)
		memset(&Adapter->WepKey[i], 0, sizeof(MRVL_WEP_KEY));
	Adapter->CurrentWepKeyIndex = 0;
	Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
	Adapter->SecInfo.Auth1XAlg = WLAN_1X_AUTH_ALG_NONE;
	Adapter->SecInfo.EncryptionMode = CIPHER_NONE;
#ifdef ADHOCAES
	Adapter->AdhocAESEnabled = FALSE;
#endif
	Adapter->InfrastructureMode = Wlan802_11Infrastructure;
	Adapter->ulNumOfBSSIDs = 0;
	Adapter->ulCurrentBSSIDIndex = 0;
	Adapter->ulAttemptedBSSIDIndex = 0;

	Adapter->bIsScanInProgress = FALSE;

	Adapter->HisRegCpy |= HIS_TxDnLdRdy;

	memset(&Adapter->CurBssParams, 0, sizeof(Adapter->CurBssParams));

	Adapter->TxRetryCount = 4;

	/* PnP and power profile */
	Adapter->SurpriseRemoved = FALSE;

	Adapter->CurrentPacketFilter =
		HostCmd_ACT_MAC_RX_ON | HostCmd_ACT_MAC_TX_ON;

	Adapter->SetSpecificScanSSID = FALSE;	// default

	/*Default: broadcast*/
	memset( Adapter->SpecificScanBSSID, 0xff, ETH_ALEN );

#ifdef PROGRESSIVE_SCAN
	Adapter->ChannelsPerScan = MRVDRV_CHANNELS_PER_SCAN;
#endif

	Adapter->RadioOn = RADIO_ON;
#ifdef REASSOCIATION
	Adapter->Reassoc_on = TRUE;
#endif /* REASSOCIATION */
	Adapter->TxAntenna = RF_ANTENNA_2;
	Adapter->RxAntenna = RF_ANTENNA_AUTO;

	Adapter->Is_DataRate_Auto = TRUE;
	
	Adapter->FragThsd = MRVDRV_FRAG_MAX_VALUE;
	Adapter->RTSThsd = MRVDRV_RTS_MAX_VALUE;

	// set default value of capInfo.
#define SHORT_PREAMBLE_ALLOWED		1
	memset(&Adapter->capInfo, 0, sizeof(Adapter->capInfo));
	Adapter->capInfo.ShortPreamble = SHORT_PREAMBLE_ALLOWED;

	Adapter->Channel = DEFAULT_CHANNEL;
	Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;

#ifdef	PS_REQUIRED
	Adapter->PSMode = Wlan802_11PowerModeCAM;
	Adapter->MultipleDtim = MRVDRV_DEFAULT_MULTIPLE_DTIM;

	Adapter->ListenInterval = MRVDRV_DEFAULT_LISTEN_INTERVAL;
	
	Adapter->PSState = PS_STATE_FULL_POWER;
	Adapter->NeedToWakeup = FALSE;
	Adapter->LocalListenInterval = 0;	/* default value in firmware will be used */
#ifdef FW_WAKEUP_METHOD
	Adapter->fwWakeupMethod = WAKEUP_FW_UNCHANGED;
#endif
#endif

#ifdef DEEP_SLEEP
	Adapter->IsDeepSleep = FALSE;
#endif // DEEP_SLEEP

#ifdef HOST_SLEEP
	Adapter->bWakeupDevRequired = FALSE;
	Adapter->bHostSleepConfigured = FALSE;
	Adapter->WakeupTries = 0;
#endif // HOST_SLEEP

	Adapter->BeaconPeriod = 100;

	Adapter->DataRate = 0; // Initially indicate the rate as auto 

#ifdef ADHOC_GRATE
	Adapter->adhoc_grate_enabled=FALSE;
#endif

	Adapter->IntCounter = Adapter->IntCounterSaved = 0;
#ifdef WMM
	memset(&Adapter->wmm, 0, sizeof(WMM_DESC));
	for (i = 0; i < MAX_AC_QUEUES; i++)
		INIT_LIST_HEAD(&Adapter->wmm.TxSkbQ[i]);
#ifdef WMM_UAPSD
	Adapter->wmm.gen_null_pkg = TRUE; /*Enable NULL Pkg generation*/
#endif

#endif /* WMM */
	INIT_LIST_HEAD(&Adapter->RxSkbQ);
#ifdef BG_SCAN
	Adapter->bgScanConfig = kmalloc(sizeof(dfltBgScanCfg), GFP_KERNEL);
	memcpy(Adapter->bgScanConfig, dfltBgScanCfg, sizeof(dfltBgScanCfg));
	Adapter->bgScanConfigSize = sizeof(dfltBgScanCfg);
#endif /* BG_SCAN */

#ifdef ADHOCAES
	init_waitqueue_head(&Adapter->cmd_EncKey);
#endif

#ifdef WPA
    	Adapter->EncryptionStatus = Wlan802_11WEPDisabled;

	/* setup association information buffer */
	{
		PWLAN_802_11_ASSOCIATION_INFORMATION pAssoInfo;

		pAssoInfo = (PWLAN_802_11_ASSOCIATION_INFORMATION)
			Adapter->AssocInfoBuffer;

		// assume the buffer has already been zero-ed
		// no variable IE, so both request and response IE are
		// pointed to the end of the buffer, 4 byte aligned

		pAssoInfo->OffsetRequestIEs =
			pAssoInfo->OffsetResponseIEs =
			pAssoInfo->Length =
			((sizeof(WLAN_802_11_ASSOCIATION_INFORMATION) + 3) / 4) * 4;
    	}
#endif
	
	spin_lock_init(&Adapter->CurrentTxLock);

	Adapter->CurrentTxSkb = NULL;
	Adapter->PktTxCtrl = 0;
	
	return;
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function initializes firmware
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_init_fw(wlan_private *priv)
{
	int  ret = WLAN_STATUS_SUCCESS;

	ENTER();
	
	/* Allocate adapter structure */
	if ((ret = wlan_allocate_adapter(priv)) != WLAN_STATUS_SUCCESS) {
		goto done;
	}
	
	/* init adapter structure */
	wlan_init_adapter(priv);

	/* init timer etc. */
	init_sync_objects(priv);

	/* download fimrware etc. */
	if ((ret = wlan_setup_station_hw(priv)) != WLAN_STATUS_SUCCESS) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#ifdef ENABLE_802_11D
	/* init 802.11d */
	wlan_init_11d( priv );
#endif

	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function frees the structure of adapter
 *    
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void wlan_free_adapter(wlan_private *priv)
{
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

	if (!Adapter) {
		PRINTM(INFO, "Why double free adapter?:)\n");
		return;
	}
	
	PRINTM(INFO, "Free Command buffer\n");
	FreeCmdBuffer(priv);

	PRINTM(INFO, "Free CommandTimer\n");
	if (Adapter->CommandTimerIsSet) {
		CancelTimer(&Adapter->MrvDrvCommandTimer);
		FreeTimer(&Adapter->MrvDrvCommandTimer);
		Adapter->CommandTimerIsSet = FALSE;
	}

#ifdef REASSOCIATION
	PRINTM(INFO, "Free MrvDrvTimer\n");
	if (Adapter->TimerIsSet) {
		CancelTimer(&Adapter->MrvDrvTimer);
		FreeTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}
#endif /* REASSOCIATION */

#ifdef BG_SCAN
	if (Adapter->bgScanConfig) {
		kfree(Adapter->bgScanConfig);
		Adapter->bgScanConfig = NULL;
	}
#endif /* BG_SCAN */

	/* The following is needed for Threadx and are dummy in linux */
	OS_FREE_LOCK(&Adapter->CurrentTxLock);
	OS_FREE_LOCK(&Adapter->QueueSpinLock);

	PRINTM(INFO, "Free BSSIDList\n");
	if (Adapter->BSSIDList) {
		kfree(Adapter->BSSIDList);
		Adapter->BSSIDList = NULL;
	}

	PRINTM(INFO, "Free Adapter\n");

	/* Free the adapter object itself */
	kfree(Adapter);
	priv->adapter = NULL;
	LEAVE();
}

/** 
 *  @brief This function handles the timeout of command sending.
 *  It will re-send the same command again.
 *  
 *  @param FunctionContext    A pointer to FunctionContext
 *  @return 	   n/a
 */
void MrvDrvCommandTimerFunction(void *FunctionContext)
{
	wlan_private	*priv = (wlan_private *)FunctionContext;
	wlan_adapter	*Adapter = priv->adapter;
	CmdCtrlNode	*pTempNode;
	u32		flags;

	ENTER();

	PRINTM(INFO, "MrvDrvCommandTimer fired.\n");

	Adapter->CommandTimerIsSet = FALSE;

	pTempNode = Adapter->CurCmd;

	if(pTempNode == NULL){
		PRINTM(INFO, "PTempnode Empty\n");
		return ;
	}

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
	Adapter->CurCmd = NULL;
	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

	PRINTM(INFO, "Re-sending same command as it timeout...!\n");
	QueueCmd(Adapter,pTempNode, FALSE);

	wake_up_interruptible(&priv->MainThread.waitQ);

	LEAVE();
	return;
}

#ifdef REASSOCIATION
/** 
 *  @brief This function triggers re-association by waking up
 *  re-assoc thread.
 *  
 *  @param FunctionContext    A pointer to FunctionContext
 *  @return 	   n/a
 */
void MrvDrvTimerFunction(void *FunctionContext)
{
	wlan_private   *priv = (wlan_private *) FunctionContext;
	wlan_adapter   *Adapter = priv->adapter;
	OS_INTERRUPT_SAVE_AREA;

	ENTER();

	PRINTM(INFO, "MrvDrvTimer fired.\n");

#ifdef PS_REQUIRED
	if (Adapter->PSState != PS_STATE_FULL_POWER) {
		/* wait until Exit_PS command returns */
		Adapter->TimerIsSet = TRUE;
		ModTimer(&Adapter->MrvDrvTimer, MRVDRV_TIMER_1S);
		PRINTM(INFO, "MrvDrvTimerFunction(PSState=%d) waiting" 
				"for Exit_PS done\n",Adapter->PSState);
		LEAVE();
		return;
	}
#endif

	PRINTM(INFO, "Waking Up the Event Thread\n");

	wake_up_interruptible(&priv->ReassocThread.waitQ);

	LEAVE();
	return;
}
#endif /* REASSOCIATION */

#ifdef PCB_REV4
/** 
 *  @brief This function provides a special handling for PCB REV_4 board.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void wlan_read_write_rfreg(wlan_private *priv)
{
	wlan_offset_value offval;
	offval.offset = RF_REG_OFFSET;
	offval.value  = RF_REG_VALUE;

	PrepareAndSendCommand( priv, HostCmd_CMD_RF_REG_ACCESS,
			HostCmd_ACT_GEN_GET,
			HostCmd_OPTION_WAITFORRSP,
			0, &offval);

	offval.value |= 0x1; /* enable special handling for PCB_REV4 */

	PrepareAndSendCommand(priv, HostCmd_CMD_RF_REG_ACCESS,
			HostCmd_ACT_GEN_SET,
			HostCmd_OPTION_WAITFORRSP,
			0, &offval);
}
#endif

