/** @file  wlanconfig.c
  * @brief Program to configure addition paramters into the wlan driver
  * 
  *  Usage: wlanconfig <ethX> <cmd> [...] 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	10/12/05: Add Doxygen format comments
	
********************************************************/
#include	<stdio.h>
#include	<unistd.h>
#include	<time.h>
#include	<ctype.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<string.h>
#include	<stdlib.h>
#include	<errno.h>
#include	<linux/if.h>
#include	<sys/ioctl.h>
#include	<linux/wireless.h>
#include	<linux/if_ether.h>
#include 	<linux/byteorder/swab.h>


typedef unsigned char		u8;
typedef unsigned short  	u16;
typedef unsigned long		u32;
typedef unsigned long long	u64;
typedef signed char		s8;
typedef signed short  		s16;
typedef signed long		s32;

#ifdef EXTSCAN
#include	"wlan_config.h"
#endif

#ifdef 	BYTE_SWAP
#define 	cpu_to_le16(x)	__swab16(x)
#else
#define		cpu_to_le16(x)	(x)
#endif

#ifndef __ATTRIB_ALIGN__
#define __ATTRIB_ALIGN__ __attribute__((aligned(4)))
#endif

#ifndef __ATTRIB_PACK__
#define __ATTRIB_PACK__   __attribute__((packed))
#endif

#include	"wlan_defs.h"
#include	"wlan_types.h"
#include	"host.h"
#ifdef WMM
#include	"wlan_wmm.h"
#endif
#include	"hostcmd.h"
#include	"wlan_wext.h"

#include    "wlan_11h.h"
#include    "wlan_meas.h"

#ifndef MIN
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif /* MIN */

enum COMMANDS {
	CMD_RDMAC,
	CMD_WRMAC,
	CMD_RDBBP,
	CMD_WRBBP,
	CMD_RDRF,
	CMD_WRRF,
	CMD_RDEEPROM,
	CMD_CMD52R,
	CMD_CMD52W,
	CMD_CMD53R,
	CMD_CMD53W,
#ifdef SUBSCRIBE_EVENT_CTRL
	CMD_SUB_EVENT,
#endif
#ifdef BG_SCAN
	CMD_BG_SCAN_CONFIG,
#endif
#ifdef WMM
	CMD_WMM_TSPEC,
	CMD_WMM_ACK_POLICY,
	CMD_WMM_AC_WPAIE,
#endif /* WMM */
#ifdef CAL_DATA	
	CMD_CAL_DATA_EXT,
#endif
	CMD_CFREGR,
	CMD_CFREGW,
	CMD_GETRATE,
	CMD_SLEEPPARAMS,
    	CMD_BCA_TS,
	CMD_SETADHOCCH,
	CMD_GETADHOCCH,
	CMD_REASSOCIATE,
#ifdef EXTSCAN	
	CMD_EXTSCAN,
	CMD_SCAN_LIST,
#endif	

#ifdef CIPHER_TEST
  	CMD_CIPHER_TEST,
#endif
	CMD_REQUESTTPC,
	CMD_CHANSWANN,
    	CMD_POWERCAP,
    	CMD_MEASREQ
};

#define IW_MAX_PRIV_DEF		128
#define IW_PRIV_IOCTL_COUNT	(SIOCIWLASTPRIV-SIOCIWFIRSTPRIV+1)
/********************************************************
		Local Variables
********************************************************/
static s8    *commands[] = {
	"rdmac",
	"wrmac",
	"rdbbp",
	"wrbbp",
	"rdrf",
	"wrrf",
	"rdeeprom",
	"sdcmd52r",
	"sdcmd52w",
	"sdcmd53r",
	"sdcmd53w",
#ifdef SUBSCRIBE_EVENT_CTRL
	"subevent",
#endif
#ifdef BG_SCAN
	"bgscanconfig",
#endif
#ifdef WMM
	"wmmtspec",
	"wmm_ack_policy",
	"wmmparaie",
#endif /* WMM */
#ifdef CAL_DATA	
	"caldataext",
#endif
	"rdcfreg",
	"wrcfreg",
	"getrate",
	"sleepparams",
	"bca-ts",
	"setadhocch",
	"getadhocch",
    	"reassociate",
#ifdef EXTSCAN	
	"extscan",
	"getscanlist",
#endif
#ifdef CIPHER_TEST
  	"cipher_test",
#endif
	"requesttpc",
    	"chanswann",
    	"powercap",
    	"measreq"
};

static s8    *usage[] = {
	"Usage: wlanconfig <ethX> <cmd> [...]",
	"where",
	"	ethX	: wireless network interface",
	"	cmd	: rdmac, wrmac, rdbbp, wrbbp, rdrf, wrrf,\n"
	"		: sdcmd52r, sdcmd52w, sdcmd53r,\n" 
	"		: caldataext, rdcfreg, wrcfreg, rdeeprom, bgscanconfig,\n"
	"		: sleepparams, bca-ts,\n"
	"		: requesttpc, setadhocch, getadhocch,\n"
	"		: wmmparaie, wmm_ack_policy, wmmtspec\n"
	"		: reassociate\n"
	"		: requesttpc, powercap, chanswann, measreq\n"
#ifdef SUBSCRIBE_EVENT_CTRL
	"		: subevent\n"
#endif
	"	[...]	: additional parameters for read registers are",
	"		:	<offset>",
	"		: additional parameters for write registers are",
	"		:	<offset> <value>",
	"		: addition parameters for caldataext",
	"		: 	< filename >",
#ifdef SUBSCRIBE_EVENT_CTRL
	"		: additonal parameter for subevent",
	"		: 	< filename >",
#endif
	"		: additional parameters for reassociate are:",
	"		:	XX:XX:XX:XX:XX:XX YY:YY:YY:YY:YY:YY < string max 32>",
	"		:	< Current BSSID > < Desired BSSID > < Desired SSID >",
};

#define	MAX_COMMANDS	(sizeof(commands)/sizeof(commands[0]))

#ifdef BG_SCAN
u16 TLVChanSize;
u16 TLVSsidSize;
u16 TLVProbeSize;
u16 TLVSnrSize;
u16 TLVBcProbeSize;
u16 TLVNumSsidProbeSize;
u16 ActualPos = sizeof(HostCmd_DS_802_11_BG_SCAN_CONFIG);
#endif


static s32      sockfd;
static s8	DevName[IFNAMSIZ + 1];
static s32	Priv_count;
static struct iw_priv_args	Priv_args[IW_MAX_PRIV_DEF];
/*
 * Add the private commands handled by the host driver 
 * it can be either subcommand or the main command
 */
static s8 *priv_ioctl_names[] = 
{
    "getrate",
    "sleepparams",
    "bca-ts",
    "setadhocch",
    "getadhocch",
    "reassociate",
    "extscan",
    "getscanlist",
    "cipher_test",
    "requesttpc",
    "chanswann",
    "powercap",
    "measreq"
};

/* 
 * These two static array contains the main command number and the
 * subcommand number respectively
 */
static int PRIV_IOCTL[sizeof(priv_ioctl_names)/sizeof(priv_ioctl_names[0])];
static int PRIV_SUBIOCTL[sizeof(priv_ioctl_names)/sizeof(priv_ioctl_names[0])];

#define IOCTL_WLANGETRATE            PRIV_IOCTL[0]
#define SUBIOCTL_WLANGETRATE         PRIV_SUBIOCTL[0]
#define IOCTL_WLANSLEEPPARAMS        PRIV_IOCTL[1]
#define SUBIOCTL_WLANSLEEPPARAMS     PRIV_SUBIOCTL[1]
#define IOCTL_WLANBCA_TS             PRIV_IOCTL[2]
#define SUBIOCTL_WLANBCA_TS          PRIV_SUBIOCTL[2]
#define IOCTL_WLANSETADHOCCH         PRIV_IOCTL[3]
#define SUBIOCTL_WLANSETADHOCCH      PRIV_SUBIOCTL[3]
#define IOCTL_WLANGETADHOCCH         PRIV_IOCTL[4]
#define SUBIOCTL_WLANGETADHOCCH      PRIV_SUBIOCTL[4]
#define IOCTL_WLANREASSOCIATE        PRIV_IOCTL[5]
#define SUBIOCTL_WLANREASSOCIATE     PRIV_SUBIOCTL[5]
#define IOCTL_WLANEXTSCAN            PRIV_IOCTL[6]
#define SUBIOCTL_WLANEXTSCAN         PRIV_SUBIOCTL[6]
#define IOCTL_WLANSCAN               PRIV_IOCTL[7]
#define SUBIOCTL_WLANSCAN            PRIV_SUBIOCTL[7]
#define IOCTL_CIPHER_TEST            PRIV_IOCTL[8]
#define SUBIOCTL_CIPHER_TEST         PRIV_SUBIOCTL[8]
#define IOCTL_WLAN_11H_REQUESTTPC    PRIV_IOCTL[9]
#define SUBIOCTL_WLAN_11H_REQUESTTPC PRIV_SUBIOCTL[9]
#define IOCTL_WLAN_11H_CHANSWANN     PRIV_IOCTL[10]
#define SUBIOCTL_WLAN_11H_CHANSWANN  PRIV_SUBIOCTL[10]
#define IOCTL_WLAN_11H_POWERCAP      PRIV_IOCTL[11]
#define SUBIOCTL_WLAN_11H_POWERCAP   PRIV_SUBIOCTL[11]
#define IOCTL_WLAN_MEAS_REQ          PRIV_IOCTL[12]
#define SUBIOCTL_WLAN_MEAS_REQ       PRIV_SUBIOCTL[12]
/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/
static int      process_read_register(s32 cmd, s8 *stroffset);
static int      process_write_register(s32 cmd, s8 *stroffset, s8 *strvalue);
static u32	a2hex(s8 *s);
static void     hexdump(s8 *prompt, void *p, s32 len, s8 delim);
static int	process_write_cfreg(s8 *stroffset, s8 *strvalue);
static int	process_read_cfreg(s8 *stroffset);

/** 
 *  @brief convert char to hex integer
 * 
 *  @param chr 		char to convert
 *  @return      	hex integer or 0
 */
static int hexval(s32 chr)
{
	if (chr >= '0' && chr <= '9')
		return chr - '0';
	if (chr >= 'A' && chr <= 'F')
		return chr - 'A' + 10;
	if (chr >= 'a' && chr <= 'f')
		return chr - 'a' + 10;

	return 0;
}



/** 
 *  @brief convert string to integer
 * 
 *  @param ptr		A pointer to data buffer
 *  @param chr 		A pointer to return integer
 *  @return      	A pointer to next data field
 */
s8 *convert2hex(s8 *ptr, u8 *chr)
{
	u8	val;

	for (val = 0; *ptr && isxdigit(*ptr); ptr++) {
		val = (val * 16) + hexval(*ptr);
	}

	*chr = val;

	return ptr;
}

/** 
 *  @brief Get private info.
 *   
 *  @param ifname   A pointer to net name
 *  @return 	    WLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int get_private_info(const s8 *ifname)
{
	/* This function sends the SIOCGIWPRIV command which is
	 * handled by the kernel. and gets the total number of
	 * private ioctl's available in the host driver.
	 */
	struct iwreq iwr;
	int s, ret = WLAN_STATUS_SUCCESS;
	struct iw_priv_args *pPriv = Priv_args ;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		perror("socket[PF_INET,SOCK_DGRAM]");
		return WLAN_STATUS_FAILURE;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) pPriv;
	iwr.u.data.length = IW_MAX_PRIV_DEF;
	iwr.u.data.flags = 0;
	      
	if (ioctl(s, SIOCGIWPRIV, &iwr) < 0) {
		perror("ioctl[SIOCGIWPRIV]");
		ret = WLAN_STATUS_FAILURE;
	} else {
		/* Return the number of private ioctls */
		ret = iwr.u.data.length;
	}

	close(s);

	return ret;
}

/** 
 *  @brief Get Sub command ioctl number
 *   
 *  @param i        command index
 *  @sub_cmd	    A pointer to return sub-ioctl number
 *  @return 	    real IOCTL number or WLAN_STATUS_FAILURE
 */
static int marvell_get_subioctl_no(s32 i, int *sub_cmd)
{
	s32 j;

	if (Priv_args[i].cmd >= SIOCDEVPRIVATE) {
		*sub_cmd = 0;
		return Priv_args[i].cmd;
	}

	j = -1;

	/* Find the matching *real* ioctl */

	while ((++j < Priv_count) && ((Priv_args[j].name[0] != '\0') ||
			(Priv_args[j].set_args != Priv_args[i].set_args) ||
			(Priv_args[j].get_args != Priv_args[i].get_args)));

	/* If not found... */
	if (j == Priv_count) {
		printf("%s: Invalid private ioctl definition for: 0x%x\n",
					DevName, Priv_args[i].cmd);
		return  WLAN_STATUS_FAILURE;
	}

	/* Save sub-ioctl number */
	*sub_cmd = Priv_args[i].cmd;

	/* Return the real IOCTL number */
	return Priv_args[j].cmd;
}

/** 
 *  @brief Get ioctl number
 *   
 *  @param ifname   	A pointer to net name
 *  @param priv_cmd	A pointer to priv command buffer
 *  @param sub_cmd	A pointer to return sub-ioctl number
 *  @return 	        IOCTL number or WLAN_STATUS_FAILURE
 */
static int marvell_get_ioctl_no(const s8 *ifname, const s8 *priv_cmd, 
							int *sub_cmd)
{
	s32	i;

	/* Are there any private ioctls? */
	if (Priv_count <= 0) {
		/* Could skip this message ? */
		printf("%-8.8s  no private ioctls.\n", ifname);
	} else {
		//printf("%-8.8s  Available private ioctl :\n", ifname);
		for (i = 0; i < Priv_count; i++) {
			if (Priv_args[i].name[0] &&
					!strcmp(Priv_args[i].name, priv_cmd)) {

				return marvell_get_subioctl_no(i, sub_cmd);
			}
		}
	}

	return WLAN_STATUS_FAILURE;
}

/** 
 *  @brief init ioctl numbers
 *   
 *  @param ifname   	A pointer to net name
 *  @return 	        NA
 */
static void marvell_init_ioctl_numbers(const s8 *ifname)
{
	s32	i;
	
	/* Read the private ioctls */
	Priv_count = get_private_info(ifname);

	for (i = 0; i < sizeof(priv_ioctl_names)/sizeof(priv_ioctl_names[0]);
								i++) {
		PRIV_IOCTL[i] = marvell_get_ioctl_no(ifname,
					priv_ioctl_names[i], &PRIV_SUBIOCTL[i]);
	}
}

#define WLAN_MAX_RATES	14
#define	GIGA		1e9
#define	MEGA		1e6
#define	KILO		1e3

/** 
 *  @brief print bit rate
 *   
 *  @param rate  	rate to be print
 *  @param current      if current is TRUE, data rate not need convert
 *  @param fixed        not used
 *  @return 	        WLAN_STATUS_SUCCESS
 */
static int print_bitrate(double rate, s32 current, s32 fixed) 
{
	s8	scale = 'k', buf[128];
	s32	divisor = KILO;
	
	if (!current)
		rate *= 500000;

	if (rate >= GIGA) {
		scale = 'G';
		divisor = GIGA;
	} else if (rate >= MEGA) {
		scale = 'M';
		divisor = MEGA;
	}

	snprintf(buf, sizeof(buf), "%g %cb/s", rate/divisor, scale);

	if (current) {
		printf("\t  Current Bit Rate%c%s\n\n",
					(fixed) ? '=' : ':', buf);
	} else {
		printf("\t  %s\n", buf);
	}

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get Rate
 *   
 *  @return      WLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int process_get_rate(void)
{
	u32	bitrate[WLAN_MAX_RATES];
	struct iwreq	iwr;
	s32		i = 0;
	
	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) bitrate;
	iwr.u.data.length = sizeof(bitrate);
	
	if (IOCTL_WLANGETRATE <= 0) {
		return -EOPNOTSUPP;
	}
	
	if (SUBIOCTL_WLANGETRATE > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANGETRATE;
	}

	if (ioctl(sockfd, IOCTL_WLANGETRATE, &iwr) < 0) {
		perror("wlanconfig");
		return WLAN_STATUS_FAILURE;
	}

	printf("%-8.16s  %d available bit-rates :\n",
					DevName, iwr.u.data.length);
	
	for (i = 0 ;  i < iwr.u.data.length; i++) {
		print_bitrate(bitrate[i], 0, 0);
	}

	if (ioctl(sockfd, SIOCGIWRATE, &iwr)) {
		perror("wlanconfig");
		return WLAN_STATUS_FAILURE;
	}

	print_bitrate(iwr.u.bitrate.value, 1, iwr.u.bitrate.fixed);
	
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Check the Hex String
 *  @param s		A pointer to the string      
 *  @return      	0--HexString, -1--not HexString
 */
static int ishexstring(s8 *s)
{
	int ret = -1;
	s32 tmp;
	
	while(*s) {
		tmp = toupper(*s);
		if (tmp >= 'A' && tmp <= 'F') {
			ret = 0;
			break;
		}
		s++;
	}
	
	return ret;
}

/** 
 *  @brief Convert String to Integer
 *  @param buf		A pointer to the string      
 *  @return      	Integer
 */
static int atoval(s8 *buf)
{
	if (!strncasecmp(buf, "0x", 2))
		return a2hex(buf+2);
	else if (!ishexstring(buf))
		return a2hex(buf);
	else 	
		return atoi(buf);
}


/** 
 *  @brief Display sleep params
 *  @param sp		A pointer to wlan_ioctl_sleep_params_config structure    
 *  @return      	NA
 */
void display_sleep_params(wlan_ioctl_sleep_params_config *sp)
{
	printf("Sleep Params for %s:\n", sp->Action ? "set" : "get");
   	printf("----------------------------------------\n");
	printf("Error		: %u\n", sp->Error);
	printf("Offset		: %u\n", sp->Offset);
	printf("StableTime	: %u\n", sp->StableTime);
	printf("CalControl	: %u\n", sp->CalControl);
	printf("ExtSleepClk	: %u\n", sp->ExtSleepClk);
	printf("Reserved	: %u\n", sp->Reserved);
}

/** 
 *  @brief Process sleep params
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sleep_params(int argc, char *argv[])
{
	struct iwreq			iwr;
	int				ret;
	wlan_ioctl_sleep_params_config 	sp;

	if (argc < 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 sleepparams get/set <p1>"
				" <p2> <p3> <p4> <p5> <p6>\n");
		exit(1);
	}

	if (IOCTL_WLANSLEEPPARAMS <= 0) {
		return -EOPNOTSUPP;
	}

	memset(&sp, 0, sizeof(wlan_ioctl_sleep_params_config));
	if (!strcmp(argv[3], "get")) {
		sp.Action = 0;
	} else if (!strncmp(argv[3], "set", 3)) {
		if (argc != 10) {
			printf("Error: invalid no of arguments\n");
			printf("Syntax: ./wlanconfig eth1 sleepparams get/set" 
					"<p1> <p2> <p3> <p4> <p5> <p6>\n");
			exit(1);
		}

		sp.Action = 1;
		if ((ret = atoval(argv[4])) < 0)
			return -EINVAL;
		sp.Error = (u16) ret;
		if ((ret = atoval(argv[5])) < 0)
			return -EINVAL;
		sp.Offset = (u16) ret;
		if ((ret = atoval(argv[6])) < 0)
			return -EINVAL;
		sp.StableTime = (u16) ret;
		if ((ret = atoval(argv[7])) < 0)
			return -EINVAL;
		sp.CalControl = (u8) ret;
		if ((ret = atoval(argv[8])) < 0)
			return -EINVAL;
		sp.ExtSleepClk = (u8) ret;
		if ((ret = atoval(argv[9])) < 0)
			return -EINVAL;
		sp.Reserved = (u16) ret;
	} else 	{
		return -EINVAL;
	}

	memset(&iwr, 0, sizeof(iwr));
	
	if (SUBIOCTL_WLANSLEEPPARAMS > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANSLEEPPARAMS;
	}

	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &sp;
	iwr.u.data.length = sizeof(wlan_ioctl_sleep_params_config);

	if (ioctl(sockfd, IOCTL_WLANSLEEPPARAMS, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	display_sleep_params(&sp);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Display BCA Time Share Params
 *  @param sp		A point to wlan_ioctl_bca_timeshare_config structure    
 *  @return      	NA
 */
static void display_bca_ts_params(wlan_ioctl_bca_timeshare_config *bca_ts)
{
	printf("BCA Time Share Params for %s:\n", bca_ts->Action?"set" : "get");
   	printf("----------------------------------------\n");
	printf("TrafficType		: %u\n", bca_ts->TrafficType);	
	printf("TimeShareInterval	: %lu\n", bca_ts->TimeShareInterval);	
	printf("BTTime			: %lu\n", bca_ts->BTTime);	
}

/** 
 *  @brief Process BCA Time Share Params
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_bca_ts(int argc, char *argv[]) 
{
	int				ret, i;
	struct iwreq			iwr;
	wlan_ioctl_bca_timeshare_config	bca_ts;

	if (argc < 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 bca_ts get/set <p1>"
								" <p2> <p3>\n");
		exit(1);
	}

	if (IOCTL_WLANBCA_TS <= 0) {
		return -EOPNOTSUPP;
	}

	memset(&bca_ts, 0, sizeof(wlan_ioctl_bca_timeshare_config));
	
	if ((ret = atoval(argv[4])) < 0)
		return -EINVAL;
	if (ret > 1)
		return -EINVAL;
	bca_ts.TrafficType = (u16) ret; // 0 or 1
	
	if (!strcmp(argv[3], "get")) {
		bca_ts.Action = 0;
	} else if (!strncmp(argv[3], "set", 3)) {
		if (argc != 7) {
			printf("Error: invalid no of arguments\n");
			printf("Syntax: ./wlanconfig eth1 bca_ts get/set" 
							" <p1> <p2> <p3>\n");
			exit(1);
		}

		bca_ts.Action = 1;
		
		if ((ret = atoval(argv[5])) < 0)
			return -EINVAL;
		/* If value is not multiple of 10 then take the floor value */
		i = ret % 10;
		ret -= i;
		/* Valid Range for TimeShareInterval: < 20 ... 60_000 > ms */
		if (ret < 20  || ret > 60000) {
			printf("Invalid TimeShareInterval Range:"
						" < 20 ... 60000 > ms\n");
			return -EINVAL;
		}
		bca_ts.TimeShareInterval = (u32) ret;
		
		if ((ret = atoval(argv[6])) < 0)
			return -EINVAL;
		/* If value is not multiple of 10 then take the floor value */
		i = ret % 10;
		ret -= i;
		
		if (ret > bca_ts.TimeShareInterval) {
			printf("Invalid BTTime"
				"  Range: < 0 .. TimeShareInterval > ms\n");
			return -EINVAL;
		}
		bca_ts.BTTime = (u32) ret;
	} else 	{
		return -EINVAL;
	}

	memset(&iwr, 0, sizeof(iwr));
	
	if (SUBIOCTL_WLANBCA_TS > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANBCA_TS;
	}

	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &bca_ts;
	iwr.u.data.length = sizeof(wlan_ioctl_bca_timeshare_config);

	if (ioctl(sockfd, IOCTL_WLANBCA_TS, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	display_bca_ts_params(&bca_ts);

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Process request TPC
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_requesttpc(int argc, char *argv[]) 
{
	struct iwreq			iwr;
    	union 
    	{
        	wlan_ioctl_11h_tpc_req  req;
        	wlan_ioctl_11h_tpc_resp resp;
    	} tpcIoctl;

    	unsigned int mac[6];
    	unsigned int idx;
    	int numToks;

    	/*
    	** TPC Request expected to be in the following format:
    	**
    	**  <xx:xx:xx:xx:xx:xx> rateIndex timeout
    	**
   	**  where xx..xx is the destination MAC address of the TPC request
    	**        rateIndex is an optional int value for the IEEE tx rate index
    	**        timeout is an optional int value in ms for the response
    	**  
    	*/

    	/* Verify the number of arguments is between 4 and 6 */
    	if (argc < 4 || argc > 6){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

	memset(&iwr, 0, sizeof(iwr));
    	memset(&tpcIoctl, 0x00, sizeof(tpcIoctl));
    
    	/*
    	**  Scan in and set the destination mac address
    	*/
	numToks = sscanf(argv[3], "%2x:%2x:%2x:%2x:%2x:%2x",
                     mac + 0, mac + 1, mac + 2, mac + 3, mac + 4, mac + 5);

    	if (numToks != 6){
		fprintf(stderr, "Invalid number of parameters in mac address!\n");
       	 	return -EINVAL;
    	}

    	for (idx = 0; idx < sizeof(mac); idx++){
        	tpcIoctl.req.destMac[idx] = (u8)mac[idx];
    	}

    	/*
    	**  Assign the rate index if present, otherwise pass lowest 11a rate
    	**     index (12 == 6mbps)
    	*/
    	tpcIoctl.req.rateIndex = (argc >= 5) ? atoi(argv[4]) : 12;

    	/*
    	**  Assign the timeout value if present, otherwise pass 1000 ms
    	*/
    	tpcIoctl.req.timeout = (argc == 6) ? atoi(argv[5]) : 1000;

    	/* 
    	** Set up and execute the ioctl call
    	*/
    	iwr.u.data.flags = SUBIOCTL_WLAN_11H_REQUESTTPC;
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)&tpcIoctl;
	iwr.u.data.length = sizeof(tpcIoctl);

	if (ioctl(sockfd, IOCTL_WLAN_11H_REQUESTTPC, &iwr) < 0) {
		perror("wlanconfig: tpcrequest ioctl");
		return -EFAULT;
	}

    	if ( tpcIoctl.resp.statusCode == 0 ){
        	printf("tpcrequest:  txPower(%d), linkMargin(%d), rssi(%d)\n",
               	tpcIoctl.resp.txPower,
               	tpcIoctl.resp.linkMargin,
               	tpcIoctl.resp.rssi);
    	}
    	else
    	{
        	printf("tpcrequest:  failure, status = %d\n", 
               	tpcIoctl.resp.statusCode);
    	}

    	/* No error return */
    	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process Channel Switch Announcment
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_chanswann(int argc, char *argv[]) 
{
	struct iwreq			iwr;
    	wlan_ioctl_11h_chan_sw_ann  chSwAnnIoctl;

    	/*
    	** Channel Switch Announcment expected to be in the following format:
    	**
    	**   newChan switchCount switchMode
    	**
    	**  where:
    	**        newChan (u8): new channel to announce in switch action frame
    	**    switchCount (u8): Number of tbtt's until switch 
    	**                      (optional: default to 50)
    	**     switchMode (u8): 1 if announcement requires no tx until switch is 
    	**                      complete (optional: default to 1)
    	**
    	*/

    	/* Verify the number of arguments is between 4 and 6 */
    	if (argc < 4 || argc > 6){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

	memset(&iwr, 0, sizeof(iwr));
    	memset(&chSwAnnIoctl, 0x00, sizeof(chSwAnnIoctl));

    	/*
    	**  Set the desired channel
    	*/
    	chSwAnnIoctl.newChan = (u8)atoi(argv[3]);
   
    	/*
    	** Set the switch count, set to 50 TBTTs if argument not present
    	*/
    	chSwAnnIoctl.switchCount = (u8)((argc >= 5) 
                                               ? atoi(argv[4]) : 50);

    	/*
    	** Set the switch mode, set to 1 (quiet during switch) if not present
    	*/
    	chSwAnnIoctl.switchMode = (u8)((argc == 6) ? atoi(argv[5]) : 1);

    	/* 
    	** Set up and execute the ioctl call
    	*/
    	iwr.u.data.flags = SUBIOCTL_WLAN_11H_CHANSWANN;
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)&chSwAnnIoctl;
	iwr.u.data.length = sizeof(chSwAnnIoctl);

	if (ioctl(sockfd, IOCTL_WLAN_11H_CHANSWANN, &iwr) < 0) {
		perror("wlanconfig: chanswann ioctl");
		return -EFAULT;
	}

    	printf("chanswann:  success\n");

    	/* No error return */
    	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process Power Capability
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_powercap(int argc, char *argv[]) 
{
	struct iwreq			 iwr;
    	wlan_ioctl_11h_power_cap powerCapIoctl;

    	/*
    	** Power Capability is expected to be in the following format:
    	**
    	**   MinPwr MaxPwr
    	**
    	**  where:
    	**    minPwr (i8): 11h minimum local power constraint
    	**    maxPwr (i8): 11h maximum local power constraint
    	*/

    	/* Verify the number of arguments is 5 */
    	if (argc != 5){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

	memset(&iwr, 0, sizeof(iwr));
    	memset(&powerCapIoctl, 0x00, sizeof(powerCapIoctl));

    	/*
    	** Set the minimum transmit power level
   	*/
    	powerCapIoctl.minTxPower = (char)atoi(argv[3]);
    
    	/*
    	** Set the maximum transmit power level
    	*/
    	powerCapIoctl.maxTxPower = (char)atoi(argv[4]);

    	/* 
    	** Set up and execute the ioctl call
    	*/
    	iwr.u.data.flags = SUBIOCTL_WLAN_11H_POWERCAP;
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)&powerCapIoctl;
	iwr.u.data.length = sizeof(powerCapIoctl);

	if (ioctl(sockfd, IOCTL_WLAN_11H_POWERCAP, &iwr) < 0) {
		perror("wlanconfig: powercap ioctl");
		return -EFAULT;
	}

    	printf("powercap:  success\n");

    	/* No error return */
    	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process measurement request
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_measreq(int argc, char *argv[]) 
{
	struct iwreq		 iwr;
    	union
    	{
        	HostCmd_DS_MEASUREMENT_REQUEST req;
        	HostCmd_DS_MEASUREMENT_REPORT  rpt;
    	} measReqIoctl;

    	HostCmd_DS_MEASUREMENT_REQUEST* pReq = &measReqIoctl.req;
    	HostCmd_DS_MEASUREMENT_REPORT*  pRpt = &measReqIoctl.rpt;
    	unsigned int mac[6];
       	int idx;

    	memset(&measReqIoctl, 0x00, sizeof(measReqIoctl));

    	/*
    	**  Command arguments: type duration channel token macaddress
    	**
    	**  where:
    	**         type (u8): measurement type - blank or 0(basic), 1 (cca), 2(rpi)
    	**     duration (u8): Measurement duration in TUs
    	**                    (blank or 0 is default driver measurement time)
    	**      channel (u8): channel to perform measurement 
    	**                    (blank or 0 - current channel)
    	**        token (u8): Dialog token for this measurement
    	**                    (blank or 0 - autonumbered)
    	**    macaddress(u8): STA address to send request to.  
    	**                    (blank for our own STA to perform measurement)
    	**         
    	*/
    	switch (argc){
    	case 8:
        	/* Absent STA address will pass as zeros; driver will use own mac */
        	if (sscanf(argv[7], "%2x:%2x:%2x:%2x:%2x:%2x",
                   	mac + 0, mac + 1, mac + 2, mac + 3, mac + 4, mac + 5) == 6)
        	{
            		for (idx = 0; idx < (sizeof(mac)/sizeof(mac[0])); idx++){
                		pReq->macAddr[idx] = (u8)mac[idx];
            		}
        	}
        	else
        	{
            		fprintf(stderr, "Invalid number of parameters!\n");
            		return -EINVAL;
        	}
    	case 7:
        	/* Absent dialog Token will pass as zero; driver will auto-number */
        	pReq->dialogToken = (u8)atoi(argv[6]);
    	case 6:
        	/* Absent channel will pass as zero; driver will use current channel */
        	pReq->req.basic.channel = (u8)atoi(argv[5]);
    	case 5:
        	/* Absent duration will pass as zero; driver will use default */
        	pReq->req.basic.duration = (u16)atoi(argv[4]);
    	case 4:
        	/* Absent type will pass as zero; driver will interpret as BASIC */
        	pReq->measType = (MeasType_t)atoi(argv[3]);
        	break;
    	default:
        	break;
    	}
    
    	/* 
    	** Set up and execute the ioctl call
   	*/
    	iwr.u.data.flags = SUBIOCTL_WLAN_MEAS_REQ;
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)&measReqIoctl;
	iwr.u.data.length = sizeof(measReqIoctl);

	if (ioctl(sockfd, IOCTL_WLAN_MEAS_REQ, &iwr) < 0) {
		perror("wlanconfig: measreq ioctl");
		return -EFAULT;
	}

    	printf("measreq: "
        	"tok:%03u type:%d sta:%02x:%02x:%02x:%02x:%02x:%02x mode:%#x rpt:",
           	pRpt->dialogToken, pRpt->measType,
           	pRpt->macAddr[0], pRpt->macAddr[1], pRpt->macAddr[2],
           	pRpt->macAddr[3], pRpt->macAddr[4], pRpt->macAddr[5],
           	*(u8 *)&pRpt->rptMode);

    	/*
    	** Display the measurement report based on type 
    	*/
    	switch (pRpt->measType){
    	case WLAN_MEAS_BASIC:
        	printf("%u %u %#x",pRpt->rpt.basic.channel,pRpt->rpt.basic.duration,
               		*(u8*)&pRpt->rpt.basic.map);
        	break;
#if WLAN_MEAS_EXTRA_11H_TYPES
       	/* Not supported in firmware */
    	case WLAN_MEAS_CCA:
        	printf("%u %u %#x",pRpt->rpt.cca.channel,pRpt->rpt.cca.duration,
               		pRpt->rpt.cca.busyFraction);
        	break;
    	case WLAN_MEAS_RPI:
        	printf("%u %u %u,%u,%u,%u,%u,%u,%u,%u",
               		pRpt->rpt.rpi.channel, pRpt->rpt.rpi.duration,
               		pRpt->rpt.rpi.density[0], pRpt->rpt.rpi.density[1],
               		pRpt->rpt.rpi.density[2], pRpt->rpt.rpi.density[3],
               		pRpt->rpt.rpi.density[4], pRpt->rpt.rpi.density[5],
               		pRpt->rpt.rpi.density[6], pRpt->rpt.rpi.density[7]);
        	break;
#endif        
    	default:
        	break;
    	}

    	printf("\n");

    	/* No error return */
    	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process reassoication
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */                              
static int process_reassociation(int argc, char *argv[]) 
{
    	wlan_ioctl_reassociation_info reassocInfo;
    	struct iwreq			iwr;
    	unsigned int mac[6];
    	u32 idx;
    	s32 numToks;

    	/*
    	** Reassociation request is expected to be in the following format:
    	**
    	**  <xx:xx:xx:xx:xx:xx>  <yy:yy:yy:yy:yy:yy>  <ssid string>
    	**
    	**  where xx..xx is the current AP BSSID to be included in the reassoc req
    	**        yy..yy is the desired AP to send the reassoc req to
    	**        <ssid string> is the desired AP SSID to send the reassoc req to.
    	**
    	**  The current and desired AP BSSIDs are required.  
    	**  The SSID string can be omitted if the desired BSSID is provided.
    	**
    	**  If we fail to find the desired BSSID, we attempt the SSID.
    	**  If the desired BSSID is set to all 0's, the ssid string is used.
    	**  
    	*/

    	/* Verify the number of arguments is either 5 or 6 */
    	if (argc != 5 && argc != 6){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

	memset(&iwr, 0, sizeof(iwr));
    	memset(&reassocInfo, 0x00, sizeof(reassocInfo));

    	/*
    	**  Scan in and set the current AP BSSID
    	*/
	numToks = sscanf(argv[3], "%2x:%2x:%2x:%2x:%2x:%2x",
                     mac + 0, mac + 1, mac + 2, mac + 3, mac + 4, mac + 5);

    	if (numToks != 6){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

    	for (idx = 0; idx < sizeof(mac); idx++){
        	reassocInfo.CurrentBSSID[idx] = (u8)mac[idx];
    	}


    	/*
    	**  Scan in and set the desired AP BSSID
    	*/
	numToks = sscanf(argv[4], "%2x:%2x:%2x:%2x:%2x:%2x",
                     mac + 0, mac + 1, mac + 2, mac + 3, mac + 4, mac + 5);

    	if (numToks != 6){
		fprintf(stderr, "Invalid number of parameters!\n");
        	return -EINVAL;
    	}

    	for (idx = 0; idx < sizeof(mac); idx++){
        	reassocInfo.DesiredBSSID[idx] = (u8)mac[idx];
    	}

    	/*
    	** If the ssid string is provided, save it; otherwise it is an empty string
    	*/
    	if (argc == 6){
        	strcpy(reassocInfo.DesiredSSID, argv[5]);
    	}

   	/* 
    	** Set up and execute the ioctl call
    	*/
    	iwr.u.data.flags = SUBIOCTL_WLANREASSOCIATE;
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)&reassocInfo;
	iwr.u.data.length = sizeof(reassocInfo);

	if (ioctl(sockfd, IOCTL_WLANREASSOCIATE, &iwr) < 0) {
		perror("wlanconfig: reassociate ioctl");
		return -EFAULT;
	}

    	/* No error return */
    	return 	WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Process set adhoc channel 
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_setadhocch(int argc, char *argv[]) 
{
	u32		vals[2];
	struct iwreq	iwr;

	if (argc != 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: wlanconfig eth1 setadhocch <band> <channel>\n");
		exit(1);
	}

	if (IOCTL_WLANSETADHOCCH <= 0) {
		return -EOPNOTSUPP;
	}

	vals[0] = *argv[3];
	vals[1] = atoval(argv[4]);

	memset(&iwr, 0, sizeof(iwr));
	
	if (SUBIOCTL_WLANSETADHOCCH > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANSETADHOCCH;
	}

	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)vals;
	iwr.u.data.length = sizeof(vals);

	if (ioctl(sockfd, IOCTL_WLANSETADHOCCH, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process Get adhoc channel 
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_getadhocch(int argc, char *argv[]) 
{
	unsigned int	vals[2];
	struct iwreq	iwr;

	if (argc != 3) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: wlanconfig eth1 getadhocch\n");
		exit(1);
	}

	if (IOCTL_WLANGETADHOCCH <= 0) {
		return -EOPNOTSUPP;
	}

	memset(&iwr, 0, sizeof(iwr));
	
	if (SUBIOCTL_WLANGETADHOCCH > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANGETADHOCCH;
	}

	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t)vals;
	iwr.u.data.length = sizeof(vals);

	if (ioctl(sockfd, IOCTL_WLANGETADHOCCH, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	printf("getadhocch: %c %d\n", vals[0], vals[1]);

	return 	WLAN_STATUS_SUCCESS;
}

#ifdef EXTSCAN
/** 
 *  @brief scan network with specific ssid
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_extscan(int argc, char *argv[]) 
{
	struct iwreq			iwr;
	WCON_SSID 			Ssid;

	if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 extscan <SSID>\n");
		exit(1);
	}

	if (IOCTL_WLANEXTSCAN <= 0) {
		return -EOPNOTSUPP;
	}

	printf("Ssid: %s\n", argv[3]);

	memset(&Ssid, 0, sizeof(Ssid));
	memset(&iwr, 0, sizeof(iwr));

	Ssid.ssid_len = strlen(argv[3]);
	memcpy(Ssid.ssid, argv[3], Ssid.ssid_len);

	if (SUBIOCTL_WLANEXTSCAN > 0) {
		iwr.u.data.flags = SUBIOCTL_WLANEXTSCAN;
	}

	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &Ssid;
	iwr.u.data.length = sizeof(Ssid);

	if (ioctl(sockfd, IOCTL_WLANEXTSCAN, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	return WLAN_STATUS_SUCCESS;
}

#if WIRELESS_EXT > 14
/** 
 *  @brief parse custom info
 *  @param pHandle	A pointer to WCON_HANDLE
 *  @param data         A pointer to iw_point structure
 *  @param idx          AP index
 *  @return      	NA
 */
static void parse_custom_info(WCON_HANDLE *pHandle, struct iw_point *data, s32 idx)
{
	s32	i = 0;
	s8 	*custom_cmd[] = { "wpa_ie", "rsn_ie", NULL };
	
	if (!data->pointer || !data->length) {
		printf("iw_point: Invalid Pointer/Length\n");
		return;
	}
	
	if (!strncmp(data->pointer, "wmm_ie", strlen("wmm_ie"))) {
			pHandle->ScanList[idx].Wmm = WCON_WMM_ENABLED;
	}

	while (custom_cmd[i]) {
		if (!strncmp(data->pointer, custom_cmd[i], 
					strlen(custom_cmd[i]))) {
			pHandle->ScanList[idx].WpaAP = WCON_WPA_ENABLED;
			break;
		}
		i++;
	}

	printf("Wpa:\t %s\n", pHandle->ScanList[idx].WpaAP ?
					"enabled" : "disabled");
	printf("Wmm:\t %s\n", pHandle->ScanList[idx].Wmm ?
					"enabled" : "disabled");
}
#endif


/** 
 *  @brief parse scan info
 *  @param pHandle	A pointer to WCON_HANDLE
 *  @param buffer       A pointer to scan result buffer
 *  @param length       length of scan result buffer
 *  @return      	NA
 */
static void parse_scan_info(WCON_HANDLE *pHandle, u8 buffer[], s32 length)
{
	s32			len = 0;
	s32			ap_index = -1;
	s8			*mode[3] = {"auto", "ad-hoc", "infra"};
	struct iw_event		iwe;

	memset(pHandle->ScanList, 0, sizeof(pHandle->ScanList));
	pHandle->ApNum = 0;

	while (len + IW_EV_LCP_LEN < length) {
		memcpy((s8 *)&iwe, buffer + len, sizeof(struct iw_event));
		switch (iwe.cmd) {
		case SIOCGIWAP:
			ap_index++;
			memcpy(pHandle->ScanList[ap_index].Bssid, 
					iwe.u.ap_addr.sa_data, ETH_ALEN);
			printf("\nBSSID:\t %02X:%02X:%02X:%02X:%02X:%02X\n",
				HWA_ARG(pHandle->ScanList[ap_index].Bssid));
			break;
			
		case SIOCGIWESSID:
			iwe.u.essid.pointer = buffer + len + IW_EV_POINT_LEN;	
			if ((iwe.u.essid.pointer) && (iwe.u.essid.length)) {
				memcpy(pHandle->ScanList[ap_index].Ssid.ssid,
						(s8 *)iwe.u.essid.pointer,
						iwe.u.essid.length);
				pHandle->ScanList[ap_index].Ssid.ssid_len = 
							iwe.u.essid.length;
			}
			printf("SSID:\t %s\n",
				pHandle->ScanList[ap_index].Ssid.ssid);
			break;
		
		case SIOCGIWENCODE:
			if (!(iwe.u.data.flags & IW_ENCODE_DISABLED)) {
				pHandle->ScanList[ap_index].Privacy =
							WCON_ENC_ENABLED;
			}
			printf("Privacy: %s\n", 
					pHandle->ScanList[ap_index].Privacy ?
					"enabled": "disabled");
			break;

		case SIOCGIWMODE:
			pHandle->ScanList[ap_index].NetMode = iwe.u.mode;
			printf("NetMode: %s\n", 
				mode[pHandle->ScanList[ap_index].NetMode]);
			break;
	
#if WIRELESS_EXT > 14
		case IWEVCUSTOM:
			iwe.u.data.pointer = buffer + len + IW_EV_POINT_LEN;
			parse_custom_info(pHandle, &iwe.u.data, ap_index);
			break;
#endif

		case IWEVQUAL:
			pHandle->ScanList[ap_index].Rssi = iwe.u.qual.level;
			printf("Quality: %d\n", 
					pHandle->ScanList[ap_index].Rssi);
			break;
		}
		
		len += iwe.len;
	}
	
	pHandle->ApNum = ap_index + 1;
	printf("\nNo of AP's = %d\n", pHandle->ApNum);

	return;
}

/* 
 *  @brief Process scan results
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_scan_results(int argc, char *argv[]) 
{
	u8		buffer[IW_SCAN_MAX_DATA];
	struct iwreq	iwr;
	WCON_HANDLE	mhandle, *pHandle = &mhandle;

	memset(pHandle, 0, sizeof(WCON_HANDLE));
	memset(&iwr, 0, sizeof(struct iwreq));
	
	iwr.u.data.pointer = buffer;
	iwr.u.data.length = sizeof(buffer);
	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);

	if ((ioctl(sockfd, SIOCGIWSCAN, &iwr)) < 0) {
		printf("Get Scan Results Failed\n");
		return -1;
	}

	parse_scan_info(pHandle, buffer, iwr.u.data.length);

	return WLAN_STATUS_SUCCESS;
}
#endif

#ifdef CIPHER_TEST
/** 
 *  @brief convert string to hex integer
 *
 *  @param d		A pointer dest buffer
 *  @param s		A pointer to source string buffer 
 *  @param dlen         max len of s buffer
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int ascii2hex(u8 *d, s8 *s, u32 dlen)
{
	s32 i;
	u8 n;

	memset(d, 0x00, dlen);

	for (i = 0; i < dlen * 2; i++) {
		if ((s[i] >= 48) && (s[i] <= 57))
			n = s[i] - 48;
		else if ((s[i] >= 65) && (s[i] <= 70))
			n = s[i] - 65;
		else if ((s[i] >= 97) && (s[i] <= 102))
			n = s[i] - 97;
		else
			break;
		if ((i % 2) == 0)
			n = n * 16;
		d[i / 2] += n;
	}

	return i;
}
/* 
 *  @brief Process cipher test
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_cipher_test(int argc, char *argv[])
{
  	s32 n, i;
  	struct iwreq iwr;
  	HostCmd_DS_802_11_KEY_ENCRYPT cmd;

  	memset(&cmd, 0, sizeof(cmd));

  	ascii2hex(cmd.KeyEncKey,  argv[4], 32);
  	ascii2hex(cmd.KeyIV,      argv[5], 32);

  	cmd.EncType = (strcmp(argv[3], "aes") == 0) ? CIPHER_TEST_AES: CIPHER_TEST_RC4;

  	cmd.Action  = HostCmd_ACT_SET;

  	for (i = 0; i < 512; i++){
    		n = getchar();
    		if (n == EOF)
      			break;
    		cmd.KeyData[i] = n;    
  	}
    	cmd.KeyDataLen = i;
  	strncpy(iwr.ifr_name, DevName, IFNAMSIZ);
  	iwr.u.data.pointer  = (caddr_t) &cmd;
 	iwr.u.data.length   = sizeof(cmd);
  	if (IOCTL_CIPHER_TEST <= 0)
    		return -EOPNOTSUPP;
  	if (SUBIOCTL_CIPHER_TEST > 0)
    		iwr.u.data.flags = SUBIOCTL_CIPHER_TEST;
  	if (ioctl(sockfd, IOCTL_CIPHER_TEST, &iwr) < 0)
    		return WLAN_STATUS_FAILURE;
  	for(i = 0; i < cmd.KeyDataLen; i++)
    		putchar(cmd.KeyData[i]);
  	return WLAN_STATUS_SUCCESS;
}

#endif


/* 
 *  @brief Process read eeprom
 *
 *  @param stroffset	A pointer to the offset string
 *  @param strnob	A pointer to NOB string
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_read_eeprom(s8 *stroffset, s8 *strnob)
{
	s8 			buffer[MAX_EEPROM_DATA];
	struct ifreq    	userdata;
	wlan_ioctl_regrdwr 	*reg = (wlan_ioctl_regrdwr *)buffer;

	memset(buffer, 0, sizeof(buffer));
	reg->WhichReg = REG_EEPROM;
	reg->Action = 0;

	if (!strncasecmp(stroffset, "0x", 2))
		reg->Offset = a2hex((stroffset + 2));
	else
		reg->Offset = atoi(stroffset);

	if (!strncasecmp(strnob, "0x", 2))
		reg->NOB = a2hex((strnob + 2));
	else
		reg->NOB = atoi(strnob);

	if (reg->NOB > MAX_EEPROM_DATA) {
		fprintf(stderr, "Number of bytes exceeds MAX EEPROM Read size\n");
		return WLAN_STATUS_FAILURE;
	}

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buffer;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: EEPROM read not possible "
			"by interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
	}

	hexdump("RD EEPROM", &reg->Value, reg->NOB, ' '); 

	return WLAN_STATUS_SUCCESS;
}



/* 
 *  @brief Display usage
 *  
 *  @return       NA
 */
static void display_usage(void)
{
	s32 i;

	for (i = 0; i < (sizeof(usage) / sizeof(usage[0])); i++)
		fprintf(stderr, "%s\n", usage[i]);
}

/* 
 *  @brief Find command
 *  
 *  @param maxcmds	max command number
 *  @param cmds		A pointer to commands buffer
 *  @param cmd		A pointer to command buffer
 *  @return      	index of command or WLAN_STATUS_FAILURE
 */
static int findcommand(s32 maxcmds, s8 *cmds[], s8 *cmd)
{
	s32 i;

	for (i = 0; i < maxcmds; i++) {
		if (!strcasecmp(cmds[i], cmd)) {
			return i;
		}
	}

	return WLAN_STATUS_FAILURE;
}


/* 
 *  @brief SD comand52 read
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd52r(int argc, char *argv[])
{
	struct ifreq    userdata;
	u8	 	buf[6];
	u32	 	tmp;

	buf[0] = 0;			//CMD52 read
	if (argc == 5) {
		buf[1] = atoval(argv[3]);	//func
		tmp = 	 atoval(argv[4]);	//reg
		buf[2] = tmp & 0xff;
		buf[3] = (tmp >> 8) & 0xff;
		buf[4] = (tmp >> 16) & 0xff;
		buf[5] = (tmp >> 24) & 0xff;
	} else {
		fprintf(stderr, "Invalid number of parameters!\n");
		return WLAN_STATUS_FAILURE;
	}
	

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLANCMD52RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD52 R/W not supported by "
				"interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
   	}
   	printf("sdcmd52r returns 0x%02X\n", buf[0]);

   	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief SD comand52 write
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd52w(int argc, char *argv[])
{
   	struct ifreq    userdata;
   	u8	 	buf[7];
   	u32	 	tmp;

	buf[0] = 1;			//CMD52 write
	if (argc == 6) {
		buf[1] = atoval(argv[3]);		//func
		tmp =    atoval(argv[4]);		//reg
		buf[2] = tmp & 0xff;
		buf[3] = (tmp >> 8) & 0xff;
		buf[4] = (tmp >> 16) & 0xff;
		buf[5] = (tmp >> 24) & 0xff;
		buf[6] = atoval(argv[5]);		//dat
	} else {
		fprintf(stderr, "Invalid number of parameters!\n");
		return WLAN_STATUS_FAILURE;
	}

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLANCMD52RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD52 R/W not supported by "
				"interface %s\n", DevName);
			return WLAN_STATUS_FAILURE;
   	}
   	printf("sdcmd52w returns 0x%02X\n",buf[0]);

   	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief SD comand53 read
 *  
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd53r(void)
{
	struct ifreq    userdata;
	s8 		buf[CMD53BUFLEN];
	int 		i;

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	for(i=0; i < sizeof(buf); i++)
		buf[i] = i & 0xff;

	if (ioctl(sockfd, WLANCMD53RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD53 R/W not supported by "
			"interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
	}

	for(i=0; i < sizeof(buf); i++) {
		if (buf[i] != (i ^ 0xff))
			printf("i=%02X  %02X\n",i,buf[i]);
	}

	return WLAN_STATUS_SUCCESS;
}

#ifdef WMM
/* 
 *  @brief Get/Set WMM ack policy
 *  
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_wmm_ack_policy(int argc, char *argv[])
{
	u8		buf[(WMM_ACK_POLICY_PRIO * 2) + 3];
	s32		count, i;
	struct ifreq	userdata;

	if ((argc != 3) && (argc != 5)) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 wmm_ack_policy\n");
		printf("Syntax: ./wlanconfig eth1 wmm_ack_policy <AC> <POLICY>\n");
		exit(1);
	}
	
	memset(buf, 0, (WMM_ACK_POLICY_PRIO * 2) + 3);

	buf[0] = WMM_ACK_POLICY;

	if (argc == 5) {
		buf[1] = HostCmd_ACT_SET;
		buf[2] = 0;

		buf[3] = atoi(argv[3]);
		if (buf[3] > WMM_ACK_POLICY_PRIO - 1) {
			printf("Invalid Priority. Should be between 0 and %d\n",
					WMM_ACK_POLICY_PRIO - 1);
			exit(1);
		}

		buf[4] = atoi(argv[4]);
		if(buf[4] > 1) {
			printf("Invalid Ack Policy. Should be 1 or 0\n");
			exit(1);
		}

		count = 5;
	} else {
		count = 2;
		buf[1] = HostCmd_ACT_GET;
	}
	
	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLAN_SETCONF_GETCONF, &userdata)) {
		fprintf(stderr, "wlanconfig: %s not supported by %s\n", 
							argv[2], DevName);
			return WLAN_STATUS_FAILURE;
   	}

	if (buf[1] == HostCmd_ACT_GET) {
		printf("AC Value    Priority\n");
		for (i=0; i < WMM_ACK_POLICY_PRIO; ++i) {
			count = SKIP_TYPE_ACTION + (i*2);
			printf("%4x       %5x\n", buf[count], buf[count+1]);
		}
	}		

	return WLAN_STATUS_SUCCESS;
}




/** 
 *  @brief read current command
 *  @param ptr		A pointer to data
 *  @param curCmd       A pointer to the buf which will hold current command    
 *  @return      	NULL or the pointer to the left command buf
 */
static s8 *readCurCmd(s8 *ptr, s8 *curCmd)
{
	s32 i = 0;
#define MAX_CMD_SIZE 64

	while(*ptr != ']' && i < (MAX_CMD_SIZE - 1))
		curCmd[i++] = *(++ptr);	

	if(*ptr != ']')
		return NULL;

	curCmd[i - 1] = '\0';

	return ++ptr;
}


/** 
 *  @brief parse command and hex data
 *  @param fp 		A pointer to FILE stream
 *  @param dst       	A pointer to the dest buf
 *  @param cmd		A pointer to command buf for search
 *  @return            	length of hex data or WLAN_STATUS_FAILURE  		
 */
static int fparse_for_cmd_and_hex(FILE *fp, u8 *dst, u8 *cmd)
{
	s8	*ptr;
	u8	*dptr;
	s8	buf[256], curCmd[64];
	s32	isCurCmd = 0;

	dptr = dst;
	while (fgets(buf, sizeof(buf), fp)) {
		ptr = buf;

		while (*ptr) {
			// skip leading spaces
			while (*ptr && isspace(*ptr))
				ptr++;

			// skip blank lines and lines beginning with '#'
			if (*ptr == '\0' || *ptr == '#')
				break;

			if(*ptr == '[' && *(ptr + 1) != '/') {
				if(!(ptr = readCurCmd(ptr, curCmd)))
					return WLAN_STATUS_FAILURE;

				if(strcasecmp(curCmd, cmd)) /* Not equal */
					isCurCmd = 0;
				else
					isCurCmd = 1;
			}

			/* Ignore the rest if it is not correct cmd */
			if(!isCurCmd)
				break;

			if(*ptr == '[' && *(ptr + 1) == '/' )
				return (dptr - dst);

			if (isxdigit(*ptr)) {
				ptr = convert2hex(ptr, dptr++);
			} else {
				/* Invalid character on data line */
				ptr++;
			}

		}
	}

	return WLAN_STATUS_FAILURE;
}

/* 
 *  @brief Config WMM parameters
 *  
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_wmm_para_conf(int argc, char *argv[], s32 cmd)
{
	s32		count;
	FILE		*fp;
	s8		buf[256];
	s8		filename[48] = "";
	struct ifreq	userdata;

	if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 %s <filename>\n",argv[2]);
		exit(1);
	}
	
	strncpy(filename, argv[3], MIN(sizeof(filename)-1, strlen(argv[3])));
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[3]);
		exit(1);
	}

	count = fparse_for_cmd_and_hex(fp, buf + SKIP_TYPE, argv[2]);
	if(count < 0) {
		printf("Invalid command parsing failed !!!\n");
		return -EINVAL;
	}

	/* This will set the type of command sent */
	buf[0] = (cmd - CMD_WMM_TSPEC) + WMM_TSPEC;

	hexdump(argv[2], buf, count + SKIP_TYPE, ' ');
	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLAN_SETCONF_GETCONF, &userdata)) {
		fprintf(stderr, "wlanconfig: %s not supported by %s\n", 
							argv[2], DevName);
			return WLAN_STATUS_FAILURE;
   	}

	hexdump(argv[2], buf, count + SKIP_TYPE, ' ');

	return WLAN_STATUS_SUCCESS;
}
#endif /* WMM */



/* 
 *  @brief convert String to integer
 *  
 *  @param value	A pointer to string
 *  @return             integer
 */
static u32 a2hex_or_atoi(s8 *value)
{
	if (value[0] == '0' && (value[1] == 'X' || value[1] == 'x'))
		return a2hex(value + 2);
	else
		return atoi(value);
}


/* 
 *  @brief Get one line from the File
 *  
 *  @param s	        Storage location for data.
 *  @param size 	Maximum number of characters to read. 
 *  @param stream 	File stream	  	
 *  @param line		A pointer to return current line number
 *  @return             returns string or NULL 
 */
static s8 * wlan_config_get_line(s8 *s, s32 size, FILE *stream, int *line)
{
	s8 *pos, *end, *sstart;

	while (fgets(s, size, stream)) {
		(*line)++;
		s[size - 1] = '\0';
		pos = s;

		while (*pos == ' ' || *pos == '\t')
			pos++;
		if (*pos == '#' || (*pos == '\r' && *(pos+1) == '\n') || 
						*pos == '\n' || *pos == '\0')
			continue;

		/* Remove # comments unless they are within a double quoted
		* string. Remove trailing white space. */
		sstart = strchr(pos, '"');
		if (sstart)
			sstart = strchr(sstart + 1, '"');
		if (!sstart)
			sstart = pos;
		end = strchr(sstart, '#');
		if (end)
			*end-- = '\0';
		else
			end = pos + strlen(pos) - 1;
		while (end > pos && (*end == '\r' || *end == '\n' || 
						*end == ' ' || *end == '\t')) {
			*end-- = '\0';
		}
		if (*pos == '\0')
			continue;
		return pos;
	}

	return NULL;
}

#ifdef BG_SCAN
/* 
 *  @brief convert hex char to integer
 *  
 *  @param c		char
 *  @return      	integer or WLAN_STATUS_FAILURE
 */
static int hex2num(s8 c)
{
	if (c >= '0' && c <= '9')
		return c - '0';
	if (c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	if (c >= 'A' && c <= 'F')
		return c - 'A' + 10;
	return WLAN_STATUS_FAILURE;
}
/* 
 *  @brief convert hex char to integer
 *  
 *  @param c		char
 *  @return      	integer or WLAN_STATUS_FAILURE
 */
static int hex2byte(const s8 *hex)
{
	s32 a, b;
	a = hex2num(*hex++);
	if (a < 0)
		return -1;
	b = hex2num(*hex++);
	if (b < 0)
		return -1;
	return (a << 4) | b;
}
/* 
 *  @brief convert hex char to integer
 *  
 *  @param hex		A pointer to hex string
 *  @param buf		buffer to storege the data
 *  @param len		
 *  @return      	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int hexstr2bin(const s8 *hex, u8 *buf, size_t len)
{
	s32	 i, a;
	const s8 *ipos = hex;
	s8	 *opos = buf;

	for (i = 0; i < len; i++) {
		a = hex2byte(ipos);
		if (a < 0)
			return WLAN_STATUS_FAILURE;
		*opos++ = a;
		ipos += 2;
	}
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse string
 *  
 *  @param value	A pointer to string
 *  @param buf		buffer to storege the data
 *  @param str		buffer to hold return string
 *  @param len 		use to return the length of parsed string
 *  @return      	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_config_parse_string(const s8 *value, s8 *str, size_t *len)
{
	if (*value == '"') {
		s8 *pos;
		value++;
		pos = strchr(value, '"');
		
		if (pos == NULL || pos[1] != '\0') {
			value--;
			return WLAN_STATUS_FAILURE;
		}
		*pos = '\0';
		*len = strlen(value);
		strcpy(str, value);
		return WLAN_STATUS_SUCCESS;
	} else {
		s32 hlen = strlen(value);

		if (hlen % 1)
			return WLAN_STATUS_FAILURE;
		*len = hlen / 2;
		if (str == NULL)
			return WLAN_STATUS_FAILURE;
		if (hexstr2bin(value, str, *len)) {
			return WLAN_STATUS_FAILURE;
   		 }
		
	    	return WLAN_STATUS_SUCCESS;
 	 }
}

/* 
 *  @brief parse bgscan action
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to Action buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_action(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->Action = (u16)a2hex_or_atoi(value);
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan enable parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to Enable buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_enable(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;

	bgscan_config->Enable = (u8)a2hex_or_atoi(value);
	
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan BssType
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to BssType buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_bsstype(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;

	bgscan_config->BssType = (u8)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan channels per scan parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to channels per scan buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_channelsperscan(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->ChannelsPerScan = (u8)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}


/* 
 *  @brief parse bgscan discardwhenfull parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to channels per scan buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_discardwhenfull(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->DiscardWhenFull = (u8)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan scan interval parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to scan interval buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_scaninterval(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->ScanInterval = (u32)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan store condition parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to store condition buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_storecondition(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->StoreCondition = a2hex_or_atoi(value);

	return 	WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan report conditions parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to report conditionsn buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_reportconditions(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->ReportConditions = a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan max scan results parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to max scan results buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_maxscanresults(u8 *CmdBuf, s32 line, s8 *value)
{
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config = 
				(HostCmd_DS_802_11_BG_SCAN_CONFIG *)CmdBuf;
	
	bgscan_config->MaxScanResults = (u16)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan ssid parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to ssid buffer
 *  @return      	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int bgscan_parse_ssid(u8 *CmdBuf, s32 line, s8 *value)
{
	static int 			ssidCnt;
	MrvlIEtypes_SsIdParamSet_t 	*SsIdParamSet = NULL;
	s8				*buf;
	size_t				len = 0;
	
	SsIdParamSet = (MrvlIEtypes_SsIdParamSet_t *)(CmdBuf + ActualPos);

	buf = (s8 *)malloc(strlen(value));	

	if(wlan_config_parse_string(value, buf, &len)) {
		printf("Invalid SSID\n");
       		free(buf);
 		return  WLAN_STATUS_FAILURE;
	}		

	ssidCnt++;

	if(!strlen(buf)) {
		printf("The %dth SSID is NULL.\n", ssidCnt);
	}

 	SsIdParamSet->Header.Type = cpu_to_le16(TLV_TYPE_SSID); /*0x0000; */
	SsIdParamSet->Header.Len  = strlen(buf);

	TLVSsidSize += SsIdParamSet->Header.Len + 
			sizeof(MrvlIEtypesHeader_t);

	ActualPos += SsIdParamSet->Header.Len + 
			sizeof(MrvlIEtypesHeader_t);
	
	SsIdParamSet->Header.Len  = cpu_to_le16(SsIdParamSet->Header.Len);
	
	memcpy(SsIdParamSet->SsId, buf, strlen(buf));

	free(buf);
	return  WLAN_STATUS_SUCCESS;
}		

/* 
 *  @brief parse bgscan probes parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to probes buffer
 *  @return      	WLAN_STATUS_SUCCESS 
 */
static int bgscan_parse_probes(u8 *CmdBuf, s32 line, s8 *value)
{
	MrvlIEtypes_NumProbes_t	*Probes = NULL;

#define PROBES_PAYLOAD_SIZE	2
	
	Probes = (MrvlIEtypes_NumProbes_t *)(CmdBuf + ActualPos);

	Probes->Header.Type = TLV_TYPE_NUMPROBES;
	Probes->Header.Len = PROBES_PAYLOAD_SIZE;
	
	Probes->NumProbes = (u16)a2hex_or_atoi(value);

	if (Probes->NumProbes) {
		TLVProbeSize += sizeof(MrvlIEtypesHeader_t) +
						Probes->Header.Len;
	} else {
		TLVProbeSize = 0;
	}

	ActualPos += TLVProbeSize;
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan channel list parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to channel list buffer
 *  @return      	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int bgscan_parse_channellist(u8 *CmdBuf, s32 line, s8 *value)
{
	MrvlIEtypes_ChanListParamSet_t  *chan;
  	char *buf, *grp0, *grp1;
  	s32 len, idx;

  	chan = (MrvlIEtypes_ChanListParamSet_t *)(CmdBuf + ActualPos);

  	len = strlen(value) + 1;
  	buf = malloc(len);

  	if (buf == NULL)
    		return WLAN_STATUS_FAILURE;

  	memset(buf, 0, len);
  	strcpy(buf, value);

  	chan->Header.Type = cpu_to_le16(TLV_TYPE_CHANLIST);
  	grp1 = buf;
  	idx = 0;
  	while ((grp1 != NULL) && (*grp1 != 0)){
    		grp0 = strsep(&grp1, "\";");

    		if ((grp0 != NULL) && (*grp0 != 0)){
      			chan->ChanScanParam[idx].RadioType    = atoi(strtok(grp0, ","));
      			chan->ChanScanParam[idx].ChanNumber   = atoi(strtok(NULL, ","));
      			chan->ChanScanParam[idx].ScanType     = atoi(strtok(NULL, ","));
      			chan->ChanScanParam[idx].MinScanTime  = atoi(strtok(NULL, ","));
      			chan->ChanScanParam[idx].MaxScanTime  = atoi(strtok(NULL, ","));
      			idx ++;
    		}
  	}

  	chan->Header.Len = (idx * sizeof(ChanScanParamSet_t));
  	TLVChanSize += (chan->Header.Len + sizeof(MrvlIEtypesHeader_t));
  	chan->Header.Len = cpu_to_le16(chan->Header.Len);
  	ActualPos += TLVChanSize;

  	free (buf);
  	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan snr threshold parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to bgscan snr threshold buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_snrthreshold(u8 *CmdBuf, s32 line, s8 *value)
{
	MrvlIEtypes_SnrThreshold_t	*SnrThreshold = NULL;
	u32 	tmp;
	
	SnrThreshold = (MrvlIEtypes_SnrThreshold_t *)(CmdBuf + ActualPos);

	SnrThreshold->Header.Type = TLV_TYPE_SNR;
	SnrThreshold->Header.Len = PROBES_PAYLOAD_SIZE;
	
	tmp = (u16)a2hex_or_atoi(value);
	SnrThreshold->SNRValue = tmp & 0xff;
	SnrThreshold->SNRFreq = (tmp >> 8) & 0xff;

    	TLVSnrSize += sizeof(MrvlIEtypesHeader_t) + SnrThreshold->Header.Len; 
	ActualPos += TLVSnrSize;
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse bgscan broadcast probe parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to broadcast probe buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_bcastprobe(u8 *CmdBuf, s32 line, s8 *value)
{
	MrvlIEtypes_BcastProbe_t *BcastProbe = NULL;

	BcastProbe = (MrvlIEtypes_BcastProbe_t *)(CmdBuf + ActualPos);

	BcastProbe->Header.Type = TLV_TYPE_BCASTPROBE;
	BcastProbe->Header.Len = PROBES_PAYLOAD_SIZE;
	
	BcastProbe->BcastProbe = (u16)a2hex_or_atoi(value);

    	TLVBcProbeSize = sizeof(MrvlIEtypesHeader_t) + BcastProbe->Header.Len;
	ActualPos += TLVBcProbeSize;
	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse number ssid probe parameter
 *  
 *  @param CmdBuf	A pointer to command buffer
 *  @param line	        line number
 *  @param value	A pointer to number ssid probe buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int bgscan_parse_numssidprobe(u8 *CmdBuf, s32 line, s8 *value)
{
	MrvlIEtypes_NumSSIDProbe_t *NumSSIDProbe = NULL;


	NumSSIDProbe = (MrvlIEtypes_NumSSIDProbe_t *)(CmdBuf + ActualPos);

	NumSSIDProbe->Header.Type = TLV_TYPE_NUMSSID_PROBE;
	NumSSIDProbe->Header.Len = PROBES_PAYLOAD_SIZE;
	
	NumSSIDProbe->NumSSIDProbe = (u16)a2hex_or_atoi(value);

    	TLVNumSsidProbeSize = sizeof(MrvlIEtypesHeader_t) + NumSSIDProbe->Header.Len;
	ActualPos += TLVNumSsidProbeSize;
	return WLAN_STATUS_SUCCESS;
}

static struct bgscan_fields {
  s8 *name;
  int (*parser)(u8 *CmdBuf, s32 line, s8 *value);
} bgscan_fields[] = {
  { "Action", bgscan_parse_action },
  { "Enable", bgscan_parse_enable },
  { "BssType", bgscan_parse_bsstype },
  { "ChannelsPerScan", bgscan_parse_channelsperscan },
  { "DiscardWhenFull", bgscan_parse_discardwhenfull },
  { "ScanInterval", bgscan_parse_scaninterval },
  { "StoreCondition", bgscan_parse_storecondition },
  { "ReportConditions", bgscan_parse_reportconditions },
  { "MaxScanResults", bgscan_parse_maxscanresults },
  { "SSID1", bgscan_parse_ssid },
  { "SSID2", bgscan_parse_ssid },
  { "SSID3", bgscan_parse_ssid },
  { "SSID4", bgscan_parse_ssid },
  { "SSID5", bgscan_parse_ssid },
  { "SSID6", bgscan_parse_ssid },
  { "Probes", bgscan_parse_probes },
  { "ChannelList", bgscan_parse_channellist },
  { "SnrThreshold", bgscan_parse_snrthreshold },
  { "BcastProbe", bgscan_parse_bcastprobe },
  { "NumSSIDProbe", bgscan_parse_numssidprobe },
};
#define NUM_BGSCAN_FIELDS (sizeof(bgscan_fields) / sizeof(bgscan_fields[0]))
/* 
 *  @brief get bgscan data
 *  
 *  @param fp			A pointer to file stream
 *  @param line	        	A pointer to line number
 *  @param bgscan_config	A pointer to HostCmd_DS_802_11_BG_SCAN_CONFIG structure
 *  @return      		WLAN_STATUS_SUCCESS
 */
static int wlan_get_bgscan_data(FILE *fp, int *line, 
				HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config)
{
	s32	errors = 0, i, end = 0;
	s8	buf[256], *pos, *pos2;


	while ((pos = wlan_config_get_line(buf, sizeof(buf), fp, line))) {
		if (strcmp(pos, "}") == 0) {
			end = 1;
			break;
		}

		pos2 = strchr(pos, '=');
		if (pos2 == NULL) {
			printf("Line %d: Invalid bgscan line '%s'.",
								*line, pos);
			errors++;
			continue;
		}

		*pos2++ = '\0';
		if (*pos2 == '"') {
			if (strchr(pos2 + 1, '"') == NULL) {
				printf("Line %d: invalid quotation '%s'.", 
								*line, pos2);
				errors++;
				continue;
			}
		}

		for (i = 0; i < NUM_BGSCAN_FIELDS; i++) {
			if (strcmp(pos, bgscan_fields[i].name) == 0) {
				if (bgscan_fields[i].parser((u8 *)bgscan_config,
								*line, pos2)) {
					printf("Line %d: failed to parse %s"
						"'%s'.", *line, pos, pos2);
					errors++;
					}
				break;
			}
		}
		if (i == NUM_BGSCAN_FIELDS) {
			printf("Line %d: unknown bgscan field '%s'.\n", 
								*line, pos);
			errors++;
		}
	}
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process bgscan config 
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_bg_scan_config(int argc, char *argv[])
{
	u8				scanCfg[512], *pos, *buf = NULL;
	s8				filename[48] = "";
	FILE				*fp;
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgscan_config;
	struct ifreq			userdata;
	int 				line = 0;
	s32				CmdNum = BG_SCAN_CONFIG;
	s32				Action;
	u16				Size;

	buf = (u8 *)malloc(512);
	
	memset(buf, 0, 512);

	bgscan_config = (HostCmd_DS_802_11_BG_SCAN_CONFIG *)
				(buf + sizeof(s32) + sizeof(u16));

	if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 bgscanconfig <filename>\n");
		exit(1);
	}
	
	strncpy(filename, argv[3], MIN(sizeof(filename)-1, strlen(argv[3])));
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", filename);
		exit(1);
	}

	while ((pos = wlan_config_get_line(scanCfg, sizeof(scanCfg), fp, 
								&line))) {
		if (strcmp(pos, "bgscan={") == 0) {
			wlan_get_bgscan_data(fp, &line, bgscan_config);
		}
	}

	fclose(fp);

	Action = bgscan_config->Action;

	Size = sizeof(HostCmd_DS_802_11_BG_SCAN_CONFIG) +
		TLVSsidSize + TLVProbeSize + TLVChanSize + TLVSnrSize + TLVBcProbeSize + TLVNumSsidProbeSize;

	memcpy(buf, &CmdNum, sizeof(s32));
	memcpy(buf + sizeof(s32), &Size, sizeof(u16));

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLAN_SETCONF_GETCONF, &userdata)) {
		fprintf(stderr, "wlanconfig: BG_SCAN is not supported by %s\n", 
								DevName);
			return -1;
   	}

	if (Action == HostCmd_ACT_GEN_GET) {
		s32 i;
		printf("BGSCAN Configuration setup:\n");
		for(i = 0; i < Size; i++) {
			if (!(i % 10)) {
				printf("\n");
			}
			printf(" 0x%x ", buf[i + 3]);
		}
		printf("\n");
	}

	free(buf);

	return WLAN_STATUS_SUCCESS;

}
#endif

#ifdef SUBSCRIBE_EVENT_CTRL
/* 
 *  @brief parse event action parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to event action buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_action(EventSubscribe *EventData, s32 line, s8 *value)
{
	
	EventData->Action = (u16)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse event parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to event buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_event(EventSubscribe *EventData, s32 line, s8 *value)
{
	
	EventData->Events = (u16)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;	
}

/* 
 *  @brief parse RSSI parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to RSSI buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_RSSI(EventSubscribe *EventData, s32 line, s8 *value)
{
	s8		*p = value, *token;
	s8		delim = ',';

	token = strtok(p, &delim);
	EventData->RSSIValue = (u8)a2hex_or_atoi(token);
	p = NULL;
	token = strtok(p, &delim);
	EventData->RSSIFreq = (u8)a2hex_or_atoi(token);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse SNR parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to RSSI buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_SNR(EventSubscribe *EventData, s32 line, s8 *value)
{
	s8		*p = value, *token;
	s8		delim = ',';

	token = strtok(p, &delim);
	EventData->SNRValue = (u8)a2hex_or_atoi(token);
	p = NULL;
	token = strtok(p, &delim);
	EventData->SNRFreq = (u8)a2hex_or_atoi(token);

	return 	WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse fail count parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to fail count buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_failcnt(EventSubscribe *EventData, s32 line, s8 *value)
{
	s8		*p = value, *token;
	s8		delim = ',';

	token = strtok(p, &delim);
	EventData->FailValue = (u8)a2hex_or_atoi(token);
	p = NULL;
	token = strtok(p, &delim);
	EventData->FailFreq = (u8)a2hex_or_atoi(token);

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief parse beacon missed parameter
 *  
 *  @param CmdBuf	A pointer to EventSubscribe structure
 *  @param line	        line number
 *  @param value	A pointer to beacon missed buffer
 *  @return      	WLAN_STATUS_SUCCESS
 */
static int sub_event_parse_beacon_missed(EventSubscribe *EventData, s32 line, s8 *value)
{
	EventData->BcnMissed = (u8)a2hex_or_atoi(value);

	return WLAN_STATUS_SUCCESS;
}

static struct sub_event_fields {
  s8 *name;
  int (*parser)(EventSubscribe *EventData, s32 line, s8 *value);
} sub_event_fields[] = {
  { "Action", sub_event_parse_action },
  { "Event", sub_event_parse_event },
  { "RSSI", sub_event_parse_RSSI },
  { "SNR", sub_event_parse_SNR },
  { "FailureCount", sub_event_parse_failcnt },
  { "BcnMissed", sub_event_parse_beacon_missed },
};

#define NUM_SUB_EVENT_FIELDS (sizeof(sub_event_fields) / sizeof(sub_event_fields[0]))
/* 
 *  @brief get event data
 *  
 *  @param fp			A pointer to file stream
 *  @param line	        	A pointer to line number
 *  @param EventData		A pointer to EventSubscribe structure
 *  @return      		WLAN_STATUS_SUCCESS
 */
static int wlan_get_subscribe_event_data(FILE *fp, int *line, EventSubscribe *EventData)
{
	s32	errors = 0, i, end = 0;
	s8	buf[256], *pos, *pos2;


	while ((pos = wlan_config_get_line(buf, sizeof(buf), fp, line))) {
		if (strcmp(pos, "}") == 0) {
			end = 1;
			break;
		}

		pos2 = strchr(pos, '=');
		if (pos2 == NULL) {
			printf("Line %d: Invalid sub_event line '%s'.",
								*line, pos);
			errors++;
			continue;
		}

		*pos2++ = '\0';
		if (*pos2 == '"') {
			if (strchr(pos2 + 1, '"') == NULL) {
				printf("Line %d: invalid quotation '%s'.", 
								*line, pos2);
				errors++;
				continue;
			}
		}

		for (i = 0; i < NUM_SUB_EVENT_FIELDS; i++) {
			if (strcmp(pos, sub_event_fields[i].name) == 0) {
				if (sub_event_fields[i].parser(EventData,
								*line, pos2)) {
					printf("Line %d: failed to parse %s"
						"'%s'.", *line, pos, pos2);
					errors++;
					}
				break;
			}
		}
		if (i == NUM_SUB_EVENT_FIELDS) {
			printf("Line %d: unknown sub_event field '%s'.", 
								*line, pos);
			errors++;
		}
	}
	return 	WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief Process event subscribe
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_event_subscribe(int argc, char *argv[])
{
	u8			*buf;
	s8			event_data[512];
	s8			filename[48] = "";
	FILE			*fp;
	EventSubscribe		EventData;
	struct ifreq		userdata;
	u8			*pos;
	int			line = 0;
	s32			CmdNum = SUBSCRIBE_EVENT; 

	memset(&EventData, 0, sizeof(EventSubscribe));
	
	if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 subevent <filename>\n");
		exit(1);
	}
	
	strncpy(filename, argv[3], MIN(sizeof(filename)-1, strlen(argv[3])));

	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[1]);
		exit(1);
	}

	while ((pos = wlan_config_get_line(event_data, sizeof(event_data), fp, 
							&line))) {
		if (strcmp(pos, "Events={") == 0) {
			wlan_get_subscribe_event_data(fp, &line, &EventData);
		}
	}

	fclose(fp);

	buf = (u8 *)malloc(512);

	memcpy(buf, &CmdNum, sizeof(s32));
	memcpy(buf + sizeof(s32), (u8 *)&EventData.Action, 
							sizeof(EventSubscribe));

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = buf;
	
	if (ioctl(sockfd, WLAN_SETCONF_GETCONF, &userdata)) {
		fprintf(stderr, "wlanconfig: SUBSCRIBE_EVENT not"
						" supported by %s\n", DevName);
			return 	WLAN_STATUS_FAILURE;
   	}
	
#define GET_EVENTS	0
#define BIT_0		(1 << 0) 
#define BIT_1		(1 << 1)
#define BIT_2		(1 << 2)
#define BIT_3		(1 << 3)
	if (EventData.Action == GET_EVENTS) {
		u16		Events;

		Events = *(u16 *)(buf + sizeof(s32));

		printf("Events Subscribed to Firmware are\n");

		if (Events & BIT_0) {
			printf("RSSI_LOW\n");
		}

		if (Events & BIT_1) {
			printf("SNR_LOW\n");
		}

		if (Events & BIT_2) {
			printf("MAX_FAIL\n");
		}

		if (Events & BIT_3) {
			printf("LINK LOSS\n");
		}
			
	}
		
	free(buf);

	return 	WLAN_STATUS_SUCCESS;
}
#endif

#ifdef CAL_DATA

/** 
 *  @brief parse hex data
 *  @param fp 		A pointer to FILE stream
 *  @param dst		A pointer to receive hex data
 *  @return            	length of hex data
 */
static int fparse_for_hex(FILE *fp, u8 *dst)
{
	s8	*ptr;
	u8	*dptr;
	s8	buf[256];
	
	dptr = dst;
	while (fgets(buf, sizeof(buf), fp)) {
		ptr = buf;

		while (*ptr) {
			// skip leading spaces
			while (*ptr && isspace(*ptr))
				ptr++;

			// skip blank lines and lines beginning with '#'
			if (*ptr == '\0' || *ptr == '#')
				break;

			if (isxdigit(*ptr)) {
				ptr = convert2hex(ptr, dptr++);
			} else {
				/* Invalid character on data line */
				ptr++;
			}
		}
	}

	return (dptr - dst);
}

/** 
 *  @brief Process calibration data
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int process_cal_data_ext(int argc, char *argv[])
{
	s32				count;
	u8				*buf, *CmdBuf;
	FILE				*fp;
	s8				filename[48] = "";
	HostCmd_DS_802_11_CAL_DATA_EXT	cal_data; 
	struct ifreq			userdata;
	s32				CmdNum = CAL_DATA_EXT_CONFIG;

	printf("Enter process_cal_data_ext()\n");
	memset(&cal_data, 0, sizeof(cal_data));

	buf = (u8 *) &cal_data.Action;
	
	if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 caldataext <filename>\n");
		exit(1);
	}
	
	strncpy(filename, argv[3], MIN(sizeof(filename)-1, strlen(argv[3])));
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[1]);
		exit(1);
	}
	
	count = fparse_for_hex(fp, buf); //copyfile data to buf+1
	fclose(fp);

	CmdBuf = malloc(sizeof(HostCmd_DS_802_11_CAL_DATA_EXT) + sizeof(s32));
	memcpy(CmdBuf, &CmdNum, sizeof(s32));
	memcpy(CmdBuf + sizeof(s32), &cal_data, 
					sizeof(HostCmd_DS_802_11_CAL_DATA_EXT));
	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = CmdBuf;

	if (ioctl(sockfd, WLAN_SETCONF_GETCONF, &userdata)) { 
		fprintf(stderr, "wlanconfig: CAL DATA not supported by %s\n", 
								DevName);
			return 	WLAN_STATUS_FAILURE;
   	}

	free(CmdBuf); //free CmdBuf
	printf("Exit process_cal_data_ext()\n"); 
	return 	WLAN_STATUS_SUCCESS;
}
#endif


/** 
 *  @brief read register
 *  @param cmd 		the type of register
 *  @param stroffset	A pointer to register index string
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_read_register(s32 cmd, s8 *stroffset)
{
	struct ifreq    userdata;
	wlan_ioctl_regrdwr reg;
	s8           *whichreg;

	switch (cmd) {
		case CMD_RDMAC:
			/*
			 * HostCmd_CMD_MAC_REG_ACCESS 
			 */
			reg.WhichReg = REG_MAC;
			whichreg = "MAC";
			break;
		case CMD_RDBBP:
			/*
			 * HostCmd_CMD_BBP_REG_ACCESS 
			 */
			reg.WhichReg = REG_BBP;
			whichreg = "BBP";
			break;
		case CMD_RDRF:
			/*
			 * HostCmd_CMD_RF_REG_ACCESS 
			 */
			reg.WhichReg = REG_RF;
			whichreg = "RF";
			break;
		default:
			fprintf(stderr, 
				"Invalid Register set specified.\n");
			return -1;
	}

	reg.Action = 0;		/* READ */

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: Register Reading not supported by"
			"interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
	}

	printf("%s[0x%04lx] = 0x%08lx\n", 
			whichreg, reg.Offset, reg.Value);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief write register
 *  @param cmd 		the type of register
 *  @param stroffset	A pointer to register index string
 *  @param strvalue	A pointer to the register value
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_write_register(s32 cmd, s8 *stroffset, s8 *strvalue)
{
	struct ifreq    	userdata;
	wlan_ioctl_regrdwr 	reg;
	s8           		*whichreg;

	switch (cmd) {
		case CMD_WRMAC:
			/*
			 * HostCmd_CMD_MAC_REG_ACCESS 
			 */
			reg.WhichReg = REG_MAC;
			whichreg = "MAC";
			break;
		case CMD_WRBBP:
			/*
			 * HostCmd_CMD_BBP_REG_ACCESS 
			 */
			reg.WhichReg = REG_BBP;
			whichreg = "BBP";
			break;
		case CMD_WRRF:
			/*
			 * HostCmd_CMD_RF_REG_ACCESS 
			 */
			reg.WhichReg = REG_RF;
			whichreg = "RF";
			break;
		default:
			fprintf(stderr, 
				"Invalid register set specified.\n");
			return -1;
	}

	reg.Action = 1;		/* WRITE */

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	if (!strncasecmp(strvalue, "0x", 2))
		reg.Value = a2hex((strvalue + 2));
	else
		reg.Value = atoi(strvalue);

	printf("Writing %s Register 0x%04lx with 0x%08lx\n", whichreg,
			reg.Offset, reg.Value);

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register Writing not supported "
			"by interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
	}

	printf("%s[0x%04lx] = 0x%08lx\n",
			whichreg, reg.Offset, reg.Value);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief read CF register
 *
 *  @param stroffset	A pointer to register index string
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_read_cfreg(s8 *stroffset)
{
	struct ifreq    	userdata;
	wlan_ioctl_cfregrdwr 	reg;
	
	reg.Action = 0; //Read register

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGCFRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register reading not supported "
			"by interface %s\n", DevName);
		return  WLAN_STATUS_FAILURE;
	}

	printf("CFREG[0x%04X] = 0x%04X\n",
				reg.Offset, reg.Value);

	return  WLAN_STATUS_SUCCESS;

}

/** 
 *  @brief write CF register
 *
 *  @param stroffset	A pointer to register index string
 *  @param strvalue	A pointer to the register value 
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_write_cfreg(s8 *stroffset, s8 *strvalue)
{
	struct ifreq    	userdata;
	wlan_ioctl_cfregrdwr 	reg;

	reg.Action = 1; //Write register

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	if (!strncasecmp(strvalue, "0x", 2))
		reg.Value = a2hex((strvalue + 2));
	else
		reg.Value = atoi(strvalue);

	strncpy(userdata.ifr_name, DevName, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGCFRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register writing not supported "
			"by interface %s\n", DevName);
		return WLAN_STATUS_FAILURE;
	}

	return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief convert char to hex integer
 *
 *  @param chr		char
 *  @return            	hex integer
 */
static u8 hexc2bin(s8 chr)
{
	if (chr >= '0' && chr <= '9')
		chr -= '0';
	else if (chr >= 'A' && chr <= 'F')
		chr -= ('A' - 10);
	else if (chr >= 'a' && chr <= 'f')
		chr -= ('a' - 10);

	return chr;
}

/** 
 *  @brief convert string to hex integer
 *
 *  @param s		A pointer string buffer
 *  @return            	hex integer
 */
static u32 a2hex(s8 *s)
{
	u32    val = 0;
	while (*s && isxdigit(*s)) {
		val = (val << 4) + hexc2bin(*s++);
	}

	return val;
}

/** 
 *  @brief Hump hex data
 *
 *  @param prompt	A pointer prompt buffer
 *  @param p		A pointer to data buffer
 *  @param len		the len of data buffer
 *  @param delim	delim char
 *  @return            	hex integer
 */
static void hexdump(s8 *prompt, void *p, s32 len, s8 delim)
{
	s32             i;
	u8  *s = p;
	
	printf("%s : ", prompt);
	for (i = 0; i < len; i++) {
		if (i != len - 1)
			printf("%02x%c", *s++, delim);
		else
			printf("%02x\n", *s);
	}
}

/********************************************************
		Global Functions
********************************************************/
/* 
 *  @brief Entry function for wlanconfig
 *  @param arac		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int main(int argc, char *argv[])
{
	s32             cmd;
	s32		PrivateCmds;

	if (argc < 3) {
		fprintf(stderr, "Invalid number of parameters!\n");
		display_usage();
		exit(1);
	}

	strncpy(DevName, argv[1], IFNAMSIZ);

	/*
	 * create a socket 
	 */
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr, "wlanconfig: Cannot open socket.\n");
		exit(1);
	}

	PrivateCmds = get_private_info(DevName);

	marvell_init_ioctl_numbers(DevName);

	switch ((cmd = findcommand(MAX_COMMANDS, commands, argv[2]))) {
	case CMD_RDMAC:
	case CMD_RDBBP:
	case CMD_RDRF:
		if (argc < 4) {
			fprintf(stderr, 
				"Register offset required!\n");
			display_usage();
			exit(1);
		}

		if (process_read_register(cmd, argv[3])) {
			fprintf(stderr,
				"Read command failed!\n");
			exit(1);
		}
		break;
	case CMD_WRMAC:
	case CMD_WRBBP:
	case CMD_WRRF:
		if (argc < 5) {
			fprintf(stderr,
				"Register offset required &" 
					"value!\n");
			display_usage();
			exit(1);
		}
		if (process_write_register(cmd, argv[3],
					argv[4])) {
			fprintf(stderr,
				"Write command failed!\n");
			exit(1);
		}
		break;
	case CMD_CMD52R:
		process_sdcmd52r(argc,argv);
		break;
	case CMD_CMD52W:
		process_sdcmd52w(argc,argv);
		break;
	case CMD_CMD53R:
		process_sdcmd53r();
		break;
#ifdef SUBSCRIBE_EVENT_CTRL
	case CMD_SUB_EVENT:
		process_event_subscribe(argc, argv);
		break;
#endif

#ifdef BG_SCAN
	case CMD_BG_SCAN_CONFIG:
		process_bg_scan_config(argc, argv);
		break;
#endif /* BG_SCAN */
#ifdef WMM
	case CMD_WMM_ACK_POLICY:
		process_wmm_ack_policy(argc, argv);
		break;
	case CMD_WMM_TSPEC:
	case CMD_WMM_AC_WPAIE:
		process_wmm_para_conf(argc, argv, cmd);
		break;
#endif /* WMM */
#ifdef CAL_DATA
	case CMD_CAL_DATA_EXT:
		process_cal_data_ext(argc, argv); 
		break;
#endif
	case CMD_CFREGR:
		printf("process read cfreg\n");
		if (argc < 4) {
			fprintf(stderr,	"Register offset required!\n");
			display_usage();
			exit(1);
		}
		if (process_read_cfreg(argv[3])) {
			fprintf(stderr, "Read CF register failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_CFREGW:
		printf("process write cfreg\n");
		if (argc < 5) {
			fprintf(stderr,	"Register offset required!\n");
			display_usage();
			exit(1);
		}
		if (process_write_cfreg(argv[3], argv[4])) {
			fprintf(stderr, "Read CF register failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_RDEEPROM:
		printf("proces read eeprom\n");

		if(argc < 5) {
			fprintf(stderr, "Register offset, number of bytes required\n");
			display_usage();
			exit(1);
		}
		
		if(process_read_eeprom(argv[3], argv[4])) {
			fprintf(stderr, "EEPROM Read failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_GETRATE:
		if (process_get_rate()) {
			fprintf(stderr, "Get Rate Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_SLEEPPARAMS:
		if (process_sleep_params(argc, argv)) {
			fprintf(stderr, "Sleep Params Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_BCA_TS:
		if (process_bca_ts(argc, argv)) {
			fprintf(stderr, "SetBcaTs Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_REASSOCIATE:
        	if (process_reassociation(argc, argv))
        	{
			exit(1);
        	}
        	break;
	case CMD_SETADHOCCH:
		if (process_setadhocch(argc, argv)) {
			fprintf(stderr, "SetAdhocCh Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_GETADHOCCH:
		if (process_getadhocch(argc, argv)) {
			fprintf(stderr, "GetAdhocCh Failed\n");
			display_usage();
			exit(1);
		}
		break;
#ifdef EXTSCAN		
	case CMD_EXTSCAN:
		if (process_extscan(argc, argv)) {
			fprintf(stderr, "ExtScan Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_SCAN_LIST:
		if (process_scan_results(argc, argv)) {
			fprintf(stderr, "getscanlist Failed\n");
			display_usage();
			exit(1);
		}
		break;

#endif		

#ifdef CIPHER_TEST
	case CMD_CIPHER_TEST:
		if (process_cipher_test(argc, argv)) {
			fprintf(stderr, "cipher_test Failed\n");
			display_usage();
			exit(1);
		}
		break;
#endif
	case CMD_REQUESTTPC:
        if (process_requesttpc(argc, argv))
        {
		exit(1);
        }
        break;
	case CMD_CHANSWANN:
        if (process_chanswann(argc, argv))
        {
		exit(1);
        }
		break;
	case CMD_POWERCAP:
        if (process_powercap(argc, argv))
        {
		exit(1);
        }
        break;
	case CMD_MEASREQ:
        if (process_measreq(argc, argv))
        {
		exit(1);
        }
        break;
	default:
		fprintf(stderr, "Invalid command specified!\n");
		display_usage();
		exit(1);
	}

	return WLAN_STATUS_SUCCESS;
}

