/** @file wlan_cmdresp.c
  * @brief This file contains the handling of command
  * responses as well as events generated by firmware.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	10/10/05: Add Doxygen format comments
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/

#ifdef FW_WAKEUP_TIME
u32 wt_pwrup_sending=0L, wt_pwrup_sent=0L, wt_int=0L,
    wt_wakeup_event=0L, wt_awake_confirmrsp=0L;
#endif

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function handles disconnect event. it
 *  reports disconnect to upper layer, clean tx/rx packets,
 *  reset link state etc.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void MacEventDisconnected(wlan_private * priv)
{
	wlan_adapter	*Adapter = priv->adapter;
	union iwreq_data wrqu;

	ENTER();

	if (Adapter->MediaConnectStatus != WlanMediaStateConnected)
		return;
	
	PRINTM(INFO, "Handles disconnect event.\n");

	memset(wrqu.ap_addr.sa_data, 0x00, ETH_ALEN);
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;

	/* 
	 * Cisco AP sends EAP failure and de-auth in less than 0.5 ms.
	 * It causes problem in the Supplicant
	 */

	os_sched_timeout(1000);
	wireless_send_event(priv->wlan_dev.netdev,
		       		SIOCGIWAP, &wrqu, NULL);

	/* Free Tx and Rx packets */
	os_free_tx_packet(priv);
	wlan_send_rxskbQ(priv);

	/* report disconnect to upper layer */
	os_stop_queue(priv);
	os_carrier_off(priv);

	/* reset SNR/NF/RSSI values */
	memset(Adapter->SNR, 0x00, sizeof(Adapter->SNR));
	memset(Adapter->NF, 0x00, sizeof(Adapter->NF));
	memset(Adapter->RSSI, 0x00, sizeof(Adapter->RSSI));
	
	PRINTM(INFO, "Current SSID=%s, Ssid Length=%u\n",
			Adapter->CurBssParams.ssid.Ssid, 
			Adapter->CurBssParams.ssid.SsidLength);
	PRINTM(INFO, "Previous SSID=%s, Ssid Length=%u\n",
			Adapter->PreviousSSID.Ssid, 
			Adapter->PreviousSSID.SsidLength);

	/* reset internal flags */
	Adapter->AdHocCreated = FALSE;

#ifdef WMM
	if(Adapter->wmm.enabled) {
		Adapter->wmm.enabled = FALSE;
		wmm_cleanup_queues(priv);
	}
#endif /* WMM */
		
#ifdef WPA
	Adapter->SecInfo.WPAEnabled = FALSE;
#ifdef WPA2
	Adapter->SecInfo.WPA2Enabled = FALSE;
#endif
	Adapter->Wpa_ie_len = 0;
#endif
	Adapter->SecInfo.Auth1XAlg = WLAN_1X_AUTH_ALG_NONE;
	Adapter->SecInfo.EncryptionMode = CIPHER_NONE;


	Adapter->MediaConnectStatus = WlanMediaStateDisconnected;
	Adapter->LinkSpeed = MRVDRV_LINK_SPEED_1mbps;
		
	/* 
	 * memorize the previous SSID and BSSID
	 * it could be used for re-assoc
	 */
	memcpy(&(Adapter->PreviousSSID), &(Adapter->CurBssParams.ssid),
			sizeof(WLAN_802_11_SSID));
	memcpy(Adapter->PreviousBSSID, Adapter->CurrentBSSID,
			MRVDRV_ETH_ADDR_LEN);

	memcpy(&Adapter->SpecificScanSSID, &Adapter->PreviousSSID,
		sizeof(WLAN_802_11_SSID));

	/* need to erase the current SSID and BSSID info */
	Adapter->ulCurrentBSSIDIndex = 0;
	memset(&Adapter->CurBssParams, 0, sizeof(Adapter->CurBssParams));
	memset(Adapter->CurrentBSSID, 0, MRVDRV_ETH_ADDR_LEN);
	memset(&(Adapter->CurrentBSSIDDescriptor), 0,
			sizeof(WLAN_802_11_BSSID));

#ifdef	PS_REQUIRED
	if (Adapter->PSState != PS_STATE_FULL_POWER) {
		/* make firmware to exit PS mode */
		PRINTM(INFO, "Disconnected, so exit PS mode.\n");
		PSWakeup(priv, 0);
	}
#endif

	LEAVE();
}

/** 
 *  @brief This function handles link lost, deauth and
 *  disassoc events.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void HandleDisconnectEvent(wlan_private * priv)
{
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		MacEventDisconnected(priv);
#ifdef REASSOCIATION
		if (Adapter->Reassoc_on == TRUE ) {
			PRINTM(INFO, "RE-ASSOC: trigger the timer\n");
			Adapter->TimerIsSet = TRUE;
			ModTimer(&Adapter->MrvDrvTimer, 0);
		}
#endif /* REASSOCIATION */
	}
}

#ifdef WPA
/** 
 *  @brief This function handles MIC failure event.
 *
 *  @param priv    A pointer to wlan_private structure
 *  @para  event   the event id
 *  @return 	   n/a
 */
static void HandleMICFailureEvent(wlan_private *priv, u32 event)
{
	u8		buf[50];
	
	ENTER();

	memset(buf, 0, sizeof(buf));
	
	sprintf(buf, "%s", "MLME-MICHAELMICFAILURE.indication ");
	
	if (event == MACREG_INT_CODE_MIC_ERR_UNICAST) {
		strcat(buf, "unicast ");
	} else {
		strcat(buf, "multicast ");
	}
	
	send_iwevcustom_event(priv, buf);

	LEAVE();
}
#endif

#ifdef BG_SCAN
/** 
 *  @brief This function sends BG_SCAN query command to firmware.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int sendBgScanQueryCmd(wlan_private *priv)
{
	wlan_adapter    *Adapter = priv->adapter;

	/* Clear the previous scan result */
	Adapter->ulNumOfBSSIDs = 0;
	memset(Adapter->BSSIDList, 0, 
		sizeof(WLAN_802_11_BSSID) * MRVDRV_MAX_BSSID_LIST);

	return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_BG_SCAN_QUERY,
			0, 0, 0, NULL);
}
#endif /* BG_SCAN */

#ifdef SUBSCRIBE_EVENT_CTRL
/** 
 *  @brief This function handles the command response of subscribe
 *  events.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_ret_802_11_subscribe_events(wlan_private *priv,
						HostCmd_DS_COMMAND *resp)
{
	HostCmd_DS_802_11_SUBSCRIBE_EVENT	*subevent = 
							&resp->params.events;
	wlan_adapter				*Adapter = priv->adapter;

	ENTER();

	if (subevent->Action == HostCmd_ACT_GET) {
		PRINTM(INFO, "Get Subscribed Events\n");
		Adapter->GetEvents = subevent->Events;
	}
	
	LEAVE();

	return WLAN_STATUS_SUCCESS;
}
#endif

/** 
 *  @brief This function interprets BSS description in scan response.
 *  
 *  @param priv    		A pointer to wlan_private structure
 *  @param ret_cmd_buf		A pointer to the buffer of command resp
 *  @param bss_descp_list_size	The size of BSS description list
 *  @return 	   		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int InterpretBSSDescriptionWithIE(wlan_private * priv,
			      		void *ret_cmd_buf,
			      		u32 bss_descp_list_size,
			      		u32 next_bssid_index)
{

	wlan_adapter			*Adapter = priv->adapter;
	u8				*pCurrentPtr, *pNextBeacon;
	u32				nBytesLeft;
	u32				ulBSSIDIndex = next_bssid_index;

	PWLAN_802_11_BSSID		pBSSIDList;
	IEEEtypes_ElementId_e		nElemID;
	IEEEtypes_FhParamSet_t		*pFH;
	IEEEtypes_DsParamSet_t		*pDS;
	IEEEtypes_CfParamSet_t		*pCF;
	IEEEtypes_IbssParamSet_t	*pIbss;
	IEEEtypes_CapInfo_t		*pCap;

	/* store the BSSID's information element */
	MRV_BSSID_IE_LIST       	IEBuffer;
	WLAN_802_11_FIXED_IEs		*pFixedIE;
	WLAN_802_11_VARIABLE_IEs	*pVariableIE;

	u16				usBeaconSize;
	u16				usBytesLeftForCurrentBeacon;
	u8				ucElemLen, ucRateSize = 0;
	BOOLEAN				bFoundDataRateIE;
	u16				usNumLineToPrint;

#ifdef WPA
	PWPA_SUPPLICANT			pwpa_supplicant;
#ifdef WPA2
	PWPA_SUPPLICANT			pwpa2_supplicant;
#endif /* WPA2 */
	IE_WPA				*pIe;
	u8				oui01[4] = { 0x00, 0x50, 0xf2, 0x01 };
#endif /* WPA */
#ifdef WMM
	static u8 oui02[4] = { 0x00, 0x50, 0xf2, 0x02 };
#endif /* WMM */

#ifdef ENABLE_802_11D
	IEEEtypes_CountryInfoSet_t 	*pcountryinfo;
#endif
	
	int				ret = WLAN_STATUS_SUCCESS;

	ENTER();

	/* Initialize the current pointer to the Command buffer */
	pNextBeacon = pCurrentPtr = (u8 *) ret_cmd_buf;

	if (!next_bssid_index) {
		/* Clean up the current BSSID List */
		memset(Adapter->BSSIDList, 0,
		sizeof(WLAN_802_11_BSSID) * MRVDRV_MAX_BSSID_LIST);
	}
    
	/* At the beginning, nBytesLeft is the total BSSDescription List */
	nBytesLeft = (LONG) bss_descp_list_size;

	/*
	 *  expected format:
	 *  
	 *  Length - 2 bytes
	 *  BSSId - 6 bytes
	 *  IE - variable length
	 */

	while (nBytesLeft > 0 && ulBSSIDIndex < Adapter->ulNumOfBSSIDs) {

		pCurrentPtr = pNextBeacon;

		memcpy(&usBeaconSize, pCurrentPtr, sizeof(u16));

		usBeaconSize = wlan_le16_to_cpu(usBeaconSize);
		
		usBytesLeftForCurrentBeacon = usBeaconSize;

		if (usBeaconSize > nBytesLeft) {
			Adapter->ulNumOfBSSIDs = ulBSSIDIndex;
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}

		usNumLineToPrint = (usBeaconSize + 15) >> 4;

		pCurrentPtr += sizeof(usBeaconSize);

		pNextBeacon = pCurrentPtr + usBeaconSize;

		pBSSIDList = &Adapter->BSSIDList[ulBSSIDIndex];
		pFixedIE = &IEBuffer.FixedIE;
		
		pBSSIDList->IELength = wlan_le32_to_cpu(pBSSIDList->IELength);
		PRINTM(INFO, "InterpretIE: pFixedIE->Capabilities=0x%X\n", pFixedIE->Capabilities);

		pVariableIE = (PWLAN_802_11_VARIABLE_IEs)IEBuffer.VariableIE;

#ifdef WPA
		pwpa_supplicant = &Adapter->BSSIDList[ulBSSIDIndex].wpa_supplicant;
#ifdef WPA2
		pwpa2_supplicant = &Adapter->BSSIDList[ulBSSIDIndex].wpa2_supplicant;
#endif //WPA2
#endif //WPA
		memcpy(pBSSIDList->MacAddress, pCurrentPtr, 
               		MRVDRV_ETH_ADDR_LEN);
		PRINTM(INFO, "InterpretIE: AP MAC Addr-%x:%x:%x:%x:%x:%x\n",
			pBSSIDList->MacAddress[0], pBSSIDList->MacAddress[1],
			pBSSIDList->MacAddress[2], pBSSIDList->MacAddress[3],
			pBSSIDList->MacAddress[4], pBSSIDList->MacAddress[5]);

		pCurrentPtr += MRVDRV_ETH_ADDR_LEN;
		usBytesLeftForCurrentBeacon -= MRVDRV_ETH_ADDR_LEN;
		pBSSIDList->Configuration.Length =
					sizeof(WLAN_802_11_CONFIGURATION);

		if (usBytesLeftForCurrentBeacon < 12) {
			PRINTM(INFO, "InterpretIE: Not enough bytes left\n");
			Adapter->ulNumOfBSSIDs = ulBSSIDIndex;
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
		
		/* rest of the current buffer are IE's */
		pBSSIDList->IELength = usBytesLeftForCurrentBeacon;
		PRINTM(INFO, "InterpretIE: IELength for this AP = %d\n", 
						(u32) pBSSIDList->IELength);

		pBSSIDList->Rssi = wlan_le32_to_cpu((LONG) (*pCurrentPtr));
		PRINTM(INFO, "InterpretIE: RSSI=%02X\n", *pCurrentPtr );
		pCurrentPtr +=1;
		usBytesLeftForCurrentBeacon -= 1;

		/*
		 * next 3 fields are time stamp, beacon interval, 
		 * and capability information
		 * time stamp is 8 byte long
		 */
		memcpy(pFixedIE->Timestamp, pCurrentPtr, 8);
		memcpy(pBSSIDList->TimeStamp, pCurrentPtr, 8);
		pCurrentPtr += 8;
		usBytesLeftForCurrentBeacon -= 8;

		/* beacon interval is 2 byte long */
		memcpy(&(pFixedIE->BeaconInterval), pCurrentPtr, 2);

		pBSSIDList->Configuration.BeaconPeriod =
				wlan_le16_to_cpu(pFixedIE->BeaconInterval);

		pCurrentPtr += 2;
		usBytesLeftForCurrentBeacon -= 2;

		/* capability information is 2 byte long */
		memcpy(&(pFixedIE->Capabilities), pCurrentPtr, 2);
		
		PRINTM(INFO, "InterpretIE: pFixedIE->Capabilities=0x%X\n", 
				pFixedIE->Capabilities);
		
		pBSSIDList->Configuration.BeaconPeriod =
				wlan_le16_to_cpu(pFixedIE->BeaconInterval);

		PRINTM(INFO, "InterpretIE: pFixedIE->Capabilities=0x%X\n", 
				pFixedIE->Capabilities);	
		
		pFixedIE->Capabilities = wlan_le16_to_cpu(pFixedIE->Capabilities);
			
		pCap = (IEEEtypes_CapInfo_t *) & (pFixedIE->Capabilities);
		
		if (pCap->Privacy) {
			PRINTM(INFO, "InterpretIE: AP WEP enabled\n");
			pBSSIDList->Privacy = Wlan802_11PrivFilter8021xWEP;
		} else {
			pBSSIDList->Privacy = Wlan802_11PrivFilterAcceptAll;
		}

		if (pCap->Ibss == 1) {
			pBSSIDList->InfrastructureMode = Wlan802_11IBSS;
		} else {
			pBSSIDList->InfrastructureMode = Wlan802_11Infrastructure;
		}

		memcpy(&pBSSIDList->Cap, pCap, sizeof(IEEEtypes_CapInfo_t));

#ifdef ENABLE_802_11H
		if (pCap->SpectrumMgmt == 1) {
			PRINTM(INFO, "InterpretIE: 11h- Spectrum Management capability bit found\n");
			pBSSIDList->wlan_11h_bss_info.sensed11h = 1;
		}
#endif
		pCurrentPtr += 2;
		usBytesLeftForCurrentBeacon -= 2;

		if (usBytesLeftForCurrentBeacon > 
					MRVDRV_SCAN_LIST_VAR_IE_SPACE) {
			/*
			 * not enough space to copy the IE for the current AP
			 * just skip it for now
			 */
			pCurrentPtr += usBytesLeftForCurrentBeacon;
			nBytesLeft -= usBeaconSize;
			Adapter->ulNumOfBSSIDs--;
			PRINTM(INFO, "InterpretIE: Variable IE size too big to fit into buffer, "
								"skip\n");
		 	continue;
		}
		/*
		 * reset are variable IE
		 * copy the entire variable IE to IE buffer
		 */
		memcpy(pVariableIE, pCurrentPtr, usBytesLeftForCurrentBeacon);

		HEXDUMP("InterpretIE: IE info", (u8 *) pVariableIE,
						usBytesLeftForCurrentBeacon);

		bFoundDataRateIE = FALSE;

		/* process variable IE */
        	while (usBytesLeftForCurrentBeacon > 2) {
            		nElemID = (IEEEtypes_ElementId_e)(*((u8 *)pCurrentPtr));
            		ucElemLen = *((u8 *) pCurrentPtr + 1);

           		 usNumLineToPrint = (ucElemLen + 15) >> 4;
            
            		if (usBytesLeftForCurrentBeacon < ucElemLen) {
                		nBytesLeft -= usBeaconSize;
               			 /* Keep AP info and only ignore the paddings
               				 Adapter->ulNumOfBSSIDs--;
               			 */
               			PRINTM(INFO, "InterpretIE: Error in processing IE, bytes left < IE length\n");
               			usBytesLeftForCurrentBeacon = 0;
                		continue;
            		}

            		if ( ucElemLen == 0x00 ) {
                		PRINTM(INFO, "InterpretIE: Ignore IE with zero Length\n");
               			pCurrentPtr += 2;
                		usBytesLeftForCurrentBeacon -= 2;
                		continue;
            		}               

           		switch (nElemID) {
            
            		case SSID:
               			pBSSIDList->Ssid.SsidLength = ucElemLen;
                		memcpy(pBSSIDList->Ssid.Ssid, (pCurrentPtr + 2), ucElemLen);

               			break;
                
            		case SUPPORTED_RATES:
                		memcpy(pBSSIDList->DataRates, (pCurrentPtr + 2), ucElemLen);
               			memmove(pBSSIDList->SupportedRates, (pCurrentPtr + 2),
                        	ucElemLen);
                		ucRateSize = ucElemLen;
                		bFoundDataRateIE = TRUE;
                		break;

            		case EXTRA_IE:
                		PRINTM(INFO, "InterpretIE: EXTRA_IE Found!\n" );
                		pBSSIDList->extra_ie = 1;
                		break;

           		case FH_PARAM_SET:
                		pFH = (IEEEtypes_FhParamSet_t *)pCurrentPtr;
                		pBSSIDList->NetworkTypeInUse = Wlan802_11FH;
               			pBSSIDList->Configuration.DSConfig = 0;
                    
                		pBSSIDList->Configuration.FHConfig.Length =
                    			sizeof(WLAN_802_11_CONFIGURATION_FH);

                		pBSSIDList->Configuration.FHConfig.HopPattern = pFH->HopSet;
                		pBSSIDList->Configuration.FHConfig.DwellTime = pFH->DwellTime;

                		memmove(&pBSSIDList->PhyParamSet.FhParamSet, pFH, 
                        		sizeof(IEEEtypes_FhParamSet_t));
                    
                		pBSSIDList->PhyParamSet.FhParamSet.DwellTime 
                    			= wlan_le16_to_cpu(pBSSIDList
                                       		->PhyParamSet.FhParamSet.DwellTime);
               			break;

           		case DS_PARAM_SET:
                		pDS = (IEEEtypes_DsParamSet_t *)pCurrentPtr;

                		pBSSIDList->NetworkTypeInUse = Wlan802_11DS;

                		pBSSIDList->Configuration.DSConfig 
                    				= DSFreqList[pDS->CurrentChan];

                		pBSSIDList->Channel = pDS->CurrentChan;

               			memcpy(&pBSSIDList->PhyParamSet.DsParamSet, pDS, 
                       				sizeof(IEEEtypes_DsParamSet_t));
                		break;

            		case CF_PARAM_SET:
                		pCF = (IEEEtypes_CfParamSet_t *)pCurrentPtr;
                    
                		memcpy(&(pBSSIDList->SsParamSet.CfParamSet), pCF, 
                       			sizeof(IEEEtypes_CfParamSet_t));
                		break;

            		case IBSS_PARAM_SET:
               			pIbss = (IEEEtypes_IbssParamSet_t *)pCurrentPtr;
                		pBSSIDList->Configuration.ATIMWindow
                    			= wlan_le32_to_cpu(pIbss->AtimWindow);

                		memmove(&pBSSIDList->SsParamSet.IbssParamSet, pIbss, 
                        		sizeof(IEEEtypes_IbssParamSet_t));

                		pBSSIDList->SsParamSet.IbssParamSet.AtimWindow 
                   			 = wlan_le16_to_cpu(pBSSIDList
                                       		->SsParamSet.IbssParamSet.AtimWindow);
                		break;

#ifdef ENABLE_802_11D
			/* Handle Country Info IE */
            		case COUNTRY_INFO: 
                		pcountryinfo = (IEEEtypes_CountryInfoSet_t *)pCurrentPtr;
                		
				if ( pcountryinfo->Len < sizeof(pcountryinfo->CountryCode) ||
                     			pcountryinfo->Len > 254 ) {
                    			PRINTM(INFO, "InterpretIE: 11D- Err CountryInfo len =%d min=%d max=254\n", 
                            		pcountryinfo->Len,
                            		sizeof(pcountryinfo->CountryCode) );
                  			ret = WLAN_STATUS_FAILURE;
					goto done;
                		}
                    
                		memcpy( &(pBSSIDList->CountryInfo),  pcountryinfo, 
                        			pcountryinfo->Len + 2 );
               			HEXDUMP("InterpretIE: 11D- CountryInfo:", (u8 *)pcountryinfo,
                        		(u32)(pcountryinfo->Len+2) );
                		break;
#endif

#ifdef ENABLE_802_11H
			case POWER_CONSTRAINT:
			case POWER_CAPABILITY:
			case TPC_REPORT:
			case CHANNEL_SWITCH_ANN:
			case QUIET:
			case IBSS_DFS:
			case SUPPORTED_CHANNELS:
			case TPC_REQUEST:
	                    wlan_11h_process_bss_elem(priv, 
                                              &pBSSIDList->wlan_11h_bss_info,
                                              pCurrentPtr);
        	            break;
#endif /* ENABLE_802_11H */
            		case EXTENDED_SUPPORTED_RATES:
                	/*
			 * only process extended supported rate
			 * if data rate is already found.
			 * data rate IE should come before
			 * extended supported rate IE
		 	 */
                	if (bFoundDataRateIE) {
                    		u8      *pRate;
                    		u8      ucBytesToCopy;
                    		if ((ucElemLen + ucRateSize) > WLAN_SUPPORTED_RATES) {
                        		ucBytesToCopy = (WLAN_SUPPORTED_RATES - ucRateSize);
                    		}
                    		else {
		                        ucBytesToCopy = ucElemLen;
                    		}

                    		pRate = (u8 *)pBSSIDList->DataRates;
                    		pRate += ucRateSize;
                    		memmove(pRate,(pCurrentPtr +2), ucBytesToCopy);

                    		pRate = (u8 *)pBSSIDList->SupportedRates;

                    		pRate += ucRateSize;
                    		memmove(pRate,(pCurrentPtr +2), ucBytesToCopy);
                	}
                	
			break;

            		case VENDOR_SPECIFIC_221:
#ifdef WPA
#define IE_ID_LEN_FIELDS_BYTES 2
                	pIe = (IE_WPA *) pCurrentPtr;

               		if (!memcmp(pIe->oui, oui01, sizeof(oui01))) {
                   		pwpa_supplicant->Wpa_ie_len 
		                        = MIN(ucElemLen + IE_ID_LEN_FIELDS_BYTES,
                		              sizeof(pwpa_supplicant->Wpa_ie));
				memcpy(pwpa_supplicant->Wpa_ie, 
		                           pCurrentPtr, pwpa_supplicant->Wpa_ie_len);
                		HEXDUMP("InterpretIE: Resp WPA_IE",
		                            pwpa_supplicant->Wpa_ie, ucElemLen);
                	}
#ifdef WMM
	                else if (!memcmp(pIe->oui, oui02, sizeof(oui02))) {
        	        	pBSSIDList->Wmm_ie_len 
                		        = MIN(ucElemLen + IE_ID_LEN_FIELDS_BYTES, 
		                              sizeof(pBSSIDList->Wmm_IE));
                	    	memcpy(pBSSIDList->Wmm_IE, pCurrentPtr,
                        	   	pBSSIDList->Wmm_ie_len);

	                    	HEXDUMP("InterpretIE: Resp WMM_IE", pBSSIDList->Wmm_IE,
        	                    	pBSSIDList->Wmm_ie_len);
                	}
#endif /* WMM */
#endif /* WPA */
#ifdef CCX
                	wlan_ccx_process_bss_elem(&pBSSIDList->ccx_bss_info,
                                          pCurrentPtr);
#endif
                	break;
#ifdef WPA2
            		case WPA2_IE:
                		pIe = (IE_WPA *) pCurrentPtr;
                		pwpa2_supplicant->Wpa_ie_len 
                   			= MIN(ucElemLen + IE_ID_LEN_FIELDS_BYTES,
                          		sizeof(pwpa2_supplicant->Wpa_ie));
                		memcpy(pwpa2_supplicant->Wpa_ie, 
                       			pCurrentPtr, pwpa2_supplicant->Wpa_ie_len);
				
                		HEXDUMP("InterpretIE: Resp WPA2_IE",
					       pwpa2_supplicant->Wpa_ie, ucElemLen);
                		break;
#endif //WPA2
            		case TIM:
               			break;
                
            		case CHALLENGE_TEXT:
                		break;
#ifdef CCX
            		default:
                		wlan_ccx_process_bss_elem(&pBSSIDList->ccx_bss_info,
                                          		pCurrentPtr);
                		break;
#endif
            		}

            		pCurrentPtr += ucElemLen + 2;
            	
			/* need to account for IE ID and IE Len */
            		usBytesLeftForCurrentBeacon =
                	usBytesLeftForCurrentBeacon - ucElemLen - 2;
        	} /* while (usBytesLeftForCurrentBeacon > 2) */
		
		/* update the length field */
		pBSSIDList->Length = sizeof(WLAN_802_11_BSSID) + pBSSIDList->IELength;

		/* need to be 4 byte aligned */
		pBSSIDList->Length = ((pBSSIDList->Length + 3) >> 2) << 2;

		nBytesLeft -= usBeaconSize;
		ulBSSIDIndex++;

	} /* while (nBytesLeft > 0 && ulBSSIDIndex <  MRVDRV_MAX_BSSID_LIST) */

done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles the command response of reg_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param type	   the type of reg access (MAC, BBP or RF)
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_reg_access(wlan_private * priv,
	       			u16 type, 
			        HostCmd_DS_COMMAND * resp)
{
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

    	switch (type) {
	    	case HostCmd_RET_MAC_REG_ACCESS:
			{
				HostCmd_DS_MAC_REG_ACCESS *reg;

				reg = (PHostCmd_DS_MAC_REG_ACCESS) & 
				resp->params.macreg;

				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
				break;
			}
	
	    	case HostCmd_RET_BBP_REG_ACCESS:
			{
				HostCmd_DS_BBP_REG_ACCESS *reg;
				reg = (PHostCmd_DS_BBP_REG_ACCESS) & 
					resp->params.bbpreg;

				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
				break;
			}

	    	case HostCmd_RET_RF_REG_ACCESS:
			{
				HostCmd_DS_RF_REG_ACCESS *reg;
				reg = (PHostCmd_DS_RF_REG_ACCESS) & 
					resp->params.rfreg;
				
				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
	    			break;
			}

		default:
			LEAVE();
			return WLAN_STATUS_FAILURE;
	}

    	LEAVE();
    	return WLAN_STATUS_SUCCESS;
}
 
/** 
 *  @brief This function handles the command response of get_hw_spec
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_get_hw_spec(wlan_private * priv, 
				HostCmd_DS_COMMAND * resp)
{
	u32			i;
	HostCmd_DS_GET_HW_SPEC	*hwspec = &resp->params.hwspec;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;

	ENTER();

	Adapter->HardwareStatus = WlanHardwareStatusReady;
	Adapter->fwCapInfo = wlan_le32_to_cpu(hwspec->fwCapInfo);

#ifdef MULTI_BANDS
	if (IS_SUPPORT_MULTI_BANDS(Adapter)) {
		Adapter->fw_bands		= GET_FW_DEFAULT_BANDS(Adapter);
		Adapter->is_multiband		= 1;
	} else {
		Adapter->adhoc_start_band	= BAND_B;
		Adapter->fw_bands		= BAND_B;
		Adapter->is_multiband		= 0;
	}

	Adapter->config_bands = Adapter->fw_bands; 

	memset(&SupportedRates, 0, sizeof(SupportedRates));

	if (Adapter->fw_bands & BAND_A) {
		Adapter->Channel = DEFAULT_CHANNEL_A;
		Adapter->adhoc_start_band = BAND_A;
		Adapter->AdhocChannel	  = DEFAULT_AD_HOC_CHANNEL_A;
	} else if (Adapter->fw_bands & BAND_G) {
		Adapter->Channel = DEFAULT_CHANNEL;
		Adapter->adhoc_start_band = BAND_G;
		Adapter->AdhocChannel	  = DEFAULT_AD_HOC_CHANNEL;
	} else if (Adapter->fw_bands & BAND_B) {
		Adapter->Channel = DEFAULT_CHANNEL;
		Adapter->adhoc_start_band = BAND_B;
		Adapter->AdhocChannel	  = DEFAULT_AD_HOC_CHANNEL;
	}
#endif

	/* permanent address should only be set once at start up */
	if (Adapter->PermanentAddr[0] == 0xff) {
		/* permanent address has not been set yet, set it */
		memcpy(Adapter->PermanentAddr, hwspec->PermanentAddr,
							MRVDRV_ETH_ADDR_LEN);
	}

	memcpy(priv->wlan_dev.netdev->dev_addr, hwspec->PermanentAddr,
								ETH_ALEN);
	Adapter->FWReleaseNumber = hwspec->FWReleaseNumber;

	PRINTM(INFO, "GET_HW_SPEC: FWReleaseVersion- 0x%X\n", Adapter->FWReleaseNumber);
	PRINTM(INFO, "GET_HW_SPEC: Permanent addr- %2x:%2x:%2x:%2x:%2x:%2x\n",
			hwspec->PermanentAddr[0], hwspec->PermanentAddr[1],
			hwspec->PermanentAddr[2], hwspec->PermanentAddr[3],
			hwspec->PermanentAddr[4], hwspec->PermanentAddr[5]);
	PRINTM(INFO, "GET_HW_SPEC: HWIfVersion=0x%X  Version=0x%X\n",hwspec->HWIfVersion,
							hwspec->Version);

#ifdef HARDCODED_REGION_CODE
	Adapter->RegionTableIndex = 0;
	Adapter->RegionCode = 0x10;
#else
	Adapter->RegionCode = wlan_le16_to_cpu(hwspec->RegionCode) >> 8;
#endif

	for (i = 0; i < MRVDRV_MAX_REGION_CODE; i++) {
		/* use the region code to search for the index */
		if (Adapter->RegionCode == RegionCodeToIndex[i]) {
			Adapter->RegionTableIndex = (u16) i;
			break;
		}
	}

	/* if it's unidentified region code, use the default (USA) */
	if (i >= MRVDRV_MAX_REGION_CODE) {
		Adapter->RegionCode = 0x10;
		Adapter->RegionTableIndex = 0;
	}

	if (Adapter->CurrentAddr[0] == 0xff) {
		memmove(Adapter->CurrentAddr, hwspec->PermanentAddr,
				MRVDRV_ETH_ADDR_LEN);
	}

#ifdef MULTI_BANDS
	if (wlan_set_regiontable(priv, Adapter->RegionCode, 
							Adapter->fw_bands)) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
 
#ifdef ENABLE_802_11D
	if (wlan_set_universaltable(priv, Adapter->fw_bands)) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif /* ENABLE_802_11D */

#else
	if (wlan_set_regiontable(priv, Adapter->RegionCode, 0)) {
			ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#ifdef ENABLE_802_11D
	if (wlan_set_universaltable(priv, 0)) {
			ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif /* ENABLE_802_11D */

#endif /* MULTI_BANDS */

done:
	LEAVE();
	return ret;
}

#ifdef PS_REQUIRED
/** 
 *  @brief This function handles the command response of ps_mode
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_ps_mode(wlan_private * priv, 
				HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

#ifdef FW_WAKEUP_METHOD
/** 
 *  @brief This function handles the command response of fw_wakeup_method
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_fw_wakeup_method(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	wlan_adapter				*Adapter = priv->adapter;
	HostCmd_DS_802_11_FW_WAKEUP_METHOD	*fwwm = &resp->params.fwwakeupmethod;
	u16					action;

	ENTER();

	action = wlan_le16_to_cpu(fwwm->Action);

	switch (action) {
	case HostCmd_ACT_GET:
	case HostCmd_ACT_SET:
		Adapter->fwWakeupMethod = wlan_le16_to_cpu(fwwm->Method);
		break;
	default:
		break;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* FW_WAKEUP_METHOD */
#endif /* PS_REQUIRED */

/** 
 *  @brief This function handles the command response of fw_wakeup_method
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param bssid_index  the index of BSSID
 *  @return 	   	n/a
 */
static void AppendCurrentSSID(wlan_private *priv, u32 bssid_index)
{
	wlan_adapter	*Adapter = priv->adapter;

	if (Adapter->SetSpecificScanSSID != TRUE) {
		
		PRINTM(INFO, "Append current SSID to " "SCAN list\n");

		if (bssid_index < MRVDRV_MAX_BSSID_LIST) {
			
			Adapter->ulCurrentBSSIDIndex = bssid_index;
			Adapter->ulNumOfBSSIDs++;
			memcpy(&Adapter->BSSIDList[bssid_index],
					&Adapter->CurrentBSSIDDescriptor,
					sizeof(WLAN_802_11_BSSID));

#ifdef MULTI_BANDS
			Adapter->BSSIDList[bssid_index].bss_band = Adapter->CurBssParams.band;
#endif
			Adapter->BSSIDList[bssid_index].Channel = Adapter->CurBssParams.channel;

			if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled) {
				Adapter->BSSIDList[bssid_index].Privacy = 
					Wlan802_11PrivFilter8021xWEP;
			} else {
				Adapter->BSSIDList[bssid_index].Privacy =
					Wlan802_11PrivFilterAcceptAll;
			}

		}
	}
}

/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_sleep_params(wlan_private * priv,
			      		HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_SLEEP_PARAMS 	*sp = &resp->params.sleep_params;
	wlan_adapter   			*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "error=%x offset=%x stabletime=%x calcontrol=%x\n" 
			" extsleepclk=%x\n", sp->Error, sp->Offset, 
			sp->StableTime, sp->CalControl, sp->ExternalSleepClk);
	Adapter->sp.sp_error = wlan_le16_to_cpu(sp->Error);
	Adapter->sp.sp_offset = wlan_le16_to_cpu(sp->Offset);
	Adapter->sp.sp_stabletime = wlan_le16_to_cpu(sp->StableTime);
	Adapter->sp.sp_calcontrol = wlan_le16_to_cpu(sp->CalControl);
	Adapter->sp.sp_extsleepclk = wlan_le16_to_cpu(sp->ExternalSleepClk);
	Adapter->sp.sp_reserved = wlan_le16_to_cpu(sp->Reserved);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_sleep_period(wlan_private * priv,
			      		HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_SLEEP_PERIOD	*sp_period = &resp->params.ps_sleeppd;
	wlan_adapter   			*Adapter = priv->adapter;

	ENTER();

	Adapter->sleep_period.period = wlan_le16_to_cpu(sp_period->Period);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef BCA
/** 
 *  @brief This function handles the command response of bca_timeshare
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_bca_timeshare(wlan_private *priv, 
					HostCmd_DS_COMMAND *resp)
{
	HostCmd_DS_802_11_BCA_TIMESHARE	*bca_ts = &resp->params.bca_timeshare;
	wlan_adapter   			*Adapter = priv->adapter;

	ENTER();

	PRINTM(MSG, "TrafficType=%x TimeShareInterva=%x BTTime=%x\n", 
			bca_ts->TrafficType, bca_ts->TimeShareInterval, 
			bca_ts->BTTime);
	
	Adapter->bca_ts.TrafficType = wlan_le16_to_cpu(bca_ts->TrafficType);
	Adapter->bca_ts.TimeShareInterval = wlan_le32_to_cpu(bca_ts->TimeShareInterval);
	Adapter->bca_ts.BTTime = wlan_le32_to_cpu(bca_ts->BTTime);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif

/** 
 *  @brief This function handles the command response of bca_timeshare
 *  
 *  @param adpt    A pointer to wlan_adapter structure
 *  @return 	   n/a
 */
static void discard_bad_ssid(wlan_adapter *adpt)
{
	u32 bad_count = 0, i = 0, j = 0;

	for (i = 0; i < adpt->ulNumOfBSSIDs; i++) {
		for (j = 0; j < adpt->BSSIDList[i].Ssid.SsidLength; j++) {
			if (adpt->BSSIDList[i].Ssid.Ssid[j] < 0x20) {
		        	bad_count ++;
	        		break;
		      	}
		}

	    	if ((j == adpt->BSSIDList[i].Ssid.SsidLength) && (bad_count > 0)) {
      			memmove(&adpt->BSSIDList[i - bad_count], 
        			&adpt->BSSIDList[i], 
        			sizeof(WLAN_802_11_BSSID));
	    	}
  	}

	adpt->ulNumOfBSSIDs -= bad_count;
	memset(&adpt->BSSIDList[adpt->ulNumOfBSSIDs], 0, sizeof(WLAN_802_11_BSSID));
	
      	return;
}

/** 
 *  @brief This function handles the command response of scan
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_scan(wlan_private * priv, 
				HostCmd_DS_COMMAND * resp)
{
	u32				i, j;
	u32				ndStat;
	HostCmd_DS_802_11_SCAN_RSP	*scan; 
	wlan_adapter			*Adapter = priv->adapter;
	u16				BSSDescriptSize;
	u16				IncrementSize = 8;
	int				ret = WLAN_STATUS_SUCCESS; 

	ENTER();

#ifdef BG_SCAN
	if (priv->adapter->bgScanConfig->Enable) {
		scan = &resp->params.bgscanqueryresp.scanresp;
		IncrementSize += sizeof(HostCmd_DS_802_11_BG_SCAN_QUERY_RSP);
        } else {
#endif /* BG_SCAN */
                scan = &resp->params.scanresp;
		IncrementSize += sizeof(HostCmd_DS_802_11_SCAN_RSP);
#ifdef BG_SCAN
	}
#endif /* BG_SCAN */

	BSSDescriptSize = wlan_le16_to_cpu(scan->BSSDescriptSize);

	PRINTM(INFO, "SCAN_RESP: BSSDescriptSize %d\n", BSSDescriptSize);
	PRINTM(INFO, "SCAN_RESP: IncrementSize %d\n", IncrementSize);
	PRINTM(INFO, "SCAN_RESP: returned %d AP before parsing\n", scan->NumberOfSets);

	if (scan->NumberOfSets > MRVDRV_MAX_BSSID_LIST) {
		PRINTM(INFO, "SCAN_RESP: Invalid number of AP returned!!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	
	/* Get number of BSS Descriptors */
	i = Adapter->ulNumOfBSSIDs;
	Adapter->ulNumOfBSSIDs += scan->NumberOfSets;
	ndStat = InterpretBSSDescriptionWithIE(priv,
			(Adapter->CurCmd->BufVirtualAddr + IncrementSize),
			BSSDescriptSize, i);

	if (ndStat != WLAN_STATUS_SUCCESS) {
		PRINTM(INFO, "SCAN_RESP: InterpretBSSDescriptionWithIE returned ERROR");
	}
	
	for (j = 0; j < scan->NumberOfSets; j++, i++) {
#ifdef MULTI_BANDS
#ifdef BG_SCAN
	    if (priv->adapter->bgScanConfig->Enable) {
		/* remove this after f/w add band info. in scan response */
		MrvlIEtypes_ChanListParamSet_t	*chanList;
		ChanScanParamSet_t		*chanScan = NULL;
		u8				band = BAND_G;	

		chanList = (MrvlIEtypes_ChanListParamSet_t *)((u8 *)Adapter->bgScanConfig + 
					sizeof(HostCmd_DS_802_11_BG_SCAN_CONFIG));
		while (!chanScan &&
			((u8 *)chanList < (u8 *)Adapter->bgScanConfig + Adapter->bgScanConfigSize)) {
			PRINTM(INFO, "SCAN_RESP: Header type=%#x  len=%d\n",chanList->Header.Type,chanList->Header.Len);
			if (chanList->Header.Type == TLV_TYPE_CHANLIST) {
				chanScan = chanList->ChanScanParam;
				break;
			} else {
				chanList = (MrvlIEtypes_ChanListParamSet_t *)
					((u8 *)chanList + chanList->Header.Len + sizeof(chanList->Header));
			}
		}

		if (chanScan) {
			PRINTM(INFO, "SCAN_RESP: radio_type=%#x  chan=%#x\n",chanScan->RadioType,chanScan->ChanNumber);
			if (chanScan->RadioType == HostCmd_SCAN_RADIO_TYPE_A)
				band = BAND_A;
			else
				band = BAND_G;
		} else {
			PRINTM(MSG, "SCAN_RESP: Cannot find valid radio_type/channel info. in channel list\n");
		}

 		Adapter->BSSIDList[i].bss_band = band;
	    } else {
#endif /* BG_SCAN */
 		Adapter->BSSIDList[i].bss_band = Adapter->cur_region_channel->Band;
#ifdef BG_SCAN
	    }
#endif /* BG_SCAN */
#endif /* MULTI_BANDS */
	}

	PRINTM(INFO, "SCAN_RESP: number of BSSID: %d\n", Adapter->ulNumOfBSSIDs);
    
#if DEBUG_VER
#ifdef PROGRESSIVE_SCAN
	if (Adapter->ScanChannelsLeft <= 0) {
#endif
		for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
			PRINTM(INFO, "%2d:%32s-%02x:%02x:%02x:%02x:%02x:%02x "
					"RSSI=%d SR=%x\n", i,
					Adapter->BSSIDList[i].Ssid.Ssid,
					Adapter->BSSIDList[i].MacAddress[0],
					Adapter->BSSIDList[i].MacAddress[1],
					Adapter->BSSIDList[i].MacAddress[2],
					Adapter->BSSIDList[i].MacAddress[3],
					Adapter->BSSIDList[i].MacAddress[4],
					Adapter->BSSIDList[i].MacAddress[5],
					(s32) Adapter->BSSIDList[i].Rssi,
					Adapter->
					BSSIDList[i].SupportedRates[i]);
		}

#ifdef PROGRESSIVE_SCAN
	}
#endif
#endif

  	discard_bad_ssid(Adapter);

	/* if currently connected to an AP or an Ad Hoc network */
	if (Adapter->CurBssParams.ssid.SsidLength != 0 &&
			Adapter->CurBssParams.ssid.Ssid[0] > 0x20) {
		/* try to find the current SSID in the new scan list */
		for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
			if (!SSIDcmp(&Adapter->BSSIDList[i].Ssid,
						&Adapter->CurBssParams.ssid) &&
					!memcmp(Adapter->CurrentBSSID,
						Adapter->BSSIDList[i].
						MacAddress,
						MRVDRV_ETH_ADDR_LEN)) {
				break;
			}
		}
		
		/* if we found matching SSID, update the index */
		if (i < Adapter->ulNumOfBSSIDs) {
			/* Set the attempted BSSID Index to current */
			Adapter->ulCurrentBSSIDIndex = i;
			/* Set the new BSSID (AP's MAC address) to current BSSID */
			memcpy(Adapter->CurrentBSSID,
					&(Adapter->BSSIDList[Adapter->
					ulCurrentBSSIDIndex].MacAddress),
					MRVDRV_ETH_ADDR_LEN);
			
			/* Make a copy of current BSSID descriptor */
			memcpy(&Adapter->CurrentBSSIDDescriptor,
					&Adapter->BSSIDList[Adapter->
					ulCurrentBSSIDIndex],
					sizeof(WLAN_802_11_BSSID));
			/* Set the new configuration to the current config */
			memcpy(&Adapter->CurrentConfiguration,
					&Adapter->BSSIDList[Adapter->
					ulCurrentBSSIDIndex].Configuration, 
					sizeof(WLAN_802_11_CONFIGURATION));
		}
		/*
		 * if the current associated SSID is not contained in the
		 * list, append it
		 */
		else {
#ifdef PROGRESSIVE_SCAN
			if (Adapter->ScanChannelsLeft == 0) {
#endif
				AppendCurrentSSID(priv, i);
#ifdef PROGRESSIVE_SCAN				
			}
#endif
		}
	}

	PRINTM(INFO, "SCAN_RESP: Scanned %2d APs\n", Adapter->ulNumOfBSSIDs);

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles the command response of mac_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_mac_control(wlan_private * priv,
	       			HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of associate
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_associate(wlan_private * priv, 
                                     HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_ASSOCIATE_RSP	*rasst = &resp->params.associatersp;
	wlan_adapter			*Adapter = priv->adapter;
	union iwreq_data		wrqu;
	u16				ResultCode = wlan_le16_to_cpu(rasst->ResultCode);
	int				ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (ResultCode) {
		PRINTM(INFO, "ASSOC_RESP: Association Failed, code = %d\n", rasst->ResultCode);
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			MacEventDisconnected(priv);
		}
	
#ifdef WPA  
	        if (ResultCode == HostCmd_Assoc_RESULT_AUTH_REFUSED) {
			send_iwevcustom_event(priv, "BAD-AUTH.indication ");
		}
#endif
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#ifdef WPA
	{
		/*
		 * copy the association result info to the 
		 * ASSOCIATION_INFO buffer
		 */
		PWLAN_802_11_ASSOCIATION_INFORMATION pInfo;
		u32 buflen;
	
		pInfo = (PWLAN_802_11_ASSOCIATION_INFORMATION)
			Adapter->AssocInfoBuffer;

		/* only currently copy the fixed IE */
		memcpy(&pInfo->ResponseFixedIEs.Capabilities, 
				&rasst->CapInfo, sizeof(rasst->CapInfo));
		pInfo->ResponseFixedIEs.StatusCode = ResultCode;
		pInfo->ResponseFixedIEs.AssociationId =
			wlan_le16_to_cpu(rasst->AssociationID);
		pInfo->AvailableResponseFixedIEs |=
				WLAN_802_11_AI_RESFI_CAPABILITIES;
		pInfo->AvailableResponseFixedIEs |=
				WLAN_802_11_AI_RESFI_STATUSCODE;
		pInfo->AvailableResponseFixedIEs |=
				WLAN_802_11_AI_RESFI_ASSOCIATIONID;
		pInfo->OffsetResponseIEs = pInfo->OffsetRequestIEs + 
				pInfo->RequestIELength;
		pInfo->ResponseIELength = wlan_le16_to_cpu(rasst->IELength);
        	buflen = sizeof(Adapter->AssocInfoBuffer) - pInfo->OffsetResponseIEs;
		memmove((u8*)pInfo + pInfo->OffsetResponseIEs, 
				(u8*)&rasst->RateID, 
                MIN(buflen, wlan_le16_to_cpu(rasst->IELength)));
		PRINTM(INFO, "ASSOC_RESP: CapInfo = 0x%x,"
				"StatusCode = 0x%x,AssociationID = 0x%x\n",
				pInfo->ResponseFixedIEs.Capabilities,
				pInfo->ResponseFixedIEs.StatusCode,
				pInfo->ResponseFixedIEs.AssociationId);

	}
#endif	/* WPA */
	
	HEXDUMP("ASSOC_RESP:", 
			(void*)rasst, sizeof(HostCmd_DS_802_11_ASSOCIATE_RSP));
	
	/* Send a Media Connected event, according to the Spec */
	Adapter->MediaConnectStatus = WlanMediaStateConnected;
	Adapter->LinkSpeed = MRVDRV_LINK_SPEED_11mbps;
	
	/* Set the attempted BSSID Index to current */
	Adapter->ulCurrentBSSIDIndex = Adapter->ulAttemptedBSSIDIndex;
	
	PRINTM(INFO, "ASSOC_RESP: BSSIDList[%d]=%s\n", Adapter->ulCurrentBSSIDIndex,
			Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex].Ssid.Ssid);
	
	/* Set the new SSID to current SSID */
	memcpy(&Adapter->CurBssParams.ssid, &Adapter->
           BSSIDList[Adapter-> ulCurrentBSSIDIndex].Ssid,
           sizeof(WLAN_802_11_SSID));
	
	/* Set the new BSSID (AP's MAC address) to current BSSID */
	memcpy(Adapter->CurrentBSSID, 
           &Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex].MacAddress,
           MRVDRV_ETH_ADDR_LEN);
	
	/* Make a copy of current BSSID descriptor */
	memcpy(&Adapter->CurrentBSSIDDescriptor,
           &Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex],
           sizeof(WLAN_802_11_BSSID));

	/* Copy associate'd bssid into Current BSS State struct */
	memcpy(&Adapter->CurBssParams.bssid, 
			&Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex],
			sizeof(WLAN_802_11_BSSID));
	
#ifdef WMM
	if (Adapter->BSSIDList[Adapter->
			ulCurrentBSSIDIndex].Wmm_IE[0] == WMM_IE) {
		Adapter->CurBssParams.wmm_enabled = TRUE;
		wmm_setup_queue_priorities(priv);
	}
	else
		Adapter->CurBssParams.wmm_enabled = FALSE;

	if (Adapter->wmm.required && Adapter->CurBssParams.wmm_enabled)
		Adapter->wmm.enabled = TRUE;
	else
		Adapter->wmm.enabled = FALSE;
#endif /* WMM */

#ifdef WMM_UAPSD
	Adapter->CurBssParams.wmm_uapsd_enabled = FALSE;

	if (Adapter->wmm.enabled == TRUE)
	{
		if(Adapter->BSSIDList[Adapter->ulCurrentBSSIDIndex].Wmm_IE[8] & 0x80)
		Adapter->CurBssParams.wmm_uapsd_enabled = TRUE;
	}
#endif

	if (Adapter->ulCurrentBSSIDIndex < MRVDRV_MAX_BSSID_LIST) {
		/* Set the new configuration to the current config */
		memcpy(&Adapter->CurrentConfiguration,
				&Adapter->BSSIDList[Adapter->
				ulCurrentBSSIDIndex].
				Configuration, 
				sizeof(WLAN_802_11_CONFIGURATION));
	}
	
	/*
	 *  Association causes the MAC to lose the flag such as
	 *  promiscuous mode on, need to set the flag again
	 */
	if (Adapter->CurBssParams.ssid.Ssid[0] == '\0' ||
			!Adapter->CurBssParams.ssid.SsidLength) {
		memcpy(&Adapter->CurBssParams.ssid, &Adapter->PreviousSSID,
				sizeof(WLAN_802_11_SSID));
	
		memcpy(&Adapter->CurrentBSSID, &Adapter->PreviousBSSID,
				ETH_ALEN);
	}	
	
	PRINTM(INFO, "ASSOC_RESP: CurrentPacketFilter is %x\n",
			Adapter->CurrentPacketFilter);
	
	Adapter->MediaConnectStatus = WlanMediaStateConnected;
	
#ifdef WPA
	if (Adapter->SecInfo.WPAEnabled
#ifdef WPA2
		|| Adapter->SecInfo.WPA2Enabled
#endif
	)
		Adapter->IsGTK_SET = FALSE;
#endif
	
	Adapter->SNR[TYPE_RXPD][TYPE_AVG] = 
		Adapter->NF[TYPE_RXPD][TYPE_AVG] = 0;

#ifdef WMM
	/* Don't enable carrier until we get the WMM_GET_STATUS event */
	if (Adapter->wmm.enabled)
		sendWMMStatusChangeCmd(priv);
	else
#endif /* WMM */
	{
		os_carrier_on(priv);
		os_start_queue(priv);
	}
	
	PRINTM(INFO, "ASSOC_RESP: Associated \n");

	memcpy(wrqu.ap_addr.sa_data, Adapter->CurrentBSSID, ETH_ALEN);
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;
	wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL); 

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles the command response of disassociate
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_disassociate(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	ENTER();
	
	MacEventDisconnected(priv);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of ad_hoc_stop
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_ad_hoc_stop(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	ENTER();
	
	MacEventDisconnected(priv);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of set_wep
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_set_wep(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of ad_hoc_start
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_ad_hoc_start(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	int				ret = WLAN_STATUS_SUCCESS;
	wlan_adapter			*Adapter = priv->adapter;
	u16				Command = wlan_le16_to_cpu(resp->Command);
	HostCmd_DS_802_11_AD_HOC_RESULT	*pAdHocResult = &resp->params.result;
	u16				Result = wlan_le16_to_cpu(resp->Result);
	union iwreq_data		wrqu;


	ENTER();

	PRINTM(INFO, "ADHOC_S_RESP: Size = %d\n", wlan_le16_to_cpu(resp->Size));
	PRINTM(INFO, "ADHOC_S_RESP: Command = %x\n", Command);
	PRINTM(INFO, "ADHOC_S_RESP: Result = %x\n", Result);

	/*
	 * Join result code 0 --> SUCCESS
	 * if join result code != 0, will return Failure later
	 */
	if (Result) {
		PRINTM(INFO, "ADHOC_RESP Failed\n");
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			MacEventDisconnected(priv);
		}

		if (Command == HostCmd_RET_802_11_AD_HOC_JOIN) {
			Adapter->m_NumAssociationAttemp++;
			if (Adapter->m_NumAssociationAttemp >= 2) {
				/* do not attemp any more */
				ret = WLAN_STATUS_SUCCESS;
				goto done;
			}

			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_AD_HOC_JOIN, 0,
					0, (WLAN_OID) 0,
					&(Adapter->AttemptedSSIDBeforeScan));

			if (ret) {
				goto done;
			}
		}
		/*
		 * If it's a START command and it fails, 
		 * remove the entry on BSSIDList
		 */
		else if (Command == HostCmd_RET_802_11_AD_HOC_START) {
			memset(&(Adapter->BSSIDList[Adapter->
						ulAttemptedBSSIDIndex]), 0,
						sizeof(WLAN_802_11_BSSID));

			Adapter->ulNumOfBSSIDs--;
		}

		Adapter->AdHocFailed = TRUE;

		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}
	/*
	 * Now the join cmd should be successful
	 * If BSSID has changed use SSID to compare instead of BSSID
	 */
	PRINTM(INFO, "ADHOC_J_RESP  %s\n",
			Adapter->BSSIDList[Adapter->
			ulAttemptedBSSIDIndex].Ssid.Ssid);

	if (memcmp(Adapter->CurBssParams.ssid.Ssid,
				Adapter->BSSIDList[Adapter->
				ulAttemptedBSSIDIndex].Ssid.Ssid,
				Adapter->BSSIDList[Adapter->
				ulAttemptedBSSIDIndex].Ssid.SsidLength)) {
		/* Send a Media Connected event, according to the Spec */
		Adapter->MediaConnectStatus = WlanMediaStateConnected;
		Adapter->LinkSpeed = MRVDRV_LINK_SPEED_11mbps;

		/*
		 * NOTE: According to spec, WLAN_STATUS_MEDIA_CONNECT should
		 * be indicated only when the first client joins, not
		 * when the network is started.  Therefore, CONNECT
		 * should not be indicated if the command was AD_HOC_START
		 * Set the attempted BSSID Index to current
		 * Adapter->ulCurrentBSSIDIndex = Adapter->ulAttemptedBSSIDIndex;
		 * Set the new SSID to current SSID
		 */
		memmove(&(Adapter->CurBssParams.ssid), 
				&(Adapter->BSSIDList[Adapter->
					ulCurrentBSSIDIndex].Ssid),
				sizeof(WLAN_802_11_SSID));

		/*
		 * If it's a Start command, 
		 * set the BSSID to the returned value
		 */
		if (Command == HostCmd_RET_802_11_AD_HOC_START) {
			memmove(&(Adapter->BSSIDList[Adapter->
						ulCurrentBSSIDIndex].
						MacAddress),
					pAdHocResult->BSSID, 
					MRVDRV_ETH_ADDR_LEN);

			Adapter->AdHocCreated = TRUE;
		}

		if (Adapter->ulCurrentBSSIDIndex < MRVDRV_MAX_BSSID_LIST) {
			/* Set the new BSSID (AP's MAC address) to current BSSID */
			memmove(Adapter->CurrentBSSID,
					&(Adapter->BSSIDList[Adapter->
						ulCurrentBSSIDIndex].
						MacAddress),
					MRVDRV_ETH_ADDR_LEN);

			/* Make a copy of current BSSID descriptor */
			memmove(&(Adapter->CurrentBSSIDDescriptor),
					&(Adapter->BSSIDList[Adapter->
						ulCurrentBSSIDIndex]),
					sizeof(WLAN_802_11_BSSID));

			/* Copy adhoc'd bssid into Current BSS State struct */
			memcpy(&Adapter->CurBssParams.bssid, &Adapter->
				BSSIDList[Adapter-> ulCurrentBSSIDIndex],
				sizeof(WLAN_802_11_BSSID));

			/* Set the new configuration to the current config */
			memmove(&Adapter->CurrentConfiguration,
					&(Adapter->BSSIDList[Adapter->
						ulCurrentBSSIDIndex].
						Configuration), 
					sizeof(WLAN_802_11_CONFIGURATION));
		}
	}

	Adapter->m_NumAssociationAttemp = 0;

	os_carrier_on(priv);
	os_start_queue(priv);
		
	memset(&wrqu, 0, sizeof(wrqu));
	memcpy(wrqu.ap_addr.sa_data, Adapter->CurrentBSSID, ETH_ALEN);
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;
	wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL); 


	PRINTM(INFO, "ADHOC_RESP: - Joined/Started Ad Hoc\n");
	PRINTM(INFO, "ADHOC_RESP: Channel = %d\n", Adapter->AdhocChannel);
	PRINTM(INFO, "ADHOC_RESP: BSSID  %x:%x:%x:%x:%x:%x\n",
			pAdHocResult->BSSID[0], pAdHocResult->BSSID[1],
			pAdHocResult->BSSID[2], pAdHocResult->BSSID[3],
			pAdHocResult->BSSID[4], pAdHocResult->BSSID[5]);

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles the command response of reset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_reset(wlan_private * priv,
	       			HostCmd_DS_COMMAND * resp)
{
    	ENTER();
    	PRINTM(INFO, "HWAC - Reset command successful\n");

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of authenticate
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_authenticate(wlan_private * priv,
			     		HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of statistics
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_stat(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_GET_STAT	*p11Stat = &resp->params.gstat;
	wlan_adapter			*Adapter = priv->adapter;
	
	/* TODO Convert it to Big endian befor copy */
	memcpy(&Adapter->wlan802_11Stat,
		       p11Stat, 
		       sizeof(HostCmd_DS_802_11_GET_STAT));
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of snmp_mib
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_snmp_mib(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_SNMP_MIB	*smib = &resp->params.smib;
	u16				OID = wlan_le16_to_cpu(smib->OID);

	ENTER();

	PRINTM(INFO, "SNMP_RESP: value of the OID = %x\n", OID);
	PRINTM(INFO, "SNMP_RESP: Buf size  = %x\n", wlan_le16_to_cpu(smib->BufSize));

	if (OID == OID_802_11_RTS_THRESHOLD) {
		PRINTM(INFO, "SNMP_RESP: RTSThsd =%u\n", priv->adapter->RTSThsd);
	} else if (OID == OID_802_11_FRAGMENTATION_THRESHOLD) {
		PRINTM(INFO, "SNMP_RESP: FragThsd =%u\n", priv->adapter->FragThsd);
	} else if (OID == OID_802_11_INFRASTRUCTURE_MODE) {
		PRINTM(INFO, "SNMP_RESP: InfrastructureMode = %x\n",
				priv->adapter->InfrastructureMode);
		PRINTM(INFO, "SNMP_RESP: set to adhoc /infra mode\n");
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of radio_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_radio_control(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	ENTER();

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef WPA
/** 
 *  @brief This function handles the command response of key_material
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_key_material(wlan_private * priv,
					HostCmd_DS_COMMAND * resp) 
{
	HostCmd_DS_802_11_KEY_MATERIAL 	*pKey = &resp->params.keymaterial;
	wlan_adapter			*Adapter = priv->adapter;

	ENTER();

	if ((Adapter->IsGTK_SET == (TRUE + 1)) && (pKey->Action == HostCmd_ACT_SET))
		Adapter->IsGTK_SET = TRUE;

	memcpy(Adapter->aeskey.KeyParamSet.Key, pKey->KeyParamSet.Key,
				sizeof(pKey->KeyParamSet.Key));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* WPA */

/** 
 *  @brief This function handles the command response of mac_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_mac_address(wlan_private * priv,
					HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_MAC_ADDRESS	*MacAdd = &resp->params.macadd;
	wlan_adapter			*Adapter = priv->adapter;
	
	ENTER();
	
	memcpy(Adapter->CurrentAddr, MacAdd->MacAdd, ETH_ALEN);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_tx_power
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_tx_power(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_RF_TX_POWER	*rtp = &resp->params.txp;
	wlan_adapter			*Adapter = priv->adapter;

	ENTER();

	Adapter->TxPowerLevel = wlan_le16_to_cpu(rtp->CurrentLevel); 
 
	PRINTM(INFO, "Current TxPower Level = %d\n", Adapter->TxPowerLevel);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_antenna
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_antenna(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	PHostCmd_DS_802_11_RF_ANTENNA	pAntenna = &resp->params.rant;
	wlan_adapter			*Adapter = priv->adapter;
	u16				Action = wlan_le16_to_cpu(pAntenna->Action);

	if (Action == HostCmd_ACT_GET_RX)
		Adapter->RxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

	if (Action == HostCmd_ACT_GET_TX)
		Adapter->TxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

	PRINTM(INFO, "RF_ANT_RESP: Action = 0x%x, Mode = 0x%04x\n",
			Action, wlan_le16_to_cpu(pAntenna->AntennaMode));

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of multicast_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_mac_multicast_adr(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rate_adapt_rateset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rate_adapt_rateset(wlan_private *priv,
						HostCmd_DS_COMMAND *resp)
{
	HostCmd_DS_802_11_RATE_ADAPT_RATESET	*rates = 
						&resp->params.rateset;
	wlan_adapter				*Adapter = priv->adapter;

	ENTER();

	if (rates->Action == HostCmd_ACT_GET) {
		Adapter->EnableHwAuto = rates->EnableHwAuto;
		Adapter->RateBitmap = rates->Bitmap;
	}

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of data_rate
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_data_rate(wlan_private * priv, 
					HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_DATA_RATE	*pDataRate = &resp->params.drate;
	wlan_adapter			*Adapter = priv->adapter;
	u8				Dot11DataRate;

	ENTER();

	HEXDUMP("DATA_RATE_RESP: data_rate- ",
			(u8 *)pDataRate,sizeof(HostCmd_DS_802_11_DATA_RATE));

	Dot11DataRate = pDataRate->DataRate[0];
	if (pDataRate->Action == HostCmd_ACT_GET_TX_RATE) 
	{
		memcpy(Adapter->SupportedRates, pDataRate->DataRate,
				sizeof(Adapter->SupportedRates));
	}
	Adapter->DataRate = index_to_data_rate(Dot11DataRate);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_channel
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_channel(wlan_private * priv,
				      HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RF_CHANNEL    *rfchannel = &resp->params.rfchannel;
    wlan_adapter                    *Adapter   = priv->adapter;
    u16 Action     = wlan_le16_to_cpu(rfchannel->Action);
    u16 newChannel = wlan_le16_to_cpu(rfchannel->CurrentChannel);

    ENTER();

    if (Action == HostCmd_OPT_802_11_RF_CHANNEL_GET
        && Adapter->Channel != newChannel )
    {
        PRINTM(INFO, "Channel Switch: %d to %d\n", Adapter->Channel, newChannel);

        /* Update the channel */
        Adapter->Channel = newChannel;

        /* Update the channel again */
        Adapter->CurBssParams.channel = newChannel;
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rssi
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rssi(wlan_private * priv, 
				HostCmd_DS_COMMAND * resp)
{
	HostCmd_DS_802_11_RSSI_RSP	*rssirsp = &resp->params.rssirsp;
	wlan_adapter		*Adapter = priv->adapter;

	/* store the non average value */
	Adapter->SNR[TYPE_BEACON][TYPE_NOAVG]	= wlan_le16_to_cpu(rssirsp->SNR);
	Adapter->NF[TYPE_BEACON][TYPE_NOAVG]	= wlan_le16_to_cpu(rssirsp->NoiseFloor);
	    
	Adapter->SNR[TYPE_BEACON][TYPE_AVG] = wlan_le16_to_cpu(rssirsp->AvgSNR);
	Adapter->NF[TYPE_BEACON][TYPE_AVG] = wlan_le16_to_cpu(rssirsp->AvgNoiseFloor);

	Adapter->RSSI[TYPE_BEACON][TYPE_NOAVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
				Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);

	Adapter->RSSI[TYPE_BEACON][TYPE_AVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_AVG] / AVG_SCALE,
			Adapter->NF[TYPE_BEACON][TYPE_AVG] / AVG_SCALE);
    
	PRINTM(INFO, "Beacon RSSI value = 0x%x\n", 
		Adapter->RSSI[TYPE_BEACON][TYPE_AVG]);

	return WLAN_STATUS_SUCCESS;
}

#ifdef	MANF_CMD_SUPPORT
/** 
 *  @brief This function handles the command response of mfg_cmd
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_mfg_cmd(wlan_private * priv,
	       			HostCmd_DS_COMMAND * resp)
{
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->mfg_cmd_flag == 1) {
		// enough buffer for the response
		if (priv->wlan_dev.upld_len <= Adapter->mfg_cmd_len) {
			memcpy(Adapter->mfg_cmd,
					Adapter->CurCmd->BufVirtualAddr,
					priv->wlan_dev.upld_len);
			Adapter->mfg_cmd_resp_len = priv->wlan_dev.upld_len;
		} else {
			memset(Adapter->mfg_cmd, 0, Adapter->mfg_cmd_len);
			Adapter->mfg_cmd_resp_len = -1;
		}
		wake_up_interruptible(&(Adapter->mfg_cmd_q));
		Adapter->mfg_cmd_flag = 0;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* MANF_CMD_SUPPORT */

#ifdef MULTI_BANDS
/** 
 *  @brief This function handles the command response of band_config
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_band_config(wlan_private *priv,
			      HostCmd_DS_COMMAND *resp) 
{
	ENTER();
	
	HEXDUMP("802_11a CmdResp",(s8 *) resp, wlan_le16_to_cpu(resp->Size));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* MULTI_BANDS */

/** 
 *  @brief This function handles the command response of eeprom_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_eeprom_access(wlan_private *priv,
	       				HostCmd_DS_COMMAND *resp)	
{
	wlan_adapter    *Adapter = priv->adapter;

	memcpy(Adapter -> pRdeeprom, (u8 *) &resp -> params.rdeeprom.Value, 
				wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount));
	HEXDUMP("Adapter", Adapter -> pRdeeprom, wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount));

	return WLAN_STATUS_SUCCESS;
}

#ifdef GSPI
/** 
 *  @brief This function handles the command response of gspi_bus_config
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_cmd_gspi_bus_config(wlan_private *priv,
					HostCmd_DS_COMMAND *resp) 
{
	ENTER();
	
	HEXDUMP("GSPI_BUS_RESP: ",(s8 *) resp, wlan_le16_to_cpu(resp->Size));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* GSPI */

#ifdef ATIMGEN
/** 
 *  @brief This function handles the command response of generate_atim
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_generate_atim(wlan_private *priv,
	       				HostCmd_DS_COMMAND *resp)	
{
	HostCmd_DS_802_11_GENERATE_ATIM *pAtim = &resp->params.genatim;
	wlan_adapter *Adapter = priv->adapter;

	ENTER();
	
	if (wlan_le16_to_cpu(pAtim->Enable))
		Adapter->ATIMEnabled = wlan_le16_to_cpu(pAtim->Enable);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* ATIMGEN */

/** 
 *  @brief This function handles the command response of get_log
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_get_log(wlan_private *priv,
	       			HostCmd_DS_COMMAND *resp)
{
	PHostCmd_DS_802_11_GET_LOG  LogMessage = 
			(PHostCmd_DS_802_11_GET_LOG)&resp->params.glog;
	wlan_adapter 	*Adapter = priv->adapter;

	ENTER();

	/* TODO Convert it to Big Endian before copy */
	memcpy(&Adapter->LogMsg, LogMessage, 
					sizeof(HostCmd_DS_802_11_GET_LOG));
#ifdef BIG_ENDIAN
	endian_convert_GET_LOG(Adapter->LogMsg);
#endif

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function handles the command response
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_process_rx_command(wlan_private * priv)
{
	u16			RespCmd;
	HostCmd_DS_COMMAND	*resp;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;
	u32			flags;
	u16			Result;

	ENTER();

	PRINTM(INFO, "CMD_RESP: @ %lu\n", os_time_get());

	/* Now we got response from FW, cancel the command timer */
	if (Adapter->CommandTimerIsSet) {
		CancelTimer(&Adapter->MrvDrvCommandTimer);
		Adapter->CommandTimerIsSet = FALSE;
	}

	if (!Adapter->CurCmd) {
		PRINTM(INFO, "CMD_RESP: NULL CurCmd=%p\n", Adapter->CurCmd);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	resp = (HostCmd_DS_COMMAND *) (Adapter->CurCmd->BufVirtualAddr);

	HEXDUMP("CMD_RESP:", Adapter->CurCmd->BufVirtualAddr,
						priv->wlan_dev.upld_len);

	RespCmd = wlan_le16_to_cpu(resp->Command);

	Result = wlan_le16_to_cpu(resp->Result);

	PRINTM(INFO, "CMD_RESP: %x Result: %d Length: %d\n", RespCmd,
			Result, priv->wlan_dev.upld_len);
	
	if (!(RespCmd & 0x8000)) {
		PRINTM(INFO, "Invalid response to command!");
		Adapter->CurCmd->retcode = WLAN_STATUS_FAILURE;
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
		
		ret = WLAN_STATUS_FAILURE;
		goto done;		
	}

	/* Store the response code in the CmdNode. 
	 * Note: CmdNode->retcode should not be reset in CleanUpCmdCtrlNode()
	 */
	Adapter->CurCmd->retcode = wlan_le16_to_cpu(resp->Result);

#ifdef  PS_REQUIRED
	if (RespCmd == HostCmd_RET_802_11_PS_MODE) {
		HostCmd_DS_802_11_PS_MODE *psmode;

		psmode = &resp->params.psmode;
		PRINTM(INFO, "CMD_RESP: PS_MODE cmd reply result=%#x action=0x%X\n",
						resp->Result, psmode->Action);
		psmode->Action = wlan_cpu_to_le16(psmode->Action);

		if (Result) {
			PRINTM(INFO, "CMD_RESP: PS command failed- %#x \n", resp->Result);
			if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
				/* 
				 * We should not re-try enter-ps command in 
				 * ad-hoc mode. It takes place in 
				 * ExecuteNextCommand().
				 */
				if (psmode->Action == HostCmd_SubCmd_Enter_PS)
					Adapter->PSMode = 
						Wlan802_11PowerModeCAM;
			}
		} else if (psmode->Action == HostCmd_SubCmd_Enter_PS) {
			Adapter->NeedToWakeup = FALSE;
			Adapter->PSState = PS_STATE_AWAKE;
			PRINTM(INFO, "CMD_RESP: Enter_PS command response\n");
			if (Adapter->MediaConnectStatus != WlanMediaStateConnected){
				/*
				 * When Deauth Event received before Enter_PS command
				 * response, We need to wake up the firmware.
				 */
				PRINTM(INFO, "Disconnected, Going to invoke PSWakeup\n");
				PSWakeup(priv, 0);
			}
		} else if (psmode->Action == HostCmd_SubCmd_Exit_PS) {
			Adapter->NeedToWakeup = FALSE;
			Adapter->PSState = PS_STATE_FULL_POWER;
			PRINTM(INFO, "CMD_RESP: Exit_PS command response\n");
		} else {
			PRINTM(INFO, "CMD_RESP: PS- Action=0x%X\n",psmode->Action);
		}

		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}
#endif

#ifdef STDCMD
	if (Adapter->CurCmd->CmdFlags & CMD_F_STDCMD) {
		/* Copy the response back to response buffer */
		memcpy(Adapter->CurCmd->pdata_buf,
				resp, resp->Size);

		Adapter->CurCmd->CmdFlags &= ~CMD_F_STDCMD;
	}
#endif
    
	/* If the command is not successful, cleanup and return failure */
	if ((Result != HostCmd_RESULT_OK || !(RespCmd & 0x8000))) {
		PRINTM(INFO, "CMD_RESP: command reply %#x result=%#x\n",
				resp->Command, resp->Result);
		/*
		 * Handling errors here
		 * In some cases we need to report hardware status
		 */
		switch (RespCmd) {
		case HostCmd_RET_MAC_REG_ACCESS:
		case HostCmd_RET_BBP_REG_ACCESS:
		case HostCmd_RET_RF_REG_ACCESS:
			break;
		
		case HostCmd_RET_HW_SPEC_INFO:
		case HostCmd_RET_802_11_RESET:
			PRINTM(INFO, "CMD_RESP: Reset command Failed\n");
			Adapter->HardwareStatus = WlanHardwareStatusNotReady;
			break;
		
		case HostCmd_RET_802_11_SCAN:
		case HostCmd_RET_802_11_ASSOCIATE:
			if (RespCmd == HostCmd_RET_802_11_SCAN) {
				Adapter->bIsScanInProgress = FALSE;
				wake_up_interruptible(&Adapter->scan_q);
#ifdef PROGRESSIVE_SCAN
				Adapter->ScanChannelsLeft = 0;
#endif
			}

		/* FALL THROUGH */
		case HostCmd_RET_802_11_REASSOCIATE:
		case HostCmd_RET_802_11_AD_HOC_JOIN:
		case HostCmd_RET_802_11_AD_HOC_START:
			break;
#ifdef BCA
		case HostCmd_RET_802_11_BCA_CONFIG_TIMESHARE:
			break;
#endif

		case HostCmd_RET_802_11_INACTIVITY_TIMEOUT:
			break;

		case HostCmd_RET_802_11_SLEEP_PERIOD: 
			break;
		}
		
                CleanupAndInsertCmd(priv, Adapter->CurCmd);
                spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

		return WLAN_STATUS_FAILURE;
	}

	switch (RespCmd) {
	case HostCmd_RET_MAC_REG_ACCESS:
	case HostCmd_RET_BBP_REG_ACCESS:
	case HostCmd_RET_RF_REG_ACCESS:
		ret = wlan_ret_reg_access(priv, RespCmd, resp);
		break;
	    
	case HostCmd_RET_HW_SPEC_INFO:
		ret = wlan_ret_get_hw_spec(priv, resp);
		break;
	    
#ifdef PS_REQUIRED
	case HostCmd_RET_802_11_PS_MODE:
		ret = wlan_ret_802_11_ps_mode(priv, resp);
		break;
#endif

#ifdef BG_SCAN
	case HostCmd_RET_802_11_BG_SCAN_QUERY:
	{
		union iwreq_data	wrqu;

		ret = wlan_ret_802_11_scan(priv, resp);
		memset(&wrqu, 0, sizeof(union iwreq_data));
#ifdef linux
		wireless_send_event(priv->wlan_dev.netdev, SIOCGIWSCAN, &wrqu , NULL);
#endif
		PRINTM(MSG, "CMD_RESP: BG_SCAN result is ready!\n");
		break;
	}
#endif /* BG_SCAN */
	case HostCmd_RET_802_11_SCAN:
		ret = wlan_ret_802_11_scan(priv, resp);

#ifdef PROGRESSIVE_SCAN
		if (Adapter->ScanChannelsLeft > 0) {
			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_SCAN, 0,
					0, 0, NULL);

			if (ret) {
				LEAVE();
				return ret;
			}
		} else
#endif
		{
			Adapter->bIsScanInProgress = FALSE;
			wake_up_interruptible(&Adapter->scan_q); 
		}

		break;

	case HostCmd_RET_MAC_CONTROL:
		ret = wlan_ret_mac_control(priv, resp);
		break;
			
	case HostCmd_RET_802_11_GET_LOG:
		ret = wlan_ret_get_log(priv, resp);		
		break;

	case HostCmd_RET_802_11_ASSOCIATE:
	case HostCmd_RET_802_11_REASSOCIATE:
		ret = wlan_ret_802_11_associate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_DISASSOCIATE:
	case HostCmd_RET_802_11_DEAUTHENTICATE:
		ret = wlan_ret_802_11_disassociate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_SET_WEP:
		ret = wlan_ret_802_11_set_wep(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_AD_HOC_START:
	case HostCmd_RET_802_11_AD_HOC_JOIN:
		ret = wlan_ret_802_11_ad_hoc_start(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RESET:
		ret = wlan_ret_802_11_reset(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_AUTHENTICATE:
		ret = wlan_ret_802_11_authenticate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_STAT:
		ret = wlan_ret_802_11_stat(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_SNMP_MIB:
		ret = wlan_ret_802_11_snmp_mib(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RF_TX_POWER:
		ret = wlan_ret_802_11_rf_tx_power(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RADIO_CONTROL:
		ret = wlan_ret_802_11_radio_control(priv, resp);
		break;

#ifdef DEEP_SLEEP
	case HostCmd_RET_802_11_DEEP_SLEEP:
		Adapter->IsDeepSleep = TRUE;
		break;
#endif // DEEP_SLEEP

#ifdef HOST_SLEEP
	case HostCmd_RET_802_11_HOST_SLEEP_CFG:
		if (Adapter->HostSleepCfgParam.conditions != HOST_SLEEP_CFG_CANCEL) {
			Adapter->bHostSleepConfigured = TRUE;
			PRINTM(MSG, "CMD_RESP: Host Sleep Cfg cmd resp 0x%04X received\n", RespCmd);
		}
		else {
			Adapter->bHostSleepConfigured = FALSE;
			memset(&Adapter->HostSleepCfgParam, 0x00, sizeof(Adapter->HostSleepCfgParam));
		}
		break;
	case HostCmd_RET_802_11_HOST_SLEEP_AWAKE_CONFIRM:
		PRINTM(INFO, "CMD_RESP: Host Sleep Awake Confirm cmd resp 0x%04X received\n", RespCmd);
#ifdef FW_WAKEUP_TIME
		wt_awake_confirmrsp = get_utimeofday();
	{
		u32 tv1, tv2;

		tv1 = wt_pwrup_sending;
		tv2 = wt_pwrup_sent;
		PRINTM(MSG, "wt_pwrup_sending: %ld.%03ld.%03ld ", tv1 / 1000000, (tv1 % 1000000) / 1000, tv1 % 1000);
		PRINTM(MSG, " -> wt_pwrup_sent: %ld.%03ld.%03ld ", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2%1000);
		tv2 -= tv1;
		PRINTM(MSG, " == %ld.%03ld.%03ld\n", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2 % 1000);

		tv1 = wt_pwrup_sent;
		tv2 = wt_int;
		PRINTM(MSG, "wt_pwrup_sent: %ld.%03ld.%03ld ", tv1 / 1000000, (tv1 % 1000000) / 1000, tv1 % 1000);
		PRINTM(MSG, " -> wt_int: %ld.%03ld.%03ld ", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2%1000);
		tv2 -= tv1;
		PRINTM(MSG, " == %ld.%03ld.%03ld\n", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2 % 1000);

		tv1 = wt_pwrup_sent;
		tv2 = wt_wakeup_event;
		PRINTM(MSG, "wt_pwrup_sent: %ld.%03ld.%03ld ", tv1 / 1000000, (tv1 % 1000000) / 1000, tv1 % 1000);
		PRINTM(MSG, " -> wt_wakeup_event: %ld.%03ld.%03ld ", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2%1000);
		tv2 -= tv1;
		PRINTM(MSG, " == %ld.%03ld.%03ld\n", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2 % 1000);

		tv1 = wt_pwrup_sent;
		tv2 = wt_awake_confirmrsp;
		PRINTM(MSG, "wt_pwrup_sent: %ld.%03ld.%03ld ", tv1 / 1000000, (tv1 % 1000000) / 1000, tv1 % 1000);
		PRINTM(MSG, " -> wt_awake_confirmrsp: %ld.%03ld.%03ld ", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2%1000);
		tv2 -= tv1;
		PRINTM(MSG, " == %ld.%03ld.%03ld\n", tv2 / 1000000, (tv2 % 1000000) / 1000, tv2 % 1000);

		wt_pwrup_sending=wt_pwrup_sent=wt_int=wt_wakeup_event=wt_awake_confirmrsp=0L;
	}
#endif
		break;
#endif

#ifdef AUTO_FREQ_CTRL
	case HostCmd_RET_802_11_SET_AFC:
	case HostCmd_RET_802_11_GET_AFC:
		memmove(Adapter->CurCmd->pdata_buf, 
			&resp->params.afc,
			sizeof(HostCmd_DS_802_11_AFC));

		break;
#endif
	case HostCmd_RET_802_11_RF_ANTENNA:
		ret = wlan_ret_802_11_rf_antenna(priv, resp);
		break;
	    
	case HostCmd_RET_MAC_MULTICAST_ADR:
		ret = wlan_ret_mac_multicast_adr(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_DATA_RATE:
		ret = wlan_ret_802_11_data_rate(priv, resp);
		break;
	case HostCmd_RET_802_11_RATE_ADAPT_RATESET:
	    	ret = wlan_ret_802_11_rate_adapt_rateset(priv, resp);
		break;
	case HostCmd_RET_802_11_RF_CHANNEL:
		ret = wlan_ret_802_11_rf_channel(priv, resp);
		break;
	    		
	case HostCmd_RET_802_11_RSSI:
		ret = wlan_ret_802_11_rssi(priv, resp);
		break;

	case HostCmd_RET_802_11_MAC_ADDRESS:
		ret = wlan_ret_802_11_mac_address(priv, resp);
		break;

#ifdef MANF_CMD_SUPPORT
	case HostCmd_RET_MFG_COMMAND:
		ret = wlan_ret_mfg_cmd(priv, resp);
		break;
#endif
	case HostCmd_RET_802_11_AD_HOC_STOP:
		ret = wlan_ret_802_11_ad_hoc_stop(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_BEACON_STOP:
		break;
			
#ifdef WPA
	case HostCmd_RET_802_11_ENABLE_RSN:
		break;
	case HostCmd_RET_802_11_KEY_MATERIAL:
		PRINTM(INFO, "CMD_RESP: KEY_MATERIAL command response\n"); 
		ret = wlan_ret_802_11_key_material(priv, resp); 
		break;
#endif //WPA
			
#ifdef MULTI_BANDS
	case HostCmd_RET_802_11_BAND_CONFIG:
		ret = wlan_ret_802_11_band_config(priv, resp); 
		break;
#endif
	case HostCmd_RET_802_11_EEPROM_ACCESS:
		ret = wlan_ret_802_11_eeprom_access(priv, resp);
		break;

#ifdef GSPI
	case HostCmd_RET_CMD_GSPI_BUS_CONFIG:
		ret = wlan_ret_cmd_gspi_bus_config(priv, resp);
		break;
#endif /* GSPI */
				
#ifdef ATIMGEN			
	case HostCmd_RET_802_11_GENERATE_ATIM:
		ret = wlan_ret_802_11_generate_atim(priv, resp);
		break;
#endif

#ifdef ENABLE_802_11D
	case HostCmd_RET_802_11D_DOMAIN_INFO:
		ret = wlan_ret_802_11d_domain_info(priv, resp);
		break;
#endif		

#ifdef ENABLE_802_11H
	case HostCmd_RET_802_11H_TPC_REQUEST:
	case HostCmd_RET_802_11H_TPC_INFO:
	case HostCmd_RET_802_11H_CHAN_SW_ANN:
		ret = wlan_11h_cmdresp_process(priv, resp);
        break;
#endif
#ifdef ENABLE_MEAS
	case HostCmd_RET_MEASUREMENT_REQUEST:
	case HostCmd_RET_MEASUREMENT_REPORT:
        ret = wlan_meas_cmdresp_process(priv, resp);
        break;
#endif                
	case HostCmd_RET_802_11_SLEEP_PARAMS:		
		ret = wlan_ret_802_11_sleep_params(priv, resp);
		break;
#ifdef BCA
	case HostCmd_RET_802_11_BCA_CONFIG_TIMESHARE:
		ret = wlan_ret_802_11_bca_timeshare(priv, resp);
		break;
#endif
	case HostCmd_RET_802_11_INACTIVITY_TIMEOUT:
		*((u16 *)Adapter->CurCmd->pdata_buf) =
			wlan_le16_to_cpu(resp->params.inactivity_timeout.Timeout);
		break;
#ifdef BG_SCAN
	case HostCmd_RET_802_11_BG_SCAN_CONFIG:
		PRINTM(INFO, "CMD_RESP: BG_CONFIG CMD response\n");
		break;
#endif /* BG_SCAN */

#ifdef PS_REQUIRED
#ifdef FW_WAKEUP_METHOD
	case HostCmd_RET_802_11_FW_WAKEUP_METHOD:
		PRINTM(INFO, "CMD_RESP: FW_WAKEUP_METHOD CMD response\n");
		ret = wlan_ret_fw_wakeup_method(priv, resp);
		break;
#endif
#endif
		
#ifdef SUBSCRIBE_EVENT_CTRL
	case HostCmd_RET_802_11_SUBSCRIBE_EVENT:
		PRINTM(INFO, "CMD_RESP: SUBSCRIBE_EVENTS response\n");
		wlan_cmd_ret_802_11_subscribe_events(priv, resp);
		break;
#endif

	case HostCmd_RET_802_11_SLEEP_PERIOD:
		ret = wlan_ret_802_11_sleep_period(priv, resp);
		break;
#ifdef WMM
	case HostCmd_RET_802_11_WMM_GET_TSPEC:
	case HostCmd_RET_802_11_WMM_ADD_TSPEC:
	case HostCmd_RET_802_11_WMM_REMOVE_TSPEC:
		HEXDUMP("CMD_RESP: TSPEC", (u8 *)&resp->params.tspec, sizeof(resp->params.tspec));
		memcpy(&Adapter->tspec, &resp->params.tspec, 
			sizeof(HostCmd_DS_802_11_WMM_TSPEC));
		break;
	case HostCmd_RET_802_11_WMM_ACK_POLICY:
		HEXDUMP("CMD_RESP: ACK_POLICY", (u8 *)&resp->params.ackpolicy, sizeof(resp->params.ackpolicy));
		memcpy(&Adapter->ackpolicy, &resp->params.ackpolicy,
			sizeof(HostCmd_DS_802_11_WMM_ACK_POLICY));
		break;
	case HostCmd_RET_802_11_WMM_GET_STATUS:
	{
		u32 i;

		PRINTM(INFO, "CMD_RESP: WMM_GET_STATUS old acstatus %#x\n", Adapter->wmm.acstatus);
		HEXDUMP("CMD_RESP: wmm get status", (u8 *)&resp->params.getstatus,
			       		sizeof(resp->params.getstatus));
		for (Adapter->wmm.acstatus = 0, i = 0; i < MAX_AC_QUEUES; i++) {
			Adapter->wmm.acstatus |= (resp->params.getstatus.
				Status[i].Disabled ? 1 : 0) << (MAX_AC_QUEUES - 1 - i);
		}
		PRINTM(INFO, "CMD_RESP: new acstatus %#x\n", Adapter->wmm.acstatus);
		memcpy(&Adapter->getstatus, &resp->params.getstatus,
			sizeof(HostCmd_DS_802_11_WMM_GET_STATUS));

		wmm_setup_ac_downgrade(priv);

		break;
	}
#endif /* WMM */

        case HostCmd_RET_802_11_TPC_CFG:

          	memmove(Adapter->CurCmd->pdata_buf, 
		            &resp->params.tpccfg,
		            sizeof(HostCmd_DS_802_11_TPC_CFG));

          	break;
#ifdef LED_GPIO_CTRL
        case HostCmd_RET_802_11_LED_GPIO_CTRL:

         	memmove(Adapter->CurCmd->pdata_buf,
		        &resp->params.ledgpio,
		        sizeof(HostCmd_DS_802_11_LED_CTRL));

        	break;	
#endif /* LED_GPIO_CTRL */
        case HostCmd_RET_802_11_PWR_CFG:
          	memmove(Adapter->CurCmd->pdata_buf,
			&resp->params.pwrcfg,
			sizeof(HostCmd_DS_802_11_PWR_CFG));

         	break;
#ifdef CIPHER_TEST
        case HostCmd_RET_802_11_KEY_ENCRYPT:

          	memmove(Adapter->CurCmd->pdata_buf,
		       	&resp->params.key_encrypt,
		        sizeof(HostCmd_DS_802_11_KEY_ENCRYPT));

	        break;
#endif
	default:
		PRINTM(INFO, "CMD_RESP: Unknown command response %#x\n",
				resp->Command);
		break;
	}

        if (Adapter->CurCmd) {
		/* Clean up and Put current command back to CmdFreeQ */
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
	}

done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles events generated by firmware
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_process_event(wlan_private * priv)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "EVENT Cause %x\n", Adapter->EventCause);

	switch (Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT) {
		case MACREG_INT_CODE_LINK_SENSED:
			PRINTM(INFO, "EVENT: MACREG_INT_CODE_LINK_SENSED\n");
			break;

		case MACREG_INT_CODE_DEAUTHENTICATED:
			PRINTM(INFO, "EVENT: Deauthenticated\n");
			HandleDisconnectEvent(priv);
			break;

		case MACREG_INT_CODE_DISASSOCIATED:
			PRINTM(INFO, "EVENT: Disassociated\n");
			HandleDisconnectEvent(priv);
			break;

		case MACREG_INT_CODE_LINK_LOSE_NO_SCAN:
			PRINTM(INFO, "EVENT: Link lost\n");
			HandleDisconnectEvent(priv);
			break;

#ifdef PS_REQUIRED
		case MACREG_INT_CODE_PS_SLEEP:
			PRINTM(INFO, "EVENT: SLEEP\n");
			PRINTM(INFO, "_");

			/* handle unexpected PS SLEEP event */
			if (Adapter->PSState == PS_STATE_FULL_POWER) {
				PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS SLEEP\n");
				break;
			}

			os_stop_queue(priv);
			os_carrier_off(priv);

			Adapter->PSState = PS_STATE_PRE_SLEEP;

			PSConfirmSleep(priv, (u16) Adapter->PSMode);

			break;

		case MACREG_INT_CODE_PS_AWAKE:
			PRINTM(INFO, "EVENT: AWAKE \n");
			PRINTM(INFO, "|");

			/* handle unexpected PS AWAKE event */
			if (Adapter->PSState == PS_STATE_FULL_POWER) {
				PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS AWAKE\n");
				break;
			}

			os_carrier_on(priv);
			os_start_queue(priv);

#ifdef WMM_UAPSD
			if ((Adapter->CurBssParams.wmm_uapsd_enabled == TRUE) && (Adapter->wmm.qosinfo != 0))
			{
				if (wmm_lists_empty(priv) && (Adapter->sleep_period.period != 0))
				{
					if ( Adapter->wmm.gen_null_pkg ) {
				 		SendNullPacket(priv, 
							MRVDRV_TxPD_POWER_MGMT_NULL_PACKET | 
							MRVDRV_TxPD_POWER_MGMT_LAST_PACKET);
					}
					Adapter->wmm.no_more_packet = 1;
				}
				else
				{
					Adapter->wmm.no_more_packet = 0;
				}
			}
#endif
			Adapter->PSState = PS_STATE_AWAKE;

			if (Adapter->NeedToWakeup == TRUE) {
				/*
				 * wait for the command processing to finish
				 * before resuming sending 
				 * Adapter->NeedToWakeup will be set to FALSE 
				 * in PSWakup()
				 */
				PRINTM(INFO, "Waking up...\n");
				PSWakeup(priv, 0);
			}
			break;
#endif	/* PS_REQURIED */

#ifdef DEEP_SLEEP
		case MACREG_INT_CODE_DEEP_SLEEP_AWAKE:
		        sbi_reset_deepsleep_wakeup(priv);
			PRINTM(INFO, "EVENT: DEEP SLEEP AWAKE Event!\n");

			Adapter->IsDeepSleep = FALSE;
			os_carrier_on(priv);
			os_start_queue(priv);

			wake_up_interruptible(&Adapter->ds_awake_q);
			break;
#endif /* DEEP_SLEEP */

#ifdef HOST_SLEEP
		case MACREG_INT_CODE_HOST_SLEEP_AWAKE:
			Adapter->bWakeupDevRequired = FALSE;
			Adapter->WakeupTries = 0;
#ifdef FW_WAKEUP_TIME
			wt_wakeup_event = get_utimeofday();
#endif /* FW_WAKEUP_TIME */
#ifdef DEEP_SLEEP
		        sbi_reset_deepsleep_wakeup(priv);
			PRINTM(INFO, "EVENT: HOST SLEEP AWAKE Event!\n");

			/*
			 * in BG SCAN mode w/ deep sleep, WAKE UP event
			 * will be sent directly, no Deep Sleep Awake will
			 * be sent. so we need to wakeup ds_awake_q here
			 */
			wake_up_interruptible(&Adapter->ds_awake_q); 
#endif /* DEEP_SLEEP */
			ret = PrepareAndSendCommand(priv,
					HostCmd_CMD_802_11_HOST_SLEEP_AWAKE_CONFIRM,
					0, 0, 0, NULL);
			break;
#endif /* HOST_SLEEP */

#ifdef WPA
		case MACREG_INT_CODE_MIC_ERR_UNICAST:
			PRINTM(INFO, "EVENT: UNICAST MIC ERROR\n");
			HandleMICFailureEvent(priv, 
					MACREG_INT_CODE_MIC_ERR_UNICAST);
			break;
		
		case MACREG_INT_CODE_MIC_ERR_MULTICAST:
			PRINTM(INFO, "EVENT: MULTICAST MIC ERROR\n");
			HandleMICFailureEvent(priv, 
					MACREG_INT_CODE_MIC_ERR_MULTICAST);
			break;
#endif
		case MACREG_INT_CODE_MIB_CHANGED:
		case MACREG_INT_CODE_INIT_DONE:
			break;

		case MACREG_INT_CODE_ADHOC_BCN_LOST:
			PRINTM(INFO, "EVENT: HWAC - ADHOC BCN LOST\n");		
			break;
#ifdef BG_SCAN
		case MACREG_INT_CODE_BG_SCAN_REPORT:
			PRINTM(INFO, "EVENT: Background SCAN Report\n");		
			ret = sendBgScanQueryCmd(priv);
			break;
#endif /* BG_SCAN */ 
#ifdef ENABLE_802_11H
    case MACREG_INT_CODE_STOP_TX:
        PRINTM(INFO, "EVENT: Stop Tx (%#x)\n",
                Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
        wlan_11h_tx_disable(priv);
        break;
    case MACREG_INT_CODE_START_TX:
        PRINTM(INFO, "EVENT: Start Tx (%#x)\n",
                Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
        wlan_11h_tx_enable(priv);
        break;
    case MACREG_INT_CODE_CHANNEL_SWITCH:
        /* Don't request the active chanel if we are performing a scan */
        if (Adapter->bIsScanInProgress == FALSE)
        {
            PRINTM(INFO, "EVENT: Channel Switch (%#x)\n",
                    Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
            
	    PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                                  HostCmd_OPT_802_11_RF_CHANNEL_GET,
                                  0, 0, NULL);
        }
        break;
#endif
#ifdef ENABLE_MEAS
    case MACREG_INT_CODE_MEAS_REPORT_RDY:
        PRINTM(INFO, "EVENT: Measurement Report Ready (%#x)\n",
                Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
        PrepareAndSendCommand(priv, HostCmd_CMD_MEASUREMENT_REPORT,
			HostCmd_ACT_GEN_SET, 0, 0, NULL);
        break;
#endif
#ifdef WMM
		case MACREG_INT_CODE_WMM_STATUS_CHANGE:
			PRINTM(MSG, "EVENT: WMM status changed\n");

			os_carrier_on(priv);
			os_start_queue(priv);

			ret = sendWMMStatusChangeCmd(priv);
			break;
#endif /* WMM */

#ifdef SUBSCRIBE_EVENT_CTRL
		case MACREG_INT_CODE_RSSI_LOW:
			PRINTM(MSG, "EVENT: RSSI_LOW\n");		
			break;
		case MACREG_INT_CODE_SNR_LOW:
			PRINTM(MSG, "EVENT: SNR_LOW\n");		
			break;
		case MACREG_INT_CODE_MAX_FAIL:
			PRINTM(MSG, "EVENT: MAX_FAIL\n");		
			break;
#endif

		default:
			PRINTM(INFO, "EVENT: unknown event id: %#x\n",
				Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
			break;
	}

	Adapter->EventCause = 0;
	LEAVE();
	return ret;
}

