/** @file wlan_cmd.c
  *  
  * @brief This file contains the handling of command.
  * it prepares command and sends it to firmware when
  * it is ready.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  * 
  */
/********************************************************
Change log:
	10/04/05: Add Doxygen format comments
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

#ifdef WMM
static u8 wmm_ie[WMM_IE_LENGTH] = { 0xdd, 0x07, 0x00,
       		0x50, 0xf2, 0x02, 0x00, 0x01, 0x00 };
#endif /* WMM */

#ifdef PS_REQUIRED
static u16 Commands_Allowed_In_PS[] = {
	HostCmd_CMD_802_11_RSSI,
#ifdef HOST_SLEEP
	HostCmd_CMD_802_11_HOST_SLEEP_AWAKE_CONFIRM,
#endif /* HOST_SLEEP */
};
#endif /* PS_REQUIRED */

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function finds out the common rates between
 *  rate1 and rate2. It will fill common rates in rate1 as
 *  output if found.
 *  
 *  @param Adapter 	A pointer to wlan_adapter structure
 *  @param rate1	the buffer which keeps input and output
 *  @param rate1_size,  the size of rate1 buffer
 *  @param rate2	the buffer which keeps rate2
 *  @param rate2_size   the size of rate2 buffer.
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
/*
 * NOTE: Setting the MSB of the basic rates need to be taken
 * 	 care, either before or after calling this function
 */

static int get_common_rates(wlan_adapter *Adapter, u8 *rate1,
	       		int rate1_size, u8 *rate2, int rate2_size)
{
	int	i;
	u8	tmp[30], *ptr = rate1;
	int 	ret = WLAN_STATUS_SUCCESS;

	memset(&tmp, 0, sizeof(tmp));
	memcpy(&tmp, rate1, MIN(rate1_size,sizeof(tmp)));
	memset(rate1, 0, rate1_size);

	/* Mask the top bit of the original values */
	for (i = 0; tmp[i] && i < sizeof(tmp); i++)
		tmp[i] &= 0x7F;

	for (i = 0; rate2[i] && i < rate2_size; i++) {
		/* Check for Card Rate in tmp, excluding the top bit */
		if (strchr(tmp, rate2[i] & 0x7F)) {
			/* Values match, so copy the Card Rate to rate1 */
			*rate1++ = rate2[i];
		}
	}

	HEXDUMP("rate1 (AP) Rates:", tmp, sizeof(tmp));
	HEXDUMP("rate2 (Card) Rates:", rate2, rate2_size);
	HEXDUMP("Common Rates:", ptr, rate1_size);
	PRINTM(INFO, "Tx DataRate is set to 0x%X\n", Adapter->DataRate);

	if (!Adapter->Is_DataRate_Auto) {
		while (*ptr) {
			if ((*ptr & 0x7f) == Adapter->DataRate) {
				ret = WLAN_STATUS_SUCCESS;
				goto done;
			}
			ptr++;
		}
		PRINTM(MSG, "Previously set fixed data rate %#x isn't "
			"compatible with the network.\n",Adapter->DataRate);

		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	ret = WLAN_STATUS_SUCCESS;
done:
	return ret;
}

/** 
 *  @brief This function checks if the commans is allowed
 *  in PS mode not.
 *  
 *  @param Command the command ID
 *  @return 	   TRUE or FALSE
 */
static BOOLEAN Is_Command_Allowed_In_PS(u16 Command)
{
	int count = sizeof(Commands_Allowed_In_PS) 
				/ sizeof(Commands_Allowed_In_PS[0]);
	int i;

	for (i = 0; i < count; i++) {
		if (Command == wlan_cpu_to_le16(Commands_Allowed_In_PS[i])) 
			return TRUE;
	}

	return FALSE;
}

#ifdef IWGENIE_SUPPORT
/** 
 *  @brief This function appends generac IE. It is called from the network
 *  join command prep. routine. If a generic IE buffer has been setup by
 *  the application/supplication, the routine appends the buffer as a
 *  passthrough TLV type to the request.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @param ppBuffer pointer to command buffer pointer
 *  @return 	    bytes added to the buffer
 */
static int wlan_cmd_append_generic_ie( wlan_private* priv, u8** ppBuffer )
{
    wlan_adapter *Adapter = priv->adapter;
    MrvlIEtypesHeader_t ieHeader;
    int retLen = 0;

    /* Null Checks */
    if (ppBuffer == 0) return 0;
    if (*ppBuffer == 0) return 0;

    /*
     * If there is a generic ie buffer setup, append it to the return
     * parameter buffer pointer. 
     */
    if (Adapter->genIeBufferLen)
    {
        PRINTM(INFO, "append generic %d to %p\n", Adapter->genIeBufferLen,
                *ppBuffer);

        /* Wrap the generic IE buffer with a passthrough TLV type */
        ieHeader.Type = TLV_TYPE_PASSTHROUGH;
        ieHeader.Len  = Adapter->genIeBufferLen;
        memcpy(*ppBuffer, &ieHeader, sizeof(ieHeader));

        /* Increment the return size and the return buffer pointer param */
        *ppBuffer += sizeof(ieHeader);
        retLen    += sizeof(ieHeader);

        /* Copy the generic IE buffer to the output buffer, advance pointer */
        memcpy(*ppBuffer, Adapter->genIeBuffer, Adapter->genIeBufferLen);

        /* Increment the return size and the return buffer pointer param */
        *ppBuffer += Adapter->genIeBufferLen;
        retLen    += Adapter->genIeBufferLen;

        /* Reset the generic IE buffer */
        Adapter->genIeBufferLen = 0;
    }

    /* return the length appended to the buffer */
    return retLen;
}
#endif

#ifdef REASSOC_CMD_SUPPORT
/** 
 *  @brief This function appends reassoc IE. It is called from the network
 *  join command prep. routine.  If a reassociation attempt is in progress
 *  (determined from flag in the wlan_priv structure), a REASSOCAP TLV is
 *  added to the association request.  This causes the firmware to send a
 *  reassociation request instead of an association request.  The wlan_priv
 *  structure also contains the current AP BSSID to be passed in the TLV and
 *  eventually in the managment frame to the new AP.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @param ppBuffer pointer to command buffer pointer
 *  @return 	    bytes added to the buffer
 */
static int wlan_cmd_append_reassoc_tlv( wlan_private* priv, u8** ppBuffer )
{
    wlan_adapter *Adapter = priv->adapter;
    int retLen = 0;
    MrvlIEtypes_ReassocAp_t reassocIe;

    /* Null Checks */
    if (ppBuffer == 0) return 0;
    if (*ppBuffer == 0) return 0;

    /*
     * If the reassocAttempt flag is set in the adapter structure, include
     *   the appropriate TLV in the association buffer pointed to by ppBuffer
     */
    if (Adapter->reassocAttempt)
    {
        PRINTM(INFO, "append current AP: %#x:%#x:%#x:%#x:%#x:%#x\n",
                Adapter->reassocCurrentAp[0], Adapter->reassocCurrentAp[1],
                Adapter->reassocCurrentAp[2], Adapter->reassocCurrentAp[3],
                Adapter->reassocCurrentAp[4], Adapter->reassocCurrentAp[5]);

        /* Setup a copy of the reassocIe on the stack */
        reassocIe.Header.Type = TLV_TYPE_REASSOCAP;
        reassocIe.Header.Len  = (sizeof(MrvlIEtypes_ReassocAp_t)
                                 - sizeof(MrvlIEtypesHeader_t));
        memcpy(&reassocIe.currentAp, 
               &Adapter->reassocCurrentAp,
               sizeof(reassocIe.currentAp));

        /* Copy the stack reassocIe to the buffer pointer parameter */
        memcpy(*ppBuffer, &reassocIe, sizeof(reassocIe));

        /* Set the return length */
        retLen = sizeof(reassocIe);

        /* Advance passed buffer pointer */
        *ppBuffer += sizeof(reassocIe);
        
        /* Reset the reassocAttempt flag, only valid for a single attempt */
        Adapter->reassocAttempt = FALSE;

        /* Reset the reassociation AP address */
        memset(&Adapter->reassocCurrentAp,
               0x00, 
               sizeof(Adapter->reassocCurrentAp));
    }

    /* return the length appended to the buffer */
    return retLen;
}
#endif

/** 
 *  @brief This function prepares command of get_hw_spec.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param cmd	   A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_hw_spec(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd)
{
	HostCmd_DS_GET_HW_SPEC *hwspec = &cmd->params.hwspec;

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_GET_HW_SPEC);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_GET_HW_SPEC) + S_DS_GEN);
	memcpy(hwspec->PermanentAddr, priv->adapter->CurrentAddr, ETH_ALEN);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef PS_REQUIRED
/** 
 *  @brief This function prepares command of ps_mode.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_ps_mode(wlan_private * priv,
					HostCmd_DS_COMMAND * cmd,
					u16 cmd_action) 
{
	HostCmd_DS_802_11_PS_MODE	*psm = &cmd->params.psmode;
	u16				Action = cmd_action;
	wlan_adapter			*Adapter = priv->adapter;

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_PS_MODE);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_PS_MODE) + S_DS_GEN);
	psm->Action = wlan_cpu_to_le16(cmd_action);
	switch (Action) {
	case HostCmd_SubCmd_Enter_PS:
		PRINTM(INFO, "PS Command:" "SubCode- Enter PS\n");
		PRINTM(INFO, "LocalListenInterval = %d\n", 
				Adapter->LocalListenInterval);
	
		psm->LocalListenInterval = 
			wlan_cpu_to_le16(Adapter->LocalListenInterval);
		break;

	case HostCmd_SubCmd_Exit_PS:
		PRINTM(INFO, "PS Command:" "SubCode- Exit PS\n");
		break;

	case HostCmd_SubCmd_Sleep_Confirmed:
		PRINTM(INFO, "PS Command: SubCode- sleep confirm\n");
		break;

	default:
		break;
	}

	if(Action == HostCmd_SubCmd_Enter_PS)	
		psm->MultipleDtim = wlan_cpu_to_le16(priv->adapter->MultipleDtim);
	else
		psm->MultipleDtim = 0;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef FW_WAKEUP_METHOD
/** 
 *  @brief This function prepares command of fw_wakeup_method.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @param pdata_buf 	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_fw_wakeup_method(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	int cmd_action,
			       	void *pdata_buf)
{
	HostCmd_DS_802_11_FW_WAKEUP_METHOD *fwwm = &cmd->params.fwwakeupmethod;
	u16 action = (u16)cmd_action;
	u16 method = *((u16 *)pdata_buf);

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_FW_WAKEUP_METHOD);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_FW_WAKEUP_METHOD) + S_DS_GEN);
	fwwm->Action = wlan_cpu_to_le16(action);
	switch (action) {
	case HostCmd_ACT_SET:
		fwwm->Method = wlan_cpu_to_le16(method);
		break;
	case HostCmd_ACT_GET:
	default:
		fwwm->Method = 0;
		break;
	}

	return WLAN_STATUS_SUCCESS;
}
#endif /* FW_WAKEUP_METHOD */
#endif /* PS_REQUIRED */

#ifdef HOST_SLEEP
/** 
 *  @brief This function prepares command of host_sleep_cfg.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param pdata_buf 	A pointer to HostCmd_DS_802_11_HOST_SLEEP_CFG structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_host_sleep_cfg(wlan_private * priv,
			HostCmd_DS_COMMAND * cmd,
			HostCmd_DS_802_11_HOST_SLEEP_CFG *pdata_buf)
{
	HostCmd_DS_802_11_HOST_SLEEP_CFG *phwuc = &cmd->params.hostsleepcfg;

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_HOST_SLEEP_CFG);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_HOST_SLEEP_CFG) + S_DS_GEN);
	phwuc->conditions = wlan_cpu_to_le32(pdata_buf->conditions);
	phwuc->gpio = pdata_buf->gpio;
	phwuc->gap = pdata_buf->gap;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of host_sleep_awake_confirm.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_host_sleep_awake_confirm(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd)
{
	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_HOST_SLEEP_AWAKE_CONFIRM);
	cmd->Size = wlan_cpu_to_le16(S_DS_GEN);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* HOST_SLEEP */

/** 
 *  @brief This function prepares command of inactivity_timeout.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   Action: GET SET
 *  @param pdata_buf 	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_inactivity_timeout(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	u16 cmd_action,
			       	void *pdata_buf)
{
	u16 *timeout = (u16 *) pdata_buf;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_INACTIVITY_TIMEOUT);
	cmd->Size    = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_INACTIVITY_TIMEOUT) + S_DS_GEN);

	cmd->params.inactivity_timeout.Action = wlan_cpu_to_le16(cmd_action);

	if (cmd_action)
		cmd->params.inactivity_timeout.Timeout = wlan_cpu_to_le16(*timeout);
	else
		cmd->params.inactivity_timeout.Timeout = 0;
		
	return WLAN_STATUS_SUCCESS;
}

#ifdef BG_SCAN
/** 
 *  @brief This function prepares command of bg_scan_config.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @param pdata_buf 	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_bg_scan_config(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	int cmd_action,
			       	void *pdata_buf)
{
	wlan_adapter *Adapter = priv->adapter;
	HostCmd_DS_802_11_BG_SCAN_CONFIG *bgcfg = &cmd->params.bgscancfg;
	BOOLEAN enable = *((BOOLEAN *)pdata_buf);

	cmd->Command = HostCmd_CMD_802_11_BG_SCAN_CONFIG;
	cmd->Size = (priv->adapter->bgScanConfigSize) + S_DS_GEN; 

	Adapter->bgScanConfig->Enable = enable;

	memcpy(bgcfg, Adapter->bgScanConfig, 
			Adapter->bgScanConfigSize);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of bg_scan_query.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_bg_scan_query(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd)
{
	HostCmd_DS_802_11_BG_SCAN_QUERY *bgquery = &cmd->params.bgscanquery;

	cmd->Command = HostCmd_CMD_802_11_BG_SCAN_QUERY;
	cmd->Size = sizeof(HostCmd_DS_802_11_BG_SCAN_QUERY) + S_DS_GEN; 

	bgquery->Flush = 1;

	return WLAN_STATUS_SUCCESS;
}
#endif /* BG_SCAN */


#ifdef SUBSCRIBE_EVENT_CTRL
/** 
 *  @brief This function prepares command of subscribe_events.
 *  
 *  @param priv    		A pointer to wlan_private structure
 *  @param cmd	   		A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 		the action: GET or SET
 *  @param pdata_buf		A pointer to data buffer
 *  @return 	   		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_subscribe_events(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	u16 cmd_action,
			       	void *pdata_buf)
{
	HostCmd_DS_802_11_SUBSCRIBE_EVENT	*events = &cmd->params.events;
	MrvlIEtypes_RssiParamSet_t		*Rssi;
	MrvlIEtypes_BeaconsMissed_t		*BcnMiss;
	MrvlIEtypes_SnrThreshold_t		*Snr;
	MrvlIEtypes_FailureCount_t		*FailCnt;
	EventSubscribe				*SubscribeEvent =
							(EventSubscribe *)pdata_buf;
	u16		ActualPos = sizeof(HostCmd_DS_802_11_SUBSCRIBE_EVENT);

	ENTER();

	HEXDUMP("SubscribeEvent-:", (u8 *)pdata_buf, sizeof(EventSubscribe));
	
#define TLV_PAYLOAD_SIZE	2	

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SUBSCRIBE_EVENT);
	
	events->Action = wlan_cpu_to_le16(SubscribeEvent->Action);
	events->Events = wlan_cpu_to_le16(SubscribeEvent->Events);

	Rssi = (MrvlIEtypes_RssiParamSet_t *)((s8 *)events + ActualPos);
	Rssi->Header.Type = wlan_cpu_to_le16(TLV_TYPE_RSSI);
	Rssi->Header.Len = wlan_cpu_to_le16(TLV_PAYLOAD_SIZE);
	Rssi->RSSIValue = SubscribeEvent->RSSIValue;
	Rssi->RSSIFreq = SubscribeEvent->RSSIFreq;
	
	ActualPos += sizeof(MrvlIEtypesHeader_t) + Rssi->Header.Len;

	Snr = (MrvlIEtypes_SnrThreshold_t *)((s8 *)events + ActualPos);
	Snr->Header.Type = wlan_cpu_to_le16(TLV_TYPE_SNR);
	Snr->Header.Len = wlan_cpu_to_le16(TLV_PAYLOAD_SIZE);
	Snr->SNRValue = SubscribeEvent->SNRValue;
	Snr->SNRFreq = SubscribeEvent->SNRFreq;

	ActualPos += sizeof(MrvlIEtypesHeader_t) + Snr->Header.Len;

	FailCnt = (MrvlIEtypes_FailureCount_t *)((s8 *)events + ActualPos);
	FailCnt->Header.Type = TLV_TYPE_FAILCOUNT;
        FailCnt->Header.Len = TLV_PAYLOAD_SIZE;
	FailCnt->FailValue = SubscribeEvent->FailValue;
	FailCnt->FailFreq = SubscribeEvent->FailFreq;
	
	ActualPos += sizeof(MrvlIEtypesHeader_t) + FailCnt->Header.Len;

	BcnMiss = (MrvlIEtypes_BeaconsMissed_t *)((s8 *)events + ActualPos);
	BcnMiss->Header.Type = wlan_cpu_to_le16(TLV_TYPE_BCNMISS);
	BcnMiss->Header.Len = wlan_cpu_to_le16(TLV_PAYLOAD_SIZE);
	BcnMiss->BeaconMissed = SubscribeEvent->BcnMissed;
	BcnMiss->Reserved = 0;

	ActualPos += sizeof(MrvlIEtypesHeader_t) + BcnMiss->Header.Len;

	cmd->Size = wlan_cpu_to_le16(ActualPos + S_DS_GEN);

	PRINTM(INFO, "ActualPos = %d\n", ActualPos);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;	
}
#endif /* SUBSCRIBE_EVENT_CTRL */

/** 
 *  @brief This function prepares command of sleep_period.
 *  
 *  @param priv    		A pointer to wlan_private structure
 *  @param cmd	   		A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 		the action: GET or SET
 *  @param pdata_buf		A pointer to data buffer
 *  @return 	   		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_sleep_period(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	u16 cmd_action,
			       	void *pdata_buf)
{
	cmd->Command =	wlan_cpu_to_le16(HostCmd_CMD_802_11_SLEEP_PERIOD);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_SLEEP_PERIOD) + 
								S_DS_GEN);
	memmove(&cmd->params.ps_sleeppd, pdata_buf,
			sizeof(HostCmd_DS_802_11_SLEEP_PERIOD));

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of sleep_params.
 *  
 *  @param priv    		A pointer to wlan_private structure
 *  @param cmd	   		A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 		the action: GET or SET
 *  @return 	   		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_sleep_params(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	u16 cmd_action)
{
	wlan_adapter *Adapter = priv->adapter;
	HostCmd_DS_802_11_SLEEP_PARAMS *sp = &cmd->params.sleep_params;

	ENTER();

	cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_SLEEP_PARAMS)) +
								S_DS_GEN);
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SLEEP_PARAMS);

	if (cmd_action == HostCmd_ACT_GEN_GET) {
		memset(&Adapter->sp, 0, sizeof(SleepParams));
		memset(sp, 0, sizeof(HostCmd_DS_802_11_SLEEP_PARAMS));
		sp->Action = wlan_cpu_to_le16(cmd_action);
	} else if (cmd_action == HostCmd_ACT_GEN_SET) {
		sp->Action = wlan_cpu_to_le16(cmd_action);
		sp->Error = wlan_cpu_to_le16(Adapter->sp.sp_error);
		sp->Offset = wlan_cpu_to_le16(Adapter->sp.sp_offset);
		sp->StableTime = wlan_cpu_to_le16(Adapter->sp.sp_stabletime);
		sp->CalControl = (u8) Adapter->sp.sp_calcontrol;
		sp->ExternalSleepClk = (u8) Adapter->sp.sp_extsleepclk;
		sp->Reserved = wlan_cpu_to_le16(Adapter->sp.sp_reserved);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of scan.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_scan(wlan_private * priv,
			HostCmd_DS_COMMAND * cmd)
{
#ifdef	PROGRESSIVE_SCAN
	int				j = 0;
#endif
	int				i = 0;
	wlan_adapter			*Adapter = priv->adapter;
	HostCmd_DS_802_11_SCAN 		*scan = &cmd->params.scan;
	CHANNEL_FREQ_POWER		*cfp;

	MrvlIEtypes_SsIdParamSet_t	*ssid;
	MrvlIEtypes_ChanListParamSet_t  *chan;
	MrvlIEtypes_RatesParamSet_t	*rates;
	MrvlIEtypes_NumProbes_t *scanprobes;

	u8				scanType;
	int ssidPos, chanPos, ratesPos, probPos =0;
#ifdef MULTI_BANDS
	int 				ret = WLAN_STATUS_SUCCESS;
	HostCmd_DS_802_11_BAND_CONFIG	bc;
#endif /* MULTI BANDS*/

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SCAN);
	
	ssidPos = 1 + MRVDRV_ETH_ADDR_LEN;

	scan->BSSType = Adapter->ScanMode;

	memcpy(scan->BSSID, Adapter->SpecificScanBSSID, ETH_ALEN);

	/* SetSsIdParamSet */
	ssid = (MrvlIEtypes_SsIdParamSet_t *)( (char *)scan + ssidPos );

 	ssid->Header.Type = wlan_cpu_to_le16(TLV_TYPE_SSID);
	ssid->Header.Len = 0; 

	if (Adapter->SetSpecificScanSSID) {
		PRINTM(INFO, "Setting SpecificScanSSID to %s\n",
			Adapter->SpecificScanSSID.Ssid);

		PRINTM(INFO, "Length of SpecificScanSSID is %u\n",
		       Adapter->SpecificScanSSID.SsidLength);

		/* Setup specific Ssid TLV */
		ssid->Header.Len = Adapter->SpecificScanSSID.SsidLength;
		memcpy(ssid->SsId, Adapter->SpecificScanSSID.Ssid,
				Adapter->SpecificScanSSID.SsidLength);
		scanType = HostCmd_SCAN_TYPE_ACTIVE;
	} else {
		scanType = Adapter->ScanType;
	}


	chanPos = ssidPos + ssid->Header.Len + sizeof( MrvlIEtypesHeader_t ); 
	chan = (MrvlIEtypes_ChanListParamSet_t *)( (char*)scan + chanPos );
	chan->Header.Type = wlan_cpu_to_le16(TLV_TYPE_CHANLIST);
	chan->Header.Len = 0;

#ifdef	PROGRESSIVE_SCAN
	{
		int k;

		if (Adapter->SetSpecificScanSSID == TRUE) 
 			k= MRVDRV_MAX_CHANNELS_PER_SCAN;
		else
			k = MRVDRV_CHANNELS_PER_SCAN;
		

		if (Adapter->ScanChannelsLeft <= 0) {
			Adapter->ScanChannelsLeft = 
				Adapter->cur_region_channel->NrCFP;
		}

		j = ((Adapter->ScanChannelsLeft - 1) / k) * k;

		cfp = Adapter->cur_region_channel->CFP;
		
		cfp = cfp + j;
		{ 
		int i;
		for ( i = 0; i < Adapter->cur_region_channel->NrCFP; i++ ) 
			PRINTM(INFO, "%x::ChNo:%x\n", i, 
				(Adapter->cur_region_channel->CFP+i)->Channel);
		}

		PRINTM(INFO, "SCAN_CMD: k = %x j=%x NrCFP=%x Left=%x\n", k, j, 
				Adapter->cur_region_channel->NrCFP,
				Adapter->ScanChannelsLeft );	
		for (i = j; (((i - j) < k) && 
				(i < Adapter->cur_region_channel->NrCFP));
								i++, cfp++) {
#ifdef MULTI_BANDS
			if ( Adapter->cur_region_channel->Band == BAND_B || 
				Adapter->cur_region_channel->Band == BAND_G ) {
				chan->ChanScanParam[i-j].RadioType = 
					HostCmd_SCAN_RADIO_TYPE_BG;
			}
			else
				chan->ChanScanParam[i-j].RadioType = 
					HostCmd_SCAN_RADIO_TYPE_A;
#else
			chan->ChanScanParam[i-j].RadioType = 
				HostCmd_SCAN_RADIO_TYPE_BG;
#endif

#ifdef ENABLE_802_11D
			if( wlan_get_state_11d( priv) == ENABLE_11D ) {
				scanType = wlan_get_scan_type_11d(
						cfp->Channel, 
						&Adapter->parsed_region_chan
						);
			}

#endif
			chan->ChanScanParam[i-j].ScanType = scanType;
		
			if ( scanType == HostCmd_SCAN_TYPE_PASSIVE ) 
				chan->ChanScanParam[i-j].MaxScanTime = 
				   wlan_cpu_to_le16(HostCmd_SCAN_PASSIVE_MAX_CH_TIME);
			else
				chan->ChanScanParam[i-j].MaxScanTime = 
				   wlan_cpu_to_le16(HostCmd_SCAN_MAX_CH_TIME);
		
			chan->ChanScanParam[i-j].ChanNumber = cfp->Channel;
			
			chan->Header.Len += sizeof( ChanScanParamSet_t);

			PRINTM(INFO, "SCAN_CMD: ch[%d] ScanType=%d\n", cfp->Channel, scanType);

			PRINTM(INFO, "SCAN_CMD: Max=%d\n", chan->ChanScanParam[i-j].MaxScanTime );
	
			PRINTM(INFO, "SCAN_CMD: chan=%d len=%d(i=%d, j=%d)\n", i-j, 
						chan->Header.Len, i, j );
		}
		
		Adapter->ScanChannelsLeft = j;
	}
#endif

	probPos = chanPos + chan->Header.Len + sizeof(MrvlIEtypesHeader_t);
	scanprobes = (MrvlIEtypes_NumProbes_t *)( (char *)scan + probPos );
  	scanprobes->Header.Len = wlan_cpu_to_le16(0);

	/* Handle number of probes */
	if (Adapter->ScanProbes != 0)
	{
		scanprobes->Header.Type = wlan_cpu_to_le16(TLV_TYPE_NUMPROBES);
		scanprobes->Header.Len = wlan_cpu_to_le16(2);
		scanprobes->NumProbes = wlan_cpu_to_le16(Adapter->ScanProbes);
	}

	/* Handle the Rates */
	if (Adapter->ScanProbes != 0)
		ratesPos = probPos+ scanprobes->Header.Len + sizeof(MrvlIEtypesHeader_t);
	else
		ratesPos = probPos;
	rates = (MrvlIEtypes_RatesParamSet_t *)( (char *)scan + ratesPos );
	rates->Header.Type = wlan_cpu_to_le16(TLV_TYPE_RATES);

#ifdef MULTI_BANDS
	if (Adapter->cur_region_channel->Band == BAND_A) {
		memcpy(rates->Rates, SupportedRates_A, 
						sizeof(SupportedRates_A));
		rates-> Header.Len  = wlan_cpu_to_le16(sizeof ( SupportedRates_A )); 
	} else if (Adapter->cur_region_channel->Band == BAND_B) {
		memcpy(rates->Rates, SupportedRates_B, 
						sizeof(SupportedRates_B));
		rates-> Header.Len  = wlan_cpu_to_le16(sizeof ( SupportedRates_B )); 
	} else if (Adapter->cur_region_channel->Band == BAND_G) {
		memcpy(rates->Rates, SupportedRates_G, 
						sizeof(SupportedRates_G));
		rates-> Header.Len  = wlan_cpu_to_le16(sizeof ( SupportedRates_G )); 
	}

	/* We had to send BAND_CONFIG for every scan command */
	if (Adapter->is_multiband) {
		bc.BandSelection = Adapter->cur_region_channel->Band;
		
		if ((Adapter->MediaConnectStatus == WlanMediaStateConnected) &&
			(Adapter->CurBssParams.band == bc.BandSelection))
			bc.Channel = Adapter->CurBssParams.channel;
		else
			bc.Channel = Adapter->cur_region_channel->CFP->Channel;

		ret = PrepareAndSendCommand(priv, 
					HostCmd_CMD_802_11_BAND_CONFIG,
					HostCmd_ACT_SET, 0, 0, &bc);

		if (ret) {
			LEAVE();
			return WLAN_STATUS_FAILURE;
		}
	}

#else
	memcpy(rates->Rates, SupportedRates, sizeof(SupportedRates));
	rates-> Header.Len  = wlan_cpu_to_le16(sizeof ( SupportedRates )); 
#endif /* MULTI_BAND */

	cmd->Size = wlan_cpu_to_le16(ratesPos + rates->Header.Len + 
				sizeof( MrvlIEtypesHeader_t ) + S_DS_GEN);

	Adapter->bIsScanInProgress = TRUE;

	PRINTM(INFO, "SCAN_CMD: SCAN command is ready.\n");

	PRINTM(INFO, "SCAN_CMD: ssidPos=%d chanPos=%d Rates=%d\n",
		       	ssidPos, chanPos, ratesPos);

	HEXDUMP("SCAN_CMD:", (u8 *)cmd, (int)cmd->Size );

	PRINTM(INFO, "SCAN_CMD: Command=%x, Size=%x, SeqNum=%x\n",
		       cmd->Command, cmd->Size, cmd->SeqNum);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of association.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param pssid	A pointer to wlan_802_11_ssid structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_associate(wlan_private * priv,
			  HostCmd_DS_COMMAND * cmd, 
			  void *pssid)
{
	int				ret = WLAN_STATUS_SUCCESS;
	int		           	i;
	wlan_adapter			*Adapter = priv->adapter;

	HostCmd_DS_802_11_ASSOCIATE 	*pAsso = &cmd->params.associate;
	u8  				*pReqBSSID = NULL;
	WLAN_802_11_BSSID		*pBSSIDList;
	u8				*card_rates;
	int				card_rates_size;
#ifdef MULTI_BANDS
	HostCmd_DS_802_11_BAND_CONFIG	bc;
#endif /* MULTI BANDS*/
	u16				TmpCap;
	u8				*pos = (u8 *)pAsso;
	MrvlIEtypes_SsIdParamSet_t	*ssid;
	MrvlIEtypes_PhyParamSet_t	*phy;
	MrvlIEtypes_SsParamSet_t	*ss;
	MrvlIEtypes_RatesParamSet_t	*rates;
#ifdef WPA
	MrvlIEtypes_RsnParamSet_t	*rsn;
#endif
#ifdef WMM
	MrvlIEtypes_WmmParamSet_t	*wmm;
#endif

	ENTER();

	if (!Adapter) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_ASSOCIATE);

	if (Adapter->RequestViaBSSID)
		pReqBSSID = Adapter->RequestedBSSID;
    
	/*
	 * find out the BSSID that matches 
	 * the SSID given in pssid
	 */
	PRINTM(INFO, "ASSOC_CMD: NumOfBSSIDs = %u\n", Adapter->ulNumOfBSSIDs);
	PRINTM(INFO, "ASSOC_CMD: Wanted SSID = %s\n", ((WLAN_802_11_SSID *)pssid)->Ssid);

	i = FindSSIDInList(Adapter, pssid, pReqBSSID, Wlan802_11Infrastructure);

	if (i >= 0) {
		memmove(pAsso->PeerStaAddr, &(Adapter->BSSIDList[i].MacAddress),
				ETH_ALEN);
	} else {
		PRINTM(INFO, "ASSOC_CMD: SSID is not in the list\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	pos += sizeof(pAsso->PeerStaAddr);

	/* Set the temporary BSSID Index */
	Adapter->ulAttemptedBSSIDIndex = i;
    	pBSSIDList = &Adapter->BSSIDList[Adapter->ulAttemptedBSSIDIndex];


	/* set preamble to firmware */
	if (Adapter->capInfo.ShortPreamble && pBSSIDList->Cap.ShortPreamble) {
		Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
	} else {
		Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
	}

	SetRadioControl(priv);
	
   	 /* set the listen interval */
	pAsso->ListenInterval = Adapter->ListenInterval;

	pos += sizeof(pAsso->CapInfo);
	pos += sizeof(pAsso->ListenInterval);
	pos += sizeof(pAsso->BcnPeriod);
	pos += sizeof(pAsso->DtimPeriod);

	ssid = (MrvlIEtypes_SsIdParamSet_t *)pos;
 	ssid->Header.Type = wlan_cpu_to_le16(TLV_TYPE_SSID);
	ssid->Header.Len = pBSSIDList->Ssid.SsidLength;
	memcpy(ssid->SsId, pBSSIDList->Ssid.Ssid, ssid->Header.Len);
	pos += sizeof(ssid->Header) + ssid->Header.Len;

	phy = (MrvlIEtypes_PhyParamSet_t *)pos;
 	phy->Header.Type = wlan_cpu_to_le16(TLV_TYPE_PHY_DS);
	phy->Header.Len = sizeof(phy->fh_ds.DsParamSet);
	memcpy(&phy->fh_ds.DsParamSet,
           		&pBSSIDList->PhyParamSet.DsParamSet.CurrentChan,
           		sizeof(phy->fh_ds.DsParamSet));
	pos += sizeof(phy->Header) + phy->Header.Len;
    
	ss = (MrvlIEtypes_SsParamSet_t *)pos;
 	ss->Header.Type = wlan_cpu_to_le16(TLV_TYPE_CF);
	ss->Header.Len = sizeof(ss->cf_ibss.CfParamSet);
	pos += sizeof(ss->Header) + ss->Header.Len;
    
	rates = (MrvlIEtypes_RatesParamSet_t *)pos;
 	rates->Header.Type = wlan_cpu_to_le16(TLV_TYPE_RATES);
    
	memcpy(&rates->Rates,&pBSSIDList->SupportedRates, WLAN_SUPPORTED_RATES);
    
#ifdef	MULTI_BANDS
	if (pBSSIDList->bss_band == BAND_A) {
		card_rates = SupportedRates_A;
		card_rates_size = sizeof(SupportedRates_A);
	}
	else if (pBSSIDList->bss_band == BAND_B) {
		card_rates = SupportedRates_B;
		card_rates_size = sizeof(SupportedRates_B);
	}
	else if (pBSSIDList->bss_band == BAND_G) {
		card_rates = SupportedRates_G;
		card_rates_size = sizeof(SupportedRates_G);
	}
	else {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#else
	card_rates = SupportedRates;
	card_rates_size = sizeof(SupportedRates);
#endif	/* MULTI_BANDS*/

	if (get_common_rates(Adapter, rates->Rates, WLAN_SUPPORTED_RATES,
                         card_rates, card_rates_size)) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	rates->Header.Len = MIN(strlen(rates->Rates), WLAN_SUPPORTED_RATES);
	Adapter->CurBssParams.NumOfRates = rates->Header.Len;	

	pos += sizeof(rates->Header) + rates->Header.Len;

#ifdef WPA
	if (Adapter->SecInfo.WPAEnabled
#ifdef WPA2
		|| Adapter->SecInfo.WPA2Enabled
#endif /* WPA2 */
		)
    	{
		rsn = (MrvlIEtypes_RsnParamSet_t *)pos;
 		rsn->Header.Type = (u16)Adapter->Wpa_ie[0];	/* WPA_IE or WPA2_IE */
 		rsn->Header.Type = wlan_cpu_to_le16(rsn->Header.Type);
 		rsn->Header.Len = (u16)Adapter->Wpa_ie[1];
		memcpy(rsn->RsnIE, &Adapter->Wpa_ie[2], rsn->Header.Len);
		HEXDUMP("ASSOC_CMD: RSN IE", (u8*)rsn, sizeof(rsn->Header) + rsn->Header.Len); 
		pos += sizeof(rsn->Header) + rsn->Header.Len;
	}
#endif /* WPA */

#ifdef WMM
	if (Adapter->wmm.required && pBSSIDList->Wmm_IE[0] == WMM_IE) {
		Adapter->CurrentPacketFilter |= HostCmd_ACT_MAC_WMM_ENABLE;
		SetMacPacketFilter(priv);
        
		wmm = (MrvlIEtypes_WmmParamSet_t *)pos;
 		wmm->Header.Type = (u16)wmm_ie[0];
 		wmm->Header.Type = wlan_cpu_to_le16(wmm->Header.Type);
 		wmm->Header.Len = (u16)wmm_ie[1];
		memcpy(wmm->WmmIE, &wmm_ie[2], wmm->Header.Len);
#ifdef WMM_UAPSD
        if((pBSSIDList->Wmm_IE[WMM_QOS_INFO_OFFSET] & WMM_QOS_INFO_UAPSD_BIT) 
#define QOS_INFO_PARA_MASK 0x0f
           && ((Adapter->wmm.qosinfo & QOS_INFO_PARA_MASK) != 0)) {
            memcpy((u8 *)wmm->WmmIE + wmm->Header.Len - sizeof(Adapter->wmm.qosinfo),
                   &Adapter->wmm.qosinfo, sizeof(Adapter->wmm.qosinfo));
        }
#endif /* WMM_UAPSD */
		HEXDUMP("ASSOC_CMD: WMM IE", (u8*)wmm, sizeof(wmm->Header) + wmm->Header.Len); 
		pos += sizeof(wmm->Header) + wmm->Header.Len;
	}
	else {
		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WMM_ENABLE;
		SetMacPacketFilter(priv);
	}
#endif /* WMM */
    
#ifdef REASSOC_CMD_SUPPORT
    	wlan_cmd_append_reassoc_tlv(priv, &pos);
#endif

#ifdef IWGENIE_SUPPORT
    	wlan_cmd_append_generic_ie(priv, &pos);
#endif

#ifdef CCX
#ifdef ENABLE_802_11H
    	wlan_ccx_process_association_req(&pos, &pBSSIDList->ccx_bss_info,
                                     !pBSSIDList->wlan_11h_bss_info.sensed11h);
#else
    	wlan_ccx_process_association_req(&pos, &pBSSIDList->ccx_bss_info, TRUE);
#endif
#endif
    
	/* update CurBssParams */
 	Adapter->CurBssParams.channel = (pBSSIDList
                                     ->PhyParamSet.DsParamSet.CurrentChan);
#ifdef MULTI_BANDS
	Adapter->CurBssParams.band = pBSSIDList->bss_band;
#endif	/* MULTI_BANDS*/
    
	/* Copy the infra. association rates into Current BSS state structure */
	memcpy(&Adapter->CurBssParams.DataRates, &rates->Rates, 
           MIN(sizeof(Adapter->CurBssParams.DataRates), rates->Header.Len));
    
	PRINTM(INFO, "ASSOC_CMD: rates->Header.Len = %d\n", rates->Header.Len);
    
	/* set IBSS field */
	if (pBSSIDList->InfrastructureMode == Wlan802_11Infrastructure) {
#define CAPINFO_ESS_MODE 1
		pAsso->CapInfo.Ess = CAPINFO_ESS_MODE;
    	}

#ifdef ENABLE_802_11D
   	if (wlan_parse_dnld_countryinfo_11d( priv )) {
       		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif

#ifdef ENABLE_802_11H
    	/*
    	 * Call 11h join API after capability bits are set so adhoc/infra 11h 
     	 * behavior can be properly triggered.  pos modified if data is appended
    	 */
    	wlan_11h_process_join(priv, &pos, &pAsso->CapInfo, 
                          pBSSIDList->PhyParamSet.DsParamSet.CurrentChan,
                          &pBSSIDList->wlan_11h_bss_info);
#endif

#ifdef MULTI_BANDS
	if (Adapter->is_multiband) {
		bc.BandSelection = Adapter->CurBssParams.band;
		bc.Channel = Adapter->CurBssParams.channel;

		ret = PrepareAndSendCommand(priv,
                                    HostCmd_CMD_802_11_BAND_CONFIG,
                                    HostCmd_ACT_SET, 
                                    HostCmd_OPTION_WAITFORRSP,
                                    0, &bc);
        
		if (ret) {
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}
#endif /* MULTI_BANDS */

	cmd->Size = wlan_cpu_to_le16((u16)(pos - (u8 *)pAsso) + S_DS_GEN);

	/* set the Capability info at last */
	memcpy(&TmpCap, &pBSSIDList->Cap, sizeof(pAsso->CapInfo));
	TmpCap &= CAPINFO_MASK;
	PRINTM(INFO, "ASSOC_CMD: TmpCap=%4X CAPINFO_MASK=%4X\n", TmpCap, CAPINFO_MASK );
	TmpCap = wlan_cpu_to_le16(TmpCap);
	memcpy(&pAsso->CapInfo, &TmpCap, sizeof(pAsso->CapInfo));

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function prepares command of deauthenticat.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_deauthenticate(wlan_private * priv,
			       		HostCmd_DS_COMMAND * cmd)
{
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	HostCmd_DS_802_11_DEAUTHENTICATE *dauth = &cmd->params.deauth;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_DEAUTHENTICATE);
	cmd->Size = 
		wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_DEAUTHENTICATE) + S_DS_GEN);

	/* set AP MAC address */
	memmove(dauth->MacAddr, Adapter->CurrentBSSID, MRVDRV_ETH_ADDR_LEN);
	
	/* Reason code 3 = Station is leaving */
#define REASON_CODE_STA_LEAVING 3
	dauth->ReasonCode = wlan_cpu_to_le16(REASON_CODE_STA_LEAVING);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#define WEP_40_BIT_LEN	5
#define WEP_104_BIT_LEN	13

/** 
 *  @brief This function prepares command of set_wep.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_oid   OID: ADD_WEP KEY or REMOVE_WEP KEY
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_set_wep(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
		       		u32 cmd_oid)
{
	HostCmd_DS_802_11_SET_WEP	*wep = &cmd->params.wep;
	wlan_adapter   			*Adapter = priv->adapter;
	int				ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (cmd_oid == OID_802_11_ADD_WEP) {
		cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SET_WEP);
		cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_SET_WEP)) + S_DS_GEN);
		wep->Action = wlan_cpu_to_le16(HostCmd_ACT_ADD);

		/* default tx key index */
		wep->KeyIndex = wlan_cpu_to_le16(Adapter->CurrentWepKeyIndex &
			       					HostCmd_WEP_KEY_INDEX_MASK);
	
		PRINTM(INFO, "Tx Key Index: %u\n", wep->KeyIndex);

		switch (Adapter->WepKey[0].KeyLength) {
			case WEP_40_BIT_LEN:
				wep->WEPTypeForKey1 = HostCmd_TYPE_WEP_40_BIT;
				memmove(wep->WEP1, Adapter->WepKey[0].KeyMaterial, Adapter->WepKey[0].KeyLength);
				break;
			case WEP_104_BIT_LEN:
				wep->WEPTypeForKey1 = HostCmd_TYPE_WEP_104_BIT;
				memmove(wep->WEP1, Adapter->WepKey[0].KeyMaterial, Adapter->WepKey[0].KeyLength);
				break;
			case 0:
				break;
			default:
				PRINTM(INFO, "Key1 Length = %d is incorrect\n", Adapter->WepKey[0].KeyLength);
				ret = WLAN_STATUS_FAILURE;
				goto done;
		}

		switch (Adapter->WepKey[1].KeyLength) {
			case WEP_40_BIT_LEN:
				wep->WEPTypeForKey2 = HostCmd_TYPE_WEP_40_BIT;
				memmove(wep->WEP2, Adapter->WepKey[1].KeyMaterial, Adapter->WepKey[1].KeyLength);
				break;
			case WEP_104_BIT_LEN:
				wep->WEPTypeForKey2 = HostCmd_TYPE_WEP_104_BIT;
				memmove(wep->WEP2, Adapter->WepKey[1].KeyMaterial, Adapter->WepKey[1].KeyLength);
				break;
			case 0:
				break;
			default:
				PRINTM(INFO, "Key2 Length = %d is incorrect\n", Adapter->WepKey[1].KeyLength);
				ret = WLAN_STATUS_FAILURE;
				goto done;
		}

		switch (Adapter->WepKey[2].KeyLength) {
			case WEP_40_BIT_LEN:
				wep->WEPTypeForKey3 = HostCmd_TYPE_WEP_40_BIT;
				memmove(wep->WEP3, Adapter->WepKey[2].KeyMaterial, Adapter->WepKey[2].KeyLength);
				break;
			case WEP_104_BIT_LEN:
				wep->WEPTypeForKey3 = HostCmd_TYPE_WEP_104_BIT;
				memmove(wep->WEP3, Adapter->WepKey[2].KeyMaterial, Adapter->WepKey[2].KeyLength);
				break;
			case 0:
				break;
			default:
				PRINTM(INFO, "Key3 Length = %d is incorrect\n", Adapter->WepKey[2].KeyLength);
				ret = WLAN_STATUS_FAILURE;
				goto done;
		}

		switch (Adapter->WepKey[3].KeyLength) {
			case WEP_40_BIT_LEN:
				wep->WEPTypeForKey4 = HostCmd_TYPE_WEP_40_BIT;
				memmove(wep->WEP4, Adapter->WepKey[3].KeyMaterial, Adapter->WepKey[3].KeyLength);
				break;
			case WEP_104_BIT_LEN:
				wep->WEPTypeForKey4 = HostCmd_TYPE_WEP_104_BIT;
				memmove(wep->WEP4, Adapter->WepKey[3].KeyMaterial, Adapter->WepKey[3].KeyLength);
				break;
			case 0:
				break;
			default:
				PRINTM(INFO, "Key4 Length = %d is incorrect\n", Adapter->WepKey[3].KeyLength);
				ret = WLAN_STATUS_FAILURE;
				goto done;
		}
	}
	else if (cmd_oid == OID_802_11_REMOVE_WEP) {
		cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SET_WEP);
		cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_SET_WEP)) + S_DS_GEN);
		wep->Action = wlan_cpu_to_le16(HostCmd_ACT_REMOVE);

		/* default tx key index */
		wep->KeyIndex = wlan_cpu_to_le16((u16) (Adapter->CurrentWepKeyIndex & 
				(u32) HostCmd_WEP_KEY_INDEX_MASK));
	}

	ret = WLAN_STATUS_SUCCESS;
done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function prepares command of ad_hoc_start.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param pssid	A pointer to WLAN_802_11_SSID structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_ad_hoc_start(wlan_private * priv,
			     	HostCmd_DS_COMMAND * cmd,
			     	void *pssid)
{
	wlan_adapter   			*Adapter = priv->adapter;
	HostCmd_DS_802_11_AD_HOC_START 	*adhs = &cmd->params.ads;
	int     	        	i, j;
#ifdef MULTI_BANDS
	HostCmd_DS_802_11_BAND_CONFIG 	bc;
#endif /* MULTI BANDS */
#ifdef ENABLE_802_11H
	u8   *pos = (u8 *)adhs + sizeof(HostCmd_DS_802_11_AD_HOC_START);
#endif
	int  cmdAppendSize = 0;
	u16				TmpCap;
	WLAN_802_11_BSSID		*pBSSIDList;
	int 				ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (!Adapter) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_START);

	if (Adapter->AdHocCreated && Adapter->ulNumOfBSSIDs)
		Adapter->ulNumOfBSSIDs--;

	i = Adapter->ulNumOfBSSIDs;

	/* add a new entry in the BSSID list */
	if (Adapter->ulNumOfBSSIDs < MRVDRV_MAX_BSSID_LIST) {
		Adapter->ulAttemptedBSSIDIndex = i;
		Adapter->ulNumOfBSSIDs++;
	}

   	pBSSIDList = &Adapter->BSSIDList[Adapter->ulAttemptedBSSIDIndex];

	/*
	 * Fill in the parameters for 2 data structures:
	 * 1. HostCmd_DS_802_11_AD_HOC_START Command
	 * 2. Adapter->BSSIDList[i]
	 * Driver will fill up SSID, BSSType,IBSS param, Physical Param,
	 * probe delay, and Cap info.
	 * Firmware will fill up 		ret = WLAN_STATUS_FAILURE;
		goto done;beacon period, DTIM, Basic rates 
	 * and operational rates. 
	 */
	for (j = 0; j < MRVDRV_MAX_SSID_LENGTH; j++) {
		if (((PWLAN_802_11_SSID) pssid)->Ssid[j] == '\0')
		break;
	}

	memset(adhs->SSID, 0, MRVDRV_MAX_SSID_LENGTH);
	
	memcpy(adhs->SSID,((PWLAN_802_11_SSID) pssid)->Ssid,
			((PWLAN_802_11_SSID) pssid)->SsidLength);

	PRINTM(INFO, "ADHOC_S_CMD: adhcs-ssid is %s\n", adhs->SSID);

	memset(pBSSIDList->Ssid.Ssid, 0, MRVDRV_MAX_SSID_LENGTH);
	memcpy(pBSSIDList->Ssid.Ssid,
			((PWLAN_802_11_SSID) pssid)->Ssid,
			((PWLAN_802_11_SSID) pssid)->SsidLength);

	pBSSIDList->Ssid.SsidLength =
		((PWLAN_802_11_SSID) pssid)->SsidLength;

	/* Set the length of Adapter->BSSIDList[i] */
	pBSSIDList->Length = sizeof(WLAN_802_11_BSSID);

	/* set the length of configuration in Adapter->BSSIDList[i] */
	pBSSIDList->Configuration.Length = sizeof(WLAN_802_11_CONFIGURATION);

	/* set the BSS type */
	adhs->BSSType = HostCmd_BSS_TYPE_IBSS;
	pBSSIDList->InfrastructureMode = Wlan802_11IBSS;

	/* set Physical param set */
#define DS_PARA_IE_ID	3
#define DS_PARA_IE_LEN  1
	
	adhs->PhyParamSet.DsParamSet.ElementId = DS_PARA_IE_ID;
	adhs->PhyParamSet.DsParamSet.Len = DS_PARA_IE_LEN;

	if (Adapter->AdhocChannel == 0) 
    	{
#ifdef MULTI_BANDS
		if (Adapter->adhoc_start_band == BAND_A) 
        	{
#ifdef ENABLE_802_11H
			Adapter->AdhocChannel = wlan_11h_get_adhoc_start_channel(priv);
#else
			Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL_A;
#endif /* ENABLE_802_11H */
        	}
       		else
#endif /* MULTI_BANDS */
        	{ 
            		Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
        	}
	}

	PRINTM(INFO, "ADHOC_S_CMD: Creating ADHOC on Channel %d\n", Adapter->AdhocChannel);
	
	Adapter->CurBssParams.channel = Adapter->AdhocChannel;
#ifdef MULTI_BANDS
	Adapter->CurBssParams.band = Adapter->adhoc_start_band;
#endif

	pBSSIDList->Channel = Adapter->AdhocChannel;
	adhs->PhyParamSet.DsParamSet.CurrentChan = Adapter->AdhocChannel;

	memcpy(&pBSSIDList->PhyParamSet,
			&adhs->PhyParamSet, sizeof(IEEEtypes_PhyParamSet_t));

	pBSSIDList->NetworkTypeInUse = Wlan802_11DS;
	pBSSIDList->Configuration.DSConfig 
        		= DSFreqList[adhs->PhyParamSet.DsParamSet.CurrentChan];

	/* set IBSS param set */
#define IBSS_PARA_IE_ID	  6
#define IBSS_PARA_IE_LEN  2
	
	adhs->SsParamSet.IbssParamSet.ElementId = IBSS_PARA_IE_ID;
	adhs->SsParamSet.IbssParamSet.Len = IBSS_PARA_IE_LEN;
	adhs->SsParamSet.IbssParamSet.AtimWindow = Adapter->AtimWindow;
	memcpy(&pBSSIDList->SsParamSet,
          	&(adhs->SsParamSet),
	       	sizeof(IEEEtypes_SsParamSet_t));

	/* set Capability info */
	adhs->Cap.Ess = 0;
	adhs->Cap.Ibss = 1;
	pBSSIDList->Cap.Ibss = 1;

	/* ProbeDelay */
	adhs->ProbeDelay = wlan_cpu_to_le16(HostCmd_SCAN_PROBE_DELAY_TIME);

	/* set up privacy in Adapter->BSSIDList[i] */
	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled
#ifdef ADHOCAES
		|| Adapter->AdhocAESEnabled
#endif
		) {

#define AD_HOC_CAP_PRIVACY_ON 1		
		PRINTM(INFO, "ADHOC_S_CMD: WEPStatus set, BSSIDList[i].Privacy to WEP\n");
		pBSSIDList->Privacy = Wlan802_11PrivFilter8021xWEP;
		adhs->Cap.Privacy = AD_HOC_CAP_PRIVACY_ON;
	} else {
		PRINTM(INFO, "ADHOC_S_CMD: WEPStatus NOTSET, Setting "
				"BSSIDList[i].Privacy to ACCEPT ALL\n");
		pBSSIDList->Privacy = Wlan802_11PrivFilterAcceptAll;
	}

  	memset(adhs->DataRate, 0, sizeof(adhs->DataRate));

#ifdef	MULTI_BANDS
	if (Adapter->adhoc_start_band & BAND_A)	{
		memcpy(adhs->DataRate, AdhocRates_A, 
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_A)));
   	} else if (Adapter->adhoc_start_band & BAND_B) {
		memcpy(adhs->DataRate, AdhocRates_B, 
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_B)));
    	} else if (Adapter->adhoc_start_band & BAND_G) {
		memcpy(adhs->DataRate, AdhocRates_G, 
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_G)));
    	}
#else 
#ifdef ADHOC_GRATE
	if (Adapter->adhoc_grate_enabled == TRUE) {
		memcpy(adhs->DataRate, AdhocRates_G, 
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_G)));
    	} else {
		memcpy(adhs->DataRate, AdhocRates_B,
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_B)));
    	}
#else
	memcpy(adhs->DataRate, AdhocRates_B, 
           MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_B)));
#endif /* ADHOC_GRATE */
#endif /* MULTI_BANDS */

	/* Find the last non zero */
	for (i = 0; i < sizeof(adhs->DataRate) && adhs->DataRate[i]; i++)
       		;

	Adapter->CurBssParams.NumOfRates = i;

	/* Copy the ad-hoc creating rates into Current BSS state structure */
	memcpy(&Adapter->CurBssParams.DataRates,
		       	&adhs->DataRate, 
           		Adapter->CurBssParams.NumOfRates);

	PRINTM(INFO, "ADHOC_S_CMD: Rates=%02x %02x %02x %02x \n", 
			adhs->DataRate[0], adhs->DataRate[1],
			adhs->DataRate[2], adhs->DataRate[3]);

	PRINTM(INFO, "ADHOC_S_CMD: AD HOC Start command is ready\n");

#ifdef MULTI_BANDS
	if (Adapter->is_multiband) {
		bc.BandSelection = Adapter->CurBssParams.band;
		bc.Channel = Adapter->CurBssParams.channel;

		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_BAND_CONFIG,
				HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
				0, &bc);

		if (ret) {
			PRINTM(INFO, "ADHOC_S_CMD: BAND_CONFIG failed\n");
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}
#endif /* MULTI_BANDS */

#ifdef ENABLE_802_11D
	if (wlan_create_dnld_countryinfo_11d( priv )) {
		PRINTM(INFO, "ADHOC_S_CMD: dnld_countryinfo_11d failed\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif

#ifdef ENABLE_802_11H
    	/*
    	 * Call 11h start API to add any 11h flags/elements as TLV parameters
    	*/
    	cmdAppendSize = wlan_11h_process_start(priv, &pos, &adhs->Cap, 
                                           Adapter->AdhocChannel,
                                           &pBSSIDList->wlan_11h_bss_info);
#endif

	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_AD_HOC_START) 
                                 + S_DS_GEN + cmdAppendSize);

	memcpy(&TmpCap, &adhs->Cap, sizeof(u16));
	TmpCap = wlan_cpu_to_le16(TmpCap);
	memcpy(&adhs->Cap, &TmpCap, sizeof(u16));

	ret = WLAN_STATUS_SUCCESS;
done: 	
	LEAVE();	
	return ret;
	
}

/** 
 *  @brief This function prepares command of ad_hoc_stop.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_ad_hoc_stop(wlan_private * priv, 
				HostCmd_DS_COMMAND * cmd)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_STOP);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_AD_HOC_STOP)
                                 + S_DS_GEN);

#ifdef ENABLE_802_11H
    	if (wlan_11h_is_active(priv)) {
	        wlan_11h_activate(priv, FALSE);
	}
#endif

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of ad_hoc_join.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param pssid	A pointer to WLAN_802_11_SSID structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_ad_hoc_join(wlan_private * priv, 
				HostCmd_DS_COMMAND * cmd,
				void *pssid)
{
	int				ret = WLAN_STATUS_SUCCESS;

	HostCmd_DS_802_11_AD_HOC_JOIN 	*pAdHocJoin = &cmd->params.adj;
	wlan_adapter   			*Adapter = priv->adapter;
	int             		i;
	u8				*pReqBSSID = NULL;
	u8				*card_rates;
	int				card_rates_size;
	u16				TmpCap;
#ifdef MULTI_BANDS
	HostCmd_DS_802_11_BAND_CONFIG bc;
#endif /* MULTI BANDS*/
#ifdef ENABLE_802_11H
	u8   *pos = (u8 *)pAdHocJoin + sizeof(HostCmd_DS_802_11_AD_HOC_JOIN);
#endif
    	int cmdAppendSize = 0;
    	WLAN_802_11_BSSID		*pBSSIDList;

	ENTER();
    
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_JOIN);
    
	if (Adapter->RequestViaBSSID)
		pReqBSSID = Adapter->RequestedBSSID;
    
	/*
	 * find out the BSSID that matches 
	 */
	for (i = 0; i < Adapter->ulNumOfBSSIDs; i++) {
		/* if found a match on SSID, get the MAC address(BSSID) */
		if (!SSIDcmp(&Adapter->BSSIDList[i].Ssid,
					(PWLAN_802_11_SSID)pssid)) {
			if (!pReqBSSID || !memcmp(Adapter->BSSIDList[i].
						MacAddress, pReqBSSID, 
						ETH_ALEN)) {
					if (Adapter->BSSIDList[i].InfrastructureMode !=
								Wlan802_11Infrastructure) {
					break;
				}
			}
		}
	}

	/* check if the matching SSID is found, if not, return */
	if (i == Adapter->ulNumOfBSSIDs) {
		PRINTM(INFO, "ADHOC_J_CMD: SSID is not found");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	
	PRINTM(INFO, "ADHOC_J_CMD: CurBss.ssid =%s\n", Adapter->CurBssParams.ssid.Ssid);
	PRINTM(INFO, "ADHOC_J_CMD: CurBss.ssid_len =%u\n", Adapter->CurBssParams.ssid.SsidLength);
	PRINTM(INFO, "ADHOC_J_CMD: pssid->ssid =%s\n",
            			((PWLAN_802_11_SSID) pssid)->Ssid);
	PRINTM(INFO, "ADHOC_J_CMD: pssid->len =%u\n",
            			((PWLAN_802_11_SSID) pssid)->SsidLength);

	/* check if the requested SSID is already joined */
	if (Adapter->CurBssParams.ssid.SsidLength &&
        		!SSIDcmp((PWLAN_802_11_SSID)pssid,
                 	&Adapter->CurBssParams.ssid)) 
	{
		if (Adapter->CurrentBSSIDDescriptor.InfrastructureMode ==
					Wlan802_11IBSS) {
			PRINTM(INFO, "ADHOC_J_CMD: New ad-hoc SSID is the same as current, "
					"not attempting to re-join");
            
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}

	/* Set the temporary BSSID Index */
	Adapter->ulAttemptedBSSIDIndex = i;
	pBSSIDList = &Adapter->BSSIDList[i];

	pAdHocJoin->BssDescriptor.BSSType = HostCmd_BSS_TYPE_IBSS;

	pAdHocJoin->BssDescriptor.BeaconPeriod = 
			pBSSIDList->Configuration.BeaconPeriod;

	memcpy(&pAdHocJoin->BssDescriptor.BSSID, 
           		&pBSSIDList->MacAddress,
           		MRVDRV_ETH_ADDR_LEN);

	memcpy(&pAdHocJoin->BssDescriptor.SSID,
           		&pBSSIDList->Ssid.Ssid,
           		pBSSIDList->Ssid.SsidLength);
    
	memcpy(&pAdHocJoin->BssDescriptor.PhyParamSet, 
           		&pBSSIDList->PhyParamSet,
           		sizeof(IEEEtypes_PhyParamSet_t));
    
	memcpy(&pAdHocJoin->BssDescriptor.SsParamSet,
           		&pBSSIDList->SsParamSet,
           		sizeof(IEEEtypes_SsParamSet_t));

	memcpy(&TmpCap, &pBSSIDList->Cap, sizeof(IEEEtypes_CapInfo_t));

	TmpCap &= CAPINFO_MASK;
	
	PRINTM(INFO, "ADHOC_J_CMD: TmpCap=%4X CAPINFO_MASK=%4X\n", TmpCap, CAPINFO_MASK);
	memcpy(&pAdHocJoin->BssDescriptor.Cap, &TmpCap, 
           		sizeof(IEEEtypes_CapInfo_t));

	/* information on BSSID descriptor passed to FW */
	PRINTM(INFO, "ADHOC_J_CMD: BSSID = %2x-%2x-%2x-%2x-%2x-%2x, SSID = %s\n",
			pAdHocJoin->BssDescriptor.BSSID[0],
			pAdHocJoin->BssDescriptor.BSSID[1],
			pAdHocJoin->BssDescriptor.BSSID[2],
			pAdHocJoin->BssDescriptor.BSSID[3],
			pAdHocJoin->BssDescriptor.BSSID[4],
			pAdHocJoin->BssDescriptor.BSSID[5],
			pAdHocJoin->BssDescriptor.SSID);

	PRINTM(INFO, "ADHOC_J_CMD: Data Rate = %x\n", (u32) pAdHocJoin->BssDescriptor.DataRates);

	/* FailTimeOut */
	pAdHocJoin->FailTimeOut = wlan_cpu_to_le16(MRVDRV_ASSOCIATION_TIME_OUT);

	/* ProbeDelay */
	pAdHocJoin->ProbeDelay = wlan_cpu_to_le16(HostCmd_SCAN_PROBE_DELAY_TIME);

	/* Copy Data Rates from the Rates recorded in scan response */
	memset(pAdHocJoin->BssDescriptor.DataRates, 0, 
           		sizeof(pAdHocJoin->BssDescriptor.DataRates));
	memcpy(pAdHocJoin->BssDescriptor.DataRates, pBSSIDList->DataRates, 
           		MIN(sizeof(pAdHocJoin->BssDescriptor.DataRates),
               				sizeof(pBSSIDList->DataRates)));

#ifdef	MULTI_BANDS
	if (pBSSIDList->bss_band == BAND_A) {
		card_rates = SupportedRates_A;
		card_rates_size = sizeof(SupportedRates_A);
	}
	else if (pBSSIDList->bss_band == BAND_B) {
		card_rates = SupportedRates_B;
		card_rates_size = sizeof(SupportedRates_B);
	}
	else if (pBSSIDList->bss_band == BAND_G) {
		card_rates = SupportedRates_G;
		card_rates_size = sizeof(SupportedRates_G);
	}
	else {
		/* should not reach here */
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	Adapter->CurBssParams.band = pBSSIDList->bss_band;
#else
	card_rates = SupportedRates;
	card_rates_size = sizeof(SupportedRates);
#endif	/* MULTI_BANDS*/

	Adapter->CurBssParams.channel = pBSSIDList->Channel;

	if (get_common_rates(Adapter, pAdHocJoin->BssDescriptor.DataRates, 
                         	sizeof(pAdHocJoin->BssDescriptor.DataRates),
                         	card_rates,
			       	card_rates_size)) {
		PRINTM(INFO, "ADHOC_J_CMD: get_common_rates returns error.\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	/* Find the last non zero */
	for (i=0; i < sizeof(pAdHocJoin->BssDescriptor.DataRates) 
             		&& pAdHocJoin->BssDescriptor.DataRates[i]; i++)
       		;
	
	Adapter->CurBssParams.NumOfRates = i;

	/* 
	 * Copy the adhoc joining rates to Current BSS State structure 
	 */
	memcpy(Adapter->CurBssParams.DataRates, 
		pAdHocJoin->BssDescriptor.DataRates,
		Adapter->CurBssParams.NumOfRates);

	pAdHocJoin->BssDescriptor.SsParamSet.IbssParamSet.AtimWindow =
	wlan_cpu_to_le16(pBSSIDList->Configuration.ATIMWindow);

	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled
#ifdef ADHOCAES
		|| Adapter->AdhocAESEnabled
#endif
		) {
		pAdHocJoin->BssDescriptor.Cap.Privacy = AD_HOC_CAP_PRIVACY_ON;
	}

#ifdef PS_REQUIRED
	if (Adapter->PSMode == Wlan802_11PowerModeMAX_PSP) {
		/* wake up first */
		WLAN_802_11_POWER_MODE LocalPSMode;

		LocalPSMode = Wlan802_11PowerModeCAM;
		ret = PrepareAndSendCommand(priv, 
			       	    HostCmd_CMD_802_11_PS_MODE,
                                    HostCmd_ACT_GEN_SET,
                                    0, 0, &LocalPSMode);

		if (ret) {
		  	ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}
#endif

#ifdef MULTI_BANDS
	if (Adapter->is_multiband) {
		bc.BandSelection = Adapter->CurBssParams.band;
		bc.Channel = Adapter->CurBssParams.channel;

		ret = PrepareAndSendCommand(priv,
                                    HostCmd_CMD_802_11_BAND_CONFIG,
                                    HostCmd_ACT_SET, 
                                    HostCmd_OPTION_WAITFORRSP,
                                    0, &bc);

		if (ret) {
	  		ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}
#endif /* MULTI_BANDS */
    
#ifdef ENABLE_802_11D
    	if (wlan_parse_dnld_countryinfo_11d( priv )) {
    		ret = WLAN_STATUS_FAILURE;
		goto done;
    	}
#endif
    
#ifdef ENABLE_802_11H
    	/*
    	 * Call 11h join API after capability bits are set so adhoc/infra 11h 
	 * behavior can be properly triggered.  pos modified if data is appended
    	 */
	cmdAppendSize = wlan_11h_process_join(priv, &pos, 
                                          &pAdHocJoin->BssDescriptor.Cap,
                                          pBSSIDList->Channel,
                                          &pBSSIDList->wlan_11h_bss_info);
#endif

	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_AD_HOC_JOIN) 
                                 + S_DS_GEN + cmdAppendSize);

	memcpy(&TmpCap, &pAdHocJoin->BssDescriptor.Cap, 
           			sizeof(IEEEtypes_CapInfo_t));
	TmpCap = wlan_cpu_to_le16(TmpCap);

	memcpy(&pAdHocJoin->BssDescriptor.Cap,
		       	&TmpCap, 
           		sizeof(IEEEtypes_CapInfo_t));

done:	
	LEAVE();
	return ret;
}

#ifdef WPA
/** 
 *  @brief This function prepares command of enable_rsn.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_enable_rsn(wlan_private * priv,
			   HostCmd_DS_COMMAND * cmd,
			   u16 cmd_action)
{
	HostCmd_DS_802_11_ENABLE_RSN *pEnableRSN = &cmd->params.enbrsn;
	wlan_adapter   *Adapter = priv->adapter;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_ENABLE_RSN);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_ENABLE_RSN) + S_DS_GEN);
	pEnableRSN->Action = wlan_cpu_to_le16(cmd_action);
	if (Adapter->SecInfo.WPAEnabled
#ifdef WPA2
		|| Adapter->SecInfo.WPA2Enabled
#endif
	) {
		pEnableRSN->Enable = wlan_cpu_to_le16(HostCmd_ENABLE_RSN);
	} else {
		pEnableRSN->Enable = wlan_cpu_to_le16(HostCmd_DISABLE_RSN);
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of key_material.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @param cmd_oid	OID: ENABLE or DISABLE
 *  @param pdata_buf    A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_key_material(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	u16 cmd_action,
			       	WLAN_OID cmd_oid,
			       	void *pdata_buf)
{
	HostCmd_DS_802_11_KEY_MATERIAL 	*pKeyMaterial = &cmd->params.keymaterial;
	PWLAN_802_11_KEY		pKey = (PWLAN_802_11_KEY)pdata_buf;
	u16				KeyParamSet_len;
	int				ret = WLAN_STATUS_SUCCESS;

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_KEY_MATERIAL);
	pKeyMaterial->Action = wlan_cpu_to_le16(cmd_action);

	if( cmd_action == HostCmd_ACT_GET ) {
		cmd->Size = wlan_cpu_to_le16(2 + S_DS_GEN);
		ret = WLAN_STATUS_SUCCESS;
		goto done;		
	}

	memset(&pKeyMaterial->KeyParamSet, 0, sizeof(MrvlIEtype_KeyParamSet_t));

	if (pKey->KeyLength == WPA_AES_KEY_LEN) {
		PRINTM(INFO, "WPA_AES\n");
		pKeyMaterial->KeyParamSet.KeyTypeId = wlan_cpu_to_le16(KEY_TYPE_ID_AES);

		if ( cmd_oid == (WLAN_OID)KEY_INFO_ENABLED )
			pKeyMaterial->KeyParamSet.KeyInfo = wlan_cpu_to_le16(KEY_INFO_AES_ENABLED);
		else
			pKeyMaterial->KeyParamSet.KeyInfo = !(wlan_cpu_to_le16(KEY_INFO_AES_ENABLED));

		if (pKey->KeyIndex & 0x40000000)	//AES pairwise key: unicast
			pKeyMaterial->KeyParamSet.KeyInfo |= wlan_cpu_to_le16(KEY_INFO_AES_UNICAST);
		else					//AES group key: multicast
			pKeyMaterial->KeyParamSet.KeyInfo |= wlan_cpu_to_le16(KEY_INFO_AES_MCAST);
	}
	else if (pKey->KeyLength == WPA_TKIP_KEY_LEN) {
		PRINTM(INFO, "WPA_TKIP\n");
		pKeyMaterial->KeyParamSet.KeyTypeId = wlan_cpu_to_le16(KEY_TYPE_ID_TKIP);
		pKeyMaterial->KeyParamSet.KeyInfo = wlan_cpu_to_le16(KEY_INFO_TKIP_ENABLED);

		if (pKey->KeyIndex & 0x40000000)	//TKIP pairwise key: unicast
			pKeyMaterial->KeyParamSet.KeyInfo |= wlan_cpu_to_le16(KEY_INFO_TKIP_UNICAST);
		else					//TKIP group key: multicast
			pKeyMaterial->KeyParamSet.KeyInfo |= wlan_cpu_to_le16(KEY_INFO_TKIP_MCAST);
	}

	if ((pKey->KeyIndex & 0x40000000) == 0)
		priv->adapter->IsGTK_SET = TRUE + 1;

	if (pKeyMaterial->KeyParamSet.KeyTypeId) {
		pKeyMaterial->KeyParamSet.Type = wlan_cpu_to_le16(TLV_TYPE_KEY_MATERIAL);
		pKeyMaterial->KeyParamSet.KeyLen = wlan_cpu_to_le16(pKey->KeyLength);
		memcpy(pKeyMaterial->KeyParamSet.Key,
			pKey->KeyMaterial, pKey->KeyLength);
		pKeyMaterial->KeyParamSet.Length = wlan_cpu_to_le16(pKey->KeyLength + 6);

#define TYPE_LEN_FIELDS_LEN 4
		KeyParamSet_len = pKeyMaterial->KeyParamSet.Length + TYPE_LEN_FIELDS_LEN;
#define ACTION_FIELD_LEN 2
		cmd->Size = wlan_cpu_to_le16(KeyParamSet_len + ACTION_FIELD_LEN + S_DS_GEN);
	}

	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();
	return ret;
}
#endif	/* end of WPA */

/** 
 *  @brief This function prepares command of reset.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action 	the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_reset(wlan_private * priv,
		     	 HostCmd_DS_COMMAND * cmd,
 			 int cmd_action)
{
	HostCmd_DS_802_11_RESET *reset = &cmd->params.reset;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RESET);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_RESET) + S_DS_GEN);
	reset->Action = wlan_cpu_to_le16(cmd_action);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of get_log.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_get_log(wlan_private * priv,
			HostCmd_DS_COMMAND * cmd)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_GET_LOG);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_GET_LOG) + S_DS_GEN);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of authenticate.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_authenticate(wlan_private * priv,
			     HostCmd_DS_COMMAND * cmd)
{
	HostCmd_DS_802_11_AUTHENTICATE 	*pAuthenticate = &cmd->params.auth;
	wlan_adapter   			*Adapter = priv->adapter;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AUTHENTICATE);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_AUTHENTICATE) + S_DS_GEN);

	pAuthenticate->AuthType = Adapter->SecInfo.AuthenticationMode;

	memmove(pAuthenticate->MacAddr,
			Adapter->CurrentBSSID, MRVDRV_ETH_ADDR_LEN);
		
	PRINTM(INFO, "AUTH_CMD: Bssid is : %x:%x:%x:%x:%x:%x\n",
				Adapter->CurrentBSSID[0],
				Adapter->CurrentBSSID[1],
				Adapter->CurrentBSSID[2],
				Adapter->CurrentBSSID[3],
				Adapter->CurrentBSSID[4],
				Adapter->CurrentBSSID[5]);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of get_stat.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_get_stat(wlan_private * priv, 
		HostCmd_DS_COMMAND * cmd)
{
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_GET_STAT);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_GET_STAT) + S_DS_GEN);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of snmp_mib.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param cmd_oid   	the OID of SNMP MIB
 *  @param pdata_buf	the pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_snmp_mib(wlan_private * priv,
			 	HostCmd_DS_COMMAND * cmd,
			       	int cmd_action,
       				int cmd_oid,
			       	void *pdata_buf)
{
	HostCmd_DS_802_11_SNMP_MIB 	*pSNMPMIB = &cmd->params.smib;
	wlan_adapter   			*Adapter = priv->adapter;
	u8              		ucTemp;

	ENTER();

	PRINTM(INFO, "SNMP_CMD: cmd_oid = 0x%x\n", cmd_oid);

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_SNMP_MIB);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_SNMP_MIB) + S_DS_GEN);

	switch (cmd_oid) {
	case OID_802_11_INFRASTRUCTURE_MODE:
		pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
		pSNMPMIB->OID = wlan_cpu_to_le16((u16) DesiredBssType_i);
		pSNMPMIB->BufSize = sizeof(u8);
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure)
			ucTemp = SNMP_MIB_VALUE_INFRA;
		else
			ucTemp = SNMP_MIB_VALUE_ADHOC;

		memmove(pSNMPMIB->Value, &ucTemp, sizeof(u8));

		break;

#ifdef ENABLE_802_11D
	case OID_802_11D_ENABLE:
	{
		u32 		ulTemp;

		pSNMPMIB->OID = wlan_cpu_to_le16((u16) Dot11D_i);

		if (cmd_action == HostCmd_PENDING_ON_SET_OID) {
			pSNMPMIB->QueryType = HostCmd_ACT_GEN_SET;
			pSNMPMIB->BufSize = sizeof(u16);
			ulTemp = *(u32*) pdata_buf;
			*((PUSHORT) (pSNMPMIB->Value)) = 
						wlan_cpu_to_le16((u16) ulTemp);
		}
		break;
	}
#endif

#ifdef ENABLE_802_11H
	case OID_802_11H_ENABLE:
    	{
		u32 		ulTemp;

		pSNMPMIB->OID = wlan_cpu_to_le16((u16) Dot11H_i);

		if (cmd_action == HostCmd_PENDING_ON_SET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			ulTemp = *(u32*)pdata_buf;
			*((PUSHORT) (pSNMPMIB->Value)) = 
						wlan_cpu_to_le16((u16) ulTemp);
		}
		break;
    	}
#endif

	case OID_802_11_FRAGMENTATION_THRESHOLD:
	{
		WLAN_802_11_FRAGMENTATION_THRESHOLD ulTemp;
		u16          usTemp;

		pSNMPMIB->OID = wlan_cpu_to_le16((u16) FragThresh_i);

		if (cmd_action == HostCmd_PENDING_ON_GET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
		} else if (cmd_action == HostCmd_PENDING_ON_SET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			ulTemp = *((WLAN_802_11_FRAGMENTATION_THRESHOLD *)
							pdata_buf);
			*((PUSHORT) (pSNMPMIB->Value)) = 
						wlan_cpu_to_le16((u16) ulTemp);
			
		} else if (cmd_action == HostCmd_PENDING_ON_NONE) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			usTemp = 
				wlan_cpu_to_le16((u16)Adapter->FragThsd);
			memmove(pSNMPMIB->Value, &usTemp, sizeof(u16));
		}

		break;
	}

	case OID_802_11_RTS_THRESHOLD:
	{
		u16          		usTemp;
		WLAN_802_11_RTS_THRESHOLD 	ulTemp;
		pSNMPMIB->OID = wlan_le16_to_cpu((u16)RtsThresh_i);

		if (cmd_action == HostCmd_PENDING_ON_GET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
		} else if (cmd_action == HostCmd_PENDING_ON_SET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			ulTemp =
				*((WLAN_802_11_RTS_THRESHOLD *) 
						pdata_buf);
			*(PUSHORT) (pSNMPMIB->Value) = 
						wlan_cpu_to_le16((u16) ulTemp);
			
		} else if (cmd_action == HostCmd_PENDING_ON_NONE) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			usTemp = 
				wlan_cpu_to_le16((u16) Adapter->RTSThsd);
			memmove(pSNMPMIB->Value, &usTemp, sizeof(u16));
		}
		break;
	}
	case OID_802_11_TX_RETRYCOUNT:
		pSNMPMIB->OID = wlan_cpu_to_le16((u16) ShortRetryLim_i);

		if (cmd_action == HostCmd_PENDING_ON_GET_OID) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
		} else if (cmd_action == HostCmd_PENDING_ON_SET_OID ||
			cmd_action == HostCmd_PENDING_ON_NONE) {
			pSNMPMIB->QueryType = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pSNMPMIB->BufSize = wlan_cpu_to_le16(sizeof(u16));
			*((PUSHORT) (pSNMPMIB->Value)) = 
				wlan_cpu_to_le16((u16)Adapter->TxRetryCount);
		}

		break;
	default:
		break;
	}

	PRINTM(INFO, "SNMP_CMD: Command=0x%x, Size=0x%x, SeqNum=0x%x, Result=0x%x\n",
				cmd->Command, cmd->Size, cmd->SeqNum, cmd->Result);

	PRINTM(INFO, "SNMP_CMD: Action=0x%x, OID=0x%x, OIDSize=0x%x, Value=0x%x\n",
				pSNMPMIB->QueryType, pSNMPMIB->OID, pSNMPMIB->BufSize, 
				*(u16 *) pSNMPMIB->Value);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of radio_control.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_radio_control(wlan_private * priv,
			      HostCmd_DS_COMMAND * cmd,
			      int cmd_action)
{
	wlan_adapter   			*Adapter = priv->adapter;
	HostCmd_DS_802_11_RADIO_CONTROL *pRadioControl = &cmd->params.radio;

	ENTER();

	cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_RADIO_CONTROL)) + S_DS_GEN);
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RADIO_CONTROL);

	pRadioControl->Action = wlan_cpu_to_le16(cmd_action);

	switch (Adapter->Preamble) {
	case HostCmd_TYPE_SHORT_PREAMBLE:
		pRadioControl->Control = wlan_cpu_to_le16(SET_SHORT_PREAMBLE);
		break;

	case HostCmd_TYPE_LONG_PREAMBLE:
		pRadioControl->Control = wlan_cpu_to_le16(SET_LONG_PREAMBLE);
		break;

	case HostCmd_TYPE_AUTO_PREAMBLE:
	default:
		pRadioControl->Control = wlan_cpu_to_le16(SET_AUTO_PREAMBLE);
		break;
	}

	if (Adapter->RadioOn)
		pRadioControl->Control |= wlan_cpu_to_le16(TURN_ON_RF);
	else
		pRadioControl->Control &= wlan_cpu_to_le16(~TURN_ON_RF);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#ifdef BCA
/** 
 *  @brief This function prepares command of bca_timeshare.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param user_bca_ts	A pointer to HostCmd_DS_802_11_BCA_TIMESHARE structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_bca_timeshare(wlan_private * priv,
				HostCmd_DS_COMMAND * cmd,
			       	u16 cmd_action, 
				HostCmd_DS_802_11_BCA_TIMESHARE *user_bca_ts)
{
	wlan_adapter 			*Adapter = priv->adapter;
	HostCmd_DS_802_11_BCA_TIMESHARE *bca_ts = &cmd->params.bca_timeshare;

	ENTER();

	cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_BCA_TIMESHARE)) +
								S_DS_GEN);
	cmd->Command = 
		wlan_cpu_to_le16(HostCmd_CMD_802_11_BCA_CONFIG_TIMESHARE);

	if (cmd_action == HostCmd_ACT_GEN_GET) {
		memset(&Adapter->bca_ts, 0, sizeof(bca_ts));
		memset(bca_ts, 0, sizeof(HostCmd_DS_802_11_BCA_TIMESHARE));
		bca_ts->Action = wlan_cpu_to_le16(cmd_action);
	} else if (cmd_action == HostCmd_ACT_GEN_SET) {
		bca_ts->Action = wlan_cpu_to_le16(cmd_action);
		bca_ts->TrafficType = 
				wlan_cpu_to_le16(user_bca_ts->TrafficType);
		bca_ts->TimeShareInterval = 
			wlan_cpu_to_le32(user_bca_ts->TimeShareInterval);
		bca_ts->BTTime = wlan_cpu_to_le32(user_bca_ts->BTTime);
	} 

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif

/** 
 *  @brief This function prepares command of rf_tx_power.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_rf_tx_power(wlan_private * priv,
			    HostCmd_DS_COMMAND * cmd,
			    u16 cmd_action,
			    void *pdata_buf)
{
	
	HostCmd_DS_802_11_RF_TX_POWER *pRTP = &cmd->params.txp;
	
	ENTER();
	
	cmd->Size = wlan_cpu_to_le16((sizeof(HostCmd_DS_802_11_RF_TX_POWER)) + S_DS_GEN);
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RF_TX_POWER);
	pRTP->Action = cmd_action;
	
	PRINTM(INFO, "RF_TX_POWER_CMD: Size:%d Cmd:0x%x Act:%d\n", cmd->Size, cmd->Command,
				pRTP->Action);

	switch (cmd_action) {
		case HostCmd_ACT_TX_POWER_OPT_GET:
			pRTP->Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
			pRTP->CurrentLevel = 0;
			break;

		case HostCmd_ACT_TX_POWER_OPT_SET_HIGH:
			pRTP->Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pRTP->CurrentLevel =
			   	 wlan_cpu_to_le16(HostCmd_ACT_TX_POWER_INDEX_HIGH);
			break;

		case HostCmd_ACT_TX_POWER_OPT_SET_MID:
			pRTP->Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pRTP->CurrentLevel =
			    	wlan_cpu_to_le16(HostCmd_ACT_TX_POWER_INDEX_MID);
			break;

		case HostCmd_ACT_TX_POWER_OPT_SET_LOW:
			pRTP->Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			pRTP->CurrentLevel = wlan_cpu_to_le16(*((u16 *)pdata_buf));
			break;
	}
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of rf_antenna.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_rf_antenna(wlan_private * priv,
			   HostCmd_DS_COMMAND * cmd,
			   u16 cmd_action,
			   void *pdata_buf)
{
	HostCmd_DS_802_11_RF_ANTENNA *rant = &cmd->params.rant;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RF_ANTENNA);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_RF_ANTENNA) + S_DS_GEN);

	rant->Action = wlan_cpu_to_le16(cmd_action);
	if ((cmd_action == HostCmd_ACT_SET_RX) || 
			(cmd_action == HostCmd_ACT_SET_TX)) {
		rant->AntennaMode =
			wlan_cpu_to_le16((u16) (*(WLAN_802_11_ANTENNA *) pdata_buf));
	}

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of rate_adapt_rateset.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_rate_adapt_rateset(wlan_private *priv, 
				HostCmd_DS_COMMAND *cmd,
			       	u16 cmd_action)
{
	HostCmd_DS_802_11_RATE_ADAPT_RATESET	
				*rateadapt = &cmd->params.rateset;
	wlan_adapter		*Adapter = priv->adapter;

	cmd->Size = 
		wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_RATE_ADAPT_RATESET) + S_DS_GEN);
	cmd->Command =
	       	wlan_cpu_to_le16(HostCmd_CMD_802_11_RATE_ADAPT_RATESET);

	ENTER();
	
	rateadapt->Action = cmd_action;
	rateadapt->EnableHwAuto = Adapter->EnableHwAuto;
	rateadapt->Bitmap = Adapter->RateBitmap;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of rate_adapt_rateset.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_data_rate(wlan_private * priv,
			  HostCmd_DS_COMMAND * cmd,
			  u16 cmd_action)
{
	HostCmd_DS_802_11_DATA_RATE *pDataRate = &cmd->params.drate;
	wlan_adapter   *Adapter = priv->adapter;
	u16		Action	= cmd_action;

	ENTER();

	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_DATA_RATE) + S_DS_GEN);

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_DATA_RATE);

	memset(pDataRate, 0, sizeof(HostCmd_DS_802_11_DATA_RATE));

	pDataRate->Action = wlan_cpu_to_le16(cmd_action);

	if (Action == HostCmd_ACT_SET_TX_FIX_RATE) {
		pDataRate->DataRate[0] = data_rate_to_index(Adapter->DataRate);
		PRINTM(INFO, "Setting FW for fixed rate 0x%02X\n", Adapter->DataRate);
	}
	else if(Action == HostCmd_ACT_SET_TX_AUTO) {
		PRINTM(INFO, "Setting FW for AUTO rate\n");
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function prepares command of mac_multicast_adr.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_mac_multicast_adr(wlan_private * priv,
			   HostCmd_DS_COMMAND * cmd,
			   u16 cmd_action)
{
	HostCmd_DS_MAC_MULTICAST_ADR *pMCastAdr = &cmd->params.madr;
	wlan_adapter   *Adapter = priv->adapter;

	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_MAC_MULTICAST_ADR) + S_DS_GEN);
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_MAC_MULTICAST_ADR);

	pMCastAdr->Action = wlan_cpu_to_le16(cmd_action);
	pMCastAdr->NumOfAdrs = wlan_cpu_to_le16((u16) Adapter->NumOfMulticastMACAddr);
	memcpy(pMCastAdr->MACList, Adapter->MulticastList,
			Adapter->NumOfMulticastMACAddr * ETH_ALEN);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of rf_channel.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_rf_channel(wlan_private * priv,
			   HostCmd_DS_COMMAND * cmd,
			   int option,
			   void *pdata_buf)
{
	HostCmd_DS_802_11_RF_CHANNEL *rfchan = &cmd->params.rfchannel;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RF_CHANNEL);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_RF_CHANNEL)
                                 + S_DS_GEN);

    	if (option == HostCmd_OPT_802_11_RF_CHANNEL_SET) {
		rfchan->CurrentChannel = wlan_cpu_to_le16(*((u16 *)pdata_buf));
	}

    	rfchan->Action = wlan_cpu_to_le16(option);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of rssi.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_rssi(wlan_private * priv, 
			HostCmd_DS_COMMAND * cmd)
{
	wlan_adapter   *Adapter = priv->adapter;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_RSSI);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_RSSI) + S_DS_GEN);
	cmd->params.rssi.N = priv->adapter->bcn_avg_factor;

	/* reset Beacon SNR/NF/RSSI values */
	Adapter->SNR[TYPE_BEACON][TYPE_NOAVG] = 0;
	Adapter->SNR[TYPE_BEACON][TYPE_AVG] = 0;
	Adapter->NF[TYPE_BEACON][TYPE_NOAVG] = 0;
	Adapter->NF[TYPE_BEACON][TYPE_AVG] = 0;
	Adapter->RSSI[TYPE_BEACON][TYPE_NOAVG] = 0;
	Adapter->RSSI[TYPE_BEACON][TYPE_AVG] = 0;

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of reg_access.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_reg_access(wlan_private * priv,
		   	 HostCmd_DS_COMMAND * CmdPtr,
		   	 u8 cmd_action,
		   	 void *pdata_buf)
{
	wlan_offset_value *offval;

	ENTER();

	offval = (wlan_offset_value *) pdata_buf;

	switch (CmdPtr->Command) {
	case HostCmd_CMD_MAC_REG_ACCESS:
		{
		HostCmd_DS_MAC_REG_ACCESS *macreg;

		CmdPtr->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_MAC_REG_ACCESS) + S_DS_GEN);
		macreg = (PHostCmd_DS_MAC_REG_ACCESS) & CmdPtr->params.macreg;

		macreg->Action = wlan_cpu_to_le16(cmd_action);
		macreg->Offset = wlan_cpu_to_le16((u16) offval->offset);
		macreg->Value = wlan_cpu_to_le32(offval->value);

		break;
		}

	case HostCmd_CMD_BBP_REG_ACCESS:
		{
		HostCmd_DS_BBP_REG_ACCESS *bbpreg;

		CmdPtr->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_BBP_REG_ACCESS) + S_DS_GEN);
		bbpreg = (PHostCmd_DS_BBP_REG_ACCESS) & CmdPtr->params.bbpreg;

		bbpreg->Action = wlan_cpu_to_le16(cmd_action);
		bbpreg->Offset = wlan_cpu_to_le16((u16) offval->offset);
		bbpreg->Value = (u8) offval->value;

		break;
		}

	case HostCmd_CMD_RF_REG_ACCESS:
		{
		HostCmd_DS_RF_REG_ACCESS *rfreg;

		CmdPtr->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_RF_REG_ACCESS) + S_DS_GEN);
		rfreg = (PHostCmd_DS_RF_REG_ACCESS) & CmdPtr->params.rfreg;

		rfreg->Action = wlan_cpu_to_le16(cmd_action);
		rfreg->Offset = wlan_cpu_to_le16((u16) offval->offset);
		rfreg->Value = (u8) offval->value;

		break;
		}

	default:
		break;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function prepares command of mac_address.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_mac_address(wlan_private * priv,
		        	HostCmd_DS_COMMAND * cmd,
				u16  cmd_action)
{
	wlan_adapter   *Adapter = priv->adapter;

	cmd->Command	= wlan_cpu_to_le16(HostCmd_CMD_802_11_MAC_ADDRESS);
	cmd->Size	= wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_MAC_ADDRESS) +
				S_DS_GEN);
	cmd->Result 	= 0;

	cmd->params.macadd.Action = wlan_cpu_to_le16(cmd_action);
	
	if (cmd_action == HostCmd_ACT_SET) {
		memcpy(cmd->params.macadd.MacAdd,
					Adapter->CurrentAddr,
					ETH_ALEN);
		HEXDUMP("SET_CMD: MAC ADDRESS-", 
					Adapter->CurrentAddr, 6);
	}
			
	return WLAN_STATUS_SUCCESS;
}

#ifdef MULTI_BANDS
/** 
 *  @brief This function prepares command of band_config.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to dta buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_band_config(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	u16  cmd_action,
		    		void *pdata_buf)
{
	HostCmd_DS_802_11_BAND_CONFIG *bc =
		(pHostCmd_DS_802_11_BAND_CONFIG)pdata_buf;

	ENTER();
	
	cmd->Command	= wlan_cpu_to_le16(HostCmd_CMD_802_11_BAND_CONFIG);
	cmd->Size	= wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_BAND_CONFIG) + S_DS_GEN);
	cmd->Result 	= 0;

	cmd->params.band.Action		= wlan_cpu_to_le16(cmd_action);
	cmd->params.band.Channel	= wlan_cpu_to_le16(bc->Channel);

	/* We cannot assign 'bc->BandSelection' directly to 
	 * 'cmd->params.band.BandSelection', since the value present in 
	 * 'bc->BandSelection' is BAND_A(0x04) or BAND_G(0x02) or BAND_B(0x01).
	 * But the value that has to be filled in 'cmd->params.band.BandSelection'
	 * is 0/1/2.
	 */
	
	if (bc->BandSelection == BAND_A)
		cmd->params.band.BandSelection = wlan_cpu_to_le16(BS_802_11A);
	else if (bc->BandSelection == BAND_G)
		cmd->params.band.BandSelection = wlan_cpu_to_le16(BS_802_11G);
	else 
		cmd->params.band.BandSelection = wlan_cpu_to_le16(BS_802_11B);
	
	HEXDUMP("BNAD_CFG_CMD:", (s8 *) cmd, cmd->Size);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

#endif

#ifdef CAL_DATA
/** 
 *  @brief This function prepares command of cal_data_ext.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_cal_data_ext(wlan_private *priv, 
				HostCmd_DS_COMMAND *cmd,
			       	void *pdata_buf)
{
	HostCmd_DS_802_11_CAL_DATA_EXT  *PCalDataext = pdata_buf; 
	ENTER(); 

	pHostCmd_DS_802_11_CAL_DATA_EXT pCmdCalData = 
		(pHostCmd_DS_802_11_CAL_DATA_EXT) &cmd->params.caldataext;

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_CAL_DATA_EXT);
	
	PRINTM(INFO, "CalDataLen = %d(d)\n",PCalDataext->CalDataLen);

#define MAX_ALLOWED_LEN	1024	
	if(PCalDataext->CalDataLen > MAX_ALLOWED_LEN) {
		PRINTM(MSG, "CAL_DATA_EXT: Cal data lenght too large!\n");
		return WLAN_STATUS_FAILURE;
	}
	
#define ACTION_REV_CALDATA_LEN_FIELDS_LEN 6	
	memcpy(pCmdCalData, PCalDataext,
		       PCalDataext->CalDataLen +
		       ACTION_REV_CALDATA_LEN_FIELDS_LEN + S_DS_GEN);
	
	cmd->Size = wlan_cpu_to_le16(PCalDataext->CalDataLen +
		       		ACTION_REV_CALDATA_LEN_FIELDS_LEN + S_DS_GEN);

	PRINTM(INFO, "CAL_DATA_EXT: cmd->Size = %d(d)\n", cmd->Size);
	
	cmd->Result = 0;

	LEAVE(); 
	return WLAN_STATUS_SUCCESS;
}
#endif /* End of CAL_DATA */

/** 
 *  @brief This function prepares command of eeprom_access.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @param pdata_buf	A pointer to data buffer
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_eeprom_access(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	int  cmd_action,
		    		void *pdata_buf)
{
	wlan_ioctl_regrdwr *ea = pdata_buf;

	ENTER();
	
	cmd->Command	= wlan_cpu_to_le16(HostCmd_CMD_802_11_EEPROM_ACCESS);
	cmd->Size	= wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_EEPROM_ACCESS) + S_DS_GEN);
	cmd->Result 	= 0;

	cmd->params.rdeeprom.Action = wlan_cpu_to_le16(ea -> Action);
	cmd->params.rdeeprom.Offset = wlan_cpu_to_le16(ea -> Offset);
	cmd->params.rdeeprom.ByteCount = wlan_cpu_to_le16(ea -> NOB);
	cmd->params.rdeeprom.Value    = 0;

	return WLAN_STATUS_SUCCESS;
}

#ifdef GSPI
/** 
 *  @brief This function prepares command of eeprom_access.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_gspi_bus_config(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	u16  cmd_action)
{
	pHostCmd_DS_CMD_GSPI_BUS_CONFIG pCmdGspiConfig = 
				(HostCmd_DS_CMD_GSPI_BUS_CONFIG *) &cmd->params.gspicfg;

	cmd->Command	= wlan_cpu_to_le16(HostCmd_CMD_GSPI_BUS_CONFIG);	
	cmd->Size 	= wlan_cpu_to_le16(sizeof(HostCmd_DS_CMD_GSPI_BUS_CONFIG)+ S_DS_GEN);
	cmd->Result 	= 0;

	pCmdGspiConfig -> Action = wlan_cpu_to_le16(cmd_action);
	pCmdGspiConfig -> BusDelayMode = wlan_cpu_to_le16(g_bus_mode_reg);
	pCmdGspiConfig -> HostTimeDelayToReadPort = wlan_cpu_to_le16(g_dummy_clk_ioport * 16);
	pCmdGspiConfig -> HostTimeDelayToReadregister = wlan_cpu_to_le16(g_dummy_clk_reg * 16);

	return WLAN_STATUS_SUCCESS;
}
#endif /* GSPI */

#ifdef ATIMGEN			
/** 
 *  @brief This function prepares command of generate_atim.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd	   	A pointer to HostCmd_DS_COMMAND structure
 *  @param cmd_action   the action: GET or SET
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_802_11_generate_atim(wlan_private *priv,
				HostCmd_DS_COMMAND *cmd,
			       	u16  cmd_action)
{
	PHostCmd_DS_802_11_GENERATE_ATIM pAtim =
		(PHostCmd_DS_802_11_GENERATE_ATIM)&cmd->params.genatim;
    	wlan_adapter  *Adapter = priv->adapter;
	
	ENTER();
	
	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_GENERATE_ATIM);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_GENERATE_ATIM) + S_DS_GEN);
	cmd->Result = 0;
	pAtim->Action = wlan_cpu_to_le16(cmd_action);

	if (Adapter->ATIMEnabled == TRUE) {
		pAtim->Enable = wlan_cpu_to_le16(HostCmd_ENABLE_GENERATE_ATIM);
	} else {
		pAtim->Enable = wlan_cpu_to_le16(HostCmd_DISABLE_GENERATE_ATIM);
	}

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}
#endif

/** 
 *  @brief This function queues the command to cmd list.
 *  
 *  @param Adapter	A pointer to wlan_adapter structure
 *  @param CmdNode   	A pointer to CmdCtrlNode structure
 *  @param addtail	specify if the cmd needs to be queued in the header or tail
 *  @return 	   	n/a
 */
void QueueCmd(wlan_adapter *Adapter, CmdCtrlNode *CmdNode, BOOLEAN addtail)
{
	u32             	flags;
	HostCmd_DS_COMMAND 	*CmdPtr;

	ENTER();
	
 	if (!CmdNode) {
		PRINTM(INFO, "QUEUE_CMD: CmdNode is NULL\n");
		goto done;
    	}
	
	CmdPtr = (HostCmd_DS_COMMAND *)CmdNode->BufVirtualAddr;
 	if (!CmdPtr) {
		PRINTM(INFO, "QUEUE_CMD: CmdPtr is NULL\n");
		goto done;
    	}

#ifdef PS_REQUIRED
	/* Exit_PS command needs to be queued in the header always. */
	if (CmdPtr->Command == HostCmd_CMD_802_11_PS_MODE) {
		HostCmd_DS_802_11_PS_MODE *psm = &CmdPtr->params.psmode;
		if (psm->Action == HostCmd_SubCmd_Exit_PS) {
			if (Adapter->PSState != PS_STATE_FULL_POWER)
				addtail = FALSE;
		}
	}
#endif

#ifdef HOST_SLEEP
	if (CmdPtr->Command == HostCmd_CMD_802_11_HOST_SLEEP_AWAKE_CONFIRM) {
		addtail = FALSE;
	}
#endif

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);

	if (addtail)
		list_add_tail((struct list_head *)CmdNode,
						&Adapter->CmdPendingQ);
	else
		list_add((struct list_head *)CmdNode,
						&Adapter->CmdPendingQ);

	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

	PRINTM(INFO, "QUEUE_CMD: Inserted node=0x%x, cmd=0x%x in CmdPendingQ\n",
		       (u32)CmdNode, 
		       ((HostCmd_DS_GEN *)CmdNode->BufVirtualAddr)->Command);

done:
	LEAVE();
	return;
}

#ifdef MANF_CMD_SUPPORT
/** 
 *  @brief This function sends general command to firmware.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param CmdNode   	A pointer to CmdCtrlNode structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int SendMfgCommand(wlan_private *priv, CmdCtrlNode *cmdnode)
{
	HostCmd_DS_GEN	*pCmdPtr;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
			
	pCmdPtr = (PHostCmd_DS_GEN) Adapter->mfg_cmd;

	pCmdPtr->Command = wlan_cpu_to_le16(HostCmd_CMD_MFG_COMMAND);

	SetCmdCtrlNode(priv, cmdnode, OID_MRVL_MFG_COMMAND,
			HostCmd_OPTION_WAITFORRSP, pCmdPtr);

	/* Assign new sequence number */
	pCmdPtr->SeqNum = wlan_cpu_to_le16(priv->adapter->SeqNum);

	PRINTM(INFO, "Sizeof CmdPtr->size %d\n", (u32) pCmdPtr->Size);

	/* copy the command from information buffer to command queue */
	memcpy((void *) cmdnode->BufVirtualAddr,
		       (void *) pCmdPtr,
		       	pCmdPtr->Size);

	Adapter->mfg_cmd_flag = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif

/*
 * TODO: Fix the issue when DownloadCommandToStation is being called the
 * second time when the command timesout. All the CmdPtr->xxx are in little
 * endian and therefore all the comparissions will fail.
 * For now - we are not performing the endian conversion the second time - but
 * for PS and DEEP_SLEEP we need to worry
 */

/** 
 *  @brief This function downloads the command to firmware.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param CmdNode   	A pointer to CmdCtrlNode structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int DownloadCommandToStation(wlan_private * priv, CmdCtrlNode * CmdNode)
{
	u32             	flags;
	HostCmd_DS_COMMAND 	*CmdPtr;
    	wlan_adapter   		*Adapter = priv->adapter;
	int	             	ret = WLAN_STATUS_SUCCESS;
	u16			CmdSize;
	u16			Command;
	
	OS_INTERRUPT_SAVE_AREA;

	ENTER();

	if (!Adapter || !CmdNode) {
		PRINTM(INFO, "DNLD_CMD: Adapter or CmdNode is NULL\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}

	CmdPtr = (HostCmd_DS_COMMAND *) CmdNode->BufVirtualAddr;
	
 	if (!CmdPtr || !CmdPtr->Size) {
		PRINTM(INFO, "DNLD_CMD: CmdPtr is Null or Cmd Size is Zero, "
							"Not sending\n");
		CleanupAndInsertCmd(priv, CmdNode);
		ret = WLAN_STATUS_FAILURE;
		goto done;
    	}

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
	Adapter->CurCmd = CmdNode;
    	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
	PRINTM(INFO, "DNLD_CMD:: Before download, Size of Cmd = %d\n", CmdPtr->Size);

	CmdSize = CmdPtr->Size;

	Command = wlan_cpu_to_le16(CmdPtr->Command);
	
	CmdNode->CmdWaitQWoken = FALSE;
	CmdSize = wlan_cpu_to_le16(CmdSize);

	ret = sbi_host_to_card(priv, MVMS_CMD, (u8 *) CmdPtr, CmdSize);

	/* clear TxDone interrupt bit */
	OS_INT_DISABLE;
	Adapter->HisRegCpy &= ~HIS_TxDnLdRdy;
	OS_INT_RESTORE;
	
	if (ret != 0) {
		PRINTM(INFO, "DNLD_CMD: Host to Card Failed\n");
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
		ret = WLAN_STATUS_FAILURE;
		goto done;
    	}

    	PRINTM(INFO, "DNLD_CMD: Sent command 0x%x @ %lu\n", Command, os_time_get());
    	HEXDUMP("DNLD_CMD: Command", CmdNode->BufVirtualAddr, CmdSize);

    	/* Setup the timer after transmit command */
       	if (Command == HostCmd_CMD_802_11_SCAN 
			|| Command == HostCmd_CMD_802_11_AUTHENTICATE
			|| Command == HostCmd_CMD_802_11_ASSOCIATE)
		ModTimer(&Adapter->MrvDrvCommandTimer, MRVDRV_TIMER_10S);
	else
		ModTimer(&Adapter->MrvDrvCommandTimer, MRVDRV_TIMER_5S);

	Adapter->CommandTimerIsSet = TRUE;

#ifdef DEEP_SLEEP
	if (Command == HostCmd_CMD_802_11_DEEP_SLEEP) {
		priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
		OS_INT_DISABLE;
		Adapter->HisRegCpy |= HIS_TxDnLdRdy;
		OS_INT_RESTORE;
		if(Adapter->IntCounter || Adapter->CurrentTxSkb)
			PRINTM(INFO, "DNLD_CMD: DS- IntCounter=%d CurrentTxSkb=%p\n",
				Adapter->IntCounter,Adapter->CurrentTxSkb);

		if(Adapter->IntCounter) {
			OS_INT_DISABLE;
			Adapter->IntCounterSaved = Adapter->IntCounter;
			Adapter->IntCounter = 0;
			OS_INT_RESTORE;
		}
#ifdef linux
		if (Adapter->CurrentTxSkb) {
			kfree_skb(Adapter->CurrentTxSkb);
			OS_INT_DISABLE;
			Adapter->CurrentTxSkb = NULL;
			OS_INT_RESTORE;
			priv->stats.tx_dropped++;
		}
#endif
		/* 1. change the PS state to DEEP_SLEEP
		 * 2. since there is no response for this command, so 
		 *    delete the command timer and free the Node. */

		Adapter->IsDeepSleep = TRUE;

		CleanupAndInsertCmd(priv, CmdNode);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

		if (Adapter->CommandTimerIsSet) {
			CancelTimer(&Adapter->MrvDrvCommandTimer);
			Adapter->CommandTimerIsSet = FALSE;
		}

#ifdef BULVERDE_SDIO
		/* stop clock to save more power */
		stop_bus_clock_2(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);
#endif

#ifdef HOST_SLEEP
		if (Adapter->bHostSleepConfigured) {
#if WIRELESS_EXT > 14
			send_iwevcustom_event(priv, "HWM_CFG_DONE.indication ");
			PRINTM(INFO, "DNLD_CMD:: DS- HWM_CFG_DONE event sent\n");
#endif //WIRELESS_EXT
		}
#endif //HOST_SLEEP

	}
#endif // DEEP_SLEEP 

	ret = WLAN_STATUS_SUCCESS;

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function prepares command of mac_control.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd		A pointer to HostCmd_DS_COMMAND structure
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_cmd_mac_control(wlan_private * priv, 
				HostCmd_DS_COMMAND * cmd)
{
	HostCmd_DS_MAC_CONTROL *mac = &cmd->params.macctrl;

	ENTER();

	cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_MAC_CONTROL);
	cmd->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_MAC_CONTROL) + S_DS_GEN);
	mac->Action = wlan_cpu_to_le16(priv->adapter->CurrentPacketFilter);

	PRINTM(INFO, "wlan_cmd_mac_control(): Action=0x%X Size=%d\n",
					mac->Action,cmd->Size);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function inserts command node to CmdFreeQ
 *  after cleans it.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param pTempCmd	A pointer to CmdCtrlNode structure
 *  @return 		n/a
 */
void CleanupAndInsertCmd(wlan_private * priv, CmdCtrlNode * pTempCmd)
{
	u32		flags;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (!pTempCmd)
		goto done;

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
	CleanUpCmdCtrlNode(pTempCmd);
        list_add_tail((struct list_head *)pTempCmd, &Adapter->CmdFreeQ);
	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

done:
	LEAVE();
}

/** 
 *  @brief This function sets radio control.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int SetRadioControl(wlan_private *priv)
{
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	ret = PrepareAndSendCommand(priv,
		       		HostCmd_CMD_802_11_RADIO_CONTROL,
				HostCmd_ACT_GEN_SET,
			       	HostCmd_OPTION_WAITFORRSP, 
				0, NULL);

	PRINTM(INFO, "RADIO_SET: on or off: 0x%X, Preamble = 0x%X\n", 
			priv->adapter->RadioOn,	priv->adapter->Preamble);

	LEAVE();
	return ret;
}

/** 
 *  @brief This function sets packet filter.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int SetMacPacketFilter(wlan_private * priv)
{
	int	ret = WLAN_STATUS_SUCCESS;

	ENTER();

	PRINTM(INFO, "SetMacPacketFilter Value = %x\n",
           		priv->adapter->CurrentPacketFilter);

	/* Send MAC control command to station */
	ret = PrepareAndSendCommand(priv,
		       		HostCmd_CMD_MAC_CONTROL,
			       	0, 0, 0, NULL);

	LEAVE();
	return ret;
}

/** 
 *  @brief This function prepare the command before send to firmware.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param cmd_no	command number
 *  @param cmd_action	command action: GET or SET
 *  @param wait_option	wait option: wait response or not
 *  @param cmd_oid	cmd oid: treated as sub command
 *  @param pdata_buf	A pointer to informaion buffer
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int PrepareAndSendCommand(wlan_private * priv,
	       		u16 cmd_no,
		       	u16 cmd_action,
		      	u16 wait_option,
		       	WLAN_OID cmd_oid,
		      	void *pdata_buf)
{
	int ret = WLAN_STATUS_SUCCESS;
	wlan_adapter   *Adapter = priv->adapter;
	CmdCtrlNode    *CmdNode;
	HostCmd_DS_COMMAND *CmdPtr;

	ENTER();

	if (!Adapter) {
		PRINTM(INFO, "PREP_CMD: Adapter is Null\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;		
	}

#ifdef DEEP_SLEEP
	if (Adapter->IsDeepSleep == TRUE) {
		PRINTM(INFO, "PREP_CMD: Deep sleep enabled\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif

	if (Adapter->SurpriseRemoved) {
		PRINTM(INFO, "PREP_CMD: Card is Removed\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	
	CmdNode = GetFreeCmdCtrlNode(priv);

	if (CmdNode == NULL) {
		PRINTM(MSG, "PREP_CMD: No free CmdNode\n");
		
		/* Wake up main thread to execute next command */
		wake_up_interruptible(&priv->MainThread.waitQ);
		ret = WLAN_STATUS_FAILURE;
		goto done;
    	}
	
#if 0	
	/* should not reach here */
	if (CmdNode->retcode) {
		PRINTM(INFO, "PREP_CMD: Command retcode not yet delivered\n");
		CleanupAndInsertCmd(priv, CmdNode);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif
	SetCmdCtrlNode(priv, CmdNode, cmd_oid, wait_option, pdata_buf);

	CmdPtr = (HostCmd_DS_COMMAND *) CmdNode->BufVirtualAddr;

	PRINTM(INFO, "PREP_CMD: Val of Cmd ptr =0x%x, command=0x%X\n", (u32) CmdPtr,cmd_no);

	if (!CmdPtr) {
		PRINTM(MSG, "PREP_CMD: BufVirtualAddr of CmdNode is NULL\n");
		CleanupAndInsertCmd(priv, CmdNode);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	/* Set sequence number, command and INT option */
	Adapter->SeqNum++;
	CmdPtr->SeqNum = wlan_cpu_to_le16(Adapter->SeqNum);

	CmdPtr->Command = cmd_no;
	CmdPtr->Result = 0; 

	TX_EVENT_FLAGS_SET(&CmdNode->cmdwait_q, 0, TX_AND);
	switch (cmd_no) {
		case HostCmd_CMD_GET_HW_SPEC:
			ret = wlan_cmd_hw_spec(priv, CmdPtr);
			break;
#ifdef PS_REQUIRED
		case HostCmd_CMD_802_11_PS_MODE:
			ret = wlan_cmd_802_11_ps_mode(priv, CmdPtr, cmd_action);
			break;
#endif

		case HostCmd_CMD_802_11_SCAN:
			ret = wlan_cmd_802_11_scan(priv, CmdPtr);
			break;

		case HostCmd_CMD_MAC_CONTROL:
			ret = wlan_cmd_mac_control(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_ASSOCIATE:
		case HostCmd_CMD_802_11_REASSOCIATE:
			ret = wlan_cmd_802_11_associate(priv, CmdPtr, pdata_buf);
			break;

		case HostCmd_CMD_802_11_DEAUTHENTICATE:
			ret = wlan_cmd_802_11_deauthenticate(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_SET_WEP:
			ret = wlan_cmd_802_11_set_wep(priv, CmdPtr, cmd_oid);
			break;

    		case HostCmd_CMD_802_11_AD_HOC_START:
			ret = wlan_cmd_802_11_ad_hoc_start(priv, CmdPtr,
				       		pdata_buf);
			break;
		case HostCmd_CMD_CODE_DNLD:
			break;

		case HostCmd_CMD_802_11_RESET:
			ret = wlan_cmd_802_11_reset(priv, CmdPtr,
				       		cmd_action);
			break;

		case HostCmd_CMD_802_11_GET_LOG:
			ret = wlan_cmd_802_11_get_log(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_AUTHENTICATE:
			ret = wlan_cmd_802_11_authenticate(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_GET_STAT:
			ret = wlan_cmd_802_11_get_stat(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_SNMP_MIB:
			ret = wlan_cmd_802_11_snmp_mib(priv, CmdPtr, 
						cmd_action, cmd_oid, pdata_buf);
			break;

		case HostCmd_CMD_MAC_REG_ACCESS:	
		case HostCmd_CMD_BBP_REG_ACCESS:	
		case HostCmd_CMD_RF_REG_ACCESS:	
			ret = wlan_cmd_reg_access(priv, CmdPtr, cmd_action, 
					pdata_buf);
			break;

		case HostCmd_CMD_802_11_RF_CHANNEL:
			ret = wlan_cmd_802_11_rf_channel(priv, CmdPtr,
					cmd_action, pdata_buf);
			break;

		case HostCmd_CMD_802_11_RF_TX_POWER:
			ret = wlan_cmd_802_11_rf_tx_power(priv, CmdPtr, 
					cmd_action, pdata_buf);
			break;

		case HostCmd_CMD_802_11_RADIO_CONTROL:
			ret = wlan_cmd_802_11_radio_control(priv, CmdPtr, 
					cmd_action);
			break;

		case HostCmd_CMD_802_11_RF_ANTENNA:
			ret = wlan_cmd_802_11_rf_antenna(priv, CmdPtr, 
					cmd_action, pdata_buf);
			break;

		case HostCmd_CMD_802_11_DATA_RATE:	
			ret = wlan_cmd_802_11_data_rate(priv, CmdPtr, 
					cmd_action);
			break;
		case HostCmd_CMD_802_11_RATE_ADAPT_RATESET:
			ret = wlan_cmd_802_11_rate_adapt_rateset(priv,
				       CmdPtr, cmd_action);
			break;

		case HostCmd_CMD_MAC_MULTICAST_ADR:
			ret = wlan_cmd_mac_multicast_adr(priv, CmdPtr, 
					cmd_action);
			break;

		case HostCmd_CMD_802_11_AD_HOC_JOIN:
			ret = wlan_cmd_802_11_ad_hoc_join(priv, CmdPtr, 
						pdata_buf);
			break;

		case HostCmd_CMD_802_11_RSSI:
			ret = wlan_cmd_802_11_rssi(priv, CmdPtr);
			break;

		case HostCmd_CMD_802_11_AD_HOC_STOP:	
			ret = wlan_cmd_802_11_ad_hoc_stop(priv, CmdPtr);
			break;

#ifdef WPA
		case HostCmd_CMD_802_11_ENABLE_RSN:
			ret = wlan_cmd_802_11_enable_rsn(priv, CmdPtr, 
					cmd_action);
			break;


		case HostCmd_CMD_802_11_KEY_MATERIAL:
			ret = wlan_cmd_802_11_key_material(priv, CmdPtr,
					cmd_action, cmd_oid, pdata_buf);
			break;

		case HostCmd_CMD_802_11_PAIRWISE_TSC:
			break;
		case HostCmd_CMD_802_11_GROUP_TSC:
			break;
#endif /* end of WPA */

		case HostCmd_CMD_802_11_MAC_ADDRESS:
			ret = wlan_cmd_802_11_mac_address(priv,
						CmdPtr, cmd_action);
			break;
#ifdef CAL_DATA
		case HostCmd_CMD_802_11_CAL_DATA_EXT:
			ret = wlan_cmd_802_11_cal_data_ext(priv, CmdPtr,
				       		pdata_buf);
			break;
#endif /* CAL_DATA */

#ifdef DEEP_SLEEP
		case HostCmd_CMD_802_11_DEEP_SLEEP:
			CmdPtr->Size = wlan_cpu_to_le16((u16)
					(sizeof(HostCmd_DS_802_11_DEEP_SLEEP)));
			break;
#endif // DEEP_SLEEP

#ifdef HOST_SLEEP
		case HostCmd_CMD_802_11_HOST_SLEEP_CFG:	
			ret = wlan_cmd_802_11_host_sleep_cfg(priv, CmdPtr,
							pdata_buf);
			break;
		case HostCmd_CMD_802_11_HOST_SLEEP_AWAKE_CONFIRM:	
			ret = wlan_cmd_802_11_host_sleep_awake_confirm(priv, CmdPtr);
			break;
#endif
#ifdef MULTI_BANDS
		case HostCmd_CMD_802_11_BAND_CONFIG:
			ret = wlan_cmd_802_11_band_config(priv, CmdPtr, 
						cmd_action, pdata_buf);
			break;
#endif
		case HostCmd_CMD_802_11_EEPROM_ACCESS:
			ret = wlan_cmd_802_11_eeprom_access(priv, CmdPtr,
						cmd_action, pdata_buf);
			break;
#ifdef GSPI
		case HostCmd_CMD_GSPI_BUS_CONFIG:
			ret = wlan_cmd_gspi_bus_config(priv, CmdPtr,
						cmd_action);
			break;
#endif /* GSPI */

#ifdef ATIMGEN			
		case HostCmd_CMD_802_11_GENERATE_ATIM:
			ret = wlan_cmd_802_11_generate_atim(priv, CmdPtr,
						cmd_action);
			break;
#endif

#ifdef MANF_CMD_SUPPORT
		case HostCmd_CMD_MFG_COMMAND:
			ret = SendMfgCommand(priv, CmdNode);
			break;
#endif

#ifdef AUTO_FREQ_CTRL
		case HostCmd_CMD_802_11_SET_AFC:
		case HostCmd_CMD_802_11_GET_AFC:

                  CmdPtr->Command = wlan_cpu_to_le16(cmd_no);
                  CmdPtr->Size    = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_AFC) + S_DS_GEN);

                  memmove(&CmdPtr->params.afc, 
	                    pdata_buf, 
        	            sizeof(HostCmd_DS_802_11_AFC));

                ret = WLAN_STATUS_SUCCESS;
		goto done;
#endif

#ifdef ENABLE_802_11D
		case HostCmd_CMD_802_11D_DOMAIN_INFO:
			ret = wlan_cmd_802_11d_domain_info(priv, CmdPtr,
				       		cmd_no, cmd_action );
			break;
#endif		


#ifdef ENABLE_802_11H
		case HostCmd_CMD_802_11H_TPC_REQUEST:
		case HostCmd_CMD_802_11H_TPC_INFO:
		case HostCmd_CMD_802_11H_CHAN_SW_ANN:
            		ret = wlan_11h_cmd_process(priv, CmdPtr, pdata_buf);
            		break;
#endif
            
#ifdef ENABLE_MEAS
		case HostCmd_CMD_MEASUREMENT_REQUEST:
		case HostCmd_CMD_MEASUREMENT_REPORT:			
	            	ret = wlan_meas_cmd_process(priv, CmdPtr, pdata_buf);
        		break;
#endif  

		case HostCmd_CMD_802_11_SLEEP_PARAMS:
			ret = wlan_cmd_802_11_sleep_params(priv, CmdPtr, 
							cmd_action);
			break;
#ifdef BCA		
		case HostCmd_CMD_802_11_BCA_CONFIG_TIMESHARE:
			ret = wlan_cmd_802_11_bca_timeshare(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
#endif
		case HostCmd_CMD_802_11_INACTIVITY_TIMEOUT:
			ret = wlan_cmd_802_11_inactivity_timeout(priv, CmdPtr,
							cmd_action, pdata_buf);
			SetCmdCtrlNode(priv, CmdNode, 0, 0, pdata_buf);
			break;
#ifdef BG_SCAN
		case HostCmd_CMD_802_11_BG_SCAN_CONFIG:
			ret = wlan_cmd_802_11_bg_scan_config(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;

		case HostCmd_CMD_802_11_BG_SCAN_QUERY:
			ret = wlan_cmd_802_11_bg_scan_query(priv, CmdPtr);
			break;
#endif /* BG_SCAN */

#ifdef PS_REQUIRED
#ifdef FW_WAKEUP_METHOD
		case HostCmd_CMD_802_11_FW_WAKEUP_METHOD:
			ret = wlan_cmd_802_11_fw_wakeup_method(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
#endif
#endif

#ifdef SUBSCRIBE_EVENT_CTRL
		case HostCmd_CMD_802_11_SUBSCRIBE_EVENT:
			ret = wlan_cmd_802_11_subscribe_events(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
#endif
			
#ifdef WMM
		case HostCmd_CMD_802_11_WMM_GET_TSPEC:
		case HostCmd_CMD_802_11_WMM_ADD_TSPEC:
		case HostCmd_CMD_802_11_WMM_REMOVE_TSPEC:
			ret = wlan_cmd_802_11_wmm_tspec(priv, CmdPtr,
							cmd_no, pdata_buf);
			break;
		case HostCmd_CMD_802_11_WMM_ACK_POLICY:
			ret = wlan_cmd_802_11_wmm_ack_policy(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
		case HostCmd_CMD_802_11_WMM_GET_STATUS:
			ret = wlan_cmd_802_11_wmm_get_status(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
		case HostCmd_CMD_802_11_WMM_PRIO_PKT_AVAIL:
			ret = wlan_cmd_802_11_wmm_prio_pkt_avail(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
#endif /* WMM */

                case HostCmd_CMD_802_11_TPC_CFG:

                	CmdPtr->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_TPC_CFG);
	                CmdPtr->Size    = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_TPC_CFG) + S_DS_GEN);

        	        memmove(&CmdPtr->params.tpccfg, 
	                    pdata_buf, sizeof(HostCmd_DS_802_11_TPC_CFG));

               		ret = WLAN_STATUS_SUCCESS;
			break;
#ifdef  LED_GPIO_CTRL
                case HostCmd_CMD_802_11_LED_GPIO_CTRL:
                {
                	MrvlIEtypes_LedGpio_t *gpio = (MrvlIEtypes_LedGpio_t *)CmdPtr->params.ledgpio.data;

                  	memmove(&CmdPtr->params.ledgpio, 
	                    	pdata_buf,
        	            	sizeof(HostCmd_DS_802_11_LED_CTRL));

                  	CmdPtr->Command = 
                    		wlan_cpu_to_le16(HostCmd_CMD_802_11_LED_GPIO_CTRL);

#define ACTION_NUMLED_TLVTYPE_LEN_FIELDS_LEN 8 
                  	CmdPtr->Size =
                    		wlan_cpu_to_le16(gpio->Header.Len + S_DS_GEN
					       + ACTION_NUMLED_TLVTYPE_LEN_FIELDS_LEN); 
                  	gpio->Header.Len = wlan_cpu_to_le16(gpio->Header.Len);
                                                    
                  	ret = WLAN_STATUS_SUCCESS;
		  	break;
                }
#endif /* LED_GPIO_CTRL */
                case HostCmd_CMD_802_11_PWR_CFG:

                 	CmdPtr->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_PWR_CFG);
                  	CmdPtr->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_PWR_CFG) + S_DS_GEN);

                  	memmove(&CmdPtr->params.pwrcfg, 
                   		pdata_buf, 
                    		sizeof(HostCmd_DS_802_11_PWR_CFG));

               	  	ret = WLAN_STATUS_SUCCESS;
		  	break;
		case HostCmd_CMD_802_11_SLEEP_PERIOD:
			ret = wlan_cmd_802_11_sleep_period(priv, CmdPtr,
							cmd_action, pdata_buf);
			break;
#ifdef CIPHER_TEST
		case HostCmd_CMD_802_11_KEY_ENCRYPT:
			CmdPtr->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_KEY_ENCRYPT);
			CmdPtr->Size = wlan_cpu_to_le16(sizeof(HostCmd_DS_802_11_KEY_ENCRYPT) + S_DS_GEN);
			memmove(
				&CmdPtr->params.key_encrypt,
				pdata_buf,
				sizeof(HostCmd_DS_802_11_KEY_ENCRYPT));
		  	ret = WLAN_STATUS_SUCCESS;
			break;
#endif

		default:
			PRINTM(INFO, "PREP_CMD: unknown command- %#x\n", cmd_no);
		   	ret = WLAN_STATUS_FAILURE;
		  	break;
	}

	/* return error, since the command preparation failed */
	if (ret != WLAN_STATUS_SUCCESS) {
		PRINTM(INFO, "PREP_CMD: Command preparation failed\n");
		CleanupAndInsertCmd(priv, CmdNode);
	 	ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	
	CmdNode->CmdWaitQWoken = FALSE;
	
	QueueCmd(Adapter, CmdNode, TRUE);
	wake_up_interruptible(&priv->MainThread.waitQ);

	sbi_reenable_host_interrupt(priv, 0x00);

	if (wait_option & HostCmd_OPTION_WAITFORRSP) {
		PRINTM(INFO, "PREP_CMD: Wait for CMD response\n");
		wait_event_interruptible(CmdNode->cmdwait_q, 
						CmdNode->CmdWaitQWoken);
	}
		
	if (CmdNode->retcode)
		PRINTM(INFO, "PREP_CMD: Command failed with retcode=%d\n", ret);

	/* Now reset the retcode to zero as we are done with it */
	CmdNode->retcode = 0;

done:	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function allocates the command buffer and link
 *  it to command free queue.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int AllocateCmdBuffer(wlan_private * priv)
{
 	int             ret = WLAN_STATUS_SUCCESS;
 	u32             ulBufSize;
 	u32             i;
 	CmdCtrlNode    *TempCmdArray;
 	u8             *pTempVirtualAddr;
 	wlan_adapter   *Adapter = priv->adapter;

 	ENTER();

 	/* Allocate and initialize CmdCtrlNode */
 	INIT_LIST_HEAD(&Adapter->CmdFreeQ);
        INIT_LIST_HEAD(&Adapter->CmdPendingQ);

 	ulBufSize = sizeof(CmdCtrlNode) * MRVDRV_NUM_OF_CMD_BUFFER;
	
 	if (!(TempCmdArray = kmalloc(ulBufSize, GFP_KERNEL))) {
		PRINTM(INFO, "ALLOC_CMD_BUF: Failed to allocate TempCmdArray\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	Adapter->CmdArray = TempCmdArray;
	memset(Adapter->CmdArray, 0, ulBufSize);

	/* Allocate and initialize command buffers */
	ulBufSize = MRVDRV_SIZE_OF_CMD_BUFFER;
	for (i = 0; i < MRVDRV_NUM_OF_CMD_BUFFER; i++) {
		if (!(pTempVirtualAddr = kmalloc(ulBufSize, GFP_KERNEL))) {
			PRINTM(INFO, "ALLOC_CMD_BUF: pTempVirtualAddr: out of memory\n");
				ret = WLAN_STATUS_FAILURE;
			goto done;
		}

		memset(pTempVirtualAddr, 0, ulBufSize);

		/* Update command buffer virtual */
		TempCmdArray[i].BufVirtualAddr = pTempVirtualAddr;
	}

	for (i = 0; i < MRVDRV_NUM_OF_CMD_BUFFER; i++) {
		init_waitqueue_head(&TempCmdArray[i].cmdwait_q);
                CleanupAndInsertCmd(priv, &TempCmdArray[i]);
    	}
 	
	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function frees the command buffer.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int FreeCmdBuffer(wlan_private * priv)
{
    	u32           ulBufSize;
    	UINT            i;
    	CmdCtrlNode    *TempCmdArray;
    	wlan_adapter   *Adapter = priv->adapter;

    	ENTER();

    	/* need to check if cmd array is allocated or not */
    	if (Adapter->CmdArray == NULL) {
		PRINTM(INFO, "FREE_CMD_BUF: CmdArray is Null\n");
		goto done;
	}

	TempCmdArray = Adapter->CmdArray;

	/* Release shared memory buffers */
	ulBufSize = MRVDRV_SIZE_OF_CMD_BUFFER;
	for (i = 0; i < MRVDRV_NUM_OF_CMD_BUFFER; i++) {
		if (TempCmdArray[i].BufVirtualAddr) {
			PRINTM(INFO, "Free all the array\n");
			kfree(TempCmdArray[i].BufVirtualAddr);
			TempCmdArray[i].BufVirtualAddr = NULL;
		}
	}

	/* Release CmdCtrlNode */
	if (Adapter->CmdArray) {
		PRINTM(INFO, "Free CmdArray\n");
		kfree(Adapter->CmdArray);
		Adapter->CmdArray = NULL;
	}

done:
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function gets a free command node if available in
 *  command free queue.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @return CmdCtrlNode A pointer to CmdCtrlNode structure or NULL
 */
CmdCtrlNode    *GetFreeCmdCtrlNode(wlan_private * priv)
{
    	CmdCtrlNode	*TempNode;
	wlan_adapter	*Adapter = priv->adapter;
	u32		flags;

    	ENTER();

    	if (!Adapter)
		return NULL;

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
	
	if (!list_empty(&Adapter->CmdFreeQ)) {
		TempNode = (CmdCtrlNode *)Adapter->CmdFreeQ.next;
		list_del((struct list_head *)TempNode);
	} else {
		PRINTM(INFO, "GET_CMD_NODE: CmdCtrlNode is not available\n");
		TempNode = NULL;
	}

	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

	if (TempNode) {
		PRINTM(INFO, "GET_CMD_NODE: CmdCtrlNode available\n");
		PRINTM(INFO, "GET_CMD_NODE: CmdCtrlNode Address = %p\n", TempNode);
		CleanUpCmdCtrlNode(TempNode);
	}
	
	LEAVE();
	return TempNode;
}

/** 
 *  @brief This function cleans command node.
 *  
 *  @param pTempNode	A pointer to CmdCtrlNode structure
 *  @return 		n/a
 */
void CleanUpCmdCtrlNode(CmdCtrlNode * pTempNode)
{
	ENTER();

	if (!pTempNode)
		return;
	pTempNode->CmdWaitQWoken = TRUE;
	wake_up_interruptible(&pTempNode->cmdwait_q);
	pTempNode->Status = 0;
	pTempNode->cmd_oid = (WLAN_OID) 0;
	pTempNode->wait_option = 0;
	pTempNode->pdata_buf = NULL;
	pTempNode->retcode = 0;

	if (pTempNode->BufVirtualAddr != NULL)
		memset(pTempNode->BufVirtualAddr, 0, MRVDRV_SIZE_OF_CMD_BUFFER);

	LEAVE();
	return;
}

/** 
 *  @brief This function initializes the command node.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param pTempNode	A pointer to CmdCtrlNode structure
 *  @param cmd_oid	cmd oid: treated as sub command
 *  @param wait_option	wait option: wait response or not
 *  @param pdata_buf	A pointer to informaion buffer
 *  @return 		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
void SetCmdCtrlNode(wlan_private * priv,
	       CmdCtrlNode * pTempNode,
	       WLAN_OID cmd_oid,
	       u16 wait_option, 
	       void *pdata_buf)
{
	ENTER();

	if (!pTempNode)
		return;

	pTempNode->cmd_oid = cmd_oid;
	pTempNode->wait_option = wait_option;
	pTempNode->pdata_buf = pdata_buf;

	LEAVE();
}

/** 
 *  @brief This function executes next command in command
 *  pending queue. It will put fimware back to PS mode
 *  if applicable.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int ExecuteNextCommand(wlan_private * priv)
{
	wlan_adapter   		*Adapter = priv->adapter;
	CmdCtrlNode		*CmdNode = NULL;
	HostCmd_DS_COMMAND	*CmdPtr;
	u32			flags;
	int 			ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (!Adapter) {
		PRINTM(MSG, "EXEC_NEXT_CMD: Adapter is NULL\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}
	
#ifdef DEEP_SLEEP
	if (Adapter->IsDeepSleep == TRUE) {
		PRINTM(MSG, "EXEC_NEXT_CMD: Device is in deep sleep mode.\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
#endif /* DEEP_SLEEP */

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);

	if (Adapter->CurCmd) {
		PRINTM(MSG, "EXEC_NEXT_CMD: there is command in processing!\n");
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	if (!list_empty(&Adapter->CmdPendingQ)) {
		CmdNode = (CmdCtrlNode *)
				Adapter->CmdPendingQ.next;
	}

	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
	
	if (CmdNode) {
		PRINTM(INFO, "EXEC_NEXT_CMD: Got next command from CmdPendingQ\n");
		CmdPtr = (HostCmd_DS_COMMAND *) CmdNode->BufVirtualAddr;

#ifdef	PS_REQUIRED
		if (Is_Command_Allowed_In_PS(CmdPtr->Command)) {
			if ((Adapter->PSState == PS_STATE_SLEEP)
				|| (Adapter->PSState == PS_STATE_PRE_SLEEP)
			) {
				PRINTM(INFO, "EXEC_NEXT_CMD: Cannot send cmd 0x%x in PSState %d\n",
					CmdPtr->Command, Adapter->PSState);
				ret = WLAN_STATUS_FAILURE;
				goto done;
			}
			PRINTM(INFO, "EXEC_NEXT_CMD: OK to send command "
					"0x%x in PSState %d\n", 
					CmdPtr->Command,Adapter->PSState);
		} else if (Adapter->PSState != PS_STATE_FULL_POWER) {
			/*
			 * 1. Non-PS command: 
			 * Queue it. set NeedToWakeup to TRUE if current state 
			 * is SLEEP, otherwise call PSWakeup to send Exit_PS.
			 * 2. PS command but not Exit_PS: 
			 * Ignore it.
			 * 3. PS command Exit_PS:
			 * Set NeedToWakeup to TRUE if current state is SLEEP, 
			 * otherwise send this command down to firmware
			 * immediately.
			 */
			if (CmdPtr->Command != wlan_cpu_to_le16(HostCmd_CMD_802_11_PS_MODE)) {				
				/*  Prepare to send Exit PS,
				 *  this non PS command will be sent later */
				if ((Adapter->PSState == PS_STATE_SLEEP)
					|| (Adapter->PSState == PS_STATE_PRE_SLEEP)
				) {
					/* w/ new scheme, it will not reach here.
						since it is blocked in main_thread. */
					Adapter->NeedToWakeup = TRUE;
				}
				else
					PSWakeup(priv, 0);
				
				ret = WLAN_STATUS_SUCCESS;
				goto done;
			} else {
				/*
				 * PS command. Ignore it if it is not Exit_PS. 
				 * otherwise send it down immediately.
				 */
				HostCmd_DS_802_11_PS_MODE *psm = 
					&CmdPtr->params.psmode;

				PRINTM(INFO, "EXEC_NEXT_CMD: PS cmd- Action=0x%x\n",psm->Action);
				if (psm->Action != wlan_cpu_to_le16(HostCmd_SubCmd_Exit_PS)) {
					PRINTM(INFO, "EXEC_NEXT_CMD: Ignore Enter PS cmd\n");
					list_del((struct list_head *) CmdNode);
					CleanupAndInsertCmd(priv,CmdNode);
					
					ret = WLAN_STATUS_SUCCESS;
					goto done;
				}
			
				if ((Adapter->PSState == PS_STATE_SLEEP)
					|| (Adapter->PSState == PS_STATE_PRE_SLEEP)
				) {
					PRINTM(INFO, "EXEC_NEXT_CMD: Ignore ExitPS cmd in sleep\n");
					list_del((struct list_head *) CmdNode);
					CleanupAndInsertCmd(priv,CmdNode);
					Adapter->NeedToWakeup = TRUE; 

					ret = WLAN_STATUS_SUCCESS;
					goto done;
				}

				PRINTM(INFO, "EXEC_NEXT_CMD: Sending Exit_PS down...\n");
			} 
		}
#endif
		list_del((struct list_head *) CmdNode);
		PRINTM(INFO, "EXEC_NEXT_CMD: Sending 0x%04X Command\n", CmdPtr->Command);
		DownloadCommandToStation(priv, CmdNode);
	}
#ifdef	PS_REQUIRED
	else {
		/*
		 * check if in power save mode, if yes, put the device back
		 * to PS mode
		 */
		if ((Adapter->PSMode != Wlan802_11PowerModeCAM) &&
				(Adapter->PSState == PS_STATE_FULL_POWER) &&
				(Adapter->MediaConnectStatus == 
				 WlanMediaStateConnected)) {
#ifdef WPA
			if(Adapter->SecInfo.WPAEnabled
#ifdef WPA2
				|| Adapter->SecInfo.WPA2Enabled
#endif
			) {
				if(Adapter->IsGTK_SET == TRUE) {
					PRINTM(INFO, "EXEC_NEXT_CMD: WPA enabled and GTK_SET"
						" go back to PS_SLEEP");
					PSSleep(priv, 0);
				}
			} else
#endif
			{
				PRINTM(INFO, "EXEC_NEXT_CMD: Command PendQ is empty," 
					" go back to PS_SLEEP");
				PSSleep(priv, 0);
			}
		}
	} 
#endif

	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();
	return ret;
}

#ifdef PS_REQUIRED
/** 
 *  @brief This function allocates buf for PS confirm command.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int AllocatePSConfirmBuffer(wlan_private * priv)
{
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	if (!(Adapter->PSConfirmSleep =
				kmalloc(sizeof(PS_CMD_ConfirmSleep),
					GFP_KERNEL))) {
			PRINTM(INFO, "Allocate ConfirmSleep out of memory \n");
			return WLAN_STATUS_FAILURE;
    	}

    	memset(Adapter->PSConfirmSleep, 0, sizeof(PS_CMD_ConfirmSleep));
	
	Adapter->PSConfirmSleep->SeqNum = wlan_cpu_to_le16(++Adapter->SeqNum);
    	Adapter->PSConfirmSleep->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_PS_MODE);
    	Adapter->PSConfirmSleep->Size = wlan_cpu_to_le16(sizeof(PS_CMD_ConfirmSleep));
    	Adapter->PSConfirmSleep->Result = 0;
    	Adapter->PSConfirmSleep->Action = wlan_cpu_to_le16(HostCmd_SubCmd_Sleep_Confirmed);

    	LEAVE();
    	return WLAN_STATUS_SUCCESS;
}

#if WIRELESS_EXT > 14
/** 
 *  @brief This function sends customized event to application.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @para str	   A pointer to event string
 *  @return 	   n/a
 */
void send_iwevcustom_event(wlan_private *priv, s8 *str)
{
	union iwreq_data iwrq;
	u8 buf[50];
  
	ENTER();

	memset(&iwrq, 0, sizeof(union iwreq_data));
	memset(buf, 0, sizeof(buf));
  
	snprintf(buf, sizeof(buf) - 1, "%s", str);
  
	iwrq.data.pointer = buf; 
	iwrq.data.length = strlen(buf) + 1 + IW_EV_LCP_LEN;

	/* Send Event to upper layer */
	PRINTM(INFO, "Event Indication string = %s\n", iwrq.data.pointer);
	PRINTM(INFO, "Event Indication String Length = %d\n", iwrq.data.length);

	PRINTM(INFO, "Sending wireless event IWEVCUSTOM for %s\n", str);
	wireless_send_event(priv->wlan_dev.netdev, IWEVCUSTOM, &iwrq, buf);

	LEAVE();
	return;
}
#endif

/** 
 *  @brief This function sends sleep confirm command to firmware.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param cmdptr  A pointer to the command
 *  @param size	   the size of command
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int SendConfirmSleep(wlan_private * priv, u8 * CmdPtr, u16 size)
{
	wlan_adapter *Adapter = priv->adapter;
	int             ret = WLAN_STATUS_SUCCESS;

	ENTER();

	PRINTM(INFO, "SEND_SLEEPC_CMD: Before download, Size of cmd = %d\n", size);

	HEXDUMP("SEND_SLEEPC_CMD: Sleep confirm Command", CmdPtr, size);

	ret = sbi_host_to_card(priv, MVMS_CMD, CmdPtr, size);
	priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
#ifndef BULVERDE_SDIO
	if(Adapter->IntCounter || Adapter->CurrentTxSkb)
		PRINTM(INFO, "SEND_SLEEPC_CMD: IntCounter=%d CurrentTxSkb=%p\n",
				Adapter->IntCounter,Adapter->CurrentTxSkb);
#endif

	if (ret) {
		PRINTM(MSG, "SEND_SLEEPC_CMD: Host to Card Failed for Confirm Sleep\n");
	} else {
#ifdef BULVERDE_SDIO
		sdio_clear_imask(((mmc_card_t)((priv->wlan_dev).card))->ctrlr);
#endif
		OS_INT_DISABLE;
		if(!Adapter->IntCounter) {
#ifdef BULVERDE_SDIO
			ps_sleep_confirmed = 1;
#endif
			Adapter->PSState = PS_STATE_SLEEP;
		}
		else {
			PRINTM(INFO, "SEND_SLEEPC_CMD: After sent,IntC=%d\n", Adapter->IntCounter);
		}
		OS_INT_RESTORE;

#ifdef HOST_SLEEP
		if (Adapter->PSState == PS_STATE_SLEEP && Adapter->bHostSleepConfigured) {
			Adapter->bWakeupDevRequired = TRUE;
#if WIRELESS_EXT > 14
			send_iwevcustom_event(priv, "HWM_CFG_DONE.indication ");
#endif /* WIRELESS_EXT */
		}
#endif /* HOST_SLEEP */

		PRINTM(INFO, "SEND_SLEEPC_CMD: Sent Confirm Sleep command\n");
		PRINTM(INFO, "+");
	}
	
	LEAVE();
	return ret;
}

/** 
 *  @brief This function sends Enter_PS command to firmware.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param wait_option	wait response or not
 *  @return 	   	n/a 
 */
void PSSleep(wlan_private *priv, int wait_option)
{
#ifndef AD_HOC_PS
	wlan_adapter   *Adapter = priv->adapter;
#endif

	ENTER();

	/* 
	 * PS is currently supported only in Infrastructure Mode 
	 * Remove this check if it is to be supported in IBSS mode also 
	 */
#ifndef AD_HOC_PS
	if (Adapter->InfrastructureMode != Wlan802_11Infrastructure) {
		PRINTM(INFO, "Enter_PS: PS mode not supported in Ad-Hoc mode\n");
		LEAVE();
		return;
	}
#endif

	PrepareAndSendCommand(priv, HostCmd_CMD_802_11_PS_MODE,
			HostCmd_SubCmd_Enter_PS,
			wait_option, 0, NULL);

	LEAVE();
	return;
}

/** 
 *  @brief This function sends Eixt_PS command to firmware.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param wait_option	wait response or not
 *  @return 	   	n/a 
 */
void PSWakeup(wlan_private *priv, int wait_option)
{
	WLAN_802_11_POWER_MODE LocalPSMode;

	ENTER();

	LocalPSMode = Wlan802_11PowerModeCAM;

	PRINTM(INFO, "Exit_PS: LocalPSMode = %d\n", LocalPSMode);

	PrepareAndSendCommand(priv, HostCmd_CMD_802_11_PS_MODE,
			HostCmd_SubCmd_Exit_PS,
			wait_option, 0, &LocalPSMode);

	LEAVE();
	return;
}

/** 
 *  @brief This function checks condition and prepares to
 *  send sleep confirm command to firmware if ok.
 *  
 *  @param priv    	A pointer to wlan_private structure
 *  @param PSMode  	Power Saving mode
 *  @return 	   	n/a 
 */
void PSConfirmSleep(wlan_private * priv, u16 PSMode)
{
	wlan_adapter   	*Adapter = priv->adapter;
	BOOLEAN 	allowed = TRUE;
 
	ENTER();

	if(priv->wlan_dev.dnld_sent) {
		allowed = FALSE;
		PRINTM(INFO, "D");
	}
	else if(Adapter->CurCmd) {
		allowed = FALSE;
		PRINTM(INFO, "C");
	}
	else if(Adapter->IntCounter > 0) {
		allowed = FALSE;
		PRINTM(INFO, "I%d",Adapter->IntCounter);
	}

	if(allowed) {
		PRINTM(INFO, "Sending PSConfirmSleep\n");
		SendConfirmSleep(priv, (u8 *) Adapter->PSConfirmSleep,
					sizeof(PS_CMD_ConfirmSleep));
	}
	else {
		PRINTM(INFO, "Sleep Confirm has been delayed\n");
	}

	LEAVE();
}
#endif
