#include "ssh.h"

static int
xreadn(int fd, char *buf, int n)
{
	int nn;
	
	werrstr("unexpected eof");
	if(readn(fd, buf, n) != n)
		return -1;
	return 0;
}

Msg*
connread(Conn *z)
{
	Msg *m;
	uchar buf[4];
	uint len, tot;

	/* read length off wire */
	if(xreadn(z->in.fd, buf, 4) < 0)
		return nil;	
	len = G32(buf);
	if(len > MaxPacketLen){
		werrstr("packet too large");
		return nil;
	}

	/* allocate and read packet off wire; length does not include mac */	
	tot = len+z->in.maclen;
	m = msgalloc(z, 4+tot);
	memmove(m->bp, buf, 4);
	if(xreadn(z->in.fd, m->bp+4, tot) < 0){
	Die:
		msgfree(m);
		return nil;
	}
	m->ep = m->bp+4+len;

	/* check mac */
	if(z->in.mac(&z->in, m, m->ep) < 0)
		goto Die;
	
	/* decrypt */
	if(z->in.crypt(&z->in, m) < 0)
		goto Die;
	
	/* decompress */
	if((m = z->in.compress(&z->in, m)) == nil)
		return nil;
	
	return m;	
}

int
msgfmt(Fmt *fmt)
{
	Msg *m;
	
	m = va_arg(fmt->args, Msg*);
	return fmtprint(fmt, "%.*H", m->ep-m->bp, m->bp);
}

Msg*
msgalloc(Conn *z, uint n)
{
	Msg *m;
	
	m = emalloc(sizeof *m+n);
	m->bp = (uchar*)(m+1);
	m->ep = m->bp+n;
	return m;
}

void
msgfree(Msg *m)
{
	free(m);
}

/*

4. Binary Packet Protocol

   Each packet is in the following format:

     uint32    packet_length
     byte      padding_length
     byte[n1]  payload; n1 = packet_length - padding_length - 1
     byte[n2]  random padding; n2 = padding_length
     byte[m]   mac (message authentication code); m = mac_length

      packet_length
         The length of the packet (bytes), not including MAC or the
         packet_length field itself.

      padding_length
         Length of padding (bytes).

      payload
         The useful contents of the packet.  If compression has been
         negotiated, this field is compressed.  Initially, compression
         MUST be "none".

      random padding
         Arbitrary-length padding, such that the total length of
         (packet_length || padding_length || payload || padding) is a
         multiple of the cipher block size or 8, whichever is larger.



Ylonen, et. al.          Expires March 21, 2003                 [Page 5]

Internet-Draft        SSH Transport Layer Protocol        September 2002


         There MUST be at least four bytes of padding.  The padding
         SHOULD consist of random bytes.  The maximum amount of padding
         is 255 bytes.

      mac
         Message authentication code.  If message authentication has
         been negotiated, this field contains the MAC bytes.  Initially,
         the MAC algorithm MUST be "none".


   Note that length of the concatenation of packet length, padding
   length, payload, and padding MUST be a multiple of the cipher block
   size or 8, whichever is larger.  This constraint MUST be enforced
   even when using stream ciphers.  Note that the packet length field is
   also encrypted, and processing it requires special care when sending
   or receiving packets.

   The minimum size of a packet is 16 (or the cipher block size,
   whichever is larger) bytes (plus MAC); implementations SHOULD decrypt
   the length after receiving the first 8 (or cipher block size,
   whichever is larger) bytes of a packet.

4.1 Maximum Packet Length

   All implementations MUST be able to process packets with uncompressed
   payload length of 32768 bytes or less and total packet size of 35000
   bytes or less (including length, padding length, payload, padding,
   and MAC.).  The maximum of 35000 bytes is an arbitrary chosen value
   larger than uncompressed size.  Implementations SHOULD support longer
   packets, where they might be needed, e.g.  if an implementation wants
   to send a very large number of certificates.  Such packets MAY be
   sent if the version string indicates that the other party is able to
   process them.  However, implementations SHOULD check that the packet
   length is reasonable for the implementation to avoid denial-of-
   service and/or buffer overflow attacks.

4.2 Compression

   If compression has been negotiated, the payload field (and only it)
   will be compressed using the negotiated algorithm.  The length field
   and MAC will be computed from the compressed payload.  Encryption
   will be done after compression.

   Compression MAY be stateful, depending on the method.  Compression
   MUST be independent for each direction, and implementations MUST
   allow independently choosing the algorithm for each direction.

   The following compression methods are currently defined:



Ylonen, et. al.          Expires March 21, 2003                 [Page 6]

Internet-Draft        SSH Transport Layer Protocol        September 2002


     none     REQUIRED        no compression
     zlib     OPTIONAL        ZLIB (LZ77) compression

   The "zlib" compression is described in [RFC1950] and in [RFC1951].
   The compression context is initialized after each key exchange, and
   is passed from one packet to the next with only a partial flush being
   performed at the end of each packet.  A partial flush means that the
   current compressed block is ended and all data will be output.  If
   the current block is not a stored block, one or more empty blocks are
   added after the current block to ensure that there are at least 8
   bits counting from the start of the end-of-block code of the current
   block to the end of the packet payload.

   Additional methods may be defined as specified in [SSH-ARCH].

4.3 Encryption

   An encryption algorithm and a key will be negotiated during the key
   exchange.  When encryption is in effect, the packet length, padding
   length, payload and padding fields of each packet MUST be encrypted
   with the given algorithm.

   The encrypted data in all packets sent in one direction SHOULD be
   considered a single data stream.  For example, initialization vectors
   SHOULD be passed from the end of one packet to the beginning of the
   next packet.  All ciphers SHOULD use keys with an effective key
   length of 128 bits or more.

   The ciphers in each direction MUST run independently of each other,
   and implementations MUST allow independently choosing the algorithm
   for each direction (if multiple algorithms are allowed by local
   policy).

   The following ciphers are currently defined:

     3des-cbc         REQUIRED          three-key 3DES in CBC mode
     blowfish-cbc     RECOMMENDED       Blowfish in CBC mode
     twofish256-cbc   OPTIONAL          Twofish in CBC mode,
                                        with 256-bit key
     twofish-cbc      OPTIONAL          alias for "twofish256-cbc" (this
                                        is being retained for
                                        historical reasons)
     twofish192-cbc   OPTIONAL          Twofish with 192-bit key
     twofish128-cbc   RECOMMENDED       Twofish with 128-bit key
     aes256-cbc       OPTIONAL          AES (Rijndael) in CBC mode,
                                        with 256-bit key
     aes192-cbc       OPTIONAL          AES with 192-bit key
     aes128-cbc       RECOMMENDED       AES with 128-bit key



Ylonen, et. al.          Expires March 21, 2003                 [Page 7]

Internet-Draft        SSH Transport Layer Protocol        September 2002


     serpent256-cbc   OPTIONAL          Serpent in CBC mode, with
                                        256-bit key
     serpent192-cbc   OPTIONAL          Serpent with 192-bit key
     serpent128-cbc   OPTIONAL          Serpent with 128-bit key
     arcfour          OPTIONAL          the ARCFOUR stream cipher
     idea-cbc         OPTIONAL          IDEA in CBC mode
     cast128-cbc      OPTIONAL          CAST-128 in CBC mode
     none             OPTIONAL          no encryption; NOT RECOMMENDED

   The "3des-cbc" cipher is three-key triple-DES (encrypt-decrypt-
   encrypt), where the first 8 bytes of the key are used for the first
   encryption, the next 8 bytes for the decryption, and the following 8
   bytes for the final encryption.  This requires 24 bytes of key data
   (of which 168 bits are actually used).  To implement CBC mode, outer
   chaining MUST be used (i.e., there is only one initialization
   vector).  This is a block cipher with 8 byte blocks.  This algorithm
   is defined in [SCHNEIER]

   The "blowfish-cbc" cipher is Blowfish in CBC mode, with 128 bit keys
   [SCHNEIER].  This is a block cipher with 8 byte blocks.

   The "twofish-cbc" or "twofish256-cbc" cipher is Twofish in CBC mode,
   with 256 bit keys as described [TWOFISH].  This is a block cipher
   with 16 byte blocks.

   The "twofish192-cbc" cipher.  Same as above but with 192-bit key.

   The "twofish128-cbc" cipher.  Same as above but with 128-bit key.

   The "aes256-cbc" cipher is AES (Advanced Encryption Standard),
   formerly Rijndael, in CBC mode.  This version uses 256-bit key.

   The "aes192-cbc" cipher.  Same as above but with 192-bit key.

   The "aes128-cbc" cipher.  Same as above but with 128-bit key.

   The "serpent256-cbc" cipher in CBC mode, with 256-bit key as
   described in the Serpent AES submission.

   The "serpent192-cbc" cipher.  Same as above but with 192-bit key.

   The "serpent128-cbc" cipher.  Same as above but with 128-bit key.

   The "arcfour" is the Arcfour stream cipher with 128 bit keys.  The
   Arcfour cipher is believed to be compatible with the RC4 cipher
   [SCHNEIER].  RC4 is a registered trademark of RSA Data Security Inc.
   Arcfour (and RC4) has problems with weak keys, and should be used
   with caution.



Ylonen, et. al.          Expires March 21, 2003                 [Page 8]

Internet-Draft        SSH Transport Layer Protocol        September 2002


   The "idea-cbc" cipher is the IDEA cipher in CBC mode [SCHNEIER].
   IDEA is patented by Ascom AG.

   The "cast128-cbc" cipher is the CAST-128 cipher in CBC mode
   [RFC2144].

   The "none" algorithm specifies that no encryption is to be done.
   Note that this method provides no confidentiality protection, and it
   is not recommended.  Some functionality (e.g.  password
   authentication) may be disabled for security reasons if this cipher
   is chosen.

   Additional methods may be defined as specified in [SSH-ARCH].

4.4 Data Integrity

   Data integrity is protected by including with each packet a message
   authentication code (MAC) that is computed from a shared secret,
   packet sequence number, and the contents of the packet.

   The message authentication algorithm and key are negotiated during
   key exchange.  Initially, no MAC will be in effect, and its length
   MUST be zero.  After key exchange, the selected MAC will be computed
   before encryption from the concatenation of packet data:

     mac = MAC(key, sequence_number || unencrypted_packet)

   where unencrypted_packet is the entire packet without MAC (the length
   fields, payload and padding), and sequence_number is an implicit
   packet sequence number represented as uint32.  The sequence number is
   initialized to zero for the first packet, and is incremented after
   every packet (regardless of whether encryption or MAC is in use).  It
   is never reset, even if keys/algorithms are renegotiated later.  It
   wraps around to zero after every 2^32 packets.  The packet sequence
   number itself is not included in the packet sent over the wire.

   The MAC algorithms for each direction MUST run independently, and
   implementations MUST allow choosing the algorithm independently for
   both directions.

   The MAC bytes resulting from the MAC algorithm MUST be transmitted
   without encryption as the last part of the packet.  The number of MAC
   bytes depends on the algorithm chosen.








Ylonen, et. al.          Expires March 21, 2003                 [Page 9]

Internet-Draft        SSH Transport Layer Protocol        September 2002


   The following MAC algorithms are currently defined:

     hmac-sha1    REQUIRED        HMAC-SHA1 (digest length = key
                                  length = 20)
     hmac-sha1-96 RECOMMENDED     first 96 bits of HMAC-SHA1 (digest
                                  length = 12, key length = 20)
     hmac-md5     OPTIONAL        HMAC-MD5 (digest length = key
                                  length = 16)
     hmac-md5-96  OPTIONAL        first 96 bits of HMAC-MD5 (digest
                                  length = 12, key length = 16)
     none         OPTIONAL        no MAC; NOT RECOMMENDED

   The "hmac-*" algorithms are described in [RFC2104] The "*-n" MACs use
   only the first n bits of the resulting value.

   The hash algorithms are described in [SCHNEIER].

   Additional methods may be defined as specified in [SSH-ARCH].

4.5 Key Exchange Methods

   The key exchange method specifies how one-time session keys are
   generated for encryption and for authentication, and how the server
   authentication is done.

   Only one REQUIRED key exchange method has been defined:

     diffie-hellman-group1-sha1       REQUIRED

   This method is described later in this document.

   Additional methods may be defined as specified in [SSH-ARCH].

4.6 Public Key Algorithms

   This protocol has been designed to be able to operate with almost any
   public key format, encoding, and algorithm (signature and/or
   encryption).

   There are several aspects that define a public key type:
   o  Key format: how is the key encoded and how are certificates
      represented.  The key blobs in this protocol MAY contain
      certificates in addition to keys.
   o  Signature and/or encryption algorithms.  Some key types may not
      support both signing and encryption.  Key usage may also be
      restricted by policy statements in e.g.  certificates.  In this
      case, different key types SHOULD be defined for the different
      policy alternatives.



Ylonen, et. al.          Expires March 21, 2003                [Page 10]

Internet-Draft        SSH Transport Layer Protocol        September 2002


   o  Encoding of signatures and/or encrypted data.  This includes but
      is not limited to padding, byte order, and data formats.

   The following public key and/or certificate formats are currently defined:

   ssh-dss              REQUIRED     sign    Simple DSS
   ssh-rsa              RECOMMENDED  sign    Simple RSA
   x509v3-sign-rsa      OPTIONAL     sign    X.509 certificates (RSA key)
   x509v3-sign-dss      OPTIONAL     sign    X.509 certificates (DSS key)
   spki-sign-rsa        OPTIONAL     sign    SPKI certificates (RSA key)
   spki-sign-dss        OPTIONAL     sign    SPKI certificates (DSS key)
   pgp-sign-rsa         OPTIONAL     sign    OpenPGP certificates (RSA key)
   pgp-sign-dss         OPTIONAL     sign    OpenPGP certificates (DSS key)

   Additional key types may be defined as specified in [SSH-ARCH].

   The key type MUST always be explicitly known (from algorithm
   negotiation or some other source).  It is not normally included in
   the key blob.

   Certificates and public keys are encoded as follows:

     string   certificate or public key format identifier
     byte[n]  key/certificate data

   The certificate part may have be a zero length string, but a public
   key is required.  This is the public key that will be used for
   authentication; the certificate sequence contained in the certificate
   blob can be used to provide authorization.

   Public key / certifcate formats that do not explicitly specify a
   signature format identifier MUST use the public key / certificate
   format identifier as the signature identifier.

   Signatures are encoded as follows:
     string    signature format identifier (as specified by the
               public key / cert format)
     byte[n]   signature blob in format specific encoding.


   The "ssh-dss" key format has the following specific encoding:

     string    "ssh-dss"
     mpint     p
     mpint     q
     mpint     g
     mpint     y




Ylonen, et. al.          Expires March 21, 2003                [Page 11]

Internet-Draft        SSH Transport Layer Protocol        September 2002


   Here the p, q, g, and y parameters form the signature key blob.

   Signing and verifying using this key format is done according to the
   Digital Signature Standard [FIPS-186] using the SHA-1 hash.  A
   description can also be found in [SCHNEIER].

   The resulting signature is encoded as follows:

     string    "ssh-dss"
     string    dss_signature_blob

   dss_signature_blob is encoded as a string containing r followed by s
   (which are 160 bits long integers, without lengths or padding,
   unsigned and in network byte order).

   The "ssh-rsa" key format has the following specific encoding:

     string    "ssh-rsa"
     mpint     e
     mpint     n

   Here the e and n parameters form the signature key blob.

   Signing and verifying using this key format is done according to
   [SCHNEIER] and [PKCS1] using the SHA-1 hash.

   The resulting signature is encoded as follows:

     string    "ssh-rsa"
     string    rsa_signature_blob

   rsa_signature_blob is encoded as a string containing s (which is an
   integer, without lengths or padding, unsigned and in network byte
   order).

   The "spki-sign-rsa" method indicates that the certificate blob
   contains a sequence of SPKI certificates.  The format of SPKI
   certificates is described in [RFC2693].  This method indicates that
   the key (or one of the keys in the certificate) is an RSA-key.

   The "spki-sign-dss".  As above, but indicates that the key (or one of
   the keys in the certificate) is a DSS-key.

   The "pgp-sign-rsa" method indicates the certificates, the public key,
   and the signature are in OpenPGP compatible binary format
   ([RFC2440]).  This method indicates that the key is an RSA-key.

   The "pgp-sign-dss".  As above, but indicates that the key is a DSS-



Ylonen, et. al.          Expires March 21, 2003                [Page 12]

Internet-Draft        SSH Transport Layer Protocol        September 2002


   key.
*/
