#include <ctype.h>
#include "dat.h"
#include "fns.h"

int	strict;
int	doabort;
int	verbose;
u32int	dbglevel;

char	Ebotch[] = "can't happen";
char	Ebadmac[] = "computed HMAC does not match HMAC from the wire";
char	Edecode[] = "error decoding packet on the wire";
char	Eencode[] = "error encoding packet for the wire (BUG)";
char	Ehungup[] = "hungup connection";
char	Ememory[] = "out of memory";
char	Enotimpl[] = "not implemented";

void*
emalloc(ulong n)
{
	void *p;

	if(n < 16)
		n = 16;
	if((p = malloc(n)) == nil)
		panic(Ememory);
	setmalloctag(p, getcallerpc(&n));
	memset(p, 0, n);
	return p;
}

void*
erealloc(void *p, ulong n)
{
	if((p = realloc(p, n)) == nil)
		panic(Ememory);
	setmalloctag(p, getcallerpc(&n));
	return p;
}

char*
estrdup(char *s)
{
	char *p;

	if(s == nil)
		s = "";
	if((p = strdup(s)) == nil)
		panic(Ememory);
	return p;
}

char*
estrcat(char *s, char *t)
{
	int n;

	if(t == nil)
		t = "";
	if(s == nil)
		s = estrdup("");
	n = strlen(s)+strlen(t)+1;
	s = erealloc(s, n);
	strcat(s, t);

	return s;
}

char*
esmprint(char *fmt, ...)
{
	va_list arg;
	char *s;

	va_start(arg, fmt);
	s = vsmprint(fmt, arg);
	va_end(arg);
	if(s == nil)
		sysfatal("no memory: esmprint(%s, ...), %lux",
			 fmt, getcallerpc(&fmt));
	setmalloctag(s, getcallerpc(&fmt));
	return s;
}

/* ye olde standard hash function */
uint
hash(char *s)
{
	uint h;
	uchar *p;

	h = 0;
	for(p=(uchar*)s; *p; p++)
		h = 37*h + *p;
	return h;
}

char*
skiptext(char *p)
{
	while(*p){
		if(isspace(*p))
			break;
		p++;
	}

	return p;
}

char*
skipwhite(char *p)
{
	while(*p){
		if(!isspace(*p))
			break;
		p++;
	}

	return p;
}

char*
trimwhite(char *s)
{
	int n;

	n = strlen(s)-1;
	while(n >= 0 && isspace(s[n]))
		n--;
	s[n+1] = '\0';

	return s;
}

char*
filterstring(char *s)
{
	char *p;

	for(p=s; *p != '\0'; p++)
		if(!isalnum(*p) && !isspace(*p) && !ispunct(*p))
			*p = '*';
	return s;
}

char*
snarfline(Biobuf *b, int *line)
{
	int n;
	char *s, *t;

	for(n=line?*line:0; (s = Brdstr(b, '\n', 1)) != nil; n++){
		t = strchr(s, '#');
		if(t != nil)
			*t = '\0';
		t = skipwhite(s);
		if(*t == '\0')
			goto Next;
		break;
 Next:
		free(s);
	}

	if(line != nil)
		*line = n;
	return s;
}

Biobuf*
Bsafeopen(char *fname, int mode)
{
	if(fname[0] != '/'){
		werrstr("Bsafeopen: path must be absolute: %s", fname);
		return nil;
	}
	return Bopen(fname, mode);
}

void
_coverage(char *file, int line)
{
	fprint(2, "COVERAGE %s:%d\n", file, line);
}

void
installfmts(void)
{
	quotefmtinstall();		/* q, Q */

	fmtinstall('B', mpfmt);
	fmtinstall('H', encodefmt);	/* hex */
	fmtinstall('<', encodefmt);	/* base 32 */
	fmtinstall('[', encodefmt);	/* base 64 */
	fmtinstall('N', namelistfmt);
}

void
installcleanup(void)
{
	/*
	 * p9atexit() makes sure handlers are called once in
	 * the proc that installed it.  What about Unix libc?
	 */
	atexit(conncleanup);
}

void
debug(u32int level, char *f, ...)
{
	Fmt fmt;
	va_list arg;
	char buf[128];

	if((dbglevel & level) == 0)
		return;

	fmtfdinit(&fmt, 2, buf, sizeof buf);
	va_start(arg, f);
	fmtprint(&fmt, "%s: ", dbgname ? dbgname : argv0);
	fmtvprint(&fmt, f, arg);
	fmtprint(&fmt, "\n");
	va_end(arg);
	fmtfdflush(&fmt);
}

static char*
dbgstrtab[] = {
	"debug",
	"crypto",
	"packet",
	"auth",
	"cfg",
	"proc",
	"proto",
	"io",
	"scp",
	nil
};

u32int
debugparse(char *s)
{
	int i;
	u32int mask = 0;

	if(cistrcmp("all", s) == 0)
		return ~0;

	for(i=0; dbgstrtab[i] != nil; i++)
		if(strstr(s, dbgstrtab[i]))
			mask |= 1<<i;
	return mask;
}

void
panic(char *fmt, ...)
{
	va_list arg;
	char buf[2048];

	va_start(arg, fmt);
	/* may be out of memory, so no vsmprint */
	vseprint(buf, buf+sizeof(buf), fmt, arg);
	va_end(arg);
	fprint(2, "%s panic: 0x%lux: %s\n", argv0, getcallerpc(&fmt), buf);
	if(doabort)
		abort();
	sysfatal("%s", buf);
	abort();
	for(;;)
		sleep(1);
}

void
warn(char *f, ...)
{
	Fmt fmt;
	va_list arg;
	char buf[128];

	if(verbose == 0)
		return;

	fmtfdinit(&fmt, 2, buf, sizeof buf);
	va_start(arg, f);
	fmtprint(&fmt, "warning: ");
	fmtvprint(&fmt, f, arg);
	fmtprint(&fmt, "\n");
	va_end(arg);
	fmtfdflush(&fmt);

	return;
}

void
cfgwarn(Conn *c, char *fmt, ...)
{
	va_list arg;

	Bprint(&c->bstderr, "warning: ");
	va_start(arg, fmt);
	Bvprint(&c->bstderr, fmt, arg);
	va_end(arg);

	Bprint(&c->bstderr, "\n");
	Bflush(&c->bstderr);

	return;
}

void
error(Conn *c, char *fmt, ...)
{
	va_list arg;

	va_start(arg, fmt);
	Bvprint(&c->bstderr, fmt, arg);
	va_end(arg);

	Bprint(&c->bstderr, "\n");
	Bflush(&c->bstderr);
	return;
}

mpint*
sha1tomp(uchar *buf, int n)	/* XXX: this should be in the library, no? */
{
	mpint *b;
	uchar digest[SHA1dlen];

	sha1(buf, n, digest, nil);
	b = betomp(digest, sizeof(digest), nil);
	return b;
}

mpint*
mpnrand(mpint *n, mpint *p)
{
	int k;

	/* Generate random p in [1, n-1] */
	k = mpsignif(n);
	do{
		p = mprand(k, genrandom, p);
	}while(mpcmp(p, n) >= 0 || mpcmp(p, mpzero) <= 0);

	return p;
}

void
mpclear(mpint *p)
{
	if(p == nil)
		return;

	memset(p->p, 0, p->size*Dbytes);
	mpfree(p);
}

void
rsacrt(RSApriv *rsa)
{
	mpint *p, *q, *kp, *kq;
	mpint *phi, *c2, *e, *d;

	p = mpnew(0);
	q = mpnew(0);
	phi = mpnew(0);

	e = rsa->pub.ek;
	d = rsa->dk;

	mpmul(e, d, phi);

	mpadd(e, mpone, p);
	mpadd(d, mpone, q);

	// CRT coefficient
	c2 = mpnew(0);
	mpinvert(p, q, c2);

	// a^k mod p = a^(k mod p-1) mod p
	kq = mpnew(0);
	kp = mpnew(0);
	mpsub(p, mpone, phi);
	mpmod(d, phi, kp);
	mpsub(q, mpone, phi);
	mpmod(d, phi, kq);
	mpfree(phi);

	rsa->p = p;
	rsa->q = q;
	rsa->kp = kp;
	rsa->kq = kq;
	rsa->c2 = c2;

}

static long
_iowriten(va_list *arg)
{
	int fd;
	void *a;
	long n;

	fd = va_arg(*arg, int);
	a = va_arg(*arg, void*);
	n = va_arg(*arg, long);
	return writen(fd, a, n);
}

long
iowriten(Ioproc *io, int fd, void *a, long n)
{
	return iocall(io, _iowriten, fd, a, n);
}
