#include "dat.h"
#include "fns.h"

/* Read and write certificates (aka: format hell)
 *
 * The format for public key files is similar to that
 * used by OpenSSH and is identical to that used by
 * OpenSSH for the known hosts database, namely:
 *
 *	host type uuencoded-key-blob
 *
 * OpenSSH public key files typically take the form:
 *
 *	type uuencoded-key-blob user@host
 *
 * For private keys, we use a format similar to that for
 * SSH1 RSA public keys.  OpenSSH uses PEM/ASN.1, but at
 * the moment we can only decode RSA keys in this format
 * and can't write either RSA or DSA keys in it.  SSH.com
 * has another private key format; it may be easier to
 * deal with, at least in the unencrypted form OpenSSH can
 * handle at this point.  The format we currently use is:
 *
 *	host type parameters
 *
 * where 'parameters' is a white-space separated list of key
 * parameters encoded in hexadecimal.  For instance, for DSA
 * keys it consists of:
 *
 *	p q alpha key secret
 *
 * whereas for RSA keys it consists of:
 *
 *	n ek dk p q kp kq c2
 *
 * where n is the modulus, ek is the encryption key (exponent),
 * dk is the decryption key (exponent), and the remaining
 * parameters are the pre-computed chinese remaindering values.
 */

int
Bwrcertpub(Biobuf *b, Cert *p, char *id)
{
	int n;

	n = skiptext(id) - id;
	Bprint(b, "%.*s ", n, id);
	p->pubprint(p, b);
	Bprint(b, "\n");

	return 0;
}

int
Brdcertpub(Biobuf *b, Cert **p, char **id)
{
	int n;
	char *s, *t;
	CertImpl *c;

	if(id != nil)
		*id = nil;
	s = snarfline(b, nil);
	if(s == nil)
		goto Error;
	t = skipwhite(s);
	n = skiptext(t)-t;
	if(id != nil)
		*id = esmprint("%.*s", n, t);

	t = skiptext(t);
	t = skipwhite(t);
	if((c = certstrsniff(t)) == nil){
		werrstr("unknown certificate type: '%.*s'", skiptext(t)-t, t);
		goto Error;
	}

	*p = c->pubparse(t);
	if(*p == nil){
		werrstr("can't decode public key: %r");
		goto Error;
	}

	return 0;

 Error:
	if(id != nil)
		free(*id);
	return -1;
}

int
Bwrcertpriv(Biobuf *b, Cert *p, char *id)
{
	int n;

	n = skiptext(id) - id;
	Bprint(b, "%.*s ", n, id);
	p->privprint(p, b);
	Bprint(b, "\n");

	return 0;
}

int
Brdcertpriv(Biobuf *b, Cert **p, char **id)
{
	int n;
	char *s, *t;
	CertImpl *c;

	if(id != nil)
		*id = nil;
	s = snarfline(b, nil);
	if(s == nil)
		goto Error;
	t = skipwhite(s);
	n = skiptext(t)-t;
	if(id != nil)
		*id = esmprint("%.*s", n, t);

	t = skiptext(t);
	t = skipwhite(t);
	if((c = certstrsniff(t)) == nil){
		werrstr("unknown certificate type: '%.*s'", skiptext(t)-t, t);
		goto Error;
	}

	*p = c->privparse(t);
	if(*p == nil){
		werrstr("can't decode public key: %r");
		goto Error;
	}

	return 0;

 Error:
	if(id != nil)
		free(*id);
	return -1;
}
