#include "dat.h"
#include "fns.h"

int	verbose;
u32int	dbglevel;
char	*dbgname = "ssh/keygen";

extern	char	*release(void);

static void
usage(void)
{
	fprint(2, "usage: ssh/keygen [-AVv?] [-T fpfmt] ");
	fprint(2, "[-b bits] [-i id] [-t type] [file]\n");
	exits("usage");
}

/* XXX: single threaded program, but I'm lazy */
void
threadmain(int argc, char *argv[])
{
	int i;
	int fpfmt;
	int fd[2];

	char *fp;
	char *ctype;
	char *dtype;
	char *s, *id;
	char *fname[2];

	Cert *cert;
	Impllist *il;
	Digest *digest;

	Biobuf buf[2];
	
	installfmts();
	id = sysname();

	cert = nil;
	digest = nil;
	fpfmt = FPHex;
	ctype = "dsa";
	dtype = "sha1";
	fname[0] = "identity";

	ARGBEGIN{
	case 'A':
		doabort = 1;
		break;

	case 'D':
		dbglevel++;
		s = ARGF();
		if(s != nil)
			dbglevel = debugparse(s);
		break;

	case 'H':
		dtype = EARGF(usage());		/* fingerprint hash type */
		break;

	case 'T':
		s = EARGF(usage());
		fpfmt = fingerprinttype(s);
		break;

	case 'V':
		print("release: %s\n", release());
		exits(nil);
		break;

	case 'b':
		sysfatal("-b (set key bits) not implemented");
		break;

	case 'i':
		id = EARGF(usage());
		break;

	case 't':
		ctype = EARGF(usage());		/* certificate type */
		break;

	case 'v':
		verbose++;
		break;

	case 'h':
	case '?':
	default:
		usage();
		break;
	}ARGEND;

	switch(argc){
	case 1:
		fname[0] = estrdup(argv[0]);
		break;

	case 0:
		break;

	default:
		usage();
		break;
	}

	fname[0] = estrdup(fname[0]);
	fname[1] = esmprint("%s.pub", fname[0]);

	/* rsa for ssh-rsa and dss/dsa for ssh-dss */
	if(strcmp(ctype, "rsa") == 0){
		cert = &certsshrsa;
	}else if(strcmp(ctype, "dss") == 0 || strcmp(ctype, "dsa") == 0){
		cert = &certsshdss;
	}else{
		il = loadimpl(nil, "cert", nil);
		for(i=0; i<il->nimpl; i++)
			if(cistrcmp(ctype, il->impl[i]->name) == 0){
				cert = (Cert*)il->impl[i];
				break;
			}
		freeimpllist(il);
	}

	if(cert == nil)
		sysfatal("unknown/unsupported certificate type: '%s'", ctype);

	il = loadimpl(nil, "digest", nil);
	for(i=0; i<il->nimpl; i++)
		if(cistrcmp(dtype, il->impl[i]->name) == 0){
			digest = (Digest*)il->impl[i];
			break;
		}
	freeimpllist(il);
	if(digest == nil)
		sysfatal("unknown/unsupported digest algorithm: '%s'", dtype);

	for(i=0; i<2; i++){
		if(access(fname[i], AEXIST) == 0){
			while(i>0)
				remove(fname[--i]);
			sysfatal("file '%s' already exists", fname[i]);
		}
		fd[i] = create(fname[i], OWRITE, 0600);
		if(fd[i] < 0)
			sysfatal("can't create '%s': %r", fname[i]);
		Binit(&buf[i], fd[i], OWRITE);
	}

	if(verbose)
		print("generating key ... ");
	cert = cert->gen(nil);
	if(verbose)
		print("done.\n");

	fp = fingerprintcert(cert, fpfmt, digest->digest, digest->digestlen);
	print("%s fingerprint: %s\n", digest->name, fp);
	free(fp);

	if(Bwrcertpriv(&buf[0], cert, id) < 0){
		for(i=nelem(fname); i>0; i--)
			remove(fname[i]);
		sysfatal("can't write private key: %r");
	}

	if(Bwrcertpub(&buf[1], cert, id) < 0){
		for(i=nelem(fname); i>0; i--)
			remove(fname[i]);
		sysfatal("can't write public key: %r");
	}

	threadexitsall(nil);
}
