#include "dat.h"
#include "fns.h"

/*
 * XXX: configuration syntax, semantics, and implementation is a total botch.
 * Actually, come to think of it, the need for a configuration file is, too.
 *
 * XXX: note that there is little error checking; typos are largely ignored
 */

/*
 * Section names:
 *	default
 *	client
 * 	server
 *	host
 *
 * Keys
 *	Key			Value
 *
 *	verbose			integer verbosity level if verbose is zero
 *	strict			strict host key checking? (default = no)
 *	debug			debug spec; used only if dbglevel is zero
 *	doabort			set doabort if 'yes'/'true'; otherwise nothing
 *
 * 	match			host name glob for host sections
 *	hostcertdb		path to host certificate database (XXX: list)
 *	usercertdb		list of private key file paths
 *
 *	kex			list of acceptable key exchange algorithms
 *	auth			list of acceptable authentication methods
 *	authsrv			list of acceptable server-side auth. methods
 *	cert			list of acceptable server certificate types
 *	sig			list of acceptable server cert. signature types
 *	cipher			list of acceptable ciphers
 *	hmac			list of acceptable message authentication codes
 *
 *	kbdmethods		preferences for keyboard-interactive
 *
 *	dhgpfile		path to DH group exchange group database
 *	dhgpmin			DH group exchange parameter
 *	dhgpmax			DH group exchange parameter
 *	dhgpdef			DH group exchange parameter
 */

typedef struct Sym Sym;
typedef struct Section Section;

enum {
	Ntab		= 127,
};

struct Cfg {
	char		*fname;
	Section		*defsec;
	Section		*seclist;
};

struct Section {
	char	*name;
	int	line;
	char	*fname;
	int	warned;
	Sym	*symtab[Ntab];
	Section	*link;
};

struct Sym {
	char	*key;
	char	*val;
	Sym	*link;
};

static	int		Brdcfgfile(Conn*, char*, Section**);
static	Section		*mksection(char*, int, char*);
static	void		freesection(Section*);
static	int		putsym(Section*, char*, char*);
static	char		*looksym(Section*, char*);

static	char		*cfglook(Conn*, char*, char*, int);

/*
 * XXX: Variable substitution for $home, $user, etc. not implemented
 */

Cfg*
mkcfg(Conn *c, char *fname, int role, char *host, char *user)
{
	Cfg *cfg;
	Section *s, *sl;

	USED(host);
	USED(user);

	if(Brdcfgfile(c, fname, &s) < 0)
		return nil;

	cfg = emalloc(sizeof *cfg);
	cfg->fname = estrdup(fname);

	/*
	 * Find default section and set Cfg->defsec
	 * XXX: should warn about duplicate client
	 *      and server sections
	 */
	cfg->seclist = s;
	for(sl=s; sl != nil; sl=sl->link){
		if(strcmp("default", sl->name) == 0){
			if(cfg->defsec != nil)
				cfgwarn(c, "%s:%d: duplicate default section ignored", sl->fname, sl->line);
			else
				cfg->defsec = sl;
		}
	}
	if(cfg->defsec == nil){
		/* create a default section and link it in */
		cfg->defsec = mksection(fname, ~0, "default");
		cfg->defsec->link = cfg->seclist;
		cfg->seclist = cfg->defsec;
	}

	/*
	 * The various defaults are entered into the
	 * default section by mkconn as necessary
	 */

	return cfg;
}

void
freecfg(Cfg *cfg)
{
	if(cfg == nil)
		return;
	free(cfg->fname);
	cfg->defsec = nil;			/* also in section list */
	freesection(cfg->seclist);
	free(cfg);

	return;
}

static Section*
mksection(char *fname, int line, char *name)
{
	Section *s;

	s = emalloc(sizeof *s);
	s->name = estrdup(name);
	s->fname = estrdup(fname);
	s->line = line;

	return s;
}

static void
freesection(Section *s)
{
	int i;
	Sym *p;
	Section *t;

	if(s == nil)
		return;

	while(s != nil){
		for(i=0; i<nelem(s->symtab); i++)
			while(s->symtab[i] != nil){
				p = s->symtab[i];
				s->symtab[i] = s->symtab[i]->link;
				free(p->key);
				free(p->val);
				free(p);
			}
		t = s->link;
		free(s);
		s = t;
	}

	return;
}

static int
putsym(Section *s, char *key, char *val)
{
	uint h;
	Sym *p;

	if(0 && verbose)
		debug(DbgCfg, "putsym 0x%lux %s --> %s", s, key, val);
	h = hash(key)%nelem(s->symtab);
	for(p=s->symtab[h]; p != nil; p=p->link)
		if(strcmp(key, p->key) == 0)
			return -1;
	p = emalloc(sizeof *p);
	p->key = estrdup(key);
	p->val = estrdup(val);
	p->link = s->symtab[h];
	s->symtab[h] = p;

	return 0;
}

static char*
looksym(Section *s, char *key)
{
	uint h;
	Sym *p;

	if(0 && verbose)
		debug(DbgCfg, "looksym 0x%lux %s", s, key);
	h = hash(key)%nelem(s->symtab);
	for(p=s->symtab[h]; p!=nil; p=p->link)
		if(strcmp(p->key, key) == 0)
			return p->val;

	return nil;
}

static char*
cfglook(Conn *c, char *secnam, char *key, int glob)
{
	char *s;
	char *fmt;
	Cfg *cfg;
	Section *sec;

	cfg = c->cfg;
	fmt = "%s:%d 'host' section lacks match clause";
	for(sec=cfg->seclist; sec != nil; sec=sec->link){
		if(glob == 0){
			if(strcmp(sec->name, secnam) == 0)
				return looksym(sec, key);
		}else{
			if(strcmp(sec->name, "host") != 0)
				continue;
			s = looksym(sec, "match");
			if(s == nil){
				if(sec->warned == 0){
					cfgwarn(c, fmt, sec->fname, sec->line);
					sec->warned = 1;
				}
				continue;
			}
			if(match(secnam, s))
				return looksym(sec, key);
		}
	}

	return nil;
}

void
cfgoptions(Conn *c)
{
	char *s;

	debug(DbgCfg, "cfgoptions: verbose=%d doabort=%d strict=%d",
	      verbose, doabort, strict);

	/* these options don't override the command line */
	if(verbose == 0)
		verbose = cfgint(c, "verbose");
	s = cfgstr(c, "debug");
	if(s != nil && dbglevel == 0)
		dbglevel = debugparse(s);
	s = cfgstr(c, "doabort");
	if(s != nil && (cistrcmp(s, "yes") == 0 || cistrcmp(s, "true") == 0))
		doabort = 1;
	s = cfgstr(c, "strict");
	if(s != nil && (cistrcmp(s, "yes") == 0 || cistrcmp(s, "true") == 0))
		strict = 1;

	debug(DbgCfg, "cfgoptions: verbose=%d doabort=%d strict=%d",
	      verbose, doabort, strict);
	return;
}

void
cfgdefault(Conn *c, char *key, char *val)
{
	char *s;
	Section *sec;

	sec = c->cfg->defsec;
	s = looksym(sec, key);
	if(s == nil)
		putsym(sec, key, val);
	return;
}

char*
cfgstr(Conn *c, char *key)
{
	char *r, *s;

	r = c->role==RClient?"client":"server";
	s = cfglook(c, c->host, key, 1);
	if(s == nil)
		s = cfglook(c, r, key, 0);
	if(s == nil)
		s = cfglook(c, "default", key, 0);
	return s;
}

int
cfgint(Conn *c, char *key)
{
	int n;
	char *s, *p;

	s = cfgstr(c, key);
	if(s == nil)
		return 0;
	n = strtol(s, &p, 0);		
	if(*s != '\0' && *p == '\0')
		return n;
	return 0;		
}

int
cfgbool(Conn *c, char *key)
{
	char *s;

	s = cfgstr(c, key);
	if(s == nil)
		return 0;
	if(cistrcmp("yes", s) == 0 || cistrcmp("true", s) == 0)
		return 1;
	return 0;
}

Namelist*
cfgnamelist(Conn *c, char *key)
{
	char *s;
	Namelist *nl;

	s = cfgstr(c, key);
	if(s == nil)
		return nil;
	nl = parsenamelist(s);
	return nl;
}

static char *sectab[] = {
	"client",
	"default",
 	"host",
	"server",
	nil
};

static int
validsec(char *s)
{
	int i;

	for(i=0; sectab[i] != nil; i++)
		if(strcmp(sectab[i], s) == 0)
			return 1;
	return 0;
}

static int 
Brdcfgfile(Conn *c, char *fname, Section **sl)
{
	int line;
	int ignore;

	char *p, *s;
	char *key, *val;

	Biobuf *b;
	Section *sec, *t;

	*sl = nil;	
	sec = nil;
	ignore = 0;

	b = Bopen(fname, OREAD);
	if(b == nil)
		return -1;

	line = 1;
	while((p = snarfline(b, &line)) != nil){
		if(*p == '['){
			ignore = 0;
			if((s = strchr(p+1, ']')) == nil){
				cfgwarn(c, "%s:%d malformed section",
					fname, line);
				ignore = 1;
				goto Next;
			}
			*s = '\0';
			if(validsec(p+1) == 0){
				cfgwarn(c, "%s:%d unknown section type '%s'",
					fname, line, p+1);
				ignore = 1;
			}
			if(ignore == 0){
				t = mksection(fname, line, p+1);
				t->link = sec;
				sec = t;
			}
			goto Next;
		}
		if(ignore)
			goto Next;
		if(sec == nil){
			sec = mksection(fname, line, "default");
			cfgwarn(c, "%s:%d unnamed section assumed default",
				fname, line);
		}

		key = skipwhite(p);
		for(s=key; *s; s++)
			if(*s == '=' || isspace(*s)){
				*s++ = '\0';
				break;
			}
		val = skipwhite(s);
		trimwhite(val);
//		val = unquotestrdup(val);
		val = estrdup(val);
		assert(val != nil);
		putsym(sec, key, val);
		free(val);
		
 Next:
		free(p);
	}

	Bterm(b);

	*sl = sec;

	return 0;
}
