#include "dat.h"
#include "fns.h"

/*
 * Certificate and signature implementations are in certfoo.c
 * Beware the private portions of those implementations.
 */

CertList*
mkcertlist(Cert *cert)
{
	CertList *cl;

	cl = emalloc(sizeof *cl);
	cl->cert = cert;

	return cl;
}

void
freecertlist(CertList *cl)
{
	CertList *tcl;

	while(cl != nil){
		tcl = cl;
		cl = cl->link;
		tcl->cert->free(tcl->cert);
		free(tcl);
	}

	return;
}

/*
 * Sniff the type of a certificate or signature and
 * find the corresponding implementation, if any.
 */

CertImpl*
certwiresniff(Conn *c, Blob *b)
{
	char *s;
	Impl *ip;

	s = getstring(b);
	ip = findimpl(c->okhostcert, s);
	return (CertImpl*)ip;
}

CertImpl*
certstrsniff(char *s)
{
	int i, n;
	CertImpl *c;
	Impllist *il;

	c = nil;
	s = skipwhite(s);
	n = skiptext(s) - s;
	il = loadimpl(nil, "cert", nil);
	for(i=0; i<il->nimpl; i++)
		if(strncmp(s, il->impl[i]->name, n) == 0){
			c = (CertImpl*)il->impl[i];
			break;
		}
	freeimpllist(il);

	return c;
}

SigImpl*
sigwiresniff(Conn *c, Blob *b)
{
	char *s;
	Impl *ip;

	s = getstring(b);
	ip = findimpl(c->okhostsig, s);
	return (SigImpl*)ip;
}

static int
matchcertname(char *host, char *id)
{
	int i, n;
	Namelist *nl;

	nl = parsenamelist(id);
	n = nl->nstr;
	for(i=0; i<n; i++)
		if(cistrcmp(nl->strtab[i], id) == 0)
			break;
	freenamelist(nl);
	return i<n;
}

/* read from certificate database */
int
certpublookup(char *fname, Conn *c, char *type, char **id, Cert **cert)
{
	int rv;
	Biobuf *b;

	assert(fname != nil);

	rv = -1;
	b = Bsafeopen(fname, OREAD);
	if(b == nil){
		warn("could not open '%s': %r", fname);
		return -1;
	}
	debug(DbgAuth, "certpublook: opened db '%s'", fname);
	while(Brdcertpub(b, cert, id) == 0){
		debug(DbgAuth, "certpublook: found certificate for '%s'", *id);
		if(matchcertname(c->host, *id) == 0 &&
		   strcmp((*cert)->name, type) == 0){
			rv = 0;
			goto Found;
		}
		free(*id);
		(*cert)->free(*cert);
	}
	werrstr("no key for host '%s' in database", c->host);
	*id = nil;
	*cert = nil;

 Found:
	Bterm(b);
	return rv;
}

static CertList*
certprivloadfile(char *fname)
{
	char *id;
	Biobuf *b;
	Cert *cert;
	CertList *cl, **l;

	cl = nil;
	b = Bsafeopen(fname, OREAD);
	if(b == nil)
		return nil;

	debug(DbgAuth, "certprivall: opened db '%s'", fname);
	for(l=&cl; Brdcertpriv(b, &cert, &id) == 0; l=&(*l)->link){
		debug(DbgAuth, "certprivall: found key id='%s'", id);
		free(id);
		*l = mkcertlist(cert);
	}

	Bterm(b);
	return cl;
}

CertList*
certprivload(Conn *c)
{
	int i;
	Namelist *nl;
	CertList *cl, **l;

	cl = nil;
	nl = cfgnamelist(c, "usercertdb");
	for(i=0, l=&cl; i<nl->nstr; i++, l=*l?&(*l)->link:l){
		*l = certprivloadfile(nl->strtab[i]);
		if(*l == nil)
			error(c, "no keys in '%s': %r", nl->strtab[i]);
	}
	freenamelist(nl);

	return cl;
}
